/* GStreamer
 * Copyright (C) 1999,2000,2001,2002 Erik Walthinsen <omega@cse.ogi.edu>
 *                    2000,2001,2002 Wim Taymans <wtay@chello.be>
 *                              2002 Steve Baker <steve@stevebaker.org>
 *                              2003 Julien Moutte <julien@moutte.net>
 *
 * gstmediaplay.c: Media Player widget for Gst-Player
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <locale.h>
#include <config.h>
#include <libgnomevfs/gnome-vfs.h>
#include <gnome.h>
#include <gst/gconf/gconf.h>
#include <gconf/gconf-client.h>
#include <glade/glade.h>

#include "gstmediaplay.h"
#include "gstvideowidget.h"
#include "gstpreferences.h"
#include "gstmediainfo.h"
#include "gstcontrol.h"
#include "gstplayer-marshal.h"
#include "gtk-playlist.h"

struct _GstMediaPlayPrivate
{
  GladeXML *media_info_xml;

  GstPlay *play;

  GtkWidget *video_widget;
  GtkWidget *control;
  GtkWidget *media_info;

  GtkWindow *control_fs_window;
  GtkVBox *control_fs_vbox;

  GdkPixbuf *logo_pixbuf;
  GdkPixbuf *play_pixbuf;

  const gchar *location;
  const gchar *location_short;

  gboolean auto_resize;
  gboolean media_has_video;
  gboolean with_visualisation;
  GstElement *vis_element;

  GstMediaPlayMode display_mode;

  gboolean fs_control_bar_visible;

  gint64 length_nanos;
  gint64 current_time_nanos;

  /* For moving the control window */
  guint32 move_id;
  guint32 hide_cursor_id;
  int control_y;

  /* for housekeeping if ui has been touched since last error */
  gboolean ui_touched;
};

static const GtkTargetEntry target_table[] = {
  {"text/uri-list", 0, 0},
};

static GtkVBoxClass *parent_class = NULL;

enum
{
  STATE_CHANGE,
  DISPLAY_MODE_CHANGE,
  CURRENT_LOCATION,
  PLAYBACK_ERROR,
  LAST_SIGNAL
};

static gint gst_media_play_signals[LAST_SIGNAL] = { 0 };


/* ============================================================= */
/*                                                               */
/*                       Private Methods                         */
/*                                                               */
/* ============================================================= */

/* =========================================== */
/*                                             */
/*                  Tool Box                   */
/*                                             */
/* =========================================== */

static gchar *
gst_media_play_get_ui_file (const gchar * fname)
{
  gchar *full_fname;

  /* Searching for glade file in uninstalled dir */

  full_fname = g_strjoin ("/", GLADEUI_UNINSTALLED_DIR, fname, NULL);
  if (g_file_test (full_fname, G_FILE_TEST_IS_REGULAR))
    return full_fname;

  if (full_fname)
    g_free (full_fname);

  /* Searching for glade file in installed dir */

  full_fname = g_strjoin ("/", GLADEUI_DIR, fname, NULL);
  if (g_file_test (full_fname, G_FILE_TEST_IS_REGULAR))
    return full_fname;

  /* Not found */

  if (full_fname)
    g_free (full_fname);

  return NULL;
}

static GladeXML *
gst_media_play_get_glade_xml_from_file (const gchar * fname,
					const gchar * root,
					const gchar * domain,
					gboolean connect_signals)
{
  GladeXML *xml;

  xml = glade_xml_new (fname, root, domain);

  if (connect_signals)
    {
      /* connect the signals in the interface */
      glade_xml_signal_autoconnect (xml);
    }

  return xml;
}

static GladeXML *
gst_media_play_get_glade_xml (const gchar * fname,
			      const gchar * root,
			      const gchar * domain, gboolean connect_signals)
{
  gchar *full_fname = NULL;
  GladeXML *xml = NULL;
  full_fname = gst_media_play_get_ui_file (fname);
  g_return_val_if_fail (full_fname != NULL, NULL);
  xml = gst_media_play_get_glade_xml_from_file (full_fname,
						root,
						domain, connect_signals);
  if (full_fname)
    g_free (full_fname);
  return xml;
}

static void
gst_media_error_dialog (const gchar * message)
{
  GtkWidget *dialog;
  /* throw up a dialog box; FIXME: we don't have a parent */
  /* FIXME: maybe even fallback the GError and do error
   * handling higher up ? */
  dialog = gtk_message_dialog_new (NULL, 0,
				   GTK_MESSAGE_ERROR,
				   GTK_BUTTONS_CLOSE, message);
  gtk_dialog_run (GTK_DIALOG (dialog));
  /*
     gtk_widget_show (dialog);
     g_signal_connect_swapped (GTK_OBJECT (dialog), "response",
     G_CALLBACK (gtk_widget_destroy),
     GTK_OBJECT (dialog));
   */
}

/* =========================================== */
/*                                             */
/*                  GFX stuff                  */
/*                                             */
/* =========================================== */

static void
gst_media_play_toggle_playlist (GtkWidget * widget, GstMediaPlay * mplay)
{
  g_return_if_fail (mplay != NULL);
  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));

  if (GTK_WIDGET_VISIBLE (mplay->playlist))
    gtk_widget_hide (mplay->playlist);
  else
    gtk_widget_show (mplay->playlist);
}

static gboolean
animate_control_window (GstMediaPlay * mplay)
{
  static gint height_offset = 0;
  gboolean anim_done = FALSE;
  g_return_val_if_fail (mplay != NULL, FALSE);
  g_return_val_if_fail (GST_IS_MEDIA_PLAY (mplay), FALSE);

  if (!mplay->_priv->fs_control_bar_visible)
    {
      height_offset = height_offset + 2;
    }
  else
    {
      height_offset = height_offset - 2;
    }

  if (height_offset >= mplay->_priv->control->allocation.height)
    {
      height_offset = mplay->_priv->control->allocation.height;
      anim_done = TRUE;
    }

  if (height_offset <= 0)
    {
      height_offset = 0;
      anim_done = TRUE;
    }

  gtk_window_move (GTK_WINDOW (mplay->_priv->control_fs_window),
		   0, 0 - height_offset);

  /* We finished our animation */

  if (anim_done)
    {
      mplay->_priv->move_id = 0;
      return FALSE;
    }
  else
    return TRUE;
}

static gboolean
gst_media_play_hide_mouse_cursor (GstMediaPlay * mplay)
{
  g_return_val_if_fail (mplay != NULL, FALSE);
  g_return_val_if_fail (GST_IS_MEDIA_PLAY (mplay), FALSE);

  gst_video_widget_set_cursor_visible (GST_VIDEO_WIDGET
				       (mplay->_priv->video_widget), FALSE);

  gtk_widget_hide (GTK_WIDGET (mplay->_priv->control_fs_window));
  return TRUE;
}

static void
gst_media_play_set_fullscreen (GstMediaPlay * mplay,
			       gboolean fullscreen, gint width, gint height)
{
  g_return_if_fail (mplay != NULL);
  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));

  if (fullscreen)
    {

      if (GTK_WIDGET_VISIBLE (mplay->playlist))
	gtk_widget_hide (mplay->playlist);

      gtk_widget_hide (GTK_WIDGET (mplay->_priv->control));

      if (!mplay->_priv->control_fs_vbox)
	mplay->_priv->control_fs_vbox = GTK_VBOX (gtk_vbox_new (TRUE, 0));

      if (!mplay->_priv->control_fs_window)
	{
	  mplay->_priv->control_fs_window =
	    GTK_WINDOW (gtk_window_new (GTK_WINDOW_POPUP));
	  gtk_container_add (GTK_CONTAINER (mplay->_priv->control_fs_window),
			     GTK_WIDGET (mplay->_priv->control_fs_vbox));
	}

      /* Refcounting our control widget as we don't want to loose it */

      g_object_ref (mplay->_priv->control);

      gtk_container_remove (GTK_CONTAINER (mplay), mplay->_priv->control);
      gtk_box_pack_start (GTK_BOX (mplay->_priv->control_fs_vbox),
			  mplay->_priv->control, FALSE, FALSE, 0);

      g_object_unref (mplay->_priv->control);

      gtk_widget_show (GTK_WIDGET (mplay->_priv->control));
      gtk_widget_show (GTK_WIDGET (mplay->_priv->control_fs_vbox));
      gtk_widget_show (GTK_WIDGET (mplay->_priv->control_fs_window));

      gtk_window_move (mplay->_priv->control_fs_window, 0, 0);
      gtk_window_resize (mplay->_priv->control_fs_window,
			 width, mplay->_priv->control->allocation.height);

      gst_control_set_display_mode (GST_CONTROL (mplay->_priv->control),
				    GST_MEDIA_PLAY_FULLSCREEN);

      gst_video_widget_set_cursor_visible (GST_VIDEO_WIDGET
					   (mplay->_priv->video_widget),
					   FALSE);

      mplay->_priv->fs_control_bar_visible = FALSE;

      gtk_widget_hide (GTK_WIDGET (mplay->_priv->control_fs_window));
    }
  else
    {

      gtk_widget_hide (GTK_WIDGET (mplay->_priv->control_fs_window));

      /* Refcounting our control widget as we don't want to loose it */

      g_object_ref (mplay->_priv->control);

      gtk_container_remove (GTK_CONTAINER (mplay->_priv->control_fs_vbox),
			    mplay->_priv->control);
      gtk_box_pack_start (GTK_BOX (mplay),
			  mplay->_priv->control, FALSE, FALSE, 0);

      g_object_unref (mplay->_priv->control);

      if (mplay->_priv->move_id)
	{
	  g_source_remove (mplay->_priv->move_id);
	  mplay->_priv->move_id = 0;
	}

      gst_control_set_display_mode (GST_CONTROL (mplay->_priv->control),
				    GST_MEDIA_PLAY_NORMAL);

      gtk_widget_show (GTK_WIDGET (mplay->_priv->control));
    }
}

/* =========================================== */
/*                                             */
/*          Event Handlers, Callbacks          */
/*                                             */
/* =========================================== */

static void
gst_media_play_error (GstPlay * play, GstElement * orig,
		      char *error_message, GstMediaPlay * mplay)
{
  static char *previous = NULL;	/* remember previous ones */

  g_return_if_fail (mplay != NULL);
  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));

  /* if we already emitted this error, and no UI has been touched since,
     there's no point in showing it again. */
  if (previous &&
      (strncmp (previous, error_message, strlen (previous)) == 0) &&
      mplay->_priv->ui_touched == FALSE)
    return;

  /* this error needs to be shown, so do some housekeeping then signal */
  mplay->_priv->ui_touched = FALSE;
  if (previous)
    g_free (previous);
  previous = g_strdup (error_message);

  g_signal_emit (mplay, gst_media_play_signals[PLAYBACK_ERROR],
		 0, error_message);
}

static void
gst_media_play_have_vis_window (GstElement * element,
				gint xid, GstMediaPlay * mplay)
{
  gint width, height;
  g_return_if_fail (mplay != NULL);
  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));

  if ((mplay->_priv->video_widget) && (!mplay->_priv->media_has_video))
    {
      width = GTK_WIDGET (mplay->_priv->video_widget)->allocation.width;
      height = GTK_WIDGET (mplay->_priv->video_widget)->allocation.height;
      /*gst_video_widget_set_source_size (GST_VIDEO_WIDGET
					(mplay->_priv->video_widget), width,
					height);*/
      gst_video_widget_set_xembed_xid (GST_VIDEO_WIDGET
				       (mplay->_priv->video_widget), xid);
    }
}

static void
gst_media_play_current_removed (GtkWidget * widget, GstMediaPlay * mplay)
{
  gchar *mrl = NULL;

  g_return_if_fail (mplay != NULL);
  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));

  gtk_playlist_set_at_start (GTK_PLAYLIST (mplay->playlist));
  mrl = gtk_playlist_get_current_mrl (GTK_PLAYLIST (mplay->playlist));

  if (gst_media_play_get_state (mplay) == GST_STATE_PLAYING)
    {
      gst_media_play_set_state (mplay, GST_STATE_READY);
      if (mrl)
	{
	  gst_media_play_set_location (mplay, mrl);
	  gst_media_play_set_state (mplay, GST_STATE_PLAYING);
	}
    }

  if (mrl)
    g_free (mrl);
}

static void
gst_media_play_playlist_changed (GtkWidget * widget, GstMediaPlay * mplay)
{
  gchar *mrl = NULL;

  g_return_if_fail (mplay != NULL);
  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));

  mrl = gtk_playlist_get_current_mrl (GTK_PLAYLIST (mplay->playlist));

  if (mplay->_priv->location &&
      mrl && g_strcasecmp (mplay->_priv->location, mrl) != 0)
    {
      if (gst_media_play_get_state (mplay) == GST_STATE_PLAYING)
	{
	  gst_media_play_set_state (mplay, GST_STATE_READY);
	  gst_media_play_set_location (mplay, mrl);
	  gst_media_play_set_state (mplay, GST_STATE_PLAYING);
	}
      else
	gst_media_play_set_location (mplay, mrl);
    }

  if (mrl)
    g_free (mrl);

}

static gboolean
gst_media_play_add_droped_files (GstMediaPlay * mplay,
				 GtkSelectionData * data, gboolean empty_pl)
{
  GList *list, *p, *file_list;
  gboolean cleared = FALSE;

  list = gnome_vfs_uri_list_parse (data->data);

  if (list == NULL)
    return FALSE;

  p = list;
  file_list = NULL;

  while (p != NULL)
    {
      file_list = g_list_prepend (file_list,
				  gnome_vfs_uri_to_string
				  ((const GnomeVFSURI *) (p->data), 0));
      p = p->next;
    }

  gnome_vfs_uri_list_free (list);
  file_list = g_list_reverse (file_list);

  if (file_list == NULL)
    return FALSE;

  for (p = file_list; p != NULL; p = p->next)
    {
      char *filename;

      if (p->data == NULL)
	continue;

      /* We can't use g_filename_from_uri, as we don't know if
       * the uri is in locale or UTF8 encoding */
      filename = gnome_vfs_get_local_path_from_uri (p->data);
      if (filename == NULL)
	filename = g_strdup (p->data);

      if (filename != NULL &&
	  (g_file_test (filename, G_FILE_TEST_IS_REGULAR
			| G_FILE_TEST_EXISTS)
	   || strstr (filename, "://") != NULL))
	{
	  if (empty_pl == TRUE && cleared == FALSE)
	    {
	      /* The function that calls us knows better
	       * if we should be doing something with the 
	       * changed playlist ... */
	      g_signal_handlers_disconnect_by_func
		(G_OBJECT (mplay->playlist),
		 gst_media_play_playlist_changed, (gpointer) mplay);
	      gtk_playlist_clear (GTK_PLAYLIST (mplay->playlist));
	      cleared = TRUE;
	    }
	  gtk_playlist_add_mrl (GTK_PLAYLIST (mplay->playlist),
				filename, NULL);
	}
      g_free (filename);
      g_free (p->data);
    }

  g_list_free (file_list);

  /* ... and reconnect because we're nice people */
  if (cleared == TRUE)
    {
      g_signal_connect (G_OBJECT (mplay->playlist),
			"changed",
			G_CALLBACK (gst_media_play_playlist_changed),
			(gpointer) mplay);
      gst_media_play_set_location (mplay,
				   gtk_playlist_get_current_mrl (GTK_PLAYLIST
								 (mplay->
								  playlist)));
      gst_media_play_toggle_play_pause (GTK_WIDGET (mplay), mplay);
    }

  return TRUE;
}

static void
gst_media_play_drop_files (GtkWidget * widget,
			   GdkDragContext * context,
			   gint x,
			   gint y,
			   GtkSelectionData * data,
			   guint info, guint time, GstMediaPlay * mplay)
{
  gboolean retval;

  retval = gst_media_play_add_droped_files (mplay, data, TRUE);
  gtk_drag_finish (context, retval, FALSE, time);
}

static gint
gst_media_play_motion_notify_callback (GtkWidget * widget,
				       GdkEventMotion * event,
				       GstMediaPlay * mplay)
{
  gint pointer_x, pointer_y;
  GdkModifierType state;

  if (mplay->_priv->hide_cursor_id)
    g_source_remove (mplay->_priv->hide_cursor_id);

  gst_video_widget_set_cursor_visible (GST_VIDEO_WIDGET (widget), TRUE);

  if (mplay->_priv->display_mode == GST_MEDIA_PLAY_FULLSCREEN)
    {
      gtk_widget_show (GTK_WIDGET (mplay->_priv->control_fs_window));
    }

  gdk_window_get_pointer (widget->window, &pointer_x, &pointer_y, &state);

  if (mplay->_priv->display_mode == GST_MEDIA_PLAY_FULLSCREEN)
    mplay->_priv->hide_cursor_id = g_timeout_add (3000,
						  (GSourceFunc)
						  gst_media_play_hide_mouse_cursor,
						  mplay);

  return TRUE;
}

static void
gst_media_play_have_video_size (GstPlay * play,
				gint width, gint height, GstMediaPlay * mplay)
{
  GtkWidget *label;
  g_return_if_fail (mplay != NULL);
  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));
  
  if (label =
      glade_xml_get_widget (mplay->_priv->media_info_xml, "dimensions"))
    {
      gtk_label_set_text (GTK_LABEL (label),
			  g_strdup_printf ("%d x %d", width, height));
    }

  if (mplay->_priv->video_widget)
    {
      gst_video_widget_set_source_size (GST_VIDEO_WIDGET
					(mplay->_priv->video_widget), width,
					height);
    }
}

static void
gst_media_play_have_vis_size (GstPlay *play, gint width,
                              gint height, GstMediaPlay *mplay)
{
  GtkWidget *label;
  g_return_if_fail (mplay != NULL);
  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));
  
  if (mplay->_priv->media_has_video)
    return;
  
  if (label =
      glade_xml_get_widget (mplay->_priv->media_info_xml, "dimensions"))
    {
      gtk_label_set_text (GTK_LABEL (label),
			  g_strdup_printf ("%d x %d", width, height));
    }

  if (mplay->_priv->video_widget)
    {
      gst_video_widget_set_source_size (GST_VIDEO_WIDGET
					(mplay->_priv->video_widget), width,
					height);
    }
}

static void
gst_media_play_have_xid (GstPlay * play, gint xid, GstMediaPlay * mplay)
{
  g_return_if_fail (mplay != NULL);
  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));
  
  mplay->_priv->media_has_video = TRUE;

  /* Disconnecting goom */

  gst_play_connect_visualisation (mplay->_priv->play, FALSE);

  if (mplay->_priv->video_widget)
    gst_video_widget_set_xembed_xid (GST_VIDEO_WIDGET
				     (mplay->_priv->video_widget), xid);
}

static void
gst_media_play_got_length (GstPlay * play,
			   gint64 length_nanos, GstMediaPlay * mplay)
{
  gint seconds;
  gchar *time_str = NULL;
  GtkWidget *label;

  g_return_if_fail (mplay != NULL);
  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));

  seconds = (gint) (length_nanos / GST_SECOND);
  mplay->_priv->length_nanos = length_nanos;
  gst_control_set_enable_seek (GST_CONTROL (mplay->_priv->control), TRUE);
  gst_control_set_length (GST_CONTROL (mplay->_priv->control), length_nanos);
  time_str = gst_control_get_time_string (seconds);
  if (label = glade_xml_get_widget (mplay->_priv->media_info_xml, "duration"))
    gtk_label_set_text (GTK_LABEL (label), time_str);
  if (time_str)
    g_free (time_str);
}

static void
gst_media_play_time_tick (GstPlay * play,
			  gint64 time_nanos, GstMediaPlay * mplay)
{
  gdouble seek_pos;
  gint seconds;

  g_return_if_fail (mplay != NULL);
  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));

  mplay->_priv->current_time_nanos = time_nanos;

  seek_pos = ((gdouble) time_nanos) / ((gdouble) mplay->_priv->length_nanos);
  seconds = (gint) (time_nanos / GST_SECOND);

  gst_control_set_seek_pos (mplay->_priv->control, seek_pos);
  gst_control_set_time (GST_CONTROL (mplay->_priv->control), time_nanos);
}

static void
gst_media_play_volume_changed (GtkWidget * widget,
			       gdouble value, GstMediaPlay * mplay)
{
  g_return_if_fail (mplay != NULL);
  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));
  gst_play_set_volume (mplay->_priv->play, value);
}

static void
gst_media_play_display_mode_changed (GtkWidget * widget, GstMediaPlay * mplay)
{
  g_return_if_fail (mplay != NULL);
  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));
  g_signal_emit (mplay,
		 gst_media_play_signals[DISPLAY_MODE_CHANGE],
		 0, mplay->_priv->display_mode);
}

static void
gst_media_play_seek_to_pos (GtkWidget * widget,
			    gdouble value, GstMediaPlay * mplay)
{
  gdouble length_nanos;
  gint64 seek_time;

  g_return_if_fail (mplay != NULL);
  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));

  length_nanos = (gdouble) mplay->_priv->length_nanos;
  seek_time = (gint64) (length_nanos * value);
  gst_play_seek_to_time (mplay->_priv->play, seek_time);
}

static void
gst_media_play_state_change (GstPlay * play,
			     GstElementState old_state,
			     GstElementState new_state, GstMediaPlay * mplay)
{
  g_return_if_fail (mplay != NULL);
  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));
  if (new_state == GST_STATE_PLAYING)
    gtk_playlist_set_playing (GTK_PLAYLIST (mplay->playlist), TRUE);
  else
    gtk_playlist_set_playing (GTK_PLAYLIST (mplay->playlist), FALSE);
  gst_control_set_state (GST_CONTROL (mplay->_priv->control), old_state,
			 new_state);
}

static void
gst_media_play_stream_end (GstPlay * play, GstMediaPlay * mplay)
{
  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));

  gst_video_widget_set_logo_focus (GST_VIDEO_WIDGET
				   (mplay->_priv->video_widget), TRUE);

  gst_media_play_set_state (mplay, GST_STATE_READY);

  if (gtk_playlist_has_next_mrl (GTK_PLAYLIST (mplay->playlist)) ||
      gtk_playlist_get_repeat (GTK_PLAYLIST (mplay->playlist)))
    {
      gtk_playlist_set_next (GTK_PLAYLIST (mplay->playlist));
      gst_media_play_set_location (mplay,
				   gtk_playlist_get_current_mrl (GTK_PLAYLIST
								 (mplay->
								  playlist)));
      gst_media_play_set_state (mplay, GST_STATE_PLAYING);
    }
  else
    {
      gtk_playlist_set_at_start (GTK_PLAYLIST (mplay->playlist));
      gst_media_play_set_location (mplay,
				   gtk_playlist_get_current_mrl (GTK_PLAYLIST
								 (mplay->
								  playlist)));
    }
}

static void
gst_media_play_information (GstPlay * play,
			    GstObject * element,
			    GParamSpec * param, GstMediaPlay * mplay)
{
  GValue value = { 0, };
  GstCaps *metadata;
  GstProps *props;
  GList *p;

  if (strcmp (param->name, "metadata") == 0)
    {
      g_value_init (&value, param->value_type);
      g_object_get_property (G_OBJECT (element), param->name, &value);
      metadata = g_value_peek_pointer (&value);
      props = gst_caps_get_props (metadata);
      p = props->properties;
      while (p)
	{
	  const gchar *name;
	  const gchar *val;
	  GtkWidget *label;
	  GstPropsEntry *entry = (GstPropsEntry *) p->data;
	  name = gst_props_entry_get_name (entry);
	  gst_props_entry_get_string (entry, &val);
	  if (label = glade_xml_get_widget (mplay->_priv->media_info_xml,
					    g_ascii_strdown (name, -1)))
	    gtk_label_set_text (GTK_LABEL (label), val);
	  p = g_list_next (p);
	}
    }
}

/* =========================================== */
/*                                             */
/*              Init & Class init              */
/*                                             */
/* =========================================== */

static void
gst_media_play_destroy (GtkObject * object)
{
  GstMediaPlay *mplay;

  g_return_if_fail (object != NULL);
  g_return_if_fail (GST_IS_MEDIA_PLAY (object));

  mplay = GST_MEDIA_PLAY (object);
  
  if ((mplay->_priv->location) && GST_IS_PLAY (mplay->_priv->play))
    gst_media_play_set_state (mplay, GST_STATE_READY);

  if (mplay->_priv->hide_cursor_id)
    {
      g_source_remove (mplay->_priv->hide_cursor_id);
      mplay->_priv->hide_cursor_id = 0;
    }

  if (mplay->_priv->logo_pixbuf)
    {
      g_object_unref (mplay->_priv->logo_pixbuf);
      mplay->_priv->logo_pixbuf = NULL;
    }

  if (GTK_IS_WIDGET (mplay->_priv->video_widget))
    {
      gtk_widget_destroy (mplay->_priv->video_widget);
      mplay->_priv->video_widget = NULL;
    }

  if (GTK_IS_WIDGET (mplay->playlist))
    {
      gtk_widget_destroy (mplay->playlist);
      mplay->playlist = NULL;
    }

  if (GTK_IS_WIDGET (mplay->_priv->control))
    {
      gtk_widget_destroy (mplay->_priv->control);
      mplay->_priv->control = NULL;
    }

  if (mplay->_priv->play)
    {
      g_object_unref (mplay->_priv->play);
      mplay->_priv->play = NULL;
    }

  if (mplay->_priv->move_id)
    g_source_remove (mplay->_priv->move_id);
  if (mplay->_priv->hide_cursor_id)
    g_source_remove (mplay->_priv->hide_cursor_id);

  /* Chain up */
  if (GTK_OBJECT_CLASS (parent_class)->destroy)
    (*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void
gst_media_play_class_init (GstMediaPlayClass * klass)
{
  GtkWidgetClass *widget_class = (GtkWidgetClass *) klass;
  GtkObjectClass *object_class = (GtkObjectClass *) klass;
  GObjectClass *gobject_class = (GObjectClass *) klass;

  object_class->destroy = gst_media_play_destroy;

  parent_class = gtk_type_class (gtk_vbox_get_type ());

  gst_media_play_signals[STATE_CHANGE] =
    g_signal_new ("state_change",
		  G_TYPE_FROM_CLASS (klass),
		  G_SIGNAL_RUN_FIRST,
		  G_STRUCT_OFFSET (GstMediaPlayClass, state_changed),
		  NULL, NULL,
		  gst_marshal_VOID__INT_INT,
		  G_TYPE_NONE, 2, G_TYPE_INT, G_TYPE_INT);
  gst_media_play_signals[DISPLAY_MODE_CHANGE] =
    g_signal_new ("display_mode_change",
		  G_TYPE_FROM_CLASS (gobject_class),
		  G_SIGNAL_RUN_FIRST,
		  G_STRUCT_OFFSET (GstMediaPlayClass, display_mode_change),
		  NULL, NULL,
		  g_cclosure_marshal_VOID__INT, G_TYPE_NONE, 1, G_TYPE_INT);
  gst_media_play_signals[CURRENT_LOCATION] =
    g_signal_new ("current_location",
		  G_TYPE_FROM_CLASS (klass),
		  G_SIGNAL_RUN_FIRST,
		  G_STRUCT_OFFSET (GstMediaPlayClass, current_location),
		  NULL, NULL,
		  gst_play_marshal_VOID__STRING_STRING,
		  G_TYPE_NONE, 2, G_TYPE_STRING, G_TYPE_STRING);
  gst_media_play_signals[PLAYBACK_ERROR] =
    g_signal_new ("playback_error",
		  G_TYPE_FROM_CLASS (klass),
		  G_SIGNAL_RUN_FIRST,
		  G_STRUCT_OFFSET (GstMediaPlayClass, playback_error),
		  NULL, NULL,
		  g_cclosure_marshal_VOID__STRING,
		  G_TYPE_NONE, 1, G_TYPE_STRING);

}

static void
gst_media_play_init (GstMediaPlay * mplay)
{
  GstElement *audio_sink, *video_sink, *vis_video_sink;
  GError *error = NULL;
  gchar *logo_path = NULL;
  GstMediaPlayPrivate *priv;

  mplay->_priv = g_new0 (GstMediaPlayPrivate, 1);
  priv = mplay->_priv;

  /* play object */
  priv->play = gst_play_new (GST_PLAY_PIPE_VIDEO_VISUALISATION, &error);
  if (error != NULL)
    {
      gst_media_error_dialog (error->message);
      g_error_free (error);
    }

  g_return_if_fail (mplay->_priv->play != NULL);

  audio_sink = gst_gconf_get_default_audio_sink ();

  if (!GST_IS_ELEMENT (audio_sink))
    {
      gst_media_error_dialog
	("Could not render default GStreamer audio output sink "
	 "from GConf /system/gstreamer/default/audiosink key. "
	 "Check if it is set correctly.");
      return;
    }

  video_sink = gst_gconf_get_default_video_sink ();
  vis_video_sink = gst_gconf_get_default_video_sink ();

  if (!GST_IS_ELEMENT (video_sink) || !GST_IS_ELEMENT (vis_video_sink))
    {
      gst_media_error_dialog
	("Could not render default GStreamer video output sink "
	 "from GConf /system/gstreamer/default/videosink key. "
	 "Check if it is set correctly.");
      return;
    }

  priv->vis_element = gst_gconf_get_default_visualisation_element ();

  gst_play_set_video_sink (priv->play, video_sink);

  gst_play_set_audio_sink (priv->play, audio_sink);

  gst_play_set_visualisation_video_sink (priv->play, vis_video_sink);

  gst_play_set_visualisation_element (priv->play, priv->vis_element);

  mplay->_priv->media_has_video = FALSE;

  g_signal_connect (G_OBJECT (priv->play), "have_vis_xid",
		    G_CALLBACK (gst_media_play_have_vis_window), mplay);

  g_signal_connect (G_OBJECT (priv->play), "stream_end",
		    G_CALLBACK (gst_media_play_stream_end), mplay);
  g_signal_connect (G_OBJECT (priv->play), "information",
		    G_CALLBACK (gst_media_play_information), mplay);
  g_signal_connect (G_OBJECT (priv->play), "time_tick",
		    G_CALLBACK (gst_media_play_time_tick), mplay);
  g_signal_connect (G_OBJECT (priv->play), "stream_length",
		    G_CALLBACK (gst_media_play_got_length), mplay);
  g_signal_connect (G_OBJECT (priv->play), "have_xid",
		    G_CALLBACK (gst_media_play_have_xid), mplay);
  g_signal_connect (G_OBJECT (priv->play), "have_video_size",
		    G_CALLBACK (gst_media_play_have_video_size), mplay);
  g_signal_connect (G_OBJECT (priv->play), "have_vis_size",
		    G_CALLBACK (gst_media_play_have_vis_size), mplay);
  g_signal_connect (G_OBJECT (priv->play), "state_change",
		    G_CALLBACK (gst_media_play_state_change), mplay);
  g_signal_connect (G_OBJECT (priv->play), "pipeline_error",
		    G_CALLBACK (gst_media_play_error), mplay);

  gtk_box_set_homogeneous (GTK_BOX (mplay), FALSE);
  gtk_box_set_spacing (GTK_BOX (mplay), 0);
  /* player widgets */

  priv->control =
    gst_control_new (gst_media_play_get_glade_xml
		     ("mediacontrol.glade", "hbox_controller", NULL, FALSE),
		     gst_media_play_get_glade_xml ("mediacontrol.glade",
						   "window_volume_popup",
						   NULL, FALSE));
  g_return_if_fail (priv->control != NULL);
  gtk_box_pack_start (GTK_BOX (mplay), GTK_WIDGET (priv->control), FALSE,
		      FALSE, 0);

  priv->control_fs_window = NULL;
  priv->control_fs_vbox = NULL;

  priv->video_widget = gst_video_widget_new ();
  g_return_if_fail (mplay->_priv->video_widget != NULL);

  gst_video_widget_set_minimum_size (GST_VIDEO_WIDGET (priv->video_widget),
				     300, 300);

  logo_path = gst_media_play_get_ui_file ("gst-player-logo.png");

  priv->logo_pixbuf = gdk_pixbuf_new_from_file (logo_path, NULL);
  if (logo_path)
    {
      g_free (logo_path);
      logo_path = NULL;
    }

  gst_video_widget_set_logo (GST_VIDEO_WIDGET (priv->video_widget),
			     priv->logo_pixbuf);

  /* Packing the video widget at the end of our Mplay */

  gtk_box_pack_end (GTK_BOX (mplay), priv->video_widget, TRUE, TRUE, 0);

  logo_path = gst_media_play_get_ui_file ("gst-player-playing.png");

  priv->play_pixbuf = gdk_pixbuf_new_from_file (logo_path, NULL);
  if (logo_path)
    {
      g_free (logo_path);
      logo_path = NULL;
    }

  mplay->playlist =
    gtk_playlist_new (gst_media_play_get_ui_file ("playlist.glade"),
		      priv->play_pixbuf);

  g_return_if_fail (mplay->playlist != NULL);

  priv->media_info_xml = gst_media_play_get_glade_xml ("mediacontrol.glade",
						       "media-properties",
						       NULL, TRUE);

  priv->media_info = glade_xml_get_widget (priv->media_info_xml,
					   "media-properties");

  /* Playlist signals */

  g_signal_connect (G_OBJECT (mplay->playlist), "delete-event",
		    G_CALLBACK (gtk_widget_hide_on_delete), NULL);

  g_signal_connect (G_OBJECT (mplay->playlist), "changed",
		    G_CALLBACK (gst_media_play_playlist_changed), mplay);
  g_signal_connect (G_OBJECT (mplay->playlist), "current-removed",
		    G_CALLBACK (gst_media_play_current_removed), mplay);

  /* VideoWidget signals */

  g_signal_connect (G_OBJECT (priv->video_widget), "motion-notify-event",
		    G_CALLBACK (gst_media_play_motion_notify_callback),
		    mplay);

  /* Control signals */

  g_signal_connect (G_OBJECT (priv->control), "fullscreen_toggled",
		    G_CALLBACK (gst_media_play_display_mode_changed), mplay);
  g_signal_connect (G_OBJECT (priv->control), "toggle_play",
		    G_CALLBACK (gst_media_play_toggle_play_pause), mplay);
  g_signal_connect (G_OBJECT (priv->control), "toggle_playlist",
		    G_CALLBACK (gst_media_play_toggle_playlist), mplay);
  g_signal_connect (G_OBJECT (priv->control), "toggle_info",
		    G_CALLBACK (gst_media_play_show_media_info), mplay);
  g_signal_connect (G_OBJECT (priv->control), "clicked_next",
		    G_CALLBACK (gst_media_play_next), mplay);
  g_signal_connect (G_OBJECT (priv->control), "clicked_previous",
		    G_CALLBACK (gst_media_play_previous), mplay);
  g_signal_connect (G_OBJECT (priv->control), "stop",
		    G_CALLBACK (gst_media_play_stop), mplay);
  g_signal_connect (G_OBJECT (priv->control), "volume_change",
		    G_CALLBACK (gst_media_play_volume_changed), mplay);
  g_signal_connect (G_OBJECT (priv->control), "seek_change",
		    G_CALLBACK (gst_media_play_seek_to_pos), mplay);

  g_signal_connect (G_OBJECT (mplay), "drag_data_received",
		    G_CALLBACK (gst_media_play_drop_files), mplay);

  gtk_drag_dest_set (GTK_WIDGET (mplay), GTK_DEST_DEFAULT_ALL,
		     target_table, 1, GDK_ACTION_COPY);

  mplay->_priv->display_mode = GST_MEDIA_PLAY_NORMAL;

  gtk_widget_show (priv->control);
  gtk_widget_show (priv->video_widget);
  priv->location = NULL;
  gst_media_gconf_get_boolean ("visualisation",
			       &(mplay->_priv->with_visualisation));
}

/* ============================================================= */
/*                                                               */
/*                       Public Methods                          */
/*                                                               */
/* ============================================================= */

/* =========================================== */
/*                                             */
/*   Play/Pause, Stop, Next, Previous, Mute    */
/*                                             */
/* =========================================== */

void
gst_media_play_toggle_play_pause (GtkWidget * widget, GstMediaPlay * mplay)
{
  GstElementState playstate;
  g_return_if_fail (mplay != NULL);
  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));

  playstate = gst_play_get_state (mplay->_priv->play);

  mplay->_priv->ui_touched = TRUE;
  if (playstate == GST_STATE_PLAYING)
    gst_media_play_set_state (mplay, GST_STATE_PAUSED);
  else if (playstate == GST_STATE_PAUSED)
    gst_media_play_set_state (mplay, GST_STATE_PLAYING);
  else if (playstate == GST_STATE_READY)
    {
      if (!gtk_playlist_get_current_mrl (GTK_PLAYLIST (mplay->playlist)))
	return;
      gst_media_play_set_location (mplay,
				   gtk_playlist_get_current_mrl (GTK_PLAYLIST
								 (mplay->
								  playlist)));
      gst_media_play_set_state (mplay, GST_STATE_PLAYING);
    }
}

void
gst_media_play_stop (GtkWidget * widget, GstMediaPlay * mplay)
{
  g_return_if_fail (mplay != NULL);
  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));
  mplay->_priv->ui_touched = TRUE;
  gst_media_play_set_state (mplay, GST_STATE_READY);
}

void
gst_media_play_next (GtkWidget * widget, GstMediaPlay * mplay)
{
  g_return_if_fail (mplay != NULL);
  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));
  mplay->_priv->ui_touched = TRUE;

  if (!gtk_playlist_get_current_mrl (GTK_PLAYLIST (mplay->playlist)))
    return;

  if (gtk_playlist_has_next_mrl (GTK_PLAYLIST (mplay->playlist)) ||
      gtk_playlist_get_repeat (GTK_PLAYLIST (mplay->playlist)))
    {
      gst_video_widget_set_logo_focus (GST_VIDEO_WIDGET
				       (mplay->_priv->video_widget), TRUE);

      gst_media_play_set_state (mplay, GST_STATE_READY);

      gtk_playlist_set_next (GTK_PLAYLIST (mplay->playlist));
      gst_media_play_set_location (mplay,
				   gtk_playlist_get_current_mrl (GTK_PLAYLIST
								 (mplay->
								  playlist)));

      gst_media_play_set_state (mplay, GST_STATE_PLAYING);
    }
}

void
gst_media_play_previous (GtkWidget * widget, GstMediaPlay * mplay)
{
  g_return_if_fail (mplay != NULL);
  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));
  mplay->_priv->ui_touched = TRUE;

  if (!gtk_playlist_get_current_mrl (GTK_PLAYLIST (mplay->playlist)))
    return;

  if (gtk_playlist_has_previous_mrl (GTK_PLAYLIST (mplay->playlist)) ||
      gtk_playlist_get_repeat (GTK_PLAYLIST (mplay->playlist)))
    {
      gst_video_widget_set_logo_focus (GST_VIDEO_WIDGET
				       (mplay->_priv->video_widget), TRUE);

      gst_media_play_set_state (mplay, GST_STATE_READY);

      gtk_playlist_set_previous (GTK_PLAYLIST (mplay->playlist));
      gst_media_play_set_location (mplay,
				   gtk_playlist_get_current_mrl (GTK_PLAYLIST
								 (mplay->
								  playlist)));

      gst_media_play_set_state (mplay, GST_STATE_PLAYING);
    }
}

void
gst_media_play_toggle_mute (GtkWidget * widget, GstMediaPlay * mplay)
{
  gboolean muted;

  g_return_if_fail (mplay != NULL);
  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));
  mplay->_priv->ui_touched = TRUE;

  muted = !gst_play_get_mute (mplay->_priv->play);
  gst_play_set_mute (mplay->_priv->play, muted);
  gst_control_set_mute (GST_CONTROL (mplay->_priv->control), muted);
}

/* =========================================== */
/*                                             */
/*                  GFX stuff                  */
/*                                             */
/* =========================================== */

void
gst_media_play_show_media_info (GtkWidget * widget, GstMediaPlay * mplay)
{
  g_return_if_fail (mplay != NULL);
  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));

  if (GTK_WIDGET_VISIBLE (mplay->_priv->media_info))
    gtk_widget_hide (mplay->_priv->media_info);
  else
    gtk_widget_show (mplay->_priv->media_info);
}

void
gst_media_play_toggle_visualisation (GstMediaPlay * mplay)
{
  g_return_if_fail (mplay != NULL);
  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));

  if (!mplay->_priv->media_has_video)
    {
      if (mplay->_priv->with_visualisation)
	mplay->_priv->with_visualisation = FALSE;
      else
	mplay->_priv->with_visualisation = TRUE;
      
      gst_media_gconf_set_boolean ("visualisation",
                                   mplay->_priv->with_visualisation);

      if (!mplay->_priv->with_visualisation)
	{
	  gst_video_widget_set_logo_focus (GST_VIDEO_WIDGET
					   (mplay->_priv->video_widget),
					   TRUE);
	}
      else
	{
	  gst_video_widget_set_logo_focus (GST_VIDEO_WIDGET
					   (mplay->_priv->video_widget),
					   FALSE);

	}

      gst_play_connect_visualisation (mplay->_priv->play,
				      mplay->_priv->with_visualisation);

    }

}

void
gst_media_play_show_about (GstMediaPlay * mplay)
{
  static GtkWidget *about;
  char *description, *about_logo_path;
  int major, minor, micro;
  GdkPixbuf *about_pixbuf = NULL;

  const gchar *authors[] = {
    "Maintainer :",
    "Julien Moutte <julien@moutte.net>",
    "",
    "Contributors :",
    "Steve Baker <stevebaker_org@yahoo.co.uk>",
    "Richard Boulton <richard@tartarus.org>",
    "Ronald Bultje <rbultje@ronald.bitfreak.net>",
    "Arik Devens <arik@gnome.org>",
    "Chris Emerson (PPC port)",
    "Benjamin Otte <in7y118@public.uni-hamburg.de>",
    "Wim Taymans <wim.taymans@tvd.be>",
    "Erik Walthinsen <omega@cse.ogi.edu>",
    "Thomas Vander Stichele <thomas@apestaart.org>",
    NULL
  };

  if (about != NULL)
    {
      gdk_window_raise (about->window);
      gdk_window_show (about->window);
      return;
    }

  gst_version (&major, &minor, &micro);

  description =
    g_strdup_printf (_("GStreamer based media player using backend version %d.%d.%d"),
		     major, minor, micro);
    
  about_logo_path = gst_media_play_get_ui_file ("about-logo.png");

  about_pixbuf = gdk_pixbuf_new_from_file (about_logo_path, NULL);
  if (about_logo_path)
    {
      g_free (about_logo_path);
      about_logo_path = NULL;
    }

  about = gnome_about_new ("GStreamer Player",
			   GST_PLAYER_VERSION,
			   "(C) 1999-2003 The GStreamer Team",
			   (const char *) description,
			   (const char **) authors,
                           NULL, NULL, about_pixbuf);

  g_free (description);

  g_signal_connect (G_OBJECT (about), "destroy",
		    G_CALLBACK (gtk_widget_destroyed), &about);

  gtk_widget_show (about);
}

void
gst_media_play_set_display_mode (GstMediaPlay * mplay,
				 GstMediaPlayMode display_mode,
				 gint width, gint height)
{
  g_return_if_fail (mplay != NULL);
  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));

  gst_control_set_display_mode (GST_CONTROL (mplay->_priv->control),
				display_mode);

  switch (display_mode)
    {
    case GST_MEDIA_PLAY_FULLSCREEN:
      if (mplay->_priv->display_mode == GST_MEDIA_PLAY_NORMAL)
	gst_media_play_set_fullscreen (mplay, TRUE, width, height);
      break;
    case GST_MEDIA_PLAY_NORMAL:
      if (mplay->_priv->display_mode == GST_MEDIA_PLAY_FULLSCREEN)
	gst_media_play_set_fullscreen (mplay, FALSE, width, height);
      break;
    case GST_MEDIA_PLAY_NOCONTROL:
      break;
    default:
      g_warning ("unknown mode %d\n", display_mode);
      break;
    }
  mplay->_priv->display_mode = display_mode;
}

GstMediaPlayMode
gst_media_play_get_display_mode (GstMediaPlay * mplay)
{
  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));
  return mplay->_priv->display_mode;
}

/* =========================================== */
/*                                             */
/*                GstPlay stuff                */
/*                                             */
/* =========================================== */

void
gst_media_play_set_video_scale (GstMediaPlay * mplay, gfloat scale_factor)
{
  g_return_if_fail (mplay != NULL);
  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));
  if (mplay->_priv->video_widget)
    {
      gst_video_widget_set_scale (GST_VIDEO_WIDGET
				  (mplay->_priv->video_widget), scale_factor);
      gst_video_widget_set_scale_override (GST_VIDEO_WIDGET
					   (mplay->_priv->video_widget),
					   TRUE);
    }
}

void
gst_media_play_set_video_scale_override (GstMediaPlay * mplay,
					 gboolean override)
{
  g_return_if_fail (mplay != NULL);
  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));
  if (mplay->_priv->video_widget)
    {
      gst_video_widget_set_scale_override (GST_VIDEO_WIDGET
					   (mplay->_priv->video_widget),
					   override);
    }
}

gboolean
gst_media_play_get_video_scale_override (GstMediaPlay * mplay)
{
  g_return_if_fail (mplay != NULL);
  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));
  if (mplay->_priv->video_widget)
    return
      gst_video_widget_get_scale_override (GST_VIDEO_WIDGET
					   (mplay->_priv->video_widget));
}

gfloat
gst_media_play_get_video_scale (GstMediaPlay * mplay)
{
  gfloat scale_factor = 1.0;
  g_return_if_fail (mplay != NULL);
  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));
  if (mplay->_priv->video_widget)
    g_object_get (G_OBJECT (mplay->_priv->video_widget),
		  "scale_factor", &scale_factor, NULL);
  return scale_factor;
}

GstElement *
gst_media_play_get_visual (GstMediaPlay * mplay)
{
  g_return_val_if_fail (GST_IS_MEDIA_PLAY (mplay), NULL);
  return mplay->_priv->vis_element;
}

gboolean
gst_media_play_set_visual (GstMediaPlay * mplay, const char *name)
{
  gboolean paused = FALSE;
  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));

  if (gst_play_get_state (mplay->_priv->play) == GST_STATE_PLAYING)
    {
      gst_play_set_state (mplay->_priv->play, GST_STATE_PAUSED);
      paused = TRUE;
    }

  mplay->_priv->vis_element =
    gst_element_factory_make (name, "vis_plugin_element");
  if (GST_IS_ELEMENT (mplay->_priv->vis_element))
    {
      gst_play_set_visualisation_element (mplay->_priv->play,
					  mplay->_priv->vis_element);
      if (paused)
	{
	  gst_play_seek_to_time (mplay->_priv->play,
				 mplay->_priv->current_time_nanos);
	  gst_play_set_state (mplay->_priv->play, GST_STATE_PLAYING);
	}
        
      return TRUE;
    }
  else
    {
      g_warning ("Failed loading visualization plugin %s", name);
      return FALSE;
    }
}

gboolean
gst_media_play_set_effect (GstMediaPlay * mplay, const char *name)
{
  gboolean paused = FALSE;
  GString *videosink_desc;
  GstElement *videosink_bin, *vis_videosink_bin;
  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));
  
  if (gst_video_widget_get_xembed_xid (
        GST_VIDEO_WIDGET (mplay->_priv->video_widget)) != 0)
    {
      g_warning ("already have a video window");
      return FALSE;
    }

  videosink_desc = g_string_sized_new (128);
 
  if (gst_play_get_state (mplay->_priv->play) == GST_STATE_PLAYING)
    {
      gst_play_set_state (mplay->_priv->play, GST_STATE_READY);
      paused = TRUE;
    }

  videosink_desc = g_string_append (videosink_desc,
                                    gst_gconf_get_string ("default/videosink"));
  
  if (strcmp (name, "None") !=0)
    {
      videosink_desc = g_string_prepend (videosink_desc, " ! colorspace ! ");
      videosink_desc = g_string_prepend (videosink_desc, name);
    }
    
  videosink_bin = gst_gconf_render_bin_from_description (videosink_desc->str);
  vis_videosink_bin = gst_gconf_render_bin_from_description (videosink_desc->str);
  
  g_string_free (videosink_desc, TRUE);
  
  if (GST_IS_ELEMENT (videosink_bin))
    {
      gst_play_set_video_sink (mplay->_priv->play,
                               videosink_bin);
      gst_play_set_visualisation_video_sink (mplay->_priv->play,
                                             vis_videosink_bin);
      if (paused)
	{
	  gst_play_seek_to_time (mplay->_priv->play,
				 mplay->_priv->current_time_nanos);
	  gst_play_set_state (mplay->_priv->play, GST_STATE_PLAYING);
	}

      return TRUE;
    }
  else
    {
      g_warning ("Failed loading effect plugin %s", name);
      return FALSE;
    }
}

void
gst_media_play_set_state (GstMediaPlay * mplay, GstElementState state)
{
  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));
  g_return_if_fail (mplay->_priv->location != NULL);
  gst_play_set_state (mplay->_priv->play, state);
}

GstElementState
gst_media_play_get_state (GstMediaPlay * mplay)
{
  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));
  return gst_play_get_state (mplay->_priv->play);
}

void
gst_media_play_set_location (GstMediaPlay * mplay, const gchar * location)
{
  GnomeVFSFileInfo info;
  GstElement *datasrc = NULL;

  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));

  if (!location)
    return;

  mplay->_priv->location_short = NULL;
  mplay->_priv->location = location;

  mplay->_priv->media_has_video = FALSE;

  gst_video_widget_set_source_size (GST_VIDEO_WIDGET
				    (mplay->_priv->video_widget), 1, 1);
  gst_video_widget_set_logo_focus (GST_VIDEO_WIDGET
				   (mplay->_priv->video_widget), TRUE);

  /* Connecting visualisation if enabled */

  if (mplay->_priv->with_visualisation)
    gst_play_connect_visualisation (mplay->_priv->play, TRUE);

  gst_play_need_new_video_window (mplay->_priv->play);

  /* get a short filename if we can */
  if (GNOME_VFS_OK !=
      gnome_vfs_get_file_info (location, &info, GNOME_VFS_FILE_INFO_DEFAULT))
    mplay->_priv->location_short = location;
  else
    mplay->_priv->location_short = info.name;

  g_signal_emit (mplay, gst_media_play_signals[CURRENT_LOCATION],
		 0, mplay->_priv->location, mplay->_priv->location_short);

  if (g_file_test (location, G_FILE_TEST_EXISTS))
    {
      datasrc = gst_element_factory_make ("filesrc", "source");
      if (GST_IS_ELEMENT (datasrc))
	gst_play_set_data_src (mplay->_priv->play, datasrc);
      gst_play_set_location (mplay->_priv->play, mplay->_priv->location);
    }
  else if (g_str_has_prefix (location, "cda://"))
    {
      datasrc = gst_element_factory_make ("cdparanoia", "source");
      if (GST_IS_ELEMENT (datasrc))
	gst_play_set_data_src (mplay->_priv->play, datasrc);
      gst_play_set_location (mplay->_priv->play, "audio-cd");
    }
  else if (g_str_has_prefix (location, "v4l://"))
    {
      datasrc = gst_element_factory_make ("v4lsrc", "source");
      if (GST_IS_ELEMENT (datasrc))
	gst_play_set_data_src (mplay->_priv->play, datasrc);
      gst_play_set_location (mplay->_priv->play, "webcam");
    }
  else if (g_str_has_prefix (location, "fd://"))
    {
      GstElement *fdsrc, *queue;
      datasrc = gst_element_factory_make ("thread", "source");
      fdsrc = gst_element_factory_make ("fdsrc", "thread_fdsrc");
      queue = gst_element_factory_make ("queue", "thread_queue");
      gst_bin_add_many (GST_BIN (datasrc), fdsrc, queue, NULL);
      gst_element_link (fdsrc, queue);

      gst_element_add_ghost_pad (datasrc,
				 gst_element_get_pad (queue, "src"), "src");

      g_object_set (G_OBJECT (queue), "max-level", 1000000, NULL);
      if (GST_IS_ELEMENT (datasrc))
	gst_play_set_data_src (mplay->_priv->play, datasrc);
      gst_play_set_location (mplay->_priv->play, "file-descriptor");
    }
  else
    {
      datasrc = gst_element_factory_make ("gnomevfssrc", "source");
      if (GST_IS_ELEMENT (datasrc))
	gst_play_set_data_src (mplay->_priv->play, datasrc);
      gst_play_set_location (mplay->_priv->play, mplay->_priv->location);
    }
}

const gchar *
gst_media_play_get_location (GstMediaPlay * mplay)
{
  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));
  return mplay->_priv->location;
}

void
gst_media_play_set_volume (GstMediaPlay * mplay, gdouble value)
{
  GtkAdjustment *adj;
  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));
  adj =
    gst_control_get_volume_adjustment (GST_CONTROL (mplay->_priv->control));

  value = value > 1.0 ? 1.0 : value;
  value = value < 0.0 ? 0.0 : value;
  gtk_adjustment_set_value (adj, value);
}

gdouble
gst_media_play_get_volume (GstMediaPlay * mplay)
{
  GtkAdjustment *adj;
  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));
  adj =
    gst_control_get_volume_adjustment (GST_CONTROL (mplay->_priv->control));
  return gtk_adjustment_get_value (adj);
}

/* =========================================== */
/*                                             */
/*                  Tool Box                   */
/*                                             */
/* =========================================== */

void
gst_media_play_ignore_playlist_changes (GstMediaPlay * mplay, gboolean ignore)
{
  g_return_if_fail (mplay != NULL);
  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));

  if (ignore)
    g_signal_handlers_disconnect_by_func (G_OBJECT (mplay->playlist),
					  gst_media_play_playlist_changed,
					  (gpointer) mplay);
  else
    g_signal_connect (G_OBJECT (mplay->playlist),
		      "changed",
		      G_CALLBACK (gst_media_play_playlist_changed),
		      (gpointer) mplay);
}

gboolean
gst_media_gconf_set_boolean (const gchar *key, gboolean value)
{
  gchar *full_key;
  GError *error = NULL;

  static GConfClient *_gst_media_gconf_client = NULL;

  if (!_gst_media_gconf_client)
    _gst_media_gconf_client = gconf_client_get_default ();

  full_key = g_strdup_printf ("%s/%s", GCONF_PREFIX, key);
  gconf_client_set_bool (_gst_media_gconf_client, full_key, value, &error);
  g_free (full_key);
  if (error)
    {
      g_warning ("GConf: %s\n", error->message);
      g_error_free (error);
      return FALSE;
    }
  return TRUE;
}

gboolean
gst_media_gconf_get_boolean (const gchar * key, gboolean * value)
{
  gchar *full_key;
  GError *error = NULL;

  static GConfClient *_gst_media_gconf_client = NULL;

  if (!_gst_media_gconf_client)
    _gst_media_gconf_client = gconf_client_get_default ();

  full_key = g_strdup_printf ("%s/%s", GCONF_PREFIX, key);
  *value = gconf_client_get_bool (_gst_media_gconf_client, full_key, &error);
  g_free (full_key);
  if (error)
    {
      g_warning ("GConf: %s\n", error->message);
      g_error_free (error);
      return FALSE;
    }
  return TRUE;
}

gint
gst_media_play_handle_keypress (GtkWidget * widget,
				GdkEventKey * event, GstMediaPlay * mplay)
{
  guint state;
  gboolean handled = FALSE;

  g_return_val_if_fail (GST_IS_MEDIA_PLAY (mplay), 0);

  state = event->state;

  switch (event->keyval)
    {
    case GDK_l:
      if (gst_video_widget_get_logo_focus
	  (GST_VIDEO_WIDGET (mplay->_priv->video_widget)))
	gst_video_widget_set_logo_focus (GST_VIDEO_WIDGET
					 (mplay->_priv->video_widget), FALSE);
      else
	gst_video_widget_set_logo_focus (GST_VIDEO_WIDGET
					 (mplay->_priv->video_widget), TRUE);
      break;
    case GDK_Up:
      gst_media_play_set_volume (mplay,
				 gst_media_play_get_volume (mplay) + 0.1);
      handled = TRUE;
      break;
    case GDK_Down:
      gst_media_play_set_volume (mplay,
				 gst_media_play_get_volume (mplay) - 0.1);
      handled = TRUE;
      break;
    case GDK_space:
    case GDK_p:
      if (gst_media_play_get_state (mplay) == GST_STATE_PLAYING)
	{
	  gst_media_play_set_state (mplay, GST_STATE_PAUSED);
	}
      else
	{
	  if (gtk_playlist_get_current_mrl (GTK_PLAYLIST (mplay->playlist)))
	    gst_media_play_set_state (mplay, GST_STATE_PLAYING);
	}
      handled = TRUE;
      break;
    case GDK_Return:
    case GDK_Escape:
      if (mplay->_priv->display_mode == GST_MEDIA_PLAY_FULLSCREEN)
	{
	  gst_media_play_set_display_mode (mplay,
					   GST_MEDIA_PLAY_NORMAL, 0, 0);
	}
      handled = TRUE;
      break;
    }

  return handled;
}

/* =========================================== */
/*                                             */
/*          Widget typing & Creation           */
/*                                             */
/* =========================================== */

GType
gst_media_play_get_type (void)
{
  static GType play_type = 0;

  if (!play_type)
    {
      static const GTypeInfo style_info = {
	sizeof (GstMediaPlayClass),
	(GBaseInitFunc) NULL,
	(GBaseFinalizeFunc) NULL,
	(GClassInitFunc) gst_media_play_class_init,
	NULL,			/* class_finalize */
	NULL,			/* class_data */
	sizeof (GstMediaPlay),
	0,			/* n_preallocs */
	(GInstanceInitFunc) gst_media_play_init,
      };

      play_type = g_type_register_static (GTK_TYPE_VBOX,
					  "GstMediaPlay", &style_info, 0);
    }

  return play_type;
}

GstMediaPlay *
gst_media_play_new (void)
{
  return GST_MEDIA_PLAY (g_object_new (GST_TYPE_MEDIA_PLAY, NULL));
}

/* ============================================================= */
/*                                                               */
/*                    Unused for the moment                      */
/*                                                               */
/* ============================================================= */

void
gst_media_play_show_preferences (GstMediaPlay * mplay)
{
  GtkWidget *preferences;

  g_return_if_fail (GST_IS_MEDIA_PLAY (mplay));

  preferences = gst_preferences_new ();

  gtk_widget_show (preferences);
}
