/* IceGradient theme for gtk, hacked over ThinIce by Tim Gerla and Tomas gren ThinIce,
   which is based on raster's Motif theme and the Metal theme.
   Authors: 
            JM Perez <jose.perez@upcnet.es>
*/

#include "icegradient_theme.h"
#include "gradient.h"


#define DETAIL(xx) (detail && g_str_equal(xx, detail))

#ifdef DEBUG
#define dump_widget() if (widget && detail) printf("Widget: %s, Detail: %s, Function: %s\n", gtk_type_name(GTK_WIDGET_TYPE(widget)), detail, __PRETTY_FUNCTION__); \
  else if (widget) printf("Widget: %s, Function: %s\n", gtk_type_name(GTK_WIDGET_TYPE(widget)), __PRETTY_FUNCTION__); \
  else printf("Null widget on function: %s\n", __PRETTY_FUNCTION__)
#else
#define dump_widget()
#endif


extern GtkStyleClass icegradient_default_class;

/* internal functions */
static void draw_hline (GtkStyle * style,
			GdkWindow * window,
			GtkStateType state_type,
			GdkRectangle * area,
			GtkWidget * widget,
			gchar * detail, gint x1, gint x2, gint y);
static void draw_vline (GtkStyle * style,
			GdkWindow * window,
			GtkStateType state_type,
			GdkRectangle * area,
			GtkWidget * widget,
			gchar * detail, gint y1, gint y2, gint x);
static void draw_shadow (GtkStyle * style,
			 GdkWindow * window,
			 GtkStateType state_type,
			 GtkShadowType shadow_type,
			 GdkRectangle * area,
			 GtkWidget * widget,
			 gchar * detail,
			 gint x, gint y, gint width, gint height);

static void draw_polygon (GtkStyle * style,
			  GdkWindow * window,
			  GtkStateType state_type,
			  GtkShadowType shadow_type,
			  GdkRectangle * area,
			  GtkWidget * widget,
			  gchar * detail,
			  GdkPoint * point, gint npoints, gint fill);
static void draw_arrow (GtkStyle * style,
			GdkWindow * window,
			GtkStateType state_type,
			GtkShadowType shadow_type,
			GdkRectangle * area,
			GtkWidget * widget,
			gchar * detail,
			GtkArrowType arrow_type,
			gint fill, gint x, gint y, gint width, gint height);
static void draw_diamond (GtkStyle * style,
			  GdkWindow * window,
			  GtkStateType state_type,
			  GtkShadowType shadow_type,
			  GdkRectangle * area,
			  GtkWidget * widget,
			  gchar * detail,
			  gint x, gint y, gint width, gint height);
static void draw_oval (GtkStyle * style,
		       GdkWindow * window,
		       GtkStateType state_type,
		       GtkShadowType shadow_type,
		       GdkRectangle * area,
		       GtkWidget * widget,
		       gchar * detail,
		       gint x, gint y, gint width, gint height);
static void draw_string (GtkStyle * style,
			 GdkWindow * window,
			 GtkStateType state_type,
			 GdkRectangle * area,
			 GtkWidget * widget,
			 gchar * detail,
			 gint x, gint y, const gchar * string);
static void draw_box (GtkStyle * style,
		      GdkWindow * window,
		      GtkStateType state_type,
		      GtkShadowType shadow_type,
		      GdkRectangle * area,
		      GtkWidget * widget,
		      gchar * detail,
		      gint x, gint y, gint width, gint height);
static void draw_flat_box (GtkStyle * style,
			   GdkWindow * window,
			   GtkStateType state_type,
			   GtkShadowType shadow_type,
			   GdkRectangle * area,
			   GtkWidget * widget,
			   gchar * detail,
			   gint x, gint y, gint width, gint height);
static void draw_check (GtkStyle * style,
			GdkWindow * window,
			GtkStateType state_type,
			GtkShadowType shadow_type,
			GdkRectangle * area,
			GtkWidget * widget,
			gchar * detail,
			gint x, gint y, gint width, gint height);
static void draw_option (GtkStyle * style,
			 GdkWindow * window,
			 GtkStateType state_type,
			 GtkShadowType shadow_type,
			 GdkRectangle * area,
			 GtkWidget * widget,
			 gchar * detail,
			 gint x, gint y, gint width, gint height);
static void draw_cross (GtkStyle * style,
			GdkWindow * window,
			GtkStateType state_type,
			GtkShadowType shadow_type,
			GdkRectangle * area,
			GtkWidget * widget,
			gchar * detail,
			gint x, gint y, gint width, gint height);
static void draw_ramp (GtkStyle * style,
		       GdkWindow * window,
		       GtkStateType state_type,
		       GtkShadowType shadow_type,
		       GdkRectangle * area,
		       GtkWidget * widget,
		       gchar * detail,
		       GtkArrowType arrow_type,
		       gint x, gint y, gint width, gint height);
static void draw_tab (GtkStyle * style,
		      GdkWindow * window,
		      GtkStateType state_type,
		      GtkShadowType shadow_type,
		      GdkRectangle * area,
		      GtkWidget * widget,
		      gchar * detail,
		      gint x, gint y, gint width, gint height);
static void draw_shadow_gap (GtkStyle * style,
			     GdkWindow * window,
			     GtkStateType state_type,
			     GtkShadowType shadow_type,
			     GdkRectangle * area,
			     GtkWidget * widget,
			     gchar * detail,
			     gint x,
			     gint y,
			     gint width,
			     gint height,
			     GtkPositionType gap_side,
			     gint gap_x, gint gap_width);
static void draw_box_gap (GtkStyle * style,
			  GdkWindow * window,
			  GtkStateType state_type,
			  GtkShadowType shadow_type,
			  GdkRectangle * area,
			  GtkWidget * widget,
			  gchar * detail,
			  gint x,
			  gint y,
			  gint width,
			  gint height,
			  GtkPositionType gap_side,
			  gint gap_x, gint gap_width);
static void draw_extension (GtkStyle * style,
			    GdkWindow * window,
			    GtkStateType state_type,
			    GtkShadowType shadow_type,
			    GdkRectangle * area,
			    GtkWidget * widget,
			    gchar * detail,
			    gint x,
			    gint y,
			    gint width,
			    gint height, GtkPositionType gap_side);
static void draw_focus (GtkStyle * style,
			GdkWindow * window,
			GdkRectangle * area,
			GtkWidget * widget,
			gchar * detail,
			gint x, gint y, gint width, gint height);
static void draw_slider (GtkStyle * style,
			 GdkWindow * window,
			 GtkStateType state_type,
			 GtkShadowType shadow_type,
			 GdkRectangle * area,
			 GtkWidget * widget,
			 gchar * detail,
			 gint x,
			 gint y,
			 gint width, gint height, GtkOrientation orientation);
static void draw_handle (GtkStyle * style,
			 GdkWindow * window,
			 GtkStateType state_type,
			 GtkShadowType shadow_type,
			 GdkRectangle * area,
			 GtkWidget * widget,
			 gchar * detail,
			 gint x,
			 gint y,
			 gint width, gint height, GtkOrientation orientation);

static void icegradient_slash_one (GdkWindow * window,
				   GdkGC * gc1,
				   GdkGC * gc2,
				   int width,
				   int height,
				   int x, int y, gboolean down_rigth);
static void icegradient_slash_two (GdkWindow * window,
				   GdkGC * gc1,
				   GdkGC * gc2,
				   int width, int height, int x, int y);
static void icegradient_tab (GtkStyle * style,
			     GdkWindow * window,
			     GtkStateType state_type,
			     GtkShadowType shadow_type,
			     GdkRectangle * area,
			     GtkWidget * widget,
			     gchar * detail,
			     gint x, gint y, gint width, gint height);

/* internal data structs */

GtkStyleClass icegradient_default_class = {
  2,
  2,
  draw_hline,
  draw_vline,
  draw_shadow,
  draw_polygon,
  draw_arrow,
  draw_diamond,
  draw_oval,
  draw_string,
  draw_box,
  draw_flat_box,
  draw_check,
  draw_option,
  draw_cross,
  draw_ramp,
  draw_tab,
  draw_shadow_gap,
  draw_box_gap,
  draw_extension,
  draw_focus,
  draw_slider,
  draw_handle
};


static void
draw_hline (GtkStyle * style,
	    GdkWindow * window,
	    GtkStateType state_type,
	    GdkRectangle * area,
	    GtkWidget * widget, gchar * detail, gint x1, gint x2, gint y)
{
  gint thickness_light;
  gint thickness_dark;
  gint i;

  g_return_if_fail (style != NULL);
  g_return_if_fail (window != NULL);

  dump_widget ();

  thickness_light = style->klass->ythickness / 2;
  thickness_dark = style->klass->ythickness - thickness_light;

  if (area) {
    gdk_gc_set_clip_rectangle (style->light_gc[state_type], area);
    gdk_gc_set_clip_rectangle (style->dark_gc[state_type], area);
  }
  for (i = 0; i < thickness_dark; i++) {
    gdk_draw_line (window, style->light_gc[state_type], x2 - i - 1, y + i,
		   x2, y + i);
    gdk_draw_line (window, style->dark_gc[state_type], x1, y + i,
		   x2 - i - 1, y + i);
  }

  y += thickness_dark;
  for (i = 0; i < thickness_light; i++) {
    gdk_draw_line (window, style->dark_gc[state_type], x1, y + i,
		   x1 + thickness_light - i - 1, y + i);
    gdk_draw_line (window, style->light_gc[state_type],
		   x1 + thickness_light - i - 1, y + i, x2, y + i);
  }
  if (area) {
    gdk_gc_set_clip_rectangle (style->light_gc[state_type], NULL);
    gdk_gc_set_clip_rectangle (style->dark_gc[state_type], NULL);
  }
}


static void
draw_vline (GtkStyle * style,
	    GdkWindow * window,
	    GtkStateType state_type,
	    GdkRectangle * area,
	    GtkWidget * widget, gchar * detail, gint y1, gint y2, gint x)
{
  gint thickness_light;
  gint thickness_dark;
  gint i;

  g_return_if_fail (style != NULL);
  g_return_if_fail (window != NULL);

  dump_widget ();

  thickness_light = style->klass->xthickness / 2;
  thickness_dark = style->klass->xthickness - thickness_light;

  if (area) {
    gdk_gc_set_clip_rectangle (style->light_gc[state_type], area);
    gdk_gc_set_clip_rectangle (style->dark_gc[state_type], area);
  }
  for (i = 0; i < thickness_dark; i++) {
    gdk_draw_line (window, style->light_gc[state_type], x + i, y2 - i - 1,
		   x + i, y2);
    gdk_draw_line (window, style->dark_gc[state_type], x + i, y1, x + i,
		   y2 - i - 1);
  }

  x += thickness_dark;
  for (i = 0; i < thickness_light; i++) {
    gdk_draw_line (window, style->dark_gc[state_type], x + i, y1, x + i,
		   y1 + thickness_light - i);
    gdk_draw_line (window, style->light_gc[state_type], x + i,
		   y1 + thickness_light - i, x + i, y2);
  }
  if (area) {
    gdk_gc_set_clip_rectangle (style->light_gc[state_type], NULL);
    gdk_gc_set_clip_rectangle (style->dark_gc[state_type], NULL);
  }
}


static void
draw_shadow (GtkStyle * style,
	     GdkWindow * window,
	     GtkStateType state_type,
	     GtkShadowType shadow_type,
	     GdkRectangle * area,
	     GtkWidget * widget,
	     gchar * detail, gint x, gint y, gint width, gint height)
{
  GdkGC *gc1 = NULL;
  GdkGC *gc2 = NULL;

  g_return_if_fail (style != NULL);
  g_return_if_fail (window != NULL);

  dump_widget ();

  if ((width == -1) && (height == -1))
    gdk_window_get_size (window, &width, &height);
  else if (width == -1)
    gdk_window_get_size (window, &width, NULL);
  else if (height == -1)
    gdk_window_get_size (window, NULL, &height);

  switch (shadow_type) {
  case GTK_SHADOW_NONE:
    return;
  case GTK_SHADOW_IN:
  case GTK_SHADOW_ETCHED_IN:
    gc1 = style->light_gc[state_type];
    gc2 = style->dark_gc[state_type];
    break;
  case GTK_SHADOW_OUT:
  case GTK_SHADOW_ETCHED_OUT:
    gc1 = style->dark_gc[state_type];
    gc2 = style->light_gc[state_type];
    break;
  }

  if (area) {
    gdk_gc_set_clip_rectangle (gc1, area);
    gdk_gc_set_clip_rectangle (gc2, area);
    if ((shadow_type == GTK_SHADOW_IN) || (shadow_type == GTK_SHADOW_OUT)) {
      gdk_gc_set_clip_rectangle (style->black_gc, area);
      gdk_gc_set_clip_rectangle (style->bg_gc[state_type], area);
    }
  }
  switch (shadow_type) {
  case GTK_SHADOW_NONE:
    break;

  case GTK_SHADOW_ETCHED_IN:
    gdk_draw_line (window, gc2, x, y, x + width - 1, y);
    gdk_draw_line (window, gc2, x, y, x, y + height - 1);
    gdk_draw_line (window, gc1, x, y + height - 1, x + width - 1,
		   y + height - 1);
    gdk_draw_line (window, gc1, x + width - 1, y, x + width - 1,
		   y + height - 1);
    gdk_draw_line (window, gc1, x + 1, y + 1, x + width - 2, y + 1);
    gdk_draw_line (window, gc1, x + 1, y + 1, x + 1, y + height - 2);
    gdk_draw_line (window, gc2, x + 1, y + height - 2, x + width - 2,
		   y + height - 2);
    gdk_draw_line (window, gc2, x + width - 2, y + 1, x + width - 2,
		   y + height - 2);
    break;
  case GTK_SHADOW_IN:
    gdk_draw_line (window, gc2, x, y, x + width - 1, y);
    gdk_draw_line (window, gc2, x, y, x, y + height - 1);
    gdk_draw_line (window, gc1, x, y + height - 1, x + width - 1,
		   y + height - 1);
    gdk_draw_line (window, gc1, x + width - 1, y, x + width - 1,
		   y + height - 1);
    break;
  case GTK_SHADOW_ETCHED_OUT:
    gdk_draw_line (window, gc2, x, y, x + width - 1, y);
    gdk_draw_line (window, gc2, x, y, x, y + height - 1);
    gdk_draw_line (window, gc1, x, y + height - 1, x + width - 1,
		   y + height - 1);
    gdk_draw_line (window, gc1, x + width - 1, y, x + width - 1,
		   y + height - 1);
    gdk_draw_line (window, gc1, x + 1, y + 1, x + width - 2, y + 1);
    gdk_draw_line (window, gc1, x + 1, y + 1, x + 1, y + height - 2);
    gdk_draw_line (window, gc2, x + 1, y + height - 2, x + width - 2,
		   y + height - 2);
    gdk_draw_line (window, gc2, x + width - 2, y + 1, x + width - 2,
		   y + height - 2);
    break;
  case GTK_SHADOW_OUT:
    gdk_draw_line (window, gc2, x, y, x + width - 1, y);
    gdk_draw_line (window, gc2, x, y, x, y + height - 1);
    gdk_draw_line (window, gc1, x, y + height - 1, x + width - 1,
		   y + height - 1);
    gdk_draw_line (window, gc1, x + width - 1, y, x + width - 1,
		   y + height - 1);
    break;
  }
  if (area) {
    gdk_gc_set_clip_rectangle (gc1, NULL);
    gdk_gc_set_clip_rectangle (gc2, NULL);
    if ((shadow_type == GTK_SHADOW_IN) || (shadow_type == GTK_SHADOW_OUT)) {
      gdk_gc_set_clip_rectangle (style->black_gc, NULL);
      gdk_gc_set_clip_rectangle (style->bg_gc[state_type], NULL);
    }
  }
}

static void
draw_polygon (GtkStyle * style,
	      GdkWindow * window,
	      GtkStateType state_type,
	      GtkShadowType shadow_type,
	      GdkRectangle * area,
	      GtkWidget * widget,
	      gchar * detail, GdkPoint * points, gint npoints, gint fill)
{
#ifndef M_PI
#define M_PI    3.14159265358979323846
#endif /* M_PI */
#ifndef M_PI_4
#define M_PI_4  0.78539816339744830962
#endif /* M_PI_4 */

  static const gdouble pi_over_4 = M_PI_4;
  static const gdouble pi_3_over_4 = M_PI_4 * 3;

  GdkGC *gc1;
  GdkGC *gc2;
  GdkGC *gc3;
  GdkGC *gc4;
  gdouble angle;
  gint xadjust;
  gint yadjust;
  gint i;

  g_return_if_fail (style != NULL);
  g_return_if_fail (window != NULL);
  g_return_if_fail (points != NULL);

  dump_widget ();

  switch (shadow_type) {
  case GTK_SHADOW_IN:
    gc1 = style->light_gc[state_type];
    gc2 = style->dark_gc[state_type];
    gc3 = style->light_gc[state_type];
    gc4 = style->dark_gc[state_type];
    break;
  case GTK_SHADOW_ETCHED_IN:
    gc1 = style->light_gc[state_type];
    gc2 = style->dark_gc[state_type];
    gc3 = style->dark_gc[state_type];
    gc4 = style->light_gc[state_type];
    break;
  case GTK_SHADOW_OUT:
    gc1 = style->dark_gc[state_type];
    gc2 = style->light_gc[state_type];
    gc3 = style->dark_gc[state_type];
    gc4 = style->light_gc[state_type];
    break;
  case GTK_SHADOW_ETCHED_OUT:
    gc1 = style->dark_gc[state_type];
    gc2 = style->light_gc[state_type];
    gc3 = style->light_gc[state_type];
    gc4 = style->dark_gc[state_type];
    break;
  default:
    return;
  }

  if (area) {
    gdk_gc_set_clip_rectangle (gc1, area);
    gdk_gc_set_clip_rectangle (gc2, area);
    gdk_gc_set_clip_rectangle (gc3, area);
    gdk_gc_set_clip_rectangle (gc4, area);
  }

  if (fill)
    gdk_draw_polygon (window, style->bg_gc[state_type], TRUE, points,
		      npoints);

  npoints--;

  for (i = 0; i < npoints; i++) {
    if ((points[i].x == points[i + 1].x)
	&& (points[i].y == points[i + 1].y)) {
      angle = 0;
    } else {
      angle = atan2 (points[i + 1].y - points[i].y,
		     points[i + 1].x - points[i].x);
    }

    if ((angle > -pi_3_over_4) && (angle < pi_over_4)) {
      if (angle > -pi_over_4) {
	xadjust = 0;
	yadjust = 1;
      } else {
	xadjust = 1;
	yadjust = 0;
      }

      gdk_draw_line (window, gc1,
		     points[i].x - xadjust, points[i].y - yadjust,
		     points[i + 1].x - xadjust, points[i + 1].y - yadjust);
      gdk_draw_line (window, gc3, points[i].x, points[i].y,
		     points[i + 1].x, points[i + 1].y);
    } else {
      if ((angle < -pi_3_over_4) || (angle > pi_3_over_4)) {
	xadjust = 0;
	yadjust = 1;
      } else {
	xadjust = 1;
	yadjust = 0;
      }

      gdk_draw_line (window, gc4,
		     points[i].x + xadjust, points[i].y + yadjust,
		     points[i + 1].x + xadjust, points[i + 1].y + yadjust);
      gdk_draw_line (window, gc2, points[i].x, points[i].y,
		     points[i + 1].x, points[i + 1].y);
    }
  }
  if (area) {
    gdk_gc_set_clip_rectangle (gc1, NULL);
    gdk_gc_set_clip_rectangle (gc2, NULL);
    gdk_gc_set_clip_rectangle (gc3, NULL);
    gdk_gc_set_clip_rectangle (gc4, NULL);
  }
}


static inline void
icegradient_slash_two (GdkWindow * window, GdkGC * gc1, GdkGC * gc2,
		       int width, int height, int x, int y)
{
  int centerx, centery, thick;
  int ax1 = 0, ax2 = 0, ay1 = 0, ay2 = 0;

  centerx = (width - 1) / 2 + x;
  centery = (height - 1) / 2 + y;
  if (width > height) {
    ax1 = -2;
    ax2 = 1;
  } else {
    ay1 = -2;
    ay2 = 1;
  }

  thick = ((width < height ? width - 1 : height - 1) >> 1) - 2;
  gdk_draw_line (window, gc2,
		 centerx - thick + ax1, centery + thick + ay1,
		 centerx + thick + ax1, centery - thick + ay1);
  gdk_draw_line (window, gc1,
		 centerx - thick + ax1 + ax2, centery + thick + ay1 + ay2,
		 centerx + thick + ax1 + ax2, centery - thick + ay1 + ay2);
  if (width > height) {
    ax1 = 2;			/* ax2 = 1; */
  } else {
    ay1 = 2;			/* ay2 = 1; */
  }
  gdk_draw_line (window, gc2,
		 centerx - thick + ax1, centery + thick + ay1,
		 centerx + thick + ax1, centery - thick + ay1);
  gdk_draw_line (window, gc1,
		 centerx - thick + ax1 + ax2, centery + thick + ay1 + ay2,
		 centerx + thick + ax1 + ax2, centery - thick + ay1 + ay2);
}


static inline void
icegradient_slash_one (GdkWindow * window, GdkGC * gc1, GdkGC * gc2,
		       int width, int height, int x, int y,
		       gboolean down_rigth)
{
  int centerx, centery, thick;

  centerx = (width - 1) / 2 + x;
  centery = (height - 1) / 2 + y;

  thick = ((width < height ? width - 1 : height - 1) >> 1) - 2;
  if (!down_rigth) {
    gdk_draw_line (window, gc2,
		   centerx - thick, centery + thick,
		   centerx + thick, centery - thick);
    gdk_draw_line (window, gc1,
		   centerx - thick, centery + thick - 1,
		   centerx + thick - 1, centery - thick);
  } else {
    gdk_draw_line (window, gc1,
		   centerx + thick, centery + thick,
		   centerx - thick, centery - thick);
    gdk_draw_line (window, gc2,
		   centerx + thick, centery + thick - 1,
		   centerx - thick + 1, centery - thick);
  }
}


static inline void
icegradient_dot (GdkWindow * window, GdkGC * gc1, GdkGC * gc2, GdkGC * gc3,
		 int width, int height, int x, int y)
{
  int centerx, centery, thick;

  centerx = (width - 1) / 2 + x;
  centery = (height - 1) / 2 + y;
  thick = ((width < height ? width - 1 : height - 1) >> 1) - 2;

/*  gdk_draw_arc(window, gc3, TRUE, centerx-thick, centery-thick, thick*2, thick*2, 0, 360 * 64);*/
  gdk_draw_arc (window, gc1, FALSE, centerx - thick, centery - thick,
		thick * 2, thick * 2, 45 * 64, 225 * 64);
  gdk_draw_arc (window, gc2, FALSE, centerx - thick, centery - thick,
		thick * 2, thick * 2, 225 * 64, 180 * 64);
}


static void
draw_arrow (GtkStyle * style,
	    GdkWindow * window,
	    GtkStateType state_type,
	    GtkShadowType shadow_type,
	    GdkRectangle * area,
	    GtkWidget * widget,
	    gchar * detail,
	    GtkArrowType arrow_type,
	    gint fill, gint x, gint y, gint width, gint height)
{
  GdkGC *gc1;
  GdkGC *gc2;
  GdkGC *gc3;
  GdkGC *gc4;
  gint half_width;
  gint half_height;
  GdkPoint points[4];
  gboolean down_right;
  diagonal_aspect_style diagonal_orientation;
  Detail *d;

  g_return_if_fail (style != NULL);
  g_return_if_fail (window != NULL);

  dump_widget ();

  switch (shadow_type) {
  case GTK_SHADOW_IN:
    gc1 = style->bg_gc[state_type];
    gc2 = style->dark_gc[state_type];
    gc3 = style->light_gc[state_type];
    gc4 = style->black_gc;
    break;
  case GTK_SHADOW_OUT:
    gc1 = style->dark_gc[state_type];
    gc2 = style->light_gc[state_type];
    gc3 = style->black_gc;
    gc4 = style->bg_gc[state_type];
    break;
  case GTK_SHADOW_ETCHED_IN:
    gc1 = style->light_gc[state_type];
    gc2 = style->dark_gc[state_type];
    gc3 = NULL;
    gc4 = NULL;
    break;
  case GTK_SHADOW_ETCHED_OUT:
    gc1 = style->dark_gc[state_type];
    gc2 = style->light_gc[state_type];
    gc3 = NULL;
    gc4 = NULL;
    break;
  default:
    return;
  }

  if ((width == -1) && (height == -1))
    gdk_window_get_size (window, &width, &height);
  else if (width == -1)
    gdk_window_get_size (window, &width, NULL);
  else if (height == -1)
    gdk_window_get_size (window, NULL, &height);

  half_width = width / 2;
  half_height = height / 2;

  if (area) {
    gdk_gc_set_clip_rectangle (gc1, area);
    gdk_gc_set_clip_rectangle (gc2, area);
    if ((gc3) && (gc4)) {
      gdk_gc_set_clip_rectangle (gc3, area);
      gdk_gc_set_clip_rectangle (gc4, area);
    }
  }

  /* KLUDGE! ... checks for vscrollbar and hscrollbar .. but two strcmp's are
     expensive... */
  if (detail && detail[0] && detail[1]
      && g_str_equal (detail + 1, "scrollbar")) {
    switch (arrow_type) {
    case GTK_ARROW_UP:
      diagonal_orientation = NORTH_EAST;
      down_right = FALSE;
      break;
    case GTK_ARROW_LEFT:
      diagonal_orientation = NORTH_EAST;
      down_right = FALSE;
      break;
    case GTK_ARROW_DOWN:
      diagonal_orientation = SOUTH_EAST;
      down_right = TRUE;
      break;
    case GTK_ARROW_RIGHT:
      diagonal_orientation = NORTH_WEST;
      down_right = TRUE;
      break;
    }
    draw_generic_gradient (style, window, state_type, shadow_type, area,
			   detail, x, y, width, height, DIAGONAL_GRADIENT,
			   diagonal_orientation);
    d = get_detail (style, detail);
    if ((d) && (d->decoration != NONE)) {
      /* Draw a single / in the button */
      icegradient_slash_one (window, gc1, gc2, width, height, x, y,
			     down_right);
    }
    gtk_paint_shadow (style, window, state_type, shadow_type, area, widget,
		      detail, x, y, width, height);
  } else {

    /* Should be more indenting, but it looks.. bad.. */
    switch (arrow_type) {
    case GTK_ARROW_UP:
      if (fill) {
	points[0].x = x + half_width;
	points[0].y = y;
	points[1].x = x;
	points[1].y = y + height - 1;
	points[2].x = x + width - 1;
	points[2].y = y + height - 1;

	gdk_draw_polygon (window, style->bg_gc[state_type], TRUE, points, 3);
      }

      switch (shadow_type) {
      case GTK_SHADOW_IN:
	points[0].x = x + half_width;
	points[0].y = y;
	points[1].x = x + 1;
	points[1].y = y + height - 1;
	points[2].x = x + width - 2;
	points[2].y = y + height - 1;

	gdk_draw_polygon (window, gc1, FALSE, points, 3);
	points[0].x--;
	points[1].x--;
	points[2].x--;
	points[1].y--;
	points[2].y--;
	gdk_draw_polygon (window, gc2, FALSE, points, 3);
	gdk_draw_polygon (window, gc2, TRUE, points, 3);
	break;

      case GTK_SHADOW_OUT:
	points[0].x = x + half_width;
	points[0].y = y;
	points[1].x = x + 1;
	points[1].y = y + height - 2;
	points[2].x = x + width - 2;
	points[2].y = y + height - 2;

	gdk_draw_polygon (window, gc2, FALSE, points, 3);
	points[0].x--;
	points[1].x--;
	points[2].x--;
	gdk_draw_polygon (window, gc1, FALSE, points, 3);
	gdk_draw_line (window, gc2,
		       points[1].x, points[1].y + 1,
		       points[2].x + 1, points[2].y + 1);
	break;


      case GTK_SHADOW_ETCHED_IN:
      case GTK_SHADOW_ETCHED_OUT:
	gdk_draw_line (window, gc1,
		       x + half_width, y + 1, x + 1, y + height - 1);
	gdk_draw_line (window, gc1,
		       x + 1, y + height - 1, x + width - 1, y + height - 1);
	gdk_draw_line (window, gc1, x + width - 1, y + height - 1,
		       x + half_width + 1, y + 1);

	points[0].x = x + half_width;
	points[0].y = y;
	points[1].x = x;
	points[1].y = y + height - 2;
	points[2].x = x + width - 2;
	points[2].y = y + height - 2;

	gdk_draw_polygon (window, gc2, FALSE, points, 3);
	break;

      default:
	break;
      }
      break;

    case GTK_ARROW_DOWN:
      if (fill) {
	points[0].x = x + width - 1;
	points[0].y = y;
	points[1].x = x;
	points[1].y = y;
	points[2].x = x + half_width;
	points[2].y = y + height - 1;

	gdk_draw_polygon (window, style->bg_gc[state_type], TRUE, points, 3);
      }
      switch (shadow_type) {
      case GTK_SHADOW_IN:
	points[0].x = x + width - 2;
	points[0].y = y + 1;
	points[1].x = x + 1;
	points[1].y = y + 1;
	points[2].x = x + half_width;
	points[2].y = y + height - 1;

	gdk_draw_polygon (window, gc1, FALSE, points, 3);
	points[0].x--;
	points[1].x--;
	points[2].x--;
	gdk_draw_polygon (window, gc2, FALSE, points, 3);
	gdk_draw_polygon (window, gc2, TRUE, points, 3);
	break;

      case GTK_SHADOW_OUT:
	points[0].x = x + width - 2;
	points[0].y = y + 1;
	points[1].x = x + 1;
	points[1].y = y + 1;
	points[2].x = x + half_width;
	points[2].y = y + height - 1;

	gdk_draw_polygon (window, gc2, FALSE, points, 3);
	points[0].x--;
	points[1].x--;
	points[2].x--;
	gdk_draw_polygon (window, gc1, FALSE, points, 3);
	break;

      case GTK_SHADOW_ETCHED_IN:
      case GTK_SHADOW_ETCHED_OUT:
	gdk_draw_line (window, gc1, x + width - 1, y + 1, x + 1, y + 1);
	gdk_draw_line (window, gc1,
		       x + 1, y + 1, x + half_width + 1, y + height - 1);
	gdk_draw_line (window, gc1, x + half_width + 1, y + height - 2,
		       x + width - 1, y);

	points[0].x = x + width - 2;
	points[0].y = y;
	points[1].x = x;
	points[1].y = y;
	points[2].x = x + half_width;
	points[2].y = y + height - 2;

	gdk_draw_polygon (window, gc2, FALSE, points, 3);
	break;

      default:
	break;
      }
      break;
    case GTK_ARROW_LEFT:
      if (fill) {
	points[0].x = x;
	points[0].y = y + half_height;
	points[1].x = x + width - 1;
	points[1].y = y + height - 1;
	points[2].x = x + width - 1;
	points[2].y = y;

	gdk_draw_polygon (window, style->bg_gc[state_type], TRUE, points, 3);
      }

      switch (shadow_type) {
      case GTK_SHADOW_IN:
	points[0].x = x + width - 2;
	points[0].y = y + 1;
	points[1].x = x;
	points[1].y = y + half_height;
	points[2].x = x + width - 2;
	points[2].y = y + height - 2;

	gdk_draw_polygon (window, gc1, FALSE, points, 3);
	points[0].y--;
	points[1].y--;
	points[2].y--;
	gdk_draw_polygon (window, gc2, FALSE, points, 3);
	gdk_draw_polygon (window, gc2, TRUE, points, 3);
	break;

      case GTK_SHADOW_OUT:
	points[0].x = x + width - 2;
	points[0].y = y + 1;
	points[1].x = x;
	points[1].y = y + half_height;
	points[2].x = x + width - 2;
	points[2].y = y + height - 2;

	gdk_draw_polygon (window, gc2, FALSE, points, 3);
	gdk_draw_line (window, gc2,
		       x + width - 1, y + 1, x + width - 1, y + height - 1);

	points[0].y--;
	points[1].y--;
	points[2].y--;
	gdk_draw_polygon (window, gc1, FALSE, points, 3);
	break;

      case GTK_SHADOW_ETCHED_IN:
      case GTK_SHADOW_ETCHED_OUT:
	gdk_draw_line (window, gc1,
		       x + width - 1, y + 1, x + 1, y + half_height);
	gdk_draw_line (window, gc1,
		       x + 1, y + half_height + 1,
		       x + width - 1, y + height - 1);
	gdk_draw_line (window, gc1,
		       x + width - 1, y + height - 1, x + width - 1, y + 1);

	points[0].x = x + width - 2;
	points[0].y = y;
	points[1].x = x;
	points[1].y = y + half_height;
	points[2].x = x + width - 2;
	points[2].y = y + height - 2;

	gdk_draw_polygon (window, gc2, FALSE, points, 3);
	break;

      default:
	break;
      }
      break;
    case GTK_ARROW_RIGHT:
      if (fill) {
	points[0].x = x + width - 1;
	points[0].y = y + half_height;
	points[1].x = x;
	points[1].y = y;
	points[2].x = x;
	points[2].y = y + height - 1;

	gdk_draw_polygon (window, style->bg_gc[state_type], TRUE, points, 3);
      }

      switch (shadow_type) {
      case GTK_SHADOW_IN:
	points[0].x = x + width - 1;
	points[0].y = y + half_height - 1 + 1;
	points[1].x = x;
	points[1].y = y + 1;
	points[2].x = x;
	points[2].y = y + height - 2 + 1;

	gdk_draw_polygon (window, gc1, FALSE, points, 3);
	points[0].y--;
	points[1].y--;
	points[2].y--;
	gdk_draw_polygon (window, gc2, FALSE, points, 3);
	gdk_draw_polygon (window, gc2, TRUE, points, 3);
	break;

      case GTK_SHADOW_OUT:
	points[0].x = x + width - 1;
	points[0].y = y + half_height - 1 + 1;
	points[1].x = x;
	points[1].y = y + 1;
	points[2].x = x;
	points[2].y = y + height - 2 + 1;

	gdk_draw_polygon (window, gc2, FALSE, points, 3);
	gdk_draw_line (window, gc2, x + 1, y + 1, x + 1, y + height - 1);

	points[0].y--;
	points[1].y--;
	points[2].y--;
	gdk_draw_polygon (window, gc1, FALSE, points, 3);
	break;

	break;

      case GTK_SHADOW_ETCHED_IN:
      case GTK_SHADOW_ETCHED_OUT:
	gdk_draw_line (window, gc1,
		       x + width - 1, y + half_height + 1, x + 1, y + 1);
	gdk_draw_line (window, gc1, x + 1, y + 1, x + 1, y + height - 1);
	gdk_draw_line (window, gc1, x + 1, y + height - 1,
		       x + width - 1, y + half_height + 1);

	points[0].x = x + width - 2;
	points[0].y = y + half_height;
	points[1].x = x;
	points[1].y = y;
	points[2].x = x;
	points[2].y = y + height - 1;

	gdk_draw_polygon (window, gc2, FALSE, points, 3);
	break;

      default:
	break;
      }
      break;
    }
  }
  if (area) {
    gdk_gc_set_clip_rectangle (gc1, NULL);
    gdk_gc_set_clip_rectangle (gc2, NULL);
    if (gc3) {
      gdk_gc_set_clip_rectangle (gc3, NULL);
      gdk_gc_set_clip_rectangle (gc4, NULL);
    }
  }
}

static void
draw_diamond (GtkStyle * style,
	      GdkWindow * window,
	      GtkStateType state_type,
	      GtkShadowType shadow_type,
	      GdkRectangle * area,
	      GtkWidget * widget,
	      gchar * detail, gint x, gint y, gint width, gint height)
{
  gint half_width;
  gint half_height;
  g_return_if_fail (style != NULL);
  g_return_if_fail (window != NULL);

  dump_widget ();

  if ((width == -1) && (height == -1))
    gdk_window_get_size (window, &width, &height);
  else if (width == -1)
    gdk_window_get_size (window, &width, NULL);
  else if (height == -1)
    gdk_window_get_size (window, NULL, &height);

  half_width = width / 2;
  half_height = height / 2;

  if (area) {
    gdk_gc_set_clip_rectangle (style->light_gc[state_type], area);
    gdk_gc_set_clip_rectangle (style->bg_gc[state_type], area);
    gdk_gc_set_clip_rectangle (style->dark_gc[state_type], area);
    gdk_gc_set_clip_rectangle (style->black_gc, area);
  }
  switch (shadow_type) {
  case GTK_SHADOW_IN:
    gdk_draw_line (window, style->light_gc[state_type],
		   x + 2, y + half_height, x + half_width, y + height - 2);
    gdk_draw_line (window, style->light_gc[state_type],
		   x + half_width, y + height - 2,
		   x + width - 2, y + half_height);
    gdk_draw_line (window, style->light_gc[state_type],
		   x + 1, y + half_height, x + half_width, y + height - 1);
    gdk_draw_line (window, style->light_gc[state_type],
		   x + half_width, y + height - 1,
		   x + width - 1, y + half_height);
    gdk_draw_line (window, style->light_gc[state_type],
		   x, y + half_height, x + half_width, y + height);
    gdk_draw_line (window, style->light_gc[state_type],
		   x + half_width, y + height, x + width, y + half_height);

    gdk_draw_line (window, style->dark_gc[state_type],
		   x + 2, y + half_height, x + half_width, y + 2);
    gdk_draw_line (window, style->dark_gc[state_type],
		   x + half_width, y + 2, x + width - 2, y + half_height);
    gdk_draw_line (window, style->dark_gc[state_type],
		   x + 1, y + half_height, x + half_width, y + 1);
    gdk_draw_line (window, style->dark_gc[state_type],
		   x + half_width, y + 1, x + width - 1, y + half_height);
    gdk_draw_line (window, style->dark_gc[state_type],
		   x, y + half_height, x + half_width, y);
    gdk_draw_line (window, style->dark_gc[state_type],
		   x + half_width, y, x + width, y + half_height);
    break;
  case GTK_SHADOW_OUT:
    gdk_draw_line (window, style->dark_gc[state_type],
		   x + 2, y + half_height, x + half_width, y + height - 2);
    gdk_draw_line (window, style->dark_gc[state_type],
		   x + half_width, y + height - 2,
		   x + width - 2, y + half_height);
    gdk_draw_line (window, style->dark_gc[state_type],
		   x + 1, y + half_height, x + half_width, y + height - 1);
    gdk_draw_line (window, style->dark_gc[state_type],
		   x + half_width, y + height - 1,
		   x + width - 1, y + half_height);
    gdk_draw_line (window, style->dark_gc[state_type],
		   x, y + half_height, x + half_width, y + height);
    gdk_draw_line (window, style->dark_gc[state_type],
		   x + half_width, y + height, x + width, y + half_height);

    gdk_draw_line (window, style->light_gc[state_type],
		   x + 2, y + half_height, x + half_width, y + 2);
    gdk_draw_line (window, style->light_gc[state_type],
		   x + half_width, y + 2, x + width - 2, y + half_height);
    gdk_draw_line (window, style->light_gc[state_type],
		   x + 1, y + half_height, x + half_width, y + 1);
    gdk_draw_line (window, style->light_gc[state_type],
		   x + half_width, y + 1, x + width - 1, y + half_height);
    gdk_draw_line (window, style->light_gc[state_type],
		   x, y + half_height, x + half_width, y);
    gdk_draw_line (window, style->light_gc[state_type],
		   x + half_width, y, x + width, y + half_height);
    break;
  default:
    break;
  }
  if (area) {
    gdk_gc_set_clip_rectangle (style->light_gc[state_type], NULL);
    gdk_gc_set_clip_rectangle (style->bg_gc[state_type], NULL);
    gdk_gc_set_clip_rectangle (style->dark_gc[state_type], NULL);
    gdk_gc_set_clip_rectangle (style->black_gc, NULL);
  }
}

static void
draw_oval (GtkStyle * style,
	   GdkWindow * window,
	   GtkStateType state_type,
	   GtkShadowType shadow_type,
	   GdkRectangle * area,
	   GtkWidget * widget,
	   gchar * detail, gint x, gint y, gint width, gint height)
{
  g_return_if_fail (style != NULL);
  g_return_if_fail (window != NULL);

  dump_widget ();

}

static void
draw_string (GtkStyle * style,
	     GdkWindow * window,
	     GtkStateType state_type,
	     GdkRectangle * area,
	     GtkWidget * widget,
	     gchar * detail, gint x, gint y, const gchar * string)
{
  g_return_if_fail (style != NULL);
  g_return_if_fail (window != NULL);

  dump_widget ();

  if (area) {
    gdk_gc_set_clip_rectangle (style->white_gc, area);
    gdk_gc_set_clip_rectangle (style->fg_gc[state_type], area);
  }
  if (state_type == GTK_STATE_INSENSITIVE)
    gdk_draw_string (window, style->font, style->white_gc, x + 1, y + 1,
		     string);
  gdk_draw_string (window, style->font, style->fg_gc[state_type], x, y,
		   string);
  if (area) {
    gdk_gc_set_clip_rectangle (style->white_gc, NULL);
    gdk_gc_set_clip_rectangle (style->fg_gc[state_type], NULL);
  }
}


static inline void
draw_default (GtkStyle * style,
	      GdkWindow * window,
	      GtkStateType state_type,
	      GtkShadowType shadow_type,
	      GdkRectangle * area,
	      GtkWidget * widget,
	      gchar * detail, gint x, gint y, gint width, gint height)
{
  GdkPoint points1[3];		/* dark */
  GdkPoint points2[3];		/* light */

  points1[0].x = x + 2;
  points1[0].y = y + 2;
  points1[1].x = x + 10;
  points1[1].y = y + 2;
  points1[2].x = x + 2;
  points1[2].y = y + 10;
  points2[0].x = x + 3;
  points2[0].y = y + 3;
  points2[1].x = x + 10;
  points2[1].y = y + 3;
  points2[2].x = x + 3;
  points2[2].y = y + 10;

  /* Paint a triangle here instead of in "buttondefault"
     which is drawn _behind_ the current button */
  gdk_draw_polygon (window, style->dark_gc[state_type], FALSE, points1, 3);
  gdk_draw_polygon (window, style->light_gc[state_type], FALSE, points2, 3);
  gdk_draw_polygon (window, style->bg_gc[GTK_STATE_SELECTED], TRUE, points2,
		    3);
}


static inline void
draw_blank_background (GtkStyle * style,
		       GdkWindow * window,
		       GtkStateType state_type,
		       GdkRectangle * area,
		       gint x, gint y, gint width, gint height)
{
  if (area) {
    gdk_gc_set_clip_rectangle (style->bg_gc[state_type], area);
  }
  gdk_draw_rectangle (window, style->bg_gc[state_type], TRUE, x + 1, y + 1,
		      width - 2, height - 2);
  if (area) {
    gdk_gc_set_clip_rectangle (style->bg_gc[state_type], NULL);
  }
}


static inline void
draw_background (GtkStyle * style,
		 GdkWindow * window,
		 GtkStateType state_type,
		 GtkShadowType shadow_type,
		 GdkRectangle * area,
		 GtkWidget * widget,
		 gchar * detail, gint x, gint y, gint width, gint height)
{
  Detail *d;

  dump_widget ();
  d = get_detail (style, detail);
  if (indexed_visual (style) ||
      (shadow_type == GTK_SHADOW_NONE) || (!d) || (d->background == CLEAN) ||
/*      (gdk_window_get_type(window) == GDK_WINDOW_PIXMAP) ||*/
      DETAIL ("trough")) {
    draw_blank_background (style, window, state_type, area, x, y, width,
			   height);
  } else if (style->bg_pixmap[state_type]) {
    gtk_style_apply_default_background (style, window, widget
					&& !GTK_WIDGET_NO_WINDOW (widget),
					state_type, area, x, y, width,
					height);
  } else {
    draw_generic_gradient (style, window, state_type, shadow_type, area,
			   detail, x, y, width, height, d->background,
			   d->diagonal_aspect);
  }
}


static inline void
draw_flat_background (GtkStyle * style,
		      GdkWindow * window,
		      GtkStateType state_type,
		      GtkShadowType shadow_type,
		      GdkRectangle * area,
		      GtkWidget * widget,
		      gchar * detail, gint x, gint y, gint width, gint height)
{
  Detail *d;

  dump_widget ();
  d = get_detail (style, detail);

  if (style->bg_pixmap[state_type]) {
    gtk_style_apply_default_background (style, window, widget
					&& !GTK_WIDGET_NO_WINDOW (widget),
					state_type, area, x, y, width,
					height);
  } else if (DETAIL ("text") && (state_type == GTK_STATE_SELECTED)
	     && (d->background != CLEAN)) {
    draw_background (style, window, GTK_STATE_SELECTED, GTK_SHADOW_OUT,
		     area, widget, detail, x, y, width, height);
    gtk_paint_shadow (style, window, GTK_STATE_SELECTED, GTK_SHADOW_OUT,
		      area, widget, detail, x, y, width, height);
  } else if (DETAIL ("text") && (state_type == GTK_STATE_SELECTED)
	     && (d->background == CLEAN)) {
    draw_background (style, window, GTK_STATE_ACTIVE, GTK_SHADOW_NONE, area,
		     widget, detail, x, y, width, height);
  } else if (DETAIL ("entry_bg")) {
    gdk_draw_rectangle (window, style->bg_gc[GTK_STATE_PRELIGHT], TRUE, x,
			y, width, height);
  } else {
    draw_blank_background (style, window, state_type, area, x, y, width,
			   height);
  }
}


static void
draw_box (GtkStyle * style,
	  GdkWindow * window,
	  GtkStateType state_type,
	  GtkShadowType shadow_type,
	  GdkRectangle * area,
	  GtkWidget * widget,
	  gchar * detail, gint x, gint y, gint width, gint height)
{
  GtkOrientation orientation;
  Detail *d;

  g_return_if_fail (style != NULL);
  g_return_if_fail (window != NULL);

  dump_widget ();

  if ((width == -1) && (height == -1))
    gdk_window_get_size (window, &width, &height);
  else if (width == -1)
    gdk_window_get_size (window, &width, NULL);
  else if (height == -1)
    gdk_window_get_size (window, NULL, &height);

  orientation = GTK_ORIENTATION_HORIZONTAL;
  if (height > width)
    orientation = GTK_ORIENTATION_VERTICAL;

  draw_background (style, window, state_type, shadow_type, area, widget,
		   detail, x, y, width, height);

  d = get_detail (style, detail);
  if (!DETAIL ("trough")) {
    switch (d->decoration) {
    case NONE:
      break;
    case SLASH:
      icegradient_slash_one (window, style->light_gc[state_type],
			     style->dark_gc[state_type], width, height, x,
			     y, FALSE);
      break;
    case BACK_SLASH:
      icegradient_slash_one (window, style->light_gc[state_type],
			     style->dark_gc[state_type], width, height, x,
			     y, TRUE);
      break;
    case DOUBLE_SLASH:
      icegradient_slash_two (window, style->light_gc[state_type],
			     style->dark_gc[state_type], width, height, x, y);
      break;
    case DOT:
      icegradient_dot (window, style->dark_gc[state_type],
		       style->light_gc[state_type],
		       style->bg_gc[state_type], width, height, x, y);
      break;
    }
  }

  if (widget && GTK_WIDGET_HAS_DEFAULT (widget) && detail
      && g_str_equal (detail, "button")) {
    draw_default (style, window, state_type, shadow_type, area, widget,
		  detail, x, y, width, height);
  }

  if (shadow_type != GTK_SHADOW_NONE) {
    gtk_paint_shadow (style, window, state_type, shadow_type, area, widget,
		      detail, x, y, width, height);
  }
}


static void
draw_flat_box (GtkStyle * style,
	       GdkWindow * window,
	       GtkStateType state_type,
	       GtkShadowType shadow_type,
	       GdkRectangle * area,
	       GtkWidget * widget,
	       gchar * detail, gint x, gint y, gint width, gint height)
{
  GtkOrientation orientation;

  g_return_if_fail (style != NULL);
  g_return_if_fail (window != NULL);

  dump_widget ();

  if ((width == -1) && (height == -1))
    gdk_window_get_size (window, &width, &height);
  else if (width == -1)
    gdk_window_get_size (window, &width, NULL);
  else if (height == -1)
    gdk_window_get_size (window, NULL, &height);

  orientation = GTK_ORIENTATION_HORIZONTAL;
  if (height > width)
    orientation = GTK_ORIENTATION_VERTICAL;

  if (DETAIL ("tooltip")) {
    draw_box (style, window, GTK_STATE_PRELIGHT, GTK_SHADOW_OUT, area,
	      widget, detail, x, y, width, height);
    gdk_draw_rectangle (window, style->black_gc, FALSE, x, y, width - 1,
			height - 1);
  } else {
    draw_flat_background (style, window, state_type, shadow_type, area,
			  widget, detail, x, y, width, height);
  }
}


static void
draw_check (GtkStyle * style,
	    GdkWindow * window,
	    GtkStateType state_type,
	    GtkShadowType shadow_type,
	    GdkRectangle * area,
	    GtkWidget * widget,
	    gchar * detail, gint x, gint y, gint width, gint height)
{
  dump_widget ();

  draw_background (style, window, state_type, shadow_type, area, widget,
		   detail, x, y, width, height);
  gtk_paint_shadow (style, window, state_type, shadow_type, area, widget,
		    detail, x, y, width, height);
}


/* Thanks to Evan Lawrence */
static void
draw_option (GtkStyle * style,
	     GdkWindow * window,
	     GtkStateType state_type,
	     GtkShadowType shadow_type,
	     GdkRectangle * area,
	     GtkWidget * widget,
	     gchar * detail, gint x, gint y, gint width, gint height)
{
  GdkGC *gc1;
  GdkGC *gc2;
  GdkGC *gc3;

  dump_widget ();

  if (shadow_type == GTK_SHADOW_IN) {
    gc1 = style->dark_gc[state_type];
    gc2 = style->light_gc[state_type];
    gc3 = style->bg_gc[state_type];
  } else {
    gc1 = style->light_gc[state_type];
    gc2 = style->dark_gc[state_type];
    gc3 = style->bg_gc[state_type];
  }

  if (area) {
    gdk_gc_set_clip_rectangle (gc1, area);
    gdk_gc_set_clip_rectangle (gc2, area);
    gdk_gc_set_clip_rectangle (gc3, area);
  }

  gdk_draw_arc (window, gc3, TRUE, x, y, width, height, 0, 360 * 64);
  gdk_draw_arc (window, gc1, FALSE, x, y, width, height, 45 * 64, 225 * 64);
  gdk_draw_arc (window, gc2, FALSE, x, y, width, height, 225 * 64, 180 * 64);

  if (area) {
    gdk_gc_set_clip_rectangle (gc1, NULL);
    gdk_gc_set_clip_rectangle (gc2, NULL);
    gdk_gc_set_clip_rectangle (gc3, NULL);
  }
}


static void
draw_cross (GtkStyle * style,
	    GdkWindow * window,
	    GtkStateType state_type,
	    GtkShadowType shadow_type,
	    GdkRectangle * area,
	    GtkWidget * widget,
	    gchar * detail, gint x, gint y, gint width, gint height)
{
  g_return_if_fail (style != NULL);
  g_return_if_fail (window != NULL);

  dump_widget ();

}


static void
draw_ramp (GtkStyle * style,
	   GdkWindow * window,
	   GtkStateType state_type,
	   GtkShadowType shadow_type,
	   GdkRectangle * area,
	   GtkWidget * widget,
	   gchar * detail,
	   GtkArrowType arrow_type, gint x, gint y, gint width, gint height)
{
  g_return_if_fail (style != NULL);
  g_return_if_fail (window != NULL);

  dump_widget ();

}


static void
draw_tab (GtkStyle * style,
	  GdkWindow * window,
	  GtkStateType state_type,
	  GtkShadowType shadow_type,
	  GdkRectangle * area,
	  GtkWidget * widget,
	  gchar * detail, gint x, gint y, gint width, gint height)
{
  g_return_if_fail (style != NULL);
  g_return_if_fail (window != NULL);

  dump_widget ();

  gtk_paint_box (style, window, state_type, shadow_type, area, widget, detail,
		 x, y, width, height);
}


static void
draw_shadow_gap (GtkStyle * style,
		 GdkWindow * window,
		 GtkStateType state_type,
		 GtkShadowType shadow_type,
		 GdkRectangle * area,
		 GtkWidget * widget,
		 gchar * detail,
		 gint x,
		 gint y,
		 gint width,
		 gint height,
		 GtkPositionType gap_side, gint gap_x, gint gap_width)
{
  GdkRectangle rect;
  g_return_if_fail (style != NULL);
  g_return_if_fail (window != NULL);

  dump_widget ();

  gtk_paint_shadow (style, window, state_type, shadow_type, area, widget,
		    detail, x, y, width, height);

  switch (gap_side) {
  case GTK_POS_TOP:
    rect.x = x + gap_x;
    rect.y = y;
    rect.width = gap_width;
    rect.height = 2;
    break;
  case GTK_POS_BOTTOM:
    rect.x = x + gap_x;
    rect.y = y + height - 2;
    rect.width = gap_width;
    rect.height = 2;
    break;
  case GTK_POS_LEFT:
    rect.x = x;
    rect.y = y + gap_x;
    rect.width = 2;
    rect.height = gap_width;
    break;
  case GTK_POS_RIGHT:
    rect.x = x + width - 2;
    rect.y = y + gap_x;
    rect.width = 2;
    rect.height = gap_width;
    break;
  }

  gtk_style_apply_default_background (style, window,
				      widget
				      && !GTK_WIDGET_NO_WINDOW (widget),
				      state_type, area, rect.x, rect.y,
				      rect.width, rect.height);
}


static void
draw_box_gap (GtkStyle * style,
	      GdkWindow * window,
	      GtkStateType state_type,
	      GtkShadowType shadow_type,
	      GdkRectangle * area,
	      GtkWidget * widget,
	      gchar * detail,
	      gint x,
	      gint y,
	      gint width,
	      gint height,
	      GtkPositionType gap_side, gint gap_x, gint gap_width)
{
  GdkRectangle rect;
  g_return_if_fail (style != NULL);
  g_return_if_fail (window != NULL);

  dump_widget ();

  gtk_paint_box (style, window, state_type, shadow_type, area, widget, detail,
		 x, y, width, height);

  switch (gap_side) {
  case GTK_POS_TOP:
    rect.x = x + gap_x;
    rect.y = y;
    rect.width = gap_width;
    rect.height = 2;
    break;
  case GTK_POS_BOTTOM:
    rect.x = x + gap_x;
    rect.y = y + height - 2;
    rect.width = gap_width;
    rect.height = 2;
    break;
  case GTK_POS_LEFT:
    rect.x = x;
    rect.y = y + gap_x;
    rect.width = 2;
    rect.height = gap_width;
    break;
  case GTK_POS_RIGHT:
    rect.x = x + width - 2;
    rect.y = y + gap_x;
    rect.width = 2;
    rect.height = gap_width;
    break;
  }

  gtk_style_apply_default_background (style, window,
				      widget
				      && !GTK_WIDGET_NO_WINDOW (widget),
				      state_type, area, rect.x, rect.y,
				      rect.width, rect.height);
}


static void
draw_extension (GtkStyle * style,
		GdkWindow * window,
		GtkStateType state_type,
		GtkShadowType shadow_type,
		GdkRectangle * area,
		GtkWidget * widget,
		gchar * detail,
		gint x,
		gint y, gint width, gint height, GtkPositionType gap_side)
{
  GdkRectangle rect;

  g_return_if_fail (style != NULL);
  g_return_if_fail (window != NULL);

  dump_widget ();

  icegradient_tab (style, window, state_type, shadow_type, area, widget,
		   detail, x, y, width, height);

  switch (gap_side) {
  case GTK_POS_TOP:
    rect.x = x + style->klass->xthickness;
    rect.y = y;
    rect.width = width - style->klass->xthickness * 2;
    rect.height = style->klass->ythickness;
    break;
  case GTK_POS_BOTTOM:
    rect.x = x + style->klass->xthickness;
    rect.y = y + height - style->klass->ythickness;
    rect.width = width - style->klass->xthickness * 2;
    rect.height = style->klass->ythickness;
    break;
  case GTK_POS_LEFT:
    rect.x = x;
    rect.y = y + style->klass->ythickness;
    rect.width = style->klass->xthickness;
    rect.height = height - style->klass->ythickness * 2;
    break;
  case GTK_POS_RIGHT:
    rect.x = x + width - style->klass->xthickness;
    rect.y = y + style->klass->ythickness;
    rect.width = style->klass->xthickness;
    rect.height = height - style->klass->ythickness * 2;
    break;
  }

  gtk_style_apply_default_background (style, window,
				      widget
				      && !GTK_WIDGET_NO_WINDOW (widget),
				      state_type, area, rect.x, rect.y,
				      rect.width, rect.height);
}


static void
draw_focus (GtkStyle * style,
	    GdkWindow * window,
	    GdkRectangle * area,
	    GtkWidget * widget,
	    gchar * detail, gint x, gint y, gint width, gint height)
{
  GdkGC *gc;
  gint8 dashes[] = { 1, 1 };

  g_return_if_fail (style != NULL);
  g_return_if_fail (window != NULL);

  dump_widget ();

  if ((width == -1) && (height == -1)) {
    gdk_window_get_size (window, &width, &height);
    width -= 1;
    height -= 1;
  } else if (width == -1) {
    gdk_window_get_size (window, &width, NULL);
    width -= 1;
  } else if (height == -1) {
    gdk_window_get_size (window, NULL, &height);
    height -= 1;
  }
  gc = gdk_gc_new (window);
  gdk_gc_copy (gc, style->black_gc);
  gdk_gc_set_line_attributes (gc, 1, GDK_LINE_ON_OFF_DASH, GDK_CAP_NOT_LAST,
			      GDK_JOIN_MITER);
  gdk_gc_set_dashes (gc, 0, dashes, 2);
  if (area) {
    gdk_gc_set_clip_rectangle (gc, area);
  }
  gdk_draw_rectangle (window, gc, FALSE, x, y, width, height);
  if (area) {
    gdk_gc_set_clip_rectangle (gc, NULL);
  }
  gdk_gc_destroy (gc);
}


static void
draw_slider (GtkStyle * style,
	     GdkWindow * window,
	     GtkStateType state_type,
	     GtkShadowType shadow_type,
	     GdkRectangle * area,
	     GtkWidget * widget,
	     gchar * detail,
	     gint x,
	     gint y, gint width, gint height, GtkOrientation orientation)
{
  g_return_if_fail (style != NULL);
  g_return_if_fail (window != NULL);

  dump_widget ();

  if ((width == -1) && (height == -1))
    gdk_window_get_size (window, &width, &height);
  else if (width == -1)
    gdk_window_get_size (window, &width, NULL);
  else if (height == -1)
    gdk_window_get_size (window, NULL, &height);

  draw_background (style, window, state_type, shadow_type, area, widget,
		   detail, x, y, width, height);
  if (orientation == GTK_ORIENTATION_HORIZONTAL) {
    draw_vline (style, window, state_type, area, widget, detail,
		style->klass->ythickness,
		height - style->klass->ythickness - 1, width / 2);
  } else {
    draw_hline (style, window, state_type, area, widget, detail,
		style->klass->xthickness,
		width - style->klass->xthickness - 1, height / 2);
  }
  gtk_paint_shadow (style, window, state_type, shadow_type, area, widget,
		    detail, x, y, width, height);
}

static void
draw_handle (GtkStyle * style,
	     GdkWindow * window,
	     GtkStateType state_type,
	     GtkShadowType shadow_type,
	     GdkRectangle * area,
	     GtkWidget * widget,
	     gchar * detail,
	     gint x,
	     gint y, gint width, gint height, GtkOrientation orientation)
{
  GdkGC *light_gc, *dark_gc;
  GdkRectangle dest;
  Detail *d;

  g_return_if_fail (style != NULL);
  g_return_if_fail (window != NULL);

  dump_widget ();

  if ((width == -1) && (height == -1))
    gdk_window_get_size (window, &width, &height);
  else if (width == -1)
    gdk_window_get_size (window, &width, NULL);
  else if (height == -1)
    gdk_window_get_size (window, NULL, &height);


  gtk_paint_box (style, window, state_type, shadow_type, area, widget,
		 detail, x, y, width, height);
  draw_background (style, window, state_type, shadow_type, area, widget,
		   detail, x, y + 1, width - 1, height - 2);

  /* Draw // in the box if so wanted */
  d = get_detail (style, detail);
  if ((d) && (d->decoration == DOUBLE_SLASH)) {
    light_gc = style->light_gc[state_type];
    dark_gc = style->dark_gc[state_type];

    dest.x = x + style->klass->xthickness;
    dest.y = y + style->klass->ythickness;
    dest.width = width - (style->klass->xthickness * 2);
    dest.height = height - (style->klass->ythickness * 2);

    gdk_gc_set_clip_rectangle (light_gc, &dest);
    gdk_gc_set_clip_rectangle (dark_gc, &dest);

    icegradient_slash_two (window, light_gc, dark_gc, width, height, x, y);

    gdk_gc_set_clip_rectangle (light_gc, NULL);
    gdk_gc_set_clip_rectangle (dark_gc, NULL);
  }

}



static void
icegradient_tab (GtkStyle * style,
		 GdkWindow * window,
		 GtkStateType state_type,
		 GtkShadowType shadow_type,
		 GdkRectangle * area,
		 GtkWidget * widget,
		 gchar * detail, gint x, gint y, gint width, gint height)
{
  GtkNotebook *notebook;
  GdkGC *lightgc, *darkgc;
  int orientation;

  if (!widget) {
    g_warning ("Call to icegradient_tab with NULL widget\n");
    return;
  }

  notebook = GTK_NOTEBOOK (widget);
  orientation = notebook->tab_pos;

  lightgc = style->light_gc[state_type];
  darkgc = style->dark_gc[state_type];

  if ((!style->bg_pixmap[state_type]) ||
      (gdk_window_get_type (window) == GDK_WINDOW_PIXMAP)) {
    if (area) {
      gdk_gc_set_clip_rectangle (style->bg_gc[state_type], area);
    }
    gdk_draw_rectangle (window, style->bg_gc[state_type], TRUE,
			x, y, width, height);
    if (area) {
      gdk_gc_set_clip_rectangle (style->bg_gc[state_type], NULL);
    }
  } else {
    gtk_style_apply_default_background (style, window,
					widget
					&& !GTK_WIDGET_NO_WINDOW (widget),
					state_type, area, x, y, width,
					height);
  }
  if (area) {
    gdk_gc_set_clip_rectangle (style->dark_gc[state_type], area);
    gdk_gc_set_clip_rectangle (style->light_gc[state_type], area);
  }
  switch (orientation) {
  case GTK_POS_TOP:
    gdk_draw_line (window, lightgc, x, y + height - 1, x, y);
    gdk_draw_line (window, lightgc, x, y, x + width - 1, y);
    gdk_draw_line (window, darkgc,
		   x + width - 1, y, x + width - 1, y + height - 1);
    break;
  case GTK_POS_BOTTOM:
    gdk_draw_line (window, lightgc, x, y, x, y + height - 1);
    gdk_draw_line (window, darkgc,
		   x, y + height - 1, x + width - 1, y + height - 1);
    gdk_draw_line (window, darkgc,
		   x + width - 1, y + height - 1, x + width - 1, y);
    break;
  case GTK_POS_LEFT:
    gdk_draw_line (window, lightgc, x, y + height - 1, x, y);
    gdk_draw_line (window, lightgc, x, y, x + width - 1, y);
    gdk_draw_line (window, darkgc,
		   x, y + height - 1, x + width - 1, y + height - 1);
    break;
  case GTK_POS_RIGHT:
    gdk_draw_line (window, lightgc, x, y, x + width - 1, y);
    gdk_draw_line (window, darkgc,
		   x + width - 1, y, x + width - 1, y + height - 1);
    gdk_draw_line (window, darkgc,
		   x, y + height - 1, x + width - 1, y + height - 1);
    break;
  }
  if (area) {
    gdk_gc_set_clip_rectangle (style->dark_gc[state_type], NULL);
    gdk_gc_set_clip_rectangle (style->light_gc[state_type], NULL);
  }
  /*gtk_paint_shadow(style, window, state_type, shadow_type, area, widget,
     detail, x, y, width, height);
   */
}
