/* Gtk Theme Switch, a fast and small Gtk theme switching utility that has a 
 * GUI and console interface.
 * Released under the GPL.
 * copyright Maher Awamy <muhri@muhri.net>
 * 	     Aaron Lehmann <aaronl@vitelus.com>
 * 	     Joshua Kwan <joshk@triplehelix.org> */

#include "expand.xpm"
#include "switch.h"
#define INIT_GTK if (!using_gtk) { gtk_init (&argc, &argv); using_gtk = 1; }

/* globals */
GHashTable *hash;
GList *glist=NULL;
GSList *kids=NULL;
gint preview_counter = 0;

gchar	*homedir, /* we use it a lot, so keep it handy */
	*execname, /* == argv[0] */
	*newfont; /* The name of a new font to use as the default if the user has
				 selected one. Otherwise NULL. */

GtkWidget *dockwin, *combo=NULL, *font_entry, *use_font_button, *box, *install_button, *browse;
gint hidden = 1;
/*end globals */

static short fgrep_gtkrc_for (gchar *needle)
{
	gchar *path_to_gtkrc = g_strdup_printf("%s/.gtkrc", homedir);
	FILE *gtkrc = fopen(path_to_gtkrc, "r");
	char tempbuf[16384], *commentsearch;
	g_free (path_to_gtkrc);

        if (!gtkrc)
          return 0;

	while (!feof (gtkrc))
	{
		fgets (tempbuf, 16383, gtkrc);
		/* Strip comments */
		for (commentsearch = tempbuf; *commentsearch != '\0'; ++commentsearch)
			if (*commentsearch == '#') { *commentsearch = '\0'; break; }
		if (strstr(tempbuf, needle))
		{
			fclose (gtkrc);
			return 1;
		}
	}
	fclose (gtkrc);
	return 0;
}

static GList*
get_dirs (void) 
{
	gchar *dirname;
	DIR *dir; 
	struct dirent *dent;
	struct stat stats;
	gchar *origdir=g_get_current_dir(); /* back up cwd */
	GList *list=0;
	
	dirname = g_strconcat(homedir,"/.themes",NULL);
	chdir (dirname);
	dir = opendir (dirname);
	if (dir)
	{
		while ((dent = readdir (dir)))
		{
                        gchar* gtkrc_path = g_strconcat(dent->d_name, "/gtk-2.0/gtkrc", NULL);
			stat(dent->d_name,&stats);
			if (!S_ISDIR(stats.st_mode)) continue;
			if (strcmp(dent->d_name,"." ) == 0) continue;
			if (strcmp(dent->d_name,"..") == 0) continue;
                        if (access(gtkrc_path, R_OK) == -1) continue;
			g_hash_table_insert (hash, dent->d_name, dirname);
			list = g_list_insert_sorted(list, g_strdup(dent->d_name), (GCompareFunc)strcmp);
                        g_free(gtkrc_path);
		}
	}
 
	/* DO NOT FREE dirname!! */
	/* We are supposed to. */
	/* But we need our own copy anyway, to store in the hash table */

	dirname = gtk_rc_get_theme_dir();
	chdir (dirname);
	dir = opendir (dirname);
	if (dir)
	{
		while ((dent = readdir (dir)))
		{
                        gchar* gtkrc_path = g_strconcat(dent->d_name, "/gtk-2.0/gtkrc", NULL);
			stat(dent->d_name,&stats);
			if (!S_ISDIR(stats.st_mode)) continue;
			if (strcmp(dent->d_name, "." ) == 0) continue;
			if (strcmp(dent->d_name, "..") == 0) continue;
                        if (access(gtkrc_path, R_OK) == -1) continue;
			g_hash_table_insert (hash, dent->d_name, dirname);
			list =  g_list_insert_sorted(list, g_strdup(dent->d_name), (GCompareFunc)strcmp);
                        g_free(gtkrc_path);
		}
	}
	
	/* DO NOT FREE dirname!! */
	/* We are supposed to. */
	/* But we need our own copy anyway, to store in the hash table */

	chdir(origdir); /* Go back to where we were */
	g_free(origdir); /* Now go play like a good little program */

	return list;
}

/* Find the first element in t2 that does not exist in t1.
 * Uses the supplied cmpfunc() for determining equality of list->data
 * strcmp() is the recommended compare function */
static GList*
compare_glists (GList *t1, GList *t2, GCompareFunc cmpfunc)
{
	GList *tmp1;
	for (; t2; t2=t2->next)
	{ 
		short matched = 0;
		for (tmp1=t1; tmp1; tmp1=tmp1->next)
			if ((*cmpfunc)(tmp1->data, t2->data) == 0) { matched = 1; break; }
		if (!matched) return t2;
	}
	return 0;
}

static void
preview_clicked(GtkWidget *button, gpointer data)
{
	gchar *entry; 
	gchar *dir; 
	gchar *rc;
	gchar *actual;

	entry = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(combo)->entry));
	dir = g_hash_table_lookup(hash,entry);
	dir = g_strconcat(dir,"/",NULL);
	rc = g_strconcat(dir,entry,NULL);
	actual = g_strconcat(dir,entry,NULL);
	rc = g_strconcat(rc,"/gtk/gtkrc",NULL);
	update_newfont ();
	preview(rc);
	g_free(rc);
	g_free(actual);
}

/* Update 'newfont' */
static void update_newfont (void)
{
	if (newfont) g_free (newfont);

	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(use_font_button)))
	{
		gchar *newerfont = gtk_entry_get_text (GTK_ENTRY(font_entry));
		if (newerfont && newerfont[0])
		{
			newfont = g_strdup(newerfont);
		}
	}
	else newfont = NULL;
}

static void 
apply_clicked(GtkWidget *button, gpointer data)
{
	gchar *entry = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(combo)->entry));
	gchar *dir = g_hash_table_lookup(hash,entry);
	gchar *name = g_strdup_printf ("%s/%s/gtk/gtkrc", dir, entry);
	GtkStyle *style;

	update_newfont ();

	ok_clicked(name);
	g_free(name);

	/* make sure we get the ClientEvent ourselves */
	while (gtk_events_pending())
		gtk_main_iteration();

	/* sync the font field with the GTK font */
	style = gtk_rc_get_style (font_entry);
	if (style && style->rc_style)
		gtk_entry_set_text (GTK_ENTRY(font_entry), style->rc_style->font_name);
}

static void
usage (void)
{
	printf("\
%s command line options:\n\
-h[elp]\t\t\t(display this help message)\n\
-d[ock]\t\t\t(open dock)\n\
file\t\t\t(switch to theme (install if theme is a tarball))\n\
-p[review] file\t\t(preview a theme (installs if file is a tarball))\n\
-i[nstall] theme.tar.gz\t(install a .tar.gz)\n\
-f[ont] fontstring\t(set GTK's main font to fontstring)\n\n\
\
Passing no command line arguments will cause %s to start in dock-mode)\n\n\
\
\"file\" represents any one of (looked for in the listed order):\n\
1) An absoulte or relative path to a GTK theme directory.\n\
   A directory is considered a theme directory if it contains a\n\
   gtk/gtkrc file.\n\
2) A gzipped tar file which expands to a GTK theme directory as explained in 1).\n\
3) A GTK theme directory with the name passed located in ~/.themes.\n\
4) A GTK theme directory with the name passed located in the global\n\
   \"gtk_themedir\"\n\
If none of these files are located and found to be correct, the program will\n\
exit with an error.\n",
execname, execname);
}

static void
write_rc_file (gchar *include_file, gchar *path)
{
	FILE *gtkrc = fopen(path, "w");
	/* the caps stuff is bullshit for gnome */
	fprintf(gtkrc, "# -- THEME AUTO-WRITTEN DO NOT EDIT\ninclude \"%s\"\n\n", include_file);
	if (newfont)
		/* User set a custom font to overrride defaults. */
		fprintf (gtkrc, "style \"user-font\"\n{\n  font=\"%s\"\n}\nwidget_class \"*\" style \"user-font\"\n\n", newfont);
	fprintf(gtkrc, "include \"%s/.gtkrc.mine\"\n\n# -- THEME AUTO-WRITTEN DO NOT EDIT\n", homedir);
	fclose (gtkrc);
}

static void
ok_clicked (gchar *rc_file)
{
	/* Write the config file to disk */
	gchar *path_to_gtkrc = g_strdup_printf("%s/.gtkrc", homedir);
	write_rc_file (rc_file, path_to_gtkrc);
	send_refresh_signal();
	g_free(path_to_gtkrc);
}

static void
preview_ok_clicked (gchar *rc_file)
{
	/* Write the config file to disk */
	gchar *path_to_gtkrc = g_strdup_printf("%s/.gtkrc", homedir);
	rename (rc_file, path_to_gtkrc);
	send_refresh_signal();
	g_free(path_to_gtkrc);
}

static void
install_clicked (GtkWidget *w, gpointer data)
{
	GtkWidget *checkbutton = gtk_check_button_new_with_label ("Switch to theme after installation");
	GtkWidget *fs = gtk_file_selection_new ("Select a GTK theme tarball");
	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button), "clicked", GTK_SIGNAL_FUNC(install_ok_clicked), fs);
	gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fs)->cancel_button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), (gpointer)fs);
	gtk_box_pack_start (GTK_BOX(GTK_FILE_SELECTION(fs)->main_vbox), checkbutton, FALSE, FALSE, 0);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(checkbutton), FALSE);
	gtk_object_set_data (GTK_OBJECT(fs), "checkbutton", checkbutton);
	gtk_widget_show(checkbutton);
	gtk_widget_show(fs);
}

static void
install_ok_clicked (GtkWidget *w, gpointer data)
{
	gchar *rc_file, *beginning_of_theme_name;
	gint i, pos;
	short slashes=0;
	gboolean cbstate=gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(gtk_object_get_data(GTK_OBJECT(data), "checkbutton")));

	search_for_theme_or_die_trying(gtk_file_selection_get_filename(GTK_FILE_SELECTION(data)), &rc_file);
	gtk_widget_destroy(GTK_WIDGET(data));
	/* ok, we're like evil or something, but that won't stop us */
	for (i=strlen(rc_file); i != 0; --i)
	{
		if (rc_file[i] == '/') ++slashes;
		if (slashes == 2) { rc_file[i] = '\0'; break; }
	}
	beginning_of_theme_name = rc_file;
	for (i=strlen(rc_file) /*different*/; i != 0; --i)
	{
		if (rc_file[i] == '/') { beginning_of_theme_name = &rc_file[i+1]; break; }
	}
	/* we've been very naugthy, but we should have the theme's NAME now..
	 * it's about time. */
	/* get the list item that contains this */
	pos = g_list_position (glist, g_list_find_custom (glist, beginning_of_theme_name, (GCompareFunc) strcmp));
	if (pos != -1)
		/* set combo's item to the newly-installed theme */
		gtk_list_select_item(GTK_LIST(GTK_COMBO(combo)->list), pos);
	
	if (cbstate) /* checkbutton pressed */
		apply_clicked(NULL, NULL);

	/* I guess we should free this... */
	g_free (rc_file);
}

static void
set_font (GtkWidget *w, GtkWidget *dialog)
{
	if (newfont) g_free (newfont);
	newfont = gtk_font_selection_dialog_get_font_name (GTK_FONT_SELECTION_DIALOG(dialog));
	gtk_entry_set_text (GTK_ENTRY(font_entry), newfont);
	gtk_widget_destroy (dialog);
}

static void
font_browse_clicked (GtkWidget *w, gpointer data)
{
	GtkWidget *font_dialog;
	font_dialog = gtk_font_selection_dialog_new ("Select Font");
	gtk_font_selection_dialog_set_preview_text (GTK_FONT_SELECTION_DIALOG(font_dialog), "Gtk Theme Switch");
	gtk_font_selection_dialog_set_font_name (GTK_FONT_SELECTION_DIALOG(font_dialog), gtk_entry_get_text(GTK_ENTRY(font_entry)));
	gtk_signal_connect (GTK_OBJECT(GTK_FONT_SELECTION_DIALOG(font_dialog)->ok_button), "clicked", GTK_SIGNAL_FUNC(set_font), (gpointer)font_dialog);
	gtk_signal_connect_object (GTK_OBJECT(GTK_FONT_SELECTION_DIALOG(font_dialog)->cancel_button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), (gpointer)font_dialog);

	gtk_widget_show (font_dialog);
}

void quit()
{
     	gint i;
     	i = g_slist_length(kids);
     	if (i) {
		     	for (i = 0 ; i < g_slist_length(kids); i++) {
			     	kill(GPOINTER_TO_INT(g_slist_nth_data(kids,i)),9);			}
	}

     	gtk_exit(0);
}

static void
dock (void)
{
	GtkWidget *label, *button, *pixmap, *evbox;
	GdkColormap *colormap;
     	GdkPixmap *gdkpixmap;
     	GdkBitmap *gdkbitmap;
       	GtkTooltips *tips;
     	
	dockwin = gtk_dialog_new();
     	gtk_widget_realize(dockwin);
	gtk_window_set_title(GTK_WINDOW(dockwin),"Theme Dock");
     	gtk_window_set_policy(GTK_WINDOW(dockwin), TRUE, TRUE, TRUE);
	gtk_signal_connect(GTK_OBJECT(dockwin),"destroy",GTK_SIGNAL_FUNC(quit),NULL);
	colormap = gtk_widget_get_colormap(dockwin);
	box = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dockwin)->vbox), box, FALSE, FALSE, 0);
	tips = gtk_tooltips_new();
     	gtk_tooltips_set_delay(tips,500);
	label = gtk_label_new("Theme: ");
	gtk_box_pack_start(GTK_BOX(box),label,FALSE,FALSE,FALSE);

	combo = gtk_combo_new();
	glist = get_dirs();
	gtk_combo_set_popdown_strings(GTK_COMBO(combo), glist);
	gtk_box_pack_start(GTK_BOX(box),combo,TRUE,TRUE,FALSE);
	gdkpixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap, &gdkbitmap, NULL, expand_xpm);
     	
     	pixmap = gtk_pixmap_new (gdkpixmap, gdkbitmap);
     	gdk_pixmap_unref(gdkpixmap);
     	gdk_bitmap_unref(gdkbitmap);
     	evbox = gtk_event_box_new();

     	gtk_tooltips_set_tip(tips,evbox,"click here for more options","private");
     	gtk_widget_set_events(evbox, GDK_BUTTON_PRESS);
     	gtk_signal_connect(GTK_OBJECT(evbox), "button_press_event", GTK_SIGNAL_FUNC(on_eventbox_click), NULL);
     
     	gtk_container_add(GTK_CONTAINER(evbox), pixmap);
     	gtk_box_pack_start(GTK_BOX(box),evbox,FALSE,FALSE,FALSE);
	gtk_widget_show_all(box);
     
     	box = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dockwin)->vbox), box, FALSE, FALSE, 0);

	use_font_button = gtk_check_button_new_with_label("Use font: ");
	gtk_box_pack_start(GTK_BOX(box), use_font_button, FALSE, FALSE, 0);
	gtk_widget_show(use_font_button);
     
	font_entry = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(box), font_entry, TRUE, TRUE, 0);
	gtk_widget_show(font_entry);
	if (newfont)
	{
		gtk_entry_set_text (GTK_ENTRY(font_entry), newfont);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(use_font_button), TRUE);
		g_free (newfont);
	}
	else
	{
		GtkStyle *style = gtk_rc_get_style (font_entry);
		if (style && style->rc_style)
			gtk_entry_set_text (GTK_ENTRY(font_entry), style->rc_style->font_name);
	}
	
	newfont = g_strdup(gtk_entry_get_text(GTK_ENTRY(font_entry)));

	if (newfont != 0 && newfont[0])
	{
		/* Very dirty hack...
		   We want to only set the checkbutton to TRUE if the user specified
		   the font. If the name occurs in their ~/.gtkrc file, they probably did.
		   If it isn't, they probably didn't. So, "grep" the file for the font string. */
		if (fgrep_gtkrc_for(newfont))
			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(use_font_button), TRUE);
	}

	browse = gtk_button_new_with_label("Browse...");
	gtk_signal_connect(GTK_OBJECT(browse), "clicked", GTK_SIGNAL_FUNC(font_browse_clicked), NULL);
	gtk_box_pack_start(GTK_BOX(box), browse, FALSE, FALSE, 0);
	
	button = gtk_button_new_with_label("Apply");
	gtk_signal_connect(GTK_OBJECT(button),"clicked",GTK_SIGNAL_FUNC(apply_clicked),NULL);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG (dockwin)->action_area),button,TRUE,TRUE,FALSE);
	gtk_widget_show(button);
     
	button = gtk_button_new_with_label("Preview");
	gtk_signal_connect(GTK_OBJECT(button),"clicked",GTK_SIGNAL_FUNC(preview_clicked),NULL);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG (dockwin)->action_area),button,TRUE,TRUE,FALSE);
	gtk_widget_show(button);
     
	install_button = gtk_button_new_with_label("Install New Theme");
	gtk_signal_connect(GTK_OBJECT(install_button), "clicked", GTK_SIGNAL_FUNC(install_clicked), NULL);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG (dockwin)->action_area),install_button,TRUE,TRUE,FALSE);

	gtk_widget_show(dockwin);

}

void on_eventbox_click()
{
     	if (hidden) {
	     	show_stuff();
	} else {
	     	hide_stuff();
	}
}

void hide_stuff()
{
     	gtk_widget_hide(box);
     	gtk_widget_hide(install_button);
     	gtk_widget_hide(browse);
	hidden = 1;
}

void show_stuff()
{
     	gtk_widget_show(box);
     	gtk_widget_show(install_button);
     	gtk_widget_show(browse);
     	hidden = 0;
}

static void
clist_insert (GtkWidget *clist)
{
	gint i;
	gchar *stuff[4][2] = { { "blah1", "blah2" },
			       { "blah1", "blah2" },
			       { "blah1", "blah2" },
			       { "blah1", "blah2" } };
	
	for (i = 0; i < 4; i++) 
		gtk_clist_append (GTK_CLIST (clist), stuff[i]);
	return;
}

static int
rightclick (GtkWidget *w, GdkEventButton *event, gpointer data)
{
	if (event->button == 3)
		gtk_menu_popup (GTK_MENU(data), NULL, NULL, NULL, NULL, 1, event->time);
	return 1;
}

static void
preview (gchar *rc_file)
{
	FILE *pipe;
     	gint got_it = 0, it_died = 0;
     	gchar *line;
     	gchar *path = g_strdup_printf ("%s/.gtkrc.tmp-%i", homedir, preview_counter);
	gchar *command = g_strdup_printf ("%s -_dont-use-this %s &", execname, path);
	write_rc_file (rc_file, path);
	
	pipe = popen(command,"r");
     	
     	if (pipe == NULL) {
	     	g_print("gts: error previewing\n");
	     	return;
	}

     	fcntl(fileno(pipe), F_SETFL, O_NONBLOCK);
     	
     	line = (gchar *)g_malloc(1024);     
     	while(!feof(pipe)) {
      		fgets(line,1024,pipe);
	     	line[strlen(line)-1] = '\0';	       
       	     	if (!got_it && !g_strncasecmp(line,"pid=",4)) {
		     	kids = g_slist_append(kids,GINT_TO_POINTER(atoi(line+4)));
		     	got_it = 1;
		} else if (!it_died && !g_strncasecmp(line,"die=",4)) {
		     	kids = g_slist_remove(kids,GINT_TO_POINTER(atoi(line+4)));
		     	it_died = 1;
		     	break;
		}
	     
		while (gtk_events_pending())
	     		gtk_main_iteration();
		usleep(50000);
	}
            
     	pclose(pipe);
	g_free (line);
	g_free (path);
	g_free (command);
	++preview_counter;
}

static void
preview_window (gchar *rc_file)
{

	GtkWidget *label;
	GtkWidget *win;
	GtkWidget *button;
	GtkWidget *toggle_button;
	GtkWidget *check_button;
	GtkWidget *sw;
	GtkWidget *clist;
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkWidget *radio;
	GtkWidget *radio2;
	GtkWidget *radio3;
	GtkWidget *ok;
	GtkWidget *cancel;
	GtkWidget *hbox2;
	GtkWidget *notebook;
	GtkWidget *text;
	GtkWidget *popup;
	GtkWidget *item;
	gint argc=1;
	gchar **argv = &execname;
	gchar *default_files[] = { rc_file, NULL};

	gchar *titles[2] = { "Column 1", "Column 2" };
	GSList *group;

	gtk_rc_set_default_files (default_files);
	gtk_init (&argc, &argv);
	
	win = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(win), "Gtk Theme Switch theme preview");
	gtk_window_set_policy(GTK_WINDOW(win), TRUE, TRUE, FALSE);
	gtk_signal_connect(GTK_OBJECT(win), "destroy", GTK_SIGNAL_FUNC(quit_preview), NULL);
	
	vbox = gtk_vbox_new(FALSE,0);
	notebook = gtk_notebook_new();
	gtk_container_add (GTK_CONTAINER(win), notebook);
	label = gtk_label_new("Page 1");
	gtk_notebook_append_page (GTK_NOTEBOOK(notebook), vbox, label);
	label = gtk_label_new("Label");
	button = gtk_button_new_with_label("Button");
	toggle_button = gtk_toggle_button_new_with_label("Toggle Button");
	check_button = gtk_check_button_new_with_label("Check Button");
	hbox = gtk_hbox_new(FALSE,0);
	radio = gtk_radio_button_new_with_label(NULL,"Radio 1");
	group = gtk_radio_button_group (GTK_RADIO_BUTTON (radio));
	radio2 = gtk_radio_button_new_with_label(group,"Radio 2");
	group = gtk_radio_button_group (GTK_RADIO_BUTTON (radio2));
	radio3 = gtk_radio_button_new_with_label(group,"Radio 3");
	gtk_box_pack_start((GtkBox*)hbox,radio,FALSE,FALSE,FALSE);
	gtk_box_pack_start((GtkBox*)hbox,radio2,FALSE,FALSE,FALSE);
	gtk_box_pack_start((GtkBox*)hbox,radio3,FALSE,FALSE,FALSE);
	sw = gtk_scrolled_window_new(NULL,NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
	clist = gtk_clist_new_with_titles(2,titles);
	clist_insert(clist);     
	ok = gtk_button_new_with_label("Ok");
	gtk_signal_connect_object(GTK_OBJECT(ok),"clicked",GTK_SIGNAL_FUNC(preview_ok_clicked), (gpointer) rc_file);
	cancel = gtk_button_new_with_label("Cancel");
	gtk_signal_connect_object(GTK_OBJECT(cancel),"clicked",GTK_SIGNAL_FUNC(quit_preview),NULL);
	hbox2 = gtk_hbox_new(FALSE,0);
	gtk_box_pack_start((GtkBox*)hbox2,ok,TRUE,TRUE,FALSE);
	gtk_box_pack_start((GtkBox*)hbox2,cancel,TRUE,TRUE,FALSE);
	gtk_container_add(GTK_CONTAINER(sw),clist);
	gtk_box_pack_start((GtkBox*)vbox,label,FALSE,FALSE,FALSE);
	gtk_box_pack_start((GtkBox*)vbox,button,FALSE,FALSE,FALSE);
	gtk_box_pack_start((GtkBox*)vbox,toggle_button,FALSE,FALSE,FALSE);
	gtk_box_pack_start((GtkBox*)vbox,check_button,FALSE,FALSE,FALSE);
	gtk_box_pack_start((GtkBox*)vbox,hbox,FALSE,FALSE,FALSE);
	gtk_box_pack_start((GtkBox*)vbox,sw,TRUE,TRUE,FALSE);
	gtk_box_pack_start((GtkBox*)vbox,hbox2,FALSE,FALSE,FALSE);

	vbox = gtk_vbox_new (FALSE, 0);
	label = gtk_label_new ("Page 2");
	gtk_notebook_append_page (GTK_NOTEBOOK(notebook), vbox, label);
	label = gtk_label_new ("Insensitive Label");
	gtk_widget_set_sensitive (label, 0);
	gtk_box_pack_start (GTK_BOX(vbox), label, FALSE, FALSE, 0);
	button = gtk_button_new_with_label ("Insensitive Button");
	gtk_widget_set_sensitive (button, 0);
	gtk_box_pack_start (GTK_BOX(vbox), button, FALSE, FALSE, 0);
	sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(sw), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	text = gtk_text_new (NULL, NULL);
	gtk_text_set_editable (GTK_TEXT(text), 1);
	gtk_container_add (GTK_CONTAINER(sw), text);
	gtk_text_insert (GTK_TEXT(text), NULL, NULL, NULL, "Type some text here\nto check if your\ntheme has a problem\nwith the text widget.\nAlso right-click here\nfor a popup-menu sample.\n", -1);
	popup = gtk_menu_new ();
	gtk_widget_show (popup);
	item = gtk_menu_item_new_with_label ("Menu Entry 1");
	gtk_widget_show (item);
	gtk_menu_append(GTK_MENU(popup), item);
	item = gtk_menu_item_new_with_label ("Menu Entry 2");
	gtk_widget_show (item);
	gtk_menu_append(GTK_MENU(popup), item);
	item = gtk_menu_item_new_with_label ("Menu Entry 3");
	gtk_widget_show (item);
	gtk_menu_append(GTK_MENU(popup), item);
	gtk_signal_connect(GTK_OBJECT(text), "button_press_event", GTK_SIGNAL_FUNC(rightclick), popup);
	gtk_box_pack_start(GTK_BOX(vbox), sw, TRUE, TRUE, 0);

	label = gtk_label_new ("About");
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_notebook_append_page (GTK_NOTEBOOK(notebook), vbox, label);
	label = gtk_label_new ("Gtk Theme Switch\nby Maher Awamy <muhri@muhri.net>\nand Aaron Lehmann <aaronl@vitelus.com>\nhttp://www.muhri.net/gthemeswitch.php3");
	gtk_box_pack_start (GTK_BOX(vbox), label, TRUE, TRUE, 0);

	gtk_widget_show_all(win);
	g_print("pid=%d\n",getpid());
	gtk_main ();
	
	unlink (rc_file);
	
	_exit (1); /* no change */
}

void quit_preview()
{
	g_print("die=%d\n",getpid());
     	gtk_main_quit();
}

static void
send_refresh_signal(void)
{
	GdkEventClient event;
    	event.type = GDK_CLIENT_EVENT;
	event.send_event = TRUE;
	event.window = NULL;
	event.message_type = gdk_atom_intern("_GTK_READ_RCFILES", FALSE);
	event.data_format = 8;
	gdk_event_send_clientmessage_toall((GdkEvent *)&event);
}

/* Sets rc_file to the rc_file of the theme if the result is true.
 * It is the caller's repsonsibility to free rc_file */
static short is_themedir (gchar *path, gchar **rc_file)
{
	gchar *test_rc_file;
	struct stat info;
	test_rc_file = g_strdup_printf ("%s/gtk/gtkrc",path);
	if (stat(test_rc_file, &info) == 0 && (S_ISREG(info.st_mode) || S_ISLNK(info.st_mode)))
	{
		/* $path/gtk/gtkrc exists, and is a regular file */
		*rc_file = test_rc_file;
		return 1;
	}
	else
	{
		g_free (test_rc_file);
		return 0;
	}
}

static short install_tarball (gchar *path, gchar **rc_file)
{
	gchar *command, *themedir;
	gint result;
	GList *new_list, *new_theme;

	themedir = g_strdup_printf ("%s/.themes", homedir);

	if (path[0] != '/')
	{
		gchar *cwd = g_get_current_dir();
		command = g_strdup_printf ("tar --directory %s -xzf %s/%s 2>/dev/null", themedir, cwd, path);
		g_free (cwd);
	}
	else
		command = g_strdup_printf ("tar --directory %s -xzf %s 2>/dev/null", themedir, path);

	/* Ensure that ~/.themes exists */
	mkdir (themedir, S_IRUSR | S_IWUSR | S_IXUSR);

	result = system(command);
	g_free (command);
	g_free (themedir);
	if (result != EXIT_SUCCESS)
		return 0;
	/* We don't do that anymore. Now we find the first new directory
	 * in either themedir, and presume that to be the name of the new theme.
	 * NOT FOOLPROOF, but good as long as you don't mess with stuff while the
	 * program is running. At least better than deriving the theme name from
	 * the tarball name. Ugh. */
	
	new_list = get_dirs();
	new_theme = compare_glists(glist, new_list, (GCompareFunc)strcmp);
	if (new_theme)
	{
		result = is_installed_theme(new_theme->data, rc_file);
		g_list_foreach (glist, (GFunc)g_free, NULL);
		g_list_free (glist);
		glist = new_list;
		/* Update combo, but only in dock mode */
		if (combo)	gtk_combo_set_popdown_strings(GTK_COMBO(combo), glist);
	}
	else
	{
		gchar *interestingpath, basename[1024];
		g_list_foreach (new_list, (GFunc)g_free, NULL);
		g_list_free (new_list);
		/* fall back to our old hack if no new dir was created, say if the theme
		 * was already installed... */
	
		/* Set rc_file to our best darn guess of the resultant gtk theme
		 * dir/gtk/gtkrc. This is very tricky. The hack that is used to is
		 * to return the segment in path between the last slash and the
		 * first dot or dash after that. */
		interestingpath = &path[strlen(path)-1];
		while (interestingpath != path && *(interestingpath-1) != '/') interestingpath--;
		strcpy (basename, interestingpath);
		for (interestingpath = &basename[0]; *interestingpath != '\0'; ++interestingpath)
		{
			if (*interestingpath == '-' || *interestingpath == '.')
			{
				*interestingpath = '\0';
				break;
			}
		}
		result = is_installed_theme(basename, rc_file);
	}
	
	return result;
}

static short
is_installed_theme (gchar *path, gchar **rc_file)
{
	gchar *gtk_rc_theme_dir = gtk_rc_get_theme_dir();
	/* don't strlen things twice when computing the size to use for fullpath */
	gint path_len = strlen(path), homedir_len = strlen(homedir);
	/* ditto with get_theme_dir */
	gint gtk_rc_theme_dir_len = strlen(gtk_rc_theme_dir);
	gchar *fullpath = (gchar *) g_malloc (MAX(homedir_len+path_len+10,
										  gtk_rc_theme_dir_len+path_len+1));
	short result;

	/* use memcpy since we already have the lengths */
	memcpy (fullpath, homedir, homedir_len);
	memcpy (fullpath+homedir_len, "/.themes/", 9);
	/* add 1 to length so we get the null char too */
	memcpy (fullpath+homedir_len+9, path, path_len+1);

	if (is_themedir(fullpath, rc_file))
	{
		g_free(fullpath);
		return 1;
	}
	memcpy (fullpath, gtk_rc_theme_dir, gtk_rc_theme_dir_len);
	/* add 1 to length so we get the null char too */
	memcpy (fullpath+gtk_rc_theme_dir_len, path, path_len+1);
	
	result = is_themedir(fullpath, rc_file);

	g_free(fullpath);

	return result;
}

static void
search_for_theme_or_die_trying (gchar *actual, gchar **rc_file)
{
	if (!is_themedir(actual, rc_file) &&
		!install_tarball(actual, rc_file) &&
		!is_installed_theme(actual, rc_file))
	{
		fprintf (stderr, "\
%s: Sorry, \"%s\" does not appear to be a valid theme directory or tarball!\n", execname, actual);
		exit (EXIT_FAILURE);
	}
}

static gint
switcheroo (gchar *actual)
{
	gchar *rc_file;
	search_for_theme_or_die_trying (actual, &rc_file);
	/* If we get here, we actually found the theme */
	ok_clicked(rc_file);
	g_free (rc_file);
	return EXIT_SUCCESS;
}



int main (int argc, char *argv[])
{
	gchar *rc_file;
	gchar *actual;
	gint i;
	gint result = EXIT_SUCCESS;
	short using_gtk = 0;

	newfont = 0;
	execname = argv[0];
	homedir = getenv("HOME");
	hash = g_hash_table_new (g_str_hash, g_str_equal);

	if (argc == 1) /* start in dock mode auto if no args */
	{
		INIT_GTK
		dock();
	}

	else if (strcmp(argv[1], "-_dont-use-this") == 0)
	{
		preview_window (argv[2]); /* GARGARGAR */
	}				  /* hehe, aaronl is crazy */
	
	for (i=1; i != argc; ++i)
	{
		if (argv[i][0] == '-')
		{
			/* Single-arg commands/options */
			if (argv[i][1] == 'd')
			{
				INIT_GTK
				dock();
				continue;
			}
			
			/* Double-arg commands/options */
			if (i+1 != argc)
			{
				if (argv[i][1] == 'p')
				{
					glist = get_dirs ();
					actual = argv[++i];
					if (!is_themedir(actual, &rc_file) && !install_tarball(actual, &rc_file) && !is_installed_theme(actual, &rc_file))
					{
						fprintf (stderr, "\
%s: Sorry, \"%s\" does not appear to be a valid theme directory or tarball!\n", execname, actual);
						result = EXIT_FAILURE;
					}
					else
					{
						preview (rc_file);
						g_free (rc_file);
					}
					continue;
				}

				if (argv[i][1] == 'i')
				{
					actual = argv[++i];
					if (!install_tarball(actual, &rc_file))
					{
						fprintf (stderr, "\
%s: Sorry, \"%s\" does not appear to be a valid theme tarball!\n", execname, actual);
					}
					result = EXIT_FAILURE;
					continue;
				}

				if (argv[i][1] == 'f')
				{
					newfont = g_strdup (argv[++i]);
					continue;
				}
			}

			/* if this starts with a minus, it's either an unrecognized command
			 * or -help. Perfect */
			usage();
			result = EXIT_FAILURE;
			continue;
		}
		/* got here... fallback and assume it's a theme */
		glist = get_dirs ();
		gtk_init (&argc, &argv);
		result |= switcheroo(argv[i]);
	}

	if (using_gtk) {
		gtk_main();
		i = g_slist_length(kids);
	     	if (i) {
		     	for (i = 0 ; i < g_slist_length(kids); i++) {
			     	kill(GPOINTER_TO_INT(g_slist_nth_data(kids,i)),9);
			}
		}
	}
     
     	return result;
}
