/*
  QtCurve (C) Craig Drummond, 2003 - 2007 Craig.Drummond@lycos.co.uk

  ----

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public
  License version 2 as published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; see the file COPYING.  If not, write to
  the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
  Boston, MA 02110-1301, USA.
 */

#include "common.h"
#include "config.h"
#define CONFIG_READ
#include "config_file.c"
#include <gtk/gtk.h>
#include <time.h>
#ifdef HAVE_LOCALE_H
#include <locale.h>
#endif
#include <gdk/gdkcolor.h>
#include <gtk/gtkenums.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/unistd.h>
#include <stdio.h>

#define QTC_DEFAULT_TO_KDE3   /* Should we default to KDE3, or KDE4 settings when not running under KDE? */

/*
#define QTC_DEBUG
*/

static void generateMidColor(GdkColor *a, GdkColor *b, GdkColor *mid, double factor);

static int strcmp_i(const char *s1, const char *s2)
{
    char c1,
         c2;

    for(;;)
    {
        c1=*s1++;
        c2=*s2++;
        if(!c1 || !c2)
            break;
        if(isupper(c1))
            c1=tolower (c1);
        if(isupper(c2))
            c2=tolower (c2);
        if(c1!=c2)
            break;
    }
    return (int)c2-(int)c1;
}

struct qt_toolbar
{
    GtkIconSize     size;
    GtkToolbarStyle style;
};

enum qt_fcolor
{
    COLOR_BACKGROUND,
    COLOR_BUTTON,
    COLOR_SELECTED,
    COLOR_WINDOW,

    COLOR_FOREGROUND,
    COLOR_MID,
    COLOR_TEXT,
    COLOR_TEXT_SELECTED,
    COLOR_BUTTON_TEXT,
    COLOR_NONE,
    COLOR_NUMCOLORS=COLOR_NONE  /* NONE does not count! */
};

typedef enum
{
    GTK_APP_UNKNOWN,
    GTK_APP_MOZILLA,
    GTK_APP_OPEN_OFFICE,
    GTK_APP_VMPLAYER,
    GTK_APP_GIMP,
    GTK_APP_GIMP_PLUGIN,
    GTK_APP_JAVA,
    GTK_APP_JAVA_SWT
    /*GTK_APP_GAIM*/
} EGtkApp;

struct QtData
{
    GdkColor          colors[COLOR_NUMCOLORS];
    char              *font,
#ifdef QTC_DO_ICON_MAPPING
                      *icons,
#endif
                      *progressfont_rc;
    struct qt_toolbar toolbar;
    gboolean          button_icons;
    EGtkApp           app;
    gboolean          qt4;
};

#include <gmodule.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <pwd.h>
#include <sys/types.h>

#define DEFAULT_KDE_COLORS "active=#000000^e#dddfe4^e#ffffff^e#ffffff^e#555555^e#c7c7c7^e#000000^e#ffffff^e#000000^e#ffffff^e#efefef^e#000000^e#678db2^e#ffffff^e#0000ee^e#52188b^e"
#define DEFAULT_KDE_FONT      "Sans Serif"
#define DEFAULT_KDE_FONT_SIZE 10.0
#define MAX_LINE_LEN 1024

struct QtData qtSettings;

static gboolean useQt3Settings()
{
    static const char *full=NULL;
    static const char *vers=NULL;

    if(!full)
        full=getenv("KDE_FULL_SESSION");

    if(full && !vers)
        vers=getenv("KDE_SESSION_VERSION");

#ifdef QTC_DEFAULT_TO_KDE3
    return !full || (!vers || atoi(vers)<4);
#else
    return full && (!vers || atoi(vers)<4);
#endif
}

#ifdef QTC_DO_ICON_MAPPING
static const char * defaultIcons()
{
    return qtSettings.qt4 ? "oxygen" : "crystalsvg";
}

static const char * kdeIconsPrefix()
{
    return !qtSettings.qt4 && strlen(KDE3_ICONS_PREFIX)>2
            ? KDE3_ICONS_PREFIX
            : qtSettings.qt4 && strlen(KDE4_ICONS_PREFIX)>2
                ? KDE4_ICONS_PREFIX
                : NULL;
}
#endif

enum
{
    SECT_NONE,
    SECT_PALETTE,
    SECT_GENERAL,
    SECT_KDE,
    SECT_ICONS,
    SECT_TOOLBAR_STYLE,
    SECT_MAIN_TOOLBAR_ICONS,
    SECT_QT
};

/*
  Qt uses the following predefined weights: 
    Light    = 25,
    Normal   = 50,
    DemiBold = 63,
    Bold     = 75,
    Black    = 87

  ...need to categorize mid way, i.e. 0->37 = light, 38->56 = normal, etc...
*/

enum
{
    WEIGHT_NORMAL   = 38,   /* Less than this = light */
    WEIGHT_DEMIBOLD = 57,
    WEIGHT_BOLD     = 69,
    WEIGHT_BLACK    = 81
};

static const char * weightStr(int w)
{
    if(w<WEIGHT_NORMAL)
        return "light";
    else if(w<WEIGHT_DEMIBOLD)
        return "";
    else if(w<WEIGHT_BOLD)
        return "demibold";
    else if(w<WEIGHT_BLACK)
        return "bold";
    else
        return "black";
}

static const char * italicStr(int i)
{
    return i ? "Italic" : "";
}

enum
{
    RD_PALETTE           = 0x01,
    RD_FONT              = 0x02,
    RD_CONTRAST          = 0x04,
    RD_ICONS             = 0x10,
    RD_TOOLBAR_STYLE     = 0x20,
    RD_TOOLBAR_ICON_SIZE = 0x40,
    RD_BUTTON_ICONS      = 0x80
};

static char * getKdeHome()
{
    static char *kdeHome=NULL;

    if(!kdeHome)
    {
        char *env=getenv("KDEHOME");

        if(env)
            kdeHome=env;
        else
        {
            static char kdeHomeStr[QTC_MAX_FILENAME_LEN+1];

            const char *home=getHome();

            if(home && strlen(home)<(QTC_MAX_FILENAME_LEN-5))
            {
                sprintf(kdeHomeStr, "%s/.kde", home);
                kdeHome=kdeHomeStr;
            }
        }
    }

    return kdeHome;
}

static void parseQtColors(char *line)
{
    int  n=-1;
    char *l=strtok(line, "#");

    while(l)
    {
        if(8==strlen(l))
            switch(n)
            {
                case 0:
                    setRgb(&qtSettings.colors[COLOR_FOREGROUND], l);
                    break;
                case 1:
                    setRgb(&qtSettings.colors[COLOR_BUTTON], l);
                    break;
                case 5:
                    setRgb(&qtSettings.colors[COLOR_MID], l);
                    break;
                case 6:
                    setRgb(&qtSettings.colors[COLOR_TEXT], l);
                    break;
                case 8:
                    setRgb(&qtSettings.colors[COLOR_BUTTON_TEXT], l);
                    break;
                case 9:
                    setRgb(&qtSettings.colors[COLOR_BACKGROUND], l);
                    break;
                case 10:
                    setRgb(&qtSettings.colors[COLOR_WINDOW], l);
                    break;
                case 12:
                    setRgb(&qtSettings.colors[COLOR_SELECTED], l);
                    break;
                case 13:
                    setRgb(&qtSettings.colors[COLOR_TEXT_SELECTED], l);
                    break;
                default:
                    break;
            }
        else
            if(n>-1)
                break;

        n++;
        if(n>13)
            break;
        l=strtok(NULL, "#");
    }
}

static int readRc(const char *rc, int rd, Options *opts, gboolean absolute, gboolean setDefaultFont, gboolean qt4)
{
    const char *home=absolute ? NULL : getHome();
    int        found=0,
               weight=WEIGHT_NORMAL,
               italic=0,
               fixedW=0;
    float      size=DEFAULT_KDE_FONT_SIZE;
    const char *family=DEFAULT_KDE_FONT,
               *foundry="*";
    char       line[QTC_MAX_INPUT_LINE_LEN+1],
               fontLine[QTC_MAX_INPUT_LINE_LEN+1];

    if(absolute || NULL!=home)
    {
        char     fname[256];
        FILE     *f;

        if(absolute)
            strcpy(fname, rc);
        else
            sprintf(fname, "%s/%s", home, rc);

        f=fopen(fname, "r");

        if(f)
        {
            int section=SECT_NONE;

            while(found!=rd && NULL!=fgets(line, QTC_MAX_INPUT_LINE_LEN, f))
                if(line[0]=='[')
                {
                    if(qt4)
                        if(0==memcmp(line, "[Qt]", 4))
                            section=SECT_QT;
                        else
                            section=SECT_NONE;
                    else
                        if(0==memcmp(line, "[Palette]", 9))
                            section=SECT_PALETTE;
                        else if(0==memcmp(line, "[General]", 9))
                            section=SECT_GENERAL;
                        else if(0==memcmp(line, "[KDE]", 5))
                            section=SECT_KDE;
                        else if(0==memcmp(line, "[Icons]", 7))
                            section=SECT_ICONS;
                        else if(0==memcmp(line, "[Toolbar style]", 15))
                            section=SECT_TOOLBAR_STYLE;
                        else if(0==memcmp(line, "[MainToolbarIcons]", 18))
                            section=SECT_MAIN_TOOLBAR_ICONS;
                        else
                            section=SECT_NONE;
                }
#ifdef QTC_DO_ICON_MAPPING
                else if (SECT_ICONS==section && rd&RD_ICONS && !(found&RD_ICONS) &&
                         0==memcmp(line, "Theme=", 6))
                {
                    char *eq=strstr(line, "=");

                    if(eq && ++eq)
                    {
                        unsigned int len=strlen(eq);

                        if(qtSettings.icons)
                        {
                            free(qtSettings.icons);
                            qtSettings.icons=NULL;
                        }

                        qtSettings.icons=(char *)malloc(len+1);
                        strcpy(qtSettings.icons, eq);
                        if('\n'==qtSettings.icons[len-1])
                            qtSettings.icons[len-1]='\0';
                    }

                    found|=RD_ICONS;
                }
#endif
                else if (SECT_TOOLBAR_STYLE==section && rd&RD_TOOLBAR_STYLE &&
                         !(found&RD_TOOLBAR_STYLE) && 0==memcmp(line, "IconText=", 9))
                {
                    char *eq=strstr(line, "=");

                    if(eq && ++eq)
                    {
                        if(0==memcmp(eq, "IconOnly", 8))
                            qtSettings.toolbar.style=GTK_TOOLBAR_ICONS;
                        else if(0==memcmp(eq, "TextOnly", 8))
                            qtSettings.toolbar.style=GTK_TOOLBAR_TEXT;
                        else if(0==memcmp(eq, "IconTextRight", 13))
                            qtSettings.toolbar.style=GTK_TOOLBAR_BOTH_HORIZ;
                        else if(0==memcmp(eq, "IconTextBottom", 14))
                            qtSettings.toolbar.style=GTK_TOOLBAR_BOTH;
                        found|=RD_TOOLBAR_STYLE;
                    }
                }
                else if (SECT_MAIN_TOOLBAR_ICONS==section && rd&RD_TOOLBAR_ICON_SIZE &&
                         !(found&RD_TOOLBAR_ICON_SIZE) && 0==memcmp(line, "Size=", 5))
                {
                    char *eq=strstr(line, "=");

                    if(eq && ++eq)
                    {
                        int tbarsize=atoi(eq);

                        qtSettings.toolbar.size= tbarsize <=16 ? GTK_ICON_SIZE_MENU :
                                                  tbarsize <=22 ? GTK_ICON_SIZE_LARGE_TOOLBAR :
                                                  tbarsize <=32 ? GTK_ICON_SIZE_DND
                                                                : GTK_ICON_SIZE_DIALOG;
                        found|=RD_TOOLBAR_ICON_SIZE;
                    }
                }
                else if (SECT_KDE==section && rd&RD_BUTTON_ICONS && !(found&RD_BUTTON_ICONS) &&
                         0==memcmp(line, "ShowIconsOnPushButtons=", 23))
                {
                    char *eq=strstr(line, "=");

                    if(eq && ++eq)
                    {
                        qtSettings.button_icons=0==memcmp(eq, "true", 4);
                        found|=RD_BUTTON_ICONS;
                    }
                }
                else if(SECT_KDE==section && rd&RD_CONTRAST && !(found&RD_CONTRAST) &&
                        0==memcmp(line, "contrast=", 9))
                {
                    char *l=strchr(line, '=');
                    l++;
                    sscanf(l, "%i", &(opts->contrast));
                    if(opts->contrast>10 || opts->contrast<0)
                        opts->contrast=7;
                    found|=RD_CONTRAST;
                }
                else if(( (!qt4 && SECT_PALETTE==section) || (qt4 && SECT_QT==section)) && rd&RD_PALETTE && !(found&RD_PALETTE) &&
                        (qt4 ? 0==memcmp(line, "Palette\\active=", 15) : 0==memcmp(line, "active=", 7)))
                {
                    parseQtColors(line);
                    found|=RD_PALETTE;
                }
                else if (( !qt4 && SECT_GENERAL==section && rd&RD_FONT && !(found&RD_FONT) &&
                            0==memcmp(line, "font=", 5)) ||
                         ( qt4 && SECT_QT & rd&RD_FONT && !(found&RD_FONT) &&
                            0==memcmp(line, "font=\"", 6)) )
                {
                    int   n=-1,
                          rc_weight=WEIGHT_NORMAL,
                          rc_italic=0,
                          rc_fixedW=0;
                    float rc_size=12.0;
                    char  *l,
                          *rc_family=NULL,
                          *rc_foundry=NULL;

                    if(qt4) /* Convert Qt4's font= syntax to Qt3 style... */
                    {
                        int len=strlen(line),
                            i;

                        strcpy(fontLine, "font=");
                        for(i=6; i<QTC_MAX_INPUT_LINE_LEN && i<len; ++i)
                            if('\"'==line[i])
                            {
                                fontLine[i-1]='\0';
                                break;
                            }
                            else
                                fontLine[i-1]=line[i];
                    }
                    else
                        memcpy(fontLine, line, QTC_MAX_INPUT_LINE_LEN+1);
                    l=strtok(fontLine, "=");
                    found|=RD_FONT;

                    while(l)
                    {
                        switch(n)
                        {
                            case 0:  /* Family - and foundry(maybe!) (ignore X11 and XFT) */
                            {
                                char *ob=NULL,
                                     *cb=NULL;

                                if(NULL!=(ob=strchr(l, '[')) && ob!=l && NULL!=(cb=strchr(l, ']')))
                                {
                                    ob[-1]='\0';
                                    *cb='\0';
                                    rc_foundry=&(ob[1]);

                                    if(0==strcmp_i(rc_foundry, "xft") ||
                                       0==strcmp_i(rc_foundry, "x11"))
                                        rc_foundry=NULL;
                                }
                                else  /* Sometimes .kderc has "adobe-helvetica" */
                                {
                                    char *dash=NULL;

                                    if(NULL!=(dash=strchr(l, '-')))
                                    {
                                        rc_foundry=l;
                                        *dash='\0';
                                        l=++dash;
                                    }
                                }

                                rc_family=l;
                                break;
                            }
                            case 1:  /* Point size */
                                sscanf(l, "%f", &rc_size);
                                break;
                            case 4:  /* Weight */
                                sscanf(l, "%d", &rc_weight);
                                break;
                            case 5:  /* Slant */
                                sscanf(l, "%d", &rc_italic);
                                break;
                            case 8:  /* Spacing */
                                sscanf(l, "%d", &rc_fixedW);
                                break;
                            default:
                                break;
                        }

                        n++;
                        if(n>8 && NULL!=family)
                        {
                            weight=rc_weight;
                            italic=rc_italic;
                            fixedW=rc_fixedW;
                            size=rc_size;
                            family=rc_family;
                            foundry=rc_foundry;
                            break;
                        }
                        l=strtok(NULL, ",");
                    }
                }
                else if(found==rd)
                    break;

            fclose(f);
        }
    }

    if(rd&RD_PALETTE && !(found&RD_PALETTE))
    {
        strncpy(line, DEFAULT_KDE_COLORS, QTC_MAX_INPUT_LINE_LEN);
        line[QTC_MAX_INPUT_LINE_LEN]='\0';
        parseQtColors(line);
    }

    if(rd&RD_FONT && (found&RD_FONT || setDefaultFont))  /* No need to check if read in */
    {
        if(qtSettings.font)
        {
            free(qtSettings.font);
            qtSettings.font=NULL;
        }
        if(qtSettings.progressfont_rc)
        {
            free(qtSettings.progressfont_rc);
            qtSettings.progressfont_rc=NULL;
        }

        qtSettings.font=(char *)malloc(
            strlen(family) + 1 +
            strlen(weightStr(weight)) + 1 +
            strlen(italicStr(italic)) + 1 +
            20+  /* point size */ +1);

        sprintf(qtSettings.font, "%s %s %s %d",
                family,
                weightStr(weight),
                italicStr(italic),
                (int)size);

        /* Qt uses a bold font for progressbars, try to mimic this... */
        if(weight>=WEIGHT_NORMAL && weight<WEIGHT_DEMIBOLD)
        {
            static const char *constPrefix="style \"qtc-pbar-font\"{ font_name=\"";
            static const char *constSuffix="\"} class \"GtkProgress\" style \"qtc-pbar-font\"\n";

            int pweight=WEIGHT_BOLD;

            qtSettings.progressfont_rc=(char *)malloc(
                strlen(constPrefix)+
                strlen(family) + 1 +
                strlen(weightStr(pweight)) + 1 +
                strlen(italicStr(italic)) + 1 +
                20+  /* point size */
                strlen(constSuffix) +1);

            sprintf(qtSettings.progressfont_rc, "%s%s %s %s %d%s",
                    constPrefix,
                    family,
                    weightStr(pweight),
                    italicStr(italic),
                    (int)size,
                    constSuffix);
        }
#ifdef QTC_DEBUG
printf("REQUEST FONT: %s\n", qtSettings.font);
#endif
    }

#ifdef QTC_DO_ICON_MAPPING
    if(rd&RD_ICONS && !qtSettings.icons)
    {
        qtSettings.icons=(char *)malloc(strlen(defaultIcons())+1);
        strcpy(qtSettings.icons, defaultIcons());
    }
#endif
    if(rd&RD_TOOLBAR_STYLE && !(found&RD_TOOLBAR_STYLE))
        qtSettings.toolbar.style=GTK_TOOLBAR_ICONS;
    if(rd&RD_TOOLBAR_ICON_SIZE && !(found&RD_TOOLBAR_ICON_SIZE))
        qtSettings.toolbar.size=GTK_ICON_SIZE_LARGE_TOOLBAR;
    if(rd&RD_BUTTON_ICONS && !(found&RD_BUTTON_ICONS))
        qtSettings.button_icons=TRUE;

    return found;
}

static int qt_refs=0;

#define MAX_APP_NAME_LEN 32
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <dirent.h>

static const char * kdeGlobals()
{
    static char kg[QTC_MAX_FILENAME_LEN+1]={'\0'};

    char *kdehome=getKdeHome();

    if(kdehome && strlen(kdehome)<(QTC_MAX_FILENAME_LEN-strlen("/share/config/kdeglobals")))
        sprintf(kg, "%s/share/config/kdeglobals", kdehome);

    return kg;
}

#ifdef QTC_DO_ICON_MAPPING
#define HICOLOR_ICONS           "hicolor"
#define HICOLOR_LEN             7
#define ICON_FOLDER             "/share/icons/"
#define ICON_FOLDER_SLEN        13
#define DEFAULT_ICON_PREFIX     "/usr/share/icons"
#define DEFAULT_ICON_PREFIX_LEN 16

static char * getIconPath()
{

    static char *path=NULL;
    char        *kdeHome=getKdeHome();
    const char  *kdePrefix=kdeIconsPrefix(),
                *defIcons=defaultIcons();
    gboolean    nonDefIcons=qtSettings.icons && strcmp(qtSettings.icons, defIcons);
    int         len=strlen("pixmap_path \""),
                kdeHomeLen=kdeHome ? strlen(kdeHome) : 0,
                kdeIconPrefixLen=kdePrefix ? strlen(kdePrefix) : 0,
                iconLen=qtSettings.icons ? strlen(qtSettings.icons) : 0,
                defIconsLen=defIcons ? strlen(defIcons) : 0;
    gboolean    addDefaultPrefix=!kdeIconPrefixLen || strcmp(kdePrefix, DEFAULT_ICON_PREFIX);

    if(nonDefIcons)
    {
        if(kdeHome)
        {
            len+=kdeHomeLen;
            len+=ICON_FOLDER_SLEN;
            len+=iconLen;
            len++;
        }
        if(kdeIconPrefixLen)
        {
            len+=kdeIconPrefixLen;
            len++;
            len+=iconLen;
            len++;
        }
        if(addDefaultPrefix)
        {
            len+=DEFAULT_ICON_PREFIX_LEN;
            len++;
            len+=iconLen;
            len++;
        }
    }

    if(kdeHome)
    {
        len+=kdeHomeLen;
        len+=ICON_FOLDER_SLEN;
        len+=defIconsLen;
        len++;
    }
    if(kdeIconPrefixLen)
    {
        len+=kdeIconPrefixLen;
        len++;
        len+=defIconsLen;
        len++;
    }
    if(addDefaultPrefix)
    {
        len+=DEFAULT_ICON_PREFIX_LEN;
        len++;
        len+=defIconsLen;
        len++;
    }
    if(kdeHome)
    {
        len+=kdeHomeLen;
        len+=ICON_FOLDER_SLEN;
        len+=HICOLOR_LEN;
        len++;
    }
    if(kdeIconPrefixLen)
    {
        len+=kdeIconPrefixLen;
        len++;
        len+=HICOLOR_LEN;
        len++;
    }
    if(addDefaultPrefix)
    {
        len+=DEFAULT_ICON_PREFIX_LEN;
        len++;
        len+=HICOLOR_LEN;
        len++;
    }
    len++;

    if(path && len!=(strlen(path)+1))
        free(path);

    if(!path)
        path=(char *)malloc(len+1);

    strcpy(path, "pixmap_path \"");

    if(nonDefIcons)
    {
        if(kdeHome)
        {
            strcat(path, kdeHome);
            strcat(path, ICON_FOLDER);
            strcat(path, qtSettings.icons);
            strcat(path, ":");
        }
        if(kdeIconPrefixLen)
        {
            strcat(path, kdePrefix);
            strcat(path, "/");
            strcat(path, qtSettings.icons);
            strcat(path, ":");
        }
        if(addDefaultPrefix)
        {
            strcat(path, DEFAULT_ICON_PREFIX);
            strcat(path, "/");
            strcat(path, qtSettings.icons);
            strcat(path, ":");
        }
    }

    if(kdeHome)
    {
        strcat(path, kdeHome);
        strcat(path, ICON_FOLDER);
        strcat(path, defIcons);
        strcat(path, ":");
    }

    if(kdeIconPrefixLen)
    {
        strcat(path, kdePrefix);
        strcat(path, "/");
        strcat(path, defIcons);
        strcat(path, ":");
    }

    if(addDefaultPrefix)
    {
        strcat(path, DEFAULT_ICON_PREFIX);
        strcat(path, "/");
        strcat(path, defIcons);
        strcat(path, ":");
    }

    if(kdeHome)
    {
        strcat(path, kdeHome);
        strcat(path, ICON_FOLDER);
        strcat(path, HICOLOR_ICONS);
        strcat(path, ":");
    }

    if(kdeIconPrefixLen)
    {
        strcat(path, kdePrefix);
        strcat(path, "/");
        strcat(path, HICOLOR_ICONS);
        strcat(path, ":");
    }

    if(addDefaultPrefix)
    {
        strcat(path, DEFAULT_ICON_PREFIX);
        strcat(path, "/");
        strcat(path, HICOLOR_ICONS);
    }

    strcat(path, "\"");

    return path;
}
#endif

#define GIMP_PLUGIN "gimpplugin"

static char * getAppNameFromPid(int pid)
{
/* CPD: There must be an easier way than this? */
    static char app_name[MAX_APP_NAME_LEN+1]="";

    int  procFile=-1;
    char cmdline[MAX_LINE_LEN+1];

    sprintf(cmdline, "/proc/%d/cmdline", pid);

    if(-1!=(procFile=open(cmdline, O_RDONLY)))
    {
        if(read(procFile, cmdline, MAX_LINE_LEN)>2)
        {
            int      len=strlen(cmdline),
                     pos=0;
            gboolean found_slash=FALSE;

            for(pos=len-1; pos>=0 && cmdline[pos] && !found_slash; --pos)
                if('/'==cmdline[pos])
                {
                    pos++;
                    found_slash=TRUE;
                }

            if(!found_slash)
                pos=0;  /* Perhaps no / */
            if(pos>=0)
            {
                if(NULL!=strstr(cmdline, "gimp/2.0/plug-ins"))
                    strcpy(app_name, GIMP_PLUGIN);
                else
                {
                    strncpy(app_name, &cmdline[pos ? pos+1 : 0], MAX_APP_NAME_LEN);
                    app_name[MAX_APP_NAME_LEN]='\0';
                }
            }
        }
        close(procFile);
    }

    return app_name;
}

static char * getAppName()
{
    char *name=getAppNameFromPid(getpid());

    if(0==strcmp(name, "perl") || 0==strcmp(name, "python"))
    {
        name=getAppNameFromPid(getppid());
        if(!name)
            return "scriptedapp";
        else if(name==strstr(name, "gimp"))
            return GIMP_PLUGIN;
    }

    return name;
}

#ifdef QTC_MODIFY_MOZILLA

#define MAX_CSS_HOME     256
#define CSS_DEFAULT      ".default"
#define CSS_DEFAULT_ALT  "default."
#define USER_CHROME_DIR  "/chrome"
#define USER_CHROME_FILE "userChrome.css"
#define USER_CHROME_CSS  USER_CHROME_DIR"/"USER_CHROME_FILE
#define MAX_DEFAULT_NAME 16+strlen(CSS_DEFAULT)+strlen(USER_CHROME_CSS)

#define QTC_GUARD_STR "Added by QtCurve -- do not remove"
#define MENU_TEXT_STR "menubar > menu { color: HighlightText !important; } menubar > menu[_moz-menuactive=\"true\"] "\
                      "{ background-color : HighlightText !important; color: HighlightText !important; } "\
                      "/* "QTC_GUARD_STR" */\n"

#define CSS_FILE_STR  "@import url(\"file://"QTC_MOZILLA_DIR"/QtCurve.css\"); /* "QTC_GUARD_STR" */\n"

static void processUserChromeCss(char *file, gboolean add_menu_colors)
{
    FILE        *f=fopen(file, "r");
    char        *contents=NULL;
    gboolean    add_css=TRUE,
                remove_menu_colors=FALSE;
    struct stat st;
    size_t      file_size=0,
                new_size=0;

    if(f)
    {
        if(0==fstat(fileno(f), &st))
        {
            file_size = st.st_size;
            new_size=file_size+strlen(MENU_TEXT_STR)+strlen(CSS_FILE_STR)+3;
            contents=(char *)malloc(new_size);

            if(contents)
            {
                char  *line=NULL;
                size_t len=0;

                contents[0]='\0';
                while(-1!=getline(&line, &len, f))
                {
                    gboolean write_line=TRUE;

                    if(0==strcmp(line, CSS_FILE_STR))
                        add_css=FALSE;
                    else
                    {
                        if(0==strcmp(line, MENU_TEXT_STR))
                        {
                            if (add_menu_colors)
                                add_menu_colors=FALSE;
                            else
                            {
                                remove_menu_colors=TRUE;
                                write_line=FALSE;
                            }
                        }
                    }
                    if(write_line)
                        strcat(contents, line);
                }
                if (line)
                     free(line);
            }
        }

        fclose(f);
    }

    if(!contents || add_css || add_menu_colors)
    {
        if(!contents)
        {
            new_size=strlen(MENU_TEXT_STR)+strlen(CSS_FILE_STR)+3;

            contents=(char *)malloc(new_size);
            if(contents)
                contents[0]='\0';
        }

        if(contents)
        {
            if(add_css)  /* CSS needs to be on 1st line */
            {
                char *css_contents=(char *)malloc(new_size);

                if(css_contents)
                {
                    css_contents[0]='\0';
                    strcat(css_contents, CSS_FILE_STR);
                    strcat(css_contents, contents);
                    free(contents);
                    contents=css_contents;
                }
            }
            if(add_menu_colors)  /* This can be on last line */
            {
                int len=strlen(contents);

                if(len && contents[len-1]!='\n')
                    strcat(contents, "\n");
                strcat(contents, MENU_TEXT_STR);
            }
        }
    }

    if(contents && (add_css || add_menu_colors || remove_menu_colors))
    {
        f=fopen(file, "w");

        if(f)
        {
            fputs(contents, f);
            fclose(f);
        }
        free(contents);
    }
}

static void processMozillaApp(gboolean add_menu_colors, char *app, gboolean under_moz)
{
    const char *home=getHome();

    if(home && (strlen(home)+strlen(app)+10+MAX_DEFAULT_NAME)<MAX_CSS_HOME)     /* 10 for .mozilla/<app>/ */
    {
        char cssHome[MAX_CSS_HOME+1];
        DIR  *dir=NULL;

        sprintf(cssHome, under_moz ? "%s/.mozilla/%s/" : "%s/.%s/", home, app);

        if((dir=opendir(cssHome)))
        {
             struct dirent *dir_ent=NULL;

             for (dir_ent=readdir(dir); dir_ent; dir_ent=readdir(dir))
             {
                 char *str=NULL;

                 if(((str=strstr(dir_ent->d_name, CSS_DEFAULT)) && str>=dir_ent->d_name &&
                    '\0'==str[strlen(CSS_DEFAULT)]) ||
                    ((str=strstr(dir_ent->d_name, CSS_DEFAULT_ALT)) && str==dir_ent->d_name &&
                    '\0'!=str[strlen(CSS_DEFAULT_ALT)]))
                 {
                     char        sub[MAX_CSS_HOME];
                     struct stat statbuf;

#ifdef QTC_MODIFY_MOZILLA_USER_JS
                     /* Add custom user.js file */
                     sprintf(sub, "%s%s/user.js", cssHome, dir_ent->d_name);
                     if(-1==lstat(sub, &statbuf))
                     {
                         FILE *in=NULL;

                         sprintf(sub, QTC_MOZILLA_DIR"/%s-user.js", app);

                         if((in=fopen(sub, "r")))
                         {
                             FILE *out=fopen(sub, "w");

                             if(out)
                             {
                                 char ch;

                                 while((ch=fgetc(in))!=EOF)
                                     fputc(ch, out);
                                 fclose(out);
                             }
                             fclose(in);
                         }
                     }
#endif

                     /* Now do userChrome.css */
                     sprintf(sub, "%s%s%s/", cssHome, dir_ent->d_name, USER_CHROME_DIR);

                     if(-1!=lstat(sub, &statbuf) && S_ISDIR(statbuf.st_mode))
                     {
                         strcat(sub, USER_CHROME_FILE);
                         processUserChromeCss(sub, add_menu_colors);
                     }
                 }
             }

             closedir(dir);
        }
    }
}
#endif

#ifdef QTC_ADD_EVENT_FILTER
/* ... Taken from Qt engine ... */

#include <gdk/gdkx.h>
#include <X11/X.h>
#include <X11/Xlib.h>
static Atom kipcCommAtom;
static Atom desktopWindowAtom;
static Atom knwmCommAtom;

static gboolean init();
static void qtExit();
static GdkFilterReturn qtGdkEventFilter(GdkXEvent *xevent, GdkEvent *gevent, gpointer data)
{
    XEvent *event =(XEvent*) xevent;

    if ( (ClientMessage==event->type && kipcCommAtom==event->xclient.message_type) ||
         (PropertyNotify==event->type && knwmCommAtom==event->xclient.message_type) )
    {
        qt_refs=1;
        qtExit();
        gtk_rc_reset_styles(gtk_settings_get_default());

        return GDK_FILTER_REMOVE;
    }

    return GDK_FILTER_CONTINUE;
}

static void qtcAddEventFilter() /* GdkWindow *widget) */
{
    static int done=0;

    if(!done)
    {
        /* Create a new window, and set the KDE_DESKTOP_WINDOW property on it
           This window will then receive events from KDE when the style changes */

        GtkWindow *top=gtk_window_new(GTK_WINDOW_TOPLEVEL);

        if(top && GTK_IS_WIDGET(top))
        {
            long data = 1;

            gtk_window_iconify(GTK_WINDOW(top));
            gtk_widget_show(top);
            gtk_window_set_skip_taskbar_hint(GTK_WINDOW(top), TRUE);
            gtk_window_set_skip_pager_hint(GTK_WINDOW(top), TRUE);

            /* Get KDE related atoms from the X server */
            kipcCommAtom = XInternAtom(gdk_x11_get_default_xdisplay(), "KIPC_COMM_ATOM", FALSE);
            knwmCommAtom = XInternAtom(gdk_x11_get_default_xdisplay(), "_KDE_NET_WM_FRAME_STRUT", FALSE);
            desktopWindowAtom = XInternAtom(gdk_x11_get_default_xdisplay(), "KDE_DESKTOP_WINDOW", FALSE);

            XChangeProperty(gdk_x11_get_default_xdisplay(), GDK_WINDOW_XID(GTK_WIDGET(top)->window),
                            desktopWindowAtom, desktopWindowAtom, 32, PropModeReplace,
                            (unsigned char *)&data, 1);
            /* This filter will intercept those events */
            gdk_window_add_filter(GTK_WIDGET(top)->window, qtGdkEventFilter, 0);
            gtk_widget_hide(top);
            done=1;
        }
    }
}

/* ... */

#endif

#define toGtkRcColor(col) \
    (((double)col)/65535.0)

#define toQtColor(col) \
    ((int)((((double)col)/256.0)+0.5))

static gboolean qtInit(Options *opts)
{
    if(0==qt_refs++)
    {
        static int lastRead=0;

        int now=time(NULL);

        qtSettings.app=GTK_APP_UNKNOWN;

        if(abs(now-lastRead)>1)
        {
            char        *app=NULL,
                        *path=NULL,
#ifdef HAVE_LOCALE_H
                        *locale=NULL,
#endif
                        *tmpStr=NULL;
            GtkSettings *settings=NULL;

#ifdef QTC_DO_ICON_MAPPING
            qtSettings.icons=NULL;
#endif
            qtSettings.progressfont_rc=NULL;
            qtSettings.font=NULL;
            lastRead=now;

            defaultSettings(opts);

            if(useQt3Settings())
            {
                qtSettings.qt4=FALSE;
                readRc("/etc/qt/qtrc", RD_PALETTE|RD_FONT|RD_CONTRAST, opts, TRUE, FALSE, FALSE);
                readRc("/etc/qt3/qtrc", RD_PALETTE|RD_FONT|RD_CONTRAST, opts, TRUE, FALSE, FALSE);
                readRc(".qt/qtrc", RD_PALETTE|RD_FONT|RD_CONTRAST, opts, FALSE, TRUE, FALSE);
            }
            else
            {
                static const char *constQt4ConfFile="Trolltech.conf";

                char *confFile=(char *)malloc(strlen(xdgConfigFolder())+strlen(constQt4ConfFile)+2);

                sprintf(confFile, "%s/%s", xdgConfigFolder(), constQt4ConfFile);
                readRc(confFile, RD_PALETTE|RD_FONT|RD_CONTRAST, opts, TRUE, TRUE, TRUE);
                free(confFile);
                qtSettings.qt4=TRUE;
            }

            readConfig(NULL, opts, opts);

            /* Check if we're firefox... */
            if((app=getAppName()))
            {
                if((0==strcmp(app, "firefox-bin") || 0==strcmp(app, "thunderbird-bin") ||
                    0==strcmp(app, "swiftfox-bin") || 0==strcmp(app, "mozilla-thunderbird-bin")))
                {
#ifdef QTC_MODIFY_MOZILLA
                    GdkColor *menu_col=SHADE_CUSTOM==opts->shadeMenubars
                                        ? &opts->customMenubarsColor
                                        : &qtSettings.colors[COLOR_SELECTED];
                    gboolean add_menu_colors=FALSE;

                    if(SHADE_BLEND_SELECTED==opts->shadeMenubars || (SHADE_CUSTOM==opts->shadeMenubars &&
                                                               TOO_DARK(*menu_col) ))
                        add_menu_colors=TRUE;

                    if(0==strcmp(app, "firefox-bin") || 0==strcmp(app, "swiftfox-bin"))
                        processMozillaApp(add_menu_colors, "firefox", TRUE);
                    else if(0==strcmp(app, "thunderbird-bin"))
                        processMozillaApp(add_menu_colors, "thunderbird", FALSE);
                    else if(0==strcmp(app, "mozilla-thunderbird-bin"))
                        processMozillaApp(add_menu_colors, "mozilla-thunderbird", FALSE);
#endif
                    qtSettings.app=GTK_APP_MOZILLA;
                }
                else if(0==strcmp(app, "soffice.bin"))
                    qtSettings.app=GTK_APP_OPEN_OFFICE;
                else if(0==strcmp(app, "vmplayer"))
                    qtSettings.app=GTK_APP_VMPLAYER;
                else if(0==strcmp(app, GIMP_PLUGIN))
                    qtSettings.app=GTK_APP_GIMP_PLUGIN;
                else if(app==strstr(app, "gimp"))
                    qtSettings.app=GTK_APP_GIMP;
                else if(0==strcmp(app, "java"))
                    qtSettings.app=GTK_APP_JAVA;
                /*else if(app==strstr(app, "gaim"))
                    qtSettings.app=GTK_APP_GAIM;*/
            }

            /* Eclipse sets a application name, so if this is set then we're not a Swing java app */
            if(GTK_APP_JAVA==qtSettings.app && g_get_application_name() && 0!=strcmp(g_get_application_name(), "<unknown>"))
                qtSettings.app=GTK_APP_JAVA_SWT;

            if(GTK_APP_MOZILLA==qtSettings.app || GTK_APP_JAVA==qtSettings.app)
            {
                /* KDE's "apply colors to non-KDE apps" messes up firefox! so need to fix this! */
                char *kdehome=getKdeHome();

                if(kdehome)
                {
                    static const char *constCfgDir="/share/config/";
                    static const char *constCfgFile="gtkrc-2.0";

                    tmpStr=(char *)realloc(tmpStr, strlen(kdehome)+strlen(constCfgDir)
                                                   +strlen(constCfgFile)+1);
                    sprintf(tmpStr, "%s%s%s", kdehome, constCfgDir, constCfgFile);

                    char *gtkEnv=getenv("GTK2_RC_FILES");

                    /* its in the users environment... */
                    if(gtkEnv && NULL!=strstr(gtkEnv, tmpStr))
                    {
                        struct stat statbuf;

                        /* Check if KDE's gtkrc file exists */
                        if(-1!=lstat(tmpStr, &statbuf))
                        {
                            static const int constVersion=1;
                            static const char *constQtcCfgFile="gtkrc-qtc";

                            tmpStr=(char *)realloc(tmpStr, strlen(kdehome)+strlen(constCfgDir)
                                                           +strlen(constQtcCfgFile)+1);
                            sprintf(tmpStr, "%s%s%s", kdehome, constCfgDir, constQtcCfgFile);

                            FILE     *f=fopen(tmpStr, "r");
                            gboolean write_file=TRUE;
                            char     version[10];

                            sprintf(version, "#%02d%02X%02X%02X",
                                             constVersion,
                                             toQtColor(qtSettings.colors[COLOR_TEXT_SELECTED].red),
                                             toQtColor(qtSettings.colors[COLOR_TEXT_SELECTED].green),
                                             toQtColor(qtSettings.colors[COLOR_TEXT_SELECTED].blue));

                            /* Read any previous file, and check previous color setting! */
                            if(f)
                            {
                                char line[10];

                                write_file=NULL==fgets(line, 9, f) || memcmp(version, line, 9);
                                fclose(f);
                            }

                            /* Either it didnt exist, or the colors are different */
                            if(write_file && (f=fopen(tmpStr, "w")))
                            {
#ifdef HAVE_LOCALE_H
                               /* We need to switch to the C locale in order
                                  to have decimal separator recognized by gtkrc engine */
                                if(!locale)
                                    locale = setlocale(LC_NUMERIC, "C");
#endif

                                fprintf(f, "%s\n"
                                           "# Fix for firefox and KDE's \"apply color to non-KDE"
                                           " apps\" setting\n"
                                           "style \"QtCMnuFix\"\n"
                                           "{ fg[ACTIVE]={ %5.3f, %5.3f, %5.3f }"
                                           " fg[PRELIGHT]={ %5.3f, %5.3f, %5.3f } }\n"
                                           "widget_class \"*Menu*\" style \"QtCMnuFix\"",
                                           version,
                                           toGtkRcColor(qtSettings.colors[COLOR_TEXT_SELECTED].red),
                                           toGtkRcColor(qtSettings.colors[COLOR_TEXT_SELECTED].green),
                                           toGtkRcColor(qtSettings.colors[COLOR_TEXT_SELECTED].blue),
                                           toGtkRcColor(qtSettings.colors[COLOR_TEXT_SELECTED].red),
                                           toGtkRcColor(qtSettings.colors[COLOR_TEXT_SELECTED].green),
                                           toGtkRcColor(qtSettings.colors[COLOR_TEXT_SELECTED].blue));
                                fclose(f);
                            }

                            /* Now get gtk to read this file *after* its other gtkrc files - this
                               allows us to undo the KDE settings! */
                            gtk_rc_add_default_file(tmpStr);
                        }
                    }
                }
            }
            else if(GTK_APP_VMPLAYER==qtSettings.app)
                opts->shadeMenubars=SHADE_NONE;

            readRc(kdeGlobals(), RD_ICONS|RD_TOOLBAR_STYLE|RD_TOOLBAR_ICON_SIZE|RD_BUTTON_ICONS,
                    opts, TRUE, FALSE, FALSE);

            /* Tear off menu items dont seem to draw they're background, and the default background
               is drawn :-(  Fix/hack this by making that background the correct color */
            if(opts->lighterPopupMenuBgnd)
            {
                static const char *format="style \"QtCLMnu\" "
                                          "{bg[NORMAL]={%5.3f, %5.3f, %5.3f}} "
                                          "class \"GtkMenu\" style \"QtCLMnu\"";
                tmpStr=(char *)realloc(tmpStr, strlen(format)+32);

                if(tmpStr)
                {
                    GdkColor col;

#ifdef HAVE_LOCALE_H
                   /* We need to switch to the C locale in order
                      to have decimal separator recognized by gtkrc engine */
                    if(!locale)
                        locale = setlocale(LC_NUMERIC, "C");
#endif
                    shade(&qtSettings.colors[COLOR_WINDOW], &col, POPUPMENU_LIGHT_FACTOR);
                    sprintf(tmpStr, format, toGtkRcColor(col.red), toGtkRcColor(col.green), toGtkRcColor(col.blue));
                    gtk_rc_parse_string(tmpStr);
                }
            }

            if((settings=gtk_settings_get_default()))
            {
                GtkSettingsValue svalue;

                if(qtSettings.font)
                    g_object_set(settings, "gtk-font-name", qtSettings.font, 0);

                g_object_set(settings, "gtk-icon-sizes",
                                       "gtk-large-toolbar=22,22:"
                                       "gtk-button=16,16:"
                                       "gtk-dnd=32,32:"
                                       "gtk-menu=16,16:"
                                       "gtk-small-toolbar=16,16:"
                                       "gtk-dialog=22,22", 0);

                svalue.origin="KDE-Settings";
                svalue.value.g_type=G_TYPE_INVALID;
                g_value_init(&svalue.value, G_TYPE_LONG);
                g_value_set_long(&svalue.value, qtSettings.toolbar.style);
                gtk_settings_set_property_value(settings, "gtk-toolbar-style", &svalue);
#ifdef QTC_DEBUG
                printf("gtk-toolbar-style %d\n", qtSettings.toolbar.style);
#endif
                g_value_set_long(&svalue.value, qtSettings.toolbar.size);
                gtk_settings_set_property_value(settings, "gtk-toolbar-icon-size", &svalue);
#ifdef QTC_DEBUG
                printf("gtk-toolbar-icon-size %d\n", qtSettings.toolbar.size);
#endif
                if(NULL==gtk_check_version(2, 4, 0)) /* The following settings only apply for GTK>=2.4.0 */
                {
                    g_value_set_long(&svalue.value, qtSettings.button_icons);
#ifdef QTC_DEBUG
                    printf("gtk-button-images %d\n", qtSettings.button_icons);
#endif
                    gtk_settings_set_property_value(settings, "gtk-button-images", &svalue);

#if 0
                    if(opts->drawStatusBarFrames)
                        gtk_rc_parse_string("style \"QtCStBar\""
                                            "{ GtkStatusbar::shadow-type = 1 }" /*GtkStatusbar::has-resize-grip = FALSE }" */
                                            "class \"GtkStatusbar\" style"
                                            " \"QtCStBar\"\n");
                    else
                        gtk_rc_parse_string("style \"QtCSBar\""
                                            "{ GtkStatusbar::shadow-type = 0 }" /*GtkStatusbar::has-resize-grip = FALSE }" */
                                            "class \"GtkStatusbar\" style"
                                            " \"QtCSBar\"\n");
#endif
                }
                if(NULL==gtk_check_version(2, 6, 0)) /* The following settings only apply for GTK>=2.6.0 */
                    g_object_set(settings, "gtk-alternative-button-order", TRUE, NULL);
            }

#ifdef QTC_DO_ICON_MAPPING
            if((path=getIconPath()))
            {
                const char *iconMapFile=qtSettings.qt4 ? GTK_THEME_DIR"/gtkrc-kde4" : GTK_THEME_DIR"/gtkrc-kde3";

                struct stat info;

                if(0==lstat(iconMapFile, &info) && ((info.st_mode&S_IFMT)==S_IFREG || (info.st_mode&S_IFMT)==S_IFLNK))
                {
                    gtk_rc_add_default_file(iconMapFile);
                    gtk_rc_parse_string(path);
                }
            }
#endif

            if(qtSettings.progressfont_rc)
            {
                gtk_rc_parse_string(qtSettings.progressfont_rc);
                free(qtSettings.progressfont_rc);
                qtSettings.progressfont_rc=NULL;
            }

            if(opts->thinnerMenuItems)
                gtk_rc_parse_string("style \"QtCMi\" {xthickness = 1 ythickness = 2 } "
                                    "class \"*MenuItem\" style \"QtCMi\"");

            if(GTK_APP_MOZILLA==qtSettings.app || GTK_APP_JAVA==qtSettings.app)
                opts->scrollbarType=SCROLLBAR_WINDOWS;
            else
            {
                static const char *constSbStrFormat="style \"QtCSBt\" "
                                                    "{ GtkScrollbar::has-backward-stepper=%d "
                                                      "GtkScrollbar::has-forward-stepper=%d "
                                                      "GtkScrollbar::has-secondary-backward-stepper=%d "
                                                      "GtkScrollbar::has-secondary-forward-stepper=%d } "
                                                    "class \"*\" style \"QtCSBt\"";
                tmpStr=(char *)realloc(tmpStr, strlen(constSbStrFormat)+1);

                if(GTK_APP_OPEN_OFFICE==qtSettings.app)
                {
                    if(SCROLLBAR_NEXT==opts->scrollbarType)
                        opts->scrollbarType=SCROLLBAR_KDE;
                    else if(SCROLLBAR_NONE==opts->scrollbarType)
                        opts->scrollbarType=SCROLLBAR_WINDOWS;
                }

                switch(opts->scrollbarType)
                {
                    default:
                    case SCROLLBAR_KDE:
                        sprintf(tmpStr, constSbStrFormat, 1, 1, 1, 0);
                        break;
                    case SCROLLBAR_WINDOWS:
                        sprintf(tmpStr, constSbStrFormat, 1, 1, 0, 0);
                        break;
                    case SCROLLBAR_PLATINUM:
                        sprintf(tmpStr, constSbStrFormat, 0, 1, 1, 0);
                        break;
                    case SCROLLBAR_NEXT:
                        sprintf(tmpStr, constSbStrFormat, 1, 0, 0, 1);
                        break;
                    case SCROLLBAR_NONE:
                        sprintf(tmpStr, constSbStrFormat, 0, 0, 0, 0);
                        break;
                }

                gtk_rc_parse_string(tmpStr);
            }

            /* Set cursor colours... */
            {
                static const char *constStrFormat="style \"QtCCrsr\" "
                                                    "{ GtkWidget::cursor-color=\"#%02X%02X%02X\" "
                                                      "GtkWidget::secondary-cursor-color=\"#%02X%02X%02X\" } "
                                                    "class \"*\" style \"QtCCrsr\"";
                tmpStr=(char *)realloc(tmpStr, strlen(constStrFormat)+1);

                sprintf(tmpStr, constStrFormat, qtSettings.colors[COLOR_TEXT].red>>8,
                                                qtSettings.colors[COLOR_TEXT].green>>8,
                                                qtSettings.colors[COLOR_TEXT].blue>>8,
                                                qtSettings.colors[COLOR_TEXT].red>>8,
                                                qtSettings.colors[COLOR_TEXT].green>>8,
                                                qtSettings.colors[COLOR_TEXT].blue>>8);
                gtk_rc_parse_string(tmpStr);
            }
            if(tmpStr)
                free(tmpStr);

#ifdef HAVE_LOCALE_H
            if(locale)
                setlocale(LC_NUMERIC, locale);
#endif
        }
        return TRUE;
    }
    return FALSE;
}

static void qtExit()
{
    qt_refs--;

    if(0==qt_refs)
    {
        if(qtSettings.font)
            free(qtSettings.font);
        qtSettings.font=NULL;
#ifdef QTC_DO_ICON_MAPPING
        if(qtSettings.icons)
            free(qtSettings.icons);
#endif
        qtSettings.font=NULL;
    }
}

#define SET_COLOR(st, rc, itm, ITEM, state, QTP_COL) \
    st->itm[state]=rc->color_flags[state]&ITEM ? rc->itm[state] : qtSettings.colors[QTP_COL];

static void qtSetColors(GtkStyle *style, GtkRcStyle *rc_style, Options *opts)
{
    SET_COLOR(style, rc_style, bg, GTK_RC_BG, GTK_STATE_NORMAL, COLOR_WINDOW)
    SET_COLOR(style, rc_style, bg, GTK_RC_BG, GTK_STATE_SELECTED, COLOR_SELECTED)
    SET_COLOR(style, rc_style, bg, GTK_RC_BG, GTK_STATE_INSENSITIVE, COLOR_WINDOW)
    SET_COLOR(style, rc_style, bg, GTK_RC_BG, GTK_STATE_ACTIVE, COLOR_MID)
    SET_COLOR(style, rc_style, bg, GTK_RC_BG, GTK_STATE_PRELIGHT, COLOR_WINDOW)

    SET_COLOR(style, rc_style, base, GTK_RC_BASE, GTK_STATE_NORMAL, COLOR_BACKGROUND)
    SET_COLOR(style, rc_style, base, GTK_RC_BASE, GTK_STATE_SELECTED, COLOR_SELECTED)
    SET_COLOR(style, rc_style, base, GTK_RC_BASE, GTK_STATE_INSENSITIVE, COLOR_WINDOW)
    SET_COLOR(style, rc_style, base, GTK_RC_BASE, GTK_STATE_ACTIVE, COLOR_SELECTED)
    SET_COLOR(style, rc_style, base, GTK_RC_BASE, GTK_STATE_PRELIGHT, COLOR_SELECTED)

    SET_COLOR(style, rc_style, text, GTK_RC_TEXT, GTK_STATE_NORMAL, COLOR_TEXT)
    SET_COLOR(style, rc_style, text, GTK_RC_TEXT, GTK_STATE_SELECTED, COLOR_TEXT_SELECTED)
    SET_COLOR(style, rc_style, text, GTK_RC_TEXT, GTK_STATE_INSENSITIVE, COLOR_MID)
    SET_COLOR(style, rc_style, text, GTK_RC_TEXT, GTK_STATE_ACTIVE, COLOR_TEXT_SELECTED)
    SET_COLOR(style, rc_style, text, GTK_RC_TEXT, GTK_STATE_PRELIGHT, COLOR_TEXT)

    SET_COLOR(style, rc_style, fg, GTK_RC_FG, GTK_STATE_NORMAL, COLOR_FOREGROUND)
    SET_COLOR(style, rc_style, fg, GTK_RC_FG, GTK_STATE_SELECTED, COLOR_TEXT_SELECTED)
    SET_COLOR(style, rc_style, fg, GTK_RC_FG, GTK_STATE_INSENSITIVE, COLOR_MID)
    SET_COLOR(style, rc_style, fg, GTK_RC_FG, GTK_STATE_ACTIVE, COLOR_FOREGROUND)
    SET_COLOR(style, rc_style, fg, GTK_RC_FG, GTK_STATE_PRELIGHT, COLOR_FOREGROUND)
}

static void qtSetFont(GtkRcStyle *rc_style)
{
    if(qtSettings.font)
    {
        if (rc_style->font_desc)
            pango_font_description_free (rc_style->font_desc);

        rc_style->font_desc = pango_font_description_from_string (qtSettings.font);
    }
}
