/*
 *     gtkatlantic - the gtk+ monopd client, enjoy network monopoly games
 *
 *
 *  Copyright (C) 2002-2003 Rochet Sylvain
 *
 *  gtkatlantic is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <gtk/gtk.h>
#include <string.h>

#include "config.h"

#include "global.h"
#include "callback.h"
#include "client.h"
#include "game.h"
#include "interface.h"
#include "trade.h"
#include "load.h"
#include "theme.h"
#include "display.h"

#include "engine.h"


/* **** Exit game */
void Callback_Exit(GtkWidget* menuitem)  {

	intro_free_pngs();
	game_free_pngs();
	gtk_exit(0);
}


/* **** Show Configuration Panel */
void Callback_ShowConfiguration(GtkWidget* menuitem)  {

	interface_create_config_panel();
}


/* **** Show all monopd servers */
void Callback_ShowMetaServer(GtkWidget* button)  {

	create_connection_metaserver(CONNECT_TYPE_META_GETLIST);
}


/* **** Get games from all monopd server */
void Callback_RefreshGameMetaServer(GtkWidget* button)  {

	create_connection_metaserver(CONNECT_TYPE_META_GETGAME);
}


/* **** Get games from another server */
void Callback_GetgamesButton(GtkWidget* button)  {

	guint32 port;
	gchar *host;

	host = g_strstrip( g_strdup( gtk_entry_get_text( GTK_ENTRY( gtk_object_get_data(GTK_OBJECT(button), "host") ) ) ) );
	port = atoi( gtk_entry_get_text( GTK_ENTRY( gtk_object_get_data(GTK_OBJECT(button), "port") ) ) );

	create_connection_get_games(host, port);
	g_free(host);
}


/* get games: event "row_select" */
void Callback_GetGame_Select(GtkWidget* clist, gint row)  {

	gchar **ptrtxt, *texts[10];
	guint32 i;


	if(global->mode_gameclist != GAMECLIST_MODE_GAMELIST)  return;

	ptrtxt = g_malloc(8192);

	for(i = 0 ; i < 8 ; i++)  {

		memset(ptrtxt, 0, 8192);
		gtk_clist_get_text(GTK_CLIST(clist), row, i, ptrtxt);
		texts[i] = g_strdup(*ptrtxt);
	}
	g_free(ptrtxt);

	if(global->selected_host)  g_free(global->selected_host);
	if(global->selected_gametype)  g_free(global->selected_gametype);

	global->selected_host = g_strdup( texts[0] );
	global->selected_port = atoi( texts[5] );
	global->selected_gametype = g_strdup( texts[6] );
	global->selected_gameid = atoi( texts[7] );

	for(i = 0 ; i < 8 ; i++)
		g_free(texts[i]);
}


/* get games: event "row_unselect" */
void Callback_GetGame_Unselect(GtkWidget* clist, gint row)  {

	if(global->selected_host)  g_free(global->selected_host);
	if(global->selected_gametype)  g_free(global->selected_gametype);
	global->selected_host = 0;
	global->selected_port = 0;
	global->selected_gametype = 0;
	global->selected_gameid = 0;
}


/* **** Join/create a game */
void Callback_ConnectButton(GtkWidget* clist)  {

	guint16 connectid;
	gchar *text;

	if(!global->selected_host) return;
	if(!global->selected_port) return;
	if(!global->selected_gametype) return;
	if(!global->selected_gameid) return;

	if(! client_connect(&connectid, global->selected_host, global->selected_port) )   return;
	connection[connectid]->type = CONNECT_TYPE_MONOPD_GAME;

	/* set client protocol */
	text = g_strdup_printf("client name=\"gtkAtlantic\" version=\"%s\" protocol=\"monopd\"\n", VERSION);
	client_send(connectid, text, strlen(text));
	g_free(text);

	/* set nickname */
	text = g_strdup_printf(".n%s\n", config->nickname);
	client_send(connectid, text, strlen(text));
	g_free(text);

	game_alloc();
	game->connectid = connectid;

	/* create a game */
	if(global->selected_gameid < 0)  {

		game->type = GAME_TYPE_CREATE;
		text = g_strdup_printf(".gn%s\n", global->selected_gametype);
		client_send(connectid, text, strlen(text));
		g_free(text);
	}

	/* join a game */
	if(global->selected_gameid > 0)  {

		game->type = GAME_TYPE_JOIN;
		text = g_strdup_printf(".gj%d\n", global->selected_gameid);
		client_send(connectid, text, strlen(text));
		g_free(text);
	}

	g_free(global->selected_host);
	g_free(global->selected_gametype);
	global->selected_host = 0;
	global->selected_port = 0;
	global->selected_gametype = 0;
	global->selected_gameid = 0;
}


/* **** Start the game :) */
void Callback_StartButton(GtkWidget* button)  {

	client_send(game->connectid, ".gs\n", 4);
}


/* **** Enter in the real program */
void Callback_EnterButton(GtkWidget* button)  {

	interface_create_getgamespage();
}


/* *** send data to server */
void Callback_SendButton(GtkWidget* Button, GtkWidget *entry) {

	guchar *tmp, *tmp2;

	tmp = g_strdup( gtk_entry_get_text( GTK_ENTRY(entry) ) );
	g_strstrip(tmp);

	if(tmp[0] != '/')   {

		tmp2 = g_strconcat(tmp, "\n", NULL);
		g_free(tmp);
		tmp = tmp2;
		client_send(game->connectid, tmp, strlen(tmp) );
	}
	else
		parse_specific_send_message(tmp);

	g_free(tmp);
	gtk_entry_set_text(GTK_ENTRY(entry), "");
}


/* *** Un/Float Server Message Box*/
void CallBack_toggle_float_message(GtkWidget* widget) {

	if(global->phase != PHASE_GAMEPLAY) return;

	game->message_float = !game->message_float;
	interface_message_and_chat_float_unfloat();
}


/* *** Un/Float Chat Box*/
void CallBack_toggle_float_chat(GtkWidget* widget) {

	if(global->phase != PHASE_GAMEPLAY) return;

	game->chat_float = !game->chat_float;
	interface_message_and_chat_float_unfloat();
}


/* *** "delete_event" signal window message */
void CallBack_delete_window_message(GtkWidget* widget) {

	gtk_widget_hide( GTK_WIDGET(global->MessageBox) );
	gtk_widget_reparent(GTK_WIDGET(global->MessageBox), global->Trick);
	gtk_widget_destroy(widget);
	widget = 0;
	game->message_float = 0;
	interface_message_and_chat_float_unfloat();
}


/* *** "delete_event" signal window chat */
void CallBack_delete_window_chat(GtkWidget* widget) {

	gtk_widget_hide( GTK_WIDGET(game->ChatBox) );
	gtk_widget_reparent(GTK_WIDGET(game->ChatBox), global->Trick);
	gtk_widget_destroy(widget);
	widget = 0;
	game->chat_float = 0;
	interface_message_and_chat_float_unfloat();
}


/* *** "destroy" signal window message */
void CallBack_destroy_window_message(GtkWidget* widget) {

}


/* *** "destroy" signal window chat */
void CallBack_destroy_window_chat(GtkWidget* widget) {

}


/* *** save and apply the config */
void CallBack_ok_button_configwindow(GtkWidget* widget)  {

	guchar *filename, *errormsg;
	FILE *file;
	guint32 token_animate_speed;

	token_animate_speed = (guint32) GTK_ADJUSTMENT( gtk_object_get_data(GTK_OBJECT(global->ConfigWin), "config_gameinterface_tokenanimatespeed") )->value;

	filename = g_strconcat(global->path_home, "gtkatlantic.conf", NULL);

	file = fopen(filename, "wt");
	if(!file)  {

		errormsg = g_strdup_printf("[ERROR] Can't write config file at \"%s\"", filename);
		text_insert_message(errormsg, strlen(errormsg));
		g_free(errormsg);
		g_free(filename);
		return;
	}
	g_free(filename);

	fprintf(file, "player nickname=\"%s\"\n",
		gtk_entry_get_text( GTK_ENTRY( gtk_object_get_data(GTK_OBJECT(global->ConfigWin), "config_player_nickname") ) )  );
	fprintf(file, "metaserver2 host=\"%s\" port=\"%d\" autogetgames=\"%d\" autogetlist=\"%d\" sendclientversion=\"%d\"\n",
		gtk_entry_get_text( GTK_ENTRY( gtk_object_get_data(GTK_OBJECT(global->ConfigWin), "config_metaserver_host") ) ),
		atoi( gtk_entry_get_text( GTK_ENTRY( gtk_object_get_data(GTK_OBJECT(global->ConfigWin), "config_metaserver_port") ) ) ),
		gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( gtk_object_get_data(GTK_OBJECT(global->ConfigWin), "config_metaserver_autogetgames") ) ),
		gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( gtk_object_get_data(GTK_OBJECT(global->ConfigWin), "config_metaserver_autogetlist") ) ),
		gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( gtk_object_get_data(GTK_OBJECT(global->ConfigWin), "config_metaserver_sendclientversion") ) )  );
	fprintf(file, "anotherserver host=\"%s\" port=\"%d\" autogetgames=\"%d\"\n",
		gtk_entry_get_text( GTK_ENTRY( gtk_object_get_data(GTK_OBJECT(global->ConfigWin), "config_anotherserver_host") ) ),
		atoi( gtk_entry_get_text( GTK_ENTRY( gtk_object_get_data(GTK_OBJECT(global->ConfigWin), "config_anotherserver_port") ) ) ),
		gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( gtk_object_get_data(GTK_OBJECT(global->ConfigWin), "config_anotherserver_autogetgames") ) )  );
	fprintf(file, "buffer message=\"%d\" chat=\"%d\"\n",
		atoi( gtk_entry_get_text( GTK_ENTRY( gtk_object_get_data(GTK_OBJECT(global->ConfigWin), "config_buffersize_servermessages") ) ) ),
		atoi( gtk_entry_get_text( GTK_ENTRY( gtk_object_get_data(GTK_OBJECT(global->ConfigWin), "config_buffersize_chat") ) ) ) );
	fprintf(file, "interface playerlist=\"%s\" token_speed=\"%d\" token_transparency=\"%d\"\n",
		gtk_entry_get_text( GTK_ENTRY( gtk_object_get_data(GTK_OBJECT(global->ConfigWin), "config_gameinterface_playerlistposition") ) ),
		token_animate_speed,
		gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( gtk_object_get_data(GTK_OBJECT(global->ConfigWin), "config_gameinterface_tokentransparency") ) )  );

	fclose(file);

	if(global->phase == PHASE_GAMEPLAY  &&  token_animate_speed != config->game_token_animation_speed)  {

		gtk_timeout_remove(game->timeout_token);
		game->timeout_token = gtk_timeout_add(token_animate_speed, game_move_tokens, NULL);
	}

	read_config_files();

	gtk_widget_destroy(global->ConfigWin);
}


/* *** reallow show config panel */
void CallBack_configwindow_destroyed(GtkWidget* widget)  {

	global->ConfigWin = 0;
}


/* *** button command pressed */
void CallBack_button_command_pressed(GtkWidget* button) {

	guchar *ptr = gtk_object_get_data(GTK_OBJECT(button), "command");
	client_send(game->connectid, ptr, strlen(ptr) );

	if(! strcmp(ptr, ".E\n") )
		display_clear_buttons();
}


/* *** button command pressed */
void Callback_AuctionButton_Absolute(GtkWidget* button, GtkWidget* entry)  {

	guchar *tosend, *auction, *money;

	auction = gtk_object_get_data(GTK_OBJECT(entry), "auctionid");

	money = g_strdup( gtk_entry_get_text( GTK_ENTRY(entry) ) );
	g_strstrip(money);

	tosend = g_strconcat(".ab", auction, ":", money, "\n", NULL);

	client_send(game->connectid, tosend, strlen(tosend) );

	g_free(money);
	g_free(tosend);

	gtk_entry_set_text(GTK_ENTRY(entry), "");
}


/* *** button command pressed */
void Callback_AuctionButton_Relative(GtkWidget* button)  {

	guchar *tosend, *auction;
	gint32 money;

	money = GPOINTER_TO_INT( gtk_object_get_data(GTK_OBJECT(game->BoardCenter), "auction_highbid") )
		+ GPOINTER_TO_INT( gtk_object_get_data(GTK_OBJECT(button), "value_add") );
	auction = gtk_object_get_data(GTK_OBJECT( gtk_object_get_data(GTK_OBJECT(button), "auction_entry") ), "auctionid");

	tosend = g_strdup_printf(".ab%s:%d\n", auction, money);
	client_send(game->connectid, tosend, strlen(tosend) );
	g_free(tosend);
}


/* *** callback clic on estate */
void Callback_ClicOnEstate(GtkWidget* widget, GdkEventButton *event)  {

	guint8 id = atoi( gtk_object_get_data(GTK_OBJECT(widget), "estateid") );
	guchar *text;

	if(event->button == 1)  { /* left clic -> build house */

		text = g_strdup_printf(".hb%d\n", id);
		client_send(game->connectid, text, strlen(text) );
		g_free(text);
	}
	if(event->button == 3)  { /* right clic -> sell house */

		text = g_strdup_printf(".hs%d\n", id);
		client_send(game->connectid, text, strlen(text) );
		free(text);
	}
	if(event->button == 2)  { /* middle clic -> toggle_mortgage */

		text = g_strdup_printf(".em%d\n", id);
		client_send(game->connectid, text, strlen(text) );
		g_free(text);
	}
}


/* *** callback enter an estate */
void Callback_EnterAnEstate(GtkWidget* widget)  {

	guint8 id = atoi( gtk_object_get_data(GTK_OBJECT(widget), "estateid") );

	guint16 frame = game->board_frame;
	guint16 pic = game->estate[id].pic;

	pic_set_bgcolor(frame, pic, 0x000000);
	pic_set_alpha(frame, pic, 170);
}


/* *** callback leave an estate */
void Callback_LeaveAnEstate(GtkWidget* widget)  {

	guint8 id = atoi( gtk_object_get_data(GTK_OBJECT(widget), "estateid") );

	guint16 frame = game->board_frame;
	guint16 pic = game->estate[id].pic;

	pic_unset_alpha(frame, pic);
}


/* *** callback clic on estate */
void Callback_ClicOnPlayer(GtkWidget* widget, GdkEvent *event)  {

	GtkWidget *menu = gtk_object_get_data(GTK_OBJECT(widget), "menu");

	if (event->type == GDK_BUTTON_PRESS)  { 

		GdkEventButton *button_event = (GdkEventButton *)event;

		if (button_event->button == 3)
			gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL, button_event->button, button_event->time);
	}
	return;
}


/* *** callback clic on estate */
void CallBack_PopupMenuPlayer(GtkWidget* widget)  {

	guchar *text;

	guint32 id = atoi( gtk_object_get_data(GTK_OBJECT(widget->parent), "playerid") );
	gchar *action = gtk_object_get_data(GTK_OBJECT(widget), "action");

	if(! strcmp(action, "trade") )  {

		/* initiate trade */
		text = g_strdup_printf(".Tn%d\n", id);
		client_send(game->connectid, text, strlen(text));
		g_free(text);
	}

	if(! strcmp(action, "version") )  {

		/* request version */
		text = g_strdup_printf("!version %s\n", game->player[ get_player_slot_with_playerid(id) ].name);
		client_send(game->connectid, text, strlen(text));
		g_free(text);
	}

	if(! strcmp(action, "date") )  {

		/* request date */
		text = g_strdup_printf("!date %s\n", game->player[ get_player_slot_with_playerid(id) ].name);
		client_send(game->connectid, text, strlen(text));
		g_free(text);
	}

	if(! strcmp(action, "ping") )  {

		/* request date */
		text = g_strdup_printf("!ping %s\n", game->player[ get_player_slot_with_playerid(id) ].name);
		client_send(game->connectid, text, strlen(text));
		g_free(text);
	}
}


/* *** callback for game option type bool */
void Callback_toggle_boolean_gameoption(GtkWidget* widget)  {

	guchar *text, *cmd;

	if(! widget)  return;

	if(game->i_am_the_master)  {

		cmd = gtk_object_get_data(GTK_OBJECT(widget), "command");
		text = g_strdup_printf("%s%d\n", cmd, gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(widget) )  );

		client_send(game->connectid, text, strlen(text) );
		g_free(text);
	}
}


/* *** callback for the entry menu change nickname */
void CallBack_menu_change_nickname(GtkWidget* widget)  {

	if(global->phase >= PHASE_GAMECREATE)
		interface_create_nicknamewin();
}


/* *** callback for the ok button of window change nickname */
void CallBack_ok_button_changenick_window(GtkWidget* widget, GtkWidget *Win)  {

	GtkWidget *Entry = gtk_object_get_data(GTK_OBJECT(Win), "nickname");
	guchar *nickname, *sendstr;

	nickname = g_strdup( gtk_entry_get_text(GTK_ENTRY( Entry ) )  );

	if(strcmp(game->player[ get_player_slot_with_playerid(game->my_playerid) ].name, nickname) )  {

		sendstr = g_strdup_printf(".n%s\n", nickname);
		client_send(game->connectid, sendstr, strlen(sendstr) );
		g_free(sendstr);
	}

	g_free(nickname);
	gtk_widget_destroy(Win);
}


/* *** callback for the entry menu reconnect */
void CallBack_menu_reconnect(GtkWidget* widget)  {

	guchar *host, *tmp, *cookie, *filename;
	guint32 port;
	FILE *file;

	guint16 connectid;
	gchar *text;

	if(global->phase > PHASE_GETGAMES) return;

	/* read cookie file */
	filename = g_strconcat(global->path_home, "cookie", NULL);
	file = fopen(filename, "rt");
	g_free(filename);

	if(!file)  return;

	host = g_malloc0(4096);
	fscanf(file, "%s\n", host );

	tmp = g_malloc0(4096);
	fscanf(file, "%s\n", tmp );
	port = atoi(tmp);
	g_free(tmp);

	cookie = g_malloc0(4096);
	fscanf(file, "%s\n", cookie);

	fclose(file);

	/* rejoin game */
	if(! client_connect(&connectid, host, port) )   {

		g_free(host);
		g_free(cookie);
		return;
	}

	g_free(host);
	connection[connectid]->type = CONNECT_TYPE_MONOPD_GAME;

	game_alloc();
	game->connectid = connectid;

	/* send cookie */
	text = g_strdup_printf(".R%s\n", cookie);
	client_send(connectid, text, strlen(text));
	g_free(text);


	g_free(cookie);
}


/* *** callback for all trades buttons */
void CallBack_trade_button(GtkWidget *button)  {

	guchar *sendstr = 0;
	guchar *command = gtk_object_get_data( GTK_OBJECT(button), "command" );
	guint32 tradeid = atoi( gtk_object_get_data( GTK_OBJECT(button), "tradeid" )  );
	guint32	tradeslot = get_trade_slot_with_tradeid(tradeid);

	if(! strcmp(command, "reject") )  {

		sendstr = g_strdup_printf(".Tr%d\n", tradeid);
		client_send(game->connectid, sendstr, strlen(sendstr));
		g_free(sendstr);
	}

	if(! strcmp(command, "accept") )  {

		sendstr = g_strdup_printf(".Ta%d\n", tradeid);
		client_send(game->connectid, sendstr, strlen(sendstr));
		g_free(sendstr);
	}

	if(! strcmp(command, "remove") )  {

		if(game->trade[tradeslot].select_type == TRADE_SELECT_MONEY)
			sendstr = g_strdup_printf(".Tm%d:%d:%d:%d\n", tradeid, game->trade[tradeslot].select_from, game->trade[tradeslot].select_to, 0);

		if(game->trade[tradeslot].select_type == TRADE_SELECT_ESTATE)
			sendstr = g_strdup_printf(".Te%d:%d:%d\n", tradeid, game->trade[tradeslot].select_value, game->trade[tradeslot].select_from);

		if(game->trade[tradeslot].select_type == TRADE_SELECT_CARD)
			sendstr = g_strdup_printf(".Tc%d:%d:%d\n", tradeid, game->trade[tradeslot].select_value, game->trade[tradeslot].select_from);

		if(sendstr)  {
			client_send(game->connectid, sendstr, strlen(sendstr));
			g_free(sendstr);
		}
	}
}


/* *** callback for all trade component */
void CallBack_trade_sub_component(GtkWidget *menuitem)  {

	guchar *component = gtk_object_get_data( GTK_OBJECT(menuitem), "component" );
	guint32 tradeslot = get_trade_slot_with_tradeid( atoi( gtk_object_get_data( GTK_OBJECT(menuitem), "tradeid" )  )  );

	if(!component) return;
	if(tradeslot < 0) return;

	if(! strcmp(component, "estate") )
		game->trade[tradeslot].current_component = TRADE_CURRENT_COMPONENT_ESTATE;

	if(! strcmp(component, "money") )
		game->trade[tradeslot].current_component = TRADE_CURRENT_COMPONENT_MONEY;

	if(! strcmp(component, "card") )
		game->trade[tradeslot].current_component = TRADE_CURRENT_COMPONENT_CARD;

	trade_rebuild_subcomponent(tradeslot);
}


/* *** callback for all update component */
void CallBack_trade_update_component(GtkWidget *button)  {

	guchar *sendstr;
	guint32 tradeid, estateid, cardid, target, from, to, tradeslot, money;

	tradeslot = get_trade_slot_with_tradeid( atoi( gtk_object_get_data( GTK_OBJECT(button), "tradeid" )  )  );
	tradeid = game->trade[tradeslot].tradeid;

	switch(game->trade[tradeslot].current_component)  {

		case TRADE_CURRENT_COMPONENT_ESTATE:

			estateid = get_estateid_by_estatename( gtk_entry_get_text( gtk_object_get_data( GTK_OBJECT(button), "name_estate" ) )  );
			target = get_playerid_by_playername( gtk_entry_get_text( gtk_object_get_data( GTK_OBJECT(button), "name_target_player" ) )  );

			sendstr = g_strdup_printf(".Te%d:%d:%d\n", tradeid, estateid, target);
			client_send(game->connectid, sendstr, strlen(sendstr));
			g_free(sendstr);
			break;

		case TRADE_CURRENT_COMPONENT_MONEY:

			from = get_playerid_by_playername( gtk_entry_get_text( gtk_object_get_data( GTK_OBJECT(button), "name_from_player" ) )  );
			to = get_playerid_by_playername( gtk_entry_get_text( gtk_object_get_data( GTK_OBJECT(button), "name_to_player" ) )  );
			money = atoi( gtk_entry_get_text( gtk_object_get_data( GTK_OBJECT(button), "amount_money" ) )  );

			sendstr = g_strdup_printf(".Tm%d:%d:%d:%d\n", tradeid, from, to, money);
			client_send(game->connectid, sendstr, strlen(sendstr));
			g_free(sendstr);
			break;

		case TRADE_CURRENT_COMPONENT_CARD:

			cardid = atoi( gtk_entry_get_text( gtk_object_get_data( GTK_OBJECT(button), "name_card" ) )  );
			target = get_playerid_by_playername( gtk_entry_get_text( gtk_object_get_data( GTK_OBJECT(button), "name_target_player" ) )  );

			sendstr = g_strdup_printf(".Tc%d:%d:%d\n", tradeid, cardid, target);
			client_send(game->connectid, sendstr, strlen(sendstr));
			g_free(sendstr);
			break;
	}
}


void CallBack_logo_page_destroyed(GtkWidget *widget)  {

	frame_destroy(global->logo_frame);
	intro_free_pngs();
}


/* *** callback for the entry menu leave game */
void CallBack_menu_LeaveGame(GtkWidget* widget)  {

	guchar *sendstr;

	if(global->phase < PHASE_GAMECREATE)  return;

	if(global->phase >= PHASE_GAMEPLAY)  {

		sendstr = g_strdup_printf(".gx\n");
		client_send(game->connectid, sendstr, strlen(sendstr));
		g_free(sendstr);
	}

	game_free();
}


/* *** callback for the entry menu close all connections */
void CallBack_menu_CloseAllConnections(GtkWidget* widget)  {

	guint32 i;

	for(i = 0 ; i < MAX_CONNECTION ; i++)  {

		if(! connection_is_open[i])  continue;
		if(game  &&  game->connectid == i)  continue;

		client_disconnect(i);
	}
}


/* *** Show estates tree */
void CallBack_estates_tree(GtkWidget* widget) {

	if(global->phase != PHASE_GAMEPLAY) return;

	interface_create_estates_tree();
}


/* *** callback for the entry menu declare bankrupcy */
void CallBack_menu_DeclareBankrupcy(GtkWidget* widget)  {

	guchar *sendstr;

	if(global->phase < PHASE_GAMEPLAY) return;

	sendstr = g_strdup_printf(".D\n");
	client_send(game->connectid, sendstr, strlen(sendstr));
	g_free(sendstr);
}


void CallBack_EstateTree_SelectRow(GtkCTree *EstateTree,  GtkCTreeNode *Node)  {

	gchar *data;
	GtkWidget *parent, *info, *Alignment, *Label;
	guchar *Text;
	gint32 id;

	data = gtk_ctree_node_get_row_data(GTK_CTREE(EstateTree), Node);
	id = atoi(data);

	parent = gtk_object_get_data(GTK_OBJECT(game->WinEstateTree), "parent_box");
	info = gtk_object_get_data(GTK_OBJECT(game->WinEstateTree), "info_box");

	if(info) gtk_widget_destroy(info);
	info = 0;

	/* info box */
	info = gtk_vbox_new(FALSE, 0);
	gtk_object_set_data(GTK_OBJECT(game->WinEstateTree), "info_box", info);
	gtk_container_set_border_width(GTK_CONTAINER(info), 10);
	gtk_box_pack_start(GTK_BOX(parent), info, TRUE, TRUE, 0);

	/* name */
	Text = g_strdup_printf("%s", game->estate[id].name);

	Alignment = gtk_alignment_new(0.0, 0.0, 0.0, 0.0);
	gtk_box_pack_start(GTK_BOX(info), Alignment, FALSE, TRUE, 0);
	Label = gtk_label_new(Text);
	gtk_container_add(GTK_CONTAINER(Alignment), Label);

	g_free(Text);


	/* group */
	if(game->estate[id].group >= 0)  Text = g_strdup_printf("%s\n", game->group[ game->estate[id].group ].name);
	else  Text = g_strdup("\n");

	Alignment = gtk_alignment_new(0.0, 0.0, 0.0, 0.0);
	gtk_box_pack_start(GTK_BOX(info), Alignment, FALSE, TRUE, 0);
	Label = gtk_label_new(Text);
	gtk_container_add(GTK_CONTAINER(Alignment), Label);

	g_free(Text);

	/* state (for sale, owned, mortgaged, unknow?) */
	if(game->estate[id].price)  {

		if(game->estate[id].owner <= 0)  Text = g_strdup("State: for sale");
		else  if(game->estate[id].owner > 0  &&  game->estate[id].mortgaged)  Text = g_strdup("State: mortgaged");
		else  if(game->estate[id].owner > 0)  Text = g_strdup("State: owned");
		else  Text = g_strdup("State: unknow");

		Alignment = gtk_alignment_new(0.0, 0.0, 0.0, 0.0);
		gtk_box_pack_start(GTK_BOX(info), Alignment, FALSE, TRUE, 0);
		Label = gtk_label_new(Text);
		gtk_container_add(GTK_CONTAINER(Alignment), Label);

		g_free(Text);
	}

	/* owner */
	if(game->estate[id].price)  {

		if(game->estate[id].owner <= 0)  Text = g_strdup("Owner: no");
		else  Text = g_strdup_printf("Owner: %s", game->player[ get_player_slot_with_playerid( game->estate[id].owner ) ].name);

		Alignment = gtk_alignment_new(0.0, 0.0, 0.0, 0.0);
		gtk_box_pack_start(GTK_BOX(info), Alignment, FALSE, TRUE, 0);
		Label = gtk_label_new(Text);
		gtk_container_add(GTK_CONTAINER(Alignment), Label);

		g_free(Text);
	}

	/* price */
	if(game->estate[id].price)  {

		Text = g_strdup_printf("\nPrice: %d", game->estate[id].price);

		Alignment = gtk_alignment_new(0.0, 0.0, 0.0, 0.0);
		gtk_box_pack_start(GTK_BOX(info), Alignment, FALSE, TRUE, 0);
		Label = gtk_label_new(Text);
		gtk_container_add(GTK_CONTAINER(Alignment), Label);

		g_free(Text);
	}

	/* money */
	if(game->estate[id].money)  {

		Text = g_strdup_printf("Money: %d", game->estate[id].money);

		Alignment = gtk_alignment_new(0.0, 0.0, 0.0, 0.0);
		gtk_box_pack_start(GTK_BOX(info), Alignment, FALSE, TRUE, 0);
		Label = gtk_label_new(Text);
		gtk_container_add(GTK_CONTAINER(Alignment), Label);

		g_free(Text);
	}

	/* rent */
	if(game->estate[id].rent[5])  {

		Text = g_strdup_printf("\nRent with no house: %d", game->estate[id].rent[0]);

		Alignment = gtk_alignment_new(0.0, 0.0, 0.0, 0.0);
		gtk_box_pack_start(GTK_BOX(info), Alignment, FALSE, TRUE, 0);
		Label = gtk_label_new(Text);
		gtk_container_add(GTK_CONTAINER(Alignment), Label);

		g_free(Text);


		Text = g_strdup_printf("Rent with 1 house: %d", game->estate[id].rent[1]);

		Alignment = gtk_alignment_new(0.0, 0.0, 0.0, 0.0);
		gtk_box_pack_start(GTK_BOX(info), Alignment, FALSE, TRUE, 0);
		Label = gtk_label_new(Text);
		gtk_container_add(GTK_CONTAINER(Alignment), Label);

		g_free(Text);


		Text = g_strdup_printf("Rent with 2 houses: %d", game->estate[id].rent[2]);

		Alignment = gtk_alignment_new(0.0, 0.0, 0.0, 0.0);
		gtk_box_pack_start(GTK_BOX(info), Alignment, FALSE, TRUE, 0);
		Label = gtk_label_new(Text);
		gtk_container_add(GTK_CONTAINER(Alignment), Label);

		g_free(Text);


		Text = g_strdup_printf("Rent with 3 houses: %d", game->estate[id].rent[3]);

		Alignment = gtk_alignment_new(0.0, 0.0, 0.0, 0.0);
		gtk_box_pack_start(GTK_BOX(info), Alignment, FALSE, TRUE, 0);
		Label = gtk_label_new(Text);
		gtk_container_add(GTK_CONTAINER(Alignment), Label);

		g_free(Text);


		Text = g_strdup_printf("Rent with 4 houses: %d", game->estate[id].rent[4]);

		Alignment = gtk_alignment_new(0.0, 0.0, 0.0, 0.0);
		gtk_box_pack_start(GTK_BOX(info), Alignment, FALSE, TRUE, 0);
		Label = gtk_label_new(Text);
		gtk_container_add(GTK_CONTAINER(Alignment), Label);

		g_free(Text);


		Text = g_strdup_printf("Rent with a hotel: %d", game->estate[id].rent[5]);

		Alignment = gtk_alignment_new(0.0, 0.0, 0.0, 0.0);
		gtk_box_pack_start(GTK_BOX(info), Alignment, FALSE, TRUE, 0);
		Label = gtk_label_new(Text);
		gtk_container_add(GTK_CONTAINER(Alignment), Label);

		g_free(Text);
	}

	/* houses prices */
	if(game->estate[id].houseprice)  {

		Text = g_strdup_printf("\nBuy house price: %d", game->estate[id].houseprice);

		Alignment = gtk_alignment_new(0.0, 0.0, 0.0, 0.0);
		gtk_box_pack_start(GTK_BOX(info), Alignment, FALSE, TRUE, 0);
		Label = gtk_label_new(Text);
		gtk_container_add(GTK_CONTAINER(Alignment), Label);

		g_free(Text);
	}

	if(game->estate[id].sellhouseprice)  {

		Text = g_strdup_printf("Sell house price: %d", game->estate[id].sellhouseprice);

		Alignment = gtk_alignment_new(0.0, 0.0, 0.0, 0.0);
		gtk_box_pack_start(GTK_BOX(info), Alignment, FALSE, TRUE, 0);
		Label = gtk_label_new(Text);
		gtk_container_add(GTK_CONTAINER(Alignment), Label);

		g_free(Text);
	}

	/* number of houses */
	if(game->estate[id].houseprice)  {

		Text = g_strdup_printf("Number of houses: %d", game->estate[id].houses);

		Alignment = gtk_alignment_new(0.0, 0.0, 0.0, 0.0);
		gtk_box_pack_start(GTK_BOX(info), Alignment, FALSE, TRUE, 0);
		Label = gtk_label_new(Text);
		gtk_container_add(GTK_CONTAINER(Alignment), Label);

		g_free(Text);
	}

	/* (un)mortgage value */
	if(game->estate[id].mortgageprice)  {

		Text = g_strdup_printf("\nMortgage price: %d", game->estate[id].mortgageprice);

		Alignment = gtk_alignment_new(0.0, 0.0, 0.0, 0.0);
		gtk_box_pack_start(GTK_BOX(info), Alignment, FALSE, TRUE, 0);
		Label = gtk_label_new(Text);
		gtk_container_add(GTK_CONTAINER(Alignment), Label);

		g_free(Text);
	}

	if(game->estate[id].unmortgageprice)  {

		Text = g_strdup_printf("Unmortgage price: %d", game->estate[id].unmortgageprice);

		Alignment = gtk_alignment_new(0.0, 0.0, 0.0, 0.0);
		gtk_box_pack_start(GTK_BOX(info), Alignment, FALSE, TRUE, 0);
		Label = gtk_label_new(Text);
		gtk_container_add(GTK_CONTAINER(Alignment), Label);

		g_free(Text);
	}


	gtk_widget_show_all(info);
}


void CallBack_EstateTree_UnSelectRow(GtkCTree *EstateTree,  GtkCTreeNode *Node)  {

	gchar *data;
	GtkWidget *parent, *info, *Alignment, *Label;
	gint32 id;

	data = gtk_ctree_node_get_row_data(GTK_CTREE(EstateTree), Node);
	id = atoi(data);

	parent = gtk_object_get_data(GTK_OBJECT(game->WinEstateTree), "parent_box");
	info = gtk_object_get_data(GTK_OBJECT(game->WinEstateTree), "info_box");

	if(info) gtk_widget_destroy(info);
	info = 0;

	/* info box */
	info = gtk_vbox_new(FALSE, 0);
	gtk_object_set_data(GTK_OBJECT(game->WinEstateTree), "info_box", info);
	gtk_container_set_border_width(GTK_CONTAINER(info), 10);
	gtk_box_pack_start(GTK_BOX(parent), info, TRUE, TRUE, 0);

	/* default text */
	Alignment = gtk_alignment_new(0.0, 0.0, 0.0, 0.0);
	gtk_box_pack_start(GTK_BOX(info), Alignment, FALSE, TRUE, 0);
	Label = gtk_label_new("Select an estate...");
	gtk_container_add(GTK_CONTAINER(Alignment), Label);

	gtk_widget_show_all(info);
}


void CallBack_EstateTree_Destroyed(GtkWidget *widget)  {

	game->WinEstateTree = 0;
}


/* proposal list: event "row_select" */
void Callback_ProposalList_Select(GtkWidget* clist, gint row)  {

	gchar **ptrtxt, *texts[10];
	guint32 i, tradeslot;

	ptrtxt = g_malloc(8192);

	for(i = 0 ; i < 10 ; i++)  {

		memset(ptrtxt, 0, 8192);
		gtk_clist_get_text(GTK_CLIST(clist), row, i, ptrtxt);
		texts[i] = g_strdup(*ptrtxt);
	}
	g_free(ptrtxt);

	tradeslot =  atoi( texts[9] );

	if(! strcmp(texts[5], "money") )
		game->trade[tradeslot].select_type = TRADE_SELECT_MONEY;
	else  if(! strcmp(texts[5], "estate") )
		game->trade[tradeslot].select_type = TRADE_SELECT_ESTATE;
	else  if(! strcmp(texts[5], "card") )
		game->trade[tradeslot].select_type = TRADE_SELECT_CARD;

	game->trade[tradeslot].select_from = atoi( texts[6] );
	game->trade[tradeslot].select_to = atoi( texts[7] );
	game->trade[tradeslot].select_value = atoi( texts[8] );

	for(i = 0 ; i < 10 ; i++)
		g_free(texts[i]);
}


/* proposal list: event "row_unselect" */
void Callback_ProposalList_Unselect(GtkWidget* clist, gint row)  {

	gchar **ptrtxt, *texts[10];
	guint32 i, tradeslot;

	ptrtxt = g_malloc(8192);

	for(i = 0 ; i < 10 ; i++)  {

		memset(ptrtxt, 0, 8192);
		gtk_clist_get_text(GTK_CLIST(clist), row, i, ptrtxt);
		texts[i] = g_strdup(*ptrtxt);
	}
	g_free(ptrtxt);

	tradeslot =  atoi( texts[9] );

	game->trade[tradeslot].select_type = 0;
	game->trade[tradeslot].select_from = 0;
	game->trade[tradeslot].select_to = 0;
	game->trade[tradeslot].select_value = 0;

	for(i = 0 ; i < 10 ; i++)
		g_free(texts[i]);
}


void CallBack_HelpWin_Destroyed(GtkWidget *widget)  {

	global->HelpWin = 0;
}


void Callback_menu_Themes(GtkWidget *widget)  {

	if(global->ThemeWin)  return;

	theme_build_selection_win();
	theme_build_database();
	theme_fill_selection_list();
}


void CallBack_ThemeWin_Delete(GtkWidget *widget)  {

	theme_preview_free();
	gtk_widget_destroy(global->ThemeWin);
}


void CallBack_ThemeWin_Destroyed(GtkWidget *widget)  {

	global->ThemeWin = 0;
}


/* theme list: event "row_select" */
void Callback_ThemeList_Select(GtkWidget* clist, gint row)  {

	gchar **ptrtxt, *texts[10];
	guint32 i, themeid;

	ptrtxt = g_malloc(8192);

	for(i = 0 ; i < 2 ; i++)  {

		memset(ptrtxt, 0, 8192);
		gtk_clist_get_text(GTK_CLIST(clist), row, i, ptrtxt);
		texts[i] = g_strdup(*ptrtxt);
	}
	g_free(ptrtxt);

	themeid = atoi( texts[1] );

	for(i = 0 ; i < 2 ; i++)
		g_free(texts[i]);

	theme_display(themeid);
}


/* theme list: event "row_unselect" */
void Callback_ThemeList_Unselect(GtkWidget* clist, gint row)  {


}


/* select/apply a theme */
void CallBack_Theme_Apply(GtkWidget* button)  {

	theme_load( atoi( gtk_object_get_data(GTK_OBJECT(global->ThemeWin), "theme_preview_selected") ) );
}
