/* GtkBalls
 * Copyright (C) 1998-1999 Eugene V. Morozov
 * Modifyed in 2001 by drF_ckoff
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#include <config.h>
#include <gtk/gtk.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>
#include <ctype.h>
#include <fcntl.h>
#include <errno.h>

#include "gtkballs.h"
#include "gtkutils.h"
#include "interface.h"
#include "scoreboard.h"
#include "preferences.h"
#include "gfx.h"

static GtkWidget *dialog 	= NULL;
static gint 	  saved_score 	= 0;
static gchar 	 *user_name 	= NULL;
static gchar 	  last_player_name[15]="";

gchar *selected_save_load=NULL;

struct point { gint x; gint y; };

struct gtkb_animarray {
	gint color, x, y, phase, time;
};

struct gtkb_rules_dialog {
  	GtkWidget *xrange, *yrange, *crange, *nrange, *drange;
};

struct gtkb_rules_dialog RD;

gint parse_save_game(gchar *sgame, GtkbGameRules *rules, gint *score, gint **board, gint **next) {
  	struct stat buf;
        gchar  *sdata, *psdata;
        FILE   *f;
        gint   i, val;

  	if(stat(sgame,&buf) != 0 || !S_ISREG(buf.st_mode) ||
	   buf.st_size < 20 ||
           (f = fopen(sgame, "r")) == NULL) {
                return 0;
        }

        sdata=g_malloc(buf.st_size);

        if(fread(sdata, 1, buf.st_size, f) != buf.st_size) {
                g_free(sdata);
                return 0;
        }

        fclose(f);

        if(sscanf(sdata, "%02d%02d%02d%02d%02d", &rules->xsize, &rules->ysize, &rules->colors, &rules->next, &rules->destroy) != 5
           || sscanf(sdata + 10, "%010d", score) !=1
           || rules->xsize < 0 || rules->xsize > 99
           || rules->ysize < 0 || rules->ysize > 99
           || rules->colors < 2 || rules->colors > 99
           || rules->next < 2 || rules->next > 99
           || rules->destroy < 3 || rules->destroy > 99
	   || score < 0
	   || buf.st_size != 20 + rules->xsize * rules->ysize * 2 + rules->next * 2) {
                g_free(sdata);
                return 0;
        }

        *board = g_malloc(rules->xsize * rules->ysize * sizeof(gint));
        *next = g_malloc(rules->next * sizeof(gint));

        psdata = sdata + 20;
        for(i=0; i<rules->xsize * rules->ysize; i++, psdata += 2) {
                if(sscanf(psdata, "%02d", &val) != 1 || val < 0 || val > rules->colors) {
                        g_free(*next);
                        g_free(*board);
                	g_free(sdata);
                	return 0;
                }
                (*board)[i] = val;
        }

        for(i=0; i<rules->next; i++, psdata += 2) {
                if(sscanf(psdata, "%02d", &val) != 1 || val < 0 || val > rules->colors) {
                        g_free(*next);
                        g_free(*board);
                	g_free(sdata);
                	return 0;
                }
                (*next)[i] = val;
        }

        g_free(sdata);

        return 1;
}

/* check that save game name is in form YYYY-MM-DD-HHMMSS.sav and have correct content
   return string "DD.MM.YYYY HH:MM:SS (score)" on success, NULL on failure */
gchar *is_valid_save_game(gchar *name, gchar *path) {
        guint  i,y,m,d,h,min,s;
        gint score, *board, *next;
        gchar  *sgame;
        GtkbGameRules rules;

        if((i=sscanf(name, "%04u-%02u-%02u-%02u%02u%02u",&y,&m,&d,&h,&min,&s))!=6 ||
           !m || m > 12 || !d || d > 31 || h > 23 || min > 59 || s > 61 ||
	   (strcmp(name+strlen(name)-4, ".sav")!=0)) {
                return NULL;
        }

        sgame=g_strconcat(path, G_DIR_SEPARATOR_S, name, NULL);

	i = parse_save_game(sgame, &rules, &score, &board, &next);
        g_free(sgame);

	if(!i) return NULL;

        g_free(board);
        g_free(next);

        return g_strdup_printf("%02d.%02d.%04d %02d:%02d:%02d (%d [%dx%d %d %d %d])",
		d, m, y, h, min, s, score, rules.xsize, rules.ysize, rules.colors, rules.next, rules.destroy);
}

/* return list of "data" strung, "file name" pairs... */
gint get_saved_games(gchar ***gamelist) {
  	gchar  *homedir, *datapath;
  	struct stat buf;
  	struct dirent *dir_entry;
  	DIR    *directory;
        gchar  **games=NULL,*game;
        gint   num=0;

  	if(!(homedir=getenv("HOME"))) {
                return -1;
        }

        datapath=g_strconcat(homedir, G_DIR_SEPARATOR_S, ".gtkballs", NULL);
  	if(stat(datapath,&buf) != 0) { /* no ~/.gtkballs */
                if(mkdir(datapath, 0700) != 0) { /* and cannot create it... */
        		g_free(datapath);
                        return -1;
                }
        } else if(!S_ISDIR(buf.st_mode)) { /* ~/.gtkballs is not a directory */
        	g_free(datapath);
                return -1;
        }
        if((directory=opendir(datapath))) {
        	while((dir_entry=readdir(directory))) {
                        if((game=is_valid_save_game(dir_entry->d_name, datapath))!=NULL) {
                                num++;
                                games=g_realloc(games, sizeof(gchar *)*num*2);
                                games[(num-1)*2]=game;
                                games[(num-1)*2+1]=g_strdup_printf("%s/%s", datapath, dir_entry->d_name);
                        }
        	}
        	closedir(directory);
        }
        g_free(datapath);
        *gamelist=games;
        return num;
}

void do_save_game(GtkWidget *widget, gpointer data) {
        FILE *f;
        gchar *fname, *errormsg;
        time_t nowtime;
        gchar  ftime[]="0000-00-00-000000";
        gint i;

        nowtime=time(NULL);
        if(selected_save_load) {
                /* TODO: alert stupid user about erasing file... */
                unlink(selected_save_load);
        }
        strftime(ftime, sizeof(ftime), "%Y-%m-%d-%H%M%S", localtime(&nowtime));
        fname=g_strconcat(getenv("HOME"), G_DIR_SEPARATOR_S, ".gtkballs", G_DIR_SEPARATOR_S, ftime, ".sav", NULL);
        if((f = fopen(fname, "w")) != NULL) {
                chmod(fname, 0600);
                /* TODO: check for errors ! */
                fprintf(f, "%02d%02d%02d%02d%02d", Rules.xsize, Rules.ysize, Rules.colors, Rules.next, Rules.destroy);
                fprintf(f, "%010d", Score);
                for(i=0; i<Rules.xsize*Rules.ysize; i++) {
                	fprintf(f, "%02d", Board[i]);
                }
                for(i=0; i<Rules.next; i++) {
                	fprintf(f, "%02d", NextColors[i]);
                }
                fclose(f);
        } else {
                errormsg=g_strdup_printf(_("Cannot save game to:\n%s\n%s"), fname, strerror(errno));
                simple_message_box(errormsg);
                g_free(errormsg);
        }

        g_free(fname);
  	if(data) {
      		gtk_widget_destroy(GTK_WIDGET(data));
  	}
}

void reinit_board(gint *newboard, gint *newnext, gint score, gint oldnext) {
        g_free(Board);
        g_free(BoardUndo);
        g_free(NextColors);
        g_free(NextColorsUndo);

        BoardUndo=g_malloc0(sizeof(gint)*Rules.xsize*Rules.ysize);
        NextColorsUndo=g_malloc(sizeof(gint)*Rules.next);

        if(newboard) {
		Board=newboard;
        } else {
                Board=g_malloc0(sizeof(gint)*Rules.xsize*Rules.ysize);
        }

        if(newnext) {
        	NextColors=newnext;
        } else {
        	NextColors=g_malloc0(sizeof(gint)*Rules.next);
        }

  	if(!read_score(Scoreboard, &FullScoreboard, &FullScoreboardNumber)) {
      		simple_message_box(_("Unable to read score.\n"));
    	}

        Score=score;
        Hi_score=score>Scoreboard[0].score ? score : Scoreboard[0].score;
        remake_board(oldnext, newnext ? 1 : 0);
        if(!newboard) {
        	new_game();
        } else {
        	set_hi_score_label();
        	set_user_score_label();
        	menu_set_sensitive(Menu_items_factory, "/Edit/Undo", FALSE);
  		if(DrawingArea != NULL) {
                	draw_next_balls();
        		draw_board(DrawingArea, Pixmap);
      			update_rectangle(DrawingArea, 0, 0, -1, -1);
    		}
        }
}

void do_load_game(GtkWidget *widget, gpointer data) {
        gint score, *board, *next, oldnext;
        gchar *errormsg;
        GtkbGameRules rules;

        if(!selected_save_load) {
                simple_message_box(_("No game selected for load.\n"));
  		if(data) {
      			gtk_widget_destroy(GTK_WIDGET(data));
	  	}
                return;
        }

	if(!parse_save_game(selected_save_load, &rules, &score, &board, &next)) {
        	errormsg=g_strdup_printf(_("Cannot load game from:\n%s\n"), selected_save_load);
        	simple_message_box(errormsg);
                g_free(errormsg);
		return;
        }

        if(rules.colors > gtkbTheme->numballs) {
        	errormsg=g_strdup_printf(_("Not enough balls(%d) in current theme.\nWe need %d balls.\nLoad another theme and try again."), gtkbTheme->numballs, rules.colors);
        	simple_message_box(errormsg);
                g_free(errormsg);
		return;
        }

        if(AnimationInProgress) {
      		gtk_timeout_remove(TimerTag);
        	AnimationInProgress=FALSE;
        }

        oldnext=Rules.next;
        memcpy(&Rules, &rules, sizeof(Rules));
	reinit_board(board, next, score, oldnext);

  	if(data) {
      		gtk_widget_destroy(GTK_WIDGET(data));
  	}
}

void save_row_activated_cb(GtkTreeView *treeview, gpointer arg1, GtkTreeViewColumn *arg2, gpointer data) {
        do_save_game(GTK_WIDGET(treeview), data);
}

void load_row_activated_cb(GtkTreeView *treeview, gpointer arg1, GtkTreeViewColumn *arg2, gpointer data) {
        do_load_game(GTK_WIDGET(treeview), data);
}

void save_load_game_list_destroy(gpointer data) {
        if(data) g_free(data);
}

void sl_row_activated(GtkTreeSelection *selection, GtkTreeModel *model) {
        GtkTreeIter iter;
        GValue value = {0, };

        if(gtk_tree_selection_get_selected(selection, NULL, &iter)) {
                gtk_tree_model_get_value(model, &iter, 1, &value);
                if(g_value_get_string(&value)) {
                        selected_save_load=(gchar *)g_value_get_string(&value);
                } else {
                        selected_save_load=NULL;
                }
        }
}

void save_load_game_dialog(GtkWidget *widget, gpointer data, gboolean is_save) {
        GtkListStore *store;
        GtkTreeIter iter;
	GtkWidget *treeview;
        GtkCellRenderer *renderer;
        GtkTreeViewColumn *column;
        GtkTreePath *path;
  	GtkWidget *window, *swindow;
  	GtkWidget *vbox, *button_box;
  	GtkWidget *ok_button, *cancel_button;
        gint      i, num;
        gchar     **gamelist;

        num=get_saved_games(&gamelist);
        if(!is_save && !num) {
                simple_message_box(_("No saved games found.\n"));
                return;
        }

	window=ut_window_new(is_save ? _("Save game") : _("Load game"),
                             is_save ? "GtkBalls_Save" : "GtkBalls_Load",
			     "GtkBalls", TRUE, TRUE, TRUE, 5);

  	vbox=gtk_vbox_new(FALSE, 0);
  	button_box=gtk_hbox_new(TRUE, 0);
  	gtk_container_set_border_width(GTK_CONTAINER(vbox), 1);

        swindow=gtk_scrolled_window_new(NULL, NULL);
        gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(swindow), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
        gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(swindow), GTK_SHADOW_ETCHED_IN);
  	gtk_box_pack_start(GTK_BOX(vbox), swindow, TRUE, TRUE, 0);

        store=gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_STRING);
        for(i=0;i<num;i++) {
                gtk_list_store_append(store, &iter);
                gtk_list_store_set(store, &iter, 0, (GValue *)gamelist[i*2], 1, (GValue *)gamelist[i*2+1], -1);
        }
        g_free(gamelist);

      	treeview=gtk_tree_view_new_with_model(GTK_TREE_MODEL(store));
	gtk_widget_set_size_request(treeview, -1, 250);

      	g_object_unref(G_OBJECT(store));
      	gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(treeview), TRUE);
      	gtk_tree_view_set_search_column(GTK_TREE_VIEW(treeview), 0);
        gtk_tree_selection_set_mode(gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview)), GTK_SELECTION_BROWSE);

  	g_signal_connect(G_OBJECT(gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview))),
		         "changed", G_CALLBACK(sl_row_activated), store);

        if(is_save) {
                gtk_list_store_append(store, &iter);
                gtk_list_store_set(store, &iter, 0, _("Empty"), -1);
        }
  	gtk_container_add(GTK_CONTAINER(swindow), treeview);

        renderer=gtk_cell_renderer_text_new();
        column=gtk_tree_view_column_new_with_attributes(_("Date (score)"), renderer, "text", 0, NULL);
        gtk_tree_view_column_set_sort_column_id(column, 0);
        gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), column);

        /* FIXME: imho this way of sorting is stupid */
        gtk_tree_view_column_clicked(column);
        gtk_tree_view_column_clicked(column);

        if(iter.stamp==store->stamp) {
                path=gtk_tree_path_new_from_string("0");
                if(path) {
			gtk_tree_selection_select_path(gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview)), path);
                	gtk_tree_view_set_cursor(GTK_TREE_VIEW(treeview), path, NULL, FALSE);
                	gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(treeview), path, NULL, TRUE, 0, 0);
                	gtk_tree_path_free(path);
                }
        }

        if(is_save) {
        	ok_button=ut_button_new(_("Save game"), do_save_game, window, button_box);
  		g_signal_connect(G_OBJECT(treeview), "row_activated", G_CALLBACK(save_row_activated_cb), window);
        } else {
        	ok_button=ut_button_new(_("Load game"), do_load_game, window, button_box);
  		g_signal_connect(G_OBJECT(treeview), "row_activated", G_CALLBACK(load_row_activated_cb), window);
        }

        cancel_button=ut_button_new_stock_swap(GTK_STOCK_CANCEL, gtk_widget_destroy, window, button_box);

  	gtk_box_pack_start(GTK_BOX(vbox), button_box, FALSE, FALSE, 4);

  	gtk_container_add(GTK_CONTAINER(window), vbox);

        gtk_widget_grab_default(ok_button);
        gtk_widget_grab_focus(ok_button);
  	gtk_widget_show_all(window);
}

void save_game_cb(GtkWidget *widget, gpointer data) {
        save_load_game_dialog(widget, data, TRUE);
}

void load_game_cb(GtkWidget *widget, gpointer data) {
        save_load_game_dialog(widget, data, FALSE);
}

void show_hide_next_balls(gboolean show) {
        gint i;

  	if(show) {
        	for(i=0;i<Rules.next;i++) {
      			gtk_widget_show(SmallBalls[i]);
                }
    	} else {
        	for(i=0;i<Rules.next;i++) {
      			gtk_widget_hide(SmallBalls[i]);
                }
    	}
}

void init_names_scores_and_dates (GtkWidget **names, GtkWidget **scores, GtkWidget **dates, int pos) {
  	gint i;
  	gchar *str;

  	if(!read_score(Scoreboard, &FullScoreboard, &FullScoreboardNumber)) {
      		simple_message_box(_("Unable to read score.\n"));
    	}

  	for (i=0;i<10;i++) {
      		str=g_strdup_printf(" %s ", Scoreboard[i].name);
      		names[i]=gtk_label_new(str);
                g_free(str);
      		if(!Scoreboard[i].score) {
                        scores[i]=gtk_label_new("");
      		} else {
			str=g_strdup_printf(" %d ", Scoreboard[i].score);
                        scores[i]=gtk_label_new(str);
                	g_free(str);
                }
      		str=g_strdup_printf(" %s ", Scoreboard[i].date);
      		dates[i]=gtk_label_new(str);
                g_free(str);
    	}
}

void show_hall_of_fame(GtkWidget *widget, gpointer data) {
  	GtkWidget *hall_of_fame;
  	GtkWidget *frame;
  	GtkWidget *table;
  	GtkWidget *vbox,*button_box;
  	GtkWidget *names[10],*scores[10],*dates[10];
  	GtkWidget *close_button;
  	gint i;

	hall_of_fame=ut_window_new(_("Hall of Fame"), "GtkBalls_Scores", "GtkBalls", TRUE, TRUE, FALSE, 5);

  	frame=gtk_frame_new(_(" Hall of Fame "));
  	vbox=gtk_vbox_new(FALSE, 0);

  	button_box=gtk_hbox_new(TRUE, 0);
  	gtk_container_set_border_width(GTK_CONTAINER(vbox), 1);
  	gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 0);

  	table=gtk_table_new(9, 3, FALSE);
  	init_names_scores_and_dates(names, scores, dates, (data!=NULL) ? (gint)data : -1);

  	for(i=0;i<10;i++) {
      		gtk_table_attach_defaults(GTK_TABLE(table), names[i],  0, 1, i, i+1);
      		gtk_table_attach_defaults(GTK_TABLE(table), scores[i], 1, 2, i, i+1);
      		gtk_table_attach_defaults(GTK_TABLE(table), dates[i],  2, 3, i, i+1);
    	}

  	gtk_container_add(GTK_CONTAINER(frame), table);

        close_button=ut_button_new_stock_swap(GTK_STOCK_CLOSE, gtk_widget_destroy, hall_of_fame, button_box);

  	gtk_box_pack_start(GTK_BOX(vbox), button_box, FALSE, FALSE, 4);

  	gtk_container_add(GTK_CONTAINER(hall_of_fame), vbox);

        gtk_widget_grab_default(close_button);
  	gtk_widget_show_all(hall_of_fame);
}

/* shows simple message box */
void simple_message_box(gchar *message) {
  	GtkWidget *dialog;

  	dialog=gtk_message_dialog_new(NULL, 0, GTK_MESSAGE_INFO, GTK_BUTTONS_CLOSE, message);
 	gtk_dialog_run(GTK_DIALOG (dialog));
 	gtk_widget_destroy(dialog);
}

/* returns board coordinate of the window position */
gint calculate_board_coord_x(gint window_coord) {
        if(gtkbTheme) {
  		return (window_coord-1)/gtkbTheme->emptycell.xsize;
        }
        return 0;
}

gint calculate_board_coord_y(gint window_coord) {
        if(gtkbTheme) {
  		return (window_coord-1)/gtkbTheme->emptycell.ysize;
        }
        return 0;
}

void update_rectangle(GtkWidget *widget, gint x, gint y, gint w, gint h) {
  	gdk_draw_drawable(widget->window, widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
			  Pixmap, x, y, x, y, w, h);
}

void set_hi_score_label(void) {
  	gchar *str;

  	str=g_strdup_printf(_("Hi-score: %i"), Hi_score);
  	gtk_label_set_text(GTK_LABEL(Hi_score_label), str);
        g_free(str);
}

void set_user_score_label(void) {
  	gchar *str;

  	if(Score>Hi_score) {
      		Hi_score=Score;
      		set_hi_score_label();
    	}

  	str=g_strdup_printf(_("Your score: %i"), Score);
  	gtk_label_set_text(GTK_LABEL(User_score_label), str);
        g_free(str);
}

void add_point_if_first(struct point *balls_to_delete, gint *number_of_balls_to_delete, gint x, gint y) {
  	gint i;

  	for(i=0;i<*number_of_balls_to_delete;i++) {
    		if((balls_to_delete[i].x==x) && (balls_to_delete[i].y==y)) {
      			return;
                }
        }
  	balls_to_delete[*number_of_balls_to_delete].x=x;
  	balls_to_delete[*number_of_balls_to_delete].y=y;
  	(*number_of_balls_to_delete)++;
}

void clear_line(gint x, gint y, gint length, gint direction, struct point *balls_to_delete, gint *number_of_balls_to_delete) {
  	gint i,x1,y1;

  	switch(direction) {
    		case 0:
      			for(i=y-length;i<y;i++) {
				add_point_if_first(balls_to_delete, number_of_balls_to_delete, x, i);
                        }
      			break;
    		case 1:
      			for(i=x-length;i<x;i++) {
				add_point_if_first(balls_to_delete, number_of_balls_to_delete, i, y);
                        }
      			break;
    		case 2:
      			/* save length */
      			i=length;
      			for(x1=x-1,y1=y-1;length>0;x1--,y1--,length--) {
				add_point_if_first(balls_to_delete, number_of_balls_to_delete, x1, y1);
                        }
      			/* restore length */
      			length=i;
      			break;
    		case 3:
      			i=length;
      			for(x1=x-1,y1=y+1;length>0;x1--,y1++,length--) {
				add_point_if_first(balls_to_delete, number_of_balls_to_delete, x1, y1);
                        }
      			length=i;
      			break;
    		case 4:
      			i=length;
      			for(x1=x-1,y1=y-1;length>0;x1--,y1--,length--) {
				add_point_if_first(balls_to_delete, number_of_balls_to_delete, x1, y1);
                        }
      			length=i;
      			break;
    		case 5:
      			i=length;
      			for(x1=x-1,y1=y+1;length>0;x1--,y1++,length--) {
				add_point_if_first(balls_to_delete, number_of_balls_to_delete, x1, y1);
                        }
      			length=i;
      			break;
    	}
}

int animsort(const void *a, const void *b) {
        if(((const struct gtkb_animarray *)a)->time == ((const struct gtkb_animarray *)b)->time) return 0;
        if(((const struct gtkb_animarray *)a)->time > ((const struct gtkb_animarray *)b)->time) return 1;
	return -1;
}

gint destroy_lines(GtkWidget *widget, GdkPixmap *pixmap, gint count_score) {
  	gint x,y,length;
  	gint x1,y1;
  	gint retval=FALSE;
  	gint number_of_balls_to_delete=0;
  	struct point *balls_to_delete;
  	gint i,j;

  	balls_to_delete=g_malloc(Rules.xsize*Rules.ysize*sizeof(struct point));

  	/* test vertical rows */
  	for(x=0;x<Rules.xsize;x++) {
      		for(length=1,y=1;y<Rules.ysize-1;y++) {
			if(Board[x+y*Rules.xsize] && (Board[x+y*Rules.xsize]==Board[x+(y-1)*Rules.xsize])) {
	    			while((Board[x+y*Rules.xsize]==Board[x+(y-1)*Rules.xsize]) && y<Rules.ysize) {
					length++;
					y++;
	      			}
	    			if(length>=Rules.destroy) {
					clear_line(x, y, length, 0, balls_to_delete, &number_of_balls_to_delete);
					retval=TRUE;
	      			}
	    			length=1;
	  		}
                }
    	}

  	/* test horizontal rows */
  	for(y=0;y<Rules.ysize;y++) {
      		for(length=1,x=1;x<Rules.xsize-1;x++) {
			if((Board[x+y*Rules.xsize] && (Board[x+y*Rules.xsize]==Board[x-1+y*Rules.xsize]))) {
	    			while((Board[x+y*Rules.xsize]==Board[x-1+y*Rules.xsize]) && x<Rules.xsize) {
					length++;
					x++;
	      			}
	    			if(length>=Rules.destroy) {
					clear_line(x, y, length, 1, balls_to_delete, &number_of_balls_to_delete);
					retval=TRUE;
	      			}
	    			length = 1;
	  		}
                }
    	}

  	/* test diagonal rows from the left side to the right */
  	for(y=0;y<Rules.ysize;y++) {
      		/* from the top to the bottom */
      		for(length=1,x1=1,y1=y+1;y1<Rules.ysize-1 && x1<Rules.xsize-1;x1++,y1++) {
			if(Board[x1+y1*Rules.xsize] && (Board[x1+y1*Rules.xsize]==Board[x1-1+(y1-1)*Rules.xsize])) {
	    			while((Board[x1+y1*Rules.xsize]==Board[x1-1+(y1-1)*Rules.xsize]) && x1<Rules.xsize && y1<Rules.ysize) {
					length++;
					x1++;
					y1++;
	      			}
	    			if(length>=Rules.destroy) {
					clear_line(x1, y1, length, 2, balls_to_delete, &number_of_balls_to_delete);
					retval=TRUE;
	      			}
	    			length = 1;
	  		}
                }

      		/* from bottom to top */
      		for (length=1,x1=1,y1=y-1;y1>=0 && x1<Rules.xsize;x1++,y1--) {
			if (Board[x1+y1*Rules.xsize] && (Board[x1+y1*Rules.xsize]==Board[x1-1+(y1+1)*Rules.xsize])) {
	    			while((Board[x1+y1*Rules.xsize]==Board[x1-1+(y1+1)*Rules.xsize]) && x1<Rules.xsize && y1>=0) {
					length++;
					x1++;
					y1--;
	      			}
	    			if(length>=Rules.destroy) {
					clear_line(x1, y1, length, 3, balls_to_delete, &number_of_balls_to_delete);
					retval=TRUE;
	      			}
	    			length = 1;
	  		}
                }
    	}

  	/* test diagonal rows from the top to the bottom */
  	for(x=0;x<Rules.xsize;x++) {
      		/* from the top to the bottom */
      		for(length=1,x1=x+1,y1=1;y1<Rules.ysize-1 && x1<Rules.xsize-1;x1++,y1++) {
			if(Board[x1+y1*Rules.xsize] && (Board[x1+y1*Rules.xsize]==Board[x1-1+(y1-1)*Rules.xsize])) {
	    			while((Board[x1+y1*Rules.xsize]==Board[x1-1+(y1-1)*Rules.xsize]) && x1<Rules.xsize && y1<Rules.ysize) {
					length++;
					x1++;
					y1++;
	      			}
	    			if(length>=Rules.destroy) {
					clear_line(x1, y1, length, 4, balls_to_delete, &number_of_balls_to_delete);
					retval=TRUE;
	      			}
	    			length = 1;
	  		}
                }

      		/* from bottom to top */
      		for(length=1,x1=x+1,y1=Rules.ysize-2;y1>=0 && x1<Rules.xsize;x1++,y1--) {
			if(Board[x1+y1*Rules.xsize] && (Board[x1+y1*Rules.xsize] == Board[x1-1+(y1+1)*Rules.xsize])) {
	    			while((Board[x1+y1*Rules.xsize]==Board[x1-1+(y1+1)*Rules.xsize]) && x1<Rules.xsize && y1>=0) {
					length++;
					x1++;
					y1--;
	      			}
	    			if(length>=Rules.destroy) {
					clear_line(x1, y1, length, 5, balls_to_delete, &number_of_balls_to_delete);
					retval=TRUE;
	      			}
	    			length = 1;
	  		}
                }
    	}

  	if(retval) {
                if(Show_destroy) {
                	gint animcadres=0,animpos=0,animtime;
        	        struct gtkb_animarray *animarray;

	                for(i=0;i<number_of_balls_to_delete;i++) {
                        	animcadres+=gtkbTheme->balls[Board[balls_to_delete[i].x+balls_to_delete[i].y*Rules.xsize]-1].destroyphases+1;
                	}
        	        animarray=g_new0(struct gtkb_animarray, animcadres);
	                for(i=0;i<number_of_balls_to_delete;i++) {
        	                gint color=Board[balls_to_delete[i].x+balls_to_delete[i].y*Rules.xsize];

	                        for(j=0,animtime=0;j<gtkbTheme->balls[color-1].destroyphases;j++) {
                                	animarray[animpos].color=color;
                        	        animarray[animpos].x=balls_to_delete[i].x;
                	                animarray[animpos].y=balls_to_delete[i].y;
        	                        animarray[animpos].phase=j;
	                                animarray[animpos].time=animtime;
                                	animtime+=gtkbTheme->balls[color-1].destroydelays[j];
                        	        animpos++;
                	        }
        	                animarray[animpos].color=0;
	                        animarray[animpos].phase=0;
                        	animarray[animpos].x=balls_to_delete[i].x;
                	        animarray[animpos].y=balls_to_delete[i].y;
        	                animarray[animpos].time=animtime;
	                        animtime+=gtkbTheme->balls[color-1].destroydelays[j];
                        	animpos++;
                	}

        	        qsort(animarray, animcadres, sizeof(struct gtkb_animarray), animsort);

	                for(animtime=0,i=0;i<animcadres;) {
                	        gint isav=i;
        	                my_usleep((animarray[i].time-animtime)*1000);
	                	for(;animtime==animarray[i].time && i<animcadres;i++) {
                        		draw_ball(widget, animarray[i].color, Pixmap, animarray[i].x, animarray[i].y, 0, animarray[i].phase+1);
                		}
                	        gdk_flush();
        	                animtime=animarray[isav].time;
	                }
                	g_free(animarray);
                }

      		for(i=0;i<number_of_balls_to_delete;i++) {
			Board[balls_to_delete[i].x+balls_to_delete[i].y*Rules.xsize]=0;
                }

      		draw_board(widget, pixmap);
      		update_rectangle(DrawingArea, 0, 0, -1, -1);

      		if(count_score) {
	  		Score=Score+Rules.destroy*2+2*(number_of_balls_to_delete-Rules.destroy)*(number_of_balls_to_delete-Rules.destroy);
	  		if (!Show_next_colors) {
	    			Score++;
                        }
	  		set_user_score_label();
		}
    	}

  	g_free(balls_to_delete);
  	return retval;
}

void read_entry(GtkWidget *widget, gpointer data) {
  	struct score_board current_entry;
  	time_t current_time;
  	struct tm *timeptr;
  	gint pos;
        gsize br, bw;
        gchar *tstr;
  
  	user_name=(gchar*)gtk_entry_get_text(GTK_ENTRY(data));
  	if(user_name && *user_name) {
    		strncpy(current_entry.name, user_name, sizeof(current_entry.name));
  	} else {
    		strncpy(current_entry.name, _("Anonymous"), sizeof(current_entry.name));
        }
  	strncpy(last_player_name, user_name, sizeof(last_player_name));
  	current_entry.score=saved_score;
  	current_time=time(NULL);
  	timeptr=localtime(&current_time);
  	if(!timeptr) {
      		simple_message_box(_("Unable to determine current date.\n"));
                strncpy(current_entry.date, _("Unknown"), sizeof(current_entry.date));
    	} else {
      		/* I use _() for format string because some people might want
	 	   format of the date more traditional for their locale */
      		if(!strftime(current_entry.date, sizeof(current_entry.date), _("%a %b %d %H:%M %Y"), timeptr)) {
	  		simple_message_box(_("Unable to determine current date.\n"));
                        strncpy(current_entry.date, _("Unknown"), sizeof(current_entry.date));
		} else {
                	tstr=g_locale_to_utf8(current_entry.date, -1, &br, &bw, NULL);
                        if(!tstr) {
                                strncpy(current_entry.date, _("Unknown"), sizeof(current_entry.date));
                        } else {
                        	strncpy(current_entry.date, tstr, sizeof(current_entry.date));
                                g_free(tstr);
                        }
                }
    	}
  
  	if(!read_score(Scoreboard, &FullScoreboard, &FullScoreboardNumber)) {
      		simple_message_box (_("Unable to read score.\n"));
    	}
  	pos=insert_entry_in_score_board(Scoreboard, current_entry);
  	if(!write_score(Scoreboard, FullScoreboard, FullScoreboardNumber)) {
      		simple_message_box(_("Unable to save score.\n"));
    	}
  	gtk_widget_destroy(dialog);
  	/* show scores to let user see if (s)he's on top ;) */
  	show_hall_of_fame(NULL, (gpointer)pos);
}

gchar *input_name_dialog(void) {
  	GtkWidget *prompt_label;
  	GtkWidget *name;
  	GtkWidget *button;
        gchar *s;

  	/* we have to save score, because they will be set to 0 in new_game() */
  	saved_score=Score;

  	dialog=gtk_dialog_new();
        gtk_window_set_wmclass(GTK_WINDOW(dialog), "GtkBalls_Inputname", "GtkBalls");

  	gtk_container_set_border_width(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), 2);

  	prompt_label=gtk_label_new(_("Enter your name"));
  	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), prompt_label, TRUE, TRUE, 0);

  	name=gtk_entry_new();
        gtk_entry_set_max_length(GTK_ENTRY(name), 14);
  	g_signal_connect(G_OBJECT(name), "activate", G_CALLBACK(read_entry), name);

  	/* restore the last player's name */
        if(!(*last_player_name)) {
                if((s = getenv("USER"))) {
  			strncpy(last_player_name, s, sizeof(last_player_name));
                }
        }
        if(*last_player_name) {
    		gtk_entry_set_text(GTK_ENTRY(name), last_player_name);
    		gtk_editable_select_region(GTK_EDITABLE(name), 0, -1);
        }
  	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), name, TRUE, TRUE, 5);

        button=ut_button_new_stock(GTK_STOCK_OK, read_entry, name, GTK_DIALOG(dialog)->action_area);

        gtk_widget_grab_focus(name);

  	gtk_widget_grab_default(button);
  	gtk_widget_show_all(dialog);

  	return NULL;
}

void rules_ok(GtkWidget *widget, gpointer data) {
        gint oldnext = Rules.next;

        Rules.xsize = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(RD.xrange));
        Rules.ysize = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(RD.yrange));
        Rules.colors = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(RD.crange));
        Rules.next = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(RD.nrange));
        Rules.destroy = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(RD.drange));

	reinit_board(NULL, NULL, 0, oldnext);

  	if(data) {
      		gtk_widget_destroy(GTK_WIDGET(data));
  	}
        save_preferences();
}

void rules_classic(void) {
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(RD.xrange), ClassicRules.xsize);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(RD.yrange), ClassicRules.ysize);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(RD.crange), ClassicRules.colors);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(RD.nrange), ClassicRules.next);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(RD.drange), ClassicRules.destroy);
}

void rules_dialog(void) {
  	GtkWidget *dialog;
  	GtkWidget *frame;
  	GtkWidget *big_vbox, *vbox, *buttons_box;
  	GtkWidget *separator;
  	GtkWidget *ok_button, *cancel_button, *classic_button;

	dialog=ut_window_new(_("Game rules"), "GtkBalls_Rules", "GtkBalls", TRUE, TRUE, TRUE, 5);

  	big_vbox=gtk_vbox_new(FALSE, 0);
  	gtk_container_add(GTK_CONTAINER(dialog), big_vbox);

  	frame=gtk_frame_new(_("Game rules"));
  	gtk_box_pack_start(GTK_BOX(big_vbox), frame, FALSE, FALSE, 0);

	vbox=gtk_vbox_new(FALSE, 0);
  	gtk_container_add(GTK_CONTAINER(frame), vbox);

	RD.xrange=ut_spin_button_new(_("Board width"), 4, 99, Rules.xsize, vbox);
	RD.yrange=ut_spin_button_new(_("Board height"), 4, 99, Rules.ysize, vbox);
	RD.crange=ut_spin_button_new(_("Number of different objects"), 3, gtkbTheme->numballs, Rules.colors, vbox);
	RD.nrange=ut_spin_button_new(_("Number of 'next' objects"), 2, 99, Rules.next, vbox);
	RD.drange=ut_spin_button_new(_("How many balls at line eliminate it"), 3, 99, Rules.destroy, vbox);

	classic_button=ut_button_new(_("Classic rules"), rules_classic, dialog, vbox);

  	separator=gtk_hseparator_new();
  	gtk_box_pack_start(GTK_BOX(big_vbox), separator, FALSE, FALSE, 5);

  	buttons_box=gtk_hbutton_box_new();
        gtk_button_box_set_layout(GTK_BUTTON_BOX(buttons_box), GTK_BUTTONBOX_SPREAD);
  	gtk_box_pack_start(GTK_BOX(big_vbox), buttons_box, TRUE, TRUE, 0);

        ok_button=ut_button_new_stock(GTK_STOCK_OK, rules_ok, dialog, buttons_box);

        cancel_button=ut_button_new_stock_swap(GTK_STOCK_CANCEL, gtk_widget_destroy, dialog, buttons_box);

        gtk_widget_grab_default(ok_button);

  	gtk_widget_show_all(dialog);
}
