/*
 *   Copyright (C) 2007 Tristan Heaven <tristanheaven@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License along
 *   with this program; if not, write to the Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <stdlib.h>
#include <stdbool.h>
#include <string.h>
#include <gtk/gtk.h>
#include <glade/glade.h>

#include "gui.h"
#include "main.h"
#include "hash.h"
#include "list.h"
#include "prefs.h"

static GladeXML *glade;

static void init_hash_funcs(void)
{
	for (int i = 0; i < HASH_N; i++) {
		hash[i].button = GTK_TOGGLE_BUTTON(
			gtk_check_button_new_with_label(hash[i].name));

		// Label the digest outputs
		char *label = g_strdup_printf("%s:", hash[i].name);
		hash[i].label = GTK_LABEL(gtk_label_new(label));
		g_free(label);

		hash[i].entry = GTK_ENTRY(gtk_entry_new());

		gtk_table_attach_defaults(GTK_TABLE(gui_widget("dialog_table")),
			GTK_WIDGET(hash[i].button),
			// Sort checkbuttons into 2 columns
			i % 2 ? 1 : 0,
			i % 2 ? 2 : 1,
			i / 2,
			i / 2 + 1);

		gtk_container_add(GTK_CONTAINER(gui_widget("vbox_labels")),
			GTK_WIDGET(hash[i].label));
		gtk_misc_set_alignment(GTK_MISC(hash[i].label), 0.0, 0.5); // Left align

		gtk_container_add(GTK_CONTAINER(gui_widget("vbox_outputdata")),
			GTK_WIDGET(hash[i].entry));
		gtk_editable_set_editable(GTK_EDITABLE(hash[i].entry), false);

		gtk_widget_show(GTK_WIDGET(hash[i].button));
	}
}

void gui_init(void)
{
	// Load glade file
	if (g_file_test("src/"PACKAGE, G_FILE_TEST_EXISTS))
		glade = glade_xml_new("src/gui.glade", NULL, PACKAGE);
	else
		glade = glade_xml_new(PKGDATADIR "/gui.glade", NULL, PACKAGE);

	g_assert(glade);
	glade_xml_signal_autoconnect(glade);

	init_hash_funcs();
}

GtkWidget *gui_widget(const char *name)
{
	return glade_xml_get_widget(glade, name);
}

int gui_view(void)
{
	if (gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(
		gui_widget("radiomenuitem_file"))))
	{
		return VIEW_FILE;

	} else if (gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(
		gui_widget("radiomenuitem_text"))))
	{
		return VIEW_TEXT;

	} else if (gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(
		gui_widget("radiomenuitem_file_list"))))
	{
		return VIEW_FILE_LIST;

	} else
		g_assert_not_reached();
}

void gui_update(void)
{
	bool has_active = false;

	for (int i = 0; i < HASH_N; i++) {
		if (gtk_toggle_button_get_active(hash[i].button)) {
			gtk_widget_show(GTK_WIDGET(hash[i].label));
			gtk_widget_show(GTK_WIDGET(hash[i].entry));
			has_active = true;
		} else {
			gtk_widget_hide(GTK_WIDGET(hash[i].label));
			gtk_widget_hide(GTK_WIDGET(hash[i].entry));
		}
	}

	list_update();

	if (!has_active)
		prefs_default();
}

void gui_clear(void)
{
	for (int i = 0; i < HASH_N; i++) {
		gtk_entry_set_text(hash[i].entry, "");
		list_clear_digests();
	}

	gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(
		gui_widget("progressbar")), 0.0);
}

void gui_busy(const bool busy)
{
	gui.busy = busy;

	gtk_widget_set_sensitive(gui_widget("menuitem_prefs"), !busy);
	gtk_widget_set_sensitive(gui_widget("radiomenuitem_text"), !busy);
	gtk_widget_set_sensitive(gui_widget("radiomenuitem_file"), !busy);
	gtk_widget_set_sensitive(gui_widget("radiomenuitem_file_list"), !busy);
	gtk_widget_set_sensitive(gui_widget("hbox_input"), !busy);
	gtk_widget_set_sensitive(gui_widget("hbox_output"), !busy);
	gtk_widget_set_sensitive(gui_widget("toolbar"), !busy);

	if (busy) {
		gtk_widget_hide(gui_widget("button_hash"));
		gtk_widget_show(gui_widget("button_stop"));
		gtk_window_set_default(GTK_WINDOW(gui_widget("window")),
			gui_widget("button_stop"));
		gtk_widget_set_sensitive(gui_widget("menuitem_save_as"), false);
	} else {
		gtk_widget_show(gui_widget("button_hash"));
		gtk_widget_hide(gui_widget("button_stop"));
		gtk_window_set_default(GTK_WINDOW(gui_widget("window")),
			gui_widget("button_hash"));
	}
}

void gui_window_resizable(const bool set)
{
	GtkWindow *window = GTK_WINDOW(gui_widget("window"));

	gtk_window_set_resizable(window, set);

	if (set) {
		gtk_window_resize(GTK_WINDOW(gui_widget("window")),
			CLAMP(prefs.width, 1, gdk_screen_width()),
			CLAMP(prefs.height, 1, gdk_screen_height()));
	} else {
		gtk_window_unmaximize(window);
	}
}

bool gui_window_maximised(void)
{
	GtkWidget *window = gui_widget("window");

	if (!window->window)
		return false;

	GdkWindowState state = gdk_window_get_state(window->window);

	return (state & GDK_WINDOW_STATE_MAXIMIZED);
}

void gui_chooser_set_filename(const char *path)
{
	if (!file_exists(path))
		return;

	gtk_file_chooser_set_filename(GTK_FILE_CHOOSER(
		gui_widget("filechooserbutton")), path);

	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(
		gui_widget("radiomenuitem_file")), true);
}

void gui_dialog_hide(void)
{
	prefs_save();
	gui_update();
	gtk_widget_hide(gui_widget("dialog"));
}
