/*
 *   Copyright (C) 2007-2009 Tristan Heaven <tristanheaven@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License along
 *   with this program; if not, write to the Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#endif

#define _LARGEFILE64_SOURCE
#include <sys/stat.h>
#include <stdlib.h>
#include <stdbool.h>
#include <gtk/gtk.h>

#include "main.h"
#include "gui.h"
#include "list.h"
#include "prefs.h"

static struct {
	bool version;
	char **files;
} opts = {
	.version = false,
	.files = NULL
};

static GOptionEntry entries[] = {
	{
		"version", 'v', 0, G_OPTION_ARG_NONE, &opts.version,
		"Display version information", NULL
	},
	{
		G_OPTION_REMAINING, 0, 0, G_OPTION_ARG_FILENAME_ARRAY, &opts.files,
		NULL, NULL
	},
	{ NULL, 0, 0, 0, NULL, NULL, NULL }
};

static void parse_opts(int *argc, char ***argv)
{
	GOptionContext *context = g_option_context_new(_("[FILE...]"));

#if ENABLE_NLS
	g_option_context_add_main_entries(context, entries, PACKAGE);
#else
	g_option_context_add_main_entries(context, entries, NULL);
#endif

	g_option_context_add_group(context, gtk_get_option_group(false));

	if (!g_option_context_parse(context, argc, argv, NULL)) {
		printf(_("Unknown option, try %s --help\n"), g_get_prgname());
		exit(EXIT_FAILURE);
	}

	if (opts.version) {
		printf("%s\n", PACKAGE_STRING);
		exit(EXIT_SUCCESS);
	}

	if (opts.files) {
		if (!opts.files[1])
			gui_chooser_set_filename(opts.files[0]);
		else
			for (int i = 0; opts.files[i]; i++)
				list_append_row(opts.files[i]);
	}

	g_option_context_free(context);
}

int main(int argc, char *argv[])
{
#if ENABLE_NLS
	bindtextdomain(PACKAGE, LOCALEDIR);
	bind_textdomain_codeset(PACKAGE, "UTF-8");
	textdomain(PACKAGE);
#endif

	gtk_init(&argc, &argv);
	gui_init();

	list_init();

	prefs_load();
	atexit(prefs_save);

	parse_opts(&argc, &argv);

	// Show window here so it doesn't resize just after it appears
	gtk_widget_show(gui_get_widget("window"));

	gtk_main();

	return EXIT_SUCCESS;
}

unsigned int get_filesize(const char *filename)
{
	struct stat64 s;

	if (stat64(filename, &s) == -1)
		return 0;
	else
		return s.st_size;
}

bool file_exists(const char *filename)
{
	return (g_file_test(filename, G_FILE_TEST_EXISTS) &&
		!g_file_test(filename, G_FILE_TEST_IS_DIR));
}
