/* ----------------------------------------------------------------------------
 * gtkpbinit.c
 * funtions that prepare program workflow and initialization
 *
 * Copyright 2002 Matthias Grimm
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 * ----------------------------------------------------------------------------*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <syslog.h>
#include <signal.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <getopt.h>
#include <sys/stat.h>

#ifdef HAVE_LANGINFO_CODESET
#  include <langinfo.h>
#endif

#include <pbb.h>

#include "gtkpbbinit.h"
#include "gtkinterface.h"
#include "audio.h"
#include "themes.h"
#include "gettext_macros.h"

extern volatile sig_atomic_t prgexit;

/* This function does the first initialization of the init-structure. Some values
   are used very early to decide if some cleanup is necessary or not. */
void init_init(struct init *md, int argc, char **argv)
{
	md->themename = DEFAULT_THEME;
	md->themedata = NULL;
	md->timeout = 0;
	md->argc = argc;   /* save command line arguments */
	md->argv = argv;

	md->dsp.active = 0;
	md->dsp.audiodev = DEFAULT_AUDIO;
	md->dsp.sample = NULL;   /* no sound to play */
}

int prg_init(struct init *md)
{
	int err;

	if ((md->themedata = theme_init(md->themename)) == NULL)
		return 1;

	if ((md->window = popup_create_window(md->themedata)) == NULL)
		return 1;
	
	if ((err = ipc_init ("GtkPBButtons", LIBMODE_CLIENT, CLIENT_REGISTER | CLIENT_REPLACEMENT)) != 0) {
		if (err == E_NOSERVER)
			print_msg (PBB_ERR, _("Server message port not found. Server isn't running.\n"));
		else if (err == E_REGISTER)
			print_msg (PBB_ERR, _("Sending registration to server failed.\n"));
		else if (err == E_MSGPORT)
			print_msg (PBB_ERR, _("Can't create message port for client.\n"));
		return err;
	}

#ifdef HAVE_SOUND
	if (md->themedata->havesound) {
		if ((init_sound_thread(&md->dsp)) == -1)
			print_msg (PBB_WARN, _("Problems initializing sound subsystem. Beep disabled.\n"));
		else
			md->dsp.active = 1;
	}
#endif
	return 0;
}

/* --- Signal Handler --- */

void
signalhandler (int signum)
{
	prgexit = 1;
}

int
install_sighandler ()
{
	struct sigaction sa = { {signalhandler}, {{0}}, SA_RESTART, 0 };

	/* install the signal handlers */
	if (!sigaction (SIGINT, &sa, NULL))
		if (!sigaction (SIGTERM, &sa, NULL))
			return 0;
	print_msg (PBB_ERR, _("Can't install signal handler\n"));
	return -1;
}

int
evaluate_args (struct init *md, struct poptArgs *args)
{
	int err;
	char *prgname;
	
#if defined(ENABLE_NLS) && defined(HAVE_LANGINFO_CODESET)
	bind_textdomain_codeset (GETTEXT_PACKAGE, nl_langinfo(CODESET));
#endif

	if (args->version == 1) {
		printf(_("%s, version %s"), PACKAGE, VERSION);
		printf(", (c) 2002-2004 Matthias Grimm\n");
		return E_INFO;
	}

	if((prgname = strrchr(md->argv[0],'/')) == NULL)
		prgname = md->argv[0];
	else prgname++;		/* ignore first slash*/
	
	if (args->detach == 1)
		prepare_daemon (prgname, args->pidfile, PBBDF_FORCE);

	if (is_theme(args->theme) != 0)
		printf(_("ERROR: %s is not a valid theme file; using default.\n"), args->theme);
	else
		md->themename = args->theme;
	
	if ((err = check_path (args->audiodev, TYPE_CHARDEV)) == 0)
		md->dsp.audiodev = args->audiodev;
	
#ifdef ENABLE_NLS
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
#endif
	return err;
}

int
check_path (char *source, int type)
{
	int rc = 0;

	rc = check_devorfile (source, type);
	switch (rc) {
	case E_NOEXIST:
		print_msg (PBB_ERR, _("Can't access the file '%s': %s.\n"), source, strerror(errno));
		break;
	case E_NOFILE:
		print_msg (PBB_ERR, _("The file '%s' is not a file.\n"), source);
		break;
	case E_NOCHAR:
		print_msg (PBB_ERR, _("The file '%s' is not a character device.\n"), source);
		break;
	case E_NOBLK:
		print_msg (PBB_ERR, _("The file '%s' is not a block device.\n"), source);
		break;
	}
	return rc;
}

