/* ----------------------------------------------------------------------------
 * theme_macosx.c
 * functions to create the layout for the MacOSX theme.
 *
 * Copyright 2002 Matthias Grimm
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 * ----------------------------------------------------------------------------*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <math.h>
#include <gtk/gtk.h>

#include "gettext_macros.h"
#include "themes.h"

struct layout_macosx {
	GtkWidget	*pbarempty_pixmap;
	GtkWidget	*pbarfull_pixmap;
	int		pbarsteps;
	gchar		*textformat;
	
	GtkWidget	*brightness_pixmap;
	GtkWidget	*volume_pixmap;
	GtkWidget	*mute_pixmap;
	GtkWidget	*mouse_pixmap;
	GtkWidget	*tpmodes[4];
	GtkWidget	*battery_pixmap;
	GtkWidget	*sleep_pixmap;
	GtkWidget	*noaudio_pixmap;
	GtkWidget	*kbdillu_pixmap;
	GtkWidget	*cdrom_pixmap;
	GtkWidget	*mirror_pixmap;

	GdkPixbuf	*pbarempty_pixbuf;
	GdkPixbuf	*pbarfull_pixbuf;

	int		current_popup;    /* popup currently prepared for */
	int		barwidth;         /* length of barimage */
	GtkWidget	*align1;
	GtkWidget	*align2;
	GtkWidget	*barimage;
	GtkWidget	*tpimage;
	GtkWidget	*message;
	GtkWidget	*pixmap;
};

/* private prototypes */
int layout_macosx_get_popup_type(int popup);

int
layout_macosx_init (struct theme *td)
{
	int err, n;
	static struct layout_macosx te;
	struct themeoption themedef[] = {
		{GLOBAL,     "pbarsteps", theme_int,   &te.pbarsteps},
		{GLOBAL,     "textfmt",   theme_string,&te.textformat},
		{GLOBAL,     "pbarempty", theme_image, &te.pbarempty_pixmap},
		{GLOBAL,     "pbarfull",  theme_image, &te.pbarfull_pixmap},
		{BRIGHTNESS, "image",     theme_image, &te.brightness_pixmap},
		{VOLUME,     "image",     theme_image, &te.volume_pixmap},
		{MUTE,       "image",     theme_image, &te.mute_pixmap},
		{MOUSE,      "image",     theme_image, &te.mouse_pixmap},
		{MOUSE,      "mode0",     theme_image, &te.tpmodes[0]},
		{MOUSE,      "mode1",     theme_image, &te.tpmodes[1]},
		{MOUSE,      "mode2",     theme_image, &te.tpmodes[2]},
		{MOUSE,      "mode3",     theme_image, &te.tpmodes[3]},
		{BATTERY,    "image",     theme_image, &te.battery_pixmap},
		{SLEEP,      "image",     theme_image, &te.sleep_pixmap},
		{NOAUDIO,    "image",     theme_image, &te.noaudio_pixmap},
		{KBDILLU,    "image",     theme_image, &te.kbdillu_pixmap},
		{CDROM,      "image",     theme_image, &te.cdrom_pixmap},
		{CRTMIRROR,  "image",     theme_image, &te.mirror_pixmap},
	};

	/* This loop works only for the same datatype; in this case
	 * GtkWidget. All other datatypes must be excluded from the
	 * loop and initialised separately.
	 */
	for (n=2; n < sizeof(themedef) / sizeof(struct themeoption)-1; n++)
		* (GtkWidget**) themedef[n].data = NULL;

	te.pbarsteps = 0;
	te.textformat= NULL;
	te.barimage  = NULL;
	te.tpimage   = NULL;
	te.message   = NULL;
	te.pixmap    = NULL;

	err = theme_load_data(td, themedef, sizeof(themedef) / sizeof(struct themeoption));
	if (err != 0) {
		layout_macosx_exit (td);
		return -1;
	} else {
		td->user_data = (void*) &te;

		if ((te.pbarempty_pixmap) && (te.pbarfull_pixmap)) {
			te.pbarempty_pixbuf = gtk_image_get_pixbuf (GTK_IMAGE(te.pbarempty_pixmap));
			te.pbarfull_pixbuf  = gtk_image_get_pixbuf (GTK_IMAGE(te.pbarfull_pixmap));
			if (!te.pbarempty_pixbuf || !te.pbarfull_pixbuf)
				te.pbarempty_pixbuf = te.pbarfull_pixbuf = NULL;
		} else
			te.pbarempty_pixbuf = te.pbarfull_pixbuf = NULL;

		return 0;
	}
}

void
layout_macosx_exit (struct theme *td)
{
	struct layout_macosx *te = (struct layout_macosx*) td->user_data;
	int n;

	if (te != NULL) {
		if (te->textformat != NULL)
			g_free(te->textformat);
		if (te->pbarempty_pixmap != NULL)
			g_object_unref(te->pbarempty_pixmap);
		if (te->pbarfull_pixmap != NULL)
			g_object_unref(te->pbarfull_pixmap);
		if (te->brightness_pixmap != NULL)
			g_object_unref(te->brightness_pixmap);
		if (te->volume_pixmap != NULL)
			g_object_unref(te->volume_pixmap);
		if (te->mute_pixmap != NULL)
			g_object_unref(te->mute_pixmap);
		if (te->mouse_pixmap != NULL)
			g_object_unref(te->mouse_pixmap);
		if (te->battery_pixmap != NULL)
			g_object_unref(te->battery_pixmap);
		if (te->sleep_pixmap != NULL)
			g_object_unref(te->sleep_pixmap);
		if (te->noaudio_pixmap != NULL)
			g_object_unref(te->noaudio_pixmap);
		if (te->kbdillu_pixmap != NULL)
			g_object_unref(te->kbdillu_pixmap);
		if (te->cdrom_pixmap != NULL)
			g_object_unref(te->cdrom_pixmap);
		if (te->mirror_pixmap != NULL)
			g_object_unref(te->mirror_pixmap);

		for(n=0; n < 3; n++)
			if (te->tpmodes[n] != NULL)
				g_object_unref(te->tpmodes[n]);

		if (te->message != NULL)
			g_object_unref(te->message);
	}
}

int
layout_macosx_attach (struct theme *td, GtkWidget *window)
{
	struct layout_macosx *te = (struct layout_macosx*) td->user_data;

	GdkPixbuf *bgpixbuf; 
	GtkWidget *vbox;
	GtkWidget *align1, *align2;
	GtkWidget *message;
	int height = 0;

	if (td->bgimage)
		if ((bgpixbuf = gtk_image_get_pixbuf (GTK_IMAGE(td->bgimage))))
			height = gdk_pixbuf_get_height (bgpixbuf); 

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (window), vbox);

	align1 = gtk_alignment_new(0.5,0.7,0.0,0.0);
	gtk_box_pack_start (GTK_BOX (vbox), align1, TRUE, TRUE, 0);
	gtk_container_add(GTK_CONTAINER(align1), te->sleep_pixmap);
	if (height)
		gtk_widget_set_size_request(GTK_WIDGET(te->sleep_pixmap), -1, (int) height * 0.7); 

	align2 = gtk_alignment_new(0.5,0.0,0.0,0.0);
	gtk_box_pack_start (GTK_BOX (vbox), align2, TRUE, TRUE, 0);
	message = gtk_label_new ("dummy");
	gtk_widget_ref(message);
	gtk_label_set_justify (GTK_LABEL(message), GTK_JUSTIFY_CENTER);
	gtk_container_add(GTK_CONTAINER(align2), message);

	te->current_popup	= POPUP_SLEEP;
	te->align1		= align1;
	te->align2		= align2;
	te->message		= message;
	te->pixmap		= te->sleep_pixmap;
	return 0;
}

void
layout_macosx_update (struct theme *td, GtkWidget *window, int popup, ...)
{
	struct layout_macosx *te = (struct layout_macosx*) td->user_data;
	GtkWidget *pixmap = NULL;
	GdkPixbuf *tmppixbuf;
	gchar *str = NULL, *tmpstr1, *tmpstr2;
	int x, y, l, arg0, arg1, tpmode = 0;
	gdouble	frac = 0.0;
	va_list list;
	char *label1line = "<span %s>%s</span>";
	char *label2line = "<span %s>%s\n%s</span>";

	va_start(list, popup);

	switch (popup) {
	case POPUP_BRIGHTNESS:
		arg0	 = va_arg(list, int);
		if (arg0 < 0) arg0 = 0;
		pixmap	 = te->brightness_pixmap;
		frac	 = (gdouble) arg0 / BACKLIGHT_MAX;
		break;
	case POPUP_VOLUME:
		arg0 = va_arg(list, int);
		frac = (gdouble) arg0 / VOLUME_MAX;
		pixmap = te->volume_pixmap;
		break;
	case POPUP_MUTE:
		pixmap = te->mute_pixmap;
		frac = 0.0;
		break;
	case POPUP_NOAUDIO:
		pixmap = te->noaudio_pixmap;
		str = g_strdup_printf(label1line, te->textformat, _("Sorry, no sound"));
		break;
	case POPUP_MOUSE:
		tpmode = va_arg(list, int) & 3;
		pixmap = te->mouse_pixmap;
		break;
	case POPUP_BATTERY:
		pixmap = te->battery_pixmap;

		arg0 = va_arg(list, int);
		arg1 = va_arg(list, int);

		tmpstr1 = g_strdup_printf(_("Warnlevel %d"), arg0);
		if (arg0 == 4)
			tmpstr2 = g_strdup(_("Shutdown now"));
		else if (arg1 == -1)
			tmpstr2 = g_strdup(_("?? minutes left"));
		else
			tmpstr2 = g_strdup_printf(dngettext (PACKAGE, "One minute left",
				"%d minutes left", arg1), arg1);
		str = g_strdup_printf(label2line, te->textformat, tmpstr1, tmpstr2);
		g_free(tmpstr1);
		g_free(tmpstr2);
		break;
	case POPUP_SLEEP:
		pixmap = te->sleep_pixmap;

		arg0 = va_arg(list, int);
		if (arg0 == 0)
			tmpstr1 = g_strdup_printf(_("Initiate sleep"));
		else {
			tmpstr1 = g_strdup_printf(dngettext (PACKAGE,
				"Sleep in %d second", "Sleep in %d seconds", arg0), arg0);
		}
		str = g_strdup_printf(label1line, te->textformat, tmpstr1);
		g_free(tmpstr1);
		break;
	case POPUP_KBDILLU:
		arg0	 = va_arg(list, int);
		pixmap	 = te->kbdillu_pixmap;
		frac	 = (gdouble) arg0 / KBDILLU_MAX;
		break;
	case POPUP_CDROM:
		pixmap = te->cdrom_pixmap;
		arg0 = va_arg(list, int);
		switch (arg0) {
		case 0:
		default:
			str = g_strdup_printf(label1line, te->textformat, _("CDROM busy."));
			break;
		case 1:
			str = g_strdup_printf(label1line, te->textformat, _("Ejecting ..."));
			break;
		case 2:
			str = g_strdup_printf(label1line, te->textformat, _("Closing Tray."));
			break;
		}
		break;
	case POPUP_CRTMIRROR:
		arg0 = va_arg(list, int) & 1;
		pixmap = te->mirror_pixmap;
		if (arg0 == 0)
			str = g_strdup_printf(label1line, te->textformat, _("Ext. Video off"));
		else
			str = g_strdup_printf(label1line, te->textformat, _("Ext. Video on"));	
		break;
	}

	arg0 = layout_macosx_get_popup_type(popup);
	arg1 = layout_macosx_get_popup_type(te->current_popup);

	if (te->current_popup != popup) { /* popup changed? */
		if (te->pixmap)
			gtk_container_remove (GTK_CONTAINER(te->align1), te->pixmap);
		if (pixmap)
			gtk_container_add(GTK_CONTAINER(te->align1), pixmap);
		te->pixmap = pixmap;

		if (arg0 != arg1) { /* popup type changed? */
			switch (arg1) {  /* remove old widget not longer needed */
			case POPUP_TYPE_PROGRESS:
				gtk_container_remove (GTK_CONTAINER(te->align2), te->barimage);
				te->barimage = NULL;
				break;
			case POPUP_TYPE_MESSAGE:
				gtk_container_remove (GTK_CONTAINER(te->align2), te->message);
				break;
			case POPUP_TYPE_IMAGE:
				if (te->tpimage) {
					gtk_container_remove (GTK_CONTAINER(te->align2), te->tpimage);
					te->tpimage = NULL;
				}
				break;
			}
			if (arg0 == POPUP_TYPE_MESSAGE)
				gtk_container_add(GTK_CONTAINER(te->align2), te->message);
		}
		te->current_popup = popup;
	}

	switch (arg0) {  /* popup type of current popup */
	case POPUP_TYPE_PROGRESS:
		if (te->pbarempty_pixbuf) {
			x = gdk_pixbuf_get_width (te->pbarempty_pixbuf);
			y = gdk_pixbuf_get_height (te->pbarempty_pixbuf);
			if (te->pbarsteps)
				frac = ceil(frac * te->pbarsteps) / te->pbarsteps;
			l = (int) x * frac; /* bar length */

			tmppixbuf = gdk_pixbuf_copy(te->pbarempty_pixbuf);
			if (tmppixbuf) {
				gdk_pixbuf_copy_area (te->pbarfull_pixbuf, 0, 0,
                	l, y, tmppixbuf, 0, 0);
				if (te->barimage)
					gtk_container_remove (GTK_CONTAINER(te->align2), te->barimage);
				if ((te->barimage = gtk_image_new_from_pixbuf (tmppixbuf)))
					gtk_container_add(GTK_CONTAINER(te->align2), te->barimage);
				g_object_unref(tmppixbuf);
			}
		}
		break;
	case POPUP_TYPE_MESSAGE:
		gtk_label_set_markup(GTK_LABEL(te->message), str);
		g_free(str);
		break;
	case POPUP_TYPE_IMAGE:
		if (te->tpimage)
			gtk_container_remove (GTK_CONTAINER(te->align2), te->tpimage);
		te->tpimage = te->tpmodes[tpmode];
		gtk_container_add(GTK_CONTAINER(te->align2), te->tpimage);
		break;
	}
	va_end(list);
}

int
layout_macosx_get_popup_type(int popup)
{
	int	popup_types[POPUPS];

	popup_types[POPUP_BRIGHTNESS]	= POPUP_TYPE_PROGRESS;
	popup_types[POPUP_VOLUME]	= POPUP_TYPE_PROGRESS;
	popup_types[POPUP_MUTE]		= POPUP_TYPE_PROGRESS;
	popup_types[POPUP_MOUSE]	= POPUP_TYPE_IMAGE;
	popup_types[POPUP_BATTERY]	= POPUP_TYPE_MESSAGE;
	popup_types[POPUP_SLEEP]	= POPUP_TYPE_MESSAGE;
	popup_types[POPUP_NOAUDIO]	= POPUP_TYPE_MESSAGE;
	popup_types[POPUP_KBDILLU]	= POPUP_TYPE_PROGRESS;
	popup_types[POPUP_CDROM]	= POPUP_TYPE_MESSAGE;
	popup_types[POPUP_CRTMIRROR] = POPUP_TYPE_MESSAGE;

	return popup_types[popup];
}
