/*
 * Common environment for most program functions
 *
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#define APPNAME PACKAGE
#else
#warning No config.h found: using fallback values
#define APPNAME "missing appname"
#define VERSION "unknown"
#endif

#include "GuessnetEnvironment.h"
#include "CommandlineParser.h"
#include "Regexp.h"

#include "GuessnetParser.h"
#include "IfaceParser.h"

#include <set>

#include <stdio.h>
#include <stdarg.h>

using namespace std;

static GuessnetEnvironment* instance;

GuessnetEnvironment& GuessnetEnvironment::get() throw ()
{
	return *instance;
}


class StandaloneEnvironment : public GuessnetEnvironment, public ScanConsumer
{
protected:
	string configFile;

public:
	StandaloneEnvironment(CommandlineParser& opts)
	{
		// Find out the configuration file to use
		if (opts.get_argc() > 2)
			configFile = opts.get_argv()[2];
		else if (opts.get("configfile").defined())
			configFile = opts.get("configfile").stringVal();

		// Set verbosity
		verbose(opts.get("verbose").defined());
		debug(opts.get("debug").defined());


		// Find out the interface to be tested
		if (opts.get_argc() > 1)
			iface(opts.get_argv()[1]);
		else
			iface("eth0");


		// Find out the default profile name
		if (opts.get("defprof").defined())
		{
			defprof(opts.get("defprof").stringVal());
			::verbose("Default profile set to `%.*s'\n", PFSTR(defprof()));
		}

		// Find out the scan timeout
		if (opts.get("timeout").defined())
			timeout(opts.get("timeout").intVal());

		// Find out the init timeout
		if (opts.get("inittime").defined())
			initTimeout(opts.get("inittime").intVal());

	}

	virtual void handleScan(const Scan* scan) throw ();

	virtual void parseConfigFile();
};

void StandaloneEnvironment::handleScan(const Scan* scan) throw ()
{
	_scans.push_back(scan);
}

void StandaloneEnvironment::parseConfigFile()
{
	/* Open the specified config file or stdin if not specified */
	FILE* input = stdin;
	if (!configFile.empty())
	{
		input = fopen(configFile.c_str(), "rt");
		if (!input)
			throw FileException(errno, "opening " + configFile);
	}

	GuessnetParser::parse(input, *this);

	if (input != stdin)
		fclose(input);
}


class Tokenizer
{
protected:
	std::string str;
	std::string::size_type s;
public:
	Tokenizer(const std::string& str) throw ()
		: str(str), s(0) {}

	std::string next()
	{
		// Skip leading spaces
		while (s < str.size() && isspace(str[s]))
			s++;

		if (s == str.size()) return string();
		
		string::size_type start = s;

		while (s < str.size() && !isspace(str[s]))
			s++;

		return str.substr(start, s - start);
	}
};


class IfupdownEnvironment : public StandaloneEnvironment
{
protected:
	set<string> ifupdownProfiles;

public:
	IfupdownEnvironment(CommandlineParser& opt) :
		StandaloneEnvironment(opt)
	{
		// We have a default config file name in ifupdown mode
		if (configFile.empty())
			configFile = "/etc/network/interfaces";

		::debug("program name is guessnet-ifupdown: enabling ifupdown mode\n");

		// Read stuff from stdin
		ExtendedRegexp null_line("^[[:blank:]]*(#.*)?$");
		ExtendedRegexp parm_line(
				"^[[:blank:]]*([A-Za-z_-]+):[[:blank:]]*(.+)$", 3);

		string line;
		int linenum = 1;
		int found = 0;
		int c;
		while ((c = fgetc(stdin)) != EOF)
		{
			if (c != '\n')
				line += c;
			else
			{
				if (null_line.match(line))
				{
					//fprintf(stderr, "EMPTY\n");
				}
				else if (parm_line.match(line))
				{
					string name = parm_line[1];
					Tokenizer t(parm_line[2]);
					string val = t.next();

					if (name == "default")
					{
						if (!val.empty())
							defprof(val);
					}
					else if (name == "verbose")
					{
						if (!val.empty())
							verbose(val == "true");
					}
					else if (name == "debug")
					{
						if (!val.empty())
							debug(val == "true");
					}
					else if (name == "timeout")
					{
						int v = atoi(val.c_str());
						if (v > 0)
							timeout(v);
					}
					else if (name == "init-time")
					{
						int v = atoi(val.c_str());
						if (v > 0)
							initTimeout(v);
					}
				}
				else
				{
					Tokenizer t(line);
					for (string w = t.next(); !w.empty(); w = t.next())
						ifupdownProfiles.insert(w);
				}
				line = string();
				linenum++;
			}
		}
	}

	virtual void handleScan(const Scan* scan) throw ();
	virtual void parseConfigFile();
};

void IfupdownEnvironment::handleScan(const Scan* scan) throw ()
{
	if (ifupdownProfiles.size() == 0 || ifupdownProfiles.find(scan->name()) != ifupdownProfiles.end())
		_scans.push_back(scan);
}

void IfupdownEnvironment::parseConfigFile()
{
	/* Open the specified config file or stdin if not specified */
	FILE* input = fopen(configFile.c_str(), "rt");;
	if (!input)
		throw FileException(errno, "opening " + configFile);

	IfaceParser::parse(input, *this);

	fclose(input);
}

void GuessnetEnvironment::init(int argc, const char* argv[])
{
	CommandlineParser opts(APPNAME, "[options] [ethernet_interface]", "Guess the current network location");
	opts.add("version", 'V', "version", "print the program version, then exit");
	opts.add("configfile", 'C', "config-file",
			"name of the configuration file to read (default: stdin or"
			"/etc/network/interfaces in ifupdown mode", "file");
	opts.add("ifupdown", 'i', "ifupdown-mode",
			"use /etc/network/interfaces file instead of the usual"
			" guessnet configuration file");
	opts.add("defprof", 'd', "default",
			"profile name to report if no known networks are found"
			" (defaults to \"none\")", "name");
	opts.add("timeout", 't', "timeout",
			"timeout (in seconds) used to wait for response packets"
			" (defaults to 5 seconds)", "seconds");
	opts.add("inittime", 0, "init-timeout",
			"time (in seconds) to wait for the interface to initialize"
			" when not found already up (defaults to 3 seconds)");
	opts.add("verbose", 'v', "verbose",
			"enable verbose operations");
	opts.add("debug", 0, "debug", 
			"enable debugging output");

	// Process the commandline
	if (!opts.parse(argc, argv))
	{
		opts.printHelp();
		exit(1);
	}
	if (opts.get("help").defined())
	{
		opts.printHelp();
		exit(0);
	}
	if (opts.get("version").defined())
	{
		printf("%s ver." VERSION "\n", APPNAME);
		exit(0);
	}

	// Check user id
	/*
	if (geteuid() != 0)
		fatal_error("You must run this command as root.");
	*/


	// Find out wether we should run in ifupdown mode
	bool ifupdown_mode = opts.get("ifupdown").defined();
	if (!ifupdown_mode)
	{
		const char* pname = strrchr(argv[0], '/');
		pname = pname ? pname + 1 : argv[0];
		if (strcmp(pname, "guessnet-ifupdown") == 0)
			ifupdown_mode = true;
	}

	if (ifupdown_mode)
	{
		instance = new IfupdownEnvironment(opts);
	} else {
		instance = new StandaloneEnvironment(opts);
	}
	Environment::init(instance);
	
	instance->parseConfigFile();
}


// vim:set ts=4 sw=4:
