#ifndef THREAD_H
#define THREAD_H

/*
 * OO encapsulation of Posix threads
 *
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <Exception.h>
#include <pthread.h>

/**
 * Encapsulates a thread
 *
 * FIXME:  C++ resource management and thread cancellation
 *         C++ uses the "resource allocation is instantiation" way of managing
 *         resources: when a function exits, either by terminating or because
 *         an exception has been raised, destructors are called.
 *         However, when a thread is cancelled, the flow of control is
 *         interrupted and no exceptions are thrown.  Cleanup should be
 *         performed by registering cleanup functions, but a cleanup function
 *         can't usefully throw exceptions nor call destructors.
 *         At the moment, I don't know what to do to correctly release
 *         resources on thread cancellation.  I'm waiting for new ideas.
 */

class Thread
{
protected:
	pthread_t thread;

	/// Short tag describing this thread, used in error messages and
	/// identification
	virtual const char* threadTag() { return "generic"; }
	
	/// Main thread function, executed in the new thread after creation.
	/// When main() exits, the new thread ends and main() result will be the
	/// thread exit result
	virtual void* main() throw () = 0;

	/// Callback function used to start the thread
	static void* Starter(void* parm) throw ();

public:
	Thread() throw () {}
	virtual ~Thread() throw () {}

	/// Start the thread
	void start() throw (SystemException);

	/// Start the thread in the detached state
	void startDetached() throw (SystemException);

	/// Join the thread
	void* join() throw (SystemException);

	/// Put the thread in the detached state
	void detach() throw (SystemException);

	/// Send a cancellation request to the thread
	void cancel() throw (SystemException);

	/// Sent a signal to the thread
	void kill(int signal) throw (SystemException);
};

// vim:set ts=4 sw=4:
#endif
