/*
 * Guess the current network location using a wide selection of scans
 *
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 * Mostly rewritten by Enrico Zini on May 2003
 * Originally based on laptop-netconf.c by Matt Kern <matt@debian.org>
 * That was in turn based on divine.c by Felix von Leitner <felix@fefe.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#define APPNAME PACKAGE
#else
#warning No config.h found: using fallback values
#define APPNAME __FILE__
#define VERSION "unknown"
#endif

#include "PeerScanner.h"
#include "ScriptScanner.h"
#include "Mutex.h"
#include "GuessnetEnvironment.h"
#include "ScanBag.h"
#include "stringf.h"

#include "IFace.h"

#include <stdio.h>
#include <ctype.h>
#include <errno.h>	/* errno */

#include <string.h>     // memcpy
#include <sys/types.h>  // socket, getpwuid, geteuid
#include <sys/socket.h> // socket
#include <sys/ioctl.h>  // ioctl
#include <net/if.h>
#include <unistd.h>     // close, geteuid
#include <pwd.h>		// getpwuid


#include <set>
#include <vector>

#include "Thread.h"

using namespace std;
using namespace stringf;

class ScanRunner : public ScannerListener, public ScanConsumer
{
protected:
	Mutex waitMutex;
	Condition waitCond;
	vector<string> candidates;

	int cand_count;

	ScanBag scans;

public:
	ScanRunner() throw () : cand_count(0) {}
		
	/// Get the number of candidate profiles
	int candidateCount() const throw () { return cand_count; }
	
	/// Add `scan' to the list of scans to perform
	virtual void handleScan(const Scan* scan) throw ()
	{
		scans.add(scan);
	}

	/// Notify that the scan `scan' has been successful
	virtual void notifyScan(const Scan* scan) throw ()
	{
		try {
			MutexLock lock(waitMutex);
			candidates = scans.notify(scan);
			if (candidates.size() == 1)
				waitCond.broadcast();
		} catch (Exception& e) {
			error("%s: %.*s\n", e.type(), PFSTR(e.desc()));
		}
	}

	/// Wait for an answer from the scanners for no more than `timeout' milliseconds
	string getResult(int timeout) throw ()
	{
		struct timeval before;
		gettimeofday(&before, 0);
		struct timespec abstime;
		abstime.tv_sec = before.tv_sec;
		abstime.tv_nsec = before.tv_usec * 1000;

		abstime.tv_sec += timeout / 1000;
		abstime.tv_nsec += (timeout % 1000) * 1000000;
		if (abstime.tv_nsec > 1000000000)
		{
			abstime.tv_sec++;
			abstime.tv_nsec -= 1000000000;
		}

		MutexLock lock(waitMutex);

		if (candidates.size() == 1)
			return candidates[0];
		//else
			//return scans.getLessSpecific();

		waitCond.wait(lock, abstime);

		if (candidates.size() == 1)
			return candidates[0];
		else
			return scans.getLessSpecific();
	}
};

class FakeScanner : public ScanRunner, Thread
{
protected:
	bool _canceled;
	
	virtual void* main() throw ()
	{
		interact();
	}
	
public:
	FakeScanner() throw () : _canceled(false) {}
	~FakeScanner() throw () {}

	bool canceled() const throw () { return _canceled; }
	bool canceled(bool val) throw () { return _canceled = val; }
	
	void shutdown() throw (Exception) { canceled(true); }
	void startScans() throw (SystemException, libnetException, pcapException)
	{
		printf("Scans contents:\n");
		scans.dump();
		
		for (list<const Scan*>::const_iterator i = scans.getScans().begin();
				i != scans.getScans().end(); i++)
		{
			const Scan* scan = *i;
			if (const PeerScan* s = dynamic_cast<const PeerScan*>(scan))
			{
				debug("Will scan network %.*s for %.*s (mac %.*s)\n",
						PFSTR(s->name()), PFSTR(fmt(s->ip())), PFSTR(fmt(s->mac())));
				cand_count++;
			}
			else if (const ScriptScan* s = dynamic_cast<const ScriptScan*>(scan))
			{
				debug("Will use script %.*s to test %.*s\n",
						PFSTR(s->cmdline()), PFSTR(s->name()));
				cand_count++;
			}
			else if (const LinkBeatScan* s = dynamic_cast<const LinkBeatScan*>(scan))
			{
				//warning("Link-beat detection currently disabled\n");
				debug("Will test for link beat.  If absent, will return %.*s\n",
						PFSTR(s->name()));
				cand_count++;
			}
			else if (const DefaultScan* s = dynamic_cast<const DefaultScan*>(scan))
			{
				debug("Default scan is %.*s\n",
						PFSTR(s->name()));
				cand_count++;
			}
			else
				//printf("Unknown scan %p\n", scan);
				printf("Unknown scan %.*s: %.*s\n", PFSTR(scan->name()), PFSTR(scan->signature()));
		}

		start();
	}

	void interact() throw ()
	{
		FILE* in = fopen("/dev/tty", "rt");

		// To be run in a separate thread
		while (!_canceled)
		{
			int num = 1;
			const list<const Scan*>& cands = scans.getScans();
			for (list<const Scan*>::const_iterator i = cands.begin();
					i != cands.end(); i++)
			{
				printf("%2d: %.*s (%.*s)\n", num++, PFSTR((*i)->name()), PFSTR((*i)->signature()));
			}
			printf(" 0: quit\n");
			printf("> ");
			int res;
			fscanf(in, "%d", &res);
			if (res == 0)
				canceled(true);
			res--;
			if (res > 0 && res < cands.size())
			{
				list<const Scan*>::const_iterator i = cands.begin();
				for (int j = 0; j < res; j++)
					i++;
				notifyScan(*i);
			}
		}
		fclose(in);
	}
};

class MainScanner : public ScanRunner
{
protected:
	// Scanning services
	NetSender* sender;
	NetWatcher* watcher;
	ProcessRunner* runner;

	PeerScanner* peerScanner;
	ScriptScanner* scriptScanner;

	IFace& iface;

	void startPeerScanner() throw (SystemException, libnetException, pcapException)
	{
		if (peerScanner)
			return;
		verbose("Starting network scan subsystem\n");
		sender = new NetSender(GuessnetEnvironment::get().iface());
		watcher = new NetWatcher(GuessnetEnvironment::get().iface());
		peerScanner = new PeerScanner(*sender, this);
		watcher->addARPListener(peerScanner);
	}

	void startScriptScanner() throw (SystemException)
	{
		if (scriptScanner)
			return;
		verbose("Starting external processes scan subsystem\n");
		runner = new ProcessRunner();
		scriptScanner = new ScriptScanner(*runner, this);
	}


public:
	MainScanner(IFace& iface) throw (Exception) : ScanRunner(),
		sender(0), watcher(0), runner(0), peerScanner(0), scriptScanner(0),
		iface(iface) {}

	void shutdown() throw (Exception)
	{
		if (watcher)
			watcher->shutdown();
		if (runner)
			runner->shutdown();
	}
	
	void startScans() throw (SystemException, libnetException, pcapException)
	{
		for (list<const Scan*>::const_iterator i = scans.getScans().begin();
				i != scans.getScans().end(); i++)
		{
			const Scan* scan = *i;
			if (const PeerScan* s = dynamic_cast<const PeerScan*>(scan))
			{
				debug("Will scan network %.*s for %.*s (%.*s)\n",
						PFSTR(s->name()), PFSTR(fmt(s->ip())), PFSTR(fmt(s->mac())));

				startPeerScanner();
				peerScanner->addCandidate(s);

				cand_count++;
			}
			else if (const ScriptScan* s = dynamic_cast<const ScriptScan*>(scan))
			{
				debug("Will use script %.*s to test %.*s\n",
						PFSTR(s->cmdline()), PFSTR(s->name()));

				startScriptScanner();
				scriptScanner->addCandidate(s);

				cand_count++;
			}
			else if (const LinkBeatScan* s = dynamic_cast<const LinkBeatScan*>(scan))
			{
				//warning("Link-beat detection currently disabled\n");
				debug("Will test for link beat.  If absent, will return %.*s\n",
						PFSTR(s->name()));

				iface.update();
				if (!iface.has_mii())
					warning("Link beat scan requested on an interface without link beat detection support\n");
				if (!iface.connected())
				{
					verbose("Link beat not detected\n");
					notifyScan(s);
				}

				cand_count++;
			}
			else if (const DefaultScan* s = dynamic_cast<const DefaultScan*>(scan))
			{
				debug("Default scan is %.*s\n",
						PFSTR(s->name()));
				cand_count++;
			}
			else
				//printf("Unknown scan %p\n", scan);
				printf("Unknown scan %.*s: %.*s\n", PFSTR(scan->name()), PFSTR(scan->signature()));
		}
	}
};

/*
bool detectIfupdown()
{
	pid_t ppid = getppid();
	string exe = "/proc/" + fmt(ppid) + "/exe";
	char buf[15];
	int size;
	if ((size = readlink(exe.c_str(), buf, 15)) == -1)
	{
		if (debug)
			fprintf(stderr, "Readlink of %.*s failed\n", PFSTR(exe));
		return false;
	}
	if (debug)
		fprintf(stderr, "Readlink of %.*s gave \"%.*s\"\n", PFSTR(exe), size, buf);
	return strncmp(buf, "/sbin/ifup", size) == 0;
}
*/

int main (int argc, const char *argv[])
{
	// Access the interface
	try {
		GuessnetEnvironment::init(argc, argv);

		IFace iface(GuessnetEnvironment::get().iface());

		// After the interface is up, we need another try run to be able to
		// shut it down, if needed, in case of problems
		bool iface_was_down;
		if_params saved_iface_cfg;
		try {
			// Install the handler for unexpected exceptions
			InstallUnexpected installUnexpected;

			vector<const Scan*> scans = GuessnetEnvironment::get().scans();
			verbose("%d candidate profiles\n", scans.size());

			/* Check if we have to bring up the interface; if yes, do it */
			//iface_was_down = iface_init(Environment::get().iface(), op_init_time);
			iface.update();
			iface_was_down = !iface.up();
			if (iface_was_down)
			{
				verbose("Interface %.*s was down: initializing for broadcast", PFSTR(iface.name()));
				saved_iface_cfg = iface.initBroadcast(GuessnetEnvironment::get().initTimeout());
			}

			// Let the signals be caught by some other process
			sigset_t sigs, oldsigs;
			sigfillset(&sigs);
			sigdelset(&sigs, SIGFPE);
			sigdelset(&sigs, SIGILL);
			sigdelset(&sigs, SIGSEGV);
			sigdelset(&sigs, SIGBUS);
			sigdelset(&sigs, SIGABRT);
			sigdelset(&sigs, SIGIOT);
			sigdelset(&sigs, SIGTRAP);
			sigdelset(&sigs, SIGSYS);
			// Don't block the termination signals: we need them
			sigdelset(&sigs, SIGTERM);
			sigdelset(&sigs, SIGINT);
			sigdelset(&sigs, SIGQUIT);
			pthread_sigmask(SIG_BLOCK, &sigs, &oldsigs);

			// Scanning methods
			MainScanner scanner(iface);
			//FakeScanner scanner;
			debug("Initialized scanning subsystems\n");

			scanner.handleScan(new DefaultScan(GuessnetEnvironment::get().defprof()));
			debug("Added \"default\" scan %.*s\n", PFSTR(GuessnetEnvironment::get().defprof()));

			for (vector<const Scan*>::const_iterator i = scans.begin(); i != scans.end(); i++)
				scanner.handleScan(*i);
			
			scanner.startScans();
			debug("Started scans\n");

			// Wait for the first result from the scans
			string profile;
			if (scanner.candidateCount() > 1)
			{
				debug("%d candidates\n", scanner.candidateCount());
				profile = scanner.getResult(GuessnetEnvironment::get().timeout() * 1000);
			} else {
				warning("no candidates provided: skipping detection");
			}

			// Shutdown the scanners
			scanner.shutdown();

			// We've shutdown the threads: restore original signals
			pthread_sigmask(SIG_SETMASK, &oldsigs, &sigs);

			// Output the profile name we found
			if (profile.size())
			{
				printf("%.*s\n", PFSTR(profile));
			} else {
				printf("%.*s\n", PFSTR(GuessnetEnvironment::get().defprof()));
			}
		} catch (Exception& e) {
			error("%s: %.*s\n", e.type(), PFSTR(e.desc()));
			if (geteuid() != 0)
			{
				struct passwd *p = getpwuid(geteuid());
				error("You can try invoking guessnet as user root instead of %s\n", p->pw_name);
			}
			return 1;
		}

		/* Bring down the interface if we need it */
		if (iface_was_down)
			iface.setConfiguration(saved_iface_cfg);

	} catch (Exception& e) {
		error("%s: %.*s\n", e.type(), PFSTR(e.desc()));
		if (geteuid() != 0)
		{
			struct passwd *p = getpwuid(geteuid());
			error("You can try invoking guessnet as user root instead of %s\n", p->pw_name);
		}
		return 1;
	}

	return 0;
}

// vim:set ts=4 sw=4:
