/*
 * Sniff network traffic to guess network data, and print it out as
 * an /etc/network/interfaces configuration profile
 *
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#define APPNAME PACKAGE
#else
#warning No config.h found: using fallback values
#define APPNAME __FILE__
#define VERSION "unknown"
#endif

#include "TrafficScanner.h"
#include "Mutex.h"
#include "Environment.h"
#include "stringf.h"

#include "CommandlineParser.h"
#include "IFace.h"

#include <stdio.h>
#include <ctype.h>
#include <errno.h>	/* errno */

#include <string.h>     // memcpy
#include <sys/types.h>  // socket
#include <sys/socket.h> // socket
#include <sys/ioctl.h>  // ioctl
#include <net/if.h>
#include <unistd.h>     // close


#include <set>

using namespace std;
using namespace stringf;

class MainScanner
{
protected:
	Mutex waitMutex;
	Condition waitCond;
	string name;

	// Scanning services
	NetSender sender;
	NetWatcher watcher;

	TrafficScanner trafficScanner;

	int cand_count;

	inline string fmt_ip(unsigned int ip) throw ()
	{
		unsigned char ipfmt[4];
		memcpy(ipfmt, &ip, 4);
		return fmt("%d.%d.%d.%d",
				(int)ipfmt[0], (int)ipfmt[1], (int)ipfmt[2], (int)ipfmt[3]);
	}

	inline string fmt_mac(long long int mac) throw ()
	{
		unsigned char macfmt[6];
		memcpy(macfmt, &mac, 6);
		return fmt("%02x:%02x:%02x:%02x:%02x:%02x",
				(int)macfmt[0], (int)macfmt[1], (int)macfmt[2],
				(int)macfmt[3], (int)macfmt[4], (int)macfmt[5]);
	}

	unsigned int netaddr_to_netmask(unsigned int na) throw ()
	{
		unsigned int res = 0;
		na = ntohl(na);

		// Count the trailing zeros
		int i = 0;
		for ( ; i < 32 && (na & (1 << i)) == 0; i++)
			;

		// Add 1s to the start of res
		for ( ; i < 32; i++)
			res |= (1 << i);

		return htonl(res);
	}

public:
	MainScanner() throw (Exception) :
		sender(Environment::get().iface()),
		watcher(Environment::get().iface()),
		trafficScanner(sender),
		cand_count(0)
	{
		watcher.addEthernetListener(&trafficScanner);
	}

	int candidateCount() const throw () { return cand_count; }
	
	void shutdown() throw (Exception)
	{
		watcher.shutdown();
	}

	void printResults() throw ()
	{
		string s;
		printf("iface <name> inet static\n");
		printf("\taddress <addr>\n");

		unsigned int network = trafficScanner.guessed_netaddr;
		s = fmt_ip(network);
		printf("\tnetwork %.*s\n", PFSTR(s));

		unsigned int netmask = netaddr_to_netmask(network);
		s = fmt_ip(netmask);
		printf("\tnetmask %.*s\n", PFSTR(s));

		s = fmt_ip(network | ~netmask);
		printf("\tbroadcast %.*s\n", PFSTR(s));

		for (set<TrafficScanner::mac_key>::const_iterator i = trafficScanner.guessed_gateways.begin();
				i != trafficScanner.guessed_gateways.end(); i++)
		{
			TrafficScanner::HostData& od = trafficScanner.scanData[*i];

			s = fmt_ip(od.addr);
			printf("\tgateway %.*s\n", PFSTR(s));
		
			string m = fmt_mac(*i);
			printf("\ttest-peer address %.*s mac %.*s\n", PFSTR(s), PFSTR(m));
		}
	}
};

int main (int argc, const char *argv[])
{
	CommandlineParser opts(APPNAME, "[options] [ethernet_interface]", "Guess the current network location");
	opts.add("version", 'V', "version", "print the program version, then exit");
	opts.add("timeout", 't', "timeout",
			"timeout (in seconds) used to wait for response packets"
			" (defaults to 5 seconds)", "seconds");
	opts.add("inittime", 0, "init-timeout",
			"time (in seconds) to wait for the interface to initialize"
			" when not found already up (defaults to 3 seconds)");
	opts.add("verbose", 'v', "verbose",
			"enable verbose operations");
	opts.add("debug", 0, "debug", 
			"enable debugging output");

	// Process the commandline
	if (!opts.parse(argc, argv))
	{
		opts.printHelp();
		return 1;
	}
	if (opts.get("help").defined())
	{
		opts.printHelp();
		return 0;
	}
	if (opts.get("version").defined())
	{
		printf("%s " VERSION "\n", APPNAME);
		return 0;
	}

	// Set verbosity
	Environment::get().verbose(opts.get("verbose").defined());
	Environment::get().debug(opts.get("debug").defined());

	// Check user id
	if (geteuid() != 0)
		fatal_error("You must run this command as root.");

	// Find out the interface to be tested
	if (argc > 1)
		Environment::get().iface(argv[1]);

	// Find out the test timeout
	if (opts.get("timeout").defined())
		Environment::get().timeout(opts.get("timeout").intVal());

	// Find out the init timeout
	if (opts.get("inittime").defined())
		Environment::get().initTimeout(opts.get("inittime").intVal());

	// Access the interface
	try {
		IFace iface(Environment::get().iface());

	bool iface_was_down;
	if_params saved_iface_cfg;

	try {
		// Install the handler for unexpected exceptions
		InstallUnexpected installUnexpected;
		FILE* input = 0;

		/* Check if we have to bring up the interface; if yes, do it */
		//iface_was_down = iface_init(Environment::get().iface(), op_init_time);
		iface.update();
		iface_was_down = !iface.up();
		if (iface_was_down)
			saved_iface_cfg = iface.initBroadcast(Environment::get().initTimeout());

		// Let the signals be caught by some other process
		sigset_t sigs, oldsigs;
		sigfillset(&sigs);
		sigdelset(&sigs, SIGFPE);
		sigdelset(&sigs, SIGILL);
		sigdelset(&sigs, SIGSEGV);
		sigdelset(&sigs, SIGBUS);
		sigdelset(&sigs, SIGABRT);
		sigdelset(&sigs, SIGIOT);
		sigdelset(&sigs, SIGTRAP);
		sigdelset(&sigs, SIGSYS);
		// Don't block the termination signals: we need them
		sigdelset(&sigs, SIGTERM);
		sigdelset(&sigs, SIGINT);
		sigdelset(&sigs, SIGQUIT);
		pthread_sigmask(SIG_BLOCK, &sigs, &oldsigs);

		// Scanning methods
		MainScanner scanner;

		debug("Started test subsystems\n");

		sleep(Environment::get().timeout());

		scanner.printResults();

		/*
		// Wait for the first result from the tests
		string profile;
		if (scanner.candidateCount() > 0)
			profile = scanner.getResult(Environment::get().timeout() * 1000);
		*/

		// Shutdown the tests
		scanner.shutdown();

		// We've shutdown the threads: restore original signals
		pthread_sigmask(SIG_SETMASK, &oldsigs, &sigs);
	} catch (Exception& e) {
		fatal_error("%s: %.*s", e.type(), PFSTR(e.desc()));
		return 1;
	}

	/* Bring down the interface if we need it */
	if (iface_was_down)
		iface.setConfiguration(saved_iface_cfg);

	} catch (Exception& e) {
		fatal_error("%s: %.*s", e.type(), PFSTR(e.desc()));
		return 1;
	}

	return 0;
}

// vim:set ts=4 sw=4:
