#ifndef BUFFER_H
#define BUFFER_H

/*
 * Variable-size, reference-counted memory buffer
 *
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

class Buffer
{
public:
	class Data
	{
	protected:
		mutable int _ref;
		unsigned int _size;
		void* _data;

	public:
		Data() throw () : _ref(0), _size(0), _data(0) {}
		Data(unsigned int size) throw ();
		// if own == true, take possession of the memory buffer, else copy it
		Data(void* buf, unsigned int size, bool own = true) throw ();
		Data(const void* buf, unsigned int size) throw ();
		~Data() throw ();

		/// Increment the reference count for this object
		void ref() const throw () { ++_ref; }

		/// Decrement the reference count for this object, returning true when it
		/// reaches 0
		bool unref() const throw () { return --_ref == 0; }

		/// Resize (enlarging or shrinking it) the buffer to `size' bytes
		void resize(unsigned int size) throw ();

		friend class Buffer;
	};

	Data* item;
	
public:
	Buffer() throw () : item(new Data()) { item->ref(); }
	Buffer(unsigned int size) throw () : item(new Data(size)) { item->ref(); }
	Buffer(void* buf, unsigned int size, bool own = true) throw ()
		: item(new Data(buf, size, own)) { item->ref(); }
	Buffer(const void* buf, unsigned int size) throw ()
		: item(new Data(buf, size)) { item->ref(); }

	Buffer(const Buffer& buf) throw ()
	{
		if (buf.item)
			buf.item->ref();
		item = buf.item;
	}
	~Buffer() throw ()
	{
		if (item && item->unref())
			delete item;
	}
	Buffer& operator=(const Buffer& buf) throw ()
	{
		if (buf.item)
			buf.item->ref();  // Do it early to correctly handle the case of x = x;
		if (item && item->unref())
			delete item;
		item = buf.item;
		return *this;
	}

	void* data() const throw () { return item->_data; }
	unsigned int size() const throw () { return item->_size; }
	void resize(unsigned int size) throw () { item->resize(size); }
};

// vim:set ts=4 sw=4:
#endif
