# -*- coding: utf-8 -*-
# Gufw 13.10.2 - http://gufw.org
# Copyright (C) 2008-2013 Marcos Alvarez Costales https://launchpad.net/~costales
#
# Gufw is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
# 
# Gufw is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with Gufw; if not, see http://www.gnu.org/licenses for more
# information.


from gi.repository import Gtk, GObject
import os, glob

from constants import *

import gettext
from gettext import gettext as _
gettext.textdomain('gufw')


class UpdateRule:
    def __init__(self, fw_frontend, utils, rules_model, ufw_row,
                 description, cmd, policy, direction, proto, from_ip, from_port, to_ip, to_port, iface, logging,
                 winMain):
        self.fw = fw_frontend
        self.utils = utils
        self.model = rules_model
        
        self.ufw_row          = str(ufw_row)
        self.rule_description = description
        self.rule_cmd         = cmd
        self.rule_policy      = policy
        self.rule_direction   = direction
        self.rule_proto       = proto
        self.rule_from_ip     = from_ip
        self.rule_from_port   = from_port
        self.rule_to_ip       = to_ip
        self.rule_to_port     = to_port
        self.rule_iface       = iface
        self.rule_logging     = logging

        self.builder = Gtk.Builder()
        self.builder.set_translation_domain("gufw")
        self.builder.add_from_file('/usr/share/gufw/ui/update.ui')
        self._set_objects_name()
        self._set_initial_values()
        self.win_update.set_transient_for(winMain)
        self.builder.connect_signals(self)
        self.win_update.show_all()
        # Set sensitive values
        self.update_from_ip.set_text(self.rule_from_ip)
        self.update_from_port.set_text(self.rule_from_port)
        self.update_to_ip.set_text(self.rule_to_ip)
        self.update_to_port.set_text(self.rule_to_port)        
    
    def _set_objects_name(self):
        self.win_update       = self.builder.get_object('UpdateRule')
        self.update_rule_name = self.builder.get_object('update_rule_name')
        self.update_policy    = self.builder.get_object('update_policy')
        self.update_direction = self.builder.get_object('update_direction')
        self.update_iface     = self.builder.get_object('update_iface')
        self.update_log       = self.builder.get_object('update_log')
        self.update_protocol  = self.builder.get_object('update_protocol')
        self.update_from_ip   = self.builder.get_object('update_from_ip')
        self.update_from_port = self.builder.get_object('update_from_port')
        self.update_to_ip     = self.builder.get_object('update_to_ip')
        self.update_to_port   = self.builder.get_object('update_to_port')

    def _set_initial_values(self):
        iface_num = 0
        i = 0
        self.update_iface.append_text(_("All Interfaces"))
        for ifaceName in self.fw.get_net_interfaces():
            self.update_iface.append_text(ifaceName)
            if ifaceName == self.rule_iface:
                iface_num = i + 1
            i += 1
        self.update_rule_name.set_text(self.rule_description)
        self.update_policy.set_active(POLICY2NUM[self.rule_policy])
        self.update_direction.set_active(DIRECTION2NUM[self.rule_direction])
        self.update_iface.set_active(iface_num)
        self.update_log.set_active(LOGGING2NUM[self.rule_logging])
        self.update_protocol.set_active(PROTO2NUM[self.rule_proto])
    
    def _set_from_port_sensitive(self, value=True):
        self.update_protocol.set_sensitive(value)
        self.update_iface.set_sensitive(value)
        self.update_from_ip.set_sensitive(value)
        self.update_to_ip.set_sensitive(value)
        self.update_to_port.set_sensitive(value)
        if value:
            if self.update_from_ip.get_text() or self.update_to_ip.get_text():
                self.update_iface.set_sensitive(True)
            else:
                self.update_iface.set_sensitive(False)
    
    def _set_to_port_sensitive(self, value=True):
        self.update_protocol.set_sensitive(value)
        self.update_iface.set_sensitive(value)
        self.update_from_ip.set_sensitive(value)
        self.update_to_ip.set_sensitive(value)
        self.update_from_port.set_sensitive(value)
        if value:
            if self.update_from_ip.get_text() or self.update_to_ip.get_text():
                self.update_iface.set_sensitive(True)
            else:
                self.update_iface.set_sensitive(False)
    
    def on_update_to_port_changed(self, widget, data=None):
        if ('/' in self.update_to_port.get_text() or
            '/' in self.update_to_port.get_text()):
            self._set_to_port_sensitive(False)
        else:
            self._set_to_port_sensitive(True)
    
    def on_update_from_port_changed(self, widget, data=None):
        if ('/' in self.update_from_port.get_text() or
            '/' in self.update_from_port.get_text()):
            self._set_from_port_sensitive(False)
        else:
            self._set_from_port_sensitive(True)
    
    def on_update_from_ip_changed(self, widget, data=None):
        if self.update_from_ip.get_text() or self.update_to_ip.get_text():
            self.update_iface.set_sensitive(True)
        else:
            self.update_iface.set_sensitive(False)
    
    def on_update_to_ip_changed(self, widget, data=None):
        if self.update_from_ip.get_text() or self.update_to_ip.get_text():
            self.update_iface.set_sensitive(True)
        else:
            self.update_iface.set_sensitive(False)
    
    def on_btnUpdateCancel_clicked(self, widget, data=None):
        self._win_close()
    
    def on_UpdateRule_delete_event(self, widget, data=None):
        self._win_close()
        
    def _win_close(self):
        self.win_update.destroy()
    
    def on_update_copy_from_IP_clicked(self, widget, data=None):
        self.update_from_ip.set_text(self.fw.get_internal_ip())
    
    def on_update_copy_to_IP_clicked(self, widget, data=None):
        self.update_to_ip.set_text(self.fw.get_internal_ip())
    
    def on_btnUpdate_clicked(self, widget, data=None):
        if not self.utils.validate_rule(self.update_from_ip.get_text(), self.update_from_port.get_text(), self.update_to_ip.get_text(), self.update_to_port.get_text()):
            return
        
        new_description = self.update_rule_name.get_text()
        new_policy      = NUM2POLICY[self.update_policy.get_active()]
        new_direction   = NUM2DIRECTION[self.update_direction.get_active()]
        new_logging     = NUM2LOGGING[self.update_log.get_active()]
        
        new_proto = ''
        if self.update_protocol.get_sensitive():
            new_proto = NUM2PROTO[self.update_protocol.get_active()]
        
        new_iface = ''
        if self.update_iface.get_sensitive() and self.update_iface.get_active_text() != _("All Interfaces"):
            new_iface = self.update_iface.get_active_text()
        
        new_from_ip = new_from_port = new_to_ip = new_to_port = ''
        if self.update_from_ip.get_sensitive():
            new_from_ip = self.update_from_ip.get_text()
        if self.update_from_port.get_sensitive():
            new_from_port = self.update_from_port.get_text()
        if self.update_to_ip.get_sensitive():
            new_to_ip = self.update_to_ip.get_text()
        if self.update_to_port.get_sensitive():
            new_to_port = self.update_to_port.get_text()
        
        if (self.rule_description == new_description and
            self.rule_policy      == new_policy      and
            self.rule_direction   == new_direction   and
            self.rule_proto       == new_proto       and
            self.rule_from_ip     == new_from_ip     and
            self.rule_from_port   == new_from_port   and
            self.rule_to_ip       == new_to_ip       and
            self.rule_to_port     == new_to_port     and
            self.rule_iface       == new_iface       and
            self.rule_logging     == new_logging):
            self.utils.set_statusbar_msg(_("No changes were made"))
            return
        
        # Delete the same rules
        same_rules_rows = self._get_same_rules(self.rule_cmd)
        for same_row in reversed(same_rules_rows):
            cmd = self.fw.delete_rule(same_row)
            self.utils.add_to_log(_("Deleted rule") + ": " + new_description + ' | ' + cmd[0] + ' > ' + cmd[1].replace('\n', ' | '))
        
        # Add new
        insert_row = ''
        cmd = self.fw.add_rule(new_description, insert_row, new_policy, new_direction, new_iface, new_logging, new_proto, new_from_ip, new_from_port, new_to_ip, new_to_port)

        self.utils.add_to_log(_("Added rule") + ": " + new_description + ' | ' + cmd[1] + ' > ' + cmd[2].replace('\n', ' | '), POLICY2COLOR[new_policy])
        self.utils.set_statusbar_msg(_("Updated rule ") + str(self.ufw_row))
        
        self.utils.print_rules(self.fw.get_rules())
        self._win_close()
    
    def _get_same_rules(self, rule_cmd):
        i = 0
        rules_rows = []
        while True:
            try:
                iter = self.model.get_iter(i,)
                cmd = self.model.get_value(iter, 2)
                if cmd == rule_cmd:
                    rules_rows.append((i+1))
            except:
                return rules_rows
            i += 1
