// ****************************************************************************
//  Project:        GUYMAGER
// ****************************************************************************
//  Programmer:     Guy Voncken
//                  Police Grand-Ducale
//                  Service de Police Judiciaire
//                  Section Nouvelles Technologies
// ****************************************************************************
//  Module:         Thread for writing data.
// ****************************************************************************

// Copyright 2008, 2009, 2010 Guy Voncken
//
// This file is part of guymager.
//
// guymager is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// guymager is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with guymager. If not, see <http://www.gnu.org/licenses/>.

#include "common.h"
#include "compileinfo.h"

#include <errno.h>
#include <zlib.h>
#include <fcntl.h>

#include <QtCore>

#include "toolconstants.h"
#include "toolsysinfo.h"

#include "util.h"
#include "config.h"
#include "device.h"
#include "main.h"
#include "qtutil.h"
#include "threadwrite.h"


const unsigned long THREADWRITE_WAIT_FOR_HANDLE_GRANULARITY =   100; // ms
const unsigned long THREADWRITE_WAIT_FOR_HANDLE             = 30000; // ms
const unsigned long THREADWRITE_SLOWDOWN_SLEEP              =   700; // ms

class t_OutputFile
{
   public:
      virtual        ~t_OutputFile  (void) {};
      virtual APIRET  Open          (t_pDevice pDevice, bool Verify) = 0;
      virtual APIRET  Write         (t_pFifoBlock pFifoBlock)        = 0;
      virtual APIRET  Verify        (t_pHashContextMD5 pHashContextMD5, t_pHashContextSHA256 pHashContextSHA256, quint64 *pPos) = 0;
      virtual APIRET  Close         (void)                           = 0;
      virtual bool    Opened        (void)                           = 0;
      virtual void *  GetFileHandle (void)                           = 0;
};

class t_OutputFileDD: public t_OutputFile
{
   public:
      t_OutputFileDD (void) :
          oFile                (-1),
          oLastCheckT          (0),
         poVerifyBuff          (NULL),
         poDevice              (NULL),
          oSplitNr             (0),
          oCurrentVerifyFileNr (0)
      {
      } //lint -esym(613, t_OutputFileDD::poFile)  Possible use of NULL pointer
        //lint -esym(668, fclose, fwrite)  Possibly passing NULL pointer

      ~t_OutputFileDD (void)
      {
         if (t_OutputFileDD::Opened())
            (void) t_OutputFileDD::Close();
      } //lint !e1740

      APIRET Open (t_pDevice pDevice, bool Verify)
      {
         QString Extension;

         poDevice = pDevice;
         if (CONFIG (WriteToDevNull))
         {
            oFilename = "/dev/null";
         }
         else
         {
            CHK (t_File::GetFormatExtension (pDevice, &Extension))
            oFilename = pDevice->Acquisition.ImagePath + pDevice->Acquisition.ImageFilename;
            if (pDevice->Acquisition.SplitFileSize == 0)
                 oFilename += Extension;
            else oSplitNrDecimals = t_File::GetDdSplitNrDecimals (pDevice->Size, pDevice->Acquisition.SplitFileSize);
         }

//       oFileFlags = O_NOATIME | Verify ? O_RDONLY : O_WRONLY; // Doesn't work... !??
         oFileFlags = O_NOATIME | O_RDWR;
         if (!pDevice->Acquisition.Clone && !Verify)
            oFileFlags |= O_CREAT;              // Create it if doesn't exist

         time (&oLastCheckT);

         return NO_ERROR;
      }

      APIRET Write (t_pFifoBlock pFifoBlock)
      {
         size_t   RemainingData = pFifoBlock->DataSize;
         size_t   Offset = 0;
         ssize_t  Written;
         time_t   NowT;

         while (RemainingData)
         {
            if (!Opened())
            {
               QString Extension;
               QString Filename;
               if (poDevice->Acquisition.SplitFileSize)
               {
                  Extension = QString(".%1").arg(oSplitNr++, oSplitNrDecimals, 10, QChar('0'));
                  oRemainingFileSize = poDevice->Acquisition.SplitFileSize;
               }
               else
               {
                  oRemainingFileSize = ULONG_LONG_MAX;
               }
               Filename = QSTR_TO_PSZ (oFilename+Extension); // Do a QSTR_TO_PSZ conversion here in order to work everywhere with exactly the same filename
               oFilenameList += Filename;
               oFile = open64 (QSTR_TO_PSZ (Filename), oFileFlags, S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH | S_IWOTH);
               if (oFile < 0)
               {
                  LOG_ERROR ("open64 on %s failed, errno=%d '%s'", QSTR_TO_PSZ (oFilename), errno, ToolErrorTranslateErrno (errno))
                  return ERROR_THREADWRITE_OPEN_FAILED;
               }
            }

            Written = write (oFile, &pFifoBlock->Buffer[Offset], GETMIN(RemainingData, oRemainingFileSize));
            if (Written < 0)
            {
               LOG_ERROR ("write failed, oFile %d, errno=%d '%s'", oFile, errno, ToolErrorTranslateErrno (errno))
               return ERROR_THREADWRITE_WRITE_FAILED;
            }
            RemainingData      -= Written;
            oRemainingFileSize -= Written;
            Offset             += Written;
            if (oRemainingFileSize == 0)
               Close();
         }

         if (poDevice->Acquisition.Clone)    // It was observed that the write or fwrite functions do not detect device removal! It is unknown
         {                                   // how this is possible... Therefore, we check the existence of the device every 2 seconds by means
            time (&NowT);                    // of a seperate fopen/flclose.
            if ((NowT - oLastCheckT) >=2)
            {
               FILE *pFile;
               bool   Error;

               pFile = fopen64 (QSTR_TO_PSZ (oFilename), "r");
               Error = (pFile == NULL);
               if (!Error)
                  Error = (fclose (pFile) != 0);
               if (Error)
               {
                  LOG_ERROR ("Check for destination clone %s failed (%d '%s')", QSTR_TO_PSZ (oFilename), errno, ToolErrorTranslateErrno (errno))
                  return ERROR_THREADWRITE_WRITE_FAILED;
               }
               oLastCheckT = NowT;
            }
         }

         return NO_ERROR;
      }

      APIRET Verify (t_pHashContextMD5 pHashContextMD5, t_pHashContextSHA256 pHashContextSHA256, quint64 *pPos)
      {
         ssize_t Read;

         if (*pPos == 0)       // First call of this function?
         {
            poVerifyBuff = UTIL_MEM_ALLOC (poDevice->FifoBlockSize);
            if (poVerifyBuff == NULL)
               CHK (ERROR_THREADWRITE_MALLOC_FAILED);
         }

         if (!Opened())
         {
            QString Filename = oFilenameList[oCurrentVerifyFileNr++];
            oFile = open64 (QSTR_TO_PSZ (Filename), oFileFlags, S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH | S_IWOTH);
            off64_t FileSize;
            bool    Err = (oFile < 0);
            if (!Err) Err = ((FileSize = lseek64 (oFile,0,SEEK_END)) == -1);
            if (!Err) Err = (lseek64 (oFile, 0, SEEK_SET) != 0);
            if (Err)
            {
               LOG_ERROR ("Opening %s failed, errno=%d '%s'", QSTR_TO_PSZ (oFilename), errno, ToolErrorTranslateErrno (errno))
               return ERROR_THREADWRITE_OPEN_FAILED;
            }
            oRemainingFileSize = (t_uint64) FileSize;
         }

         Read = read (oFile, (char*)poVerifyBuff, GETMIN(poDevice->FifoBlockSize, oRemainingFileSize));
         if (Read < 0)
         {
            QString Filename = oFilenameList[oCurrentVerifyFileNr-1];
            LOG_ERROR ("read failed, oFile %d [%s], seek pos %llu, errno %d '%s'", oFile, QSTR_TO_PSZ (Filename), *pPos, errno, ToolErrorTranslateErrno (errno))
            return ERROR_THREADWRITE_VERIFY_FAILED;
         }
         oRemainingFileSize -= Read;

         if (oRemainingFileSize == 0)
            Close0 (false);

         if (pHashContextMD5)    CHK_EXIT (HashMD5Append    (pHashContextMD5   , poVerifyBuff, Read))
         if (pHashContextSHA256) CHK_EXIT (HashSHA256Append (pHashContextSHA256, poVerifyBuff, Read))

         *pPos += Read;

         return NO_ERROR;
      }

      APIRET Close0 (bool ReleaseMem)
      {
         int Res;

         if (oFile == -1)
            CHK (ERROR_THREADWRITE_NOT_OPENED)

         Res = close (oFile);
         oFile = -1;
         if (Res != 0)
         {
            LOG_ERROR ("close failed, errno=%d '%s'", errno, ToolErrorTranslateErrno (errno))
            return ERROR_THREADWRITE_CLOSE_FAILED;
         }

         if (poVerifyBuff && ReleaseMem)
         {
            UTIL_MEM_FREE (poVerifyBuff);
            poVerifyBuff = NULL;
         }
         return NO_ERROR;
      }

      APIRET Close (void)
      {
         return Close0 (true);
      }

      void * GetFileHandle (void)
      {
         if (oFile == -1)
              return NULL;
         else return &oFile;
      }

      inline bool Opened (void)
      {
         return (oFile != -1);
      }

   private:
      QStringList oFilenameList;
      int         oFile;
      time_t      oLastCheckT; // Only used when cloning, see above
      QString     oFilename;
      void      *poVerifyBuff;
      t_pDevice  poDevice;
      int         oSplitNr;
      int         oSplitNrDecimals;
      int         oFileFlags;
      t_uint64    oRemainingFileSize;
      int         oCurrentVerifyFileNr;
};

#define CHK_LIBEWF(Fn)                                                 \
{                                                                      \
   int rclibewf = (Fn);                                                \
   if (rclibewf != 1)                                                  \
   {                                                                   \
      LOG_ERROR ("Error in libewf function: %s, rc=%d", #Fn, rclibewf) \
      return ERROR_THREADWRITE_LIBEWF_FAILED;                          \
   }                                                                   \
}

class t_OutputFileEWF: public t_OutputFile
{
   public:
      t_OutputFileEWF (void)
      {
         poFile                  = NULL;
         poDevice                = NULL;   //lint -esym(613,t_OutputFileEWF::poDevice)  Prevent lint from telling us about possible null pointers in the following code
         poVerifyBuff            = NULL;
         poImageFilenameArr      = NULL;
          oImageFileCount        = 0;
          oHasCompressionThreads = false;
          oVerification          = false;
      }

      ~t_OutputFileEWF (void)
      {
         if (t_OutputFileEWF::Opened())
            (void) t_OutputFileEWF::Close();
      } //lint !e1740

      APIRET Open (t_pDevice pDevice, bool Verify)
      {
         QString         Uname;
         QString         GuymagerVersion;
         LIBEWF_HANDLE *pFile;
         char          *pAsciiFileName;
         QByteArray      AsciiFileName = (pDevice->Acquisition.ImagePath + pDevice->Acquisition.ImageFilename).toAscii();

         oHasCompressionThreads = pDevice->HasCompressionThreads();
         poDevice               = pDevice;
         oVerification          = Verify;

         if (Verify)
         {
            QFileInfoList FileInfoList;
            QString       ExtensionImage;
            QFileInfo     FileInfo;
            QDir          Dir(pDevice->Acquisition.ImagePath);
            int           i;

            CHK (t_File::GetFormatExtension (pDevice, &ExtensionImage))
            FileInfoList = Dir.entryInfoList (QStringList(pDevice->Acquisition.ImageFilename + ExtensionImage), QDir::Files, QDir::Name);
            oImageFileCount = FileInfoList.count();
            poImageFilenameArr = (char **) malloc (oImageFileCount * sizeof (char*));
            for (i=0; i<oImageFileCount; i++)
            {
               FileInfo       = FileInfoList.takeFirst();
               AsciiFileName  = FileInfo.absoluteFilePath().toAscii();
               pAsciiFileName = AsciiFileName.data();
               poImageFilenameArr[i] = (char *) malloc (strlen(pAsciiFileName)+1);
               strcpy (poImageFilenameArr[i], pAsciiFileName);
            }
            pFile = libewf_open (poImageFilenameArr, oImageFileCount, libewf_get_flags_read());
            if (pFile == NULL)
            {
               LOG_INFO ("Error while reopening EWF for verification. The files are:")
               for (i=0; i<oImageFileCount; i++)
                  LOG_INFO ("%s", poImageFilenameArr[i])
               return ERROR_THREADWRITE_OPEN_FAILED;
            }
         }
         else
         {
            char *pAsciiFileName = AsciiFileName.data();

            pFile = libewf_open (&pAsciiFileName, 1, LIBEWF_OPEN_WRITE);
            if (pFile == NULL)
               return ERROR_THREADWRITE_OPEN_FAILED;

            #define STR_AND_LEN(QStr) QStr.toAscii().data(), strlen(QStr.toAscii().data())
            libewf_set_notify_values(stderr, 1);

            CHK_LIBEWF (libewf_set_format             (pFile, (uint8_t) CONFIG (EwfFormat)))
            CHK_LIBEWF (libewf_set_media_size         (pFile, pDevice->Size))
            CHK_LIBEWF (libewf_set_bytes_per_sector   (pFile, (unsigned int) pDevice->SectorSize))
            CHK_LIBEWF (libewf_set_sectors_per_chunk  (pFile, 64))
            CHK_LIBEWF (libewf_set_segment_file_size  (pFile, pDevice->Acquisition.SplitFileSize))
            CHK_LIBEWF (libewf_set_compression_values (pFile, CONFIG (EwfCompression), 1))
            CHK_LIBEWF (libewf_set_media_type         (pFile, pDevice->Removable ? LIBEWF_MEDIA_TYPE_REMOVABLE : LIBEWF_MEDIA_TYPE_FIXED))
            CHK_LIBEWF (libewf_set_volume_type        (pFile, LIBEWF_VOLUME_TYPE_PHYSICAL))

            CHK_LIBEWF (libewf_set_header_value (pFile, (char *)"case_number"    , STR_AND_LEN(pDevice->Acquisition.CaseNumber    )))
            CHK_LIBEWF (libewf_set_header_value (pFile, (char *)"description"    , STR_AND_LEN(pDevice->Acquisition.Description   )))
            CHK_LIBEWF (libewf_set_header_value (pFile, (char *)"examiner_name"  , STR_AND_LEN(pDevice->Acquisition.Examiner      )))
            CHK_LIBEWF (libewf_set_header_value (pFile, (char *)"evidence_number", STR_AND_LEN(pDevice->Acquisition.EvidenceNumber)))
            CHK_LIBEWF (libewf_set_header_value (pFile, (char *)"notes"          , STR_AND_LEN(pDevice->Acquisition.Notes         )))

            CHK (ToolSysInfoUname (Uname))
            GuymagerVersion = QString("guymager ") + QString(pCompileInfoVersion);
            CHK_LIBEWF (libewf_set_header_value (pFile, (char *)"acquiry_operating_system", STR_AND_LEN(Uname)))
            CHK_LIBEWF (libewf_set_header_value (pFile, (char *)"acquiry_software_version", STR_AND_LEN(GuymagerVersion)))

            #undef STR_AND_LEN

         }
         poFile = pFile; // Only set poFile at the very end, so the CompressionThreads won't use it until everything is initialised
         return NO_ERROR;
      }

      APIRET Write (t_pFifoBlock pFifoBlock)
      {
         int Written;
         int Size;

         if (oHasCompressionThreads != pFifoBlock->EwfPreprocessed)
            CHK (ERROR_THREADWRITE_WRONG_BLOCK)

         if (oHasCompressionThreads)
         {
            Size = pFifoBlock->EwfDataSize;
            Written = libewf_raw_write_buffer (poFile, pFifoBlock->Buffer, Size, pFifoBlock->DataSize,
                                                                                 pFifoBlock->EwfCompressionUsed,
                                                                                 pFifoBlock->EwfChunkCRC,
                                                                                 pFifoBlock->EwfWriteCRC);
         }
         else
         {
            Size = pFifoBlock->DataSize;
            Written = libewf_write_buffer (poFile, pFifoBlock->Buffer, Size);
         }

         if (Written != Size)
         {
            LOG_ERROR ("Written %d/%d bytes", Written, Size)
            return ERROR_THREADWRITE_WRITE_FAILED;
         }

         return NO_ERROR;
      }

      APIRET Verify (t_pHashContextMD5 pHashContextMD5, t_pHashContextSHA256 pHashContextSHA256, quint64 *pPos)
      {
         quint64      Remaining;
         unsigned int ToRead;
         ssize_t      Read;


         if (*pPos == 0)
         {
            poVerifyBuff = UTIL_MEM_ALLOC (poDevice->FifoBlockSize);
            if (poVerifyBuff == NULL)
               CHK (ERROR_THREADWRITE_MALLOC_FAILED);
         }
         Remaining = poDevice->Size - *pPos;
         ToRead = GETMIN (Remaining, poDevice->FifoBlockSize);
         Read = libewf_read_buffer (poFile, poVerifyBuff, ToRead);
         if (Read != (ssize_t)ToRead)
         {
            LOG_INFO ("Reading from EWF file failed (%zu)", Read);
            return ERROR_THREADWRITE_VERIFY_FAILED;
         }
         if (pHashContextMD5)    CHK_EXIT (HashMD5Append    (pHashContextMD5   , poVerifyBuff, ToRead))
         if (pHashContextSHA256) CHK_EXIT (HashSHA256Append (pHashContextSHA256, poVerifyBuff, ToRead))

         *pPos += ToRead;

         return NO_ERROR;
      }

      APIRET Close (void)
      {
         int rc;
         int i;

         if (poFile == NULL)
            CHK (ERROR_THREADWRITE_NOT_OPENED)

         if (!oVerification && poDevice->HasCompressionThreads() && poDevice->Acquisition.CalcMD5)
            CHK_LIBEWF (libewf_set_md5_hash (poFile, (uint8_t*)&poDevice->MD5Digest, HASH_MD5_DIGEST_LENGTH))

         rc = libewf_close (poFile);
         if (rc != 0)
         {
            LOG_ERROR ("Error in libewf function: libewf_close, rc=%d", rc)
            return ERROR_THREADWRITE_LIBEWF_FAILED;
         }

         if (poVerifyBuff)
         {
            UTIL_MEM_FREE (poVerifyBuff);
            poVerifyBuff = NULL;
         }

         for (i=0; i<oImageFileCount; i++)
            free (poImageFilenameArr[i]);
         free (poImageFilenameArr);

         poFile = NULL;
         return NO_ERROR;
      }

      void * GetFileHandle (void)
      {
         return poFile;
      }

      bool Opened (void)
      {
         return (poFile != NULL);
      }

   private:
      LIBEWF_HANDLE *poFile;
      t_pDevice      poDevice;
      bool            oVerification;
      void          *poVerifyBuff;
      int             oImageFileCount;
      char         **poImageFilenameArr;
      bool            oHasCompressionThreads;
};


class t_OutputFileAFF: public t_OutputFile
{
   public:
      t_OutputFileAFF (void) :
         poFile                (NULL ),
         poDevice              (NULL ),
         poVerifyBuff          (NULL ),
         oHasCompressionThreads(false),
         oVerification         (false)
      {
      }

      ~t_OutputFileAFF (void)
      {
         if (t_OutputFileAFF::Opened())
            (void) t_OutputFileAFF::Close();
      } //lint !e1740

      APIRET Open (t_pDevice pDevice, bool Verify)
      {
         QString               Extension;
         QString               FileName;
         QString               Mac;
         QString               DateTimeStr;
         t_ToolSysInfoMacAddr  MacAddr;
         char                *pCommandLine;
         t_pAaff              pFile;
         APIRET                rc;

         poDevice      = pDevice;
         oVerification =  Verify;
         oHasCompressionThreads = pDevice->HasCompressionThreads();

         CHK (t_File::GetFormatExtension (pDevice, &Extension))
         FileName = pDevice->Acquisition.ImagePath + pDevice->Acquisition.ImageFilename + Extension;

         if (oVerification)
         {
            rc = AaffOpen (&pFile, QSTR_TO_PSZ(FileName));
            if (rc)
            {
               LOG_INFO ("AFF image verification failed: %s", ToolErrorMessage (rc))
               return ERROR_THREADWRITE_VERIFY_FAILED;
            }
         }
         else
         {
            CHK (AaffOpen (&pFile, QSTR_TO_PSZ(FileName), pDevice->Size, pDevice->SectorSize, pDevice->FifoBlockSize))

            rc = ToolSysInfoGetMacAddr (&MacAddr);
            if (rc == TOOLSYSINFO_ERROR_NO_ADDR)
            {
               Mac = "none";
            }
            else
            {
               CHK (rc)
               Mac = &MacAddr.AddrStr[0];
            }

            DateTimeStr  = poDevice->StartTimestamp.toString ("yyyy-MM-dd hh:mm:ss");
            DateTimeStr += " localtime";
            CHK (MainGetCommandLine (&pCommandLine))

            CHK (AaffWriteSegmentStr (pFile, AAFF_SEGNAME_COMMAND_LINE, 0, pCommandLine))
            CHK (AaffWriteSegmentStr (pFile, AAFF_SEGNAME_MACADDR     , 0, QSTR_TO_PSZ(Mac                                 )))
            CHK (AaffWriteSegmentStr (pFile, AAFF_SEGNAME_DATE        , 0, QSTR_TO_PSZ(DateTimeStr                         )))
            CHK (AaffWriteSegmentStr (pFile, AAFF_SEGNAME_DEVICE      , 0, QSTR_TO_PSZ(poDevice->LinuxDevice               )))
            CHK (AaffWriteSegmentStr (pFile, AAFF_SEGNAME_MODEL       , 0, QSTR_TO_PSZ(poDevice->Model                     )))
            CHK (AaffWriteSegmentStr (pFile, AAFF_SEGNAME_SN          , 0, QSTR_TO_PSZ(poDevice->SerialNumber              )))
            CHK (AaffWriteSegmentStr (pFile, "CaseNumber"             , 0, QSTR_TO_PSZ(poDevice->Acquisition.CaseNumber    )))
            CHK (AaffWriteSegmentStr (pFile, "EvidenceNumber"         , 0, QSTR_TO_PSZ(poDevice->Acquisition.EvidenceNumber)))
            CHK (AaffWriteSegmentStr (pFile, "Examiner"               , 0, QSTR_TO_PSZ(poDevice->Acquisition.Examiner      )))
            CHK (AaffWriteSegmentStr (pFile, "Description"            , 0, QSTR_TO_PSZ(poDevice->Acquisition.Description   )))
            CHK (AaffWriteSegmentStr (pFile, "Notes"                  , 0, QSTR_TO_PSZ(poDevice->Acquisition.Notes         )))
         }

         poFile = pFile; // Only set poFile at the very end, so the CompressionThreads won't use it until everything is initialised
         return NO_ERROR;
      }

      //lint -save -esym(613,t_OutputFileAFF::poDevice)   Possible use of null pointer
      APIRET Write (t_pFifoBlock pFifoBlock)
      {
         if (!oHasCompressionThreads) // Do preprocessing if not already done in compression threads
         {
            t_pFifoBlock pPreprocessBlock;

            CHK_EXIT (t_Fifo::Create (poDevice->pFifoMemory, pPreprocessBlock, poDevice->FifoAllocBlockSize))
            pPreprocessBlock->DataSize = pFifoBlock->DataSize;
            CHK_EXIT (AaffPreprocess (&pPreprocessBlock->pAaffPreprocess, pFifoBlock->Buffer, pFifoBlock->DataSize, pPreprocessBlock->Buffer, pPreprocessBlock->BufferSize))
            if (pPreprocessBlock->pAaffPreprocess->Compressed)
            {
               CHK_EXIT (t_Fifo::Destroy (poDevice->pFifoMemory, pFifoBlock))
               pFifoBlock = pPreprocessBlock;
            }
            else
            {
               pFifoBlock->pAaffPreprocess = pPreprocessBlock->pAaffPreprocess;
               CHK_EXIT (t_Fifo::Destroy (poDevice->pFifoMemory, pPreprocessBlock))
            }
         }

         CHK (AaffWrite (poFile, pFifoBlock->pAaffPreprocess, pFifoBlock->Buffer, pFifoBlock->DataSize))

         return NO_ERROR;
      }
      //lint -restore

      APIRET Verify (t_pHashContextMD5 pHashContextMD5, t_pHashContextSHA256 pHashContextSHA256, quint64 *pPos)
      {
         unsigned int DataLen = poDevice->FifoBlockSize;
         APIRET       rc;

         if (*pPos == 0)
         {
            poVerifyBuff = UTIL_MEM_ALLOC (poDevice->FifoBlockSize);
            if (poVerifyBuff == NULL)
               CHK (ERROR_THREADWRITE_MALLOC_FAILED);
         }
         rc = AaffReadNextPage (poFile, (unsigned char *) poVerifyBuff, &DataLen);
         if (rc)
         {
            LOG_INFO ("AFF image verification failed: %s", ToolErrorMessage (rc))
            return ERROR_THREADWRITE_VERIFY_FAILED;
         }

         if (pHashContextMD5)    CHK_EXIT (HashMD5Append    (pHashContextMD5   , poVerifyBuff, DataLen))
         if (pHashContextSHA256) CHK_EXIT (HashSHA256Append (pHashContextSHA256, poVerifyBuff, DataLen))

         *pPos += DataLen;

         return NO_ERROR;
      }

      //lint -save -esym(613,t_OutputFileAFF::poDevice)   Possible use of null pointer
      APIRET Close (void)
      {
         QList<quint64> BadSectors;
         int            Seconds;

         if (poFile == NULL)
            CHK (ERROR_THREADWRITE_NOT_OPENED)

         if (oVerification)
         {
            CHK (AaffClose (&poFile))
            if (poVerifyBuff)
            {
               UTIL_MEM_FREE (poVerifyBuff);
               poVerifyBuff = NULL;
            }
         }
         else
         {
            CHK (poDevice->GetBadSectors (BadSectors, false))
            Seconds  = poDevice->StartTimestamp.secsTo (QDateTime::currentDateTime());

            CHK (AaffClose (&poFile, BadSectors.count(), (unsigned char*)&poDevice->MD5Digest, (unsigned char*)&poDevice->SHA256Digest, Seconds))
         }

         poFile = NULL;

         return NO_ERROR;
      }
      //lint -restore

      void *GetFileHandle (void)
      {
         return poFile;
      }

      bool Opened (void)
      {
         return (poFile != NULL);
      }

   private:
      t_pAaff    poFile;
      t_pDevice  poDevice;
      void      *poVerifyBuff;
      bool        oHasCompressionThreads;
      bool        oVerification;
};


class t_ThreadWriteLocal
{
   public:
      t_ThreadWriteLocal (void) :
         pOutputFile        (NULL),
         pSlowDownRequest   (NULL),
         pDevice            (NULL),
         FileHandleRequests (0)
      {}

   public:
      t_OutputFile *pOutputFile;
      bool         *pSlowDownRequest;
      t_pDevice     pDevice;
      int            FileHandleRequests;
      QMutex         SemHandle;
};


t_ThreadWrite::t_ThreadWrite(void)
{
   CHK_EXIT (ERROR_THREADWRITE_CONSTRUCTOR_NOT_SUPPORTED)
} //lint !e1401 not initialised

t_ThreadWrite::t_ThreadWrite (t_pDevice pDevice, bool *pSlowDownRequest)
   :pOwn(NULL)
{
   static bool Initialised = false;

   if (!Initialised)
   {
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_THREADWRITE_OPEN_FAILED              ))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_THREADWRITE_WRITE_FAILED             ))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_THREADWRITE_CLOSE_FAILED             ))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_THREADWRITE_NOT_OPENED               ))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_THREADWRITE_INVALID_FORMAT           ))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_THREADWRITE_CONSTRUCTOR_NOT_SUPPORTED))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_THREADWRITE_WRONG_BLOCK              ))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_THREADWRITE_OUT_OF_SEQUENCE_BLOCK    ))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_THREADWRITE_HANDLE_NOT_YET_AVAILABLE ))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_THREADWRITE_HANDLE_TIMEOUT           ))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_THREADWRITE_LIBEWF_FAILED            ))

      Initialised = true;
   }

   pOwn = new t_ThreadWriteLocal;
   pOwn->pDevice            = pDevice;
   pOwn->pOutputFile        = NULL;
   pOwn->pSlowDownRequest   = pSlowDownRequest;
   pOwn->FileHandleRequests = 0;

   CHK_QT_EXIT (connect (this, SIGNAL(finished()), this, SLOT(SlotFinished())))
}

t_ThreadWrite::~t_ThreadWrite (void)
{
   delete pOwn;
}


static APIRET ThreadWriteDebugCheckLibewfData (t_pFifoBlock pFifoBlock)
{
   unsigned char *pBufferUncompressed;
   uLongf          UncompressedSize;
   int             zrc=99;
   uint32_t        CRC1;
   uint32_t        CRC2;
   bool            Error;

   Error = !pFifoBlock->EwfPreprocessed;

   // Get the original data
   // ---------------------
   if (pFifoBlock->EwfCompressionUsed)
   {
      UncompressedSize = pFifoBlock->DataSize + 4096; // DataSize should be enough for the uncompression buffer, as the uncompressed data originally had that
                                                      // size, but as this fn is used for debugging and we do not know what the bug might be, we add 4K for safety
      pBufferUncompressed = (unsigned char *) UTIL_MEM_ALLOC (UncompressedSize);
      if (pBufferUncompressed == NULL)
      {
         LOG_ERROR ("malloc returned NULL")
         CHK_EXIT (1967)
      }
      zrc = uncompress ((Bytef*)pBufferUncompressed, &UncompressedSize, pFifoBlock->Buffer, pFifoBlock->EwfDataSize);
      Error = Error || (zrc != Z_OK);
      Error = Error || (UncompressedSize != (unsigned) pFifoBlock->DataSize);

      ((char *)&CRC1)[0] = ((char *)&pFifoBlock->EwfChunkCRC)[3];
      ((char *)&CRC1)[1] = ((char *)&pFifoBlock->EwfChunkCRC)[2];
      ((char *)&CRC1)[2] = ((char *)&pFifoBlock->EwfChunkCRC)[1];
      ((char *)&CRC1)[3] = ((char *)&pFifoBlock->EwfChunkCRC)[0];
   }
   else
   {
      pBufferUncompressed = pFifoBlock->Buffer;
      UncompressedSize = pFifoBlock->DataSize;
      CRC1 = pFifoBlock->EwfChunkCRC;
   }

   // Get the CRC
   // -----------
   CRC2 = adler32 (1, (Bytef*)pBufferUncompressed, UncompressedSize);
   Error = Error || (CRC1 != CRC2);

   // Clean up
   // --------
   if (pFifoBlock->EwfCompressionUsed)
   {
      UTIL_MEM_FREE (pBufferUncompressed);  //lint !e673 Possibly inappropriate deallocation
   }

   if (Error)
   {
      LOG_ERROR ("zrc=%d CRC1=%08X CRC2=%08X Size1=%d Size2=%lu EwfPreprocessed=%d, EwfCompressionUsed=%d EwfWriteCRC=%d ",
                  zrc, CRC1, CRC2, pFifoBlock->DataSize, UncompressedSize,
                                   pFifoBlock->EwfPreprocessed?1:0,
                                   pFifoBlock->EwfCompressionUsed,
                                   pFifoBlock->EwfWriteCRC)
   }
   return NO_ERROR;
}

//lint -save -esym(613,pOutputFile)   Possible use of null pointer
void t_ThreadWrite::run (void)
{
   t_pDevice     pDevice;
   t_pFifoBlock  pFifoBlock;
   t_OutputFile *pOutputFile= NULL;
   bool           Finished  = false;
   bool           FileHandleReleased;
   quint64        Blocks    = 0;
   quint64        Written   = 0;
   APIRET         rc;

   LOG_INFO ("Acquisition of %s: Write thread started", QSTR_TO_PSZ (pOwn->pDevice->LinuxDevice))

   pDevice = pOwn->pDevice;
   pDevice->SetCurrentWritePos (0LL);

   CHK_EXIT (DeleteImageFiles (false))

   switch (pDevice->Acquisition.Format)
   {
      case t_File::DD:  pOutputFile = new t_OutputFileDD;  break;
      case t_File::EWF: pOutputFile = new t_OutputFileEWF; break;
      case t_File::AFF: pOutputFile = new t_OutputFileAFF; break;
      default: CHK_EXIT (ERROR_THREADWRITE_INVALID_FORMAT)
   }

   rc = pOutputFile->Open (pDevice, false);
   if (rc == ERROR_THREADWRITE_OPEN_FAILED)
   {
      LOG_INFO ("Could not open destination file %s", QSTR_TO_PSZ (pOwn->pDevice->Acquisition.ImageFilename))
      pDevice->AbortReason  = t_Device::ThreadWriteWriteError;
      pDevice->AbortRequest = true;
   }
   else
   {
      CHK_EXIT (rc)
   }

   pOwn->SemHandle.lock ();
   pOwn->pOutputFile = pOutputFile;  // Only set pOwn->pOutputFile here, after initialisation completed successfully. The reason is,
   pOwn->SemHandle.unlock ();        // that other threads should remain blocked in t_ThreadWrite::GetpFileHandle until this point.

   while (!Finished && !pDevice->AbortRequest)
   {
      if (*(pOwn->pSlowDownRequest))
         msleep (THREADWRITE_SLOWDOWN_SLEEP);

      CHK_EXIT (pDevice->pFifoWrite->Get (pFifoBlock))
      if (pFifoBlock)
      {
//         t_Fifo::LogBlock (pDevice->pFifoMemory, pFifoBlock, 'w');
         if (pFifoBlock->Nr != Blocks)
         {
            LOG_ERROR ("Fifo block number out of sequence. Expected: %Ld Received: %Ld", Blocks, pFifoBlock->Nr)
//            t_Fifo::LogBlock (pDevice->pFifoMemory, pFifoBlock, 'e');
            CHK_EXIT (ERROR_THREADWRITE_OUT_OF_SEQUENCE_BLOCK)
        }
         Blocks++;
         rc = pOwn->pOutputFile->Write (pFifoBlock);

         pDevice->IncCurrentWritePos (pFifoBlock->DataSize);
         if ((rc == ERROR_THREADWRITE_WRITE_FAILED) || 
             (rc == ERROR_THREADWRITE_OPEN_FAILED))
         {
            const char *pAction = (rc == ERROR_THREADWRITE_WRITE_FAILED) ? "write to" : "open";
            LOG_ERROR ("Could not %s destination file %s", pAction, QSTR_TO_PSZ (pOwn->pDevice->Acquisition.ImageFilename))
            LOG_INFO ("Last block sizes: %d - %d - %zd ", pFifoBlock->BufferSize, pFifoBlock->DataSize, pFifoBlock->EwfDataSize)
            pDevice->AbortReason  = t_Device::ThreadWriteWriteError;
            pDevice->AbortRequest = true;
         }
         else
         {
            CHK_EXIT (rc)
         }
         if (pDevice->HasCompressionThreads() && (pDevice->Acquisition.Format == t_File::EWF) && CONFIG(CheckEwfData))
            CHK_EXIT (ThreadWriteDebugCheckLibewfData (pFifoBlock))

         Written += pFifoBlock->DataSize;
         Finished = (Written >= pDevice->Size);

         CHK_EXIT (t_Fifo::Destroy (pDevice->pFifoMemory, pFifoBlock))
      }
      else
      {
         LOG_INFO ("Dummy block")
         Finished = true;
      }
   }

   // Wait until file handle released and close it
   // --------------------------------------------
   LOG_INFO ("Waiting for all other threads using the file handle to finish")
   do
   {
      bool AskedForRelease = false;

      pOwn->SemHandle.lock();
      FileHandleReleased = (pOwn->FileHandleRequests == 0);
      pOwn->SemHandle.unlock();
      if (!FileHandleReleased)
      {
         if (!AskedForRelease)
         {
            emit SignalFreeMyHandle (pOwn->pDevice);
            AskedForRelease = true;
         }
         msleep (THREADWRITE_WAIT_FOR_HANDLE_GRANULARITY);
      }
   } while (!FileHandleReleased);

   LOG_INFO ("Closing output file")
   if (pOwn->pOutputFile->Opened())
   {
      rc = pOwn->pOutputFile->Close ();
      if (rc == ERROR_THREADWRITE_CLOSE_FAILED)
      {
         pDevice->AbortReason  = t_Device::ThreadWriteWriteError;
         pDevice->AbortRequest = true;
      }
   }

   // Image verification
   // ------------------
   if (pDevice->Acquisition.VerifyDst && !pDevice->AbortRequest)
   {
      t_HashContextMD5      HashContextMD5;
      t_HashContextSHA256   HashContextSHA256;
      t_pHashContextMD5    pHashContextMD5    = &HashContextMD5;
      t_pHashContextSHA256 pHashContextSHA256 = &HashContextSHA256;
      quint64               Pos = 0;

      if (pDevice->Acquisition.CalcMD5)
           CHK_EXIT (HashMD5Init (pHashContextMD5))
      else pHashContextMD5 = NULL;

      if (pDevice->Acquisition.CalcSHA256)
           CHK_EXIT (HashSHA256Init (pHashContextSHA256))
      else pHashContextSHA256 = NULL;

      LOG_INFO ("Reopening output file for verification")

      rc = pOutputFile->Open (pDevice, true);
      if (rc == ERROR_THREADWRITE_OPEN_FAILED)
      {
         LOG_INFO ("Could not open destination file %s for verification", QSTR_TO_PSZ (pOwn->pDevice->Acquisition.ImageFilename))
         pDevice->AbortReason  = t_Device::ThreadWriteVerifyError;
         pDevice->AbortRequest = true;
      }
      else
      {
         CHK_EXIT (rc)
      }

      Finished = false;
      while (!Finished && !pDevice->AbortRequest)
      {
         rc = pOwn->pOutputFile->Verify (pHashContextMD5, pHashContextSHA256, &Pos);
         if (rc == ERROR_THREADWRITE_VERIFY_FAILED)
         {
            LOG_ERROR ("Could not verify image %s", QSTR_TO_PSZ (pOwn->pDevice->Acquisition.ImageFilename))
            pDevice->AbortReason  = t_Device::ThreadWriteVerifyError;
            pDevice->AbortRequest = true;
         }
         else
         {
            CHK_EXIT (rc)
            pDevice->SetCurrentVerifyPosDst (Pos);
            Finished = (Pos >= pDevice->Size);
            if (Finished)
            {
               if (pDevice->Acquisition.CalcMD5)    CHK_EXIT (HashMD5Digest    (pHashContextMD5   , &pDevice->MD5DigestVerifyDst   ))
               if (pDevice->Acquisition.CalcSHA256) CHK_EXIT (HashSHA256Digest (pHashContextSHA256, &pDevice->SHA256DigestVerifyDst))
            }
         }
      }
      LOG_INFO ("Verification finished, closing output file")
      if (pOwn->pOutputFile->Opened())
      {
         rc = pOwn->pOutputFile->Close ();
         if (rc == ERROR_THREADWRITE_CLOSE_FAILED)
         {
            pDevice->AbortReason  = t_Device::ThreadWriteVerifyError;
            pDevice->AbortRequest = true;
         }
      }
   }

   // Finish
   // ------
   delete pOwn->pOutputFile;
   pOwn->pOutputFile = NULL;

   if (pDevice->AbortRequest && pDevice->DeleteAfterAbort)
   {
      pDevice->State = t_Device::Cleanup;
      CHK_EXIT (DeleteImageFiles (true))
   }

   LOG_INFO ("Write thread exits now (device %s, %Ld blocks processed, %Ld bytes written to output file)", QSTR_TO_PSZ (pOwn->pDevice->LinuxDevice), Blocks, pDevice->GetCurrentWritePos ())
}
//lint -restore

static APIRET DeleteImageFiles0 (t_pDevice pDevice, const QDir &Dir, const QStringList &NameFilter)
{
   QFileInfoList  FileInfoList;
   QFileInfo      FileInfo;
   QString        Info;
   bool           Success;

   FileInfoList = Dir.entryInfoList (NameFilter, QDir::Files, QDir::Name);
   while (!FileInfoList.isEmpty())
   {
      FileInfo = FileInfoList.takeFirst();
      CHK (pDevice->SetMessage (QObject::tr("Deleting %1") .arg(FileInfo.fileName())))

      Info = "Deleting " + FileInfo.absoluteFilePath() + " - ";
      Success = QFile::remove (FileInfo.absoluteFilePath());
      if (Success)
           Info += "successfull";
      else Info += "could not be deleted";
      LOG_INFO ("%s", QSTR_TO_PSZ(Info))
   }

   return NO_ERROR;
}

APIRET t_ThreadWrite::DeleteImageFiles (bool AlsoDeleteInfoFile)
{
   t_pDevice pDevice = pOwn->pDevice;
   QDir       DirImage (pDevice->Acquisition.ImagePath);
   QDir       DirInfo  (pDevice->Acquisition.InfoPath );
   QString    ExtensionImage;

   if (!pDevice->Acquisition.Clone)
   {
      LOG_INFO ("Deleting existing image files of the same name")
      CHK (t_File::GetFormatExtension (pDevice, &ExtensionImage))
      CHK (DeleteImageFiles0 (pDevice, DirImage, QStringList(pDevice->Acquisition.ImageFilename + ExtensionImage)))
   }
   if (AlsoDeleteInfoFile)
   {
      LOG_INFO ("Deleting existing info file of the same name")
      CHK (DeleteImageFiles0 (pDevice, DirImage, QStringList(pDevice->Acquisition.InfoFilename + t_File::pExtensionInfo)))
   }
   CHK (pDevice->SetMessage (QString()))

   return NO_ERROR;
}

void t_ThreadWrite::SlotFinished (void)
{
   emit SignalEnded (pOwn->pDevice);
}

APIRET t_ThreadWrite::GetpFileHandle0 (void **ppHandle)
{
   *ppHandle = NULL;

   if (!isRunning())       return ERROR_THREADWRITE_HANDLE_NOT_YET_AVAILABLE;   // This should normally never happen, as the threads are all created together and only started after that
   if (!pOwn->pOutputFile) return ERROR_THREADWRITE_HANDLE_NOT_YET_AVAILABLE;

   *ppHandle = pOwn->pOutputFile->GetFileHandle();
   if (!*ppHandle)         return ERROR_THREADWRITE_HANDLE_NOT_YET_AVAILABLE;

   return NO_ERROR;
}

APIRET t_ThreadWrite::GetpFileHandle (void **ppHandle)
{
   unsigned int Wait=0;
   APIRET       rc;

   *ppHandle = NULL;
   do
   {
      if (pOwn->pDevice->AbortRequest)  // May happen, for example, if the write thread wasn't able to open the destination file.
         break;

      pOwn->SemHandle.lock();
      rc = GetpFileHandle0 (ppHandle);
      if ((rc != ERROR_THREADWRITE_HANDLE_NOT_YET_AVAILABLE) && (rc != NO_ERROR))
      {
         pOwn->SemHandle.unlock();
         CHK (rc)
      }
      if (*ppHandle == NULL)
      {
         pOwn->SemHandle.unlock();

         Wait += THREADWRITE_WAIT_FOR_HANDLE_GRANULARITY;
         if (Wait > THREADWRITE_WAIT_FOR_HANDLE)
            CHK_EXIT (ERROR_THREADWRITE_HANDLE_TIMEOUT)

         msleep (THREADWRITE_WAIT_FOR_HANDLE_GRANULARITY);
      }
   } while (*ppHandle == NULL);

   pOwn->FileHandleRequests++;
   pOwn->SemHandle.unlock();

   return NO_ERROR;
}

APIRET t_ThreadWrite::ReleaseFileHandle (void)
{
   pOwn->SemHandle.lock();
   pOwn->FileHandleRequests--;
   pOwn->SemHandle.unlock();

   return NO_ERROR;
}

