""" RUR-PLE: Roberge's Used Robot - a Python Learning Environment
    rur_robot_factory.py - see description below
    Version 0.7
    Author: Andre Roberge    Copyright  2005
    andre.roberge@gmail.com

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

rur_robot_factory includes four classes:

    Robot_brain1(), which incorporates the basic logic of the robot, including
    a limited ability to sense the outside world.

    Robot_brain2(), which incorporates an "advanced" version, capable of more
    remote sensing ability, akin to Pattis's original Karel as well as
    turning right directly.

    Used_robot() subclasses Robot_brain1 and adds a body (i.e. physical
    representation: images for visual display, time delay for execution
    of movements, etc.)

    New_improved_robot() subclasses both Used_robot (for physical display) and
    Robot_brain2 for the improved logic.
    """
import os
import rur_misc,utils
if rur_misc.WX_VERSION == 0:
    try:
        import wxversion     # if multiple versions are installed
        wxversion.select("2.5")
        rur_misc.WX_VERSION = 2.5
    except Exception, info:
        print info
        print "Can not import wxversion in RobotFactory.py "
        print "Will attempt to import existing version of wxPython."
        #most likely only version 2.4 (or earlier) installed
        rur_misc.WX_VERSION = 2 # placeholder for some other version

from wxPython.wx import *
import rur_dialogs

#---------------------------------------------------------------------------

class Robot_brain1(object):

    _directions = [ (0, 1), (-1, 0), (0, -1), (1, 0) ]
    _orient_dict = { 'E': 3, 'S': 2, 'W': 1, 'N': 0}

    def __init__(self, parent=None, avenues=1, streets=1,
                 orient_key = 'E', beepers=0):
        # East; value by default - tied to lessons
        self.parent = parent
        #--- Basic variables
        self._beeper_bag = beepers
        self._x = avenues
        self._y = streets
        self._facing = self._orient_dict[orient_key.upper()]

    #--- "public" getter
    def getPos(self):
        """ returns current robot position; intended to be
            accessed by user-defined program."""
        return self._x, self._y

    #--- "private" getters and setters
    def _setPos(self, x, y):
        """ sets robot position (teleport); intended to be
            accessed only through GUI, not by user-defined program."""
        self._x = x
        self._y = y

    def _getOrientation(self):
        """ returns orientation (0, 1, 2, 3); needed for drawing trace.
            Not intended to be accessed by user-defined program."""
        return self._facing

    def _getOrientationKey(self):
        """ returns orientation key ('N', 'E', 'S', W').
            Not intended to be accessed by user-defined program."""
        for key in self._orient_dict.keys():
            if self._orient_dict[key] == self._facing:
                return key

    def _getInfoTuple(self):
        """ returns (avenue, street, orientation, beepers).
            Not intended to be accessed by user-defined program."""
        return self._x, self._y, self._getOrientationKey(), self._beeper_bag

    def _getInfoString(self):
        """ returns (avenue, street, orientation, beepers).
            Not intended to be accessed by user-defined program."""
        return str(self._getInfoTuple())

    #--- built-in tests

    def front_is_clear(self):
        ''' True if no wall or border in front of robot'''
        col = 2*self._x - 1
        row = 2*self._y - 1
        xx, yy = self._directions[self._facing]
        return self.parent.isClear(col+xx, row+yy)

    def facing_North(self):
        ''' True if Robot facing North'''
        if self._facing == 0:
            return True
        else:
            return False

    def any_beepers_in_beeper_bag(self):
        '''True if some beepers are left in Robot's bag'''
        if self._beeper_bag == 0:
            return False
        else:
            return True

    def next_to_a_beeper(self):
        '''True if beepers are present at current robot position.'''
        if (self._x, self._y) in self.parent.beepers_dict:
            return True
        else:
            return False

    #--- Actions

    def move(self):
        '''Robot moves one street/avenue in direction where it is facing'''
        if self.front_is_clear():
            xx, yy = self._directions[self._facing]
            self._x += xx
            self._y += yy
        else:
            mesg = _("move failed.\n") + \
                   _("You must make sure that there is no wall in front of me!")
            raise rur_dialogs.HitWallException(mesg)

    def turn_off(self):    # is this really needed?
        mesg = _("  I obey your command: turning myself off.")
        raise rur_dialogs.NormalEnd(mesg)

    def turn_left(self):
        '''Robot turns left by 90 degrees.'''
        self._facing += 1
        self._facing %= 4

    def put_beeper(self):
        '''Robot put one beeper down at current location.'''
        if self.any_beepers_in_beeper_bag():
            self._beeper_bag -= 1
            self.parent.addOneBeeper(self._x, self._y)
        else:
            mesg = _("put_beeper failed.\n You are not carrying any beepers.")
            raise rur_dialogs.PutBeeperException(mesg)

    def pick_beeper(self):
        '''Robot picks one beeper up at current location.'''
        if self.next_to_a_beeper():
            self.parent.removeOneBeeper(self._x, self._y)
            self._beeper_bag += 1
        else:
            mesg = _("pick_beeper failed.\n") + \
                   _("You must be next to a beeper before you can pick it up.")
            raise rur_dialogs.PickBeeperException(mesg)


class Robot_brain2(Robot_brain1):

    """ This class has NOT been fully tested yet."""
    def __init__(self, parent=None, avenues=1, streets=1, orient_key = 'E',
                 beepers=0):
        Robot_brain1.__init__(self, parent, avenues, streets,
                              orient_key, beepers)

    #--- Additional built-in tests

    def left_is_clear(self):
        '''Returns True if no walls or borders are to the immediate left
           of the robot.'''
        col = 2*self._x - 1
        row = 2*self._y - 1
        facing = self._facing + 1
        facing %= 4
        xx, yy = self._directions[facing]
        if (col+xx, row+yy) in self.parent.walls_list:
            return False
        if (col+xx, row+yy) in self.parent.borders:
            return False
        else:
            return True

    def right_is_clear(self):
        '''Returns True if no walls or borders are to the immediate
           right of the robot.'''
        col = 2*self._x - 1
        row = 2*self._y - 1
        facing = self._facing + 3
        facing %= 4
        xx, yy = self._directions[facing]
        if (col+xx, row+yy) in self.parent.walls_list:
            return False
        if (col+xx, row+yy) in self.parent.borders:
            return False
        else:
            return True

    def facing_East(self):
        if self._facing == 3:
            return True
        else:
            return False

    def facing_South(self):
        if self._facing == 2:
            return True
        else:
            return False

    def facing_West(self):
        if self._facing == 1:
            return True
        else:
            return False

    #--- Additional action
    def turn_right(self):
        self._facing += 3
        self._facing %= 4

class Used_robot(Robot_brain1):
    """ Adds physical attributes """

    def __init__(self, parent=None, avenues=1, streets=1, orient_key = 'E',
                 beepers=0, name = 'robot', colour = 'grey'):
        Robot_brain1.__init__(self, parent, avenues, streets,
                              orient_key, beepers)
        self._delay = 0.3
        self.name = name
        self.colour = colour

        # The following are used to follow the robot trail
        self.line_trace = []
        self.set_trace_style(1, "sea green")  # default
    #--- Robot images
        # create a list of four objects
        self._image = [0, 1, 2, 3]
        base = utils.get_rootdir()
        # re-initialise it to robot images
        try:
            self._image[2] = wxImage(os.path.join(base,'bitmaps','robot_s.png'),
                              wxBITMAP_TYPE_PNG).ConvertToBitmap()
            self._image[0] = wxImage(os.path.join(base,'bitmaps','robot_n.png'),
                              wxBITMAP_TYPE_PNG).ConvertToBitmap()
            self._image[3] = wxImage(os.path.join(base,'bitmaps','robot_e.png'),
                              wxBITMAP_TYPE_PNG).ConvertToBitmap()
            self._image[1] = wxImage(os.path.join(base,'bitmaps','robot_w.png'),
                              wxBITMAP_TYPE_PNG).ConvertToBitmap()
        except Exception,info:
            print __name__, info
            print " Problem loading robot_?.png in RobotFactory.py"
        # offset based on above images for proper display
        # [obtained through trial and error]
        self.imageOffset = (12, 9)

        # image size (x, y) [all images equal]; for use in automatic scrolling
        self._image_size = self._image[0].GetWidth(), \
                           self._image[0].GetHeight()
        ## Note: for some reason, GetSize() did not work  using
        ## wxPython 2.4, which is why I used GetWidth() and GetHeight()

        # selecting the right image based on initial orientation
        self.robot_image = self._image[self._facing]

    #--- Action over-riden to handle images
    def turn_left(self):
        '''Robot turns left by 90 degrees, and image is updated.'''
        Robot_brain1.turn_left(self)
        self.robot_image = self._image[self._facing]

    def set_trace_style(self, style, colour = "sea green"):
        if style == 1:
           self.trace_offset = [(3, 3), (3, -3), (-3, -3), (-3, 3)]
           self.trace_width = 1
        elif style == 2:
           self.trace_offset = [(5, 5), (5, -5), (-5, -5), (-5, 5)]
           self.trace_width = 1
        elif style == 3:
           self.trace_offset = [(3, 3), (3, -3), (-3, -3), (-3, 3)]
           self.trace_width = 3
        elif style == 4:
           self.trace_offset = [(5, 5), (5, -5), (-5, -5), (-5, 5)]
           self.trace_width = 3
        elif style == 5:
            self.trace_offset = [(0, 0), (0, 0), (0, 0), (0, 0)]
            self.trace_width = 3
        else:
            self.trace_offset = [(0, 0), (0, 0), (0, 0), (0, 0)]
            self.trace_width = 0

        self.trace_colour = colour

## todo: transform the error message to be an exception as it
## doesn't display as it is.
    def set_delay(self, delay):
        '''Sets the delay value between robot actions.'''
        if delay >= 0 and delay <= 10:
            self._delay = delay
        else:
            mesg = "Setting delay failed.\n" + \
                   "Accepted values are between 0 and 10."
            rur_dialogs.rurMessageDialogs(mesg, "error")

    def get_delay(self):
        return self._delay

    delay = property(get_delay, set_delay, None, "Time between robot actions")

class New_improved_robot(Used_robot, Robot_brain2):
    """ Adds physical attributes and better logic."""
    # TODO: design "better looking" images for this robot.
    #--- Action over-riden to handle images
    def turn_right(self):
        '''Robot turns right by 90 degrees, and image is updated.'''
        Robot_brain2.turn_right(self)
        self.robot_image = self._image[self._facing]
