/*  GWhere
 *  Copyright (C) 2000  Sbastien LECACHEUR
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


/*!
 * @header	gwdiskinfo.h
 *		GWhere<br>
 *		Copyright (C) 2000  Sbastien LECACHEUR<br><br>
 *		This Distributed under the terms of the <a href="http://www.gnu.org/copyleft/gpl.html">GNU General Public Licence</a><br><br>
 *		This component was developed for the GWhere project.<br>
 *		It's an object for disk representation. His structure gets many informations as name, capacity, size, label,...<br><br>
 * @abstract	This is the disk data structure.
 * @author	Sbastien LECACHEUR <zero@gwhere.org>
 * @version	1.0 2001/08/21
 * @see		gtk/gtk.h
 * @see		gwsupport.h
 * @see		gwcategory.h
 * @link		GWhere Web Site <http://www.gwhere.org>
 * @link		GTK+ Web Site <http://www.gtk.org>
 * @ref		gwdiskinfo.c
 * @author	Author <zero@gwhere.org>
 */


#ifndef GW_DATA_DISK_INFO_H
#define GW_DATA_DISK_INFO_H


#include <gtk/gtk.h>
#include "gwcategory.h"


/*!
 * @struct	disk_info
 * @abstract	Disk information data structure
 * @discussion	Disk information is an opaque data structure containing information about disk properties and informations.
 */
struct disk_info;


/*!
 * @function	disk_info_new
 * @abstract	Creates a new disk information.
 * @discussion	This function creates a new disk information. It's an allocation function.
 * @result	a pointer to the disk information structure.<br>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
struct disk_info * disk_info_new ( void);


/*!
 * @function	disk_info_set_name
 * @abstract	Sets the disk name.
 * @discussion	This function sets the disk name used in the catalog.<br>
 * @param	p		the disk information to set the name
 * @param	name		the name
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint disk_info_set_name ( struct disk_info *p, gchar *name);


/*!
 * @function	disk_info_set_fsname
 * @abstract	Sets the disk file system name.
 * @discussion	This function sets the disk file system name.<br>
 * @param	p		the disk information to set the file system name
 * @param	fsname		the file system name
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint disk_info_set_fsname ( struct disk_info *p, gchar *fsname);


/*!
 * @function	disk_info_set_dir
 * @abstract	Sets the disk directory.
 * @discussion	This function sets the disk directory. This information is the mount point where was the disk when it was added to the catalog.<br>
 * @param	p		the disk information to set the directory
 * @param	dir		the directory
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint disk_info_set_dir ( struct disk_info *p, gchar *dir);


/*!
 * @function	disk_info_set_num
 * @abstract	Sets the disk number archive.
 * @discussion	This function sets the disk number archive used in the catalog. The number archive is a single index.<br>
 * @param	p		the disk information to set the number archive
 * @param	num		the number archive
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint disk_info_set_num ( struct disk_info *p, gint num);


/*!
 * @function	disk_info_set_type
 * @abstract	Sets the disk type.
 * @discussion	This function sets the disk type. It's the types used in the system as <i>ext2, iso9660, vfat,...</i><br>
 * @param	p		the disk information to set the type
 * @param	type		the type
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint disk_info_set_type ( struct disk_info *p, gchar *type);


/*!
 * @function	disk_info_set_volume
 * @abstract	Sets the disk volume name.
 * @discussion	This function sets the disk volume name. The disk volume is the label of the disk.<br>
 * @param	p		the disk information to set the volume name
 * @param	volume		the volume name
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint disk_info_set_volume ( struct disk_info *p, gchar *volume);


/*!
 * @function	disk_info_set_full
 * @abstract	Sets the disk full space.
 * @discussion	This function sets the disk full space.<br>
 * @param	p		the disk information to set the full space
 * @param	full		the full space
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint disk_info_set_full ( struct disk_info *p, guint64 full);


/*!
 * @function	disk_info_set_free
 * @abstract	Sets the disk free space.
 * @discussion	This function sets the disk free space.<br>
 * @param	p		the disk information to set the free space
 * @param	free		the full capacity
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint disk_info_set_free ( struct disk_info *p, guint64 free);


/*!
 * @function	disk_info_set_date
 * @abstract	Sets the disk update date.
 * @discussion	This function sets the disk update date.<br>
 * @param	p		the disk information to set the update date
 * @param	date		the update date
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint disk_info_set_date ( struct disk_info *p, gulong date);


/*!
 * @function	disk_info_set_serial
 * @abstract	Sets the disk serial number.
 * @discussion	This function sets the disk serial number.<br>
 * @param	p		the disk information to set the serial number
 * @param	serial		the serial number
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint disk_info_set_serial ( struct disk_info *p, gchar *serial);


/*!
 * @function	disk_info_set_category
 * @abstract	Sets the disk category.
 * @discussion	This function sets the disk category.<br>
 * @param	p		the disk information to set the category
 * @param	category	the category
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint disk_info_set_category ( struct disk_info *p, struct category *category);


/*!
 * @function	disk_info_set_description
 * @abstract	Sets the disk description.
 * @discussion	This function sets the disk description.<br>
 * @param	p		the disk information to set the description
 * @param	description	the description
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint disk_info_set_description ( struct disk_info *p, gchar *description);


/*!
 * @function	disk_info_get_name
 * @abstract	Gets the disk's name.
 * @discussion	This function gets the disk's name.<br>
 * @param	p		the disk information to get the name
 * @result	the name.<br>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
gchar *	disk_info_get_name ( struct disk_info *p);


/*!
 * @function	disk_info_get_fsname
 * @abstract	Gets the disk's file system name.
 * @discussion	This function gets the disk's file system name.<br>
 * @param	p		the disk information to get the file system name
 * @result	the file system name.<br>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
gchar *	disk_info_get_fsname ( struct disk_info *p);


/*!
 * @function	disk_info_get_dir
 * @abstract	Gets the disk's directory.
 * @discussion	This function gets the disk's directory.<br>
 * @param	p		the disk information to get the directory
 * @result	the directory.<br>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
gchar *	disk_info_get_dir ( struct disk_info *p);


/*!
 * @function	disk_info_get_num
 * @abstract	Gets the disk's number archive.
 * @discussion	This function gets the disk's number archive.<br>
 * @param	p		the disk information to get the number archive
 * @result	the number archive.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint	disk_info_get_num ( struct disk_info *p);


/*!
 * @function	disk_info_get_type
 * @abstract	Gets the disk's type.
 * @discussion	This function gets the disk's type.<br>
 * @param	p		the disk information to get the type
 * @result	the type.<br>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
gchar *	disk_info_get_type ( struct disk_info *p);


/*!
 * @function	disk_info_get_volume
 * @abstract	Gets the disk's volume name.
 * @discussion	This function gets the disk's volume name.<br>
 * @param	p		the disk information to get the volume name
 * @result	the volume name.<br>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
gchar *	disk_info_get_volume ( struct disk_info *p);


/*!
 * @function	disk_info_get_capacity
 * @abstract	Gets the disk's capacity.
 * @discussion	This function gets the disk's capacity.<br>
 * @param	p		the disk information to get the capacity
 * @result	the capacity.<br>
 *		Returns <i>0</i> when an error occured.<br>
 */
guint64	disk_info_get_capacity ( struct disk_info *p);


/*!
 * @function	disk_info_get_full
 * @abstract	Gets the disk's full space.
 * @discussion	This function gets the disk's full space.<br>
 * @param	p		the disk information to get the full space
 * @result	the full  space.<br>
 *		Returns <i>0</i> when an error occured.<br>
 */
guint64 disk_info_get_full ( struct disk_info *p);


/*!
 * @function	disk_info_get_free
 * @abstract	Gets the disk's free space.
 * @discussion	This function gets the disk's free space.<br>
 * @param	p		the disk information to get the free space
 * @result	the free.<br>
 *		Returns <i>0</i> when an error occured.<br>
 */
guint64 disk_info_get_free ( struct disk_info *p);


/*!
 * @function	disk_info_get_date
 * @abstract	Gets the disk's date.
 * @discussion	This function gets the disk's date in seconds since 1970.<br>
 * @param	p		the disk information to get the date
 * @result	the date.<br>
 *		Returns <i>0</i> when an error occured.<br>
 */
gulong	disk_info_get_date ( struct disk_info *p);


/*!
 * @function	disk_info_get_date_to_gchar
 * @abstract	Gets the disk's date.
 * @discussion	This function gets the disk's date in char format.<br>
 * @param	p		the disk information to get the date
 * @result	the date. Must be freed with g_free().<br>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
gchar * disk_info_get_date_to_gchar ( struct disk_info *p);


/*!
 * @function	disk_info_get_serial
 * @abstract	Gets the disk's serial number.
 * @discussion	This function gets the disk's serial number.<br>
 * @param	p		the disk information to get the serial number
 * @result	the serial number.<br>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
gchar *	disk_info_get_serial ( struct disk_info *p);


/*!
 * @function	disk_info_get_category
 * @abstract	Gets the disk's category.
 * @discussion	This function gets the disk's category.<br>
 * @param	p		the disk information to get the category
 * @result	the category.<br>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
struct category * disk_info_get_category ( struct disk_info *p);


/*!
 * @function	disk_info_get_category_name
 * @abstract	Gets the disk's category name.
 * @discussion	This function gets the disk's category name.<br>
 * @param	p		the disk information to get the category name
 * @result	the category name.<br>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
gchar * disk_info_get_category_name ( struct disk_info *p);


/*!
 * @function	disk_info_get_category_index
 * @abstract	Gets the disk's category index.
 * @discussion	This function gets the disk's category index.<br>
 * @param	p		the disk information to get the category index
 * @result	the category index.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint disk_info_get_category_index ( struct disk_info *p);


/*!
 * @function	disk_info_get_description
 * @abstract	Gets the disk's description.
 * @discussion	This function gets the disk's description.<br>
 * @param	p		the disk information to get the description
 * @result	the description.<br>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
gchar *	disk_info_get_description ( struct disk_info *p);


/*!
 * @function	disk_info_get_pixmap
 * @abstract	Gets the disk's pixmap.
 * @discussion	This function gets the disk's pixmap for graphical represention.<br>
 * @param	p		the disk information to get the pixmap
 * @result	the pixmap.<br>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
GdkPixmap * disk_info_get_pixmap ( struct disk_info *p);


/*!
 * @function	disk_info_get_bitmap
 * @abstract	Gets the disk's bitmap.
 * @discussion	This function gets the disk's bitmap for graphical representation.<br>
 * @param	p		the disk information to get the bitmap
 * @result	the bitmap.<br>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
GdkBitmap * disk_info_get_bitmap ( struct disk_info *p);


/*!
 * @function	disk_info_free
 * @abstract	Frees datas structure.
 * @discussion	This function frees the disk information's datas.
 *		This function doesn't desallocate the category datas structure (because it's a global information stored in catalog file information datas structure).
 * @param	p		the disk information to free
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint disk_info_free ( struct disk_info *p);


#endif
