/* 
 * Copyright (C) 2003-2004 the xine project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: play_item.c,v 1.21 2005/01/29 00:54:45 dsalt Exp $
 *
 * playlist item / media mark:
 * an mrl + options (e.g. volume, start time, brightness/contrast ...)
 */

#include "globals.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <glib.h>
#include <pthread.h>

#include "play_item.h"
#include "globals.h"
#include "utils.h"
#include "playlist.h"
#include "script_engine.h"
#include "engine.h"
#include "player.h"

play_item_t *play_item_new (const char *title, const char *mrl, 
			    int start_time)
{
  play_item_t *play_item = (play_item_t *) malloc (sizeof (play_item_t));

  play_item->untitled = !title;
  if (!title)
  {
    /* find the leafname of the MRL */
    char *end, *retitle = strdup (mrl);
    if ((end = strchr (retitle, '#')))
      *end = 0; /* lose parameters suffix (if present) */

    title = strrchr (retitle, '/');
    if (title && !title[1]) /* ends in '/' - look for preceding '/' */
    {
      while (*title == '/')
	if (--title == retitle)
	  break;
      while (*title != '/')
	if (--title < retitle)
	  break;
    }

    /* at this point, title >= retitle - 1
     * retitle[-1] being invalid doesn't matter
     */

    if (title[1])
    {
      int i = 0, j = -1;

      /* Strip the extension and suffix (if present) */
      if ((end = strrchr (title + 1, '.')))
	*end = 0;

      /* URL-decode the leafname */
      while (title[++i])
      {
	if (title[i] == '%')
	{
	  char c[3] = { title[i+1], title[i+2], 0 };
	  i += 2;
	  retitle[++j] = (char) strtol (c, NULL, 16);
	}
	else
	  retitle[++j] = title[i];
      }
      /* NUL-terminate, strip trailing / */
      retitle[j + (retitle[j] != '/')] = 0;
      play_item->title = realloc (retitle, strlen (retitle) + 1);
    }
    else
    {
      free (retitle);
      play_item->title = strdup (mrl);
    }
  }
  else
    play_item->title    = strdup (title);

  play_item->mrl        = strdup (mrl);
  play_item->start_time = start_time;
  play_item->options    = NULL;
  play_item->played     = 0;

  return play_item;
}

play_item_t *play_item_copy (play_item_t *src) {

  play_item_t *item;
  GList       *option;

  item = play_item_new (src->title, src->mrl, src->start_time);

  option = src->options;

  while (option) {

    item->options = g_list_append (item->options, strdup (option->data));

    option = g_list_next (option);
  }

  item->untitled = src->untitled;

  return item;
}

void play_item_dispose (play_item_t *play_item) {
  GList *option;

  free (play_item->mrl);
  free (play_item->title);

  option = play_item->options;

  while (option) {

    free (option->data);

    option = g_list_next (option);
  }

  free (play_item);
}

void play_item_add_option (play_item_t *item, const char *option) {

  item->options = g_list_append (item->options, strdup (option));

}

void play_item_save (play_item_t *item, FILE *fh, int depth)
{
  char *title = xml_escape_string (item->title, XML_ESCAPE_NO_QUOTE);
  char *mrl = xml_escape_string (item->mrl, XML_ESCAPE_DOUBLE_QUOTE);
  char *indent = malloc (depth * 2 + 1);

  memset (indent, ' ', depth * 2);
  indent[depth * 2] = 0;

  fprintf (fh,  "%s<ENTRY>\n"
		"%s  <TITLE%s>%s</TITLE>\n"
		"%s  <REF HREF=\"%s\"/>\n"
		"%s  <TIME start=\"%d\"/>\n"
		"%s</ENTRY>\n",
		indent,
		indent, item->untitled ? " default=\"y\"" : "", title,
		indent, mrl,
		indent, item->start_time,
		indent);
  free (title);
  free (mrl);
  free (indent);
}

play_item_t *play_item_load (xml_node_t *node)
{
  play_item_t *play_item = malloc (sizeof (play_item_t));

  memset (play_item, 0, sizeof (play_item_t));
  play_item->title      = NULL;
  play_item->options    = NULL;
  play_item->mrl        = NULL;

  while (node) {

    if (!strcasecmp (node->name, "title")) {
      play_item->untitled = xml_parser_get_property (node, "default") != NULL;
      play_item->title = strdup (node->data);
      logprintf ("play_item: title = %s\n", play_item->title);
    } else if (!strcasecmp (node->name, "ref")) {
      play_item->mrl = strdup(xml_parser_get_property (node, "href"));
      logprintf ("play_item: mrl   = %s\n", play_item->mrl);
    } else if (!strcasecmp (node->name, "time")) {
      play_item->start_time = xml_parser_get_property_int (node, "start", 0);
      logprintf ("play_item: start = %d\n", play_item->start_time);
    } else 
      printf (_("play_item: error while loading, unknown node %s\n"), node->name);

    node = node->next;
  }

  return play_item;
}

void play_item_play (play_item_t *play_item) {

  GList       *option;

  /*
   * execute any optional commands which may be associated with this play item
   * (e.g. set volume etc.)
   */

  option = play_item->options;

  while (option) {

    char *cmd = (char *) option->data;

    engine_exec (cmd, NULL, NULL);

    option = g_list_next (option);
  }

  player_launch (play_item->title, play_item->mrl, 0, play_item->start_time);
}

/* gui part */

static GtkWidget *dlg;
static GtkWidget *entry, *mrl_entry, *time_entry;
static int        is_visible;

static play_item_t *edit_item = NULL;
static const char *item_title;
static int edited = 0;

static void play_item_set_text (void)
{
  char time_str[256];
  gtk_entry_set_text (GTK_ENTRY (mrl_entry), edit_item->mrl);
  gtk_entry_set_text (GTK_ENTRY (entry), edit_item->title);
  int_to_timestring ((edit_item->start_time + 500) / 1000,
		     time_str, sizeof (time_str));
  gtk_entry_set_text (GTK_ENTRY (time_entry), time_str);
}

static gboolean close_cb (GtkWidget* widget, gpointer data) {
  gtk_widget_hide (dlg);
  is_visible = 0;

  gtk_main_quit();

  return TRUE;
}

static void response_cb (GtkDialog *dbox, int response, gpointer data)
{
  const char *tmp;
  switch (response)
  {
  case GTK_RESPONSE_REJECT:
    play_item_set_text ();
    break;
  case 1: /* title */
    if (item_title)
      gtk_entry_set_text (GTK_ENTRY (entry), item_title);
    break;
  case GTK_RESPONSE_OK:
    tmp = gtk_entry_get_text (GTK_ENTRY (entry));
    edit_item->untitled &= !strcmp (tmp, edit_item->title);
    free (edit_item->mrl);
    free (edit_item->title);
    edit_item->mrl   = strdup(gtk_entry_get_text (GTK_ENTRY (mrl_entry)));
    edit_item->title = strdup(tmp);
    edit_item->start_time = parse_timestring (gtk_entry_get_text (GTK_ENTRY (time_entry)))*1000;
    edited = 1;
  default:
    is_visible = 0;
    gtk_widget_hide (dlg);
    gtk_main_quit();
  }
}

void play_item_init (void) {

  GtkWidget *table;

  dlg = gtk_dialog_new_with_buttons ("", NULL, 0,
				_("_Title from stream"), 1,
				GTK_STOCK_UNDO, GTK_RESPONSE_REJECT,
				GTK_STOCK_CANCEL, GTK_RESPONSE_DELETE_EVENT,
				GTK_STOCK_OK, GTK_RESPONSE_OK,
				NULL);
  gtk_dialog_set_default_response (GTK_DIALOG(dlg), GTK_RESPONSE_OK);
  gtk_window_set_default_size (GTK_WINDOW (dlg), 500, 150);
  gtk_signal_connect( GTK_OBJECT (dlg), "delete_event",
		      GTK_SIGNAL_FUNC (close_cb), NULL );
  g_signal_connect (GTK_OBJECT(dlg), "response",
		    GTK_SIGNAL_FUNC(response_cb), NULL);

  table = gtk_table_new (3,2,FALSE);

  add_table_row_items (table, 0, "MRL", 1, mrl_entry = gtk_entry_new ());
  add_table_row_items (table, 1, _("Name"), 1, entry = gtk_entry_new ());
  add_table_row_items (table, 2, _("Time"), 1, time_entry = gtk_entry_new ());

  gtk_widget_grab_focus (entry);

  gtk_box_pack_start_defaults (GTK_BOX (GTK_DIALOG (dlg)->vbox), table);

  is_visible = 0 ;

}

int play_item_edit (play_item_t *item, play_item_type_t type,
		    const char *newtitle)
{
  static const char *const title[] = {
    N_("Edit play item"),
    N_("Edit media mark"),
    N_("Add media mark"),
  };
  gboolean   was_modal;

  edit_item = item;
  item_title = newtitle;
  gtk_dialog_set_response_sensitive (GTK_DIALOG(dlg), 1, newtitle != NULL);
  play_item_set_text ();

  gtk_window_set_title (GTK_WINDOW(dlg), gettext (title[type]));
  gtk_widget_show_all (dlg);

  /* run as a modal dialog */

  was_modal = GTK_WINDOW(dlg)->modal;
  if (!was_modal) 
    gtk_window_set_modal(GTK_WINDOW(dlg),TRUE);

  gtk_main();

  return edited;
}

