/*
 * Copyright (C) 2004-2005 the xine-project
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * $Id: post.c,v 1.20.2.1 2005/03/12 13:17:09 dsalt Exp $
 *
 * post-plugin handling
 */

#include "globals.h"

#include <xine.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <math.h>
#include <pango/pango-context.h>

#include "post.h"
#include "gtkvideo.h"
#include "utils.h"

typedef enum {
  POST_CHAIN_VIDEO,
  POST_CHAIN_AUDIO,
  POST_CHAIN_LAST,
} post_chain_t;

typedef struct {
  const post_chain_t type;
  GtkWidget *dbox;
  GSList *plugins; /* list of gxine_post_t */
  const char **names;
} gxine_chain_t;

struct gxine_post_s {
  gxine_chain_t *parent;
  char *name, *parameters;	/* name, text config */
  xine_post_t *plugin;
  xine_post_in_t *info;
  GtkWidget *box, *enabled, *pref, *up;	/* in parent dbox */
  GtkWidget *dbox, **conf_w;	/* this plugin's dbox */
  char *params;			/* raw config */
  void (*reconfigure)(struct gxine_post_s *);
};
typedef struct gxine_post_s gxine_post_t;

static gxine_chain_t chains[] = { { POST_CHAIN_VIDEO }, { POST_CHAIN_AUDIO } };

static const char *cfgname[] = {
  "gui.post_plugins.video",
  "gui.post_plugins.audio"
};

static const void *const empty[] = { NULL };

static const char *const chain_title[] = {
  N_("Configure video post-plugins"),
  N_("Configure audio post-plugins")
};

static GtkTooltips *tips = NULL;

static char *
post_parse_parameters (xine_post_t *plugin, const char *paramstr)
{
  const xine_post_in_t *in = xine_post_input (plugin, "parameters");
  const xine_post_api_t *api;
  const xine_post_api_descr_t *param_desc;

  if (!in)
    return NULL;

  api = in->data;
  param_desc = api->get_param_descr ();

  char *params = malloc (param_desc->struct_size);
  api->get_parameters (plugin, params);

  for (;;)
  {
    const char *arg;
    const char *next = strchr (paramstr, ',');

    if (!next)
      next = paramstr + strlen (paramstr);

    while (isspace (*paramstr))
      ++paramstr;
    if (!*paramstr)
      break;

    arg = strchr (paramstr, '=');
    if (arg && ++arg < next)
    {
      const xine_post_api_parameter_t *param = param_desc->parameter - 1;
      while ((++param)->type != POST_PARAM_TYPE_LAST)
      {
	if (strncasecmp (paramstr, param->name, arg - paramstr - 1) ||
	    param->name[arg - paramstr - 1] || param->readonly)
	  continue;

	switch (param->type)
	{
	case POST_PARAM_TYPE_INT:
	  if (param->enum_values)
	  {
	    const char *const *values = param->enum_values;
	    int i = -1;
	    while (values[++i])
	    {
	      if (strncasecmp (values[i], arg, next - arg) ||
		  values[i][next - arg])
		continue;
	      *(int *) (params + param->offset) = i;
	      break;
	    }
	    if (!values[i])
	      *(int *) (params + param->offset) = atoi (arg);
	  }
	  else
	    *(int *) (params + param->offset) = atoi (arg);
	  logprintf ("post: setting %s to %d\n", param->name,
		     *(int *) (params + param->offset));
	  goto next_param;

	case POST_PARAM_TYPE_DOUBLE:
	  sscanf (arg, "%lf", (double *) (params + param->offset));
	  logprintf ("post: setting %s to %lf\n", param->name,
		     *(double *) (params + param->offset));
	  goto next_param;

	case POST_PARAM_TYPE_CHAR:
	  memset (params + param->offset, 0, param->size);
	  memcpy (params + param->offset, arg,
		  (next - arg) < param->size ? (next - arg) : param->size);
	  logprintf ("post: setting %s to \"%.*s\"\n", param->name,
		     param->size, params + param->offset);
	  goto next_param;

	case POST_PARAM_TYPE_STRING:
	  fprintf (stderr,
		   "gxine: post-parameter type %s not yet supported\n",
		   "STRING");
	  goto next_param;

	case POST_PARAM_TYPE_STRINGLIST:
	  fprintf (stderr,
		   "gxine: post-parameter type %s not yet supported\n",
		   "STRINGLIST");
	  goto next_param;

	case POST_PARAM_TYPE_BOOL:
	  *(int *) (params + param->offset) = atoi (arg) ? 1 : 0;
	  logprintf ("post: setting %s to %s\n", param->name,
		     *(int *) (params + param->offset) ? "true" : "false");
	  goto next_param;

	default:
	  fprintf (stderr,
		   "gxine: post-parameter type %s not yet supported\n",
		   "???");
	  goto next_param;
	}
      }
      next_param: /* jump here after recognising and processing a parameter */;
    }

    if (!*next)
      break;
    paramstr = next + 1;
  }

  return params;
}

#if 0
static char *
post_validate_parameters (xine_post_t *plugin, const char *paramstr)
{
  const xine_post_in_t *in = xine_post_input (plugin, "parameters");
  const xine_post_api_t *api = in->data;
  const xine_post_api_descr_t *param_desc = api->get_param_descr ();

  for (;;)
  {
    const char *arg;
    const char *next = strchr (paramstr, ',');

    if (!next)
      next = paramstr + strlen (paramstr);

    while (isspace (*paramstr))
      ++paramstr;
    if (!*paramstr)
      break;

    arg = strchr (paramstr, '=');
    if (arg && ++arg < next)
    {
      const xine_post_api_parameter_t *param = param_desc->parameter - 1;
      while ((++param)->type != POST_PARAM_TYPE_LAST)
      {
	if (!strncasecmp (paramstr, param->name, arg - paramstr - 1) &&
	    !param->name[arg - paramstr - 1])
	{
	  if (!param->readonly)
	    break; /* recognised & writeable */
	  return g_strdup_printf (_("parameter %s is read-only"), param->name);
	}
      }
      if (param->type == POST_PARAM_TYPE_LAST)
	return g_strdup_printf (_("parameter %.*s not recognised"),
				arg - paramstr - 1, paramstr);
    }

    if (!*next)
      break;
    paramstr = next + 1;
  }

  return NULL;
}
#endif

void
post_parse_set_parameters (xine_post_t *plugin, const char *paramstr)
{
  const xine_post_in_t *in = xine_post_input (plugin, "parameters");
  char *params = post_parse_parameters (plugin, paramstr);
  if (params)
  {
    ((xine_post_api_t *)in->data)->set_parameters (plugin, params);
    free (params);
  }
}

/* plugin config window */

static char *
post_make_parameter_string (const gxine_post_t *info)
{
  const xine_post_api_parameter_t *param =
    ((xine_post_api_t *)info->info->data)->get_param_descr ()->parameter - 1;
  char *new_paramstr = strdup ("");

  while ((++param)->type != POST_PARAM_TYPE_LAST)
  {
    switch (param->type)
    {
    case POST_PARAM_TYPE_INT:
      if (param->enum_values)
        asreprintf
	  (&new_paramstr, "%s,%s=%s", new_paramstr, param->name,
	   param->enum_values[*(int *) (info->params + param->offset)]);
      else
	asreprintf (&new_paramstr, "%s,%s=%d", new_paramstr, param->name,
		    *(int *) (info->params + param->offset));
      break;

    case POST_PARAM_TYPE_DOUBLE:
      asreprintf (&new_paramstr, "%s,%s=%lf", new_paramstr, param->name,
		  *(double *) (info->params + param->offset));
      break;

    case POST_PARAM_TYPE_CHAR:
      asreprintf (&new_paramstr, "%s,%s=%.*s", new_paramstr, param->name,
		  param->size, info->params + param->offset);
      break;

    case POST_PARAM_TYPE_BOOL:
      asreprintf (&new_paramstr, "%s,%s=%d", new_paramstr, param->name,
		  *(int *) (info->params + param->offset));
      break;

    default:
      continue;
    }
  }
  /* remove the leading comma */
  memmove (new_paramstr, new_paramstr + 1, strlen (new_paramstr));

  return new_paramstr;
}

static void
post_config_post_revert (gxine_post_t *info)
{
  const xine_post_api_t *api = info->info->data;
  const xine_post_api_descr_t *param_desc = api->get_param_descr ();
  const xine_post_api_parameter_t *param = param_desc->parameter;
  char *params;
  int i;

  params = post_parse_parameters (info->plugin, info->parameters);
  memcpy (info->params, params, param_desc->struct_size);
  free (params);

  for (i = 0; info->conf_w[i]; ++i)
  {
    gpointer param_p = info->params + param[i].offset;
    switch (param[i].type)
    {
    case POST_PARAM_TYPE_INT:
      if (param[i].enum_values)
	gtk_option_menu_set_history (GTK_OPTION_MENU(info->conf_w[i]),
				     *(int *)param_p);
      else
	gtk_spin_button_set_value (GTK_SPIN_BUTTON(info->conf_w[i]),
				   *(int *)param_p);
      break;

    case POST_PARAM_TYPE_DOUBLE:
      gtk_range_set_value (GTK_RANGE(info->conf_w[i]),*(double *)param_p);
      break;

    case POST_PARAM_TYPE_CHAR:
      {
	char *contents = alloca (param->size + 1);
	sprintf (contents, "%.*s", param->size, (char *)param_p);
	gtk_entry_set_text (GTK_ENTRY(info->conf_w[i]), contents);
      }
      break;

    case POST_PARAM_TYPE_BOOL:
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(info->conf_w[i]),
				    *(char *)param_p);
      break;

    default:;
    }
  }
}

static void
post_config_post_dispose_dbox (gxine_post_t *info)
{
  if (info->dbox)
    gtk_widget_destroy (info->dbox);
  free (info->params);
  free (info->conf_w);
  info->dbox = NULL;
  info->params = NULL;
  info->conf_w = NULL;
}

static void
post_config_post_response (GtkDialog *dbox, gint response, gxine_post_t *info)
{
  switch (response)
  {
  case GTK_RESPONSE_REJECT: /* undo */
    post_config_post_revert (info);
    break;
  case GTK_RESPONSE_APPLY:
    info->reconfigure (info);
    break;
  case GTK_RESPONSE_OK:
    info->reconfigure (info);
    /* fall through */
  default:
    post_config_post_dispose_dbox (info);
    if (info->pref)
      gtk_widget_set_sensitive (info->pref, TRUE);
    break;
  }
}

static void
enum_cb (GtkOptionMenu *widget, int *param)
{
  *param = gtk_option_menu_get_history (widget);
}

static void
spin_cb (GtkSpinButton *widget, int *param)
{
  *param = gtk_spin_button_get_value_as_int (widget);
}

static void
range_cb (GtkAdjustment *widget, double *param)
{
  *param = widget->value;
}

static void
help_realise_cb (GtkWidget *help, gpointer data)
{
  int i;
  for (i = 0; i < 5; ++i)
  {
    gtk_widget_modify_text (help, i, help->style->fg + i);
    gtk_widget_modify_base (help, i, help->style->bg + i);
  }
}

static void
text_insert_cb (GtkEditable *widget, const gchar *text, gint length,
		gint *position, gpointer data)
{
  int i, j;
  gchar *newtext = malloc (length);
  /* we can't allow ';' as it has a special meaning */
  for (i = j = 0; i < length; ++i)
    if (text[i] >= ' ' && text[i] != ';')
      newtext[j++] = text[i];
  g_signal_handlers_block_by_func (widget, (gpointer)text_insert_cb, data);
  gtk_editable_insert_text (widget, newtext, j, position);
  g_signal_handlers_unblock_by_func (widget, (gpointer)text_insert_cb, data);
  g_signal_stop_emission_by_name (widget, "insert-text");
  free (newtext);
}

static gboolean
char_entry_cb (GtkEditable *widget, GdkEventFocus *event,
	       const xine_post_api_parameter_t *param)
{
  const gxine_post_t *info = g_object_get_data (G_OBJECT(widget), "post");
  memset (info->params + param->offset, 0, param->size);
  snprintf (info->params + param->offset, param->size,
	    gtk_editable_get_chars (widget, 0, -1));
  return FALSE;
}

static void
check_box_cb (GtkToggleButton *widget, int *param)
{
  *param = gtk_toggle_button_get_active (widget);
}

static void
post_config_post_reconfigure (gxine_post_t *info)
{
  char *params = post_make_parameter_string (info);
  logprintf ("post_config: plugin configuration\n  was: %s\n  now: %s\n",
	     info->parameters, params);
  free (info->parameters);
  info->parameters = params;
}

static JSBool
post_config_post_window (GtkButton *widget, gxine_post_t *info)
{
  const xine_post_api_t *api;
  const xine_post_api_descr_t *param_desc;
  const xine_post_api_parameter_t *param;
  char *title;
  GtkWidget *table;
  int param_count = 0;

  if (info->dbox)
  {
    gtk_window_present (GTK_WINDOW(info->dbox));
    return JS_FALSE;
  }

  info->info = xine_post_input (info->plugin, "parameters");
  api = info->info->data;
  param_desc = api->get_param_descr ();

  title = g_strdup_printf (_("Configure plugin %s"), info->name);
  info->dbox =
    gtk_dialog_new_with_buttons (title, NULL, 0,
				 GTK_STOCK_UNDO, GTK_RESPONSE_REJECT,
				 GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				 GTK_STOCK_APPLY, GTK_RESPONSE_APPLY,
				 GTK_STOCK_OK, GTK_RESPONSE_OK,
				 NULL);
  g_signal_connect (G_OBJECT(info->dbox), "response",
		    G_CALLBACK(post_config_post_response), info);
  free (title);
  table = gtk_table_new (1, 2, FALSE);
  gtk_box_pack_start (GTK_BOX(GTK_DIALOG(info->dbox)->vbox), table,
		      FALSE, TRUE, 0);

  info->params = post_parse_parameters (info->plugin, info->parameters);
  param = param_desc->parameter - 1;

  while ((++param)->type != POST_PARAM_TYPE_LAST)
  {
    GtkWidget *widget;
    info->conf_w = realloc (info->conf_w,
			    (param_count + 2) * sizeof (GtkWidget *));
    gpointer param_p = info->params + param->offset;
    switch (param->type)
    {
    case POST_PARAM_TYPE_INT:
      if (param->enum_values)
      {
	GtkWidget *items = gtk_menu_new ();
	char *v;
	int i = -1;
	while ((v = param->enum_values[++i]))
	  gtk_menu_shell_append (GTK_MENU_SHELL(items),
				 gtk_menu_item_new_with_label (v));
	widget = gtk_option_menu_new ();
	gtk_option_menu_set_menu (GTK_OPTION_MENU(widget), items);
	gtk_option_menu_set_history (GTK_OPTION_MENU(widget), *(int *)param_p);
	g_signal_connect (GTK_OBJECT(widget), "changed",
			  G_CALLBACK(enum_cb), param_p);
      }
      else
      {
	GtkObject *adj =
	  gtk_adjustment_new (*(int *)param_p, INT_MIN, INT_MAX, 1, 10, 0);
	widget = gtk_spin_button_new (GTK_ADJUSTMENT(adj), 1, 0);
	gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON(widget),
					   GTK_UPDATE_ALWAYS);
	gtk_spin_button_set_numeric (GTK_SPIN_BUTTON(widget), TRUE);
	g_signal_connect (GTK_OBJECT(widget), "value-changed",
			  G_CALLBACK(spin_cb), param_p);
      }
      break;

    case POST_PARAM_TYPE_DOUBLE:
      {
	GtkObject *adj;
	gdouble range = param->range_max - param->range_min;
	gboolean floating = range <= 10 || range != floor (range) ||
			    param->range_min != floor (param->range_min);
	gdouble step = pow (10, floor (log10 (range / 25)));
	gdouble page = pow (10, floor (log10 (range / 10)));

	adj = gtk_adjustment_new (*(double *)param_p,
				  param->range_min, param->range_max,
				  step, page, 0);
	widget = gtk_hscale_new (GTK_ADJUSTMENT(adj));
	gtk_widget_set_size_request (widget, 100, -1);
	gtk_scale_set_draw_value (GTK_SCALE(widget), TRUE);
	gtk_scale_set_value_pos (GTK_SCALE(widget), GTK_POS_TOP);
	gtk_scale_set_digits (GTK_SCALE(widget), floating ? 2 : 0); /* hmm... */
	g_signal_connect (GTK_OBJECT (adj), "value-changed",
			  G_CALLBACK (range_cb), param_p);
      }
      break;

    case POST_PARAM_TYPE_CHAR:
      {
	char *contents = alloca (param->size + 1);
	widget = gtk_entry_new ();
	sprintf (contents, "%.*s", param->size, (char *)param_p);
	gtk_entry_set_text (GTK_ENTRY(widget), contents);
	g_object_set_data (G_OBJECT(widget), "post", info);
	g_signal_connect (GTK_OBJECT(widget), "insert-text",
			  G_CALLBACK(text_insert_cb), NULL);
	g_signal_connect (GTK_OBJECT(widget), "focus-out-event",
			  G_CALLBACK(char_entry_cb), (gpointer)param);
      }
      break;

    case POST_PARAM_TYPE_BOOL:
      widget = gtk_check_button_new();
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(widget),
				    *(char *)param_p);
      g_signal_connect (GTK_OBJECT(widget), "toggled",
			G_CALLBACK(check_box_cb), param_p);
      break;

    default:
      continue; /* unknown */
    }

    info->conf_w[param_count] = widget;
    gtk_table_attach (GTK_TABLE(table), widget,
		      1, 2, param_count, param_count + 1, 7, 7, 2, 2);

    widget = gtk_label_new (param->name);
    gtk_misc_set_alignment (GTK_MISC (widget), 1, 0.5);
    gtk_table_attach (GTK_TABLE(table), widget,
		      0, 1, param_count, param_count + 1,
		      GTK_FILL, GTK_FILL, 2, 2);

    ++param_count;
  }
  info->conf_w[param_count] = NULL;

  if (api->get_help)
  {
    char *helptext = strdup (api->get_help ());
    size_t p = strlen (helptext);

    /* lose trailing whitespace (and control codes) */
    while (p >= 0 && helptext[p] < 33)
      --p;
    helptext[p + 1] = 0;

    {
      GtkWidget *window = gtk_scrolled_window_new (NULL, NULL);
      GtkTextBuffer *buffer = gtk_text_buffer_new (NULL);
      GtkWidget *help = gtk_text_view_new_with_buffer (buffer);
      PangoFontDescription *font;

      gtk_container_add (GTK_CONTAINER(window), help);
      gtk_text_view_set_editable ((GtkTextView *)help, FALSE);
      gtk_text_view_set_wrap_mode ((GtkTextView *)help, 3);
	/* earlier GTK+2.2 apparently doesn't define GTK_WRAP_WORD_CHAR */

      /* use the normal fg/bg colours, rather than the text display colours */
      g_signal_connect (G_OBJECT(help), "realize",
			G_CALLBACK(help_realise_cb), NULL);

      /* use normal font or, if there's at least one tab, monospaced;
       * either way, reduce the size a little
       */
      font = strchr (helptext, '\t')
             ? pango_font_description_from_string ("mono")
	     : pango_font_description_copy_static (help->style->font_desc);
      p = pango_font_description_get_size (help->style->font_desc);
      pango_font_description_set_size (font, p * 5 / 6);
      gtk_widget_modify_font (help, font);
      pango_font_description_free (font);

      g_object_set ((GObject *)help, "indent", 4, "right-margin", 4, NULL);

      gtk_text_buffer_set_text (buffer, helptext, -1);
      gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(window),
				      GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
      gtk_widget_set_size_request (window, 1, 16);
      gtk_box_pack_end (GTK_BOX(GTK_DIALOG(info->dbox)->vbox), window,
			TRUE, TRUE, 2);

    }
    free (helptext);

    gtk_box_pack_end (GTK_BOX(GTK_DIALOG(info->dbox)->vbox),
		      gtk_hseparator_new (), FALSE, TRUE, 2);
  }

  hide_on_delete (info->dbox, NULL);
  gtk_widget_show_all (info->dbox);

  {
    GdkGeometry geom;
    geom.width_inc = 0;
    geom.height_inc = 1;
    gtk_window_set_geometry_hints (GTK_WINDOW(info->dbox), info->dbox, &geom,
				   GDK_HINT_RESIZE_INC);
  }

  if (info->pref)
    gtk_widget_set_sensitive (info->pref, FALSE);
  return JS_TRUE;
}

static xine_post_t *
post_config_post_init_plugin (post_chain_t chain_type, const char *name)
{
  logprintf ("post_config: creating plugin '%s'\n", name);
  switch (chain_type)
  {
  case POST_CHAIN_VIDEO:
    {
      xine_video_port_t *port = gtk_video_get_port ((GtkVideo *)gtv);
      return xine_post_init (xine, name, 0, NULL, &port);
    }
  case POST_CHAIN_AUDIO:
    return xine_post_init (xine, name, 0, &audio_port, NULL);
  default:
    return NULL; /* shouldn't happen */
  }
}

/* plugin chain config window */

static void
reconfigure_plugins (gxine_chain_t *chain)
{
  char *params = NULL;
  GSList *info = chain->plugins;
  xine_cfg_entry_t entry;

  logprintf ("post_config: configuring plugins\n");

  if (!xine_config_lookup_entry (xine, cfgname[chain->type], &entry))
    return;

  for (; info; info = info->next)
  {
    const gxine_post_t *post = info->data;
    gboolean enabled =
      gtk_toggle_button_get_active ((GtkToggleButton *)post->enabled);
    asreprintf (&params, "%s%s%s%s:%s", params ? : "", params ? ";" : "",
		enabled ? "" : "-", post->name, post->parameters);
  }

  entry.str_value = params ? : "";
  xine_config_update_entry (xine, &entry);

  free (params);
}

static void
post_config_post_chain_set_plugin (GtkOptionMenu *menu, gxine_post_t *info)
{
  gxine_chain_t *chain = info->parent;
  const char *name = chain->names[gtk_option_menu_get_history (menu)];
  xine_post_t *newplugin;

  if (!strcmp (name, info->name))
  {
    logprintf ("post_config: replacing plugin '%s' with itself\n", name);
    return;
  }

  if (info->dbox)
    post_config_post_dispose_dbox (info);

  logprintf ("post_config: replacing old plugin '%s'\n", info->name);
  newplugin = post_config_post_init_plugin (chain->type, name);
  if (!newplugin)
  {
    display_error (gettext (chain_title[chain->type]), NULL,
		   _("Couldn't initialise plugin '%s' for configuration"),
		   name);
    return;
  }

  free (info->name);
  free (info->parameters);
  xine_post_dispose (xine, info->plugin);
  info->name = strdup (name);
  info->parameters = strdup ("");
  info->plugin = newplugin;
  gtk_widget_set_sensitive
    (info->pref, xine_post_input (info->plugin, "parameters") != NULL);
}

static void
post_config_post_chain_delete_plugin (GtkButton *widget, gxine_post_t *info)
{
  logprintf ("post_config: destroying plugin '%s'\n", info->name);
  if (info->dbox)
    g_signal_handlers_disconnect_by_func
      (G_OBJECT(info->dbox), G_CALLBACK(post_config_post_response), info);
  post_config_post_dispose_dbox (info);
  free (info->name);
  free (info->parameters);
  xine_post_dispose (xine, info->plugin);
  gtk_container_remove (GTK_CONTAINER(info->box->parent), info->box);
  info->parent->plugins = g_slist_remove (info->parent->plugins, info);
  free (info);
}

static void
post_config_post_chain_move_plugin (GtkButton *widget, gxine_post_t *info)
{
  gint pos = g_slist_index (info->parent->plugins, info);
  if (--pos < 0)
    return;
  logprintf ("post_config: moving plugin '%s' up\n", info->name);
  g_slist_remove (info->parent->plugins, info); /* this is not the first item */
  info->parent->plugins = g_slist_insert (info->parent->plugins, info, pos);
  gtk_box_reorder_child (GTK_BOX(info->box->parent), info->box, pos);
  gtk_widget_set_sensitive (info->up, pos);
  gtk_widget_set_sensitive
    (((gxine_post_t *)g_slist_nth_data (info->parent->plugins, pos + 1))->up,
     TRUE);
}

extern const char *xine_get_post_plugin_description () __attribute__((weak));

static void
post_config_post_add_plugin (gxine_chain_t *chain, xine_post_t *post,
			     gboolean enabled, char *name, char *params)
{
  gxine_post_t *info = malloc (sizeof (gxine_post_t));
  GtkWidget *w, *m;
  int i, history = 0;

  logprintf ("post_config: adding plugin '%s'\n", name);
  info->parent = chain;
  info->name = name;
  info->parameters = params;
  info->plugin = post;
  info->info = NULL;
  info->dbox = NULL;
  info->conf_w = NULL;
  info->params = NULL;
  info->reconfigure = post_config_post_reconfigure;
  info->box = gtk_hbox_new (FALSE, 2);

  m = gtk_menu_new ();
  for (i = 0; chain->names[i]; ++i)
  {
    const char *desc = xine_get_post_plugin_description
		     ? xine_get_post_plugin_description (xine, chain->names[i])
		     : NULL;
    GtkWidget *item;
    if (!strcmp (chain->names[i], name))
      history = i;
    gtk_menu_shell_append (GTK_MENU_SHELL(m),
			item =   gtk_menu_item_new_with_label (chain->names[i]));
    if (desc)
      gtk_tooltips_set_tip (tips, item, desc, NULL);
  }

  info->enabled = w = gtk_check_button_new ();
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), enabled);
  gtk_box_pack_start (GTK_BOX(info->box), w, FALSE, TRUE, 0);

  w = gtk_option_menu_new ();
  gtk_option_menu_set_menu (GTK_OPTION_MENU(w), m);
  gtk_option_menu_set_history (GTK_OPTION_MENU(w), history);
  gtk_box_pack_start_defaults (GTK_BOX(info->box), w);
  g_signal_connect (G_OBJECT(w), "changed",
		    G_CALLBACK(post_config_post_chain_set_plugin), info);

  info->up = w = gtk_button_new ();
  gtk_container_add (GTK_CONTAINER(w),
		     gtk_image_new_from_stock (GTK_STOCK_GO_UP,
					       GTK_ICON_SIZE_BUTTON));
  gtk_widget_set_sensitive (w, !!chain->plugins); /* disable for first plugin */
  gtk_box_pack_end (GTK_BOX(info->box), w, FALSE, TRUE, 0);
  g_signal_connect (G_OBJECT(w), "clicked",
		    G_CALLBACK(post_config_post_chain_move_plugin), info);

  w = gtk_button_new_from_stock (GTK_STOCK_REMOVE);
  gtk_box_pack_end (GTK_BOX(info->box), w, FALSE, TRUE, 0);
  g_signal_connect (G_OBJECT(w), "clicked",
		    G_CALLBACK(post_config_post_chain_delete_plugin), info);

  info->pref = w = gtk_button_new_from_stock (GTK_STOCK_PROPERTIES);
  gtk_box_pack_end (GTK_BOX(info->box), w, FALSE, TRUE, 0);
  if (!xine_post_input (post, "parameters"))
    gtk_widget_set_sensitive (w, FALSE);
  g_signal_connect (G_OBJECT(w), "clicked",
		    G_CALLBACK(post_config_post_window), info);

  gtk_box_pack_start (GTK_BOX(GTK_DIALOG(chain->dbox)->vbox), info->box,
		      FALSE, TRUE, 2);
  gtk_widget_show_all (info->box);

  chain->plugins = g_slist_append (chain->plugins, info);
}

static void
post_config_post_chain_add (GtkButton *button, gxine_chain_t *chain)
{
  xine_post_t *plugin =
    post_config_post_init_plugin (chain->type, chain->names[0]);
  if (plugin)
    post_config_post_add_plugin (chain, plugin, TRUE, strdup (chain->names[0]),
				 strdup (""));
  else
    display_error (gettext (chain_title[chain->type]), NULL,
		   _("Couldn't initialise plugin '%s' for configuration"),
		   chain->names[0]);
}

static void
post_config_post_chain_build (gxine_chain_t *chain)
{
  xine_cfg_entry_t entry;
  char *config;

  if (!xine_config_lookup_entry (xine, cfgname[chain->type], &entry))
    return;

  config = entry.str_value;
  while (config)
  {
    char *first = strchr (config, ':');
    char *next = strchr (config, ';');
    char *name;
    gboolean enabled;
    xine_post_t *post;

    if (!first)
      break;
    if (!next)
      next = (char *)config + strlen (config);

    enabled = *config != '-';
    if (!enabled)
      ++config;
    name = g_strndup (config, first - config);
    post = post_config_post_init_plugin (chain->type, name);
    if (post)
      post_config_post_add_plugin (chain, post, enabled, name,
				   g_strndup (first + 1, next - first - 1));
    else
      free (name);

    if (!*next)
      break;
    config = next + 1;
  }
}

static void
post_config_post_chain_dispose_dbox (gxine_chain_t *chain)
{
  logprintf ("post_config: done\n");
  while (chain->plugins)
    post_config_post_chain_delete_plugin (NULL, chain->plugins->data);
  if (chain->dbox)
    gtk_widget_destroy (chain->dbox);
  free (chain->names);
  chain->names = NULL;
  chain->dbox = NULL;
}

static void
post_config_post_chain_response (GtkDialog *dbox, gint response,
				 gxine_chain_t *chain)
{
  switch (response)
  {
  case GTK_RESPONSE_REJECT: /* undo */
    while (chain->plugins)
      post_config_post_chain_delete_plugin (NULL, chain->plugins->data);
    post_config_post_chain_build (chain);
    break;
  case GTK_RESPONSE_APPLY:
    reconfigure_plugins (chain);
    break;
  case GTK_RESPONSE_CLOSE:
    reconfigure_plugins (chain);
    /* fall through */
  default:
    post_config_post_chain_dispose_dbox (chain);
    break;
  }
}

static JSBool
post_config_post_chain_window (gxine_chain_t *chain)
{
  static const int post_type[] = {
    XINE_POST_TYPE_VIDEO_FILTER,
    XINE_POST_TYPE_AUDIO_FILTER
  };
  GtkWidget *hbox, *w;
  const char *const *plugin_list;
  int i;

  if (chain->dbox)
  {
    gtk_window_present (GTK_WINDOW(chain->dbox));
    return JS_TRUE;
  }

  plugin_list = xine_list_post_plugins_typed (xine, post_type[chain->type]);
  if (!plugin_list || !plugin_list[0])
  {
    display_error (gettext (chain_title[chain->type]), NULL,
		   _("No available plugins - nothing to configure"));
    return JS_FALSE;
  }

  /* clone the plugin list (pointers only); xine-lib may reuse that memory */
  for (i = 0; plugin_list[i]; ++i) /**/;
  chain->names = malloc (i = (i + 1) * sizeof (const char *));
  memcpy (chain->names, plugin_list, i);

  logprintf ("post_config: start\n");

  chain->dbox = gtk_dialog_new_with_buttons
		  (gettext (chain_title[chain->type]), NULL, 0,
		   GTK_STOCK_UNDO, GTK_RESPONSE_REJECT,
		   GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
		   GTK_STOCK_APPLY, GTK_RESPONSE_APPLY,
		   GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
		   NULL);
  hide_on_delete (chain->dbox, NULL);
  g_signal_connect (G_OBJECT(chain->dbox), "response",
		    G_CALLBACK(post_config_post_chain_response), chain);
  if (!tips)
    tips = gtk_tooltips_new ();

  hbox = gtk_hbox_new (FALSE, 2);
  gtk_box_pack_end (GTK_BOX(GTK_DIALOG(chain->dbox)->vbox), hbox,
		    FALSE, TRUE, 5);

  w = gtk_button_new_from_stock (GTK_STOCK_ADD);
  g_signal_connect (G_OBJECT(w), "clicked",
		    G_CALLBACK(post_config_post_chain_add), chain);
  gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 5);

  post_config_post_chain_build (chain);

  gtk_widget_show_all (chain->dbox);
  return JS_TRUE;
}

static void
deinterlace_reconfigure (gxine_post_t *info)
{
  xine_cfg_entry_t entry;
  char *cfg, *params = post_make_parameter_string (info);
  logprintf ("post_config: deinterlace configuration\n  was: %s\n  now: %s\n",
	     info->parameters, params);
  free (info->parameters);
  asprintf (&cfg, "tvtime:%s", info->parameters = params);
  gtk_video_set_tvtime (GTK_VIDEO(gtv), cfg);
  if (xine_config_lookup_entry (xine, "gui.post_plugins.deinterlace", &entry))
  {
    entry.str_value = cfg;
    xine_config_update_entry (xine, &entry);
  }
  free (cfg);
}

static JSBool
js_deinterlace_show (JSContext *cx, JSObject *obj, uintN argc,
		     jsval *argv, jsval *rval)
{
  xine_cfg_entry_t entry;
  static gxine_post_t info = {
    NULL, "tvtime", .reconfigure = deinterlace_reconfigure
  };

  se_log_fncall ("deinterlace_show");
  if (info.dbox)
    return JS_FALSE;

  if (!xine_config_lookup_entry (xine, "gui.post_plugins.deinterlace", &entry))
    return JS_FALSE;

  info.name = _("deinterlace (tvtime)");
  info.plugin = gtk_video_get_deinterlacer (GTK_VIDEO(gtv));
  free (info.parameters);
  info.parameters = strdup (entry.str_value +
			    (strncmp (entry.str_value, "tvtime:", 7) ? 0 : 7));

  return post_config_post_window (NULL, &info);
}

static JSBool
js_pp_video_show (JSContext *cx, JSObject *obj, uintN argc,
		  jsval *argv, jsval *rval)
{
  se_log_fncall ("postproc_video_show");
  return post_config_post_chain_window (&chains[POST_CHAIN_VIDEO]);
}

static JSBool
js_pp_audio_show (JSContext *cx, JSObject *obj, uintN argc,
		  jsval *argv, jsval *rval)
{
  se_log_fncall ("postproc_audio_show");
  return post_config_post_chain_window (&chains[POST_CHAIN_AUDIO]);
}

void
post_init (void)
{
  static const se_f_def_t defs[] = {
    { "deinterlace_show", js_deinterlace_show, 0, 0,
      SE_GROUP_DIALOGUE, NULL, NULL },
    { "postproc_video_show", js_pp_video_show, 0, 0,
      SE_GROUP_DIALOGUE, NULL, NULL },
    { "postproc_audio_show", js_pp_audio_show, 0, 0,
      SE_GROUP_DIALOGUE, NULL, NULL },
    { NULL }
  };
  se_defuns (gse, NULL, defs);
}
