/* 
 * Copyright (C) 2003-2005 the xine project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: script_engine.c,v 1.42 2005/01/12 02:45:37 dsalt Exp $
 *
 * gate to the spider monkey javascript interpreter
 *
 * provides functions to generate interpreter instances
 * customized for controlling the xine engine and the gxine
 * frontend. this is done by defining a set of global objects
 * and functions so any xine and gxine function can be invoked from
 * (java)scripts.
 *
 */

#include "globals.h"

#include <string.h>
#include <stdlib.h>
#include <pthread.h>

#include "script_engine.h"
#include "utils.h"
#include "mediamarks.h"
#include "playlist.h"
#include "settings.h"
#include "preferences.h"
#include "key_events.h"
#include "log_window.h"
#include "stream_info.h"
#include "open_mrl.h"
#include "server.h"
#include "wizards.h"
#include "snapshot.h"
#include "engine.h"
/*
#define LOG
*/
static JSRuntime *rt=NULL; /* global */

static GSList *se_chain;

int se_eval (se_t *se, const gchar *script, se_o_t *obj,
	     se_print_cb_t print_cb, void *print_cb_data) {

  JSBool     ok;
  int        l;

  se->print_cb      = print_cb;
  se->print_cb_data = print_cb_data;

  l = strlen (script);

  if (obj) 
    ok = JS_EvaluateScript (se->cx, obj->obj, script, l,
			    "gxine", 0, &se->rval);
  else
    ok = JS_EvaluateScript (se->cx, se->global, script, l,
			    "gxine", 0, &se->rval);

  return ok;
}

gchar *se_result_str (se_t *se) {

  if (!JSVAL_IS_STRING (se->rval))
    return NULL;

  se->str = JS_ValueToString (se->cx, se->rval);
  if (verbosity)
    printf(_("script_engine: script result: %s\n"), JS_GetStringBytes (se->str));
  return JS_GetStringBytes (se->str);
}

int se_result_num (se_t *se, JSInt32 *num) {

  if (!JSVAL_IS_INT (se->rval))
    return 0;

  return JS_ValueToInt32 (se->cx, se->rval, num);
}

/* 
 * methods
 */

static JSBool controls_exit (JSContext *cx, JSObject *obj, uintN argc, 
			     jsval *argv, jsval *rval) {

  gchar *fname;

  se_log_fncall ("controls.exit");

#if 0
  stop_server();
  printf ("script_engine: server stopped\n");
#endif

  if (gtk_main_level() < 1)
  {
    engine_exec ("settings_clear ();", NULL, NULL);
    exit (2);
  }

  mm_save (); 
  playlist_save (NULL);
  save_key_bindings ();

  logprintf ("script_engine: saving config...\n");

#if 0
  {
    xine_cfg_entry_t entry;
    if (xine_config_lookup_entry (xine, "audio.volume.remember_volume", &entry) &&
	entry.num_value &&
	xine_config_lookup_entry (xine, "audio.volume.mixer_volume", &entry))
    {
      entry.num_value = xine_get_param (stream, XINE_PARAM_AUDIO_VOLUME);
      xine_config_update_entry (xine, &entry);
    }
  }
#endif

  fname = g_strconcat(g_get_home_dir(), "/.gxine/config", NULL);
  xine_config_save (xine, fname);
  g_free (fname);

  engine_exec ("settings_clear ();", NULL, NULL);

  gtk_main_quit();

  exit (0);

  return JS_TRUE;
}

static GtkWidget *about = NULL;
static gboolean about_visible = FALSE;

static void about_cb (GtkDialog *d, gint arg, gpointer data)
{
  about_visible = FALSE;
  gtk_widget_hide (about);
}

static JSBool show_about (JSContext *cx, JSObject *obj, uintN argc, 
			  jsval *argv, jsval *rval) {

  if (about == NULL)
  {
    static const gchar content_src[] =
     N_("<big><b>About gxine %s</b></big>\n"
	"\n"
	"A GTK media player frontend for xine\n"
	"\n"
	"© 2002-2004 Guenter Bartsch\n"
	"© 2004-2005 Darren Salt\n"
	"© 2002-2005 the xine project team\n");
    static gchar *content = NULL;

    GtkWidget *layout, *text;
    gchar *file;

    about = gtk_dialog_new_with_buttons (_("About gxine"), NULL, 0,
				 GTK_STOCK_CLOSE, GTK_RESPONSE_DELETE_EVENT,
				 NULL);
    gtk_dialog_set_default_response (GTK_DIALOG(about),
				     GTK_RESPONSE_DELETE_EVENT);
    gtk_dialog_set_has_separator (GTK_DIALOG(about), FALSE);

    layout = gtk_table_new (1, 2, FALSE);

    file = g_strconcat (pixmapdir, "/gxine-logo.png", NULL);
    gtk_table_attach (GTK_TABLE (layout), gtk_image_new_from_file (file),
		      0, 1, 0, 1, 0, 0, 8, 8);
    g_free (file);

    if (!content)
      content = g_strdup_printf (gettext (content_src), VERSION);

    text = gtk_label_new (content);
    gtk_label_set_use_markup (GTK_LABEL(text), TRUE);
    gtk_label_set_line_wrap (GTK_LABEL(text), TRUE);
    gtk_table_attach (GTK_TABLE (layout), text, 1, 2, 0, 1,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 8, 8);

    gtk_box_pack_start (GTK_BOX(GTK_DIALOG(about)->vbox), layout,
			TRUE, TRUE, 0);

    g_signal_connect (GTK_OBJECT(about), "response",
		      G_CALLBACK(about_cb), NULL);
    g_signal_connect (GTK_OBJECT(about), "delete-event",
		      G_CALLBACK(about_cb), NULL);
  }

  if (about_visible) {
    gtk_widget_hide (about);
    about_visible = FALSE;
  } else {
    gtk_widget_show_all (about);
    about_visible = TRUE;
  }

  return JS_TRUE;
}



#if 0
static JSBool dummy (JSContext *cx, JSObject *obj, uintN argc, 
		     jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */

  se_log_fncall ("dummy");

  /*
    argc

    JSString *str;
    str = JS_ValueToString (cx, argv[0]);
    mrl = JS_GetStringBytes (str);

    JS_ValueToInt32 (cx, argv[0], &pos);

    se->print_cb (se->print_cb_data, 
		  "error: \n");

    JSCustomer *p = JS_GetPrivate(cx, obj);
    *rval = INT_TO_JSVAL(5);
  */

  return JS_TRUE;
}
#endif

static char *show_help_int (se_t *se, se_o_t *obj, se_group_t selector)
{
  const GList *objs = obj->functions;
  char *help = NULL;
  while (objs)
  {
    se_f_t *cmd = objs->data;
    if (cmd->group == selector)
    {
      char *prefix = NULL;
      se_o_t *parent = obj;
      while (parent && strcmp (parent->id, "_global_"))
      {
	asreprintf (&prefix, "%s.%s", parent->id, prefix ? : "");
	parent = parent->parent;
      }
      asreprintf (&help, "%s    %s%s (%s)%s%s%s\n",
		  help ? : "", prefix ? prefix : "", cmd->id,
		  cmd->arghelp ? : "", cmd->funchelp ? "  /* " : "",
		  cmd->funchelp ? : "", cmd->funchelp ? " */" : "");
      free (prefix);
    }
    objs = objs->next;
  }

  objs = obj->children;
  while (objs)
  {
    char *group = show_help_int (se, objs->data, selector);
    if (group)
    {
      asreprintf (&help, "%s%s", help ? : "", group);
      free (group);
    }
    objs = objs->next;
  }

  return help;
}

static JSBool show_help (JSContext *cx, JSObject *obj, uintN argc, 
			  jsval *argv, jsval *rval)
{
  static char *const group_id[] = {
    NULL,
    N_("Engine control:"),
    N_("Dialogue boxes:"),
    N_("Playlist control:"),
    N_("File access:"),
    N_("Status:"),
    N_("Properties:"),
    N_("Input event generation (DVD menus etc.):"),
    N_("External program support:"),
    NULL
  };

  int i;
  se_t *se = (se_t *) JS_GetContextPrivate(cx);

  se->print_cb (se->print_cb_data, "\n%s",
		_("Available commands (gxine Javascript interface):"));

  i = 0;
  while (group_id[++i])
  {
    char *help = NULL;
    const GSList *pse = se_chain;

    while (pse)
    {
      char *group = show_help_int (se, ((se_t *)pse->data)->g, i);
      if (group)
      {
	asreprintf (&help, "%s%s  %s\n%s", help ? : "", help ? "\n" : "",
		    gettext (group_id[i]), group);
	free (group);
      }
      pse = pse->next;
    }

    if (!help)
      continue;
    help[strlen (help) - 1] = 0;
    se->print_cb (se->print_cb_data, "%s", help);
    free (help);
  }

  /* set_audio_channel (int);
   * set_spu_channel (int);
   * int get_pos ();
   * int get_time ();
   * int get_length ();
   */
  return JS_TRUE;
}

static jsval get_prop_jsval (se_t *se, se_o_t *o, se_prop_t *p) {
 
  int n;

  logprintf ("script_engine: getting value of %s=='%s'\n", p->id, p->value);

  switch (p->se_type) {
  case SE_TYPE_INT:
 
    if (sscanf (p->value, "%d", &n) == 1) {
      logprintf ("script_engine: this is a simple integer %d\n", n);
      return INT_TO_JSVAL (n);
    } else {
 
      if (!strncasecmp (p->value, "jscript:", 8)) {
 
        char *script;
 
        script = p->value+8;
 
	logprintf ("script_engine: this is a jscript => evaluating '%s'...\n",
		   script);
 
        se_eval (se, script, o, NULL, NULL);
 
        if (se_result_num (se, &n) ) {
          logprintf ("script_engine: result: %d\n", n);
          return INT_TO_JSVAL (n);
        } else
          printf ("script_engine: error, non-numeric result in %s=='%s'\n",
                  p->id, p->value);
      } else {
        printf ("script_engine: error, non-numeric int property %s=='%s'\n",
                p->id, p->value);
      }
    }
    return INT_TO_JSVAL (0);
 
    break;
 
  case SE_TYPE_BOOL:
    if (sscanf (p->value, "%d", &n) == 1) {
      logprintf ("script_engine: this is a simple boolean %d\n", n);
      return BOOLEAN_TO_JSVAL (n);
    } else if (!strcasecmp (p->value, "false")) {
      return BOOLEAN_TO_JSVAL (0);
    } else
      return BOOLEAN_TO_JSVAL (1);
    break;
 
  case SE_TYPE_STRING:
    {
 
      JSString *jstr;
      char     *bytes;
      int       len;

      logprintf ("script_engine: this is a string\n");

      len = strlen (p->value);
      bytes = JS_malloc (se->cx, len);
      memcpy (bytes, p->value, len);
       
      jstr = JS_NewString (se->cx, bytes, len);
       
      return STRING_TO_JSVAL(jstr);
    }
 
    break;
  default:
     
    printf ("script_engine: error, property %s has unknown type %d\n",
            p->id, p->se_type);
  }
 
 
  return BOOLEAN_TO_JSVAL (0);
}
 
  
/*
 * function to create and maintain js objects
 */

JSBool generic_JSGetProperty (JSContext *cx, JSObject *obj, jsval id, jsval *vp) {
 
  se_t   *se = (se_t *) JS_GetContextPrivate(cx);
  se_o_t *o;
 
  se_log_fncall ("generic get property");
 
  o = JS_GetPrivate (cx, obj);
 
  if (JSVAL_IS_STRING (id)) {
    JSString *str;
    char     *prop;
    GList    *n;
 
    str = JS_ValueToString (cx, id);
    prop = JS_GetStringBytes (str);

    logprintf ("script_engine: looking for generic property '%s' in '%s'\n",
	       prop, o->id);

    /* look through properties */
 
    n = g_list_first (o->properties);
    while (n) {
      se_prop_t *p;
       
      p = (se_prop_t *) n->data;
       
      if (!strcasecmp (p->id, prop)) {
        *vp = get_prop_jsval (se, o, p);
        return JS_TRUE;
      }
       
      n = g_list_next (n);
    }
     
    /* maybe an object ? */
 
    n = g_list_first (o->children);
    while (n) {
      se_o_t *c;
       
      c = (se_o_t *) n->data;
       
      if (!strcasecmp (c->id, prop)) {
         
        *vp = OBJECT_TO_JSVAL (c->obj);
        return JS_TRUE;
         
      }
       
      n = g_list_next (n);
    }
 
    if (o->parent)
      return generic_JSGetProperty (cx, o->parent->obj, id, vp);
 
    return JS_TRUE;
  }
 
  *vp = JSVAL_VOID;
 
  return JS_TRUE;
}
 

JSBool generic_JSSetProperty(JSContext *cx, JSObject *obj, jsval id, jsval *vp) {
  se_log_fncall ("generic set property");
  return JS_TRUE;
}

static void generic_JSDestructor (JSContext *cx, JSObject *obj) {
  se_log_fncall ("generic destructor");
}

static JSClass generic_JSClass = 
  {
    "view", JSCLASS_HAS_PRIVATE,
    JS_PropertyStub, JS_PropertyStub,
    generic_JSGetProperty, generic_JSSetProperty,
    JS_EnumerateStub, JS_ResolveStub, 
    JS_ConvertStub, generic_JSDestructor
  };

se_o_t *se_create_object (se_t *se, se_o_t *parent /* may be NULL */,
			  const gchar *name,
			  void *user_data) {

  se_o_t *o;

#ifdef LOG
  { 
    se_o_t *p;
    printf ("script_engine: creating object %s parent:", name);

    p = parent;
    while (p) {
      printf (".%s", p->id);

      p = p->parent; 
    }
    printf ("\n");
  }
#endif  
  

  o = (se_o_t *) malloc (sizeof (se_o_t));

  if (!parent)
    o->obj = JS_DefineObject (se->cx, se->global, name,
			      &generic_JSClass, NULL, 0);
  else
    o->obj = JS_DefineObject (se->cx, parent->obj, name,
			      &generic_JSClass, NULL, 0);

  o->user_data   = user_data;
  o->parent      = parent;
  o->id          = strdup (name);
  o->children    = NULL;
  o->properties  = NULL;
  o->functions   = NULL;

  JS_SetPrivate (se->cx, o->obj, o);

  if (parent)
    parent->children = g_list_append (parent->children, o);
  else
    se->g->children = g_list_append (se->g->children, o);

  return o;
}

static se_prop_t *se_prop_find (se_t *se, se_o_t *o, const char *id) {
 
  GList *node;
 
  if (!o)
    o = se->g;
 
  node = o->properties;
  while (node) {
    se_prop_t *p;
 
    p = node->data;
 
    if (!strcasecmp (p->id, id))
      return p;
 
    node = g_list_next (node);
  }
 
  return NULL;
}
 
char *se_prop_get (se_t *se, se_o_t *o, const char *id) {
 
  se_prop_t *p;
 
  if (!o)
    o = se->g;
 
  p = se_prop_find (se, o, id);
 
  if (!p) {
    printf ("script_engine: se_prop_get error, property %s.%s doesn't exist\n",
            o->id, id);
    return NULL;
  }

  return p->value;
}

int se_prop_get_int (se_t *se, se_o_t *o, const char *id) {

  se_prop_t *p;
  int        n;

  if (!o)
    o = se->g;

  p = se_prop_find (se, o, id);
  if (!p) {
    printf ("script_engine: se_prop_get_int error, property %s.%s doesn't exist\n",
            o->id, id);
    return 0;
  }

  logprintf ("script_engine: getting value of %s.%s=='%s'\n",
	     o->id, p->id, p->value);
 
  switch (p->se_type) {
  case SE_TYPE_INT:
 
    if (sscanf (p->value, "%d", &n) == 1) {
      logprintf ("script_engine: this is a simple integer %d\n", n);
      return n;
       
    } else {
       
      if (!strncasecmp (p->value, "jscript:", 8)) {

        char *script = p->value+8;
	logprintf ("script_engine: this is a jscript => evaluating '%s'...\n",
		   script);
        se_eval (se, script, o, NULL, NULL);
        if (se_result_num (se, &n) ) {
          logprintf ("script_engine: result: %d\n", n);
          return n;
        } else
          printf ("script_engine: error, non-numeric result for %s.%s\n",
                  o->id, id);
      } else {
        printf ("script_engine: error, non-numeric property %s.%s='%s'\n",
                o->id, id, p->value);
      }
    }
    break;
  case SE_TYPE_BOOL:
    if (sscanf (p->value, "%d", &n) == 1) {
      logprintf ("script_engine: this is a simple integer %d\n", n);
      return n;
    } else if (!strcasecmp (p->value, "false")) {
      return 0;
    } else {
      return 1;
    }
    break;
 
  default:
    printf ("script_engine: error in se_get_prop_int, %s.%s is not an integer or bool\n",            o->id, id);
  }
  return 0;
}

void se_prop_create (se_t *se, se_o_t *o,
                     const char *id, const char *value,
                     int se_type) {
  se_prop_t *p;
 
  if (!o)
    o = se->g;

  logprintf ("script_engine: setting %s.%s to '%s'\n", o->id, id, value);

  p = se_prop_find (se, o, id);
  if (!p) {
    logprintf ("script_engine: creating new property %s.%s\n", o->id, id);
    p = malloc (sizeof (se_prop_t));
    p->id = strdup (id);
    p->value = strdup (value);
    p->listeners = NULL;
    o->properties = g_list_append (o->properties, p);
  }

  p->se_type = se_type;
}

void se_prop_create_int (se_t *se, se_o_t *o,
                         const char *id, int value) {
  char str[24];
  snprintf (str, sizeof (str), "%d", value);
  se_prop_create (se, o, id, str, SE_TYPE_INT);
}

void se_prop_create_bool (se_t *se, se_o_t *o,
                          const char *id, int value) {
  char str[24];
  snprintf (str, sizeof (str), "%d", value);
  se_prop_create (se, o, id, str, SE_TYPE_BOOL);
}


static void error_reporter (JSContext *cx, const char *message,
			    JSErrorReport *report) {

  se_t *se = (se_t *) JS_GetContextPrivate(cx);

  logprintf ("script_engine: JSError '%s'\n", message);

  if (se->print_cb) 
    se->print_cb (se->print_cb_data, "%s", (char *) message);

}

se_f_t *se_defun (se_t *se, se_o_t *o /* may be NULL */,
                  const char *name, JSNative fun, uintN nargs, uintN attrs,
                  se_group_t group, const char *arghelp, const char *funchelp)
{
  se_f_t *f;
  f = malloc (sizeof(se_f_t));
  f->id = strdup (name);

  if (o) {
    f->fun = JS_DefineFunction (se->cx, o->obj, name, fun, nargs, attrs);
    o->functions = g_list_append (o->functions, f);
  } else {
    f->fun = JS_DefineFunction (se->cx, se->global, name, fun, nargs, attrs);
    se->g->functions = g_list_append (se->g->functions, f);
  }

  f->group = group;
  f->arghelp = arghelp ? gettext (arghelp) : NULL;
  f->funchelp = funchelp ? gettext (funchelp) : NULL;

  return f;
}

void se_defuns (se_t *se, se_o_t *o /* may be NULL */, const se_f_def_t defs[])
{
  int i = -1;
  while (defs[++i].name)
    se_defun (se, o, defs[i].name, defs[i].func, defs[i].nargs, defs[i].attrs,
	      defs[i].group, defs[i].arghelp, defs[i].funchelp);
}

se_t *se_new (void) {

  se_t    *se;
  static JSClass global_class = {
    "global",0,
    JS_PropertyStub,JS_PropertyStub,JS_PropertyStub,JS_PropertyStub,
    JS_EnumerateStub,JS_ResolveStub,JS_ConvertStub,JS_FinalizeStub
  };

  /*
   * create a js engine instance 
   */

  if (!rt) 
    rt = JS_NewRuntime(0x100000);

  se = malloc (sizeof (se_t));

  se->cx       = JS_NewContext (rt, 0x1000);
  se->global   = JS_NewObject (se->cx, &global_class, NULL, NULL);
  JS_InitStandardClasses (se->cx, se->global);
  JS_SetErrorReporter (se->cx, error_reporter);
  JS_SetContextPrivate (se->cx, se);

  /*
   * create fake global se_o
   */
 
  se->g = malloc (sizeof(se_o_t));
  se->g->obj        = se->global;
  se->g->parent     = NULL;
  se->g->id         = strdup("_global_");
  se->g->children   = NULL;
  se->g->functions  = NULL;
  se->g->properties = NULL;

  se_chain = g_slist_append (se_chain, se);

  {
    static const se_f_def_t defs[] = {
      { "exit", controls_exit, 0, 0, SE_GROUP_ENGINE, NULL, NULL },
      { "about_show", show_about, 0, 0, SE_GROUP_DIALOGUE, NULL, NULL },
      { "help", show_help, 0, 0, SE_GROUP_HIDDEN, NULL, NULL },
      { NULL }
    };
    se_defuns (se, se->g, defs);
  }

  return se;
}
