/*
 * Copyright (C) 2003-2004 the xine-project
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * $Id: engine.c,v 1.24 2005/11/10 18:55:29 dsalt Exp $
 *
 * init xine engine, set up script engine
 */

#include "globals.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pthread.h>
#include <stdarg.h>
#include <errno.h>

#include "engine.h"
#include "ui.h"
#include "utils.h"

se_t *gse; /* global script engine */
xine_t *xine; /* global xine instance */
pthread_mutex_t engine_lock;

gboolean initialised = FALSE;

static GtkWidget *se_startup_window, *se_startup_textarea;
static GtkTextBuffer *se_startup_buf;
static GtkTextTag *se_startup_tag;
static gint se_startup_insert;

static char *se_startup_cmds = NULL;
static gboolean modified = FALSE;


static void print_cb (void *user_data, const char *str, ...)
{
  if (verbosity)
  {
    va_list ap;
    printf (_("engine: < "));
    va_start (ap, str);
    vprintf (str, ap);
    va_end (ap);
  }
}

int engine_exec_ext (const char *cmd, se_print_cb_t cb, void *cb_data,
		     se_error_cb_t ecb, const char *src)
{
  struct {
    JSInt32 i;
    JSFloat64 d;
    JSBool b;
  } num; 
  char *str;

  if (verbosity)
    printf (_("engine: > executing '%s'...\n"), cmd);

  if (!cb) {
    cb = print_cb;
    cb_data = NULL;
  }

  se_eval_ext (gse, cmd, NULL, cb, cb_data, ecb, src);

  if ( (str = se_result_str (gse)) ) {
    cb (cb_data, _("result: "));
    cb (cb_data, "%s", str);
    return 1;
  }

  if (se_result_double (gse, &num.d))
  {
    char str[32];
    snprintf (str, sizeof (str), _("result: %lf"), num.d);
    cb (cb_data, "%s", str);
    return 1;
  }

  if (se_result_int (gse, &num.i))
  {
    char str[32];
    snprintf (str, sizeof (str), _("result: %d"), num.i);
    cb (cb_data, "%s", str);
    return 1;
  }

  if (se_result_bool (gse, &num.b))
  {
    char str[32];
    snprintf (str, sizeof (str), _("result: %s"), 
	      num.b == JS_TRUE ? _("true") : _("false"));
    cb (cb_data, "%s", str);
    return 1;
  }

  return 0;
}

int engine_exec (const char *cmd, se_print_cb_t cb, void *cb_data,
		 const char *src)
{
  return engine_exec_ext (cmd, cb, cb_data, se_error_cb, src);
}

int v_engine_exec(const char *command, se_print_cb_t cb, void *cb_data,
		  const char *src, ...)
{
  char *buf;
  va_list va;
  int ret;

  va_start (va, src);
  buf = g_strdup_vprintf (command, va);
  va_end (va);
  ret = engine_exec (buf, cb, cb_data, src);
  free (buf);
  return ret;
}

int v_engine_exec_ext (const char *command, se_print_cb_t cb, void *cb_data,
		       se_error_cb_t ecb, const char *src, ...)
{
  char *buf;
  va_list va;
  int ret;

  va_start (va, src);
  buf = g_strdup_vprintf (command, va);
  va_end (va);
  ret = engine_exec_ext (buf, cb, cb_data, ecb, src);
  free (buf);
  return ret;
}

static char *
load_user_startup_script (void)
{
  char *file = g_strconcat (g_get_home_dir (), "/.gxine/startup", NULL);
  char *text = read_entire_file (file, NULL);
  if (!text)
  {
    text = strdup ("");
    if (errno != ENOENT)
      display_error (FROM_GXINE, _("Couldn't load startup script"), "%s: %s",
		     file, strerror (errno));
  }
  free (file);

  g_strstrip (text); /* strips \n added by read_entire_file() */
  return strcat (text, "\n"); /* ... so re-add it */
}

static void
reset_startup_buffer (const char *text)
{
  GtkTextIter start, end;
  gtk_text_buffer_get_iter_at_offset (se_startup_buf, &start,
				      se_startup_insert);
  gtk_text_buffer_get_end_iter (se_startup_buf, &end);
  gtk_text_buffer_delete (se_startup_buf, &start, &end);
  gtk_text_buffer_get_iter_at_offset (se_startup_buf, &start,
				      se_startup_insert);
  gtk_text_buffer_insert (se_startup_buf, &start, text, -1);
  gtk_text_buffer_set_modified (se_startup_buf, FALSE);
}

static void
se_startup_tag_cb (GtkWidget *widget, GtkStyle *oldstyle, gpointer data)
{
  if (se_startup_tag)
    g_object_set (se_startup_tag, "foreground-gdk",
		  &widget->style->text[GTK_STATE_INSENSITIVE], NULL);
}

static void
se_startup_response_cb (GtkWidget *widget, int response, gpointer data)
{
  GtkTextIter start, end;

  switch (response)
  {
  case 1:
    {
      char *file = g_strconcat (g_get_home_dir (), "/.gxine/startup", NULL);
      char *tmp = read_entire_file (file, NULL);
      if (!tmp && errno != ENOENT)
	display_error (FROM_GXINE, _("Couldn't load startup script"), "%s: %s",
		       file, strerror (errno));
      else
	modified = FALSE;
      free (file);
      reset_startup_buffer (tmp ? : "");
      free (tmp);
    }
    gtk_widget_grab_focus (se_startup_textarea);
    break;
  case GTK_RESPONSE_REJECT:
    reset_startup_buffer (se_startup_cmds);
    gtk_widget_grab_focus (se_startup_textarea);
    break;
  case GTK_RESPONSE_OK:
    free (se_startup_cmds);
    gtk_text_buffer_get_iter_at_offset (se_startup_buf, &start,
					se_startup_insert);
    gtk_text_buffer_get_end_iter (se_startup_buf, &end);
    se_startup_cmds = gtk_text_buffer_get_text (se_startup_buf, &start, &end,
						FALSE);
    g_strstrip (se_startup_cmds);
    modified |= gtk_text_buffer_get_modified (se_startup_buf);
  default:
    gtk_widget_hide (se_startup_window);
  }
}

static JSBool
show_startup (JSContext *cx, JSObject *obj, uintN argc,
	      jsval *argv, jsval *rval)
{
  if (!GTK_WIDGET_VISIBLE (se_startup_window))
    reset_startup_buffer (se_startup_cmds);
  window_show (se_startup_window, NULL);
  return JS_TRUE;
}

void engine_init (void)
{
  char *cfgfilename;
  GtkWidget *w;
  
  /*
   * some xine engine functions may block for a long
   * time - that's why it is often not called directly
   * but by using an extra thread so the gui will stay
   * responsive
   */

  pthread_mutex_init (&engine_lock, NULL);

  /*
   * init the global xine engine
   */

  xine = xine_new ();

  cfgfilename = g_strconcat (g_get_home_dir(), "/.gxine/config", NULL);
  xine_config_load (xine, cfgfilename);
  free (cfgfilename);
  xine_engine_set_param (xine, XINE_ENGINE_PARAM_VERBOSITY, verbosity);

  xine_init (xine);
  ui_preferences_register (xine);

  /* init the javascript engine */
  
  gse = se_new ();

  {
    static const se_f_def_t defs[] = {
      { "startup_cmds_show", show_startup, 0, 0, SE_GROUP_DIALOGUE, NULL, NULL },
      { NULL }
    };
    se_defuns (gse, gse->g, defs);
  }

  /*
   * create the startup commands editor window
   */
   
  se_startup_window = gtk_dialog_new_with_buttons (_("Startup script"), NULL, 0,
				GTK_STOCK_REVERT_TO_SAVED, 1,
				GTK_STOCK_UNDO, GTK_RESPONSE_REJECT,
				GTK_STOCK_CANCEL, GTK_RESPONSE_DELETE_EVENT,
				GTK_STOCK_OK, GTK_RESPONSE_OK,
				NULL);
  gtk_window_set_default_size (GTK_WINDOW (se_startup_window), 400, 250);
  gtk_dialog_set_default_response (GTK_DIALOG (se_startup_window),
				   GTK_RESPONSE_OK);
  g_object_connect (se_startup_window,
	"signal::delete-event", G_CALLBACK (gtk_widget_hide_on_delete), NULL,
	"signal::response", G_CALLBACK (se_startup_response_cb), NULL,
	NULL);
  w = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (w),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  se_startup_buf = gtk_text_buffer_new (NULL);
  se_startup_textarea = gtk_text_view_new_with_buffer (se_startup_buf);
  gtk_widget_set_name (se_startup_textarea, "gxine_js_edit");
  gtk_container_add (GTK_CONTAINER (w), se_startup_textarea);
  gtk_box_pack_start_defaults (GTK_BOX (GTK_DIALOG (se_startup_window)->vbox),
			       w);
  g_signal_connect (se_startup_textarea, "style-set",
		    G_CALLBACK (se_startup_tag_cb), NULL);
}

void
engine_startup_script (void)
{
  gchar *file = g_strconcat (g_get_home_dir (), "/.gxine/startup", NULL);
  GtkTextIter pos;

  file = read_entire_file (GXINE_CONFDIR "/startup", NULL);
  if (!file && errno != ENOENT)
    display_error (FROM_GXINE, _("Couldn't load startup script"), "%s: %s",
		   GXINE_CONFDIR "/startup", strerror (errno));
  else if (file)
  {
    se_eval_ext (gse, file, NULL, NULL, NULL, se_error_cb,
		 _("Startup script (system)"));
    se_startup_tag = gtk_text_buffer_create_tag (se_startup_buf, "global",
		"editable", FALSE, "foreground-gdk",
		&se_startup_textarea->style->text[GTK_STATE_INSENSITIVE],
		"scale", 0.9, NULL);
    gtk_text_buffer_get_start_iter (se_startup_buf, &pos);
    gtk_text_buffer_insert_with_tags (se_startup_buf, &pos, file, -1,
				      se_startup_tag, NULL);
  }
  free (file);
  gtk_text_buffer_get_end_iter (se_startup_buf, &pos);
  se_startup_insert = gtk_text_buffer_get_char_count (se_startup_buf);

  se_startup_cmds = load_user_startup_script ();
  se_eval (gse, se_startup_cmds, NULL, NULL, NULL, _("Startup script"));
}

void
save_startup_script (void)
{
  char *fname;

  /* only save if the buffer is modified */
  if (!modified)
    return;

  fname = g_strconcat (g_get_home_dir(), "/.gxine/startup", NULL);
  FILE *f = fopen (fname, "w");
  if (f)
  {
    fputs (se_startup_cmds, f);
    if (ferror (f))
      display_error_modal (FROM_GXINE, _("Failed to save startup script"),
			   _("Error while writing to '%s': %s"),
			   fname, strerror (errno));
    if (fclose (f))
      display_error_modal (FROM_GXINE, _("Failed to save startup script"),
			   _("Error when closing '%s': %s"),
			   fname, strerror (errno));
  }
  else
    display_error_modal (FROM_GXINE, _("Failed to save startup script"),
			 _("Can't open file '%s': %s"),fname, strerror (errno));
  free (fname);
}

