/* 
 * Copyright (C) 2003-2004 the xine project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: noskin_window.c,v 1.90 2005/11/02 23:52:10 dsalt Exp $
 *
 * standard, non-skinned main window
 */

#include "globals.h"

#include "noskin_window.h"

#include <stdio.h>
#include <gtk/gtk.h>
#include <gdk/gdkx.h>
#include <stdlib.h>
#include <string.h>

#include "ui.h"
#include "utils.h"
#include "engine.h"
#include "key_events.h"
#include "player.h"
#include "playlist.h"
#include "mediamarks.h"
#include "menu.h"
#include "drag_drop.h"
#include "gtkvideo.h"
#include "gtkflipbutton.h"
#include "info_widgets.h"
#include "vis.h"
#include "xml_widgets.h"

/*
#define VIS_WIDGET
*/
#undef USE_NET_WM_STATE_SHADED

/* global */
GtkWidget *gtv;

/* private/local */
static GtkTooltips    *tips;
static int             have_video = -1, have_audio = -1;

static GtkWidget *fs_toolbar, *wm_toolbar = NULL;

static void gtk_window_set_shade (GtkWindow *w, gboolean shade)
{
  GdkWindow *window = ((GtkWidget *)w)->window;
  Display *display = GDK_WINDOW_XDISPLAY (window);
  XEvent xev;

  xev.xclient.type = ClientMessage;
  xev.xclient.serial = 0;
  xev.xclient.send_event = True;
  xev.xclient.window = GDK_WINDOW_XID (window);
  xev.xclient.message_type = XInternAtom (display, "_NET_WM_STATE", False);
  xev.xclient.format = 32;
  xev.xclient.data.l[0] = shade ? 1 : 0; /* _NET_WM_STATE_{ADD,REMOVE} */
  xev.xclient.data.l[1] = XInternAtom (display, "_NET_WM_STATE_SHADED", False);
  xev.xclient.data.l[2] = 0;
  xev.xclient.data.l[3] = 0;
  xev.xclient.data.l[4] = 0;

  XSendEvent (display, DefaultRootWindow (display) /*FIXME:screen*/, False,
              SubstructureRedirectMask | SubstructureNotifyMask, &xev);
}

/*
 * audio only / vis check
 */

void window_check_vis (gboolean force)
{
  int hv_new = xine_get_stream_info (stream, XINE_STREAM_INFO_HAS_VIDEO);
  int ha_new = xine_get_stream_info (stream, XINE_STREAM_INFO_HAS_AUDIO);
  gboolean fs = gtk_video_is_fullscreen (GTK_VIDEO(gtv));
  gboolean was_shown = GTK_WIDGET_DRAWABLE (gtv);
  gboolean is_shown = was_shown;

  if (force || have_video != hv_new || have_audio != ha_new)
  {
    have_video = hv_new;
    have_audio = ha_new;
    if (hv_new) {
      logprintf ("gxine: stream has video\n");
      vis_hide (GTK_VIDEO(gtv), &audio_port);
#ifdef USE_NET_WM_STATE_SHADED
      if (wm_toolbar)
	gtk_window_set_shade (GTK_WINDOW (app), FALSE);
      else
#endif
	gtk_widget_show (gtv);
      is_shown = TRUE;
    }
    else if (!ha_new)
    {
      logprintf ("gxine: stream has no audio\n");
      /* stream probably not initialised yet - don't show or hide */
    }
    else if (vis_show (GTK_VIDEO(gtv), &audio_port))
    {
      logprintf ("gxine: stream has audio (visualisation active)\n");
#ifdef USE_NET_WM_STATE_SHADED
      if (wm_toolbar)
	gtk_window_set_shade (GTK_WINDOW (app), FALSE);
      else
#endif
	gtk_widget_show (gtv);
      is_shown = TRUE;
    }
    else
    {
      logprintf ("gxine: stream has audio\n");
      if (fs)
      {
	fs = FALSE;
	gtk_action_activate ((GtkAction *)action_items.fullscreen);
      }
#ifdef USE_NET_WM_STATE_SHADED
      if (wm_toolbar)
	gtk_window_set_shade (GTK_WINDOW (app), TRUE);
      else
#endif
	gtk_widget_hide (gtv);
      is_shown = FALSE;
    }
    if (action_items.fullscreen)
      gtk_action_set_sensitive ((GtkAction *)action_items.fullscreen, is_shown);
    /* Kludge: resize if the video widget visibility has been changed.
     * If you have a better way of doing this, submit a patch...
     */
#ifdef USE_NET_WM_STATE_SHADED
    if (!fs && !wm_toolbar && is_shown != was_shown)
      gtk_window_resize (GTK_WINDOW(app), 10, 10);
#else
    if (!fs && is_shown != was_shown)
      gtk_window_resize (GTK_WINDOW(app), 10, 10);
#endif
  }
}

/*
 * slider
 */

static gboolean update_slider_cb (gpointer data)
{
  gint pos_stream, pos_time, length_time;

  window_check_vis (FALSE);
  if (xine_get_status (stream) == XINE_STATUS_PLAY
      && xine_get_pos_length (stream, &pos_stream, &pos_time, &length_time))
  {
    ui_set_control_adjustment (Control_SEEKER, pos_stream);
    /* update the control buttons while we're here */
    ui_set_status (UI_CURRENT_STATE);
  }

  return TRUE;
}

gint close_application (GtkWidget *widget, GdkEvent *event, gpointer data)
{
  engine_exec ("exit();", NULL, NULL, NULL);
  return FALSE;
}

/*
 * full-screen toolbar
 */

static gint cw_height = 1;

static gboolean window_fs_toolbar_move (GdkEventConfigure *event)
{
  int h = fs_toolbar_at_top
	  ? 1
	  : (event ? event->height : app->allocation.height) - cw_height - 1;
  gtk_window_move (GTK_WINDOW(fs_toolbar),
		   event ? event->x : app->allocation.x,
		   (event ? event->y : app->allocation.y) + h);
  return FALSE;
}

static gboolean window_fs_toolbar_show_int (void)
{
  window_fs_toolbar_move (NULL);
  window_show (fs_toolbar, NULL);
  gtk_window_stick (GTK_WINDOW(fs_toolbar));
  return FALSE;
}

void window_fs_toolbar_show (gboolean show)
{
  fs_toolbar_visible = show;

  if (!show)
    window_fs_toolbar_reset ();
  else if (gtk_video_is_fullscreen (GTK_VIDEO(gtv)))
    window_fs_toolbar_show_int ();

  ui_set_status (UI_FS_TOOLBAR);
}

void window_fs_toolbar_reset (void)
{
  gtk_widget_hide (fs_toolbar);
}

void window_fs_toolbar_restore (void)
{
  if (fs_toolbar_visible)
    window_fs_toolbar_show_int ();
}

void window_fs_toolbar_position (gboolean top)
{
  fs_toolbar_at_top = top;
  window_fs_toolbar_move (NULL);
  ui_set_status (UI_FS_TOOLBAR_POS);
}

static void window_fs_toolbar_set_geometry (void)
{
  GdkGeometry    cw_geom = {0};
  cw_geom.max_width = cw_geom.min_width = gdk_screen_width ();
  cw_geom.max_height = cw_geom.min_height = -1;
  cw_geom.win_gravity = GDK_GRAVITY_SOUTH_WEST;
  gtk_window_set_geometry_hints
    (GTK_WINDOW(fs_toolbar), fs_toolbar, &cw_geom,
     GDK_HINT_POS | GDK_HINT_MIN_SIZE | GDK_HINT_MAX_SIZE | GDK_HINT_BASE_SIZE
     | GDK_HINT_USER_POS);
}

static void cw_show_cb (GtkWidget *widget)
{
  if (gtk_toggle_action_get_active (GTK_TOGGLE_ACTION(action_items.fs_toolbar)))
    window_fs_toolbar_show (TRUE);
}

static gboolean fs_toolbar_configure_cb (GtkWidget *widget,
					 GdkEventConfigure *event,
					 gpointer data)
{
  if (cw_height == event->height)
    return FALSE;
  cw_height = event->height;
  if (fs_toolbar_visible)
    g_idle_add ((GSourceFunc) window_fs_toolbar_move, NULL);
  return FALSE;
}

/*
 * windowed-mode toolbar
 */

void window_wm_toolbar_show (gboolean show)
{
  if (!wm_toolbar)
    return;

  wm_toolbar_visible = show;

  if (!show)
    window_wm_toolbar_reset ();
  else if (!gtk_video_is_fullscreen (GTK_VIDEO(gtv)))
    window_wm_toolbar_restore ();

  ui_set_status (UI_WM_TOOLBAR);
}

void window_wm_toolbar_reset (void)
{
  if (wm_toolbar)
    gtk_widget_hide (wm_toolbar);
}

void window_wm_toolbar_restore (void)
{
  if (wm_toolbar)
    gtk_window_present (GTK_WINDOW (wm_toolbar));
}

void window_wm_toolbar_set_snap (gboolean snap)
{
  if (wm_toolbar)
  {
    wm_toolbar_snap = snap;
    if (snap)
      window_wm_toolbar_snap ();
  }
}

/* callbacks */

/* whether gdk_window_lower() is used; present in case the wm toolbar window
 * should somehow gain the input focus, since calling this may move the focus
 * elsewhere
 */
static gboolean can_lower = FALSE;

gboolean window_wm_toolbar_snap (void)
{
  GdkRectangle appframe, tbframe;
  int x, y;

  if (!wm_toolbar)
    return FALSE;

  gdk_window_get_frame_extents (app->window, &appframe);
  gdk_window_get_frame_extents (wm_toolbar->window, &tbframe);

  /* centre horizontally wrt video window */
  x = appframe.x + (appframe.width - tbframe.width) / 2;
  /* ensure fully on-screen unless the video window isn't fully on-screen */
  if (appframe.width >= tbframe.width)
  {
    if (x < 0)
      x = MIN (0, appframe.x + appframe.width - tbframe.width);
    else if (x + tbframe.width >
	     (y = gdk_screen_get_width (((GtkWindow *)app)->screen)))
      x = MAX (y - tbframe.width, appframe.x);
  }
  else
  {
    if (x < 0)
      x = MIN (0, appframe.x);
    else if (x + tbframe.width >
	     (y = gdk_screen_get_width (((GtkWindow *)app)->screen)))
      x = MAX (y - tbframe.width, appframe.x + appframe.width - tbframe.width);
  }

  /* position just below the video window, or above if insufficient space */
  y = appframe.y + appframe.height;
  if (y + tbframe.height > gdk_screen_get_height (((GtkWindow *)app)->screen))
    y = appframe.y - tbframe.height;

  /* reposition if needed */
  if (x != tbframe.x || y != tbframe.y)
  {
    if (can_lower)
      gdk_window_lower (wm_toolbar->window);
    gtk_window_move (GTK_WINDOW (wm_toolbar), x, y);
  }

  return FALSE;
}

static gboolean popup_cb (GtkWidget *widget, gpointer data)
{
  if (GTK_WIDGET_VISIBLE (wm_toolbar ? : menubar))
    gtk_menu_shell_select_first (GTK_MENU_SHELL (menubar), FALSE);
  else
    gtk_menu_popup (GTK_MENU (popup_menu), NULL, NULL, NULL, NULL, 0,
		    gtk_get_current_event_time ());
  return TRUE;
}

static void wm_close_cb (GtkWidget *widget, gpointer data)
{
  if (gtk_toggle_action_get_active (action_items.wm_toolbar))
    gtk_action_activate (&action_items.wm_toolbar->parent);
}

static void fs_show_hide (GtkWidget *widget, gpointer show)
{
  if (widget == gtv)
    return;
  if (show)
    gtk_widget_show (widget);
  else
    gtk_widget_hide (widget);
}

static gboolean app_state_cb (GtkWindow *window, GdkEventWindowState *event,
			      gpointer data)
{
  static gboolean sticky = FALSE, lock = FALSE;

  if (event->changed_mask & GDK_WINDOW_STATE_STICKY)
  {
    if (!lock)
    {
      if (gtk_video_is_fullscreen ((GtkVideo *)gtv))
	gtk_window_stick (window); /* prevent unstick in fullscreen (bug) */
      else
	sticky = !!(event->new_window_state & GDK_WINDOW_STATE_STICKY);
    } 
    lock = FALSE;
  }

  if (event->changed_mask & GDK_WINDOW_STATE_FULLSCREEN)
  {
    gboolean notfs = !(event->new_window_state & GDK_WINDOW_STATE_FULLSCREEN);

    if (notfs)
    {
      if (!sticky)
	gtk_window_unstick (window); /* may have been stuck by us, not user */
      gtk_window_set_keep_above (window, FALSE);
      if (!wm_toolbar)
	gtk_container_foreach (GTK_CONTAINER (window->bin.child),
			       fs_show_hide, window);
      else if (wm_toolbar_visible)
	window_wm_toolbar_restore ();
    }
    else
    {
      lock = !sticky; /* we may be triggering a state change: lock it out */
      gtk_window_set_shade (window, FALSE);
      gtk_window_stick (window);
      if (!wm_toolbar)
	gtk_container_foreach (GTK_CONTAINER (window->bin.child),
			       fs_show_hide, NULL);
      else
	window_wm_toolbar_reset ();
      gtk_window_set_keep_above (window, TRUE);
      XSetInputFocus (GDK_WINDOW_XDISPLAY (gtv->window),
		      GDK_WINDOW_XWINDOW (gtv->window),
		      RevertToParent, CurrentTime);
    }
    
    gtk_toggle_action_set_active (action_items.fullscreen, !notfs);
  }

  return FALSE;
}

static gboolean refocus_cb (gpointer widget)
{
  gtk_widget_grab_focus (widget);
  return FALSE;
}

static gboolean appfocus_cb (GtkWidget *w, GdkEventFocus *e, gpointer d)
{
  puts ("Eek, toolbar has focus");
  g_idle_add (refocus_cb, app);
  return TRUE;
}

static GdkFilterReturn
xevent_filter_cb (GdkXEvent *gx, GdkEvent *e, gpointer d)
{
  const XEvent *x = (XEvent *)gx;
  switch (x->type)
  {
  case ClientMessage:
    /* Lower the toolbar */
    if (wm_toolbar && wm_toolbar_snap && can_lower)
      gdk_window_lower (wm_toolbar->window);
    break;

  case FocusIn:
    return GDK_FILTER_REMOVE; /* else the screen goes black :-| */

  case FocusOut:
    if (gtk_video_is_fullscreen ((GtkVideo *) gtv))
    {
      Display *xd = GDK_WINDOW_XDISPLAY (((GtkWidget *) d)->window);
      Window fw = None;
      int fr;
      XGetInputFocus (xd, &fw, &fr);
      GdkWindow *gw = gdk_window_lookup (fw);
      if (!gw || gw == fs_toolbar->window)
	XSetInputFocus (xd, x->xany.window, RevertToParent, CurrentTime);
    }
    return GDK_FILTER_REMOVE;
  }

  return GDK_FILTER_CONTINUE;
}

static gboolean js_queue_cb (GtkWidget *widget, GdkEventClient *event,
			    gpointer data)
{
  char **js;
  while ((js = g_async_queue_try_pop (js_queue)))
  {
    engine_exec (js[0], NULL, NULL, js[1]);
    free (js[1]);
    free (js);
  }
  return TRUE;
}

static gboolean noskin_main_init_cb (void)
{
  gtk_widget_add_events (app, GDK_KEY_RELEASE_MASK);
  g_object_connect (G_OBJECT (app),
	"signal::key-press-event", G_CALLBACK (keypress_cb), wm_toolbar,
	"signal::popup-menu", G_CALLBACK(popup_cb), NULL,
	NULL);
  ui_set_status (xine_get_param (stream, XINE_PARAM_AUDIO_MUTE)
		 ? UI_AUDIO_MUTE : UI_AUDIO_UNMUTE);
  gdk_window_set_events (app->window, GDK_ALL_EVENTS_MASK); /* seems to work */
  gdk_window_add_filter (app->window, xevent_filter_cb, app);
  return FALSE;
}

/* initialisation */

gboolean app_configure_cb (GtkWidget *widget, GdkEventConfigure *event,
			   gpointer data)
{
  /* redisplay/reposition the toolbar (XRANDR support) */
  if (gtk_video_is_fullscreen (GTK_VIDEO (gtv)))
  {
#ifdef HAVE_XRANDR
    if (fs_toolbar_visible)
      window_fs_toolbar_move (event);
    window_fs_toolbar_set_geometry ();
    gtk_window_resize (GTK_WINDOW (fs_toolbar), event->width,
		       fs_toolbar->allocation.height);
#endif
  }
  else if (wm_toolbar_snap)
  {
    window_wm_toolbar_snap ();
    if (event->width != GTK_VIDEO_MIN_WIDTH &&
	event->height != GTK_VIDEO_MIN_HEIGHT)
      can_lower = TRUE; /* we've done the initial resize */
  }
  return FALSE;
}

static gboolean noskin_post_init (gboolean unused)
{
  xine_cfg_entry_t entry;
  if (xine_config_lookup_entry (xine, "gui.window_size", &entry))
  {
    static const gdouble scale[] = { 50, 100, 200 };
    gtk_video_rescale ((GtkVideo *)gtv, scale[entry.num_value]);
  }
  if (xine_config_lookup_entry (xine, "gui.magnify_lowres_video", &entry))
    gtk_video_set_auto_rescale ((GtkVideo *)gtv, entry.num_value);
  return FALSE;
}


void noskin_main_init (void)
{
  GtkWidget     *vbox;
  GtkWindow     *W;
#ifdef VIS_WIDGET
  GtkWidget     *vis;
  xine_post_out_t *vis_out;
#endif
  xine_cfg_entry_t entry;

  tips = gtk_tooltips_new ();

  /* create app windows */

  app = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  fs_toolbar = gtk_window_new (GTK_WINDOW_TOPLEVEL);

  if (!xine_config_lookup_entry (xine, "gui.windowedmode_separate_toolbar",
				 &entry)
      || entry.num_value)
    wm_toolbar = gtk_window_new (GTK_WINDOW_TOPLEVEL);

  /*
   * toolbar window (full-screen mode)
   */

  g_object_connect (G_OBJECT (fs_toolbar),
	"signal::delete-event", G_CALLBACK (wm_close_cb), NULL,
	"signal::configure-event", G_CALLBACK (fs_toolbar_configure_cb), NULL,
	/* cope with WMs which allow other means of closing windows */
	"signal::unmap-event", G_CALLBACK (cw_show_cb), NULL,
	NULL);

  W = GTK_WINDOW (fs_toolbar);

  gtk_window_set_title (W, _("gxine controls"));
  gtk_window_set_resizable (W, FALSE);
  gtk_window_set_type_hint (W, GDK_WINDOW_TYPE_HINT_TOOLBAR);
  gtk_window_set_decorated (W, FALSE);
  /* I'd like to have the toolbar window not accept the input focus, but
   * some programs show but don't re-hide their toolbars if that's so...
   */
  /* gtk_window_set_focus_on_map (W, FALSE); */
  /* gtk_window_set_accept_focus (W, FALSE); */
  gtk_window_set_position (W, GTK_WIN_POS_NONE);
  gtk_window_set_skip_pager_hint (W, TRUE);
  gtk_window_set_skip_taskbar_hint (W, TRUE);

  window_fs_toolbar_set_geometry ();

  gtk_container_add (GTK_CONTAINER (fs_toolbar),
		     widget_create_from_xml ("toolbar-fullscreen.xml",
					     "toolbar_fs", NULL, FALSE));

  /*
   * video window
   */

  gtk_window_set_default_size (GTK_WINDOW (app), 320, 240);

  g_object_set (G_OBJECT (app), "allow-shrink", TRUE, NULL);
  g_object_connect (G_OBJECT (app),
	"signal::delete-event", G_CALLBACK (close_application), NULL,
	"signal::configure-event", G_CALLBACK (app_configure_cb), NULL,
	"signal::window-state-event", G_CALLBACK (app_state_cb), NULL,
	"signal::client-event", G_CALLBACK (js_queue_cb), NULL,
	NULL);
  drag_drop_setup (app);

  /*
   * video widget
   */

  gtv = gtk_video_new (xine, stream, xine_get_video_source (stream), 
  	               video_driver_id, 320, 240,
                       0x04 /* press: button 2 */,
                       0x08 /* release: button 3 */);
  drag_drop_setup (gtv);
  g_object_connect (G_OBJECT (gtv),
	/*"signal::key-press-event", G_CALLBACK (keypress_cb), wm_toolbar,*/
	"signal::button-press-event", G_CALLBACK (buttonpress_cb), NULL,
	"signal::button-release-event", G_CALLBACK (buttonrelease_cb), NULL,
	"signal::scale-factor-changed", G_CALLBACK (scale_changed_cb), NULL,
	NULL);

  gtk_init_add (noskin_post_init, NULL);

  if (!wm_toolbar)
  {
    /*
     * main vbox, menu bar (combined window)
     * also add the video widget
     */

    /* this vbox will also contain the info widgets and controls */
    vbox = gtk_vbox_new (0, 0);
    gtk_container_add (GTK_CONTAINER (app), vbox);
    create_menus (app);
    gtk_box_pack_start (GTK_BOX (vbox), menubar, FALSE, FALSE, 0);

    gtk_container_add (GTK_CONTAINER (vbox), gtv);
  }
  else
  {
    /*
     * menu/info/tools window (separate windows)
     * also add the video widget
     */

    W = GTK_WINDOW (wm_toolbar);

    gtk_window_set_title (W, _("gxine controls"));
    gtk_window_set_type_hint (W, GDK_WINDOW_TYPE_HINT_TOOLBAR);
    gtk_window_set_decorated (W, TRUE);
    gtk_window_set_transient_for (W, GTK_WINDOW (app));
    gtk_window_set_position (W, GTK_WIN_POS_NONE);
    gtk_window_set_focus_on_map (W, FALSE);
    gtk_window_set_accept_focus (W, FALSE);
    gtk_window_set_skip_pager_hint (W, FALSE);
    gtk_window_set_skip_taskbar_hint (W, TRUE);

    {
      GdkGeometry geom;
      geom.min_width = -1;
      geom.min_height = -1;
      geom.max_width = 32767; /* hmm... */
      geom.max_height = -1;
      gtk_window_set_geometry_hints (W, wm_toolbar, &geom,
				     GDK_HINT_MIN_SIZE | GDK_HINT_MAX_SIZE);
    }

    g_object_connect (G_OBJECT (wm_toolbar),
        "signal::grab-focus", G_CALLBACK (appfocus_cb), NULL,
	"signal::key-press-event", G_CALLBACK (keypress_cb), NULL,
	"signal::map-event", G_CALLBACK (window_wm_toolbar_snap), NULL,
	"signal::delete-event", G_CALLBACK (wm_close_cb), NULL,
	NULL);

    /* this vbox will also contain the info widgets and controls */
    vbox = gtk_vbox_new (0, 0);
    gtk_container_add (GTK_CONTAINER (wm_toolbar), vbox);

    create_menus (wm_toolbar);
    gtk_box_pack_start (GTK_BOX (vbox), menubar, FALSE, FALSE, 0);

    gtk_container_add (GTK_CONTAINER (app), gtv);
  }

  /* time_widget / infobar */

  gtk_box_pack_start (GTK_BOX (vbox),
		      widget_create_from_xml ("toolbar-window.xml",
					      "toolbar_wm", NULL, FALSE),
		      FALSE, FALSE, 0);

  /* Show the app window (and ensure that it's resized properly) */

  gtk_widget_show_all (app);
  gtk_window_resize (GTK_WINDOW (app),
		     GTK_VIDEO_MIN_WIDTH, GTK_VIDEO_MIN_HEIGHT);

  if (wm_toolbar)
  {
    gtk_widget_show_all (wm_toolbar);
    XSelectInput (GDK_WINDOW_XDISPLAY (app->window), GDK_WINDOW_XID (app->window),
		  VisibilityChangeMask);
  }

  have_video = 1;

  g_idle_add ((GSourceFunc) noskin_main_init_cb, NULL);
  g_timeout_add (500, update_slider_cb, NULL);
}
