/*
 * Copyright (C) 2002-2005 the xine-project
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * $Id: playlist.c,v 1.148 2005/11/10 18:55:29 dsalt Exp $
 *
 * playlist implementation
 *
 * .pls parser stolen from totem (C) 2002 Bastien Nocera
 */

#include "globals.h"

#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <ctype.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>

#include <X11/Xlib.h>

#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <glib.h>
#include <xine/xmlparser.h>

#include <pthread.h>

#include "playlist.h"
#include "http.h"
#include "ui.h"
#include "utils.h"
#include "play_item.h"
#include "mediamarks.h"
#include "engine.h"
#include "player.h"
#include "drag_drop.h"

/*
#define LOG
*/

static int             is_visible;
static GtkUIManager   *pl_ui;
static GtkListStore   *pl_store;
static GtkWidget      *dlg, *tree_view;
static GtkTreeSelection *sel;
static GtkWidget      *repeat_button, *random_button;
static char            logo_mrl[1024];
static int             logo_mode;

static play_item_t    *cur_item = NULL;
static int             cur_list_pos;
static int	       ref_list_pos;

static pthread_mutex_t cur_item_lock = PTHREAD_MUTEX_INITIALIZER;
#define CUR_ITEM_LOCK()   pthread_mutex_lock (&cur_item_lock)
#define CUR_ITEM_UNLOCK() pthread_mutex_unlock (&cur_item_lock)

#define unmark_play_item() set_mark_play_item (FALSE)
#define mark_play_item()   set_mark_play_item (TRUE)
static void set_mark_play_item (gboolean state);
static void play_next (void);
static int playlist_clip (int);
static void check_list_empty (void);

static inline void remove_trailing_cr (char *str)
{
  str = strrchr (str, '\r');
  if (str && str[1] == 0)
    *str = 0;
}

static inline void cur_item_dispose (void)
{
  CUR_ITEM_LOCK ();
  if (cur_item)
  {
    play_item_dispose (cur_item);
    cur_item = NULL;
  }
  CUR_ITEM_UNLOCK ();
}

static int item_marked_current (GtkTreeIter *iter)
{
  const char *p;
  GValue v = {0};
  int ret;
  gtk_tree_model_get_value (GTK_TREE_MODEL (pl_store), iter, 3, &v);
  p = g_value_peek_pointer (&v);
  ret = p && *p;
  g_value_unset (&v);
  return ret;
}

static gboolean close_cb (GtkWidget* widget, gpointer data) {
  is_visible = FALSE;
  gtk_widget_hide (dlg);

  return TRUE;
}

void playlist_clear (void)
{
  if (!logo_mode)
    se_eval (gse, "stop()", NULL, NULL, NULL, NULL);
  gtk_list_store_clear (pl_store);
  check_list_empty ();
}

static void clear_cb (GtkWidget* widget, gpointer data) {
  playlist_clear ();
  playlist_logo ();
}

static void make_mediamark_cb (GtkWidget* widget, gpointer data)
{
  GtkTreeIter iter;

  if (gtk_tree_selection_get_selected (sel, NULL, &iter))
  {
    GValue v = {0};
    gtk_tree_model_get_value (GTK_TREE_MODEL (pl_store), &iter, 2, &v);
    mm_add (play_item_copy (g_value_peek_pointer (&v)));
    g_value_unset (&v);
  }
}

static void del_cb (GtkWidget* widget, gpointer data)
{
  GtkTreeIter iter;  

  if (gtk_tree_selection_get_selected (sel, NULL, &iter)) {

    gint        *indices;
    GtkTreePath *path;
    gint         pos;

    path = gtk_tree_model_get_path (GTK_TREE_MODEL (pl_store), &iter);
    indices = gtk_tree_path_get_indices (path);
    pos = indices[0];

    if (pos == cur_list_pos && !logo_mode)
      play_next ();

    if (pos <= cur_list_pos)
      --cur_list_pos;
    gtk_list_store_remove (pl_store, &iter);

    gtk_tree_selection_select_path (sel, path);
    gtk_tree_path_free (path);
    check_list_empty ();
  }
}

static void copy_cb (GtkWidget* widget, gpointer data)
{
  clip_set_play_item_from_selection (tree_view);
}

static void cut_cb (GtkWidget* widget, gpointer data)
{
  copy_cb (widget, data);
  del_cb (widget, data);
}

static void paste_cb (GtkWidget* widget, gpointer data)
{
  play_item_t *item = clip_get_play_item ();
  if (item)
  {
    GtkTreeIter iter;

    if (gtk_tree_selection_get_selected (sel, NULL, &iter))
    {
      GtkTreePath *path;
      path = gtk_tree_model_get_path (GTK_TREE_MODEL (pl_store), &iter);
      playlist_add (item, gtk_tree_path_get_indices (path)[0] + 1);
      gtk_tree_path_free (path);
    }
    else
      playlist_add (item, -1);
  }
}

void playlist_add_list (GSList *fnames, gboolean play)
{
  if (fnames)
  {
    GSList *iter = fnames;
    int pos = playlist_add_mrl (iter->data, -1);

    foreach_glist (iter, iter->next)
    {
      playlist_add_mrl (iter->data, -1);
      free (iter->data);
    }
    g_slist_free (fnames);

    if (play)
      playlist_play (pos);
  }
}

static void add_cb (GtkWidget* widget, gpointer data)
{
  playlist_add_list
    (modal_multi_file_dialog (_("Select files to add to playlist"),
			      FALSE, NULL, dlg),
     FALSE);
}

static void edit_cb (GtkWidget* widget, gpointer data)
{
  GtkTreeIter iter;

  if (gtk_tree_selection_get_selected (sel, NULL, &iter)) {
    GValue       v = {0};
    play_item_t *play_item, *item_copy;
    const char  *title;

    gtk_tree_model_get_value (GTK_TREE_MODEL (pl_store), &iter, 2, &v);
    play_item = g_value_peek_pointer (&v);

    item_copy = play_item_copy (play_item); /* in case of edit */

    title = item_marked_current (&iter)
	    ? xine_get_meta_info (stream, XINE_META_INFO_TITLE) : NULL;

    if (play_item_edit (play_item, PLAY_ITEM_LIST, title, dlg))
    {
      if (item_marked_current (&iter))
      {
	CUR_ITEM_LOCK ();
	cur_item = item_copy;
	CUR_ITEM_UNLOCK ();
      }
      else
	play_item_dispose (item_copy);

      gtk_list_store_set (pl_store, &iter,
			  0, play_item->title,
			  1, play_item->mrl, 2, play_item, -1);
    }
    else
      play_item_dispose (item_copy);

    g_value_unset (&v);
  }
}

static void playlist_next_cb (GtkWidget *widget, gpointer data)
{
  int pos = playlist_get_list_pos ();
  int newpos = playlist_clip (pos + 1);
  if (newpos != pos || logo_mode)
    playlist_play (newpos);
}

static void playlist_prev_cb (GtkWidget *widget, gpointer data)
{
  int pos = playlist_get_list_pos ();
  int newpos = playlist_clip (pos - 1);
  if (newpos != pos || logo_mode)
    playlist_play (newpos);
}

static int playlist_load (char *fname)
{
  char  *plfile;
  xml_node_t *node;
  gboolean defaultfile = !fname;

  if (!fname)
    fname = g_strconcat(g_get_home_dir(), "/.gxine/playlist", NULL);

  plfile = read_entire_file (fname, NULL);

  if (!plfile)
  {
    if (errno != ENOENT && (defaultfile && errno != ENOTDIR))
      display_error (FROM_GXINE, _("Loading of playlist file failed."),
		     _("Failed to open file '%s'\n%s"), fname, strerror (errno));
    goto ret0;
  }

  xml_parser_init (plfile, strlen (plfile), XML_PARSER_CASE_INSENSITIVE);

  if (xml_parser_build_tree (&node) < 0)
  {
    display_error (FROM_GXINE, _("Loading of playlist file failed."),
		   _("'%s' is not a valid XML/ASX file"), fname);
    goto ret0;
  }

  if (!strcasecmp (node->name, "ASX"))
  {
    /* Attributes: VERSION, PREVIEWMODE, BANNERBAR
     * Child elements: ABSTRACT, AUTHOR, BASE, COPYRIGHT, DURATION, ENTRY,
     ENTRYREF, MOREINFO, PARAM, REPEAT, TITLE
     */
    //const char *base_href = NULL;

    const char *version = xml_parser_get_property (node, "VERSION") ? : "";
    int version_major, version_minor = 0;

    if (!((sscanf (version, "%d.%d", &version_major, &version_minor) == 2 ||
	   sscanf (version, "%d", &version_major) == 1) &&
	  (version_major == 3 && version_minor == 0)))
    {
      display_error (FROM_GXINE, _("Loading of playlist file failed."),
		     _("Unknown or incorrect ASX version number in '%s'"),
		     fname);
      goto ret0;
    }

    playlist_clear ();

    foreach_glist (node, node->child)
    {
      //const char *ref_base_href = base_href;

      if (!strcasecmp (node->name, "SETTINGS"))
      {
        /* gxine-specific */
	gtk_toggle_button_set_active
	  (GTK_TOGGLE_BUTTON (random_button),
	   xml_parser_get_property_bool (node, "RANDOM", 0));
	gtk_toggle_button_set_active
	  (GTK_TOGGLE_BUTTON (repeat_button),
	   xml_parser_get_property_bool (node, "REPEAT", 0));
      }

      else if (!strcasecmp (node->name, "ENTRY"))
      {
	play_item_t *play_item = play_item_load (node->child);
	GtkTreeIter  iter;
	gtk_list_store_append (pl_store, &iter);
	gtk_list_store_set (pl_store, &iter, 0, play_item->title, 1, 
			    play_item->mrl, 2, play_item, 3, "", -1);
      }

      else if (!strcasecmp (node->name, "ENTRYREF"))
      {
	/* Attributes: HREF, CLIENTBIND */
	play_item_t *play_item =
	  play_item_new (NULL, xml_parser_get_property (node, "HREF"), 0);
	GtkTreeIter  iter;
	gtk_list_store_append (pl_store, &iter);
	gtk_list_store_set (pl_store, &iter, 0, play_item->title, 1, 
			    play_item->mrl, 2, play_item, 3, "", -1);
      }
/*
      else if (!strcasecmp (node->name, "BASE"))
	base_href = xml_parser_get_property (node, "HREF");
*/
    }
  }
  else
  {
    display_error (FROM_GXINE, _("Loading of playlist file failed."),
		   _("'%s' is not an ASX file"), fname);
    goto ret0;
  }

  if (defaultfile)
    free (fname);
  check_list_empty ();
  return 1;

  ret0:
  if (defaultfile)
    free (fname);
  check_list_empty ();
  return 0;
}

void playlist_save (char *file)
{
  FILE *f;
  char *fname = file ? : g_strconcat(g_get_home_dir(), "/.gxine/playlist", NULL);

  f = fopen (fname, "w");
  if (f)
  {
    static const char tf[][8] = { "false", "true" };

    GtkTreeIter iter;

    fprintf
      (f, "<ASX VERSION=\"3.0\">\n"
          "  <SETTINGS REPEAT=\"%s\" RANDOM=\"%s\"/>\n",
       tf[!!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(repeat_button))],
       tf[!!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(random_button))]);

    if (gtk_tree_model_get_iter_first (GTK_TREE_MODEL (pl_store), &iter)) {

      do {
	GValue vi = {0};
	char *xml;
	gtk_tree_model_get_value (GTK_TREE_MODEL (pl_store),
				  &iter, 2, &vi);
        xml = play_item_xml (g_value_peek_pointer (&vi), 1);
        fputs (xml, f);
        free (xml);
	g_value_unset (&vi);
      } while (gtk_tree_model_iter_next (GTK_TREE_MODEL(pl_store), &iter));
    }

    fprintf (f, "</ASX>\n");
    if (ferror (f))
      display_error_modal (FROM_GXINE, _("Failed to save playlist"),
			   _("Error while writing to '%s': %s"),
			   fname, strerror (errno));
    if (fclose (f))
      display_error_modal (FROM_GXINE, _("Failed to save playlist"),
			   _("Error when closing '%s': %s"),
			   fname, strerror (errno));
  } else 
    display_error_modal (FROM_GXINE, _("Failed to save playlist"),
                         _("Can't open file '%s': %s"),
			 fname, strerror (errno));

  if (fname != file)
    free (fname);
}

static void open_cb (GtkWidget* widget, gpointer data)
{
  char *filename = modal_file_dialog (_("Select playlist..."), TRUE, TRUE,
				      FALSE, "*.asx", NULL, dlg);
  if (filename)
    playlist_load (filename);
  free (filename);
}

static void save_as_cb (GtkWidget* widget, gpointer data)
{
  char *filename = modal_file_dialog (_("Save playlist as..."), FALSE, TRUE,
				      FALSE, "*.asx", FALSE, dlg);
  if (filename)
    playlist_save (filename);
  free (filename);
}

static int playlist_clip (int pos)
{
  int items = gtk_tree_model_iter_n_children (GTK_TREE_MODEL (pl_store), NULL);
  return (pos == -1) ? (items - (items != 0))
		     : (pos < 0 || pos >= items) ? 0 : pos;
}

play_item_t *playlist_get_item (int pos)
{
  GtkTreeIter        iter;  

  if ((pos == -2 /* highlighted item or first item */
       && gtk_tree_selection_get_selected (sel, NULL, &iter))
      ||
      ((pos = playlist_clip (pos)) >= 0 /* nth item */
       && gtk_tree_model_iter_nth_child
	    (GTK_TREE_MODEL (pl_store), &iter, NULL, pos)))
  {
    GValue       v = {0};
    play_item_t *play_item;

    gtk_tree_model_get_value (GTK_TREE_MODEL (pl_store),
			      &iter, 2, &v);
    play_item = g_value_peek_pointer (&v);
    g_value_unset (&v);
      
    return play_item;
  } else
    return NULL;
}

int playlist_get_list_pos (void) {
  return cur_list_pos;
}

play_item_t *playlist_get_current_item (void)
{
  play_item_t *i;
  CUR_ITEM_LOCK ();
  i = play_item_copy (cur_item ? : playlist_get_item (cur_list_pos));
  CUR_ITEM_UNLOCK ();
  return i;
}

int playlist_size (void) {
  return gtk_tree_model_iter_n_children (GTK_TREE_MODEL (pl_store), NULL);
}

static int read_ini_line_int (char **lines, const char *key) {
  int retval = -1;
  int i;

  if (lines == NULL || key == NULL)
    return -1;

  for (i = 0; (lines[i] != NULL && retval == -1); i++) {
    if (g_ascii_strncasecmp (lines[i], key, strlen (key)) == 0)
    {
      char **bits;

      bits = g_strsplit (lines[i], "=", 2);
      if (bits[0] == NULL || bits [1] == NULL) {
	g_strfreev (bits);
	return -1;
      }

      retval = (gint) g_strtod (bits[1], NULL);
      g_strfreev (bits);
    }
  }

  return retval;
}

static char* read_ini_line_string (char **lines, const char *key) {
  char *retval = NULL;
  int i;

  if (lines == NULL || key == NULL)
    return NULL;

  for (i = 0; (lines[i] != NULL && retval == NULL); i++) {
    if (g_ascii_strncasecmp (lines[i], key, strlen (key)) == 0) {
      char **bits;

      bits = g_strsplit (lines[i], "=", 2);
      if (bits[0] == NULL || bits [1] == NULL) {
	g_strfreev (bits);
	return NULL;
      }

      retval = g_strdup (bits[1]);

      g_strfreev (bits);
    }
  }

  return retval;
}

static int playlist_load_m3u (const char *mrl, gint ins_pos) {
  gboolean  retval = FALSE;
  char     *contents, **lines;
  int       size, i, first;
  char     *extension;
  char     *suffix = NULL;

  first = -1;

  if (! (contents = read_entire_file (mrl, &size)) )
    return -1;

  lines = g_strsplit (contents, "\n", 0);
  g_free (contents);

  for (i = 0; lines[i] != NULL; ++i) {
    remove_trailing_cr (lines[i]);
    /* Either it's a URI, or it has a proper path */
    if (strstr(lines[i], "://") != NULL
	|| lines[i][0] == G_DIR_SEPARATOR)  {
      char *tmp;
      extension = strrchr(lines[i], '.');
      if (extension && !strncasecmp (extension, ".mp3", 4)) {
          suffix="#demux:mp3";
      }
      else if (extension && !strncasecmp (extension, ".ogg", 4)) {
          suffix="#demux:ogg";
      }

      tmp = g_strconcat (lines[i], suffix, NULL);
      if (first >= 0)
	playlist_add_mrl (tmp, ins_pos);
      else
	first = playlist_add_mrl (tmp, ins_pos);
      free (tmp);

      retval = TRUE;
    }
  }

  g_strfreev (lines);

  if (retval)
    return first;
  else
    return -1;
}

static int playlist_load_smil (const char *mrl, gint ins_pos) {
  gboolean  retval = FALSE;
  char     *contents, **lines;
  int       size, i, first;
  char     *begin, *end;

  first = -1;

  if (! (contents = read_entire_file (mrl, &size)) )
    return -1;

  lines = g_strsplit (contents, "\n", 0);
  g_free (contents);

  for (i = 0; lines[i] != NULL; ++i) {

    if (strstr(lines[i], "<audio src=") != NULL
	|| lines[i][0] == G_DIR_SEPARATOR)  {
      begin=strstr(lines[i], "<audio src=")+12;
      end=strchr(begin, '"');
      if (!end) end=strchr(begin, '\'');
      if (!end) return -1;
      *end=0;
      logprintf("playlist: got '%s'\n", begin);
      if (first >= 0) {
	playlist_add_mrl (begin, ins_pos);
      }
      else {
	first = playlist_add_mrl (begin, ins_pos);
      }

      retval = TRUE;
    }
    if (strstr(lines[i], "<video src=") != NULL
	|| lines[i][0] == G_DIR_SEPARATOR)  {
      begin=strstr(lines[i], "<video src=")+12;
      end=strchr(begin, '"');
      if (!end) end=strchr(begin, '\'');
      if (!end) return -1;
      *end=0;
      logprintf ("playlist: got '%s'\n", begin);
      if (first >= 0) {
	playlist_add_mrl (begin, ins_pos);
      }
      else {
	first = playlist_add_mrl (begin, ins_pos);
      }

      retval = TRUE;
    }

  }

  g_strfreev (lines);

  if (retval)
    return first;
  else
    return -1;
}

int playlist_add (play_item_t *play_item, gint ins_pos) {

  GtkTreeIter  iter;
  GtkTreePath *path;
  gint        *indices;
  gint         ret;

  if (ins_pos<0)
    gtk_list_store_append (pl_store, &iter);
  else
    gtk_list_store_insert (pl_store, &iter, ins_pos);
    
  gtk_list_store_set (pl_store, &iter, 0, play_item->title, 
		      1, play_item->mrl, 2, play_item, 3, "", -1);

  /*
   * find out entry number in list
   */

  path = gtk_tree_model_get_path (GTK_TREE_MODEL (pl_store), &iter);
  indices = gtk_tree_path_get_indices (path);
  ret = indices[0];
  gtk_tree_path_free (path);

  check_list_empty ();
  return ret;
}

static gint playlist_add_int (const char *mrl, gint ins_pos)
{
  play_item_t *play_item;
  int i;

  /* don't add duplicate items */
  if (ins_pos < 0)
    for (i = 0; i < playlist_size (); ++i)
      if (!strcmp (playlist_get_item (i)->mrl, mrl))
        return i;

  play_item = play_item_new (NULL, mrl, 0);
  return playlist_add (play_item, ins_pos);
}

static int playlist_load_ram (const char *mrl, gint ins_pos) {
  gboolean  retval = FALSE;
  char      contents[1024], mime_type[256], **lines;
  char      *contents_ptr;
  int       size, i, first;
  int       fd;
  int       pos; 

  first = -1;
  sprintf(mime_type,"none");

  /* get first 4 bytes to decide whether it is a real media
   * file or a playlist.
   */
  size = 4;

  if (!strncasecmp (mrl, "http://", 7)) {
  
    size = http_peek (mrl,size, contents, mime_type);
    if (size <= 0)
      return -1;

  } else {

    fd = open (mrl, O_RDONLY);
    if (fd<0)
      return -1;

    size = read (fd, contents, size);
    close (fd);
  }
  if (contents[0]=='.' 
      && contents[1]=='R' 
      && contents[2]=='M' 
      && contents[3]=='F')
  {
    /* this is a real media file ... */
    /* add it directly to the playlist. */

    pos = playlist_add_int (mrl, ins_pos);

    return pos;
  }
    
  if (! (contents_ptr = read_entire_file (mrl, &size)) )
    return -1;

  lines = g_strsplit (contents_ptr, "\n", 0);
  g_free (contents_ptr);

  for (i = 0; lines[i] != NULL; ++i) {
    
    /* ignore comments */
    if (lines[i][0]=='#') continue;
    
    /* honor --stop-- lines */
    if (strstr(lines[i], "--stop--") != NULL) break;
    
    /* Either it's a rtsp, or a pnm mrl, but we also match http mrls here. */
    if (strstr(lines[i], "rtsp://") != NULL
	|| strstr(lines[i], "pnm://") != NULL
	|| strstr(lines[i], "http://") != NULL)  {
      /* we assume real media */

      remove_trailing_cr (lines[i]);

      if (first >= 0) {
	playlist_add_mrl (lines[i], ins_pos);
      }
      else {
	first = playlist_add_mrl (lines[i], ins_pos);
      }
      retval = TRUE;
    }
    
  }

  g_strfreev (lines);

  if (retval)
    return first;
  else
    return -1;
}

static int playlist_load_pls (const char *mrl, gint ins_pos) {

  gboolean  retval = FALSE;
  char     *contents=NULL, **lines;
  int       size, i, num_entries;
  char      mime_type[256];

  /* http or local file ? */

  if (!strncmp (mrl, "http://", 7)) {

    /* get mime type */
    http_peek(mrl, 0, contents, mime_type);

    if (!strcmp(mime_type, "audio/mpeg")) {
      /* add directly to playlist */

      return playlist_add_int (mrl, ins_pos);
    }

    if (! (contents = http_download (mrl, &size)))
      return -1;
  } else {

    if (! (contents = read_entire_file (mrl, &size)) )
      return -1;
  }

  lines = g_strsplit (contents, "\n", 0);
  g_free (contents);

  /* [playlist] */
  if (g_ascii_strncasecmp (lines[0], "[playlist]", 10))
    goto bail;

  /* numberofentries=? */
  num_entries = read_ini_line_int (lines, "numberofentries");
  if (num_entries == -1)
    goto bail;

  for (i = 1; i <= num_entries; i++) {
    char *file, *title;
    char *file_key, *title_key;

    file_key = g_strdup_printf ("file%d", i);
    title_key = g_strdup_printf ("title%d", i);

    file = read_ini_line_string (lines, (const char*)file_key);
    title = read_ini_line_string (lines, (const char*)title_key);

    g_free (file_key);
    g_free (title_key);

    if (file != NULL) {

      int pos = strlen(file)-1;

      while ( (pos>0) && (file[pos]<32) ) {
	file[pos] = 0;
	pos --;
      }

      return playlist_add_mrl (file, ins_pos);

      g_free (file);
      g_free (title);
    } else {
      g_free (title);
    }
  }

bail:
  g_strfreev (lines);

  return retval;
}

static int playlist_load_asx (const char *mrl, gint ins_pos) {

  char             *contents;
  char            **lines;
  int               size, res, ret, i;
  xml_node_t       *xml_tree;
  xml_node_t       *current_node_l1;
  xml_node_t       *current_node_l2;
  xml_property_t   *property;
  int               asx_version;

  ret = -1;

  if (! (contents = read_entire_file (mrl, &size)) )
    return ret;

  xml_parser_init (contents, size, XML_PARSER_CASE_INSENSITIVE);
  res = xml_parser_build_tree (&xml_tree);

  if (res) {
    /* humm, maybe the asx contains no xml? */
    /* try to find mms urls by hand */
    lines = g_strsplit (contents, "\n", 0);
    g_free (contents);

    for (i = 0; lines[i] != NULL; ++i) {
      /* look for mms:// in a line */
      if (strstr(lines[i], "mms://"))  {
        remove_trailing_cr (lines[i]);
        /* add to playlist */
	if (ret >= 0)
          playlist_add_mrl (strstr(lines[i], "mms://"), ins_pos);
	else
	  ret = playlist_add_mrl (strstr(lines[i], "mms://"), ins_pos);
      }
    }
    return ret;
  }

  /* check ASX */
  if (strcmp(xml_tree->name, "ASX") == 0) {
    logprintf("playlist: ASX tag detected\n");

    /* check version */
    asx_version = xml_parser_get_property_int (xml_tree, "VERSION", 3);

    if (asx_version == 3) {

      logprintf("playlist: VERSION 3.0 detected\n");

      /* play each entry */
      foreach_glist (current_node_l1, xml_tree->child)
      {
	/* entry */
	if (strcmp(current_node_l1->name, "ENTRY") == 0) {
	  char *title = NULL;

	  logprintf("playlist: ENTRY detected\n");

	  /* play the first ref which is playable */
	  current_node_l2 = current_node_l1->child;
	  while (current_node_l2) {
	    if (strcmp(current_node_l2->name, "REF") == 0) {
	      logprintf("playlist: REF detected\n");
	      /* find href property */
	      property = current_node_l2->props;
	      while ((property) && (strcmp(property->name, "HREF"))) {
		property = property->next;
	      }
	      if (property) {
		play_item_t *item;

		logprintf("playlist: HREF property detected\n"
			  "playlist: found an mrl: %s\n", property->value);

		item = play_item_new (title, property->value, 0);

		if (ret >= 0)
		  playlist_add (item, ins_pos);
		else
		  ret = playlist_add (item, ins_pos);
		
		/* jump to next entry */
		current_node_l2 = NULL;
		
#if 0 /* FIXME */
		/* try to play REF */
		if (demux_asx_play(this, property->value) == 1) {
		  /* jump to next entry */
		  printf("playlist: play next entry or entryref\n");
		  current_node_l2 = NULL;
		} else {
		  /* try next REF */
		  printf("playlist: try next REF\n");
		}
		
#endif
		
	      } else {
		fprintf(stderr, _("playlist: no HREF property\n"));
	      } /* if (property) */
	      
	    } else if (strcmp(current_node_l2->name, "TITLE") == 0) {
	      title = current_node_l2->data;
	    } else {
	      logprintf("playlist: unknown tag %s detected\n", 
			current_node_l2->name);
	    } /* end if (strcmp(current_node_l2->name, "REF") == 0) */
	    if (current_node_l2) {
	      current_node_l2 = current_node_l2->next;
	    }
	  } /* end while */
	} else {
	  
	  /* entryref */
	  if (strcmp(current_node_l1->name, "ENTRYREF") == 0) {
	    logprintf("playlist: ENTRYREF detected\n");
	    property = current_node_l1->props;
	    while ((property) && (strcmp(property->name, "HREF"))) {
	      property = property->next;
	    }
	    if (property) {
	      logprintf ("playlist: HREF property detected\n"
			 "playlist: found an MRL: %s\n", property->value);

	      if (ret >= 0)
		playlist_add_mrl (property->value, ins_pos);
	      else
		ret = playlist_add_mrl (property->value, ins_pos); 
	      
	      /* jump to next entry */
	      current_node_l2 = NULL;
	      
#if 0 /* FIXME */
	      
	      /* try to play HREF */
	      if (demux_asx_play(this, property->value) == 1) {
		/* jump to next entry */
		printf("playlist: play next entry or entryref\n");
		current_node_l1 = NULL;
	      } else {
		/* try next REF */
		printf("playlist: try next REF\n");
	      }
#endif
	    } else {
	      fprintf (stderr, _("playlist: no HREF property\n"));
	    } /* if (property) */
	  } else {
	    
	    /* title */
	    if (strcmp(current_node_l1->name, "TITLE") == 0) {
	      logprintf("playlist: TITLE detected\n");
	      /* change title */
	    } else {
	      fprintf(stderr, _("playlist: unknown tag %s detected\n"),
		     current_node_l1->name);
	    }
	  }
	}
      }
      
    } else {
      fprintf(stderr, _("playlist: sorry, ASX version %d not implemented yet\n"),
	     asx_version);
    }
  } else {
    fprintf(stderr, _("playlist: no ASX tag\n"));
  }
  return ret;
}

static int playlist_add_playlist_mrl (const char *mrl, gint ins_pos)
{
  const char *extension = strrchr(mrl, '.');
  if (extension)
  {
    if (!strncasecmp (extension, ".pls", 4))
      return playlist_load_pls (mrl, ins_pos);

    if (!strncasecmp (extension, ".m3u", 4))
      return playlist_load_m3u (mrl, ins_pos);

    if (!strcasecmp (extension, ".ra") ||
	!strcasecmp (extension, ".rm") ||
	!strncasecmp (extension, ".ram", 4) ||
	!strncasecmp (extension, ".rpm", 4) ||
	!strncasecmp (extension, ".lsc", 4) ||
	!strncasecmp (extension, ".pl", 3))
      return playlist_load_ram (mrl, ins_pos);

    if (!strncasecmp (extension, ".asx", 4))
      return playlist_load_asx (mrl, ins_pos);

    if (!strncasecmp (extension, ".smi", 4))
      return playlist_load_smil (mrl, ins_pos);
  }

  return -2; /* not recognised */
}

static int playlist_add_mrl_internal (const char *mrl, gint ins_pos)
{
  /* Helper function. Call only from playlist_add_mrl(). */
  int ret;
  const char *tmp;

  /*
   * is this mrl in fact a simple filename?
   */

  tmp = mrl;
  while (isalpha (*tmp))
    ++tmp;
  if (tmp[0] != ':' || tmp[1] != '/')
  {
    char *uri;

    /* is it a playlist file? */
    ret = playlist_add_playlist_mrl (mrl, ins_pos);
    if (ret != -2)
      return ret;

    if (!strncasecmp (mrl, "cdda:", 5))
      return playlist_add_int (mrl, ins_pos);

    if (mrl[0]!='/') {

      /* make filename an absolute pathname */

      char  buf[PATH_MAX];
      char *pathname;
      int   l1, l2;

      getcwd (buf, sizeof (buf));

      l1 = strlen(buf);
      l2 = strlen(mrl);

      pathname = g_malloc (l1+l2+2);
      memcpy (pathname, buf, l1);
      pathname[l1]='/';
      memcpy (&pathname[l1+1], mrl, l2);
      pathname[l1+l2+1]=0;

      uri = g_filename_to_uri (pathname, NULL, NULL);
      free (pathname);
    }
    else
      uri = g_filename_to_uri (mrl, NULL, NULL);

    ret = playlist_add_int (uri, ins_pos);
    free (uri);
    return ret;
  }

  if (!strncasecmp (mrl, "mmshttp://", 10))
  {
    logprintf ("playlist: adding regular mrl from mmshttp %s\n", mrl);
    mrl += 3;
    goto handle_http;
  }

  if (!strncasecmp (mrl, "http://", 7)) {
    /*
     * could still be a pls/asx playlist, but if so we
     * need to download it first
     */

    handle_http:
    /* is it a playlist file? */
    ret = playlist_add_playlist_mrl (mrl, ins_pos);
    if (ret != -2)
      return ret;

    return playlist_add_int (mrl, ins_pos);
  }

  logprintf ("playlist: adding regular mrl %s\n", mrl);
  return playlist_add_int (mrl, ins_pos);
}

int playlist_add_mrl (const char *mrl, gint ins_pos /* -1 => append */)
{
  static GSList *queued = NULL;
  int ret;

  /* printf ("playlist_add called >%s<\n", mrl); */

  /* handle recursive playlists */
  if (g_slist_find_custom (queued, mrl, (GCompareFunc) strcmp))
  {
    fprintf (stderr, _("recursion in playlist: %s\n"), mrl);
    return 0;
  }

  queued = g_slist_prepend (queued, (char *)mrl);
  ret = playlist_add_mrl_internal (mrl, ins_pos);
  queued = g_slist_remove (queued, mrl);

  return ret;
}

void playlist_play (int list_pos)
{
  play_item_t *item = playlist_get_item (list_pos);
  if (!item) {
    playlist_logo ();
    return;
  }  

  playlist_play_from (list_pos, 0, item->start_time); 
}

static void set_mark_play_item (gboolean state)
{
  if (cur_list_pos >= 0 && cur_list_pos < playlist_size ())
  {
    GtkTreeIter iter;
    gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (pl_store), &iter, NULL,
				   cur_list_pos);
    gtk_list_store_set (pl_store, &iter, 3, state ? "•" : "", -1);
  }
}

void playlist_logo (void)
{
  unmark_play_item ();
  cur_item_dispose ();
  logo_mode = 1;
  player_launch (NULL, logo_mrl, 0, 0);
}

void playlist_show (void) {

  if (is_visible) {
    is_visible = FALSE;
    gtk_widget_hide (dlg);
  } else {
    is_visible = TRUE;
    window_show (dlg, NULL);
    gtk_tree_view_columns_autosize (GTK_TREE_VIEW(tree_view));
  }
}

int playlist_showing_logo (void)
{
  return logo_mode;
}

static gboolean row_activated_lcb (GtkWidget *widget, GtkTreePath *path,
				   GtkTreeViewColumn *col, gpointer data)
{
  GtkTreeIter iter;

  if (gtk_tree_selection_get_selected (sel, NULL, &iter))
    playlist_play (gtk_tree_path_get_indices (path)[0]);

  return FALSE;
}

static int has_been_played (int pos) {
  return playlist_get_item (pos)->played;
}

static void play_next (void) {

  int is_repeat, is_random;

  is_repeat = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (repeat_button));
  is_random = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (random_button));

  unmark_play_item ();
  ref_list_pos = -1;

  if (is_random) {

    int count;

    cur_list_pos = rand() % playlist_size();

    count = 0;
    while ( (count<playlist_size()) && has_been_played (cur_list_pos) ) {
      cur_list_pos = (cur_list_pos + 1) % playlist_size();
      
      count++;
    }

    if (has_been_played (cur_list_pos))
      cur_list_pos = playlist_size()+1;

  } else {
    cur_list_pos++;
  }

  if ( is_repeat && (cur_list_pos>=playlist_size())) {
    if (is_random) {
      int i;

      /* reset all played entries */

      for (i=0; i<playlist_size(); i++) {
	play_item_t *item;
	item = playlist_get_item (i);
	item->played = 0;
      }

      cur_list_pos = rand() % playlist_size();
      
    } else {
      cur_list_pos = 0; 
    }
  }

  if (cur_list_pos<playlist_size()) {
    play_item_t *item;
    item = playlist_get_item (cur_list_pos);
    item->played = 1;
    mark_play_item ();
    playlist_play (cur_list_pos);
  } else {
    cur_list_pos = 0;
    playlist_logo ();
  }
}

static void xine_event_cb (void *user_data, const xine_event_t *event) {

  /*
  printf ("playlist: got an event %d\n", event->type); 
  */
  static gboolean mrl_ext = FALSE;

  switch (event->type) {
  case XINE_EVENT_UI_PLAYBACK_FINISHED:
    gdk_threads_enter();
    if (!logo_mode)
      play_next ();
    gdk_threads_leave();
    break;

  case XINE_EVENT_PROGRESS:
    {
      xine_progress_data_t *prg = event->data;

      gdk_threads_enter();
      if (prg->percent>99)
	infobar_show_metadata (infobars);
      else {
	struct timeval  tv;
	int             age;

	/* filter out old events */
	gettimeofday (&tv, NULL);
	age = abs (tv.tv_sec - event->tv.tv_sec);
	if (age>0) {
	  gdk_threads_leave();
	  break;
	}

	gxineinfo_update_line (infobars, 0, "%s %d%%", prg->description,
			       prg->percent);
      }
      gdk_flush();
      gdk_threads_leave();
    }
    break;

  case XINE_EVENT_UI_SET_TITLE:
  case XINE_EVENT_FRAME_FORMAT_CHANGE: /* in case of DVB etc. */
    player_update_metadata ();
    break;

#ifndef XINE_EVENT_MRL_REFERENCE_EXT
/* present in 1.1.0 but not 1.0.2 */
#define XINE_EVENT_MRL_REFERENCE_EXT 13
typedef struct {
  int alternative, start_time, duration;
  const char mrl[1];
} xine_mrl_reference_data_ext_t;
#endif
  case XINE_EVENT_MRL_REFERENCE_EXT:
    {
      xine_mrl_reference_data_ext_t *ref = event->data;
      const char *title = ref->mrl + strlen (ref->mrl) + 1;

      mrl_ext = TRUE;

      logprintf ("playlist: got a reference event (ref MRL: %s) (title: %s)\n",
		 ref->mrl, title);

      if (ref->alternative == 0)
      {
	play_item_t *item = play_item_new (*title ? title : NULL, ref->mrl,
					   ref->start_time);
	gdk_threads_enter ();

	if (ref_list_pos == -1)
	{
	  GtkTreeIter iter;
	  ref_list_pos = cur_list_pos + 1;
	  playlist_add (item, ref_list_pos);
	  /* remove the item which triggered this reference event */
	  gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (pl_store), &iter,
					 NULL, cur_list_pos--);
	  gtk_list_store_remove (pl_store, &iter);
	}
	else
	  playlist_add (item, ++ref_list_pos);

	check_list_empty ();
	gdk_threads_leave ();
      }
    }
    break;
  case XINE_EVENT_MRL_REFERENCE:
    if (!mrl_ext)
    {
      xine_mrl_reference_data_t *ref = (xine_mrl_reference_data_t *) event->data;

      logprintf ("playlist: got a reference event (ref MRL: %s)\n", ref->mrl);

      if (ref->alternative == 0)
      {
	play_item_t *item = play_item_new (NULL, ref->mrl, 0);
	gdk_threads_enter ();

	if (ref_list_pos == -1)
	{
	  GtkTreeIter iter;
	  ref_list_pos = cur_list_pos + 1;
	  playlist_add (item, ref_list_pos);
	  /* remove the item which triggered this reference event */
	  gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (pl_store), &iter,
					 NULL, cur_list_pos--);
	  gtk_list_store_remove (pl_store, &iter);
	}
	else
	  playlist_add (item, ++ref_list_pos);

	check_list_empty ();
	gdk_threads_leave ();
      }
    }
    break;

  case XINE_EVENT_UI_MESSAGE:
    {
      xine_ui_message_data_t *data = (xine_ui_message_data_t *) event->data;
      typeof (display_info) *msg;

      gdk_threads_enter();

      switch (data->type)
      {
      default:
        puts ("xine_event_cb: unknown XINE_MSG_*");
      case XINE_MSG_AUDIO_OUT_UNAVAILABLE:
      case XINE_MSG_ENCRYPTED_SOURCE:
        msg = display_info;
        break;

      case XINE_MSG_GENERAL_WARNING:
      case XINE_MSG_NETWORK_UNREACHABLE:
      case XINE_MSG_UNKNOWN_HOST:
        msg = display_warning;
        break;

      case XINE_MSG_READ_ERROR:
      case XINE_MSG_LIBRARY_LOAD_ERROR:

      case XINE_MSG_CONNECTION_REFUSED:
      case XINE_MSG_FILE_NOT_FOUND:
      case XINE_MSG_PERMISSION_ERROR:
      case XINE_MSG_SECURITY:
      case XINE_MSG_UNKNOWN_DEVICE:
        msg = display_error;
        break;
      }

      /* TODO: provide customized messages, hints... */
      if (data->num_parameters) {
        int i;
	const char *param = data->messages + strlen (data->messages) + 1;
	char *txt = strdup (data->messages);
	for (i = 0; i < data->num_parameters; ++i)
	{
	  asreprintf (&txt, "%s %s", txt, param);
	  param += strlen (param) + 1;
	}
	msg (FROM_XINE, NULL, "%s", txt);
	free (txt);
      }
      else
        msg (FROM_XINE, NULL, "%s", data->messages);
      gdk_threads_leave();
    }
    break;

  case XINE_EVENT_AUDIO_LEVEL:
    {
      xine_audio_level_data_t *data = event->data;
      gdk_threads_enter ();
      ui_set_control_adjustment (Control_VOLUME,
				 (data->left + data->right) / 2);
      ui_set_control_button (Control_MUTE, !data->mute);
      gdk_threads_leave ();
    }
    break;
  }
}

void playlist_check_set_title (void)
{
  const char *title;
  if ((title = xine_get_meta_info (stream, XINE_META_INFO_TITLE)))
  {
    GtkTreeIter iter;
    play_item_t *play_item;
    GValue v = {0};

    gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (pl_store), &iter,
				   NULL, cur_list_pos);

    gtk_tree_model_get_value (GTK_TREE_MODEL (pl_store), &iter, 2, &v);
    play_item = g_value_peek_pointer (&v);

    if (play_item->untitled)
    {
      free (play_item->title);
      play_item->title = strdup (title);
      play_item->untitled = FALSE;
      gtk_list_store_set (pl_store, &iter, 0, play_item->title, -1);
    }

    CUR_ITEM_LOCK ();
    if (cur_item && cur_item->untitled)
    {
      free (cur_item->title);
      cur_item->title = strdup (title);
      cur_item->untitled = FALSE;
    }
    CUR_ITEM_UNLOCK ();

    g_value_unset (&v);
  }
}

/*
 * drag and drop
 * FIXME: should subclass GtkListStore
 */

static void drop_cb (GtkTreeView	*widget,
		     GdkDragContext     *context,
		     gint                x,
		     gint                y,
		     GtkSelectionData   *data,
		     guint               info,
		     guint		 time)
{
  gint ins_pos = treeview_get_drop_index (widget, context, x, y);

  logprintf ("drag_drop: drop callback, length=%d, format=%d, pos=%d,%d, insert=%d\n",
	     data->length, data->format, x, y, ins_pos);

  if (data->format == 8)
  {
    /* Drag is from an external source */
    dnd_add_mrls (data, &cur_list_pos, ins_pos);
    gtk_drag_finish (context, TRUE, FALSE, time);
    return;
  }

  if (drag_mrl.index == -1)
  {
    /* below here, we're only interested in intra-app drags */
    gtk_drag_finish (context, FALSE, FALSE, time);
    return;
  }

  if (drag_mrl.model == (GtkTreeModel *)pl_store)
  {
    /* Moving a playlist item */
    GtkTreeIter from, to;

    if (drag_mrl.index == ins_pos)
    {
      /* no need to move the item */
      gtk_drag_finish (context, TRUE, FALSE, time);
      return;
    }

    gtk_tree_model_iter_nth_child (drag_mrl.model, &from, NULL, drag_mrl.index);
    if (ins_pos == -1)
    {
      gtk_list_store_move_before (pl_store, &from, NULL);
      ins_pos = gtk_tree_model_iter_n_children (drag_mrl.model, NULL);
    }
    else
    {
      gtk_tree_model_iter_nth_child (drag_mrl.model, &to, NULL, ins_pos);
      gtk_list_store_move_before (pl_store, &from, &to);
    }

    /* adjust index of currently-being-played item if necessary */
    if (cur_list_pos != -1)
    {
      if (cur_list_pos == drag_mrl.index)
	cur_list_pos = ins_pos;
      if (drag_mrl.index < ins_pos && drag_mrl.index <= cur_list_pos
	  && cur_list_pos <= ins_pos)
	--cur_list_pos;
      if (drag_mrl.index > ins_pos && drag_mrl.index >= cur_list_pos
	  && cur_list_pos >= ins_pos)
	++cur_list_pos;
    }

    gtk_drag_finish (context, TRUE, FALSE, time);
    return;
  }

  gtk_drag_finish (context, FALSE, FALSE, time);
}

/*
 * js functions
 */

static JSBool js_playlist_get_item (JSContext *cx, JSObject *obj, uintN argc, 
				    jsval *argv, jsval *rval)
{
  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */
  se_log_fncall ("playlist_get_item");
  *rval = INT_TO_JSVAL (playlist_get_list_pos());
  return JS_TRUE;
}

static JSBool js_playlist_clear (JSContext *cx, JSObject *obj, uintN argc, 
				 jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */

  se_log_fncall_checkinit ("playlist_clear");

  playlist_clear ();

  return JS_TRUE;
}

static JSBool js_playlist_add (JSContext *cx, JSObject *obj, uintN argc, 
			       jsval *argv, jsval *rval) {

  se_t     *se = (se_t *) JS_GetContextPrivate(cx);
  JSString *str;
  char     *mrl;
  int       item;

  se_log_fncall_checkinit ("playlist_add");

  se_argc_check (1, "playlist_add");
  se_arg_is_string (0, "playlist_add");

  str = JS_ValueToString (cx, argv[0]);
  mrl = JS_GetStringBytes (str);

  logprintf ("playlist_add: MRL=%s\n", mrl);
  item = playlist_add_mrl (mrl, -1);

  *rval = INT_TO_JSVAL (item);

  return JS_TRUE;
}

static JSBool js_playlist_play (JSContext *cx, JSObject *obj, uintN argc, 
				jsval *argv, jsval *rval) {

  se_t *se = (se_t *) JS_GetContextPrivate(cx);
  int   item;

  se_log_fncall_checkinit ("playlist_play");

  se_argc_check (1, "playlist_play");
  se_arg_is_int (0, "playlist_play");

  JS_ValueToInt32 (cx, argv[0], &item);

  playlist_play (item);
  
  return JS_TRUE;
}

static JSBool js_playlist_show (JSContext *cx, JSObject *obj, uintN argc, 
				jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */

  se_log_fncall_checkinit ("playlist_show");

  playlist_show ();
  
  return JS_TRUE;
}

void playlist_play_from (int list_pos, int pos, int pos_time)
{
  /* int          err; */
  GtkTreePath *path;
  char         indices[10];
  play_item_t *play_item;

  logo_mode = 0;

  unmark_play_item ();
  cur_item_dispose ();

  list_pos = playlist_clip (list_pos);
  play_item = playlist_get_item (list_pos);

  if (!play_item) {
    cur_list_pos = 0;
    playlist_logo ();
    return;
  }

  play_item_play (play_item, pos, pos_time);
  
  snprintf (indices, 10, "%d", list_pos);
  path = gtk_tree_path_new_from_string (indices);
  gtk_tree_view_set_cursor (GTK_TREE_VIEW (tree_view), 
			    path, NULL, FALSE);
  gtk_tree_path_free (path);

  cur_list_pos = list_pos;
  ref_list_pos = -1;

  mark_play_item ();
}

static const GtkActionEntry buttons_data[] = {
  { "open",	GTK_STOCK_OPEN,		NULL,			NULL,		N_("Open a playlist file"),			open_cb },
  { "saveas",	GTK_STOCK_SAVE_AS,	NULL,			NULL,		N_("Save the playlist file"),			save_as_cb },
  { "clear",	GTK_STOCK_CLEAR,	NULL,			NULL,		N_("Clear the playlist"),			clear_cb },
  { "fromfile",	GTK_STOCK_NEW,		NULL,			NULL,		N_("Add files to the playlist"),		add_cb },
  { "edit",	GTK_STOCK_EDIT,		NULL,			"<Control>E",	N_("Edit the selected item"),			edit_cb },
  { "delete",	GTK_STOCK_DELETE,	NULL,			"Delete",	N_("Delete the selected item"),			del_cb },
  { "copy",	GTK_STOCK_COPY,		NULL,			NULL,		NULL,						copy_cb },
  { "cut",	GTK_STOCK_CUT,		NULL,			NULL,		NULL,						cut_cb },
  { "paste",	GTK_STOCK_PASTE,	NULL,			NULL,		NULL,						paste_cb },
  { "mm",	GXINE_MEDIA_MARK,	N_("Add to _media marks"), "<Control>M",N_("Add the selected item to the media marks list"), make_mediamark_cb },
  { "prev",	GTK_STOCK_MEDIA_PREVIOUS, NULL,			NULL,		N_("Play previous item"),			playlist_prev_cb },
  { "next",	GTK_STOCK_MEDIA_NEXT,	NULL,			NULL,		N_("Play next item"),				playlist_next_cb },
};

static const char buttons_structure[] =
  "<ui>\n"
    "<toolbar>\n"
      "<toolitem action='open' />\n"
      "<toolitem action='saveas' />\n"
      "<toolitem action='clear' />\n"
      "<separator />\n"
      "<toolitem action='fromfile' />\n"
      "<toolitem action='edit' />\n"
      "<toolitem action='delete' />\n"
      "<toolitem action='copy' />\n"
      "<toolitem action='cut' />\n"
      "<toolitem action='paste' />\n"
      "<toolitem action='mm' />\n"
      "<separator />\n"
      "<toolitem action='prev' />\n"
      "<toolitem action='next' />\n"
    "</toolbar>\n"
  "</ui>";

static void sel_changed_cb (GtkTreeSelection *sel, gpointer data)
{
  static const char *const items[] = {
    "edit", "delete", "mm", "copy", "cut", NULL
  };
  ui_mark_active (pl_ui, items, gtk_tree_selection_get_selected (sel, NULL, NULL));
}

static void check_list_empty (void)
{
  static const char *const items[] = {
    "saveas", "clear", "prev", "next", NULL
  };
  ui_mark_active (pl_ui, items,
		  gtk_tree_model_iter_n_children (GTK_TREE_MODEL (pl_store),
						  NULL));
}

void playlist_init (void)
{
  GtkWidget *hbox, *scrolled_window;
  GtkBox *vbox;
  GtkCellRenderer      *cell;
  GtkTreeViewColumn    *column;
  GError *error;

  is_visible     = 0;
  cur_list_pos   = -3; /* this will be clipped to 0 when Play is pressed */

  srand (time (NULL));

  /*
   * window
   */

  dlg = gtk_dialog_new_with_buttons (_("Edit playlist..."), NULL, 0,
				     GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
				     NULL);
  gtk_window_set_default_size (GTK_WINDOW (dlg), 500, 400);
  g_object_connect (G_OBJECT (dlg),
	"signal::delete-event", G_CALLBACK (close_cb), NULL,
	"signal::response", G_CALLBACK(close_cb), NULL,
	NULL);

  vbox = GTK_BOX (GTK_DIALOG (dlg)->vbox);

  /*
   * init tree store & view
   */

  pl_store = gtk_list_store_new (4, G_TYPE_STRING, G_TYPE_STRING,
				 G_TYPE_POINTER, G_TYPE_STRING);

  tree_view = gtk_tree_view_new_with_model (GTK_TREE_MODEL(pl_store));  
  gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (tree_view), TRUE);
  g_signal_connect (G_OBJECT(tree_view), "row-activated",
		    G_CALLBACK(row_activated_lcb), NULL);

  sel = gtk_tree_view_get_selection (GTK_TREE_VIEW (tree_view));
  g_signal_connect (G_OBJECT (sel), "changed",
		    G_CALLBACK (sel_changed_cb), NULL);

  /*
   * buttons
   */

  pl_ui = ui_create_manager ("file", dlg);
  gtk_action_group_add_actions (ui_get_action_group (pl_ui), buttons_data,
				G_N_ELEMENTS (buttons_data), tree_view);
  gtk_ui_manager_add_ui_from_string (pl_ui, buttons_structure, -1, &error);
  gtk_box_pack_start (vbox, gtk_ui_manager_get_widget (pl_ui, "/toolbar"),
		      FALSE, FALSE, 0);
  gtk_action_group_connect_accelerators (ui_get_action_group (pl_ui));

  sel_changed_cb (sel, NULL);

  /*
   * install tree view
   */

  cell = gtk_cell_renderer_text_new ();
  column = gtk_tree_view_column_new_with_attributes (" ", cell,
						     "text", 3, NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (tree_view),
			       GTK_TREE_VIEW_COLUMN (column));

  cell = gtk_cell_renderer_text_new ();
  g_object_set (G_OBJECT(cell),
		"ellipsize", 2, "width_chars", 16, NULL); /* foo...bar */
  column = gtk_tree_view_column_new_with_attributes (_("Title"), cell,
						     "text", 0, NULL);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN(column), TRUE);
  gtk_tree_view_append_column (GTK_TREE_VIEW (tree_view),
			       GTK_TREE_VIEW_COLUMN (column));

  cell = gtk_cell_renderer_text_new ();
  g_object_set (G_OBJECT(cell),
		"ellipsize", 1, "width_chars", 16, NULL); /* ...bar */
  column = gtk_tree_view_column_new_with_attributes (_("MRL"), cell,
						     "text", 1, NULL);
  gtk_tree_view_column_set_sizing (GTK_TREE_VIEW_COLUMN(column),
				   GTK_TREE_VIEW_COLUMN_AUTOSIZE);
  gtk_tree_view_append_column (GTK_TREE_VIEW (tree_view),
			       GTK_TREE_VIEW_COLUMN (column));

  treeview_drag_drop_setup (GTK_TREE_VIEW(tree_view), drop_cb);

  gtk_widget_show (tree_view);

  scrolled_window = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
				  GTK_POLICY_AUTOMATIC, 
				  GTK_POLICY_AUTOMATIC);
  gtk_container_add (GTK_CONTAINER (scrolled_window), tree_view);

  gtk_box_pack_start (vbox, scrolled_window, TRUE, TRUE, 2);

  /*
   * checkboxes (repeat, random)
   */

  hbox = gtk_hbox_new (0,2);
  
  repeat_button = gtk_check_button_new_with_mnemonic (_("_Repeat"));
  gtk_box_pack_start (GTK_BOX (hbox), repeat_button, FALSE, FALSE, 2);

  random_button = gtk_check_button_new_with_mnemonic (_("_Random"));
  gtk_box_pack_start (GTK_BOX (hbox), random_button, FALSE, FALSE, 2);

  gtk_box_pack_start (vbox, hbox, FALSE, FALSE, 2);

  sprintf (logo_mrl, "%s/logo.mpv", logodir);
  logo_mode = 1;

  playlist_load (NULL);

  /*
   * event handling
   */
  xine_event_create_listener_thread (xine_event_new_queue (stream),
				     xine_event_cb, NULL);

  {
    static const se_f_def_t defs[] = {
      { "playlist_show", js_playlist_show, 0, 0,
	SE_GROUP_DIALOGUE, NULL, NULL },
      { "playlist_clear", js_playlist_clear, 0, 0,
	SE_GROUP_PLAYLIST, NULL, NULL },
      { "playlist_add", js_playlist_add, 0, 0,
	SE_GROUP_PLAYLIST, N_("MRL"), NULL },
      { "playlist_play", js_playlist_play, 0, 0,
	SE_GROUP_PLAYLIST, N_("int"), N_("playlist entry number") },
      { "playlist_get_item", js_playlist_get_item, 0, 0,
	SE_GROUP_PLAYLIST, NULL, NULL },
      { NULL }
    };
    se_defuns (gse, NULL, defs);
  }
}
