/*
 * Hardware Information, version 0.3.2
 * Copyright (C) 2003 Leandro Pereira <leandro@linuxmag.com.br>
 *
 * May be modified and/or distributed under the terms of GNU GPL version 2.
 *
 */

#include "hardinfo.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>

#include "pixmaps/pci.xpm"
#include "pixmaps/usb.xpm"
#include "pixmaps/hdd.xpm"
#include "pixmaps/gen_connector.xpm"
#include "pixmaps/scsi.xpm"
#include "pixmaps/processor.xpm"

#include "computer.h"
#include "status.h"
#include "stock.h"

GenericDevice *generic_devices = NULL;

void hi_enable_details_button(GtkCTree * tree, GList * node,
			 gint column, gpointer user_data);
void hi_enable_details_button_real(MainWindow *mainwindow, GenericDevice *dev);
void hi_disable_details_button(GtkCTree * tree, GList * node,
			 gint column, gpointer user_data);
void hi_scan_all(MainWindow * mainwindow);

void
main_window_refresh(GtkWidget * widget, gpointer data)
{
	MainWindow *mainwindow = (MainWindow *) data;

	if (!mainwindow)
		return;

	hi_disable_details_button(NULL, NULL, 0, mainwindow);

	memory_update(mainwindow);
	uptime_update(mainwindow);
	net_update(mainwindow);

	hi_scan_all(mainwindow);
}

static void
about_window_create(void)
{
	GtkAbout *about;
	const gchar *sysinfo = HOSTNAME " (" PLATFORM KERNEL ")";
	const gchar *authors[] = {
		">Written by:",
		"Leandro A. F. Pereira (leandro@linuxmag.com.br)",
		">Disclaimer:",
		"This is free software; you can modify and/or distribute it",
		"under the terms of GNU GPL version 2. See http://www.fsf.org/",
		"for more information.",
		">Compiled on:",
		sysinfo,
		NULL
	};

	about = gtk_about_new("HardInfo", VERSION,
			      _("System information tool for GNU/Linux.\n"),
			      authors, IMG_PREFIX "logo.png");

}

MainWindow *
main_window_create(void)
{
	GtkWidget *window, *mbox, *vbox, *ctree, *scroll;
	GtkWidget *notebook, *label, *hbox, *btn, *hbbox;
	MainWindow *mainwindow;

	mainwindow = g_new0(MainWindow, 1);

	mainwindow->det_window = detail_window_new();

	window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_container_set_border_width(GTK_CONTAINER(window), 4);
	gtk_window_set_title(GTK_WINDOW(window), _("System Information"));

	g_signal_connect(G_OBJECT(window), "delete-event", gtk_main_quit, NULL);
	gtk_window_set_icon_from_file(GTK_WINDOW(window), IMG_PREFIX "logo.png", NULL);

	mbox = gtk_vbox_new(FALSE, 5);
	gtk_widget_show(mbox);
	gtk_container_add(GTK_CONTAINER(window), mbox);

	notebook = gtk_notebook_new();
	gtk_box_pack_start(GTK_BOX(mbox), notebook, TRUE, TRUE, 0);

	/*
	 * Computer tab
	 */
	vbox = gtk_vbox_new(FALSE, 5);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 4);
	gtk_widget_show(vbox);
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox,
				 gtk_label_new(_("Computer")));

	label = gtk_label_new(_("<b><big>Operating System</big></b>"));
	gtk_label_set_use_markup(GTK_LABEL(label), TRUE);
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
	gtk_widget_show(label);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), os_get_widget(mainwindow),
			   FALSE, FALSE, 0);

	label = gtk_label_new(_("<b><big>Processor</big></b>"));
	gtk_label_set_use_markup(GTK_LABEL(label), TRUE);
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
	gtk_widget_show(label);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), processor_get_widget(), FALSE,
			   FALSE, 0);

	label = gtk_label_new(_("<b><big>Memory Usage</big></b>"));
	gtk_label_set_use_markup(GTK_LABEL(label), TRUE);
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
	gtk_widget_show(label);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), memory_get_widget(mainwindow),
			   FALSE, FALSE, 0);

	/*
	 * Devices tab
	 */
	vbox = gtk_vbox_new(FALSE, 5);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 4);
	gtk_widget_show(vbox);
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox,
				 gtk_label_new(_("Details")));

	hbbox = gtk_hbutton_box_new();
	gtk_container_set_border_width(GTK_CONTAINER(hbbox), 4);
	gtk_widget_show(hbbox);
	gtk_box_pack_end(GTK_BOX(vbox), hbbox, FALSE, FALSE, 0);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbbox), 6);
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbbox), GTK_BUTTONBOX_END);

	btn = gtk_button_new_from_stock(HI_DETAILS);
	g_signal_connect(G_OBJECT(btn), "clicked",
			 (GCallback) detail_window_show, mainwindow);
	gtk_widget_show(btn);
	gtk_box_pack_end(GTK_BOX(hbbox), btn, FALSE, FALSE, 0);
	mainwindow->details_button = btn;

	scroll = gtk_scrolled_window_new(NULL, NULL);
	gtk_box_pack_start(GTK_BOX(vbox), scroll, TRUE, TRUE, 0);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll),
				       GTK_POLICY_AUTOMATIC,
				       GTK_POLICY_AUTOMATIC);

	ctree = gtk_ctree_new(2, 0);
	gtk_container_add(GTK_CONTAINER(scroll), ctree);
	gtk_widget_set_usize(GTK_WIDGET(ctree), 400, 300);
	gtk_ctree_set_expander_style(GTK_CTREE(ctree),
				     GTK_CTREE_EXPANDER_TRIANGLE);
	gtk_ctree_set_line_style(GTK_CTREE(ctree), GTK_CTREE_LINES_NONE);
	gtk_clist_set_column_width(GTK_CLIST(ctree), 0, 32);
	gtk_clist_set_column_width(GTK_CLIST(ctree), 1, 32);
	gtk_clist_set_row_height(GTK_CLIST(ctree), 18);
	g_signal_connect(G_OBJECT(ctree), "tree-select-row",
			 (GCallback) hi_enable_details_button, mainwindow);
	g_signal_connect(G_OBJECT(ctree), "tree-unselect-row",
			 (GCallback) hi_disable_details_button, mainwindow);
			 
	/*
	 * Environment tab
	 */
	vbox = gtk_vbox_new(FALSE, 5);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 4);
	gtk_widget_show(vbox);
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox,
				 gtk_label_new(_("Environment")));

	label = gtk_label_new(_("<b><big>X-Window System</big></b>"));
	gtk_label_set_use_markup(GTK_LABEL(label), TRUE);
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
	gtk_widget_show(label);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
//	gtk_box_pack_start(GTK_BOX(vbox), x11_get_widget(mainwindow),
//			   FALSE, FALSE, 0);

	/*
	 * Network tab
	 */
	vbox = gtk_vbox_new(FALSE, 5);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 4);
	gtk_widget_show(vbox);
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox,
				 gtk_label_new(_("Network")));

	label = gtk_label_new(_("<b><big>Interfaces</big></b>"));
	gtk_label_set_use_markup(GTK_LABEL(label), TRUE);
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
	gtk_widget_show(label);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), net_get_widget(mainwindow),
			   TRUE, TRUE, 0);

	/*
	 * Buttons
	 */
	hbox = gtk_hbox_new(FALSE, 5);
	gtk_widget_show(hbox);
	gtk_box_pack_start(GTK_BOX(mbox), hbox, FALSE, FALSE, 0);
	
	hbbox = gtk_hbutton_box_new();
	gtk_container_set_border_width(GTK_CONTAINER(hbbox), 4);
	gtk_widget_show(hbbox);
	gtk_box_pack_start(GTK_BOX(hbox), hbbox, FALSE, FALSE, 0);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbbox), 6);
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbbox), GTK_BUTTONBOX_START);

	btn = gtk_button_new_from_stock(HI_ABOUT);
	g_signal_connect(G_OBJECT(btn), "clicked",
			 (GCallback) about_window_create, NULL);
	gtk_widget_show(btn);
	gtk_box_pack_start(GTK_BOX(hbbox), btn, FALSE, FALSE, 0);

	hbbox = gtk_hbutton_box_new();
	gtk_container_set_border_width(GTK_CONTAINER(hbbox), 4);
	gtk_widget_show(hbbox);
	gtk_box_pack_end(GTK_BOX(hbox), hbbox, FALSE, FALSE, 0);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbbox), 6);
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbbox), GTK_BUTTONBOX_END);

	btn = gtk_button_new_from_stock(GTK_STOCK_REFRESH);
	g_signal_connect(G_OBJECT(btn), "clicked",
			 (GCallback) main_window_refresh, mainwindow);
	gtk_widget_show(btn);
	gtk_box_pack_start(GTK_BOX(hbbox), btn, FALSE, FALSE, 0);

	btn = gtk_button_new_from_stock(GTK_STOCK_CLOSE);
	g_signal_connect(G_OBJECT(btn), "clicked", gtk_main_quit, NULL);
	gtk_widget_show(btn);
	gtk_box_pack_start(GTK_BOX(hbbox), btn, FALSE, FALSE, 0);

	gtk_widget_show_all(window);

	mainwindow->window = window;
	mainwindow->ctree = ctree;

	return mainwindow;
}

static GdkColor
gdk_color_from_rgb(gint r, gint g, gint b)
{
	GdkColormap *cmap = NULL;
	GdkColor color;

	if (!cmap) {
		cmap = gdk_colormap_get_system();
		gdk_colormap_ref(cmap);
	}

	color.red = r * 255;
	color.green = g * 255;
	color.blue = b * 255;

	gdk_color_alloc(cmap, &color);
	gdk_colormap_unref(cmap);

	return color;
}

GtkCTreeNode *
tree_group_new(MainWindow * mainwindow, const gchar * name, DeviceType type)
{
	GtkCTreeNode *node;
	const gchar *text[] = { NULL, name, NULL };
	GdkPixmap *pixmap = NULL;
	GdkBitmap *mask = NULL;
	GdkColor color;
	GdkColormap *colormap = NULL;

	if (!name)
		return NULL;
	if (type == NONE)
		return NULL;
	if (!mainwindow)
		return NULL;

	colormap = gtk_widget_get_colormap(mainwindow->window);

	switch (type) {
	case PARPORT:
	case SERIAL:
		pixmap = gdk_pixmap_colormap_create_from_xpm_d
		    (NULL, colormap, &mask, NULL, gen_connector_xpm);
		break;
	case V4L:
	case PCI:
	case ISAPnP:
		pixmap = gdk_pixmap_colormap_create_from_xpm_d
		    (NULL, colormap, &mask, NULL, pci_xpm);
		break;
	case SCSI:
		pixmap = gdk_pixmap_colormap_create_from_xpm_d
		    (NULL, colormap, &mask, NULL, scsi_xpm);
		break;
	case IDE:
		pixmap = gdk_pixmap_colormap_create_from_xpm_d
		    (NULL, colormap, &mask, NULL, hdd_xpm);
		break;
	case USB:
		pixmap = gdk_pixmap_colormap_create_from_xpm_d
		    (NULL, colormap, &mask, NULL, usb_xpm);
		break;
	case MODULE:
	case PROCESSOR:
		pixmap = gdk_pixmap_colormap_create_from_xpm_d
		    (NULL, colormap, &mask, NULL, processor_xpm);
		break;
	default:
		mask = pixmap = NULL;
		break;
	}

	node = gtk_ctree_insert_node(GTK_CTREE(mainwindow->ctree),
				     NULL, NULL, text, 0, pixmap, mask, pixmap,
				     mask, FALSE, TRUE);

	color = gdk_color_from_rgb(0xcc, 0xcc, 0xcc);
	gtk_ctree_node_set_background(GTK_CTREE(mainwindow->ctree),
				      node, &color);

	gdk_pixmap_unref(pixmap);
	gdk_bitmap_unref(mask);

	return node;
}

void
tree_insert_item(MainWindow * mainwindow, GtkCTreeNode * group, gchar * name,
		 gpointer data)
{
	GtkCTreeNode *node;
	gchar *text[] = { NULL, name, NULL };

	if (!mainwindow)
		return;
	if (!name)
		return;
	if (!group)
		return;

	node = gtk_ctree_insert_node(GTK_CTREE(mainwindow->ctree),
				     group, NULL,
				     text, 0,
				     NULL, NULL, NULL, NULL, FALSE, TRUE);
	gtk_ctree_node_set_row_data(GTK_CTREE(mainwindow->ctree), node, data);
}

void
hi_insert_generic(gpointer device, DeviceType type)
{
	GenericDevice *generic;

	generic = g_new0(GenericDevice, 1);

	generic->next = generic_devices;
	generic_devices = generic;

	generic_devices->device = device;
	generic_devices->type = type;
}

void
hi_disable_details_button(GtkCTree * tree, GList * node,
		    gint column, gpointer user_data)
{
	MainWindow *mainwindow = (MainWindow *) user_data;

	gtk_widget_set_sensitive(GTK_WIDGET(mainwindow->details_button), FALSE);
}

void hi_enable_details_button(GtkCTree * tree, GList * node,
		    gint column, gpointer user_data)
{
	MainWindow *mainwindow = (MainWindow *) user_data;

	gtk_widget_set_sensitive(GTK_WIDGET(mainwindow->details_button), TRUE);
}

void hi_show_device_info_real(MainWindow *mainwindow, GenericDevice *dev)
{
	if (!dev)
		return;

#define dev_info(type,name,func) \
	{ \
		type *name; \
		name = (type *)dev->device; \
		func(mainwindow, name); \
	} \
	break;

	switch (dev->type) {
	case MODULE:
		dev_info(ModInfo,   mod, hi_show_module_info);
	case PROCESSOR:
		dev_info(CPUDevice, cpu, hi_show_cpu_info);
	case PCI:
		dev_info(PCIDevice, pci, hi_show_pci_info);
	case ISAPnP:
		dev_info(ISADevice, isa, hi_show_isa_info);
	case USB:
		dev_info(USBDevice, usb, hi_show_usb_info);
	case SERIAL:
		dev_info(SerialDevice, serial, hi_show_serial_info);
	case PARPORT:
		dev_info(ParportDevice, parport, hi_show_parport_info);
	case IDE:
		dev_info(IDEDevice, ide, hi_show_ide_info);
	case SCSI:
		dev_info(SCSIDevice, scsi, hi_show_scsi_info);
	case V4L:
		dev_info(V4LDevice, v4l, hi_show_v4l_info);
	default:
		g_print("Showing info of device type %d not implemented yet.\n",
			dev->type);
		return;
		break;
	}
}

void
hi_scan_all(MainWindow * mainwindow)
{
	myStatus	*status;
	PCIDevice	*pci; 
	ISADevice	*isa; 
	IDEDevice	*ide;
	CPUDevice	*cpu;
	SCSIDevice	*scsi;
	V4LDevice	*v4l;
	ParportDevice	*pp;
	SerialDevice	*sd;
	GtkCTreeNode	*node;
	ModInfo		*mod;
	GenericDevice	*gd = generic_devices;
	gchar		*buf;

	status = my_status_new(_("Scanning Devices"), _("Scanning devices..."));

#define DEVICE_SCAN(name,ptr,func) \
		ptr = func(); \
		buf = g_strdup_printf(_("Scanning %s devices..."), name); \
		my_status_set_text(status, buf); \
		g_free(buf); \
		my_status_pulse(status)
	
	DEVICE_SCAN("CPU",	cpu,	computer_get_info);
	DEVICE_SCAN("PCI",	pci,	hi_scan_pci);
	DEVICE_SCAN("ISA PnP",	isa,	hi_scan_isapnp);
	DEVICE_SCAN("IDE",	ide,	hi_scan_ide);
	DEVICE_SCAN("SCSI",	scsi,	hi_scan_scsi);
	DEVICE_SCAN("V4L",	v4l,	hi_scan_v4l);
	DEVICE_SCAN("Parallel",	pp,	hi_scan_parport);
	DEVICE_SCAN("Serial",	sd,	hi_scan_serial);
	DEVICE_SCAN("modules",	mod,	hi_scan_modules);

	gtk_clist_freeze(GTK_CLIST(mainwindow->ctree));

	/*
	 * Free the generic_devices stuff and remove all device-related
	 * nodes.
	 */
	if (gd != NULL) {
		for (; gd != NULL; gd = gd->next) {
			g_free(gd->device);
			gtk_ctree_remove_node(GTK_CTREE(mainwindow->ctree),
					      gd->node);
			g_free(gd);
		}
		generic_devices = NULL;
	}
	
#define CHECK_INSERT(a,b,c,d) \
	if (a) { \
		node = tree_group_new(mainwindow, b, c); \
		for (; a != NULL; a = a->next) { \
			if (a->d) { \
				hi_insert_generic(a, c); \
				tree_insert_item(mainwindow, node, a->d, \
					generic_devices); \
			} \
		} \
	}


	/*
	 * Processor info
	 */
	node = tree_group_new(mainwindow, _("Processor"), PROCESSOR);
	hi_insert_generic(cpu, PROCESSOR);
	tree_insert_item(mainwindow, node, cpu->processor, generic_devices);

	CHECK_INSERT(pci, _("PCI Devices"), PCI, name);
	CHECK_INSERT(isa, _("ISA PnP Devices"), ISAPnP, card);

	/*
	 * USB is different since it's Plug'n'Pray, so we have a
	 * routine just for it :)
	 */
	usb_update(mainwindow);

	CHECK_INSERT(ide, _("ATA/IDE Block Devices"), IDE, model);
	CHECK_INSERT(scsi, _("SCSI Devices"), SCSI, model);
	CHECK_INSERT(v4l, _("Video for Linux"), V4L, name);
	CHECK_INSERT(sd, _("Communication Ports"), SERIAL, name);
	CHECK_INSERT(pp, _("Parallel Ports"), PARPORT, name);
 
 	CHECK_INSERT(mod, _("Kernel Modules"), MODULE, description);
 	
	gtk_clist_thaw(GTK_CLIST(mainwindow->ctree));

	my_status_destroy(status);
}

static void
usage(char *argv0)
{
	g_print("%s [--help] [--prefix <prefix>]\n", argv0);
	exit(1);
}

int
main(int argc, char **argv)
{
	MainWindow *mainwindow;
	gint i;

#ifdef ENABLE_NLS
	intl_init();
#endif

	g_print("HardInfo " VERSION);
	if (strstr(VERSION, "pre")) {
		g_print(" *** PRE-RELEASE ***");
	}
	g_print
	    ("\nCopyright (c) 2003 Leandro Pereira <leandro@linuxmag.com.br>\n\n");
	g_print(_
		("This is free software; you can modify and/or distribute it\n"));
	g_print(_
		("under the terms of GNU GPL version 2. See http://www.fsf.org/\n"));
	g_print(_("for more information.\n\n"));

	gtk_init(&argc, &argv);

	hi_stock_init();

	for (i = 1; i < argc; i++) {
		if (!strncmp(argv[i], "--help", 6) ||
		    !strncmp(argv[i], "-h", 2)) {
			usage(argv[0]);
		} else
		    if (!strncmp(argv[i], "--prefix", 8) ||
			!strncmp(argv[i], "-p", 2)) {
			i++;
			if (argv[i][0] == '-')
				usage(argv[0]);

			g_print("prefix = %s\n", argv[i]);
		}
	}

	mainwindow = main_window_create();
	main_window_refresh(NULL, mainwindow);

	gtk_main();

	return 0;
}
