module Main where	-- HatAnim main program

import Char               (toLower)
import LowLevel           (openHatFile,getBridgeValue,hatVersionNumber
                          ,FileNode(..),nil,peekTrace,getResult,getParentNode
                          ,getErrorLoc,getErrorMessage
                          ,getSrcRef,getDefnRef)
import VSExp              (VSExp(..),NodeExp(..),nodeExpForNode,findLambdas
                          ,limitNExpDepth,compressClosures
                          ,removeHorizontalCycles,removeVerticalCycles
                          ,limitDepth,hideFunction,unevalVSExp,partEvalVSExp
                          ,fullEvalVSExp,cutoffEvalVSExp,condEvalVSExp
                          ,limitDepth,vsExp2sExp,cutoffFindMinOffset)
import SExp               (SExp(..),prettySExp,notCondParens)
import HighlightStyle     (goto,cls,clearDown,clearUp,cleareol,highlightOff
                          ,highlight,Highlight(..),Colour(..)
                          ,enableScrollRegion,getTerminalSize
                          ,savePosition,restorePosition)
import System             (system,getArgs,getProgName,getEnv
                          ,exitWith,ExitCode(..))
import List               (isPrefixOf,isSuffixOf,group,groupBy)
import IO                 (hSetBuffering,BufferMode(..),stdin,stdout,stderr
                          ,hPutStrLn,hFlush)
import Char               (digitToInt)
import FFIExtensions      (withCString,showHex)
import CommonUI           (hatTrail,hatObserve,hatDetect,hatAnim,hatView
                          ,hatExplore
                          ,Options(..),initialOptions
                          ,OptionCmd(..),optionCmd,onOff,number
                          ,optionsUpdate,showOption,showOnOff)

main = do
    args    <- System.getArgs
    prog    <- System.getProgName
    hatfile <- case args of (f:_) -> return (rectify f)
                            _     -> do hPutStrLn stderr
                                                  ("hat-anim: no trace file")
                                        exitWith (ExitFailure 1)
    withCString prog (\p-> withCString hatfile (openHatFile p))
    errloc  <- getErrorLoc
    errmsg  <- getErrorMessage
    output  <- readOutputFile hatfile
    bridge  <- readBridgeFile
    let (start,end) = case args of
                       (_:s:e:_) -> (atoi s, atoi e)
                       (_:s:_)   -> (atoi s, 0)
                       _         -> (0,0)
{-    (style0,style1) <- catch (do home <- getEnv "HOME"
                                 f <- readFile (home++"/.hatanimrc")
                                 return (read f))
                             (\e-> return ( [Dim,Foreground Red]
                                          , [Bold,Foreground Magenta] ))-}
    hSetBuffering stdin NoBuffering
    hSetBuffering stdout NoBuffering
    System.system ("stty -icanon min 1 -echo")
    (columns,lines) <- getTerminalSize
    --putStr (show (nodeExpForNode (FileNode start)))
    {-putStr (show
             (removeHorizontalCycles
               (cutoffEvalVSExp (FileNode end)
                 (removeVerticalCycles
                   (findLambdas
                     (nodeExpForNode (FileNode start)))))))-}
    animLoop (columns,lines) 10 [] True [FileNode start] (FileNode start) ""

simpleFetchFunction :: FileNode -> VSExp
simpleFetchFunction
  = removeHorizontalCycles . (cutoffEvalVSExp (FileNode 804)) . fetchNode
fetchNode :: FileNode -> NodeExp
fetchNode = compressClosures . removeVerticalCycles . nodeExpForNode

animLoop :: (Int,Int) -> Int -> [String] -> Bool -> [FileNode] -> FileNode -> String -> IO ()
animLoop (cols,lines) depth hides colouringOn cutoffs maxCutoff com =
  animLoopRedraw (cols, lines) depth hides colouringOn cutoffs maxCutoff com True
  where
    animLoopRedraw :: (Int,Int) ->
                      Int ->
                      [String] ->
                      Bool ->
                      [FileNode] ->
                      FileNode ->
                      String ->
                      Bool -> IO ()
    animLoopRedraw (cols,lines) depth hides colouringOn cutoffs maxCutoff com fullDraw =
      do
        let
          animation :: String
          animation = if fullDraw then
                        interface (cols,lines)
                        ++ (goto 1 5)
                        ++ animText (cols,lines) depth hides colouringOn cutoffs
                      else ""
        putStr animation
        putStr (goto 1 lines ++ com ++ take (cols - length com) (repeat ' '))
        x <- getChar
        if x == '\n'
          then runCommand (cols,lines)
                          depth
                          hides
                          colouringOn
                          cutoffs
                          maxCutoff
                          com
          else if x == '\b' then
            if com == "" then
              animLoopRedraw (cols,lines)
                             depth
                             hides
                             colouringOn
                             (removeCutoff cutoffs)
                             (last (removeCutoff cutoffs))
                             ""
                             True
            else
              animLoopRedraw (cols,lines)
                             depth
                             hides
                             colouringOn
                             cutoffs
                             maxCutoff
                             (removeLast com)
                             False
          else
            animLoopRedraw (cols,lines)
                           depth
                           hides
                           colouringOn
                           cutoffs
                           maxCutoff
                           (com ++ [x])
                           False

animText :: (Int,Int) -> Int -> [String] -> Bool -> [FileNode] -> String
animText (cols,lines) depth hides colouringOn cutoffs =
  (force . unlines . (map indent)) text
  where
    text :: [String]
    text = fittingLines outputLines

    fittingLines :: [String] -> [String]
    fittingLines lns
      = drop ((length lns) - numLines) revLines
        where
          revLines = (reverse lns)
          numLines = countLines 0 lns
          countLines :: Int -> [String] -> Int
          countLines l [] = 0
          countLines l (y:ys)
            = if (takenLines y) > (lines - l - 6) then 0
              else (countLines (l + takenLines y) ys) + 1
          takenLines :: String -> Int
          takenLines = (+1) . length . (filter (=='\n'))

    outputLines :: [String]
    outputLines = map (prettyNode startNode) cutoffs
    
    prettyNode :: FileNode -> FileNode -> String
    prettyNode start end =
      prettySExp ""
                 (cols - 3)
                 Options {listSugar=True, showQual=False, colourBracks=colouringOn}
                 (notCondParens
                   (vsExp2sExp
                     (limitDepth depth
                       (foldr ((flip hideFunction)
                                ((condEvalVSExp (\x -> True)) . fetchNode))
                              (simpleFetchFunction start)
                              hides))))
      where
        simpleFetchFunction :: FileNode -> VSExp
        simpleFetchFunction
          = removeHorizontalCycles . (cutoffEvalVSExp end) . fetchNode
        fetchNode :: FileNode -> NodeExp
        fetchNode = compressClosures . removeVerticalCycles . nodeExpForNode

    startNode :: FileNode
    startNode = last cutoffs

force :: [a] -> [a]
-- force evaluation of whole list spine
force = reverse . reverse

indent :: String -> String
indent xs = "-> " ++ (indAux xs)
            where
              indAux :: String -> String
              indAux [] = []
              indAux ('\n':xs) = "\n   " ++ (indAux xs)
              indAux (x:xs) = x:(indAux xs)

runCommand :: (Int,Int) -> Int -> [String] -> Bool -> [FileNode] -> FileNode -> String -> IO ()
runCommand (cols,lines) depth hides colouringOn cutoffs maxCutoff com =
  getCommand command args (cols,lines) depth hides colouringOn cutoffs maxCutoff
  where
    (command, args) =
      case wrds of 
        []        -> ("", [])
        otherwise -> (head wrds, tail wrds)
    wrds :: [String]
    wrds = words com

getCommand :: String -> [String] ->
                        (Int,Int) ->
                        Int ->
                        [String] ->
                        Bool ->
                        [FileNode] ->
                        FileNode -> IO()
getCommand "" args (cols,lines) depth hides colouringOn cutoffs _ =
  let newCutoffs = (addCutoff cutoffs hides)
  in animLoop (cols,lines)
              depth
              hides
              colouringOn
              newCutoffs
              (head newCutoffs)
              ""

getCommand ":q" [] _ _ _ _ _ _ =
  exitWith ExitSuccess

getCommand ":r" _ _ depth hides colouringOn cutoffs maxCutoff =
  do (newCols,newLines) <- getTerminalSize
     animLoop (newCols,newLines)
              depth
              hides
              colouringOn
              cutoffs
              maxCutoff
              ""
              
getCommand ":d" [] (cols,lines) depth hides colouringOn cutoffs maxCutoff =
  do putStr (goto 1 lines ++ "depth: " ++ (show depth))
     getChar
     animLoop (cols,lines)
              depth
              hides
              colouringOn
              cutoffs
              maxCutoff
              ""

getCommand ":d" [newDepth] (cols,lines) depth hides colouringOn cutoffs maxCutoff =
  if isInteger newDepth then
    animLoop (cols,lines)
             (atoi newDepth)
             hides
             colouringOn
             cutoffs
             maxCutoff
             ""
  else
    do invalid (cols,lines)
       animLoop (cols,lines)
                depth
                hides
                colouringOn
                cutoffs
                maxCutoff
                "INVALID COMMAND"

getCommand ":h" [] (cols,lines) depth hides colouringOn cutoffs maxCutoff =
  do putStr (goto 1 4 ++ clearDown)
     putStr (goto 1 4
            ++ "Hidden items: "
            ++ (take (cols - 15) (repeat '-')))
     putStr (goto 1 5 ++ (unlines hides))
     getChar
     animLoop (cols,lines)
              depth
              hides
              colouringOn
              cutoffs
              maxCutoff
              ""

getCommand ":h" args (cols,lines) depth hides colouringOn cutoffs maxCutoff =
  let newHides = (hides ++ args)
  in animLoop (cols,lines)
              depth
              newHides
              colouringOn
              (reloadCutoffs newHides (last cutoffs) maxCutoff)
              maxCutoff
              ""

getCommand ":uh" args (cols,lines) depth hides colouringOn cutoffs maxCutoff =
  let newHides = (removeAll args hides)
  in animLoop (cols,lines)
              depth
              newHides
              colouringOn
              (reloadCutoffs newHides (last cutoffs) maxCutoff)
              maxCutoff
              ""

getCommand ":?" _ (cols,lines) depth hides colouringOn cutoffs maxCutoff =
  do putStr (goto 1 4 ++ clearDown)
     putStr (goto 1 4
            ++ "Help: "
            ++ (take (cols - 5) (repeat '-')))
     putStr (goto 1 5 ++
            "Press return to advance one reduction step, and backspace to go back a reduction step.\n" ++
            (highlight [Bold, Foreground Blue] "Commands:\n") ++
            ":r     Resize the display.\n" ++
            ":d     Display expression cutoff depth.\n" ++
            ":d n   Set cutoff depth to n.\n" ++
            ":h     Display all hidden expressions.\n"  ++
            ":h x*  Hide application of all functions listed.\n" ++
            ":uh x* Unhide application of all functions listed.\n" ++
            ":c     Toggle bracket colouring.\n" ++
            ":q     Exit hat-anim")
     getChar
     animLoop (cols,lines)
              depth
              hides
              colouringOn
              cutoffs
              maxCutoff
              ""

getCommand ":c" [] (cols,lines) depth hides colouringOn cutoffs maxCutoff =
  animLoop (cols,lines)
           depth
           hides
           (not colouringOn)
           cutoffs
           maxCutoff
           ""

getCommand _ _ (cols,lines) depth hides colouringOn cutoffs maxCutoff =
  do invalid (cols,lines)
     animLoop (cols,lines)
              depth
              hides
              colouringOn
              cutoffs
              maxCutoff
              ""

invalid :: (Int,Int) -> IO ()
invalid (_,lines) =
  do putStr (goto 1 lines ++ "INVALID COMMAND")
     getChar
     putStr ""

interface :: (Int,Int) -> String
interface (cols,lines) =
    (goto 1 1 ++ clearDown)
    ++ (goto 1 1 ++ "Output: " ++ take (cols - 8) (repeat '-'))
    ++ (goto 1 4 ++ "Animation: " ++ take (cols - 11) (repeat '-'))
    ++ (goto 1 (lines - 1) ++ take cols (repeat '-'))

addCutoff :: [FileNode] -> [String] -> [FileNode]
addCutoff (x:xs) hides =
  if isJust nextNode
    then ((unMaybe nextNode) + (FileNode 1)):x:xs
    else x:xs
  where
    nextNode = cutoffFindMinOffset x
                                   (removeVerticalCycles
                                     (nodeExpForNode (last (x:xs))))
                                   hides

removeCutoff :: [FileNode] -> [FileNode]
removeCutoff [x] = [x]
removeCutoff (x:xs) = xs

removeLast :: [a] -> [a]
removeLast [] = []
removeLast [x] = []
removeLast (x:xs) = x:(removeLast xs)

reloadCutoffs :: [String] -> FileNode -> FileNode -> [FileNode]
reloadCutoffs hides start maxCutoff =
  reloadCutoffsAux hides [start] maxCutoff
  where
    reloadCutoffsAux hs xs max =
      if (head newCutoffs) > max
        then xs
        else reloadCutoffsAux hs newCutoffs max
      where
        newCutoffs = addCutoff xs hs

isJust :: Maybe a -> Bool
isJust (Just x) = True
isJust _ = False

unMaybe :: Maybe a -> a
unMaybe (Just x) = x

rectify :: FilePath -> FilePath
rectify f | ".hat" `isSuffixOf` f = f
          | otherwise = f ++ ".hat"

readOutputFile :: FilePath -> IO String
readOutputFile hat = do readFile (hat++".output")

readBridgeFile :: IO [FileNode]
readBridgeFile = do until (==nil) getBridgeValue
      where
        until :: (a->Bool) -> IO a -> IO [a]
        until pred action = do
          v <- action
          if pred v then return [] else do vs <- until pred action
                                           return (v:vs)

remove :: Eq a => a -> [a] -> [a]
remove x = filter (/= x)

removeAll :: Eq a => [a] -> [a] -> [a]
removeAll xs ys = foldr remove ys xs

stripWhiteSpace :: String -> String
stripWhiteSpace (' ':xs) = xs
stripWhiteSpace x = x

isInteger :: String -> Bool
isInteger [] = True
isInteger (x:xs) = isDigit x && isInteger xs

isDigit :: Char -> Bool
isDigit '0' = True
isDigit '1' = True
isDigit '2' = True
isDigit '3' = True
isDigit '4' = True
isDigit '5' = True
isDigit '6' = True
isDigit '7' = True
isDigit '8' = True
isDigit '9' = True
isDigit _ = False

atoi :: String -> Int
atoi [] = 0
atoi (x:xs) = (digitToInt x) * (10 ^ length xs) + atoi xs