/* hduprestore.c
 * restore a filesystem stored with hdup
 * 
 * $Id: hduprestore.c,v 1.36 2004/04/19 11:44:38 miekg Exp $
 *
 * Copyright:
 *
 *   This package is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; version 2 dated June, 1991.
 * 
 *   This package is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 * 
 *   You should have received a copy of the GNU General Public License
 *   along with this package; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 *   02111-1307, USA.
 *
 *   Changelog:
 *    06 mar 2003: miekg: reworked the entire restore stuff 
 *    16 mar 2003: miekg: reworked must of it again,for version 1.6.6 
 *    17 mar 2003: miekg: match name: tar.* -> tar* (catches none compression)
 *    05 may 2003: miekg: did not add syslogging here: seems a bit pointless
 *
 */

#include"hdup.h"
#include"prototype.h"

void backup_restore(phost_t host[], int which, struct argument_t *arg ) {
	/* restore an archive, possibly a remote one */
	char *restore[MONTHLY + 1]; char *what;
	int i=0; int stat=0; 
	int failure=1; 
	int date_format = 0;
	int how_many = -1;

	/* run the prerun script - do this way early. People like to mount their
         * backup filesystem before doing anything */
        if ( host[which]->prerun != NULL ) {
                if ( hdup_runpre(host[which], scheme, arg->scheme, NULL) != 0 ) {
                        SYSLOG_HW("%s","Prescript executed with errors");
                        FATAL("%s: %s",host[which]->name, "Prescript executed with errors");
                }
        }
	
	/* some basic values need to be filled in */
	hdup_setup(host,which);

	/* set the archive to null */
	restore[MONTHLY] = NULL; restore[WEEKLY] = NULL; restore[DAILY] = NULL;

	/* check for restoring to / */
	if ( strncmp(arg->extractdir,"/",1) == 0 && strlen(arg->extractdir) == 1 ) {
		if ( host[which]->force == 0 ) 
			FATAL("%s", "Restoring to '/' is not a good idea");
		else 
			LOG("%s", "Forcing restore to '/'");
	}

	/* check the date format */
	if ( strncmp(host[which]->datespec, "iso", 3) == 0 ) { 
		date_format = DATE_ISO;
	} else if ( strncmp(host[which]->datespec, "american", 8) == 0 ) {
		date_format = DATE_US;
	} else {
		date_format = DATE_DEF;
	}
	/* run the prerun script */
	if ( host[which]->prerun != NULL ) {
		if ( hdup_runpre(host[which], scheme, arg->scheme, NULL) != 0 )
			FATAL("%s", "Prescript executed with errorsx");
	}

	if ( (how_many = find_restore(arg,host[which],restore, date_format) ) == -1 ) 
		FATAL("%s", "No archives found: nothing restored");

	if ( how_many < 2 )
		LOG("%s", "No monthly archive found. Trying to restore from what I do have");


	(void) hdup_lock(host[which], LOCK);

	/* the find_restore function return the archive in restore[]. I must
	 * walk that array in the reverse order: 3 2 1. If it's not null I
	 * should extract it. */

	/* if only 1 stat is successfull we're in. If not there really is an error */

	if (host[which]->alg != NULL ) {
		what = (char*)xmalloc(10 + 
				strlen(host[which]->alg) + strlen(host[which]->keypath) );
		sprintf(what,"yes (%s, %s)", host[which]->alg, host[which]->keypath);
		hdup_overview("Encryption", what);
	} else 
		hdup_overview("Encryption", "no");


	for ( i = how_many; i >= DAILY; i-- ) {
		host[which]->date = arg->date;
		if ( restore[i] != NULL ) {
			VERBOSE("%s %s", "Restoring from",restore[i]);
			stat = hdup_dountar(host[which], arg, restore[i]);
			if ( stat != 0 && host[which]->remote != NULL )
				/* no point in trying do something next */
				break;

			if ( i != MONTHLY )
				hdup_overview(hdup_lastname(restore[i]),NULL);
			else 
				hdup_overview("Archives",hdup_lastname(restore[i]));
		}
		/* some tar has been lucky */
		if ( stat == 0 ) 
			failure = 0;
	}

	if ( failure == 0 ) 
		hdup_overview("Status","successfully restored backup");
	else 
		hdup_cleanup(1,host[which]);
}

int find_restore(struct argument_t *arg, phost_t host, char *restore[], int date_format) {
	/* restore strategy:
	 * first look for daily, if found look for weekly, if found look for
	 * monthly. This all goes backwards in time (yes we can do that)
	 * At most we will have 3 files to untar! Thank you Boris for pointing
	 * the obvious out to me...
	 */
	int t,p,h,j;char *d;
	char *directory,*backupdir;
	char *archivename;char *matchname;
	int found;

	backupdir = host->archive;
	p = strlen(backupdir); h = strlen(arg->host);
	d=(char*) xmalloc(12);
	directory = (char*)xmalloc( p + h + 11 + 5);

	/* make sure its long enough 
	 * 2 times host: h+h, 2 times date: 12+12, scheme.tar.* */
	/* p + h + 11 + 5 =  strlen(directory) - bugfix: Frank Behrendt */
	archivename = (char*) xmalloc(p + h + 11 + 5 + h + 12 + 8 + 15 +40);
	/* should be enough */
	/* p + h + 11 + 5 =  strlen(directory) - bugfix: Frank Behrendt */
	matchname   = (char*) xmalloc(p + h + 11 + 5 + h + 12 + 8 + 15 +40);

	if ( host->history == 1 ) {
		/* don't keep a history of files */
		strcpy(d,STATIC);
		strncpy(directory,host->archive,strlen(host->archive));
	} else {
		/* sanitize the date? */
		strncpy(d,arg->date,11);
		strncpy(directory,host->archive,strlen(host->archive));
	}

	found = -1;
	for (t=0;t<=MAXRESTORE;t++) {
		/* create the dirname */
		sprintf(directory,"%s/%s/%s",backupdir,arg->host,d);

		/* the date dir at least exists */
		VERBOSE("%s %s","restore: looking in",directory);
		if (hdup_checkdir(directory) == 1) {

			/* start with daily then weekly then monthly */
			for (j=found; j <= MONTHLY; j++) {
				/* if a 'static' archive is upped to a remote machine and placed
				 * in a normal date directory hdup will not find it
				 * make the date in archive * and it will keep on working
				 * Miek 16 Apr 2003
				 */
				sprintf(matchname,"%s/%s.*.%s.tar*", directory,arg->host,scheme[j]);
				if ( hdup_globfilecheck(matchname,archivename) == 1) {
					if ( restore[j] == NULL ) {
						restore[j] = (char*) xmalloc(strlen(archivename)+2);
						strcpy(restore[j],archivename);
						found = j;
					}
				}
			}
		}
		if ( found == MONTHLY ) {
			/* found them all, or at least a montly */
			return MONTHLY; /* found something */
		}
		if ( host->history == 1 ) {
			/* no history only 1 dir to look in */
			return found;
		} else {
			/* the previous day */
			d = day(d, PREV, date_format);
		}
	}
	return found; /* nothing found */
}

char * day(char *date, int direction , int date_format) {
	/* substract or add  one day from date and return the result */
	struct tm *ltime = (struct tm*)xmalloc(sizeof(struct tm));
	time_t abstime;
	char *mydate = xmalloc(11);
	int day,month,year;

	/* unwind it */
	hdup_parsedate(date,date_format,&day,&month,&year);

	/* do your calculations */
	ltime->tm_mday = day;
	ltime->tm_mon  = (month -1);
	ltime->tm_year = (year - 1900);

	ltime->tm_sec = 0; ltime->tm_min = 0;
	ltime->tm_hour = 0; ltime->tm_wday = 0;
	ltime->tm_yday = 0; ltime->tm_isdst = 0;

	abstime = mktime(ltime);        
	/* substract a day 
	 * if direction is NEXT (+1) this add one day
	 * if direction is PREV (-1) this substracts one day
	 */
	abstime = abstime + ( direction * SECDAY );
	ltime = localtime(&abstime);

	day = ltime->tm_mday;
	month = (ltime->tm_mon + 1);
	year = (ltime->tm_year + 1900);

	/* put is back together */
	hdup_makedate(mydate, date_format, day, month, year);

	return mydate;
}
