/* hduptarlib.c, do all the tar things here
 *
 * $Id: hduptarlib.c,v 1.50 2004/06/23 17:31:52 miekg Exp $
 *
 * Copyright:
 *
 *   This package is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; version 2 dated June, 1991.
 * 
 *   This package is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 * 
 *   You should have received a copy of the GNU General Public License
 *   along with this package; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 *   02111-1307, USA.
 *  
 *   ChangeLog:
 *   17-mar-2003: removed all the fgets and fputs replaced them with getc/putc
 *   20-mar-2003: MUTE works again
 *
 */

#include"hdup.h"
#include"prototype.h"

extern unsigned int ignore_tar_err;
extern unsigned int dryrun;

int hdup_dountar(phost_t host, struct argument_t * arg, char *restore) {
	/* handle all the untarring */
	char *out = NULL;
	char *tar = NULL;
	char *ssh = NULL;
	char *cat = NULL;
	char *decrypt = NULL;
	FILE *in = NULL;

	pheader_t header = NULL;

	if ( host->remote == NULL ) {
		/* stay local */
		if (hdup_compression(restore) == CRYPT ) {
			if (host->alg == NULL ) { VERBOSE("%s", "Archive appears to be encrypted - skipping"); return 1 ; }

			decrypt = setup_decryptcmd(host);

			tar = setup_untarcmd(host,arg->extractdir,arg->extractfile, hdup_compression_crypt(restore));
			out = (char*)xmalloc(strlen(decrypt) + strlen(tar) + 4);

			sprintf(out,"%s | %s",decrypt,tar);
		} else {
			tar = setup_untarcmd(host,arg->extractdir,arg->extractfile, hdup_compression(restore));
			out = tar;
		}
	} else {
		/* move to the other side */
		/* allocate space, here and only here */
		header = (pheader_t) xmalloc (sizeof(header_t));
		hdup_fillheader(header, host, RESTORE, arg->extractdir);
		ssh = setup_transportcmd(host);
		if (hdup_compression(restore) == CRYPT ) {
			if (host->alg == NULL ) { VERBOSE("%s", "Archive appears to be encrypted - skipping"); return 1; }

			decrypt = setup_decryptcmd(host);

			out = (char*)xmalloc(strlen(decrypt) + strlen(ssh) + 4);

			sprintf(out,"%s | %s", decrypt, ssh);
		} else {
			out = ssh;
		}
	}
	/* out contains the OTHER side of the pipe, in is the archive we're dealing 
	 * with */
	/* everything is setup, now check for compression stuff */
	if ( hdup_chunk(restore) == SPLIT ) {
		/* oh no, split up archive */
		/* remote or local doesn't matter */
		cat = setup_unsplitcmd(restore);
		LOGDEBUG("%s",cat);

		if (dryrun) {
			VVERBOSE("Faking %s", cat);
			return 0;
		}
		
		if (!ignore_tar_err) 
			return ( pipe2pipe(cat, out, header) );
		else {
			/* discard any errors */
			pipe2pipe(cat, out, header);
			return 0;
		}

	} else {
		/* ok, normal people still exist */
		in = fopen(restore, "r");
		if ( in == NULL ) {
			VERBOSE("%s", "Could not open the archive for reading");
			perror(NULL);
			return 1;
		} else 
			if (dryrun) {
				VVERBOSE("Faking %s", restore);
				return 0;
			}
			if (!ignore_tar_err)
				return ( stream2pipe(in, out, header) );
			else {
				stream2pipe(in, out, header);
				return 0;
			}
	}
	/* should not reached this */
	VERBOSE("%s", "restore: nothing to do?!");
	return 1;
}

int hdup_dotar(phost_t host, int s) {
	/* this handles all the tars we should ever have to run. Everything is
	 * done with pipes */
	/* we are tarring to stdout, catch it with a pipe and putting it
	 * somewhere different*/

	char *out = NULL;
	char *in  = NULL;
	char *tar = NULL; /* hold entire tar command */
	char *ssh = NULL; /* hold entire ssh command */
	char *crypt = NULL; /* hold entire mcrypt command */
	char *split = NULL; /* hold entire split command */
	pheader_t header = NULL;
	unsigned int t = 0; char * tmpstr;

	/* get tar ready */
	tar = setup_tarcmd(host);
	LOGDEBUG("tar: %s",tar);

	if ( host->remote != NULL ) {
		/* allocate space here, keeps header NULL otherwise */
		header = (pheader_t) xmalloc (sizeof(header_t));
		ssh = setup_transportcmd(host);
	}

	if ( host->alg != NULL )
		crypt = setup_cryptcmd(host);

	if ( host->chunksize != NULL )
		split = setup_splitcmd(host);

	/* normal case: backup locally */
	if ( host->alg == NULL ) {
		/* no crypto */
		if ( host->remote == NULL ) {
			/* tar cvfz - > archive */
			if ( host->chunksize == NULL ) {
				out = host->archivename;
				if (dryrun) {
					VVERBOSE("Faking %s", tar);
					return 0;
				}
				if ( pipe2file(tar, out) != 0 ) 
					return 1;

				(void)hdup_chown (out, host->user, host->group);
				(void)hdup_chmod (out);
				return 0;
			} else {
				/* split */
				out = split;
				LOGDEBUG("split: %s",split);
				if (dryrun) {
					VVERBOSE("Faking %s", tar);
					return 0;
				}
				if ( pipe2pipe(tar, out, header) != 0 )
					return 1;

				t = hdup_chown_dir(host->dirname_date, s, host->user, 
						host->group);
				tmpstr = (char*)xmalloc(31);
				snprintf(tmpstr, 30, "%d", t);
				hdup_overview("Chunks", tmpstr);
				free(tmpstr);
				hdup_chmod_dir(host->dirname_date, s);
				return 0;
			}

			/* remote: tar cvf - | ssh hdup ... */
			/* no splitting up - happens at the remote side */
		} else	{
			if (dryrun) {
				VVERBOSE("Faking %s", tar);
				return 0;
			}
			hdup_fillheader(header, host, s, NULL);

			out = ssh;
			if ( pipe2pipe(tar, out, header) != 0 )
				return 1;

			return 0;
		}
	} else {
		/* crypto */
		if ( host->remote == NULL ) {
			if ( host->chunksize == NULL ) {
				/* tar cvfz - | crypto - > archive */
				/* add .nc extension */
				strncat(host->archivename, MCRYPT_EXT, 3);
				out = (char*)xmalloc(strlen(host->archivename) +
						strlen(crypt) + 4);

				sprintf(out,"%s > %s",crypt,host->archivename);
				
				if (dryrun) {
					VVERBOSE("Faking %s", tar);
					return 0;
				}
				if (pipe2pipe(tar,out, header) != 0 ) 
					return 1;

				(void)hdup_chown (host->archivename, host->user, host->group);
				(void)hdup_chmod (host->archivename);
				return 0;
			} else {
				/* also do splitting */
				strncat(host->archivename, MCRYPT_EXT, 3);

				/* refill spit -- archivename has changed */
				free(split); split = setup_splitcmd(host);
				if ( split == NULL ) 
					return 1;

				out = (char*)xmalloc(LEN(host->archivename) +
						LEN(crypt) + LEN(split) + 6);        
				sprintf(out,"%s | %s",crypt,split);
				LOGDEBUG("out: %s",out);

				if (dryrun) {
					VVERBOSE("Faking %s", out);
					return 0;
				}
				if (pipe2pipe(tar, out, header) != 0 ) 
					return 1;

				/* it could be that the whole dir is too much */
				t = hdup_chown_dir(host->dirname_date, s, host->user,
						host->group);
				tmpstr = (char*)xmalloc(31);
				snprintf(tmpstr, 30, "%d", t);
				hdup_overview("Chunks", tmpstr);
				free(tmpstr);
				hdup_chmod_dir(host->dirname_date, s);
			}
		} else	{
			/* tar cvfz - | crypto | ssh */
			/* no splitting up - happens at the remote side */
			hdup_fillheader(header, host, s, NULL);

			in = (char*)xmalloc(strlen(tar) + strlen(crypt) + 4);
			sprintf(in,"%s | %s",tar,crypt);
			if (dryrun) {
				VVERBOSE("Faking %s", tar);
				return 0;
			}
			if ( pipe2pipe(in,ssh, header) != 0 ) 
				return 1;

			return 0;
		}
	}
	return 0;
}

int pipe2file(char *tar, char *out) {
	/* write the tar to a file */
	FILE *pin;
	FILE *archivefile;
	int c; int stat;

	archivefile = fopen(out,"w");
	pin = popen(tar,"r");

	LOGDEBUG("full tar: %s",tar);
	LOGDEBUG("out file: %s",out);
	VVERBOSE("Running: %s > %s",tar, out);

	if ( archivefile == NULL ) { 
		VERBOSE("%s","Could not open archive for writing");
		perror(NULL);
		return 1; 
	}
	if ( pin == NULL ) { 
		VERBOSE("%s","Could not setup tar for archiving"); 
		perror(NULL);
		return 1; 
	}

	/* Processing loop */
	while( (c = getc(pin)) != EOF )
		putc(c, archivefile);

	fclose(archivefile); 

	stat = pclose(pin);
	LOGDEBUG("%s: %d", "tar exit status",stat);
	/* 512; /bin/tar: Error is not recoverable: exiting now */
	if (stat == -1 || stat == 512) 
		return 1;

	if ( WIFEXITED(stat) == 0 )
		if ( WEXITSTATUS(stat) > 0 )
			return 1;

	return 0;
}

/* cat a stream to a file
 * return the number of bytes written
 * return 0 on error
 */
unsigned long int stream2file(FILE *in, char *file) {
	/* read from the stream and write to file */
	FILE *handle;
	int c;
	unsigned long int b = 0;

	handle = fopen(file,"w");

	if ( handle == NULL ) { 
		VERBOSE("%s","Could not open archive for writing"); 
		perror(NULL);
		return 0; 
	}
	if ( in == NULL ) { 
		VERBOSE("%s", "Could not read from stream"); 
		perror(NULL);
		return 0; 
	}

	/* Processing loop */
	while( ( c = getc(in)) != EOF ) {
		++b;
		putc(c, handle);
	}

	fclose(handle);

	return b;

}

int stream2pipe(FILE *in, char *out, pheader_t head) {
	/* write from the stream and to the pipe */
	FILE *pout;
	int c; int stat;

	pout = popen(out,"w");

	if ( pout == NULL ) { 
		VERBOSE("%s","Could not open pipe for writing"); 
		perror(NULL);
		return 1; 
	}
	if ( in == NULL ) { 
		VERBOSE("%s", "Could not setup stream for reading"); 
		perror(NULL);
		return 1;
	}

	if ( head != NULL ) 
		/* print the header first */
		hdup_putheader(head, pout);

	/* Processing loop */
	/* if nothing is waiting on the other side, a SIGPIPE is thrown */
	while( (c = getc(in)) != EOF ) {
		if ( hdup_pipe == 1 ) 
			return 1;
		putc(c, pout);
	}

	stat = pclose(pout);
	if ( stat == -1 ) 
		return 1;

	if ( WIFEXITED(stat) == 0 )
		return 1;
	if ( WEXITSTATUS(stat) > 0 ) 
		return 1;

	return 0;
}

int pipe2pipe(char *tar, char *out, pheader_t head) {
	/* write the tar to a pipe */
	FILE *pin;
	FILE *pout;
	int c; int stat;

	pout = popen(out,"w");
	pin = popen(tar,"r");

	VVERBOSE("Running: %s | %s", tar, out);

	if ( pout == NULL ) { 
		VERBOSE("%s", "Could not open pipe for writing"); 
		perror(NULL);
		return 1;
	}
	if ( pin == NULL ) { 
		VERBOSE("%s", "Could not setup tar for archiving"); 
		perror(NULL);
		return 1;
	}

	if ( head != NULL ) 
		/* print the header first */
		hdup_putheader(head, pout);

	/* Processing loop */
	while( (c=getc(pin)) != EOF ) {
		if ( hdup_pipe == 1 ) 
			return 1;
		putc(c, pout);
	}

	/* out pipe */
	stat = pclose(pout);
	if (stat == -1 ) 
		return 1;

	/* with ssh 255 is returned when something fails */
	/* otherwise hdup's error is returned, both are more than 0 */
	if ( WIFEXITED(stat) == 0 )
		return 1;
	if ( WEXITSTATUS(stat) > 0  ) 
		return 1;

	/* in pipe */
	stat = pclose(pin);
	if ( stat == -1 ) 
		return 1;

	if ( WIFEXITED(stat) == 0 )
		return 1;

	return 0;
}

char * setup_transportcmd(phost_t host) {
	/* create a string with the ssh command */
	/* ssh ssh_options user@host hdup remote <hostname> */

	char *ssh;

	ssh = (char*)xmalloc ( strlen(host->proto) + strlen(host->proto_opt) +
			strlen(host->remote) + strlen(host->name) +
			strlen(host->remote_hdup) + strlen(host->remote_hdup_opt) + 15);

	sprintf(ssh,"%s %s %s %s %s remote %s", host->proto, host->proto_opt,
			host->remote, host->remote_hdup, host->remote_hdup_opt, host->name);

	return ssh;
}

char * setup_cryptcmd(phost_t host) {
	/* create a string with the mcrypt command */
	/* mcrypt -a alg -f keypath or
	 * gpg -e blabla */

	char *crypt;

	if ( strcasecmp(host->alg, GPG_ALG) == 0 ) {
		/* yep GPG */
		crypt = (char*)xmalloc(strlen(host->gpg) +
			strlen(GPG_CRYPT) +
			strlen(host->keypath) + /* now acting as ID */
			strlen(MUTE) + 6);

		sprintf(crypt,"%s " GPG_CRYPT ,
			host->gpg,
			host->keypath);
		LOGDEBUG("%s\n",crypt);

	} else {
		crypt = (char*)xmalloc(strlen(host->mcrypt) +
			strlen(host->alg) +
			strlen(host->keypath) +
			strlen(MUTE) + 
			strlen("-u -q -a -f") + 6);

		sprintf(crypt,"%s -u -q -a %s -f %s",
			host->mcrypt,
			host->alg,
			host->keypath);
	}

	/* f*cking mute still works, even in a pipe :) */
	if ( quiet > 0 ) 
		strncat(crypt, MUTE, strlen(MUTE));

	return crypt;
}

char * setup_decryptcmd(phost_t host) {
	/* create a string with the mcrypt command */
	/* mcrypt -a alg -f keypath */

	char *crypt;

	if ( strcasecmp(host->alg, GPG_ALG) == 0 ) {
		/* yep GPG */
		crypt = (char*)xmalloc(strlen(host->gpg) +
			strlen(GPG_DECRYPT) +
			strlen(host->keypath) + /* now acting as ID */
			strlen(MUTE) + 6);

		sprintf(crypt,"%s " GPG_DECRYPT ,
			host->gpg,
			host->keypath);
		LOGDEBUG("%s\n",crypt);
	} else {
		crypt = (char*)xmalloc(strlen(host->mcrypt) +
			strlen(host->alg) +
			strlen(host->keypath) +
			strlen(MUTE) +
			strlen("-d -q -a -f") + 6);

		sprintf(crypt,"%s -d -q -a %s -f %s",
			host->mcrypt,
			host->alg,
			host->keypath);
	}

	/* mute still works, even in a pipe */
	if ( quiet > 0 ) 
		strncat(crypt, MUTE, strlen(MUTE));

	return crypt;
}

char * setup_tarcmd(phost_t host) {
	/* create a string with the tar command */

	char *tar; char *comp;

	int compmax_len = HDUP_MAX( LEN(BZIP_PROG) , LEN(GZIP_PROG));
	compmax_len = HDUP_MAX( compmax_len, LEN(TAR_LZOP));

	/* make room for tar and _all_ the option even though it will 
	 * not always be used to the max */
	tar = (char*) xmalloc(LEN(host->tar) +
			LEN(TAR_NONE) +
			LEN("--files-from") +
			LEN(host->filelist) +
			LEN("--listed-incremental") +
			LEN(host->inclist) +
			LEN("--exclude-from") +
			LEN(host->excludelist) +
			LEN(SPARSE) + 2 +
			LEN("--ignore-failed-read") + compmax_len + 
			compmax_len + 4 + /* strlen(comp) */
			LEN(ONEFILE) + 
			LEN(MUTE) + 18 );

	/* the pipe prog */
	comp = (char*) xmalloc( compmax_len + LEN(" | ") + 3); /* 3 = _-6 */

	/* tar will always look like this */
	sprintf(tar,"%s %s - %s --files-from %s --listed-incremental %s --exclude-from %s --ignore-failed-read",
			host->tar, TAR_NONE, SPARSE,
			host->filelist, host->inclist,
			host->excludelist);

	if ( strncmp(host->compression, TAR_GZIP, LEN(TAR_GZIP)) == 0 ) {
		sprintf(comp," |%s -%s",GZIP_PROG, host->complevel);

		/* add the onefile option */
		if ( host->onefile == 1 )
			strcat(tar,ONEFILE);

		if ( quiet > 0 )
			strcat(tar, MUTE);

		/* add the rest of the string */
		strcat(tar, comp);

		LOGDEBUG("tar prog: [%s]", tar);

		return tar;
	}
	if ( strncmp(host->compression, TAR_BZIP, LEN(TAR_BZIP)) == 0 ) {
		sprintf(comp," |%s -%s",BZIP_PROG, host->complevel);

		if ( host->onefile == 1 )
			strcat(tar,ONEFILE);

		if ( quiet > 0 )
			strcat(tar, MUTE);

		/* add the rest of the string */
		strcat(tar, comp);

		LOGDEBUG("tar prog: [%s]", tar);
		return tar;
	}
	if ( strncmp(host->compression, TAR_LZOP, LEN(TAR_LZOP)) == 0 ) {
		sprintf(comp," |%s -%s",LZOP_PROG, host->complevel);

		if ( host->onefile == 1 )
			strcat(tar,ONEFILE);

		if ( quiet > 0 )
			strcat(tar, MUTE);

		/* add the rest of the string */
		strcat(tar, comp);

		LOGDEBUG("tar prog: [%s]", tar);
		return tar;
	}

	/* no compression */
	if ( host->onefile == 1 )
		strcat(tar,ONEFILE);

	if ( quiet > 0 )
		strcat(tar, MUTE);

	LOGDEBUG("tar prog: [%s]", tar);
	VVERBOSE("Setup: %s", tar);

	return tar;
}

char * setup_untarcmd(phost_t host, char * extractdir, char *extractfile, int compression) {
	/* create a string with the untar command */

	char *tar; char *tar_opt = NULL;

	/* figure out what options to use */
	switch (compression) {
		case NONE:
			tar_opt = UNTAR_NONE;
			break;
		case GZIP:
			tar_opt = UNTAR_GZIP;
			break;
		case BZIP:
			tar_opt = UNTAR_BZIP;
			break;
		case LZOP:
			tar_opt = UNTAR_LZOP;
			break;
		default:
			tar_opt = UNTAR_GZIP;
			break;
	}    
	if ( extractfile == NULL ) {
		/* completely untar it */
		tar = (char*) xmalloc(strlen(host->tar) +
				strlen(tar_opt) +
				strlen(UNTAR_OPT) +
				strlen(extractdir) +
				LEN(SPARSE) + 2 +
				LEN(MUTE) + /* just in case */
				strlen("--incremental") + 
				strlen("--ignore-failed-read") + 12) ;

		sprintf(tar,"%s %s - %s %s %s --ignore-failed-read --incremental",
				host->tar, UNTAR_OPT, extractdir, SPARSE, tar_opt);
	} else {
		/* extract a single file */
		tar = (char*) xmalloc(strlen(host->tar) +
				strlen(tar_opt) +
				strlen(UNTAR_OPT) +
				strlen(extractdir) +
				LEN(SPARSE) + 2 +
				strlen(MUTE) + 
				strlen("--incremental") + 
				strlen("--ignore-failed-read") +
				strlen(extractfile) + 12) ;

		sprintf(tar,"%s %s - %s %s %s --ignore-failed-read --incremental %s",
				host->tar, UNTAR_OPT,  
				extractdir, extractfile, SPARSE, tar_opt);
	}
	if ( quiet > 0 )
		strncat(tar, MUTE, strlen(MUTE));

	LOGDEBUG("untar %s\n",tar);
	VVERBOSE("Setup: %s", tar);

	return tar;
}

char * setup_splitcmd(phost_t host) {
	/* hold the split cmd */

	char *split;

	split = (char*)xmalloc(LEN(SPLIT_PROG) +
			LEN(host->chunksize) +
			LEN(host->dirname_date) +
			LEN(hdup_lastname(host->archivename)) +
			LEN(SPLIT_EXT) +
			LEN("-b k - /") + 4);

	sprintf(split,"%s -b %s - %s/%s%s",
			SPLIT_PROG,
			host->chunksize,
			host->dirname_date,
			hdup_lastname(host->archivename),
			SPLIT_EXT);

	return split;
}

char * setup_unsplitcmd (char * archive) { 
	/* create a string with cat */

	/* lappie.2003-04-24.monthly.tar__split__aa
	 * must be transformed in
	 * lappie.2003-04-24.monthly.tar*__split__??
	 */

	char *unsplit; char *pos;

	unsplit = (char*)xmalloc(LEN(UNSPLIT_PROG) +
			LEN(archive) + LEN(SPLIT_EXT) + LEN("??") + 4);

	pos = strstr(archive, SPLIT_EXT);
	if ( pos != NULL ) 
		*pos='\0'; /* cut string short */
	else 
		return NULL;

	/* put new stuff behind it */
	strcat(archive, "*");
	strcat(archive, SPLIT_EXT);

	/* the ?? denotes the aa suffix from split ( see split(1) ) */
	sprintf(unsplit,"%s %s??",
			UNSPLIT_PROG, archive);

	return unsplit; 
}
