// -*- C++ -*-

/*

  Heap Layers: An Extensible Memory Allocation Infrastructure
  
  Copyright (C) 2000-2004 by Emery Berger
  http://www.cs.umass.edu/~emery
  emery@cs.umass.edu
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#if !defined (_WINMMAPHEAP_H_)
#define _WINMMAPHEAP_H_

#include <windows.h>

#include <assert.h>

#include "sassert.h"
#include "bitstring.h"

using namespace HL;

#if defined(_WIN32)

/**
 * @class  WinMmapHeap
 * @brief  A Windows-specific heap that provides aligned large chunks.
 * @author Emery Berger <http://www.cs.umass.edu/~emery>
 */


template <int Alignment_,
	  int BigThreshold>
class WinMmapHeap {

  enum { AddressableMemory = 1UL << 31 };

public:

  enum { PageSize = 4096 };
  enum { AllocationGranularity = 65536 };

private:

  // A number of static assertions that verify certain important
  // properties, such as that page size, allocation grain, and
  // alignment are all powers of two, and that alignment is at least
  // as large as the size of a page.

  HL::sassert<((PageSize & (PageSize - 1)) == 0)>                          EnsurePageSizeIsPowerOfTwo;
  HL::sassert<((AllocationGranularity & (AllocationGranularity-1)) == 0)>  EnsureGrainIsPowerOfTwo;
  HL::sassert<((Alignment_ & (Alignment_ - 1)) == 0)>                      EnsureAlignmentIsPowerOfTwo;
  HL::sassert<(Alignment_ >= PageSize)>                                    EnsureAlignmentIsAtLeastAPage;

public:

  // Note: we will enforce the desired alignment.
  enum { Alignment = Alignment_ };

  WinMmapHeap (void)
    : _end (NULL)
  {}

  ~WinMmapHeap (void) {
    // We could consider freeing all reserved memory,
    // but this destructor is going to get called only
    // when the process is exiting, so it's unnecessary.
  }

  /// Return an aligned chunk of the requested size (or slightly larger).
  void * malloc (size_t sz) {
    char * startAddress = NULL;

    // Round the requested size up to the next page size, which must
    // be a power of two.
    size_t newSz = (sz + PageSize - 1) & ~(PageSize - 1);

    if (newSz < BigThreshold) {
      // If this is a 'small' object request, try to allocate the
      // space from the bitmap.

      int position = -1;

      // Convert the request and alignment from bytes to pages.
      const int requestedPages = newSz / PageSize;
      const int alignmentPages = Alignment / PageSize;

      // Iterate through the bitmap, adding memory until we get a
      // large enough chunk to satisfy the current request, or until
      // we run out of memory.

      while (position == -1) {
	position = getBitString().allocate (requestedPages, alignmentPages);
	if (position == -1) {
	  // We didn't have enough space in the bitmap.  Reserve more
	  // address space and add it to the bitmap.
	  // Round up the request to the allocation grain and get more memory.
	  newSz = (newSz + AllocationGranularity - 1) & ~(AllocationGranularity - 1);
	  void * ptr = VirtualAlloc (_end, newSz, MEM_RESERVE | MEM_COMMIT, PAGE_READWRITE);
	  if (ptr == NULL) {
	    // Reset _end and try again.
	    _end = NULL;
	    ptr = VirtualAlloc (NULL, newSz, MEM_RESERVE | MEM_COMMIT, PAGE_READWRITE);
	  }

	  if (ptr) {

	    // Update the end of the current allocation area.
	    void * endOfRange = (char *) ptr + newSz;
	    if ((size_t) endOfRange > (size_t) _end) {
	      _end = endOfRange;
	    }

	    // The request for memory was successful: make this
	    // allocated memory available for use by subsequent
	    // allocations.
	    getBitString().free ((unsigned long) ptr / PageSize, newSz / PageSize);

	  } else {
	    // Out of memory!
	    return NULL;
	  }
	}
      }
      
      // Convert the position in the bitmap into the starting address for the object request.
      startAddress = (char *) (position * PageSize);

    } else {

      // This is a big object request. We'll get it directly from the OS.

      // Round the size up to the allocation grain (must be a power of two).
      newSz = (sz + AllocationGranularity - 1) & ~(AllocationGranularity - 1);

      // Get the memory.
      startAddress = (char *) VirtualAlloc (NULL, newSz, MEM_RESERVE | MEM_COMMIT, PAGE_READWRITE);
    }

    // Verify its alignment.
    assert ((char *) ((startAddress + Alignment - 1) & ~(Alignment - 1)) == startAddress);
	
    // Record the size of this chunk and return it.
    getSizeTable()[(unsigned long) startAddress / PageSize] = newSz;
    return startAddress;
  }

  /// Return the size of the requested chunk.
  inline size_t getSize (void * ptr) {
    return getSizeTable()[(unsigned long) ptr / PageSize];
  }

  /// Deallocate an object.
  void free (void * ptr) {
    // Find out how big this chunk was.
    size_t sz = getSizeTable()[(unsigned long) ptr / PageSize];

    if (sz < BigThreshold) {
      // It was a 'small' one. Mark this object as 'free' in the
      // bitstring.
      getBitString().free ((unsigned long) ptr / PageSize, sz / PageSize);
      
      // Declare its current contents to be of no interest (i.e., so they won't be backed).
      VirtualAlloc (ptr, sz, MEM_RESET, 0);

    } else {

      // It was a big one. Free it directly back to the system.
      VirtualFree (ptr, 0, MEM_RELEASE);

    }
  }

private:

  /// Compute the total number of addressable pages in the system.
  enum { NumberOfPages = (AddressableMemory / (unsigned long) PageSize) };

  /// A bitmap with one bit for every page in the system.
  BitString<NumberOfPages>& getBitString (void) {
    static BitString<NumberOfPages> bitString;
    return bitString;
  }

  /// A size lookup table. We could probably be smarter about this data structure...

  typedef size_t SizeTable[NumberOfPages];

  SizeTable& getSizeTable (void) {
    static SizeTable sizes;
    return sizes;
  }

  void * _end;

};



#endif


#endif // _WINMMAPHEAP_H_
