/* $Id: complex.c,v 1.58 2005/09/05 18:37:16 andrew Exp $ */
/* 
 * Copyright (C) 2004 Andrew Beekhof <andrew@beekhof.net>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <portability.h>

#include <pengine.h>
#include <pe_utils.h>
#include <pe_rules.h>
#include <crm/msg_xml.h>

gboolean update_node_weight(rsc_to_node_t *cons,const char *id,GListPtr nodes);
gboolean is_active(rsc_to_node_t *cons);
gboolean constraint_violated(
	resource_t *rsc_lh, resource_t *rsc_rh, rsc_colocation_t *constraint);
void order_actions(action_t *lh, action_t *rh, order_constraint_t *order);

gboolean has_agent(node_t *a_node, lrm_agent_t *an_agent);

extern gboolean rsc_colocation_new(const char *id, enum con_strength strength,
				   resource_t *rsc_lh, resource_t *rsc_rh);
extern rsc_to_node_t *rsc2node_new(
	const char *id, resource_t *rsc, double weight, node_t *node,
	pe_working_set_t *data_set);

rsc_to_node_t *generate_location_rule(
	resource_t *rsc, crm_data_t *location_rule, pe_working_set_t *data_set);

void populate_hash(crm_data_t *nvpair_list, GHashTable *hash,
		   const char **attrs, int attrs_length);

resource_object_functions_t resource_class_functions[] = {
	{
		native_unpack,
		native_find_child,
		native_num_allowed_nodes,
		native_color,
		native_create_actions,
		native_internal_constraints,
		native_agent_constraints,
		native_rsc_colocation_lh,
		native_rsc_colocation_rh,
		native_rsc_order_lh,
		native_rsc_order_rh,
		native_rsc_location,
		native_expand,
		native_dump,
		native_printw,
		native_html,
		native_active,
		native_resource_state,
		native_create_notify_element,
		native_free
	},
	{
		group_unpack,
		group_find_child,
		group_num_allowed_nodes,
		group_color,
		group_create_actions,
		group_internal_constraints,
		group_agent_constraints,
		group_rsc_colocation_lh,
		group_rsc_colocation_rh,
		group_rsc_order_lh,
		group_rsc_order_rh,
		group_rsc_location,
		group_expand,
		group_dump,
		group_printw,
		group_html,
		group_active,
		group_resource_state,
		group_create_notify_element,
		group_free
	},
	{
		clone_unpack,
		clone_find_child,
		clone_num_allowed_nodes,
		clone_color,
		clone_create_actions,
		clone_internal_constraints,
		clone_agent_constraints,
		clone_rsc_colocation_lh,
		clone_rsc_colocation_rh,
		clone_rsc_order_lh,
		clone_rsc_order_rh,
		clone_rsc_location,
		clone_expand,
		clone_dump,
		clone_printw,
		clone_html,
		clone_active,
		clone_resource_state,
		clone_create_notify_element,
		clone_free
	}
};

/* resource_object_functions_t resource_variants[] = resource_class_functions; */


int get_resource_type(const char *name)
{
	if(safe_str_eq(name, XML_CIB_TAG_RESOURCE)) {
		return pe_native;

	} else if(safe_str_eq(name, XML_CIB_TAG_GROUP)) {
		return pe_group;

	} else if(safe_str_eq(name, XML_CIB_TAG_INCARNATION)) {
		return pe_clone;
	}
	
	return pe_unknown;
}

gboolean
is_active(rsc_to_node_t *cons)
{
	/* todo: check constraint lifetime */
	return TRUE;
}

static void dup_attr(gpointer key, gpointer value, gpointer user_data)
{
	add_hash_param(user_data, key, value);
}

gboolean	
common_unpack(crm_data_t * xml_obj, resource_t **rsc,
	      GHashTable *defaults, pe_working_set_t *data_set)
{
	int lpc = 0;
	const char *id    = crm_element_value(xml_obj, XML_ATTR_ID);
	const char *value = NULL;

	const char *allowed_attrs[] = {
		XML_CIB_ATTR_PRIORITY,
		XML_RSC_ATTR_INCARNATION_MAX,
		XML_RSC_ATTR_INCARNATION_NODEMAX,
		"resource_stickiness"
	};

	const char *rsc_attrs[] = {
		XML_RSC_ATTR_STOPFAIL,
		XML_RSC_ATTR_RESTART,
		"resource_stickiness",
		"multiple_active",
		"start_prereq",
		"is_managed",
		"notify"
	};	
	
	crm_log_xml_debug_2(xml_obj, "Processing resource input...");
	
	if(id == NULL) {
		pe_err("Must specify id tag in <resource>");
		return FALSE;
		
	} else if(rsc == NULL) {
		pe_err("Nowhere to unpack resource into");
		return FALSE;
		
	}
	crm_malloc0(*rsc, sizeof(resource_t));
	
	if(*rsc == NULL) {
		return FALSE;
	}
	
	(*rsc)->id  = id;
	(*rsc)->xml = xml_obj;
	(*rsc)->ops_xml = find_xml_node(xml_obj, "operations", FALSE);
	(*rsc)->variant = get_resource_type(crm_element_name(xml_obj));
	
	if((*rsc)->variant == pe_unknown) {
		pe_err("Unknown resource type: %s", crm_element_name(xml_obj));
		crm_free(*rsc);
		return FALSE;
	}
	
	(*rsc)->fns = &resource_class_functions[(*rsc)->variant];
	crm_debug_3("Unpacking resource...");
	
	(*rsc)->parameters = g_hash_table_new_full(
		g_str_hash,g_str_equal, g_hash_destroy_str,g_hash_destroy_str);

	for(lpc = 0; lpc < DIMOF(rsc_attrs); lpc++) {
		value = crm_element_value(xml_obj, rsc_attrs[lpc]);
		if(value != NULL) {
			add_hash_param(
				(*rsc)->parameters, rsc_attrs[lpc], value);
		}
	}
	
	unpack_instance_attributes(
		xml_obj, XML_TAG_ATTR_SETS, NULL, (*rsc)->parameters,
		allowed_attrs, DIMOF(allowed_attrs), data_set);

	if(defaults != NULL) {
		g_hash_table_foreach(defaults, dup_attr, (*rsc)->parameters);
	}
	
	(*rsc)->runnable	   = TRUE; 
	(*rsc)->provisional	   = TRUE; 
	(*rsc)->start_pending	   = FALSE; 
	(*rsc)->starting	   = FALSE; 
	(*rsc)->stopping	   = FALSE; 

	(*rsc)->candidate_colors   = NULL;
	(*rsc)->rsc_cons	   = NULL; 
	(*rsc)->actions            = NULL;
	(*rsc)->is_managed	   = TRUE;

	(*rsc)->recovery_type      = recovery_stop_start;
	(*rsc)->stickiness         = data_set->default_resource_stickiness;

	value = g_hash_table_lookup((*rsc)->parameters, XML_CIB_ATTR_PRIORITY);
	(*rsc)->priority	   = atoi(value?value:"0"); 
	(*rsc)->effective_priority = (*rsc)->priority;

	value = g_hash_table_lookup((*rsc)->parameters, "notify");
	(*rsc)->notify		   = crm_is_true(value); 
	
	value = g_hash_table_lookup((*rsc)->parameters, "is_managed");
	if(value != NULL && crm_is_true(value) == FALSE) {
		(*rsc)->is_managed = FALSE;
		crm_warn("Resource %s is currently not managed", (*rsc)->id);
#if 0		
		rsc_to_node_t *new_con = NULL;
		/* prevent this resource from running anywhere */
		new_con = rsc2node_new(
			"is_managed_default", *rsc, -INFINITY, NULL, data_set);
		new_con->node_list_rh = node_list_dup(data_set->nodes, FALSE);
#endif	
	}
	if((*rsc)->is_managed && data_set->symmetric_cluster) {
		rsc_to_node_t *new_con = rsc2node_new(
			"symmetric_default", *rsc, 0, NULL, data_set);
		new_con->node_list_rh = node_list_dup(data_set->nodes, FALSE);
	}
	
	crm_debug_2("Options for %s", id);
	
	value = g_hash_table_lookup((*rsc)->parameters, XML_RSC_ATTR_RESTART);
	if(safe_str_eq(value, "restart")) {
		(*rsc)->restart_type = pe_restart_restart;
		crm_debug_2("\tDependancy restart handling: restart");

	} else {
		(*rsc)->restart_type = pe_restart_ignore;
		crm_debug_2("\tDependancy restart handling: ignore");
	}

	value = g_hash_table_lookup((*rsc)->parameters, "multiple_active");
	if(safe_str_eq(value, "stop_only")) {
		(*rsc)->recovery_type = recovery_stop_only;
		crm_debug_2("\tMultiple running resource recovery: stop only");

	} else if(safe_str_eq(value, "block")) {
		(*rsc)->recovery_type = recovery_block;
		crm_debug_2("\tMultiple running resource recovery: block");

	} else {		
		(*rsc)->recovery_type = recovery_stop_start;
		crm_debug_2("\tMultiple running resource recovery: stop/start");
		
	}

	value = g_hash_table_lookup((*rsc)->parameters, "resource_stickiness");
	if(value != NULL) {
		(*rsc)->stickiness = char2score(value);
	}
	if((*rsc)->stickiness > 0) {
		crm_debug_2("\tPlacement: prefer current location%s",
			    value == NULL?" (default)":"");
	} else if((*rsc)->stickiness < 0) {
		crm_warn("\tPlacement: always move from the current location%s",
			    value == NULL?" (default)":"");
	} else {
		crm_debug_2("\tPlacement: optimal%s",
			    value == NULL?" (default)":"");
	}

	crm_debug_2("\tNotification of start/stop actions: %s",
		    (*rsc)->notify?"required":"not required");
	
	(*rsc)->fns->unpack(*rsc, data_set);

	return TRUE;
}

void
order_actions(action_t *lh_action, action_t *rh_action, order_constraint_t *order) 
{
	action_wrapper_t *wrapper = NULL;
	GListPtr list = NULL;
	
	crm_debug_2("Ordering %d: Action %d before %d",
		  order?order->id:-1, lh_action->id, rh_action->id);

	log_action(LOG_DEBUG_4, "LH (order_actions)", lh_action, FALSE);
	log_action(LOG_DEBUG_4, "RH (order_actions)", rh_action, FALSE);

	
	crm_malloc0(wrapper, sizeof(action_wrapper_t));
	if(wrapper != NULL) {
		wrapper->action = rh_action;
		wrapper->type = order->type;
		
		list = lh_action->actions_after;
		list = g_list_append(list, wrapper);
		lh_action->actions_after = list;
		wrapper = NULL;
	}
	if(order->type != pe_ordering_recover) {
		crm_malloc0(wrapper, sizeof(action_wrapper_t));
		if(wrapper != NULL) {
			wrapper->action = lh_action;
			wrapper->type = order->type;
			list = rh_action->actions_before;
			list = g_list_append(list, wrapper);
			rh_action->actions_before = list;
		}
	}
}

void common_html(resource_t *rsc, const char *pre_text, FILE *stream)
{
	const char *prov = crm_element_value(rsc->xml, XML_AGENT_ATTR_PROVIDER);
	
	fprintf(stream, "%s%s (%s%s%s:%s):\t",
		pre_text?pre_text:"", rsc->id,
		prov?prov:"", prov?"::":"",
		crm_element_value(rsc->xml, XML_AGENT_ATTR_CLASS),
		crm_element_value(rsc->xml, XML_ATTR_TYPE));
}

void common_printw(resource_t *rsc, const char *pre_text, int *index)
{
#if CURSES_ENABLED
	const char *prov = crm_element_value(rsc->xml, XML_AGENT_ATTR_PROVIDER);
	
	move(*index, 0);
	printw("%s%s (%s%s%s:%s):\t",
	       pre_text?pre_text:"", rsc->id,
	       prov?prov:"", prov?"::":"",
	       crm_element_value(rsc->xml, XML_AGENT_ATTR_CLASS),
	       crm_element_value(rsc->xml, XML_ATTR_TYPE));
#else
	crm_err("printw support requires ncurses to be available during configure");
#endif
}

void common_dump(resource_t *rsc, const char *pre_text, gboolean details)
{
	crm_debug_4("%s%s%s%sResource %s: (variant=%s, priority=%f)",
		  pre_text==NULL?"":pre_text,
		  pre_text==NULL?"":": ",
		  rsc->provisional?"Provisional ":"",
		  rsc->runnable?"":"(Non-Startable) ",
		  rsc->id,
		  crm_element_name(rsc->xml),
		  (double)rsc->priority);
}

void common_free(resource_t *rsc)
{
	if(rsc == NULL) {
		return;
	}
	
	crm_debug_5("Freeing %s", rsc->id);

	while(rsc->rsc_cons) {
 		pe_free_rsc_colocation(
			(rsc_colocation_t*)rsc->rsc_cons->data);
		rsc->rsc_cons = rsc->rsc_cons->next;
	}
	if(rsc->rsc_cons != NULL) {
		g_list_free(rsc->rsc_cons);
	}
	if(rsc->parameters != NULL) {
		g_hash_table_destroy(rsc->parameters);
	}
	pe_free_shallow_adv(rsc->candidate_colors, TRUE);
	crm_free(rsc->variant_opaque);
	crm_free(rsc);
	crm_debug_5("Resource freed");
}

void
unpack_instance_attributes(
	crm_data_t *xml_obj, const char *set_name, node_t *node,
	GHashTable *hash, const char **attrs, int attrs_length,
	pe_working_set_t *data_set)
{
	crm_data_t *attributes = NULL;
	
	if(xml_obj == NULL) {
		crm_debug_4("No instance attributes");
		return;
	}

	if(attrs != NULL && attrs[0] == NULL) {
		/* none allowed */
		crm_debug_2("No instance attributes allowed");
		return;
	}
	
	crm_debug_2("Checking for attributes");
	xml_child_iter(
		xml_obj, attr_set, set_name,

		/* check any rules */
		if(test_ruleset(attr_set, node, data_set) == FALSE) {
			continue;
		}
		
		crm_debug_2("Adding attributes");
		attributes = cl_get_struct(attr_set, XML_TAG_ATTRS);
		if(attributes == NULL) {
			pe_err("%s with no %s child", set_name, XML_TAG_ATTRS);
		} else {
			populate_hash(attributes, hash, attrs, attrs_length);
		}
		
		);
}

void
populate_hash(crm_data_t *nvpair_list, GHashTable *hash,
	      const char **attrs, int attrs_length) 
{
	int lpc = 0;
	gboolean set_attr = FALSE;
	const char *name = NULL;
	const char *value = NULL;
	xml_child_iter(
		nvpair_list, an_attr, XML_CIB_TAG_NVPAIR,
		
		name  = crm_element_value(an_attr, XML_NVPAIR_ATTR_NAME);
		
		set_attr = TRUE;

		if(attrs != NULL) {
			set_attr = FALSE;
		}
		
		for(lpc = 0; set_attr == FALSE && lpc < attrs_length
			    && attrs[lpc] != NULL; lpc++) {
			if(safe_str_eq(name, attrs[lpc])) {
				set_attr = TRUE;
			}
		}
		
		if(set_attr) {
			crm_debug_4("Setting attribute: %s", name);
			value = crm_element_value(
				an_attr, XML_NVPAIR_ATTR_VALUE);
			
			add_hash_param(hash, name, value);
			
		} else {
			crm_debug_4("Skipping attribute: %s", name);
		}
		
		);
}



void
add_rsc_param(resource_t *rsc, const char *name, const char *value)
{
	CRM_DEV_ASSERT(rsc != NULL);
	if(crm_assert_failed) {
		return;
	}
	add_hash_param(rsc->parameters, name, value);
}

void
add_hash_param(GHashTable *hash, const char *name, const char *value)
{
	CRM_DEV_ASSERT(hash != NULL);
	if(crm_assert_failed) {
		return;
	}

	crm_debug_3("adding: name=%s value=%s", crm_str(name), crm_str(value));
	if(name == NULL || value == NULL) {
		return;
		
	} else if(g_hash_table_lookup(hash, name) == NULL) {
		g_hash_table_insert(hash, crm_strdup(name), crm_strdup(value));
	}
}
	
const char *
get_rsc_param(resource_t *rsc, const char *name)
{
	CRM_DEV_ASSERT(rsc != NULL);
	if(crm_assert_failed) {
		return NULL;
	}
	return g_hash_table_lookup(rsc->parameters, name);
}

void
hash2nvpair(gpointer key, gpointer value, gpointer user_data) 
{
	const char *name    = key;
	const char *s_value = value;

	crm_data_t *xml_node  = user_data;
	crm_data_t *xml_child = create_xml_node(xml_node, XML_CIB_TAG_NVPAIR);

	crm_xml_add(xml_child, XML_NVPAIR_ATTR_NAME, name);
	crm_xml_add(xml_child, XML_NVPAIR_ATTR_VALUE, s_value);

	crm_debug_3("dumped: name=%s value=%s", name, s_value);
}

