# 
# ***** BEGIN LICENSE BLOCK *****
# Source last modified: $Id: bif.py,v 1.19 2004/07/07 22:00:04 hubbe Exp $
# 
# Portions Copyright (c) 1995-2004 RealNetworks, Inc. All Rights Reserved.
# 
# The contents of this file, and the files included with this file,
# are subject to the current version of the RealNetworks Public
# Source License (the "RPSL") available at
# http://www.helixcommunity.org/content/rpsl unless you have licensed
# the file under the current version of the RealNetworks Community
# Source License (the "RCSL") available at
# http://www.helixcommunity.org/content/rcsl, in which case the RCSL
# will apply. You may also obtain the license terms directly from
# RealNetworks.  You may not use this file except in compliance with
# the RPSL or, if you have a valid RCSL with RealNetworks applicable
# to this file, the RCSL.  Please see the applicable RPSL or RCSL for
# the rights, obligations and limitations governing use of the
# contents of the file.
# 
# Alternatively, the contents of this file may be used under the
# terms of the GNU General Public License Version 2 or later (the
# "GPL") in which case the provisions of the GPL are applicable
# instead of those above. If you wish to allow use of your version of
# this file only under the terms of the GPL, and not to allow others
# to use your version of this file under the terms of either the RPSL
# or RCSL, indicate your decision by deleting the provisions above
# and replace them with the notice and other provisions required by
# the GPL. If you do not delete the provisions above, a recipient may
# use your version of this file under the terms of any one of the
# RPSL, the RCSL or the GPL.
# 
# This file is part of the Helix DNA Technology. RealNetworks is the
# developer of the Original Code and owns the copyrights in the
# portions it created.
# 
# This file, and the files included with this file, is distributed
# and made available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY
# KIND, EITHER EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS
# ALL SUCH WARRANTIES, INCLUDING WITHOUT LIMITATION, ANY WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, QUIET
# ENJOYMENT OR NON-INFRINGEMENT.
# 
# Technology Compatibility Kit Test Suite(s) Location:
#    http://www.helixcommunity.org/content/tck
# 
# Contributor(s):
# 
# ***** END LICENSE BLOCK *****
# 
"""Functions and classes used in *.bif file loading, compiling, error
checking, and parsing."""

import os
import sys
import string
import stat
import cPickle
from xmllib import XMLParser

from module import CreateModule
import outmsg
import err
import fnmatch
import sysinfo
import types
import module
    

bif_shadows=[]

def AddShadow(file):
    bif_shadows.append(file)

## load a BIF file, returning the bif_data and saving the compiled
## version to a compiled BIF file (BIC file)
def load_bif(filename, branchlist = None, include_shadows=1):
    bif_data = BIFData()
    BIFParser(filename, bif_data, branchlist)

    orig_modules = bif_data.module_hash.keys()

    if include_shadows:
        id=bif_data.build_id
        for shadow in bif_shadows:
            for mod in bif_data.module_hash.values():
                mod.inherited=1

            if not branchlist:
                import branchlist
                branchlist=branchlist.BranchList()

            shadow_file_name=branchlist.file(shadow)

            if not shadow_file_name:
                if include_shadows > 1:
                    continue
                
                e=err.Error()
                e.Set("Unable to find shadow BIF file \"%s\"." % shadow)
                raise err.error, e

            BIFParser(shadow_file_name, bif_data, branchlist)

        tmp={}
        for m in orig_modules:
            tmp[m]=1
            
        for m in bif_data.module_hash.keys():
            if not tmp.has_key(m):
                bif_data.module_hash[m].set_attribute("from_shadow_only")
            
        bif_data.build_id=id

    return bif_data


## load a BIFData class either from the compiled BIC file or from
## the BIF file
def load_bif_data(filename, branchlist = None, shadow=1):
    bif_data = load_bif(filename, branchlist, shadow)
    return bif_data


class Default:
    def __init__(self,
                 profile=None,
                 target=None,
                 options=None,
                 system_id = None):
        self.profile=profile
        self.target=target
        self.options=options
        self.system_id=system_id

    def write(self):
        ret="  <default"
        if self.target:
            ret = ret +' target="%s"' % self.target
        if self.profile:
            ret = ret +' profile="%s"' % self.profile
        if self.options:
            ret = ret +' options="%s"' % self.options
        if self.system_id:
            ret = ret +' for="%s"' % self.system_id
        return ret+"/>"
        

## data structure for information contained in BIF files
class BIFData:
    def __init__(self):
        self.build_id = ''
        self.default_cvs_tag = ''
        self.default_cvs_tag_type = 'branch'
        self.default_cvs_root = ''
        self.default_cvs_date = ''
        self.module_hash = {}

        self.defaults = []

        self.default_target = ''
        self.default_profile = ''
        self.default_options = ''

    def set_build_id(self, build_id):
        self.build_id = build_id

    def set_default_cvs_tag(self, default_cvs_tag):
        self.default_cvs_tag = default_cvs_tag

    def set_default_cvs_tag_type(self, default_cvs_tag_type):
        self.default_cvs_tag_type = default_cvs_tag_type

    def set_default_cvs_root(self, default_cvs_root):
        self.default_cvs_root = default_cvs_root

    def add_module(self, module):
        if self.module_hash.has_key(module.id):
            mod = self.module_hash[module.id]
            if not mod.__dict__.has_key("inherited"):
                e = err.Error()
                e.Set("Two or more modules have id=\"%s\" in the bif file." % (
                    module.id))
                raise err.error, e
        self.module_hash[module.id] = module

    def write(self, all=0):
        ret = [
            "<?xml version=\"1.0\" ?>",
            "<!-- $Id: -->",
            "<build id=\"%s\">" % (self.build_id),
            ]

        ret.append("  <!-- defaults --> ")

        addnl=0
        if self.default_cvs_root:
            ret.append('  <cvs root="%s"/>' % self.default_cvs_root)
            addnl=1

        if self.default_cvs_tag:
            ret.append('  <cvs %s="%s"/>' % (self.default_cvs_tag_type,
                                             self.default_cvs_tag))
            addnl=1
            
        if self.default_cvs_date:
            ret.append('  <cvs date="%s"/>' % (self.default_cvs_date))
            addnl=1

        if addnl:
            ret.append("")

        for d in self.defaults:
            ret.append(d.write())

        ret.append("  <targets>")

        ids = self.module_hash.keys()
        ids.sort()
        for id in ids:
            m=self.module_hash[id]
            if (all or
                not m.attributes.get("from_shadow_only") or
                m.attributes.get("sdk_depend_only")):

                ret.append(self.module_hash[id].write())

        ret.append("  </targets>")
        ret.append("</build>")
        
        return string.join(ret,"\n") + "\n"
            
## the BIF file XML Parser
TAG_build = "build"
TAG_cvstag = "cvstag"
TAG_default = "default"
TAG_cvs = "cvs"
TAG_targets = "targets"
TAG_module = "module"
TAG_description = "description"
TAG_attribute = "attribute"
TAG_includeplatforms = "includeplatforms"
TAG_excludeplatforms = "excludeplatforms"
TAG_includeprofiles = "includeprofiles"
TAG_excludeprofiles = "excludeprofiles"
TAG_dependlist = "dependlist"
TAG_source_dependlist = "source_dependlist"
TAG_checkin_dependlist = "checkin_dependlist"
TAG_halt = "halt"
TAG_sdk = "sdk"
TAG_errmsg = "checkout-error-message"
TAG_defines = "defines"
TAG_ifdef = "ifdef"
TAG_ifndef = "ifndef"
TAG_umake_includefiles = "umake_includefiles"


class BIFParser(XMLParser):
    def __init__(self, filename, bif_data = None, branchlist = None):
        XMLParser.__init__(self)

        if not bif_data:
            bif_data = BIFData()
        self.bif_data = bif_data
        self.branchlist = branchlist
        
        ## intermediate
        self.module = None
        
        ## parsing state stack
        self.tag_stack = []
        self.current_tag = None

        self.linenum = 0
        self.filename=filename
        self.parse_bif(filename)

        ## Delete the 'inherited' flag
        for mod in self.bif_data.module_hash.values():
            try:
                del mod.inherited
            except AttributeError:
                pass

    def parse_bif(self, filename):
        fil = open(filename, "r")
        while 1:
            line = fil.readline()
            if not line:
                break

            self.linenum = self.linenum + 1
            self.feed(line)

    def error(self, text):
        e = err.Error()
        e.Set("bif parser(line %d): %s" % (self.linenum, text))
        raise err.error, e

    def warning(self, text):
        outmsg.send("[WARNING] bif parser(line %d): %s" % (self.linenum, text))

    def push_tag(self, tag):
        #print "%d: <%s>" % (self.linenum, tag)
        self.tag_stack.append(tag)
        self.current_tag = tag

    def pop_tag(self, tag):
        #print "%d: </%s>" % (self.linenum, tag)
        if len(self.tag_stack) == 0:
            self.error("pop xml tag with empty stack")
        
        if self.tag_stack[-1] != tag:
            self.error("pop xml tag=\"%s\" but expected tag=\"%s\"" % (
                tag, self.tag_stack[-1]))
        
        self.tag_stack = self.tag_stack[:-1]

        if len(self.tag_stack):
            self.current_tag = self.tag_stack[-1]
        
    def handle_data(self, data):
        data = string.strip(data)
        if not data:
            return

        if self.current_tag == TAG_description:
            self.module.set_description(data)
            
        elif self.current_tag == TAG_includeplatforms:
            self.module.set_platform_include_list(data)
            
        elif self.current_tag == TAG_excludeplatforms:
            self.module.set_platform_exclude_list(data)
            
        elif self.current_tag == TAG_includeprofiles:
            self.module.set_profile_include_list(data)
            
        elif self.current_tag == TAG_excludeprofiles:
            self.module.set_profile_exclude_list(data)
            
        elif self.current_tag == TAG_ifdef:
            self.module.set_define_include_list(data)
            
        elif self.current_tag == TAG_ifndef:
            self.module.set_define_exclude_list(data)
            
        elif self.current_tag == TAG_dependlist:
            self.module.set_dependancy_id_list(data)

        elif self.current_tag == TAG_source_dependlist:
            self.module.set_source_dependancy_id_list(data)

        elif self.current_tag == TAG_checkin_dependlist:
            self.module.set_checkin_dependancy_id_list(data)

        elif self.current_tag == TAG_umake_includefiles:
            self.module.umake_includefiles.extend(string.split(data))

        elif self.current_tag == TAG_defines:
            for d in string.split(data):
                val = "1"
                ind = string.find(d, "=")
                if ind != -1:
                    val = d[ind+1:]
                    d=d[:ind]
                self.module.defines[d]=val

        elif self.current_tag == TAG_sdk:
            self.module.sdks[-1].error_message = self.module.sdks[-1].error_message + data + "\r\n"

        elif self.current_tag == TAG_errmsg:
            self.module.error_message = self.module.error_message + data + "\r\n"

        else:
            self.warning("invalid data=\"%s\"" % (data))

    ## <build>
    def start_build(self, attr):
        self.push_tag(TAG_build)

        try:
            self.bif_data.set_build_id(attr['id'])
        except KeyError:
            self.error("<build> requires \"id\"")

    ## </build>
    def end_build(self):
        self.pop_tag(TAG_build)


    def start_inherit(self, attr):
        idtmp=self.bif_data.build_id
        try:
            tid = attr["id"]
        except KeyError:
            self.error("<inherit> requires \"id\"")

        if not self.branchlist:
            import branchlist
            self.branchlist = branchlist.BranchList()
            
        file_name=self.branchlist.file(tid)

        if not file_name:
            self.error("Unable to find BIF file \"%s\" for inherit." % tid)

        BIFParser(file_name, self.bif_data, self.branchlist)
        for mod in self.bif_data.module_hash.values():
            mod.inherited=1
                
        self.bif_data.build_id=idtmp


    ## <sdk name="sdk_name" [ for="system_id_glob" ] [ path="default_path" ] [ ifexists="file or dir" ]/>
    def start_sdk(self, attr):
        self.push_tag(TAG_sdk)

        self.module.sdks.append(
            module.SDK(attr.get("name"),
                       attr.get("path"),
                       "",
                       attr.get("for"),
                       attr.get("ifexists")))


    def end_sdk(self):
        self.pop_tag(TAG_sdk)
        

    ## <default [for="system_id_glob"] [profile=".."] [target=".."] [options=".."] />
    def start_default(self, attr):
        if self.current_tag != TAG_build:
            self.error("<default> in wrong place")

        self.bif_data.defaults.append(
            Default(attr.get("profile"),
                    attr.get("target"),
                    attr.get("options"),
                    attr.get("for")))
                    

        if attr.has_key("for"):
            if not fnmatch.fnmatch(sysinfo.id ,attr['for']):
                match =1
                for l in sysinfo.family_list:
                    if fnmatch.fnmatch(l ,attr['for']):
                        match=1
                        break
                if not match:
                    return
            
        if attr.has_key("profile"):
            self.bif_data.default_profile = attr['profile']
            
        if attr.has_key("target"):
            self.bif_data.default_target = attr['target']
        
        if attr.has_key("options"):
            self.bif_data.default_options = attr['options']

        
    ## <cvstag id="..."/>
    def start_cvstag(self, attr):
        try:
            tid = attr["id"]
        except KeyError:
            self.error("<cvstag> requires \"id\"")

        ## set global default CVS tag
        if self.current_tag == TAG_build:
            self.bif_data.set_default_cvs_tag(tid)
            if attr.has_key('type'):
                self.bif_data.set_default_cvs_tag_type(attr['type'])

        ## set the CVS tag/type for module
        elif self.current_tag == TAG_module:
            if attr.has_key('type'):
                self.module.set_cvs_tag(tid, attr['type'])
            else:
                self.module.set_cvs_tag(tid)

        else:
            self.error("<cvstag> in wrong place")

    ## <cvs root="..." tag="..." branch="..." path="..." date="..."/>
    def start_cvs(self, attr):
        tid = None
        type = None
        root = None
        path = None
        date = None
        
        try:
            root = attr["root"]
        except KeyError:
            pass

        try:
            tid = attr["tag"]
            type = "tag"
        except KeyError:
            pass

        try:
            tid = attr["branch"]
            type = "branch"
        except KeyError:
            pass

        try:
            path = attr["path"]
        except KeyError:
            pass

        try:
            date = attr["date"]
        except KeyError:
            pass

        if not root and tid == None and path == None and date == None:
            self.error('<cvs> requires "root", "tag", "branch", "path" or "date"')
            
        ## set global default CVS tag
        if self.current_tag == TAG_build:
            if root:
                self.bif_data.set_default_cvs_root(root)
            if tid != None:
                self.bif_data.set_default_cvs_tag(tid)
                if type:
                    self.bif_data.set_default_cvs_tag_type(type)
            if path:
                self.error("<cvs path=.../> in wrong place")

            if date:
                self.bif_data.default_cvs_date = date

        ## set the CVS tag/type for module
        elif self.current_tag == TAG_module:
            if root:
                self.module.set_cvs_root(root)

            if tid != None:
                if type:
                    self.module.set_cvs_tag(tid, type)
                else:
                    self.module.set_cvs_tag(tid)

            if path:
                self.module.cvs_path = path

            if date:
                self.module.set_cvs_date(date)
                
        else:
            self.error("<cvs> in wrong place")

    ## <targets>
    def start_targets(self, attr):
        self.push_tag(TAG_targets)

    ## </targets>
    def end_targets(self):
        self.pop_tag(TAG_targets)


    ## <checkout_error_message>
    def start_checkout_error_message(self, attr):
        self.push_tag(TAG_errmsg)

    ## </checkout_error_message>
    def end_checkout_error_message(self):
        self.pop_tag(TAG_errmsg)

## <module id="..." name="..." group="..." type="...">
    def start_module(self, attr):
        self.push_tag(TAG_module)

        try:
            mid = attr["id"]
        except KeyError:
            self.error("<module> requires \"id\"")

        mname = None
        if attr.has_key('name'):
            mname = attr['name']
        self.module = CreateModule(mid, mname, self.filename, self.linenum)

        ## Bind root/date/tag from global defaults
        if self.bif_data.default_cvs_tag:
            if self.bif_data.default_cvs_tag_type:
                self.module.set_cvs_tag(self.bif_data.default_cvs_tag,
                                        self.bif_data.default_cvs_tag_type)
            else:
                self.module.set_cvs_tag(self.bif_data.default_cvs_tag)

        if self.bif_data.default_cvs_date:
            self.module.set_cvs_date(self.bif_data.default_cvs_date)

        if self.bif_data.default_cvs_root:
            self.module.set_cvs_root(self.bif_data.default_cvs_root)


        if attr.has_key('group'):
            self.module.set_group(attr['group'])

        if attr.has_key('type'):
            mtype = attr["type"]
            
            if mtype == 'distribution':
                self.module.set_type(self.module.MODULE_DISTRIBUTION)
            elif mtype == 'name_only':
                self.module.set_type(self.module.MODULE_NAME_ONLY)
            elif mtype == 'installer':
                self.module.set_type(self.module.MODULE_INSTALLER)
            elif mtype == 'profile':
                self.module.set_type(self.module.MODULE_PROFILE)
            else:
                self.error("unsupported module type=\"%s\"" % (mtype))

    ## </module>
    def end_module(self):
        self.pop_tag(TAG_module)
        self.bif_data.add_module(self.module)
        self.module = None

    ## <description>
    def start_description(self, attr):
        self.push_tag(TAG_description)

    ## </description>
    def end_description(self):
        self.pop_tag(TAG_description)

    ## <attribute id="..."/>
    def start_attribute(self, attr):
        if self.current_tag != TAG_module:
            self.error("<attribute> in wrong place")

        try:
            aid = attr["id"]
        except KeyError:
            self.error("<attribute> requires \"id\"")

        if aid == 'build_number':
            self.module.set_build_number()
        elif aid == 'has_version_file':
            self.module.set_version_file()
        elif aid == 'update_platform_header':
            self.module.set_update_platform_header()
        elif aid == 'static_build':
            self.module.set_build_static()
        elif aid == 'static_build_only':
            self.module.set_build_static_only()
        elif aid == 'dynamic_build_only':
            self.module.set_build_dynamic_only()
        elif aid == 'no_build':
            self.module.set_no_build()
        else:
            self.module.set_attribute(aid)
        

    ## <halt priority="..."/>
    def start_halt(self, attr):
        if self.current_tag != TAG_module:
            self.error("<halt> in wrong place")

        try:
            priority = attr["priority"]
        except KeyError:
            self.error("<halt> requires \"priority\"")

        if priority not in ["red", "yellow", "green"]:
            self.error("<halt priority=\"%s\"> invalid, must be: "\
                       "red, yellow, green" % (prioriey))

        self.module.set_halt_priority(priority)

            
    ## <includeplatforms>
    def start_includeplatforms(self, attr):
        self.push_tag(TAG_includeplatforms)

    ## </includeplatforms>
    def end_includeplatforms(self):
        self.pop_tag(TAG_includeplatforms)

    ## <excludeplatforms>
    def start_excludeplatforms(self, attr):
        self.push_tag(TAG_excludeplatforms)

    ## </excludeplatforms>
    def end_excludeplatforms(self):
        self.pop_tag(TAG_excludeplatforms)

    ## <includeprofiles>
    def start_includeprofiles(self, attr):
        self.push_tag(TAG_includeprofiles)

    ## </includeprofiles>
    def end_includeprofiles(self):
        self.pop_tag(TAG_includeprofiles)

    ## <excludeprofiles>
    def start_excludeprofiles(self, attr):
        self.push_tag(TAG_excludeprofiles)

    ## </excludeprofiles>
    def end_excludeprofiles(self):
        self.pop_tag(TAG_excludeprofiles)

    ## <dependlist>
    def start_dependlist(self, attr):
        self.push_tag(TAG_dependlist)

    ## </dependlist>
    def end_dependlist(self):
        self.pop_tag(TAG_dependlist)

    ## <source_dependlist>
    def start_source_dependlist(self, attr):
        self.push_tag(TAG_source_dependlist)

    ## </source_dependlist>
    def end_source_dependlist(self):
        self.pop_tag(TAG_source_dependlist)

    ## <checkin_dependlist>
    def start_checkin_dependlist(self, attr):
        self.push_tag(TAG_checkin_dependlist)

    ## </checkin_dependlist>
    def end_checkin_dependlist(self):
        self.pop_tag(TAG_checkin_dependlist)

    ## <defines>
    def start_defines(self, attr):
        self.push_tag(TAG_defines)

    ## </defines>
    def end_defines(self):
        self.pop_tag(TAG_defines)

    ## <ifdef>
    def start_ifdef(self, attr):
        self.push_tag(TAG_ifdef)

    ## </ifdef>
    def end_ifdef(self):
        self.pop_tag(TAG_ifdef)

    ## <ifndef>
    def start_ifndef(self, attr):
        self.push_tag(TAG_ifndef)

    ## </ifndef>
    def end_ifndef(self):
        self.pop_tag(TAG_ifndef)

    ## <umake_includefiles>
    def start_umake_includefiles(self, attr):
        self.push_tag(TAG_umake_includefiles)

    ## </umake_includefiles>
    def end_umake_includefiles(self):
        self.pop_tag(TAG_umake_includefiles)

## useful...
def module_list_to_module_id_list(module_list):
    module_id_list = []
    for module in module_list:
        module_id_list.append(module.id)
    return module_id_list


## check for circular dependancies
class CircularDependCheck:
    def __init__(self, bif_data):
        self.bif_data = bif_data
        
        self.good_module_list = []
        self.bad_module_list = []

        ## flag that allows us to abort the recursion
        self.abort = 0
        
        ## working stack
        self.stack = []

        ## list of discovered cycles
        self.cycle_list = []

        ## list of module IDs without modules
        self.unknown_module_id_list = []
        
    def run(self):
        for module in self.bif_data.module_hash.values():
            outmsg.verbose("checking module id=\"%s\"" % (module.id))
            self.rc_check(module)

    def rc_check(self, _module):
        ## somthing happened
        if self.abort:
            return
        
        ## if this module is in the good_module_list, then we've
        ## been certified good
        if self.good_module_list.count(_module):
            return
        
        ## check for cycle (circular dependancy)
        if self.stack.count(_module):

            ## add entire stack to the bad module list
            for module in self.stack:
                if not self.bad_module_list.count(module):
                    self.bad_module_list.append(module)

            ## record the discovered cycle
            cycle = self.stack[self.stack.index(_module):]
            if not self.check_duplicate_in_cycle_list(cycle):
                self.cycle_list.append(cycle)

            self.abort = 1
            return

        ## push current module onto the stack
        self.stack.append(_module)

        ## check for a dependancy in the stack
        for module_id in _module.dependancy_id_list:
            try:
                chk_module = self.bif_data.module_hash[module_id]
            except KeyError:
                ## log modules that do not exist
                self.unknown_module_id_list.append((_module.id, module_id))
                continue

            self.rc_check(chk_module)        

        ## if we're not bad, then we're good!
        if not self.bad_module_list.count(_module):
            self.good_module_list.append(_module)

        ## pop off the stack
        self.stack.remove(_module)
        
    def compare_lists(self, list1, list2):
        ## if the lengths don't match, then the lists
        ## don't match
        if len(list1) != len(list2):
            return 0

        for index in range(len(list1)):
            if list1[index] != list2[index]:
                return 0

        return 1

    def check_duplicate_in_cycle_list(self, cycle):
        for temp in self.cycle_list:
            if self.compare_lists(temp, cycle):
                return 1

        return 0
                
    def print_module_list(self, list):
        print '--- MODULE DUMP ---'
        
        for module in list:
            print module.id

        print '------------------'

def CheckBIFData(bif_data):
    cdep_check = CircularDependCheck(bif_data)
    cdep_check.run()

    ## print any circular dependancies
    if len(cdep_check.cycle_list):
        print '*** found BIF circular dependancies ***'
        
        for cycle in cdep_check.cycle_list:
            temp = "bif dependancy cycle=\"%s\"" % (
                string.join(module_list_to_module_id_list(cycle), '->'))
            outmsg.error(temp)

    ## print unresolved module ids
    if len(cdep_check.unknown_module_id_list):
        for unknown in cdep_check.unknown_module_id_list:
            outmsg.error('in dependlist of %s found unknown dependancy %s' % (
                unknown[0], unknown[1]))

def CheckBIFFile(filename):
    if not os.path.isfile(filename):
        print 'file not found'
        sys.exit(1)

    ## load BIF file
    print 'loading...'
    bif_data = load_bif_data(filename)


def rdiff(a, b, done):
    if a == b:
        return 1

    if done.has_key( repr( (a,b) ) ):
        return 1

    done[ repr( (a,b) ) ] = 1

    ta=type(a)
    tb=type(b)

    if ta == tb:
        if ta == types.InstanceType:
            if not rdiff(a.__class__,b.__class__, done):
                print "Class differs"
                return 0
            
            if not rdiff(a.__dict__,b.__dict__, done):
                print "Dict differs"
                return 0

            return 1
            
        if ta == types.DictType:
            for k in a.keys():
                if not b.has_key(k):
                    print "Only in a: %s" % repr(k)
                    return 0
                
            for k in b.keys():
                if not a.has_key(k):
                    print "Only in b: %s" % repr(k)
                    return 0

            for k in a.keys():
                if not rdiff(a[k], b[k], done):
                    print "Value for key %s differs." % repr(k)
                    return 0

            return 1

        if ta == types.TupleType or ta==types.ListType:
            if not rdiff(len(a), len(b), done):
                print "length differs"
                return 0
            n=0
            while n < len(a):
                if not rdiff(a[n], b[n], done):
                    print "index %d differs" % n
                    return 0
                n = n + 1
            return 1

    print "%s != %s " % (repr(a), repr(b))
    return 0

if __name__ == '__main__':

    def test_bif(bif_path, branchlist):
        ## get the bif_data sturcture
        print "loading %s" % (bif_path)
        try:
            bif_data = load_bif_data(bif_path, branchlist)
        except err.error, e:
            print "Didn't load...."
            print e.Text()
            return
        except:
            e = err.Error()
            e.Set("BIF file %s didn't load..." % bif_path)
            e.SetTraceback(sys.exc_info())
            print
            print e.Text()
            return
            
        open("./biftestfile-tmp.bif","w").write(bif_data.write())
        bif_data2 = load_bif_data("./biftestfile-tmp.bif")

        if not rdiff(bif_data, bif_data2, {}):
            print "Bif file %s not reproducable" % bif_path
            sys.exit(1)

        return bif_data

    import buildmenu
    buildmenu.call_buildrc()
    
    import getopt

    def main():
        bif_path = ''
        file_flag = 0
        all_flag=0
        branch_list=0
        (opt_list, arg_list) = getopt.getopt(sys.argv[1:], 'fa')

        ## check for the remote build argument
        for opt in opt_list:
            if opt[0] == '-f':
                file_flag = 1
            if opt[0] == '-a':
                all_flag = 1

        if all_flag:
            import branchlist
            branch_list = branchlist.BranchList()
            for branch_name in branch_list.list:
                fname = branch_list.file(branch_name)
                test_bif(fname, branch_list)

            print "Successful test"
            sys.exit(0)

        ## check that there was a argument specified
        if len(arg_list) < 1:
            pname = os.path.basename(sys.argv[0])
            print '%s: invalid usage' % (pname)
            print 'python %s [-f] build-branch' % (pname)
            print 'python %s -a' % (pname)
            print '-f: take argument as path instead of branch specification'
            sys.exit(1)

        ## if we've been given a branch specification, then find
        ## the build information file path for it
        if file_flag:
            bif_path = arg_list[0]
        else:
            import branchlist
            branch_list = branchlist.BranchList()
            bif_path = branch_list.file(arg_list[0])
            if not bif_path:
                print '%s invalid branch' % (arg_list[0])
                sys.exit(1)

        print test_bif(bif_path, branch_list).write()

    main()
    #import profile
    #profile.run('main()')
    
