/* ***** BEGIN LICENSE BLOCK *****
 * Source last modified: $Id: theora_page2pkt.cpp,v 1.2.2.3 2004/07/09 01:58:26 hubbe Exp $
 * 
 * Portions Copyright (c) 1995-2004 RealNetworks, Inc. All Rights Reserved.
 * 
 * The contents of this file, and the files included with this file,
 * are subject to the current version of the RealNetworks Public
 * Source License (the "RPSL") available at
 * http://www.helixcommunity.org/content/rpsl unless you have licensed
 * the file under the current version of the RealNetworks Community
 * Source License (the "RCSL") available at
 * http://www.helixcommunity.org/content/rcsl, in which case the RCSL
 * will apply. You may also obtain the license terms directly from
 * RealNetworks.  You may not use this file except in compliance with
 * the RPSL or, if you have a valid RCSL with RealNetworks applicable
 * to this file, the RCSL.  Please see the applicable RPSL or RCSL for
 * the rights, obligations and limitations governing use of the
 * contents of the file.
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL") in which case the provisions of the GPL are applicable
 * instead of those above. If you wish to allow use of your version of
 * this file only under the terms of the GPL, and not to allow others
 * to use your version of this file under the terms of either the RPSL
 * or RCSL, indicate your decision by deleting the provisions above
 * and replace them with the notice and other provisions required by
 * the GPL. If you do not delete the provisions above, a recipient may
 * use your version of this file under the terms of any one of the
 * RPSL, the RCSL or the GPL.
 * 
 * This file is part of the Helix DNA Technology. RealNetworks is the
 * developer of the Original Code and owns the copyrights in the
 * portions it created.
 * 
 * This file, and the files included with this file, is distributed
 * and made available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS
 * ALL SUCH WARRANTIES, INCLUDING WITHOUT LIMITATION, ANY WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, QUIET
 * ENJOYMENT OR NON-INFRINGEMENT.
 * 
 * Technology Compatibility Kit Test Suite(s) Location:
 *    http://www.helixcommunity.org/content/tck
 * 
 * Contributor(s):
 * 
 * ***** END LICENSE BLOCK ***** */

#include "theora_page2pkt.h"

#include "debug.h" // DPRINTF()
#define D_THEORA_P2P 0 //0x4000000

#define THEORA_MIME_TYPE "video/x-rn-theora"

#define FMTP_CONFIG_PREFIX "a=fmtp:0 config="

COggPageToPacket* TheoraPageToPacket::Build(ogg_packet* pPkt)
{
#ifdef _DEBUG
    debug_level() |= D_THEORA_P2P;
#endif /* _DEBUG */
    COggPageToPacket* pRet = NULL;
    
    if (IsTheoraHeader(pPkt) &&   // Is it a Theora header?
        (pPkt->packet[0] == 0x80)) // Is it the ident header?
    {
        pRet = new TheoraPageToPacket;
    }

    DPRINTF(D_THEORA_P2P, ("TheoraPageToPacket::Build() : %p\n", pRet));

    return pRet;
}

TheoraPageToPacket::TheoraPageToPacket() :
    m_bHaveIdentInfo(FALSE),
    m_ulFPSNum(0),
    m_ulFPSDenom(0),
    m_ulBitrate(0),
    m_uKeyFrameShift(0),
    m_pFMTPString(NULL),
    m_bCurrentGranuleValid(FALSE),
    m_currentGranule(0)
{
    DPRINTF(D_THEORA_P2P, ("TheoraPageToPacket::TheoraPageToPacket()\n"));
}

TheoraPageToPacket::~TheoraPageToPacket()
{
    DPRINTF(D_THEORA_P2P, ("TheoraPageToPacket::~TheoraPageToPacket()\n"));
    HX_RELEASE(m_pFMTPString);
}

STDMETHODIMP TheoraPageToPacket::GetStreamHeader(REF(IHXValues*) pHeader)
{
    DPRINTF(D_THEORA_P2P, ("TheoraPageToPacket::GetStreamHeader()\n"));
    HX_RESULT res = HXR_FAILED;

    if (stReady == GetState())
    {
	IHXCommonClassFactory* pCCF = GetCCF();
	if (pCCF)
	{
	    res = pCCF->CreateInstance(CLSID_IHXValues, (void **)&pHeader);
        
	    if (HXR_OK == res) 
	    {
		// 2 seconds of preroll should be plenty
		pHeader->SetPropertyULONG32("Preroll", 2000);
		
		if ((0xffffff == m_ulBitrate) ||
		    (0 == m_ulBitrate))
		{
		    // Wild guess
		    pHeader->SetPropertyULONG32("AvgBitRate", 80000);
		}
		else
		{
		    pHeader->SetPropertyULONG32("AvgBitRate", m_ulBitrate);
		}
		
		if (m_pFMTPString)
		{
		    pHeader->SetPropertyCString("SDPData", m_pFMTPString);
		}

		IHXBuffer* pMimeType = NULL;
		
		res = pCCF->CreateInstance(CLSID_IHXBuffer,
					   (void**)&pMimeType);
		if (HXR_OK == res)
		{
		    res = pMimeType->Set((UINT8*)THEORA_MIME_TYPE,
					 strlen(THEORA_MIME_TYPE) + 1);
		    
		    if (HXR_OK == res)
		    {
			res = pHeader->SetPropertyCString("MimeType", 
							  pMimeType);
		    }
		    
		    HX_RELEASE(pMimeType);
		}
	    }
	    
	    if (HXR_OK != res)
	    {
		HX_RELEASE(pHeader);
	    }

	    HX_RELEASE(pCCF);
	}
    }

    return res;
}

STDMETHODIMP TheoraPageToPacket::OnSeek(UINT32 ulSeekPoint)
{
    m_bCurrentGranuleValid = FALSE;
    m_currentGranule = 0;

    return CBasePageToPacket::OnSeek(ulSeekPoint);
}

STDMETHODIMP TheoraPageToPacket::GetTimestamp(ogg_int64_t granulepos, 
					     REF(UINT32) ulTimestamp) const
{
    HX_RESULT res = HXR_FAILED;

    if (HasStartTime())
    {
	ogg_int64_t frame = Granule2Frame(granulepos);

        // Subtract off the frame count indicated by the start time
        frame -= Granule2Frame(StartTime());

	ogg_int64_t q = frame / m_ulFPSNum;
	ogg_int64_t r = frame - (q * m_ulFPSNum);
	ogg_int64_t mult = ((ogg_int64_t)m_ulFPSDenom) * 1000;
	ogg_int64_t ts = ((q * mult) + 
			  ((r * mult) / m_ulFPSNum));
	ulTimestamp = INT64_TO_UINT32(ts);
	res = HXR_OK;
    }

    DPRINTF(D_THEORA_P2P, ("TheoraPageToPacket::GetTimestamp( %lld ) : %u\n",
			   granulepos, ulTimestamp));

    
    return res;
}

UINT32 TheoraPageToPacket::HeaderCount() const
{
    return 3;
}

BOOL TheoraPageToPacket::IsHeader(ogg_packet* pPkt) const
{
    return IsTheoraHeader(pPkt);
}

void TheoraPageToPacket::StartStream()
{
    DPRINTF(D_THEORA_P2P, ("TheoraPageToPacket::StartStream()\n"));

}

void TheoraPageToPacket::InvalidateCurrentTime()
{
    m_bCurrentGranuleValid = FALSE;
    m_currentGranule = 0;
}

ogg_int64_t TheoraPageToPacket::CurrentTime() const
{
    DPRINTF(D_THEORA_P2P, ("TheoraPageToPacket::CurrentTime() : %u\n",
			   Granule2Frame(m_currentGranule)));
    return m_currentGranule;
}

static 
UINT32 gcd(UINT32 a, UINT32 b)
{
    // Compute the greatest common denominator
    while(b != 0)
    {
        UINT32 tmp = b;
        b = a % b;
        a = tmp;
    }

    return a;
}

HX_RESULT TheoraPageToPacket::OnHeader(ogg_packet* pPkt)
{
    DPRINTF(D_THEORA_P2P, ("TheoraPageToPacket::OnHeader() : %d\n", 
			   pPkt->bytes));

    HX_RESULT res = HXR_FAILED;

    if (pPkt && pPkt->packet && IsTheoraHeader(pPkt))
    {
	if ((pPkt->packet[0] == 0x80) && (pPkt->bytes >= 42))
	{
	    const unsigned char* pIdent = (const unsigned char*)pPkt->packet;

	    // Ident Header
	    m_ulFPSNum = ((pIdent[0x16] << 24) | (pIdent[0x17] << 16) |
			  (pIdent[0x18] << 8) | (pIdent[0x19]));
	    m_ulFPSDenom = ((pIdent[0x1a] << 24) | (pIdent[0x1b] << 16) |
			    (pIdent[0x1c] << 8) | (pIdent[0x1d]));

            // Attempt to reduce the numerator and denominator
            // by finding the greatest common denominator.
            UINT32 tmp = gcd(m_ulFPSNum, m_ulFPSDenom);
            if (tmp > 1)
            {
                m_ulFPSNum /= tmp;
                m_ulFPSDenom /= tmp;
            }

	    m_ulBitrate = ((pIdent[0x25] << 16) | (pIdent[0x26] << 8) |
			   (pIdent[0x27]));
	    m_uKeyFrameShift = (((pIdent[0x28] & 0x3) << 3) | 
				(pIdent[0x29] >> 5));

	    DPRINTF(D_THEORA_P2P, ("TheoraPageToPacket::OnHeader() : FPSnum %u FPSdenom %u bitrate %u keyframeShift %d\n",
				   m_ulFPSNum, m_ulFPSDenom, m_ulBitrate,
				   m_uKeyFrameShift));

	    if (m_ulFPSDenom == 0)
	    {
		m_ulFPSDenom = 1;
	    }

	    res = CreateFMTPString(pPkt);

	    if (HXR_OK == res)
	    {
		m_bHaveIdentInfo = TRUE;
	    }
	}
	else
	{
	    res = HXR_OK;
	}
    }

    return res;
}

HX_RESULT TheoraPageToPacket::OnDataPacket(ogg_packet* pPkt)
{
    DPRINTF(D_THEORA_P2P, ("TheoraPageToPacket::OnDataPacket() : %d\n", pPkt->bytes));

    HX_RESULT res = HXR_FAILED;

    if ((pPkt->packet[0] & 0x80) == 0)
    {
	// This is a video packet

	if ((pPkt->packet[0] & 0x40) == 0)
	{
	    // This is an I Frame
	    ogg_int64_t frames = 
		m_currentGranule & ((1 << m_uKeyFrameShift)-1);
	    m_currentGranule >>= m_uKeyFrameShift;
	    m_currentGranule += frames + 1;
	    m_currentGranule <<= m_uKeyFrameShift;
	}
	else
	{
	    // This is a P Frame
	    m_currentGranule++;
	}

	res = HXR_OK;
    }

    return res;
}

HX_RESULT TheoraPageToPacket::OnPageEnd(ogg_page* pPage)
{
    DPRINTF(D_THEORA_P2P, ("TheoraPageToPacket::OnPageEnd()\n"));

    HX_RESULT res = HXR_FAILED;

    // Only called when we are trying to establish the 
    // current time

    if (!m_bCurrentGranuleValid) 
    {
	ogg_int64_t pageEndTime = ogg_page_granulepos(pPage);
    
	if (pageEndTime > 0)
	{
	    ogg_int64_t upper = Granule2Frame(pageEndTime) + 1;

	    // At this point m_currentGranule just contains
	    // a 0 based granule position
	    ogg_int64_t lower = Granule2Frame(m_currentGranule);
	    
	    DPRINTF(D_THEORA_P2P, ("OPE() : %lld %lld %lld \n",
				   upper, lower, 
				   upper - lower));
	    
	    // Update m_currentGranule with the new current time
	    m_bCurrentGranuleValid = TRUE;
	    m_currentGranule = (upper - lower) << m_uKeyFrameShift;

	    if (!HasStartTime())
	    {
		SetStartTime(m_currentGranule);
	    }
	    OnCurrentTimeValid();
	}

	res = HXR_OK;
    }
    
    return res;
}

ogg_int64_t TheoraPageToPacket::Granule2Frame(ogg_int64_t granule) const
{
    ogg_int64_t iframes = granule >> m_uKeyFrameShift;
    ogg_int64_t pframes = granule - (iframes << m_uKeyFrameShift);
    DPRINTF(D_THEORA_P2P, ("TheoraPageToPacket::Granule2Frame() : %lld %lld %lld\n",
			   granule, iframes, pframes));    
    return iframes + pframes;
}

static UINT8 Nibble2Char(UINT8 nibble)
{
    static const char z_hexTbl[] = "0123456789abcdef";
    
    return z_hexTbl[nibble & 0xf];
}

HX_RESULT TheoraPageToPacket::CreateFMTPString(ogg_packet* pPkt)
{
    HX_RESULT res = HXR_FAILED;
    
    if (pPkt)
    {
	IHXCommonClassFactory* pCCF = GetCCF();
	if (pCCF)
	{
	    HX_RELEASE(m_pFMTPString);

	    res = pCCF->CreateInstance(CLSID_IHXBuffer, 
				       (void **)&m_pFMTPString);

	    if (HXR_OK == res)
	    {
		UINT32 ulFMTPSize = 
		    ( 1 + // Null terminator
                      2 + // \r\n
		      strlen(FMTP_CONFIG_PREFIX) + pPkt->bytes * 2);
		
		res = m_pFMTPString->SetSize(ulFMTPSize);

		if (HXR_OK == res)
		{
		    // Handle config field
		    char* pTmp = (char*)m_pFMTPString->GetBuffer();

		    strcpy(pTmp, FMTP_CONFIG_PREFIX);
		    pTmp += strlen(pTmp);

		    for (int i = 0; i < pPkt->bytes; i++)
		    {
			*pTmp++ = Nibble2Char(pPkt->packet[i] >> 4);
			*pTmp++ = Nibble2Char(pPkt->packet[i]);
		    }

                    // Add line ending
                    *pTmp++ = '\r';
                    *pTmp++ = '\n';

		    // Add null terminator
		    *pTmp = '\0';
		}
		else
		{
		    HX_RELEASE(m_pFMTPString);
		}
	    }

	    HX_RELEASE(pCCF);
	}
    }

    return res;
}
BOOL TheoraPageToPacket::IsTheoraHeader(ogg_packet* pPkt)
{
    BOOL bRet = FALSE;

    if (pPkt && pPkt->packet && (pPkt->bytes > 7) &&
        ((pPkt->packet[0] & 0x80) == 0x80) &&
        !memcmp(pPkt->packet + 1, "theora", 6))
        
    {
        bRet = TRUE;
    }

    return bRet;
}
