{- Copyright (c) 2007 John Goerzen <jgoerzen@complete.org>
   Please see the COPYRIGHT file -}

import System.Console.GetOpt
import System.Console.GetOpt.Utils
import Utils
import HSH
import Text.Printf
import Control.Monad
import System.Path
import Types
import System.Path

helpinfo = "Usage:\n\
\hg-importorig [-v] [-s n] orig-name package version\n\
\\n\
\Imports the orig-name into ../${package}.upstream\n\
\\n\
\Where:\n\
\\n\
\  orig-name is the name of a tar.gz, tar.bz2, zip, or directory\n\
\            to import\n\
\\n\
\  package is the name of the Debian source package\n\
\\n\
\  version is the upstream version being imported\n"

crashusage = die 5 $ usageInfo helpinfo importargs

main = 
  do (args, files) <- parseCmdLine RequireOrder importargs helpinfo
     (origname, package, version) <- case files of
                                           [o, p, v] -> return (o, p, v)
                                           _ -> crashusage
     let tag = Tag "UPSTREAM" package version
     let isverbose = (Verbose `elem` args)
     when isverbose setverbose
     let verbargs = if isverbose then ["-v"] else []
     upstreamrepo <- readlinkabs (printf "../%s.upstream" package)
     assertvalidhg "."
     assertvalidhg upstreamrepo

     v $ printf "Original name %s, package %s, version %s" origname package
         version

     validtag <- bracketCWD upstreamrepo (hastag tag)
     when (validtag) $ die 0 $ printf 
          "Version %s already exists in upstream repo; skipping import" version
     
     v $ printf "Importing %s into %s..." origname upstreamrepo
     bracketCWD upstreamrepo (istagok tag) >>= 
            (flip unless) (die 1 $ printf "%s is not newer than all upstreams"
                                 version)
     
     case [s | S s <- args] of
       [] -> runIO $ ("hg_load_dirs", verbargs ++ ["--wc=" ++ upstreamrepo,
              printf "--summary=Import upstream %s version %s" package version,
              origname])
       (x:_) -> do runIO $ ("hg_load_dirs", verbargs ++ 
                            ["--wc=" ++ upstreamrepo, "-n", "-f", origname])
                   bracketCWD upstreamrepo $ do
                     runIO $ ("hg", verbargs ++ ["addremove", "-s", x])
                     runIO $ ("hg", verbargs ++ ["commit", "-m",
                      printf "Import upstream %s version %s" package version])
     bracketCWD upstreamrepo $ runIO $ ("hg", ["tag", show tag])
     bracketCWD upstreamrepo $ runIO $ ("hg", ["tag", show tag ++ "_TAG"])
     putStrLn ""
     printf "%s imported into %s\n" origname upstreamrepo
     printf "Remember to run hg fetch %s\n" upstreamrepo
