/***************************************************************************
                          highlightmodule.cpp  -  description
                             -------------------
    begin                : 07.06.2004
    copyright            : (C) 2004 by Andre Simon
    email                : andre.simon1@gmx.de

  Python 2.3 binding for highlight
   
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


#include <Python.h>
#include <string>
#include "structmember.h"
#include "codegenerator.h"
#include "htmlgenerator.h"
#include "xhtmlgenerator.h"
#include "rtfgenerator.h"
#include "latexgenerator.h"
#include "texgenerator.h"
#include "xslfogenerator.h"
#include "enums.h"
#include "version.h"


using namespace highlight;


typedef struct {
    PyObject_HEAD
    /* Type-specific fields go here. */
    CodeGenerator *generator;
    OutputType outputFormat;
} highlight_generatorObject;


static void
Highlight_dealloc(highlight_generatorObject* self)
{

    CodeGenerator::deleteInstance();
    self->ob_type->tp_free((PyObject*)self);
}

/*static PyObject *
Highlight_new(PyTypeObject *type, PyObject *args, PyObject *kwds)
{
std::cout << "__new__\n";
    highlight_generatorObject *self;

    self = (highlight_generatorObject *)type->tp_alloc(type, 0);
    if (self != NULL) {
        
    }

    return (PyObject *)self;
}*/

static int
Highlight_init(highlight_generatorObject *self, PyObject *args, PyObject *kwds)
{
    static char *kwlist[] = {"outputformat","stylepath", 
                             "cssinfile","cssoutfile", "includecss", "attachanchors",
                             "fopcompatible", "replacequotes", 
			     "linenumbers","fillzeroes", 
			     "fragment", "numberspaces",
                             "wrappingstyle", 
			     NULL};
				
    self->generator=NULL;
    const char * cssInfile="";
    const char * cssOutfile="highlight.css";
    const char * stylePath="";
    int includeCss=0, attachAnchors=0, fopCompatible=0, replaceQuotes=0;	
    int printLineNumbers=0, fragment=0, fillZeroes=0, numberSpaces=2;        
    WrapMode wrappingStyle=WRAP_DISABLED;			
              
    if (!PyArg_ParseTupleAndKeywords(args, kwds, "is|ssiiiiiiiii",kwlist, 
                                     &self->outputFormat,
                                     &stylePath, 
				     &cssInfile, 
                                     &cssOutfile, 
				     &includeCss, 
				     &attachAnchors, 
				     &fopCompatible, 
				     &replaceQuotes,
    				     &printLineNumbers, 
				     &fillZeroes,
				     &fragment, 
				     &numberSpaces,
				     (int*)&wrappingStyle )) {
         return -1;
     }
	
     self->generator = CodeGenerator::getInstance(self->outputFormat,
                                                  stylePath,
						  string(cssInfile),
                                                  string(cssOutfile),
	       	                                  includeCss,
 		                                  attachAnchors,
                                                  replaceQuotes,
                                                  fopCompatible,
			                          numberSpaces,
                                                  wrappingStyle,
                                                  printLineNumbers, 
                                                  fillZeroes, 
              				          fragment);	
	    
    return 0;
}

/** 
   @param infile          String, Input source file (uses stdin if empty)
   @param outfile         String, Destination file (uses stdout if empty)
   @param linenumbers     bool, Set to true if line numbers should be printed; default:0
   @param fillzeroes      bool, Set to true if leading space of line numbers should be filled with 0's; default:0
   @param fragment        bool, Set to true if document header and footer should be omitted; default:0
   @param numberspace     integer, Number of spyce that replace a tab; values: 0..8; default: 2
   @param wrappingstyle   integer, Values: WRAP_DISABLED, WRAP_SIMPLE, WRAP_DEFAULT
   @return highlight.PARSE_OK,           success
           highlight.PARSE_BAD_INPUT,    could not write inFile 
           highlight.PARSE_BAD_OUTPUT,   could not write outFile 
           highlight.PARSE_BAD_STYLE     stylePath not found
           highlight.MODULE_ERROR:       function call failed 
*/
static PyObject *
Highlight_PrintOutput(PyObject *self, PyObject *args, PyObject *kwds)
{   

    static char *kwlist[] = { "infile", "outfile", 
                              NULL};

    const char* infile="";
    const char* outfile="";

    if (!PyArg_ParseTupleAndKeywords(args, kwds, "ss",kwlist,
                                     &infile, 
				     &outfile)){
        return NULL;
    }
    highlight_generatorObject* genObject = (highlight_generatorObject *)self;

    if (genObject->generator!=NULL){
       ParseError err = genObject->generator->printOutput(string(infile), string(outfile));       
       return Py_BuildValue("i",(int)err);       
    }  
    return Py_BuildValue("i", -1);
}
	


/** int getOutputFormat()
   @return output format
*/
static PyObject *
Highlight_GetOutputFormat(PyObject *self, PyObject *args)
{     
    return Py_BuildValue("i", ((highlight_generatorObject *)self)->outputFormat);
}



/** int initLanguage(string langDef)
   @param langDef Path of highlight language definition
   @return  highlight.LOAD_FAILED: definition not found   
            highlight.LOAD_NEW:    definition loaded successfully   
            highlight.LOAD_NONE:   reload not necessary
            highlight.MODULE_ERROR: function call failed 
*/

static PyObject *
Highlight_initLanguage(PyObject *self, PyObject *args)
{   
    const char* langdef="";

    if (!PyArg_ParseTuple(args, "s", &langdef))
        return NULL;

    highlight_generatorObject* genObject = (highlight_generatorObject *)self; 
    if (genObject->generator!=NULL){
       LoadResult loadRes= genObject->generator->initLanguage(langdef);  
       return Py_BuildValue("i",(int)loadRes);       
    }  
    return Py_BuildValue("i", -1);
}


/** bool initIndentationScheme(string indentationScheme)
   @param indentationScheme Path of highlight indent scheme file
   @return  true : success
            false: failure
*/

static PyObject *
Highlight_initIndentation(PyObject *self, PyObject *args)
{   
    const char* indentationScheme="";

    if (!PyArg_ParseTuple(args, "s", &indentationScheme))
        return NULL;
	
    highlight_generatorObject* genObject = (highlight_generatorObject *)self; 
    if (genObject->generator!=NULL){
       bool loadRes= genObject->generator->initIndentationScheme(indentationScheme);  
       return Py_BuildValue("i",(int)loadRes);       
    }  
    return Py_BuildValue("i", -1);
}





/** bool printCSSFile(string cssIn, string cssOut)
   @param cssIn File to be included in cssOut 
   @param cssOut Destination css file
   @return True if successful
*/
static PyObject *
Highlight_PrintExternalStyle(PyObject *self, PyObject *args)
{   
    const char* cssoutfile="";

    if (!PyArg_ParseTuple(args, "s",  &cssoutfile ))
        return NULL;

    highlight_generatorObject* genObject = (highlight_generatorObject *)self; 
    if (genObject->generator!=NULL){
       int currentFormat=genObject->outputFormat;
       if (currentFormat!=HTML && currentFormat!=XHTML){
          return Py_BuildValue("i", 0);
       }
       int res = genObject->generator->printExternalStyle(string(cssoutfile));       
       return Py_BuildValue("i",res);       
    }  
    return Py_BuildValue("i", -1);
}



static PyMethodDef highlight_methods[] = {
    {"getOutputFormat",      (PyCFunction)Highlight_GetOutputFormat,    METH_NOARGS,  "Get current output format"},
    {"initLanguage",         (PyCFunction)Highlight_initLanguage,       METH_VARARGS, "Set language definition."},
    {"initIndentationScheme",(PyCFunction)Highlight_initIndentation,    METH_VARARGS, "Init indentation and reformatting mode."},
    {"printOutput",          (PyCFunction)Highlight_PrintOutput,        METH_VARARGS | METH_KEYWORDS, "Generate output."},
    {"printExternalStyle",   (PyCFunction)Highlight_PrintExternalStyle, METH_VARARGS, "Generate external style file."},
    {NULL}  
};


static PyTypeObject highlight_generator = {
    PyObject_HEAD_INIT(NULL)
    0,                         /*ob_size*/
    "highlight.CodeGenerator",             /*tp_name*/
    sizeof(highlight_generatorObject), /*tp_basicsize*/
    0,                         /*tp_itemsize*/
     (destructor)Highlight_dealloc,                         /*tp_dealloc*/
    0,                         /*tp_print*/
    0,                         /*tp_getattr*/
    0,                         /*tp_setattr*/
    0,                         /*tp_compare*/
    0,                         /*tp_repr*/
    0,                         /*tp_as_number*/
    0,                         /*tp_as_sequence*/
    0,                         /*tp_as_mapping*/
    0,                         /*tp_hash */
    0,                         /*tp_call*/
    0,                         /*tp_str*/
    0,                         /*tp_getattro*/
    0,                         /*tp_setattro*/
    0,                         /*tp_as_buffer*/
    Py_TPFLAGS_DEFAULT,        /*tp_flags*/
    "Highlight objects",           /* tp_doc */
    0,		               /* tp_traverse */
    0,		               /* tp_clear */
    0,		               /* tp_richcompare */
    0,		               /* tp_weaklistoffset */
    0,		               /* tp_iter */
    0,		               /* tp_iternext */
    highlight_methods,             /* tp_methods */
    0,//Noddy_members,             /* tp_members */
    0,                         /* tp_getset */
    0,                         /* tp_base */
    0,                         /* tp_dict */
    0,                         /* tp_descr_get */
    0,                         /* tp_descr_set */
    0,                         /* tp_dictoffset */
    (initproc)Highlight_init,      /* tp_init */
    0,                         /* tp_alloc */
    0//Highlight_new                 /* tp_new */
};


static void addIntConstant(PyObject *m,  char* name, int val){
   PyObject *tmp, *d;
    d = PyModule_GetDict(m);

    tmp = Py_BuildValue("i", val);
    PyDict_SetItemString(d, name, tmp);
    Py_DECREF(tmp);
}

static void addStrConstant(PyObject *m,  char* name, char* val){
   PyObject *tmp, *d;
    d = PyModule_GetDict(m);

    tmp = Py_BuildValue("s", val);
    PyDict_SetItemString(d, name, tmp);
    Py_DECREF(tmp);
}


#ifndef PyMODINIT_FUNC	/* declarations for DLL import/export */
#define PyMODINIT_FUNC void
#endif
PyMODINIT_FUNC
inithighlight(void) 
{
    PyObject* m;

    highlight_generator.tp_new = PyType_GenericNew;
    if (PyType_Ready(&highlight_generator) < 0)
        return;

    m = Py_InitModule3("highlight", highlight_methods,
                       "A module to highlight and indent various source code.");

    addIntConstant(m, "HTML", HTML);
    addIntConstant(m, "XHTML", XHTML);
    addIntConstant(m, "TEX", TEX);
    addIntConstant(m, "LATEX", LATEX);
    addIntConstant(m, "XSL", XSLFO);
    addIntConstant(m, "RTF", RTF);
    addIntConstant(m, "ANSI", ANSI);
    addStrConstant(m, "VERSION", HIGHLIGHT_VERSION);

    
    addIntConstant(m, "WRAP_DISABLED", (int)WRAP_DISABLED );
    addIntConstant(m, "WRAP_SIMPLE", (int)WRAP_SIMPLE );
    addIntConstant(m, "WRAP_DEFAULT", (int)WRAP_DEFAULT );
    
    addIntConstant(m, "LOAD_FAILED", (int) LOAD_FAILED);
    addIntConstant(m, "LOAD_NEW", (int) LOAD_NEW);
    addIntConstant(m, "LOAD_NONE", (int)LOAD_NONE );
    
    addIntConstant(m, "PARSE_OK", (int)PARSE_OK );
    addIntConstant(m, "PARSE_BAD_INPUT", (int)BAD_INPUT );
    addIntConstant(m, "PARSE_BAD_OUTPUT", (int) BAD_OUTPUT);
    addIntConstant(m, "PARSE_BAD_STYLE", (int)BAD_STYLE );
    
    addIntConstant(m, "MODULE_ERROR", -1 );

    Py_INCREF(&highlight_generator);
    PyModule_AddObject(m, "CodeGenerator", (PyObject *)&highlight_generator);
}
