/*
 *  JLib - Jacob's Library.
 *  Copyright (C) 2003, 2004  Juan Carlos Seijo Prez
 * 
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 * 
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 * 
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 *  Juan Carlos Seijo Prez
 *  jacob@mainreactor.net
 */

/** Imagen genrica
 * @file    JImage.cpp.
 * @author  Juan Carlos Seijo Prez
 * @date    14/10/2003
 * @version 0.0.1 - 14/10/2003 - Primera varsin.
 * @version 0.0.2 - 01/06/2004 - Adicin de mtodos de referencia y copia.
 */

#include <JLib/Graphics/JImage.h>

// Constructor
JImage::JImage() : surface(0)
{
}

JImage::JImage(u32 w, u32 h, u32 _bpp)
: surface(0)
{
  Create(w, h, _bpp);
}

// Constructor copia
JImage::JImage(JImage &img)
{
  surface = SDL_ConvertSurface(img.Surface(), img.Surface()->format, img.Surface()->flags);
}

// Carga la imagen desde fichero. Soporta '.tga', '.bmp', '.jpg' y otros
bool JImage::Load(const JString &filename, bool toDisplayFormat, u32 cKey)
{
  surface = IMG_Load(filename);
  if (surface != 0)
  {
    if (toDisplayFormat)
    {
      SDL_Surface *surf = SDL_DisplayFormat(surface);

      if (0 != surf)
      {
        SDL_FreeSurface(surface);
        surface = surf;
        ColorKey(cKey);
      }
    }

    return true;
  }

  return false;
}

// Crea la superficie vaca
// Si data no es cero carga una copia en la superficie.
bool  JImage::Create(u32 w, u32 h, u32 _bpp, void *data, u32 rMask, u32 gMask, u32 bMask, u32 aMask)
{
  Destroy();

  if (_bpp == 0)
  {
    _bpp = SDL_GetVideoSurface()->format->BitsPerPixel;
  }

  if (rMask == 0 &&
      gMask == 0 &&
      bMask == 0 &&
      aMask == 0)
  {
    rMask = SDL_GetVideoSurface()->format->Rmask;
    gMask = SDL_GetVideoSurface()->format->Gmask;
    bMask = SDL_GetVideoSurface()->format->Bmask;
    aMask = SDL_GetVideoSurface()->format->Amask;
  }

  // Si hay datos los carga a una superficie temporal...
  if (data)
  {
    SDL_Surface *tmp;
    tmp = SDL_CreateRGBSurfaceFrom(data, w, h, _bpp, _bpp/8 * w,
                                   rMask, gMask, bMask, aMask);

    if (tmp == 0)
    {
      printf("JImage::Create(): Error al crear tmp!\n");
      return false;
    }

    surface = SDL_DisplayFormat(tmp);

    SDL_FreeSurface(tmp);
    if(surface == 0)
    {
      fprintf(stderr, "CreateRGBSurface failed: %s\n", SDL_GetError());
      return false;
    }
		
		return true;
  }
  else
  {
    surface = SDL_CreateRGBSurface(SDL_SWSURFACE, w, h, _bpp,
                                  rMask, gMask, bMask, aMask);
    if(surface == 0)
    {
      fprintf(stderr, "CreateRGBSurface failed: %s\n", SDL_GetError());
      return false;
    }
		
		Fill(ColorKey());
  }
  
  return true;
}

// Dibuja la imagen
void JImage::Draw()
{
  SDL_Rect rc;
  rc.x = (s16)pos.x;
  rc.y = (s16)pos.y;
  rc.w = (u16)surface->w;
  rc.h = (u16)surface->h;

  SDL_BlitSurface(surface, 0, SDL_GetVideoSurface(), &rc);
}

// Dibuja la imagen en la posicin dada.
void JImage::Draw(s32 x, s32 y)
{
  SDL_Rect rc;
  rc.x = (s16)x;
  rc.y = (s16)y;
  rc.w = (u16)surface->w;
  rc.h = (u16)surface->h;
  
  SDL_BlitSurface(surface, 0, SDL_GetVideoSurface(), &rc);
}

// Pega el contenido de una imagen en esta.
bool JImage::Paste(JImage *srcImg, s32 xSrc, s32 ySrc, s32 wSrc, s32 hSrc, s32 xDst, s32 yDst)
{
  SDL_Rect srcRc, dstRc;
  srcRc.x = (s16)xSrc;
  srcRc.y = (s16)ySrc;
  srcRc.w = (u16)wSrc;
  srcRc.h = (u16)hSrc;
  dstRc.x = (s16)xDst;
  dstRc.y = (s16)yDst;
  dstRc.w = (u16)wSrc;
  dstRc.h = (u16)hSrc;
  
  return 0 == SDL_BlitSurface(srcImg->surface, &srcRc, surface, &dstRc);
}

// Devuelve el valor del pixel en la posicin dada.
// La imagen debe estar bloqueada.
u32 JImage::GetPixel(s32 x, s32 y)
{
  s32 bpp = surface->format->BytesPerPixel;
  u8 *p = (u8 *)surface->pixels + y * surface->pitch + x * bpp;

  switch(bpp)
  {
    case 1:
      return *p;

    case 2:
      return *(u16 *)p;

    case 3:
      if(SDL_BYTEORDER == SDL_BIG_ENDIAN)
      {
				return p[0] << 16 | p[1] << 8 | p[2];
      }
      else
      {
				return p[0] | p[1] << 8 | p[2] << 16;
      }

    case 4:
      return *(u32 *)p;

    default:
      return 0;
  }
}

// Establece el valor del pixel en la posicin dada.
// La imagen debe estar bloqueada.
void JImage::PutPixel(s32 x, s32 y, u32 color)
{
  s32 bpp = surface->format->BytesPerPixel;
  u8 *p = (u8 *)surface->pixels + y * surface->pitch + x * bpp;

  switch(bpp)
  {
    case 1:
      *p = (u8)color;
      break;

    case 2:
      *(u16 *)p = (u16)color;
      break;

    case 3:
      if(SDL_BYTEORDER == SDL_BIG_ENDIAN) {
          p[0] = ((u8)(color >> 16)) & 0xff;
          p[1] = ((u8)(color >> 8)) & 0xff;
          p[2] = ((u8)color) & 0xff;
      } else {
          p[0] = ((u8)color) & 0xff;
          p[1] = ((u8)(color >> 8)) & 0xff;
          p[2] = ((u8)(color >> 16)) & 0xff;
      }
      break;

    case 4:
      *(u32 *)p = color;
      break;
  }
}

// Guarda la imagen como BMP
bool JImage::SaveAsBMP(const JString &file)
{
  return (0 == SDL_SaveBMP(surface, file));
}

// Carga la imagen.
u32 JImage::Load(JFile &f)
{
	Destroy();

  s32 w, h;
  u32 ckey, sz, Rmask, Gmask, Bmask, Amask;
  u8 bpp, *data;
  if (f.ReadLE32(&w) &&
      f.ReadLE32(&h) &&
      f.Read(&bpp, 1) &&
			f.ReadLE32(&Rmask) &&
			f.ReadLE32(&Gmask) &&
			f.ReadLE32(&Bmask) &&
			f.ReadLE32(&Amask) &&
      f.ReadLE32(&ckey) &&
      f.ReadLE32(&sz))
  {
    data = new u8[sz];
		
		int count;
		
		switch (bpp)
		{
		case 8:
			if (f.Read(data, sz))
			{
				if (Create(w, h, bpp, data, Rmask, Gmask, Bmask, Amask))
				{
					ColorKey(ckey);
					
					delete[] data;
					
					return 0;
				}
			}
			break;
			
		case 16:
			for (s32 i = 0; i < sz; i += 2)
			{
				if (!f.ReadLE16(&data[i]))
				{
					delete[] data;
					return 2;
				}
			}
			
			if (Create(w, h, bpp, data, Rmask, Gmask, Bmask, Amask))
			{
				ColorKey(ckey);
				
				delete[] data;
				
				return 0;
			}
			break;			

		case 32:
			for (s32 i = 0; i < sz; i += 4)
			{
				if (!f.ReadLE32(&data[i]))
				{
					delete[] data;
					return 3;
				}
			}
			
			if (Create(w, h, bpp, data, Rmask, Gmask, Bmask, Amask))
			{
				ColorKey(ckey);
				
				delete[] data;
				
				return 0;
			}
			break;			
		}

		delete[] data;
  }

  return 1;
}

// Salva la imagen.
u32 JImage::Save(JFile &f)
{
  u32 sz = Size();
  if (f.WriteLE32(&surface->w) &&
      f.WriteLE32(&surface->h) &&
      f.Write(&surface->format->BitsPerPixel, 1) &&
      f.WriteLE32(&surface->format->Rmask) &&
      f.WriteLE32(&surface->format->Gmask) &&
      f.WriteLE32(&surface->format->Bmask) &&
      f.WriteLE32(&surface->format->Amask) &&
      f.WriteLE32(&surface->format->colorkey) &&
      f.WriteLE32(&sz))
  {
    if (0 == Lock())
    {
			u8 *data;
			s32 W = surface->w * BytesPP();

			switch (surface->format->BitsPerPixel)
			{
			default:
			case 8:
				// Salva cada lnea, pues hemos de tener en cuenta el pitch en memoria
				for (s32 j = 0; j < surface->h; ++j)
				{
					data = Line(j);
					if (0 == f.Write(data, W))
					{
						// Error E/S
						Unlock();
						return 1;
					}
				}
				break;

			case 16:
				// Salva cada lnea, pues hemos de tener en cuenta el pitch en memoria
				for (s32 j = 0; j < surface->h; ++j)
				{
					data = Line(j);
					
					for (s32 i = 0; i < W; i += 2)
					{
						if (0 == f.WriteLE16(&data[i]))
						{
							// Error E/S
							Unlock();
							return 1;
						}
					}
				}
				break;

			case 32:
				// Salva cada lnea, pues hemos de tener en cuenta el pitch en memoria
				for (s32 j = 0; j < surface->h; ++j)
				{
					data = Line(j);
					
					for (s32 i = 0; i < W; i += 4)
					{
						if (0 == f.WriteLE32(&data[i]))
						{
							// Error E/S
							Unlock();
							return 1;
						}
					}
				}
				break;
			}
      
			Unlock();
      
      return 0;
    }
  }

  return 2;
}

JImage * JImage::Scale(float xp, float yp)
{
	if (-1 == Lock())
	{
		return 0;
	}

	s32 newW = (s32)(surface->w * xp), newH = (s32)(surface->h * yp);
	s32 w = surface->w, h = surface->h;
	
	float dx = 1.0f/xp, dy = 1.0f/yp;                // Tamao de 1 pixel original en coordenadas de la nueva imagen
	

	float sNew = 1.0f/(xp * yp);                     // Superficie de un punto de la nueva imagen en coordenadas de la imagen original
	float sOrg = xp * yp;                            // Superficie de un punto de la imagen original en coordenadas de la imagen nueva
	SDL_Color samp;                                  // Muestra actual

	JImage *img = new JImage(newW, newH, surface->format->BitsPerPixel);
	
	if (-1 == img->Lock())
	{
		return 0;
	}

	// Recorre los puntos de la nueva imagen
	float nx0, ny0;                                  // Primer pixel interviniente en la imagen original para un pixel de la nueva
	float rAccum, gAccum, bAccum, aAccum;            // Acumulados de las componentes de la imagen
	float xSize, ySize, ratio, x1, x2, y1, y2;

	for (s32 j = 0; j < newH; ++j)
	{
		for (s32 i = 0; i < newW; ++i)
		{
			nx0 = float(i)/xp;
			ny0 = float(j)/yp;
			rAccum = gAccum = bAccum = aAccum = 0.0f;

			for (float ny = ny0; ny < ny0 + dy; ++ny)     // Acumula los pxeles intervinientes de la imagen original en el nuevo pixel
			{
				for (float nx = nx0; nx < nx0 + dx; ++nx)
				{
					SDL_GetRGBA(GetPixel((s32)nx, (s32)ny), surface->format, &samp.r, &samp.g, &samp.b, &samp.unused);
					
					// Averigua la cuota de participacin del pixel.
					// Para ello vemos el cociente entre la superficie de pixel original en la nueva imagen
					// y la supeficie total de un pixel en unidades de la nueva imagen
					x1 = JMax(nx * xp, i);
					x2 = JMin((nx + 1) * xp, (i + 1));
					y1 = JMax(ny * yp, j);
					y2 = JMin((ny + 1) * yp, (j + 1));
					
					xSize = x2 - x1;
					ySize = y2 - y1;
					ratio = (xSize * ySize);
					
					rAccum += float(samp.r) * ratio;
					gAccum += float(samp.g) * ratio;
					bAccum += float(samp.b) * ratio;
					aAccum += float(samp.unused) * ratio;
				}
			}

			// Finalmente aade el pixel a la nueva imagen
			img->PutPixel(i, j, SDL_MapRGBA(surface->format, (u8)rAccum, (u8)gAccum, (u8)bAccum, (u8)aAccum));
		}
	}
	
	Unlock();
	img->Unlock();

	return img;
}

// Destructor
void JImage::Destroy()
{
	if (surface != 0)
  {
    SDL_FreeSurface(surface);
    surface = 0;
  }
}

void JImage::Ref(JImage &img)
{
	Destroy();

	/**< @todo Quitar esto cuando SDL soporte crear referencias con alguna funcin. */
  surface = img.surface;

	if (surface != 0)
	{
		++surface->refcount;
		pos = img.Pos();
	}
}

void JImage::operator =(JImage &img)
{
	Destroy();

	/**< @todo Quitar esto cuando SDL soporte crear referencias con alguna funcin. */
  surface = img.surface;

	if (surface != 0)
	{
		++surface->refcount;
		pos = img.Pos();
	}
}

void JImage::Copy(JImage &img)
{
	Destroy();
	surface = SDL_CreateRGBSurfaceFrom(img.Pixels(), img.Width(), img.Height(), img.BitsPP(), img.Surface()->pitch, 
																		 img.Format()->Rmask, img.Format()->Gmask, img.Format()->Bmask, img.Format()->Amask);

}

bool JImage::Convert(SDL_PixelFormat *fmt, u32 flags)
{
	SDL_Surface *tmp;
  if (0 != (tmp = SDL_ConvertSurface(surface, fmt, flags)))
	{
		Destroy();
		surface = tmp;
		
		return true;
	}
	
	return false;
}
