/*
 * hostapd / EAP-SIM database/authenticator gateway
 * Copyright (c) 2005, Jouni Malinen <jkmaline@cc.hut.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * Alternatively, this software may be distributed under the terms of BSD
 * license.
 *
 * See README and COPYING for more details.
 *
 * This is an example implementation of the EAP-SIM database/authentication
 * gateway interface that is expected to be replaced with an implementation of
 * SS7 gateway to GSM authentication center (HLR/AuC) or a local
 * implementation of SIM triplet generator.
 *
 * The example implementation here reads triplets from a text file in
 * IMSI:Kc:SRES:RAND format, IMSI in ASCII, other fields as hex strings. This
 * is used to simulate an HLR/AuC. As such, it is not very useful for real life
 * authentication, but it is useful both as an example implementation and for
 * EAP-SIM testing.
 */

#include "includes.h"

#include "common.h"
#include "eap_sim_common.h"
#include "eap_sim_db.h"

#ifdef TEST_PENDING_GET
#include "eloop.h"
#endif /* TEST_PENDING_GET */

struct eap_sim_pseudonym {
	struct eap_sim_pseudonym *next;
	u8 *identity;
	size_t identity_len;
	char *pseudonym;
};

struct eap_sim_db_data {
	char *fname;
	void (*get_complete_cb)(void *ctx, void *session_ctx);
	void *ctx;
	struct eap_sim_pseudonym *pseudonyms;
	struct eap_sim_reauth *reauths;
};

#define KC_LEN 8
#define SRES_LEN 4
#define RAND_LEN 16


/**
 * eap_sim_db_init - Initialize EAP-SIM DB / authentication gateway interface
 * @config: Configuration data (e.g., file name)
 * @get_complete_cb: Callback function for reporting availability of triplets
 * @ctx: Context pointer for get_complete_cb
 * Returns: Pointer to a private data structure or %NULL on failure
 */
void * eap_sim_db_init(const char *config,
		       void (*get_complete_cb)(void *ctx, void *session_ctx),
		       void *ctx)
{
	struct eap_sim_db_data *data;

	data = malloc(sizeof(*data));
	if (data == NULL) {
		return NULL;
	}

	memset(data, 0, sizeof(*data));
	data->get_complete_cb = get_complete_cb;
	data->ctx = ctx;
	data->fname = strdup(config);
	if (data->fname == NULL) {
		free(data);
		return NULL;
	}

	return data;
}


static void eap_sim_db_free_pseudonym(struct eap_sim_pseudonym *p)
{
	free(p->identity);
	free(p->pseudonym);
	free(p);
}


static void eap_sim_db_free_reauth(struct eap_sim_reauth *r)
{
	free(r->identity);
	free(r->reauth_id);
	free(r);
}


/**
 * eap_sim_db_deinit - Deinitialize EAP-SIM DB/authentication gw interface
 * @priv: Private data pointer from eap_sim_db_init()
 */
void eap_sim_db_deinit(void *priv)
{
	struct eap_sim_db_data *data = priv;
	struct eap_sim_pseudonym *p, *prev;
	struct eap_sim_reauth *r, *prevr;
	free(data->fname);
	p = data->pseudonyms;
	while (p) {
		prev = p;
		p = p->next;
		eap_sim_db_free_pseudonym(prev);
	}
	r = data->reauths;
	while (r) {
		prevr = r;
		r = r->next;
		eap_sim_db_free_reauth(prevr);
	}
	free(data);
}


#ifdef TEST_PENDING_GET
static void eap_sim_db_get_complete(void *eloop_ctx, void *timeout_ctx)
{
	struct eap_sim_db_data *data = eloop_ctx;
	if (data->get_complete_cb)
		data->get_complete_cb(data->ctx, timeout_ctx);
}
#endif /* TEST_PENDING_GET */


/**
 * eap_sim_db_get_gsm_triplets - Get GSM triplets
 * @priv: Private data pointer from eap_sim_db_init()
 * @identity: User name identity
 * @identity_len: Length of identity in bytes
 * @max_chal: Maximum number of triplets
 * @rand: Buffer for RAND values
 * @kc: Buffer for Kc values
 * @sres: Buffer for SRES values
 * @cb_session_ctx: Session callback context for get_complete_cb()
 * Returns: Number of triplets received (has to be less than or equal to
 * max_chal), -1 (EAP_SIM_DB_FAILURE) on error (e.g., user not found), or
 * -2 (EAP_SIM_DB_PENDING) if results are not yet available. In this case, the
 * callback function registered with eap_sim_db_init() will be called once the
 * results become available.
 *
 * In most cases, the user name is '1' | IMSI, i.e., 1 followed by the IMSI in
 * ASCII format.
 */
int eap_sim_db_get_gsm_triplets(void *priv, const u8 *identity,
				size_t identity_len, int max_chal,
				u8 *rand, u8 *kc, u8 *sres,
				void *cb_session_ctx)
{
	struct eap_sim_db_data *data = priv;
	FILE *f;
	int count, i;
	char buf[80], *pos, *next;

#ifdef TEST_PENDING_GET
	{
		/*
		 * Example code for testing pending get request. When using an
		 * external server for GSM triplets, this function can always
		 * start a request and return EAP_SIM_DB_PENDING immediately if
		 * authentication triplets are not available. Once the triplets
		 * are received, callback function registered with
		 * eap_sim_db_init() is called to notify EAP state machine to
		 * reprocess the message. This eap_sim_db_get_gsm_triplets()
		 * function will then be called again and the newly received
		 * triplets will then be given to the caller.
		 */
		static int test_pending = 1;
		if (test_pending) {
			test_pending = 0;
			eloop_register_timeout(1, 0, eap_sim_db_get_complete,
					       data, cb_session_ctx);
			return EAP_SIM_DB_PENDING;
		}
	}
#endif /* TEST_PENDING_GET */

	f = fopen(data->fname, "r");
	if (f == NULL) {
		wpa_printf(MSG_DEBUG, "EAP-SIM DB: could not open triplet "
			   "file '%s'", data->fname);
		return EAP_SIM_DB_FAILURE;
	}

	if (identity_len < 2 || identity[0] != EAP_SIM_PERMANENT_PREFIX) {
		wpa_hexdump_ascii(MSG_DEBUG, "EAP-SIM DB: unexpected identity",
				  identity, identity_len);
		fclose(f);
		return EAP_SIM_DB_FAILURE;
	}
	identity++;
	identity_len--;
	for (i = 0; i < identity_len; i++) {
		if (identity[i] == '@') {
			identity_len = i;
			break;
		}
	}
	wpa_hexdump_ascii(MSG_DEBUG, "EAP-SIM DB: get triplets for IMSI",
			  identity, identity_len);

	count = 0;
	while (count < max_chal && fgets(buf, sizeof(buf), f)) {
		/* Parse IMSI:Kc:SRES:RAND and match IMSI with identity. */
		buf[sizeof(buf) - 1] = '\0';
		pos = buf;
		while (*pos != '\0' && *pos != '\n')
			pos++;
		if (*pos == '\n')
			*pos = '\0';
		if (pos - buf < 60 || pos[0] == '#')
			continue;

		pos = strchr(buf, ':');
		if (pos == NULL)
			continue;
		*pos++ = '\0';
		if (strlen(buf) != identity_len ||
		    memcmp(buf, identity, identity_len) != 0)
			continue;

		next = strchr(pos, ':');
		if (next == NULL)
			continue;
		*next++ = '\0';
		if (hexstr2bin(pos, &kc[count * KC_LEN], KC_LEN) < 0)
			continue;

		pos = next;
		next = strchr(pos, ':');
		if (next == NULL)
			continue;
		*next++ = '\0';
		if (hexstr2bin(pos, &sres[count * SRES_LEN], SRES_LEN) < 0)
			continue;

		if (hexstr2bin(next, &rand[count * RAND_LEN], RAND_LEN) < 0)
			continue;

		count++;
	}

	fclose(f);

	if (count == 0) {
		wpa_printf(MSG_DEBUG, "EAP-SIM DB: no triplets found");
		count = EAP_SIM_DB_FAILURE;
	}

	return count;
}


static struct eap_sim_pseudonym *
eap_sim_db_get_pseudonym(struct eap_sim_db_data *data, const u8 *identity,
			 size_t identity_len)
{
	char *pseudonym;
	size_t len;
	struct eap_sim_pseudonym *p;

	if (identity_len == 0 ||
	    (identity[0] != EAP_SIM_PSEUDONYM_PREFIX &&
	     identity[0] != EAP_AKA_PSEUDONYM_PREFIX))
		return NULL;

	/* Remove possible realm from identity */
	len = 0;
	while (len < identity_len) {
		if (identity[len] == '@')
			break;
		len++;
	}

	pseudonym = malloc(len + 1);
	if (pseudonym == NULL)
		return NULL;
	memcpy(pseudonym, identity, len);
	pseudonym[len] = '\0';

	p = data->pseudonyms;
	while (p) {
		if (strcmp(p->pseudonym, pseudonym) == 0)
			break;
		p = p->next;
	}

	free(pseudonym);

	return p;
}


static struct eap_sim_pseudonym *
eap_sim_db_get_pseudonym_id(struct eap_sim_db_data *data, const u8 *identity,
			    size_t identity_len)
{
	struct eap_sim_pseudonym *p;

	if (identity_len == 0 ||
	    (identity[0] != EAP_SIM_PERMANENT_PREFIX &&
	     identity[0] != EAP_AKA_PERMANENT_PREFIX))
		return NULL;

	p = data->pseudonyms;
	while (p) {
		if (identity_len == p->identity_len &&
		    memcmp(p->identity, identity, identity_len) == 0)
			break;
		p = p->next;
	}

	return p;
}


static struct eap_sim_reauth *
eap_sim_db_get_reauth(struct eap_sim_db_data *data, const u8 *identity,
		      size_t identity_len)
{
	char *reauth_id;
	size_t len;
	struct eap_sim_reauth *r;

	if (identity_len == 0 ||
	    (identity[0] != EAP_SIM_REAUTH_ID_PREFIX &&
	     identity[0] != EAP_AKA_REAUTH_ID_PREFIX))
		return NULL;

	/* Remove possible realm from identity */
	len = 0;
	while (len < identity_len) {
		if (identity[len] == '@')
			break;
		len++;
	}

	reauth_id = malloc(len + 1);
	if (reauth_id == NULL)
		return NULL;
	memcpy(reauth_id, identity, len);
	reauth_id[len] = '\0';

	r = data->reauths;
	while (r) {
		if (strcmp(r->reauth_id, reauth_id) == 0)
			break;
		r = r->next;
	}

	free(reauth_id);

	return r;
}


static struct eap_sim_reauth *
eap_sim_db_get_reauth_id(struct eap_sim_db_data *data, const u8 *identity,
			 size_t identity_len)
{
	struct eap_sim_pseudonym *p;
	struct eap_sim_reauth *r;

	if (identity_len == 0)
		return NULL;

	p = eap_sim_db_get_pseudonym(data, identity, identity_len);
	if (p == NULL)
		p = eap_sim_db_get_pseudonym_id(data, identity, identity_len);
	if (p) {
		identity = p->identity;
		identity_len = p->identity_len;
	}

	r = data->reauths;
	while (r) {
		if (identity_len == r->identity_len &&
		    memcmp(r->identity, identity, identity_len) == 0)
			break;
		r = r->next;
	}

	return r;
}


/**
 * eap_sim_db_identity_known - Verify whether the given identity is known
 * @priv: Private data pointer from eap_sim_db_init()
 * @identity: User name identity
 * @identity_len: Length of identity in bytes 
 * Returns: 0 if the user is found or -1 on failure
 *
 * In most cases, the user name is ['0','1'] | IMSI, i.e., 1 followed by the
 * IMSI in ASCII format, ['2','3'] | pseudonym, or ['4','5'] | reauth_id.
 */
int eap_sim_db_identity_known(void *priv, const u8 *identity,
			      size_t identity_len)
{
	struct eap_sim_db_data *data = priv;
	FILE *f;
	char buf[80], *pos;
	int i;

	if (identity_len < 2)
		return -1;

	if (identity[0] == EAP_SIM_PSEUDONYM_PREFIX ||
	    identity[0] == EAP_AKA_PSEUDONYM_PREFIX) {
		struct eap_sim_pseudonym *p =
			eap_sim_db_get_pseudonym(data, identity, identity_len);
		return p ? 0 : -1;
	}

	if (identity[0] == EAP_SIM_REAUTH_ID_PREFIX ||
	    identity[0] == EAP_AKA_REAUTH_ID_PREFIX) {
		struct eap_sim_reauth *r =
			eap_sim_db_get_reauth(data, identity, identity_len);
		return r ? 0 : -1;
	}

	if (identity[0] == EAP_AKA_PERMANENT_PREFIX) {
		/* TODO: EAP-AKA identities. For now, just assume the IMSI
		 * is known. */
		return 0;
	}

	if (identity[0] != EAP_SIM_PERMANENT_PREFIX)
		return -1;

	f = fopen(data->fname, "r");
	if (f == NULL) {
		wpa_printf(MSG_DEBUG, "EAP-SIM DB: could not open triplet "
			   "file '%s'", data->fname);
		return -1;
	}

	identity++;
	identity_len--;
	for (i = 0; i < identity_len; i++) {
		if (identity[i] == '@') {
			identity_len = i;
			break;
		}
	}

	while (fgets(buf, sizeof(buf), f)) {
		/* Parse IMSI:Kc:SRES:RAND and match IMSI with identity. */
		buf[sizeof(buf) - 1] = '\0';
		pos = buf;
		while (*pos != '\0' && *pos != '\n')
			pos++;
		if (*pos == '\n')
			*pos = '\0';
		if (pos - buf < 60 || pos[0] == '#')
			continue;

		pos = strchr(buf, ':');
		if (pos == NULL)
			continue;
		*pos++ = '\0';
		if (strlen(buf) != identity_len ||
		    memcmp(buf, identity, identity_len) != 0)
			continue;

		fclose(f);
		return 0;
	}

	/* IMSI not found */

	fclose(f);
	return -1;
}


static char * eap_sim_db_get_next(struct eap_sim_db_data *data, char prefix)
{
	char *id, *pos, *end;
	u8 buf[10];
	int i;

	if (hostapd_get_rand(buf, sizeof(buf)))
		return NULL;
	id = malloc(sizeof(buf) * 2 + 2);
	if (id == NULL)
		return NULL;

	pos = id;
	end = id + sizeof(buf) * 2 + 2;
	*pos++ = prefix;
	for (i = 0; i < sizeof(buf); i++)
		pos += snprintf(pos, end - pos, "%02x", buf[i]);
	
	return id;
}


/**
 * eap_sim_db_get_next_pseudonym - EAP-SIM DB: Get next pseudonym
 * @priv: Private data pointer from eap_sim_db_init()
 * @aka: Using EAP-AKA instead of EAP-SIM
 * Returns: Next pseudonym (allocated string) or %NULL on failure
 *
 * This function is used to generate a pseudonym for EAP-SIM. The returned
 * pseudonym is not added to database at this point; it will need to be added
 * with eap_sim_db_add_pseudonym() once the authentication has been completed
 * successfully. Caller is responsible for freeing the returned buffer.
 */
char * eap_sim_db_get_next_pseudonym(void *priv, int aka)
{
	struct eap_sim_db_data *data = priv;
	return eap_sim_db_get_next(data, aka ? EAP_AKA_PSEUDONYM_PREFIX :
				   EAP_SIM_PSEUDONYM_PREFIX);
}


/**
 * eap_sim_db_get_next_reauth_id - EAP-SIM DB: Get next reauth_id
 * @priv: Private data pointer from eap_sim_db_init()
 * @aka: Using EAP-AKA instead of EAP-SIM
 * Returns: Next reauth_id (allocated string) or %NULL on failure
 *
 * This function is used to generate a fast re-authentication identity for
 * EAP-SIM. The returned reauth_id is not added to database at this point; it
 * will need to be added with eap_sim_db_add_reauth() once the authentication
 * has been completed successfully. Caller is responsible for freeing the
 * returned buffer.
 */
char * eap_sim_db_get_next_reauth_id(void *priv, int aka)
{
	struct eap_sim_db_data *data = priv;
	return eap_sim_db_get_next(data, aka ? EAP_AKA_REAUTH_ID_PREFIX :
				   EAP_SIM_REAUTH_ID_PREFIX);
}


/**
 * eap_sim_db_add_pseudonym - EAP-SIM DB: Add new pseudonym
 * @priv: Private data pointer from eap_sim_db_init()
 * @identity: Identity of the user (may be permanent identity or pseudonym)
 * @identity_len: Length of identity
 * @pseudonym: Pseudonym for this user. This needs to be an allocated buffer,
 * e.g., return value from eap_sim_db_get_next_pseudonym(). Caller must not
 * free it.
 * Returns: 0 on success, -1 on failure
 *
 * This function adds a new pseudonym for EAP-SIM user. EAP-SIM DB is
 * responsible of freeing pseudonym buffer once it is not needed anymore.
 */
int eap_sim_db_add_pseudonym(void *priv, const u8 *identity,
			     size_t identity_len, char *pseudonym)
{
	struct eap_sim_db_data *data = priv;
	struct eap_sim_pseudonym *p;
	wpa_hexdump_ascii(MSG_DEBUG, "EAP-SIM DB: Add pseudonym for identity",
			  identity, identity_len);
	wpa_printf(MSG_DEBUG, "EAP-SIM DB: Pseudonym: %s", pseudonym);

	/* TODO: could store last two pseudonyms */
	p = eap_sim_db_get_pseudonym(data, identity, identity_len);
	if (p == NULL)
		p = eap_sim_db_get_pseudonym_id(data, identity, identity_len);

	if (p) {
		wpa_printf(MSG_DEBUG, "EAP-SIM DB: Replacing previous "
			   "pseudonym: %s", p->pseudonym);
		free(p->pseudonym);
		p->pseudonym = pseudonym;
		return 0;
	}

	p = malloc(sizeof(*p));
	if (p == NULL) {
		free(pseudonym);
		return -1;
	}

	memset(p, 0, sizeof(*p));
	p->next = data->pseudonyms;
	p->identity = malloc(identity_len);
	if (p->identity == NULL) {
		free(p);
		free(pseudonym);
		return -1;
	}
	memcpy(p->identity, identity, identity_len);
	p->identity_len = identity_len;
	p->pseudonym = pseudonym;
	data->pseudonyms = p;

	wpa_printf(MSG_DEBUG, "EAP-SIM DB: Added new pseudonym entry");
	return 0;
}


/**
 * eap_sim_db_add_reauth - EAP-SIM DB: Add new re-authentication entry
 * @priv: Private data pointer from eap_sim_db_init()
 * @identity: Identity of the user (may be permanent identity or pseudonym)
 * @identity_len: Length of identity
 * @reauth_id: reauth_id for this user. This needs to be an allocated buffer,
 * e.g., return value from eap_sim_db_get_next_reauth_id(). Caller must not
 * free it.
 * @mk: 16-byte MK from the previous full authentication
 * Returns: 0 on success, -1 on failure
 *
 * This function adds a new re-authentication entry for an EAP-SIM user.
 * EAP-SIM DB is responsible of freeing reauth_id buffer once it is not needed
 * anymore.
 */
int eap_sim_db_add_reauth(void *priv, const u8 *identity,
			  size_t identity_len, char *reauth_id, u16 counter,
			  const u8 *mk)
{
	struct eap_sim_db_data *data = priv;
	struct eap_sim_reauth *r;
	wpa_hexdump_ascii(MSG_DEBUG, "EAP-SIM DB: Add reauth_id for identity",
			  identity, identity_len);
	wpa_printf(MSG_DEBUG, "EAP-SIM DB: reauth_id: %s", reauth_id);

	r = eap_sim_db_get_reauth(data, identity, identity_len);
	if (r == NULL)
		r = eap_sim_db_get_reauth_id(data, identity, identity_len);

	if (r) {
		wpa_printf(MSG_DEBUG, "EAP-SIM DB: Replacing previous "
			   "reauth_id: %s", r->reauth_id);
		free(r->reauth_id);
		r->reauth_id = reauth_id;
	} else {
		r = malloc(sizeof(*r));
		if (r == NULL) {
			free(reauth_id);
			return -1;
		}

		memset(r, 0, sizeof(*r));
		r->next = data->reauths;
		r->identity = malloc(identity_len);
		if (r->identity == NULL) {
			free(r);
			free(reauth_id);
			return -1;
		}
		memcpy(r->identity, identity, identity_len);
		r->identity_len = identity_len;
		r->reauth_id = reauth_id;
		data->reauths = r;
		wpa_printf(MSG_DEBUG, "EAP-SIM DB: Added new reauth entry");
	}

	r->counter = counter;
	memcpy(r->mk, mk, EAP_SIM_MK_LEN);

	return 0;
}


/**
 * eap_sim_db_get_permanent - EAP-SIM DB: Get permanent identity
 * @priv: Private data pointer from eap_sim_db_init()
 * @identity: Identity of the user (may be permanent identity or pseudonym)
 * @identity_len: Length of identity
 * @len: Buffer for length of the returned permanent identity
 * Returns: Pointer to the permanent identity, or %NULL if not found
 */
const u8 * eap_sim_db_get_permanent(void *priv, const u8 *identity,
				    size_t identity_len, size_t *len)
{
	struct eap_sim_db_data *data = priv;
	struct eap_sim_pseudonym *p;

	p = eap_sim_db_get_pseudonym(data, identity, identity_len);
	if (p == NULL)
		p = eap_sim_db_get_pseudonym_id(data, identity, identity_len);
	if (p == NULL)
		return NULL;

	*len = p->identity_len;
	return p->identity;
}


/**
 * eap_sim_db_get_reauth_entry - EAP-SIM DB: Get re-authentication entry
 * @priv: Private data pointer from eap_sim_db_init()
 * @identity: Identity of the user (may be permanent identity, pseudonym, or
 * reauth_id)
 * @identity_len: Length of identity
 * @len: Buffer for length of the returned permanent identity
 * Returns: Pointer to the re-auth entry, or %NULL if not found
 */
struct eap_sim_reauth *
eap_sim_db_get_reauth_entry(void *priv, const u8 *identity,
			    size_t identity_len)
{
	struct eap_sim_db_data *data = priv;
	struct eap_sim_reauth *r;

	r = eap_sim_db_get_reauth(data, identity, identity_len);
	if (r == NULL)
		r = eap_sim_db_get_reauth_id(data, identity, identity_len);
	return r;
}


/**
 * eap_sim_db_remove_reauth - EAP-SIM DB: Remove re-authentication entry
 * @priv: Private data pointer from eap_sim_db_init()
 * @reauth: Pointer to re-authentication entry from
 * eap_sim_db_get_reauth_entry()
 */
void eap_sim_db_remove_reauth(void *priv, struct eap_sim_reauth *reauth)
{
	struct eap_sim_db_data *data = priv;
	struct eap_sim_reauth *r, *prev = NULL;
	r = data->reauths;
	while (r) {
		if (r == reauth) {
			if (prev)
				prev->next = r->next;
			else
				data->reauths = r->next;
			eap_sim_db_free_reauth(r);
			return;
		}
		prev = r;
		r = r->next;
	}
}


/**
 * eap_sim_db_get_aka_auth - Get AKA authentication values
 * @priv: Private data pointer from eap_sim_db_init()
 * @identity: User name identity
 * @identity_len: Length of identity in bytes
 * @rand: Buffer for RAND value
 * @autn: Buffer for AUTN value
 * @ik: Buffer for IK value
 * @ck: Buffer for CK value
 * @res: Buffer for RES value
 * @res_len: Buffer for RES length
 * @cb_session_ctx: Session callback context for get_complete_cb()
 * Returns: 0 on success, -1 (EAP_SIM_DB_FAILURE) on error (e.g., user not
 * found), or -2 (EAP_SIM_DB_PENDING) if results are not yet available. In this
 * case, the callback function registered with eap_sim_db_init() will be
 * called once the results become available.
 *
 * In most cases, the user name is '0' | IMSI, i.e., 0 followed by the IMSI in
 * ASCII format.
 */
int eap_sim_db_get_aka_auth(void *priv, const u8 *identity,
			    size_t identity_len, u8 *rand, u8 *autn, u8 *ik,
			    u8 *ck, u8 *res, size_t *res_len,
			    void *cb_session_ctx)
{
#ifdef TEST_PENDING_GET
	struct eap_sim_db_data *data = priv;

	{
		/*
		 * Example code for testing pending get request. When using an
		 * external server for AKA authentication, this function can
		 * always start a request and return EAP_SIM_DB_PENDING
		 * immediately if authentication triplets are not available.
		 * Once the authentication data are received, callback function
		 * registered with eap_sim_db_init() is called to notify EAP
		 * state machine to reprocess the message. This
		 * eap_sim_db_get_aka_auth() function will then be called again
		 * and the newly received triplets will then be given to the
		 * caller.
		 */
		static int test_pending = 1;
		if (test_pending) {
			test_pending = 0;
			eloop_register_timeout(1, 0, eap_sim_db_get_complete,
					       data, cb_session_ctx);
			return EAP_SIM_DB_PENDING;
		}
	}
#endif /* TEST_PENDING_GET */

	if (identity_len < 2 || identity[0] != EAP_AKA_PERMANENT_PREFIX) {
		wpa_hexdump_ascii(MSG_DEBUG, "EAP-SIM DB: unexpected identity",
				  identity, identity_len);
		return EAP_SIM_DB_FAILURE;
	}

	/*
	 * TOOD: implement example mechanism for generating changing
	 * RAND/AUTN/IK/CK values. For example, could use 3GPP example
	 * functions (Milenage).
	 */

	memset(rand, '0', EAP_AKA_RAND_LEN);
	memset(autn, '1', EAP_AKA_AUTN_LEN);
	memset(ik, '3', EAP_AKA_IK_LEN);
	memset(ck, '4', EAP_AKA_CK_LEN);
	memset(res, '2', EAP_AKA_RES_MAX_LEN);
	*res_len = EAP_AKA_RES_MAX_LEN;

	return 0;
}


/**
 * eap_sim_db_get_aka_auth - Get AKA authentication values
 * @priv: Private data pointer from eap_sim_db_init()
 * @identity: User name identity
 * @identity_len: Length of identity in bytes
 * @auts: AUTS value from the peer
 * Returns: 0 on success, -1 on failure
 *
 * this function is called when the peer reports synchronization failure in the
 * AUTN value by sending AUTS. The AUTS value should be to HLR/AuC to allow it
 * to resynchronize with the peer. After this, eap_sim_db_get_aka_auth() will
 * be called again to to fetch updated RAND/AUTN values for the next challenge.
 */
int eap_sim_db_resynchronize(void *priv, const u8 *identity,
			     size_t identity_len, const u8 *auts)
{
	/* TODO */
	return 0;
}
