/*
 * commands.c - Mail protocol independent command handlers.
 *
 * Created by: Espeleta Tomas <espeleta@libero.it>, 12-Apr-2001
 * POP3 code based on nupop-0.4, by Kevin Stone <kstone@nuvox.net> 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include "libghttp-1.0.9-mod/ghttp.h"
#include "libghttp-1.0.9-mod/ghttp_constants.h"
#include "hotwayd.h"
#include "httpmail.h"
#include "inet.h"
#include <sys/time.h>
#include "commands.h" 
#include "xmlstuff.h"

/* Global Vars */
URL_STRUCT *trash = NULL; /* where we store the trash href */
URL_STRUCT *smtpurl = NULL; /* where we store the smtpurl href */
static KEYPAIR **cookies = NULL;
ghttp_request *request = NULL;
time_t time_getlist; /* time when getlist() was last called, used to update LIST */
#define GETLIST_TIMEOUT 120 /* 2 minute timeout for list command to refresh */
const char *badpwdtxt[] = { "Bad Password", "Invalid login." };
const int badpwdcnt = sizeof(badpwdtxt) / sizeof(char *);
extern char *prog;

char buffer[N_BUFFER];
char *serverurl=NULL;
char xml_length[N_XML_LENGTH];
char *UserName = NULL;  /* allocate space depending on size later */
char *PassWord = NULL;
char *WWWAuthenticate;
int unauth_cnt = 0; /* check for unauth loop on proxy */

/* proxy variables are setup via command line arguments */
static char *proxy=NULL;  /* proxy address:port */
static char *proxy_username=NULL;
static char *proxy_password=NULL;
static char *access_list=NULL;
extern char *folder;


/* Some function prototypes */
static int clear_authinfo(ghttp_request *request);
static int check_cpmv_response(int reply);
static int proppatch(char *href, char *xml_request);
static int allowed_user(char *user);


/* general counters/flags */
int oldstatus      = 0;
int cookiecount    = 0;
int auth_required  = 0;
extern int log_level;


const HTTP_Servers default_server[]= {
	{ "hotmail.com", "http://oe.hotmail.com/cgi-bin/hmdata" },
	{ "msn.com", "http://oe.msn.msnmail.hotmail.com/cgi-bin/hmdata" },
	{ "lycos.co.uk", "http://webdav.lycos.co.uk/httpmail.asp" },
	{ "lycos.ch", "http://webdav.lycos.de/httpmail.asp" },
	{ "lycos.de", "http://webdav.lycos.de/httpmail.asp" },
	{ "lycos.es", "http://webdav.lycos.es/httpmail.asp" },
	{ "lycos.it", "http://webdav.lycos.it/httpmail.asp" },
	{ "lycos.at", "http://webdav.lycos.at/httpmail.asp" },
	{ "lycos.nl", "http://webdav.lycos.nl/httpmail.asp" },
	{ "spray.se", "http://webdav.spray.se/httpmail.asp" },
};



const Auth_Method auth_method[] = {
  { "Basic", BASIC },
  { "Digest", DIGEST },
};

int default_servers= sizeof(default_server)/sizeof(HTTP_Servers);
int auth_methods   = sizeof(auth_method)/sizeof(Auth_Method);

/* This is the authorisation type we recieved from the server */
int auth_name = 0;

/* User Agent: was hotwayd, but M$ maybe does'nt like it */
const char *myAgent = "Outlook-Express/5.5 (MSIE 5.0; Windows 98; TmstmpExt)";

/* Additional Header, not defined in ghttp.h */
const char *http_hdr_Brief				= "Brief";
const char *http_hdr_Translate		= "Translate";
const char *http_hdr_Allow_Rename = "Allow-Rename";

/* ---------------------------------------------------------------------
 *	Here begin internal functions that handle message-popping
 *	from HTTPMail server
 * ---------------------------------------------------------------------
 */

/* ---------------------------------------------------------------------
 * SET Cookies, taking them from array of structures and verifing domain
 */
static void my_set_cookies(ghttp_request *request, char *url)
{
	int i;
	char mybuf[N_MYBUF];

	buffer[0] = '\0';

	for (i = 0; i < cookiecount; i++)		/* control if exists */
			/* 20020923: need to test for cookies[i]!=NULL (in case of manual retry-to-login?!) */
		if( cookies[i] && cookies[i]->domain && strstr(url, cookies[i]->domain) != NULL) {
			snprintf(mybuf, N_MYBUF, "%s=%s; ", cookies[i]->key, cookies[i]->value);
			strlcat(buffer, mybuf, N_BUFFER);
		}

	i = strlen(buffer);
	if (i > 0) {
		buffer[i-1] = '\0'; /* truncate last space */
		ghttp_set_header(request, "Cookie", buffer);
	}
}
/* ---------------------------------------------------------------------
 * Parse and save all cookies passed by response
 */
static void save_cookies(ghttp_request *request)
{
	int i, j = -1, n;
	char *k, *key, *value, *cookiedomain = NULL;
	char *tok, *which, pair[N_PAIR];
	char **hdrs_list;

	/* We try to find Set-Cookie headers... */

	if ((hdrs_list = ghttp_get_header_values(request, http_hdr_Set_Cookie)) != NULL) {
		for (n = 0; hdrs_list[n] != NULL; n++) {
			j=-1;
			strlcpy(buffer, hdrs_list[n],N_BUFFER);

			/* Explode key/value; domain; path */
			for (which = buffer; (tok = strtok(which, ";")) != NULL; which = NULL) {

	/* Now tok contains a pair <key="value">	*/
	strlcpy(pair, tok, N_PAIR);

	if ((k = index(pair, '=')) != NULL) {
		*k = '\0';
		value = k + 1;

		for (key = pair; *key == ' '; ) key++;
		if (value[strlen(value)-1] == '\"') value[strlen(value)-1] = '\0';
		if (value[0] == '\"') value++;

		if (!strcmp(key, "domain")) { /* Read domain for this cookie */
			cookiedomain = strdup(value);
			if (j != -1) cookies[j]->domain = cookiedomain;
		}
		else if (strcmp(key, "path")) { /* Read key/value for this cookie, ignoring path*/

			for (i = 0; i < cookiecount; i++)		/* control if exists */
				if( cookies[i] && !strcmp(key, cookies[i]->key)){
			j = i;
		 }

			if (j == -1) {		/* it does not exist: we stack it, else overwrite */
				j = cookiecount;
				cookies[j] = (KEYPAIR*) calloc (1, sizeof(KEYPAIR));
				cookiecount++;
			}
			cookies[j]->key = strdup(key);	/* let's store it */
			cookies[j]->value = strdup(value);
			cookies[j]->domain = cookiedomain;
		}
	}
			}
		}
	}
}
/* ---------------------------------------------------------------------
 * Propfind makes a PROPFIND call on myurl 
 */
int propfind(ghttp_request *request, char *myurl, int whichrequest)
{
	int status;
	
	/* Settings of our connection	 */
	ghttp_clean(request);
	ghttp_set_uri(request, myurl);

	if (ghttp_set_type(request, ghttp_type_propfind) == -1)
		return -1;

	/* Set Body request & Depth header */
	switch(whichrequest) {
	case MSGFOLDERROOT:
	  if (ghttp_set_body(request, (char *)dir_names_request, (int)(strlen(dir_names_request)) ) == -1)
		return -1;
	  ghttp_set_header(request, http_hdr_Depth, "0");
	  snprintf(xml_length, N_XML_LENGTH, "%d", (int)(strlen(dir_names_request)));
	  break;
	case GETFOLDERPROPS:
	  if (ghttp_set_body(request, (char *)dir_props_request, (int)(strlen(dir_props_request)) ) == -1)
		return -1;
	  ghttp_set_header(request, http_hdr_Depth, "1, noroot");
	  snprintf(xml_length, N_XML_LENGTH, "%d", (int)(strlen(dir_props_request)));
	  break;
	case MSGLIST:
	  if (ghttp_set_body(request, (char *)msg_props_request, (int)(strlen(msg_props_request)) ) == -1)
		return -1;
	  ghttp_set_header(request, http_hdr_Depth, "1, noroot");
	  snprintf(xml_length, N_XML_LENGTH, "%d", (int)(strlen(msg_props_request)));
	  break;
	default:
	  if (log_level > 0) 
	    LOG("Internal error: Received invalid request to propfind in %s",__FILE__);
	  httpmail_destroy();
	  exit(-1);
	}
	  
	/* Set Headers */
	ghttp_set_header(request, http_hdr_Content_Type, "text/xml");
	ghttp_set_header(request, http_hdr_Brief, "t");
	ghttp_set_header(request, http_hdr_Content_Length, xml_length);
	/* clear_authinfo(request); */
	if (auth_required) {
	  /* First we have to find out what type of auth the server requested */
	  int i;
	  for (i=0; i<auth_methods && !auth_name; i++) {
	    if (!strncasecmp(auth_method[i].wwwauthstr, WWWAuthenticate, strlen(auth_method[i].wwwauthstr))) {
	      auth_name=auth_method[i].auth_type;
	      break;
	    }
	  }
	  set_authinfo(request, UserName, PassWord, WWWAuthenticate, 0);

	}
	if (whichrequest == MSGLIST)
	  set_authinfo(request, UserName, PassWord, NULL, 1);

	/* now prepare and send our http request that we've built */
	prepare_and_send(request, myurl);
	
	/* Save response cookies */
	save_cookies(request);
				
	if ((status = ghttp_status_code(request)) == AUTHORIZATION) {
	  int i;
	  if (unauth_cnt++ > 3) /* make it 4 times just to be sure :-) */
	    return PASSWORDFAILED;
	  /* check for password failure via ghttp_reason_phrase, this is the only
	   * way I can see to distinguish the various AUTHORIZATION states. Doing
	   * it this way prevents invalid password being returned when the
	   * response was actually unauthorized */
	  for (i = 0; i < badpwdcnt; i++) {
	    if (!strcmp(badpwdtxt[i], ghttp_reason_phrase(request))) 
		return PASSWORDFAILED;
	  }
	} else
	  auth_required = 0;

	return status;
}

/* ---------------------------------------------------------------------
 * Handles replies: redirections and authorizations are recognized!
 * returns reply on break, else returns 0;
 */
int handlereplies(ghttp_request *request, int reply)
{
	const char *wwwauth;

	if (reply != AUTHORIZATION)
	  unauth_cnt=0; /* reset the unauthorised counter */
	
#ifdef DEBUG
	fprintf( stderr, "handlereplies(%d): ", reply ) ; fflush( stderr );
#endif
	if (reply == AUTHORIZATION) {
		if ((wwwauth = ghttp_get_header(request, http_hdr_WWW_Authenticate)) != NULL) {
			if (WWWAuthenticate)
			  free(WWWAuthenticate);
			WWWAuthenticate = strdup(wwwauth);
			auth_required = 1;
#ifdef DEBUG
	 fprintf( stderr, "reply=AUTHORIZATION; %s\n", wwwauth );
#endif
		}
		return 0;
	}
	else if (reply == MULTISTATUS || reply == PASSWORDFAILED || reply == INVALIDUSER) {
#ifdef DEBUG
		fprintf( stderr, "reply=%d\n", reply );
#endif
		return reply;
	}
	else if (reply == OKSTATUS || reply == MOVED) {
#ifdef DEBUG
	  fprintf( stderr, "reply=%d\n", reply );
#endif
	  /* need to reset the authorization info if we move to a new server */
	  clear_authinfo(request); 
	  return 0; /* Sometimes we get 200. We will ignore it. */
	}
	else {
#ifdef DEBUG
		fprintf( stderr, "ERROR reply=%d\n", reply );
#endif
		return ERROR; /* FIXME: this isn't getting passed thu */
	}
}


/* use this function to fill in the URL if we get moved response */
char *get_moved_url(ghttp_request *req) {
  char *new_url=NULL;
  
  if (ghttp_get_header(request, http_hdr_Location) != NULL) {
	new_url=strdup(ghttp_get_header(request, http_hdr_Location));
#ifdef DEBUG
	fprintf( stderr, "reply=MOVED; %s\n", new_url );
#endif
  }

  return new_url;
}


int set_authinfo(ghttp_request *request, char *UserName, char *Password,
			char *WWWAuthenticate, int maintainlogged) {
  int ret = -1;
  clear_authinfo(request);
  switch(auth_name) {
  case DIGEST:
    ret=ghttp_set_digestinfo(request, UserName, PassWord,
			     WWWAuthenticate, maintainlogged);
    break;
  case BASIC:
    ret=ghttp_set_authinfo(request, UserName, PassWord);
    break;
  case UNKNOWN:
  default:
    /* opps couldn't figure out auth... */
    if (log_level > 0) 
      LOG("%s:%d - couldn't figure out auth type from server request",__FILE__,__LINE__);
    break;
  }
  
  return ret;    
}

static int clear_authinfo(ghttp_request *request) {
  int ret;
  ret=ghttp_set_digestinfo(request, NULL, NULL, NULL, 0);
  ret=ghttp_set_authinfo(request, NULL, NULL);
  return ret;
}

/* ---------------------------------------------------------------------
 * Login, try to connect and login to the hotmail server
 * returns 1 on success and 0 on failure
 */
int dologin(ghttp_request *request) 
{
  int reply = 0;
  char *root_href = NULL;
  
  /* A cycle of propfind requests, until MULTISTATUS is returned */
  while ((reply = propfind(request, serverurl, MSGFOLDERROOT)) != MULTISTATUS) {
    if (reply == MOVED) {
      free(serverurl);
      serverurl = get_moved_url(request);
    }
    if (handlereplies(request, reply)) 
      break;
  }
  /* So here we have either logged in or bombed out */
  unauth_cnt=0; /* reset unauth counter */

  /* lets try to find the root href of the mailbox so we can call dir_props */
  if (reply == MULTISTATUS &&	(root_href = ParseXMLFolderRoot(request, trash, smtpurl)) != NULL) {
    if (serverurl) free(serverurl);
    serverurl = strdup(root_href);  /* copy folder root location */
    if (root_href) free(root_href);

    return 1;
  }

  /* otherwise return the status reply which caused login to fail */
  return reply;
}

/* Function: request_folder_props()
 * Send a HTTP request for the mailbox folder properties.
 */
void request_folder_props(ghttp_request *request)
{
  int reply = 0;
  while ((reply = propfind(request, serverurl, GETFOLDERPROPS)) != MULTISTATUS) {
    if (reply == MOVED) {
      free(serverurl);
      serverurl = get_moved_url(request);
    }
    if (handlereplies(request, reply))
      break;
  }
  unauth_cnt=0; /* reset unauth counter */
  return;
}

/* ---------------------------------------------------------------------
 * Get List of messages from server for specified folder
 */
int getlist(ghttp_request *request, FOLDER_STRUCT *folder_props)
{
	int reply = 0, downloaded = 0;

	if (!folder_props || !folder_props->href)
	  return downloaded;
	
	time(&time_getlist);
	
	/* A cycle of propfind requests, until MULTISTATUS is returned */
	while ((reply = propfind(request, folder_props->href, MSGLIST)) != MULTISTATUS) {
	  if (reply == MOVED) {
		free(folder_props->href);
		folder_props->href = get_moved_url(request);
	  }
	  if (handlereplies(request, reply))
		break;
	}
	
	if (reply == MULTISTATUS && ParseXMLMsgList(request, folder_props))
		downloaded = 1;
	unauth_cnt=0; /* reset unauth counter */
	return downloaded;
}


/** Function cp_mv_msg()<br>
 * This function performs the copying or moving of messages between folders on
 * the HTTPMail server. Possible values for cmd are COPY, MOVE or TRASH. If
 * TRASH is given as cmd then the dest_href is ignore and we use the global
 * trash->href address. Copying and moving are essentially the same operation
 * just that we send a COPY or MOVE respectively in the HTTP request. We can
 * set this using the libghttp functions.
 */
int cp_mv_msg(ghttp_request *request, int num_msg, FOLDER_STRUCT *folder_props, char *dest_href, int cmd)
{
  char *msg_name;
  int reply=0;
  
  if (!folder_props || !folder_props->list_props || !folder_props->list_props[num_msg]->href )
    return E_CPMV_NOFOLDERPROPS;

  ghttp_clean(request); /* clean up the request structure */
  
  /* get the file name of the message */
  msg_name = strrchr(folder_props->list_props[num_msg]->href, '/');
  msg_name++;

  /* Now we will put the destination href into the buffer and following that
   * append to file name of the message. At the same time we will set the
   * type of operation (either copy or move) depending on the value cmd. */
  switch(cmd) {
  case TRASH:
    if (!trash || !trash->href)
      return E_MOVE_TRASH_NOT_SET;
    strlcpy(buffer, trash->href, N_BUFFER);
    ghttp_set_type(request, ghttp_type_move);
    break;
  case MOVE:
    if (!dest_href)
      return E_MOVE_NO_DEST;
    strlcpy(buffer, dest_href, N_BUFFER);
    ghttp_set_type(request, ghttp_type_move);
    break;
  case COPY:
    if (!dest_href)
      return E_COPY_NO_DEST;
    strlcpy(buffer, dest_href, N_BUFFER);
    ghttp_set_type(request, ghttp_type_copy);
    break;
  default:
    return E_UNKNOWN_ERR;
  }

  /* append the message name to the buffer */
  strlcat(buffer, msg_name, N_BUFFER);
    
  /* Setup the rest of the connection. Send request to message href, set
   * HTTP destination header to where we want to move/copy msg to. */
  ghttp_set_uri(request, folder_props->list_props[num_msg]->href);
  ghttp_set_header(request, http_hdr_Destination, buffer);
  ghttp_set_header(request, http_hdr_Allow_Rename, "t");
  ghttp_set_header(request, http_hdr_Content_Length, "0");
  set_authinfo(request, UserName, PassWord, NULL, 1);
  prepare_and_send(request, folder_props->list_props[num_msg]->href);
		
  /* is response ok? (should be 201) */
  reply=ghttp_status_code(request);

  return check_cpmv_response(reply);
}

/** Function: delhref()<br>
 * Essentially the same as cp_mv_msg just that it takes an already formed href
 * and msg_name. This is useful if the folder_structs variable is not filled
 * in (basically if we need to cleanup after a failed copy).
 */
int delhref(ghttp_request *request, char *to_delete, char *msg_name) {
  int reply;
  
  ghttp_clean(request);
  if (!trash || !trash->href)
    return E_MOVE_TRASH_NOT_SET;

  strlcpy(buffer, trash->href, N_BUFFER);
  strlcat(buffer, msg_name, N_BUFFER);
  ghttp_set_type(request, ghttp_type_move);
  ghttp_set_uri(request, to_delete);
  ghttp_set_header(request, http_hdr_Destination, buffer);
  ghttp_set_header(request, http_hdr_Allow_Rename, "t");
  ghttp_set_header(request, http_hdr_Content_Length, "0");
  set_authinfo(request, UserName, PassWord, NULL, 1);
  prepare_and_send(request, to_delete);
		
  /* is response ok? (should be 201) */
  reply=ghttp_status_code(request);

  return check_cpmv_response(reply);
}

/* This function takes as inputs the MSG_STRUCT and the desired read state and
 * updates the read flag on the HTTPMail server. It does this by building a
 * D:propertyupdate XML request using the href from the MSG_STRUCT. It then
 * calls the proppatch function to wrap the HTTP request around the XML
 * instruction. */
int update_readmark(MSG_STRUCT *msg_props, int new_readmark) {
  char *xml_request;
  int reply;
  
  if (!msg_props)
    return E_NOMSGPROPS;

  if (msg_props->readmark == new_readmark)
    return E_OK; /* no point updating if it is already the same */

  switch(new_readmark) {
  case 0:
    xml_request = build_propertyupdate_str("0");
    break;
  case 1:
    xml_request = build_propertyupdate_str("1");
    break;
  default:
    return E_INVALID_READMARK;
  }

  while ((reply = proppatch(msg_props->href, xml_request)) != OKSTATUS) {
    if (reply == MULTISTATUS)
      break;
    if (reply == MOVED) {
      free(msg_props->href);
      msg_props->href = get_moved_url(request);
    }
    if (handlereplies(request, reply))
      break;
  }

  free(xml_request);

  return (reply == OKSTATUS || reply == MULTISTATUS) ? E_OK : E_UPDATE_FAILED;
}
		    

static int proppatch(char *href, char *xml_request) {
  int status, len = strlen(xml_request);

  ghttp_clean(request);

  ghttp_set_uri(request, href);

  if (ghttp_set_type(request, ghttp_type_proppatch) == -1)
    return E_GHTTP_ERR;

  if (ghttp_set_body(request, xml_request, len) == -1) {
    free(xml_request);
    return E_GHTTP_ERR; 
  }

  ghttp_set_header(request, http_hdr_Content_Type, "text/xml");
  ghttp_set_header(request, http_hdr_Brief, "t");
  snprintf(xml_length, N_XML_LENGTH, "%d", (int)(strlen(xml_request)));
  ghttp_set_header(request, http_hdr_Content_Length, xml_length);
  set_authinfo(request, UserName, PassWord, NULL, 1);
  prepare_and_send(request, href);
  save_cookies(request);

  if ((status = ghttp_status_code(request)) == AUTHORIZATION) {
    int i;
    if (unauth_cnt++ > 3) /* make it 4 times just to be sure :-) */
      return PASSWORDFAILED;
    for (i = 0; i < badpwdcnt; i++) {
      if (!strcmp(badpwdtxt[i], ghttp_reason_phrase(request))) 
	return PASSWORDFAILED;
    }
  } else
    auth_required = 0;
  
  return status;
}

/** Function check_cpmv_response()
 * Checks the status code returned from the last request and returns an error
 * message according to what happened.
 */
static int check_cpmv_response(int reply) {
  if (reply >= 200 && reply <= 201)
    return E_CPMV_OK;
  else if (reply >= 202 && reply < 300)
    return E_CPMV_WEIRD_OK;
  else if (reply == 403)
    return E_CPMV_FORBIDDEN;
  else if (reply == 409)
    return E_CPMV_CONFLICT;
  else if (reply == 412)
    return E_CPMV_PRECOND_FAILED;
  else if (reply == 423)
    return E_CPMV_LOCKED;
  else if (reply == 503)
    return E_CPMV_BADGATEWAY;
  else if (reply == 507)
    return E_CPMV_INSUFFICIENT_STORAGE;
  else
    return E_UNKNOWN_ERR;
}


/* ---------------------------------------------------------------------
 * Functions called by hotwayd.c. They are a translation POP3 -> HTTPMail
 * ---------------------------------------------------------------------
 */


void httpmail_destroy(void)
{
	/* ensure that we reset cookiecount back to 0! */
	while(cookiecount>0) {
		cookiecount--;
		if( cookies[cookiecount] ){
			free(cookies[cookiecount]->key);
			free(cookies[cookiecount]->value);
			free(cookies[cookiecount]->domain);
			free(cookies[cookiecount]);
			cookies[cookiecount]= NULL;
		}
	}
	
	if (UserName)  free(UserName);
	UserName=NULL;
	if (PassWord)  free(PassWord);
	PassWord=NULL;
	if (serverurl) free(serverurl);
	serverurl=NULL;
	if (cookies)   free(cookies);
	cookies= NULL;
	if (request)   ghttp_request_destroy(request);
	if (trash && trash->href) {
	  free(trash->href);
	  trash->href=NULL;
	}
	if (trash) free(trash);
	trash=NULL;
	if (smtpurl && smtpurl->href) {
	  free(smtpurl->href);
	  smtpurl->href=NULL;
	}
	if (smtpurl) free(smtpurl);
	smtpurl=NULL;
}

/* ---------------------------------------------------------------------
 * Setup the data structures and global variables then call dologin
 * Return 1 on successful login and 0 on failure (via dologin)
 */
int httpmail_authenticate_user(char *user, char *pass)
{
  /* Allocate a new empty request object if we don't have one yet */
  if (!request) request = ghttp_request_new();

	/* Allocate cookies structure and inbox props if not already allocated */
	if (!cookies) cookies = (KEYPAIR **) calloc (MAX_COOKIES, sizeof(KEYPAIR *));
	if (!trash) trash = (URL_STRUCT *) calloc (1, sizeof(URL_STRUCT));
	if (!smtpurl) smtpurl = (URL_STRUCT *) calloc (1, sizeof(URL_STRUCT));
	/* if (!inbox_props) inbox_props = (FOLDER_STRUCT *) calloc(1,
	   sizeof(FOLDER_STRUCT)); */

	if (UserName) free(UserName); /* get rid of any stale username */
	UserName = (char *) malloc( (strlen(user) + 1) * sizeof(char) );
	/* Set username/password for authorization-header */	
	strlcpy(UserName, user, (strlen(user) + 1));

	if (PassWord) free(PassWord);
	PassWord = (char *) malloc( (strlen(pass) + 1) * sizeof(char) );	
	strlcpy(PassWord, pass, (strlen(pass) + 1));

	return dologin(request);
}

/* this function checks we have a user name of the form xxx@hotmail.com
 * or xxx@msn.com or xxx%hotmail.com or xxx%msn.com */
int validate_username(char *user)
{
  char *domain= index( user, '@');
  char *folder_pos= index( user, '/');
  if (serverurl) free(serverurl);
  serverurl=NULL;

  if (!domain) domain=index(user,'%'); /* check for % sign as well */
  if (!domain) {
    return E_LOGIN_NO_DOMAIN; /* neither an @ nor % was there */
  } else {
		int i;
		*domain='@'; /* make sure the @ sign is there, overwrite any % */
		domain+= 1;
		/* first we will ignore the folder section */
		if (folder_pos) *folder_pos='\0';
		/* now check if we are allowed to login */
		if (!allowed_user(user))
		  return E_LOGIN_NOTALLOWED;
		/* now check for a server corresponding to the domain */
		for( i= 0; i< default_servers && !serverurl; i++ ){
		  /* 20030128 RJVB: do case-insensitive comparison of the servername...! */
		  if( strcasecmp( default_server[i].domain, domain)== 0 ){
			serverurl=strdup(default_server[i].URL);
		  }
		}
		if (folder_pos) *folder_pos='/'; /* put folder back */
  }
  
  if( !serverurl ){
	    return E_LOGIN_UNRECOGNISED_DOMAIN;
  }
  return E_LOGIN_OK; /* username validates */
}





char *gen_uidl(char *dest, char *source)
{
	int i, j;

	if( !source ){
		strcpy( dest, "<DELETED>" );
	}
	else{
		for (j = 0, i = strlen(source) - 1; i >= 0 && j < 31; i--, j++)
			dest[j] = (source[i] != '/') ? source[i] : 'Z';
		dest[j] = '\0';
	}
	return dest;
}


char *next_eol( const char *line, int *len )
{ char *l= NULL;
	*len= 0;
	if( line && *line ){
		if( (l= strstr(line, "\n")) ){
			*len= 1;
		}
		else if( (l= strstr(line, "\r\n")) ){
			*len= 2;
		}
		else if( (l= strstr(line, "\r")) ){
			*len= 1;
		}
	}
	return(l);
}

char *next_emptyline( const char *line, int *len )
{ char *l=NULL;
	*len= 0;
	if( line && *line ){
		if( (l= strstr(line, "\n\n")) ){
			*len= 2;
		}
		else if( (l= strstr(line, "\r\n\r\n")) ){
			*len= 4;
		}
		else if( (l= strstr(line, "\r\r")) ){
			*len= 2;
		}
	}
	return(l);
}

char *next_dotline( const char *line, int *len )
{ char *l= NULL;
	*len= 0;
	if( line && *line ){
		if( (l= strstr(line, "\n.")) ){
			*len= 2;
		}
		else if( (l= strstr(line, "\r\n.")) ){
			*len= 3;
		}
		else if( (l= strstr(line, "\r.")) ){
			*len= 2;
		}
	}
	return(l);
}

char *concat( char *text, char *addition )
{
	if( !addition ){
		return( text );
	}
	if( text ){
		if( (text= (char*) realloc( text, (strlen(text) + strlen(addition)+ 1) * sizeof(char) )) ){
			strcat( text, addition );
		}
	}
	else{
		text= strdup(addition);
	}
	return(text);
}

/* 20020927 RJVB: lowlevel message sending routine. Takes care of byte-stuffing, i.e.
 \ prepending a period to lines that start with a period.
 \ 20030128 RJVB: returns the output message.
 */
char *send_message( char *msg, char *response )
{ char *ldot= NULL, D= 0;
  int len;
	if( msg ){
		ldot= next_dotline(msg, &len);
		if( msg[0]== '.' || ldot ){
			while( msg && (*msg== '.' || ldot) ){
				if( ldot ){
					ldot+= len-1;
					D= *ldot;
					*ldot= '\0';
				}
				if( *msg== '.' ){
					PSOUT( "." );
					if( response ){
						response= concat( response, "." );
					}
				}
				PSOUT( msg );
				if( response ){
					response= concat( response, msg );
				}
				if( ldot ){
					*ldot= D;
					msg= ldot;
					ldot= next_dotline(msg, &len);
				}
				else{
					msg= NULL;
				}
			}
			  /* an additional newline seems to be required here to be compatible with other servers: */
			PCRLF;
			if( response ){
				response= concat( response, "\015\012" );
			}
		}
		else{
			PSOUT( msg );
			if( response ){
				response= concat( response, msg );
			}
		}
	}
	return(response);
}


/* This function does the generic work of preparing a http request and sending
 * it. It handles any proxy servers that have been specified on the command
 * line and checks for weird errors as a result of the proxy server being
 * incorrect or unavailable. This will catch some other strange responses that
 * hotmail gives, like when you try to enter an incorrect password for an
 * account many times, perhaps better error reporting will be required here in
 * the future. */
void prepare_and_send(ghttp_request *request, char *url)
{
   ghttp_set_header(request, http_hdr_User_Agent, myAgent); 
   ghttp_set_header(request, http_hdr_Connection, "Keep-Alive"); 
   if (proxy) ghttp_set_proxy(request, proxy);
   if (proxy_username && proxy_password) 
	 ghttp_set_proxy_authinfo(request, proxy_username, proxy_password);   
   my_set_cookies(request, url);
   
   /* Prepare the connection */
   ghttp_prepare(request);

   /* Process the request and exit on error */
   if (ghttp_process(request) == ghttp_error) {
	 snprintf(buffer, N_BUFFER, "-ERR Fatal connection error: %s\015\012", ghttp_get_error(request));
	 PSOUT(buffer);
	 PCRLF;
	 if (log_level > 0) 
	   LOG("Fatal connection error: %s", ghttp_get_error(request));
	 httpmail_destroy();
	 exit(-1);
   }
#ifdef DEBUG
   /* here you can see the status returned along with the "reason" */
   printf("Status returned: %d, reason: %s\n",ghttp_status_code(request), ghttp_reason_phrase(request));
#endif
   return;
}

/* This function checks a access list specified as argument to the daemon in
 * order to see if the user trying to login is allowed to use hotwayd on this
 * system. This code is based on a patch provided by Domingo Alczar. */
static int allowed_user(char *user) {
  FILE *f;
  char buf[1000];

  /* if there is no access list specified then anyone can login */
  if (access_list == NULL)
    return 1;

  /* otherwise open the file and check the list */
  f=fopen(access_list, "r");
  if (f != NULL) {
    while (!feof(f)) {
      buf[0] = '\0';
      fscanf(f, "%999s", &buf);
      if (buf && user && !strcmp(buf, user)) {
	fclose(f);
	return 1;	  
      }
    }
    fclose(f);
  }
  return 0;
}

char *grow_and_copy(char *dest, int curpos, char *src, int i)
{
  if(!(dest = realloc(dest, curpos+i+1)))
    return NULL;

  strlcat(dest, src, curpos+i+1);

  return dest;
}

int proxy_sanity_check(void) {
  if (proxy_username && !proxy_password) {
	free(proxy_username);
	printf("Invalid args: Proxy user name supplied, but no password!\n");
	return 0;
  } else if (!proxy_username && proxy_password) {
	free(proxy_password);
	printf("Invalid args: Proxy password supplied but no user name!\n");
	return 0;	  
  } else if ((proxy_username || proxy_password) && !proxy) {
	if (proxy_username) free(proxy_username);
	if (proxy_password) free(proxy_password);
	printf("Invalid args: Proxy user name or password supplied but no proxy given\n");
	return 0;
  }	
  return 1; /* arguments are sane */
}

void set_proxy(char *proxy_url) {
  proxy = proxy_url;
}

void set_proxy_username(char *p_uname) {
  proxy_username = p_uname;
}

void set_proxy_password(char *p_pword) {
  proxy_password = p_pword;
}

void set_access_list(char *a_list) {
  access_list = a_list;
}

/* little function for parsing command line arguments */
int str2val(register const char *str, register const char *what,
	    register int mi, register int ma)
{
  register const char *cp;
  register int val;
  char *ep;

  if (str[0] == '0' && (str[1] == 'x' || str[1] == 'X')) {
    cp = str + 2;
    val = (int)strtol(cp, &ep, 16);
  } else
    val = (int)strtol(str, &ep, 10);
  if (*ep != '\0') {
    fprintf(stderr, "%s: \"%s\" bad value for %s \n",
	    prog, str, what);
    exit(1);
  }
  if (val < mi && mi >= 0) {
    if (mi == 0)
      fprintf(stderr, "%s: %s must be >= %d\n",
	      prog, what, mi);
    else
      fprintf(stderr, "%s: %s must be > %d\n",
	      prog, what, mi - 1);
    exit(1);
  }
  if (val > ma && ma >= 0) {
    fprintf(stderr, "%s: %s must be <= %d\n", prog, what, ma);
    exit(1);
  }
  return (val);
}
