#!/usr/bin/perl
#
# HTTPush v0.9b11 by Lluis Mora <llmora@s21sec.com>
#
# Usage: httpush -p port [ -i ] [ -l ] [ -d level ] [ -I address ] [ -e meta ] [ -h ] [ -t ] 
#                [ -P url ] [ -S url ] [ -n noise ] [ -x intrusive ] [ -A allowed ]
#
# -p port    Listen on port "port"
# -i         In interactive mode (default), allow some data through (images, javascript...)
# -l         Learning mode. Allows transparent browsing while recording data sent and received
# -d level   Debug mode. Show debug information.
# -I address Listen on local IP address (default 127.0.0.1)
# -e meta    Meta-characters not to escape
# -h         Display usage.
# -t				 Traditional mode (just the request interception, no plugins, no XML)
# -P url     Remote HTTP proxy URL
# -S url     Remote HTTPS proxy URL
# -n noise   Noise level threshold
# -x intr    Intrusive level threshold
# -A allowed Regexp describing hostnames we are allowed to play with

use strict;
no strict 'subs';

use Socket;
use IO::Socket;
use POSIX qw(strftime);
use FileHandle;
use Getopt::Std;

use IPC::Open2;

require LWP;
require MIME::Base64;

use URI::Escape qw (%escapes);
use URI::Escape;

$HTTP::Headers::TRANSLATE_UNDERSCORE=0;

# Include our files

require httpush;

# Globals

my $VERSION="0.9b11";
my $DATADIR="data/";
my %HTTPUSH_OPTIONS;
my %CURRENT;

my $UA;

my @EXT_IMAGES=("gif","jpg","jpeg","png");
my @EXT_CLIENTSIDE=("js","css");
my @EXT_APP=("jsp","php","php3","shtml","asp","cgi","exe","bat","pl","sh");

my @EXT_INTERESTING=(@EXT_APP, @EXT_CLIENTSIDE);
my @EXT_UNINTERESTING=(@EXT_IMAGES, @EXT_CLIENTSIDE);

# Initialization

&main;
exit 0;

sub main {

my (%cmdline_opts, $i, $c, $plugin);

$HTTPUSH_OPTIONS{'ssl'}=1;
$HTTPUSH_OPTIONS{'xml'}=1;
$HTTPUSH_OPTIONS{'plugin_version'}="1.0";

$HTTPUSH_OPTIONS{'slave_count'}=8;
$HTTPUSH_OPTIONS{'max_per_slave'}=4;

$HTTPUSH_OPTIONS{'ssl'} = eval { require Net::SSLeay; require Crypt::SSLeay; 1;};


$|=1;

$HTTPUSH_OPTIONS{'title'}="HTTPush $VERSION";

$SIG{INT} = \&getout;
$SIG{CHLD}='IGNORE';
$SIG{PIPE}='IGNORE';

# Create the user agent object
$UA = LWP::UserAgent->new;

$HTTPUSH_OPTIONS{'images_through'}=0;
$HTTPUSH_OPTIONS{'learning'}=0;

getopts('A:d:e:hiI:ln:p:P:S:tx:', \%cmdline_opts);

if($cmdline_opts{'h'}) {
  usage();
  }

if($cmdline_opts{'d'}) {
  lib::set_debug_level($cmdline_opts{'d'});
}

if($cmdline_opts{'A'}) {
  $HTTPUSH_OPTIONS{'allowed'}=$cmdline_opts{'A'};

  my $re_test=eval { RecordAllowed("test"); 1;};

if(!defined $re_test) {
  die("ERROR: Invalid regexp \"$HTTPUSH_OPTIONS{'allowed'}\": $@");
}

}

if($cmdline_opts{'n'}) {
  $HTTPUSH_OPTIONS{'noise'}=$cmdline_opts{'n'};
  } else {
  $HTTPUSH_OPTIONS{'noise'}=100;
  }

if($cmdline_opts{'x'}) {
  $HTTPUSH_OPTIONS{'intrusive'}=$cmdline_opts{'x'};
  } else {
  $HTTPUSH_OPTIONS{'intrusive'}=100;
  }

$HTTPUSH_OPTIONS{'listen_port'}=$cmdline_opts{'p'} || usage();

if($cmdline_opts{'i'}) {
  $HTTPUSH_OPTIONS{'images_through'}=1;
  }

if($cmdline_opts{'I'}) {
  $HTTPUSH_OPTIONS{'listen_ip'}=$cmdline_opts{'I'};
  } else {
  $HTTPUSH_OPTIONS{'listen_ip'}="127.0.0.1";
  }

if(! $cmdline_opts{'t'}) {
$HTTPUSH_OPTIONS{'xml'} = eval { require XML::Twig; 1;};
} else {
$HTTPUSH_OPTIONS{'xml'}=0;
}

if($cmdline_opts{'l'}) {
  $HTTPUSH_OPTIONS{'learning'}=1;
  if(! $HTTPUSH_OPTIONS{'xml'}) {
    die("ERROR: Learning mode is only useful when XML support is installed, exiting.");
  }
}

# Check if we have something to not-escape (specified in $cmdline_opts{'e'})

if(defined $cmdline_opts{'e'}) {
  for($i=0;$i<length($cmdline_opts{'e'});$i++) {
    $c=substr($cmdline_opts{'e'},$i,1);
    $escapes{$c}=$c;
  }
}

# Set up proxies

$UA->env_proxy();

if($cmdline_opts{'P'}) {
  $UA->proxy('http', "$cmdline_opts{'P'}");
}

if($HTTPUSH_OPTIONS{'ssl'} && $cmdline_opts{'S'}) {
  $UA->proxy('https', "$cmdline_opts{'S'}");
}

$HTTPUSH_OPTIONS{'http_proxy'}=$UA->proxy('http');
$HTTPUSH_OPTIONS{'https_proxy'}=$UA->proxy('https');

lib::announce("Initializing HTTPush v$VERSION");

# Load plugins
$plugin=undef;

if($HTTPUSH_OPTIONS{'xml'}) {
  lib::announce("Loading plugins");

  $plugin = new lib::plugin((basedir => 'plugins/'));

  if(! $plugin || ! $plugin->load(\%HTTPUSH_OPTIONS)) {
    lib::debug("ERROR LOADING PLUGINS");
    lib::announce("Critical error loadin plugins");
    exit;
  }
}


# Initialize SSL

if($HTTPUSH_OPTIONS{'ssl'}) {

  require Net::SSLeay; import Net::SSLeay qw(die_now die_if_ssl_error);

  Net::SSLeay::load_error_strings();
  Net::SSLeay::SSLeay_add_ssl_algorithms();
  Net::SSLeay::randomize();

  $CURRENT{'ctx'} = Net::SSLeay::CTX_new() or die_now("CTX_new ($CURRENT{'ctx'}): $!");

  Net::SSLeay::CTX_set_options($CURRENT{'ctx'}, &Net::SSLeay::OP_ALL)
      and die_if_ssl_error("ssl ctx set options");

#  Net::SSLeay::CTX_set_cipher_list($CURRENT{'ctx'}, "ALL:!ADH:!EXPORT56:RC4+RSA:+HIGH:+MEDIUM:+LOW:+SSLv2:+EXP")
#      and die_if_ssl_error("ssl ctx set cipher list");

  # Following will ask password unless private key is not encrypted
  Net::SSLeay::CTX_use_RSAPrivateKey_file ($CURRENT{'ctx'}, 'httpush.pem', &Net::SSLeay::FILETYPE_PEM);
  die_if_ssl_error("private key");

  Net::SSLeay::CTX_use_certificate_file ($CURRENT{'ctx'}, 'httpush.pem', &Net::SSLeay::FILETYPE_PEM);
  die_if_ssl_error("certificate");
} else {
  lib::announce("SSL Support disabled.");
}

if(! $HTTPUSH_OPTIONS{'xml'}) {
    lib::announce("XML Support disabled.");
}

if($HTTPUSH_OPTIONS{'http_proxy'}) {
  lib::announce("Using remote HTTP proxy $HTTPUSH_OPTIONS{'http_proxy'}");
}

if($HTTPUSH_OPTIONS{'https_proxy'}) {
  lib::announce("Using remote HTTP proxy $HTTPUSH_OPTIONS{'https_proxy'}");
}

# Reindex data files

if($HTTPUSH_OPTIONS{'xml'}) {
  &StoredReindex();
}

# Set up port

my $proto = getprotobyname('tcp');
$HTTPUSH_OPTIONS{'listen_port'} = $1 if $HTTPUSH_OPTIONS{'listen_port'} =~ /(\d+)/; # untaint port number

my $Server = IO::Socket::INET->new(Listen    => 10,
  LocalAddr => "$HTTPUSH_OPTIONS{'listen_ip'}",
  LocalPort => $HTTPUSH_OPTIONS{'listen_port'},
  Reuse     => 1,
  Proto     => 'tcp') || die "Can't create server socket: $!";

  lib::announce("HTTPush v$VERSION initialized, proxy listening on $HTTPUSH_OPTIONS{'listen_ip'}:$HTTPUSH_OPTIONS{'listen_port'}");

my $paddr;
my %kids;


# Launch prefork stuff, based on code by Randal L. Schwartz

$0="httpush [manager]";

for(1..$HTTPUSH_OPTIONS{'slave_count'}) {
  $kids{&fork_a_slave($plugin, $Server)} = "slave";
}

{
  my $pid = wait;

  my $was = delete ($kids{$pid}) || "?unknown?";
  lib::debug("child $pid ($was) terminated status $?");

  if($was eq "slave") {      # oops, lost a slave
#    sleep 1;                  # don't replace it right away (avoid thrash)
    $kids{&fork_a_slave($plugin, $Server)} = "slave";
  }
} continue { redo };          # semicolon for cperl-mode

return 0;

}

############################# Functions ######################################

sub fork_a_slave {              # return int (pid)
  my ($plugin, $master) = @_;

  my $pid;

  defined ($pid = fork) or die "Cannot fork: $!";
  &child_work($plugin, $master) unless $pid;
  $pid;
}


sub child_work {
  my ($plugin, $master) = @_;

  my $did = 0;                  # processed count

#  lib::debug("child started");
  {
    $0="httpush [worker ($did)]";
    open(FL,"> httpush.lck");

#    flock($master, 2);          # 2

    flock(FL, 2);
    lib::debug("Successfully got lock");
    my $paddr = accept(Client,$master) || die("Urgh: $!");

#    lib::debug("Accepted");
#    flock($master, 8);          # 8

     flock(FL, 8);
     close(FL);

    my ($port,$iaddr) = sockaddr_in($paddr);
    my $addr=inet_ntoa($iaddr);

    open(STDIN,  "<&Client") || die "can't dup client to stdin: $!";
    open(STDOUT, ">&Client") || die "can't dup client to stdout: $!";

    &proxy_process($plugin, *Client, undef, 1, $addr, $port); # closes $slave at right time

    close(Client);
    close(STDIN);
    close(STDOUT);

  } continue { redo if ++$did < $HTTPUSH_OPTIONS{'max_per_slave'} };
  lib::debug("child terminating");
  exit 0;
}

sub proxy_process {
  my ($plugin, $client_fd, $clientRequest, $record, $who_address, $who_port)=@_;
  my $did = 0;
  my ($proxyRequest, $client_data, $pid, $path, $query, $url_ext, $cookie_text);

  $0="httpush [accepting request]";

  my $is_ssl=0;
  my $ssl_fd=0;
  my $end=0;
  my $final=0;

############################# Start processing ################################

  # Read request from client

  if(! defined $clientRequest) {
    $clientRequest = GetRequest();
  }

  if(! defined $clientRequest) {
    &my_error(1);
    return undef;
  }

  my ($url_host, $url_path)=SplitURI($clientRequest->uri());

  if($url_path=~/\.([^\.\?]+)\??/) {
    $url_ext=lc($1);
  }

  # Continue

    if(! $url_host) {
      # This is a plain web request, something to do with a special mode.
      $final=1;

      ($path,$query)=split("\\?",$url_path, 2);

      if($path) {
        $path=substr($path,1);
        my ($ws_action, $ws_file, $ws_id)=split('/',$path);
        my %query=QuerySplit($query);

        if(defined $ws_file) {
          $ws_file=~y/0-9/0-9/cd;
        }

        if(defined $ws_id) {
          $ws_id=~y/0-9/0-9/cd;
        }

        # Hook for extensions

        lib::debug("WSAction <$ws_action>");

        if($ws_action eq "Browse" ) {
          # Browse stored data
          if(! $HTTPUSH_OPTIONS{'xml'}) {
            &my_error(6,"",501);
          } else {
            $0="httpush [server - list]";
            StoredList($ws_file, $ws_id, \%query);
          }
          return undef;

        } elsif($ws_action eq "View") {
          # View an item of stored data
          if(! $HTTPUSH_OPTIONS{'xml'}) {
            my_error(6,"",501);
          } else {
            $0="httpush [server - view]";
            StoredView($ws_file, $ws_id, \%query);
          }
          return undef;
        } elsif((! $ws_action) && ($clientRequest->method() eq "GET")) {
          # Display admin menu
          $final=3;
        } else { # Fallback to processing result of an intermediate post
          ($final, $proxyRequest, $client_data)=ParsePostRequest($clientRequest->content());
        }
      }
    } elsif(($record!=1) || $HTTPUSH_OPTIONS{'learning'} || ($HTTPUSH_OPTIONS{'images_through'} && grep($url_ext eq $_, @EXT_UNINTERESTING))) {

       lib::debug("UNINTERESTING");

      $final=2;
      $proxyRequest=$clientRequest;
    } else {
      ($final, $proxyRequest, $client_data)=ParsePostRequest($clientRequest->content());
    }

    if(! $final) {
      $final=0;
    }

    if($final == 2) {

    # If the user set an authentication header in the client request, set it in the proxy request (wrong?)

    if($clientRequest->header("authorization")) {
      $proxyRequest->header("authorization",$clientRequest->header("authorization"));
      }

    # Retrieve page

    my $url_port=80;

    if($url_host=~/^([^:]+):(\d+)/) {
      $url_host=$1;
      $url_port=$2;
      }

    if(defined $proxyRequest->content()) {
    lib::debug("Retrieving ".$proxyRequest->uri()." - [".$proxyRequest->content()."]");
    } else {
    lib::debug("Retrieving ".$proxyRequest->uri());
    }
    # Print stuff

#    lib::debug("Sent:\n<<".$proxyRequest->as_string.">>");

    $UA->agent(""); # Clean the agent header, as the request will set it (if there is one)

    $0="httpush [requesting]";

    my $response = $UA->simple_request($proxyRequest);

#    lib::debug("Received: <".$response->as_string.">");

    if($response->code==500) {
      if($response->as_string=~/^500 \(Internal Server Error\)\s+([^,]*)/) {
        &my_error(3, $1, 502);
        return undef;
        }
    }


    if($response->is_error || $response->code==304) {
      # There's been an error, forget about the cookies
    } else {
      if($client_data) {
        $cookie_text="HTTPUSHDATA=".$client_data."; path=/";
      } else {
        # Delete cookie
        $cookie_text="HTTPUSHDATA=Deleting; path=/; expires=Thursday, 01-Jan-1970 00:00:00 GMT";
      }

      # Why do we do this? Because if we want to insert the Set-Cookie amongst the
      # other cookies we must first recover the original headers, which
      # HTTP::Message will happily fold them with "," into a single header, thus making it compliant with RFC2109,
      # written by Netscape:
      #
      # "An origin server may include multiple Set-Cookie headers in a
      # response.  Note that an intervening gateway could fold multiple such
      # headers into a single header."
      #
      # but making it incompatible with Netscape Browsers (!)
      #
      # HTTP::Message also doesn't allow us to insert a duplicate header, so we fool it by making it believe it's a
      # different header name while still being valid for us.

      $response->header("Set-Cookie: $cookie_text\r\nSet-Cookie", $cookie_text);
    }


    if(defined $client_fd) {
      if($response->code==304) {
        my $tmp=$response->as_string;

        $tmp=substr($tmp,0, length($tmp) - 1);

        lib::debug("Sending modified 304: <$tmp>");

        hprint($ssl_fd, $tmp);
      } else {
        hprint($ssl_fd, $response->as_string);
      }

    close $client_fd;
    close STDIN;
    close STDOUT;
    }

    if($record) {
      if(RecordAllowed($proxyRequest->uri)) {
      Record($plugin, "request", time(), $who_address, $who_port, $proxyRequest, $response);
      } else {
      lib::debug("We are not allowed to record ".$proxyRequest->uri);
      }
    }

    return $response;

    } elsif($final==3) { # Display admin menu
    print "HTTP/1.1 200 OK\r\n";
    print "Content-Type: text/html\r\n";
    print "Connection: Close\r\n";
    print "\r\n";
    print "<HTML>";

    print "<HEAD>";
    print HTMLPrintStyle();

    print "<TITLE>HTTPush admin interface</TITLE></HEAD><BODY>";
    print "<H1>HTTPush admin interface</H1>";
    print '<A HREF="/Browse">Audited Sites</A><BR>';
    print '<P>';
    print '<A HREF="http://httpush.sourceforge.net/">HTTPush website</A><BR>';
    print '<A HREF="http://sourceforge.net/tracker/?func=add&group_id=29026&atid=395271">Submit a bug</A><BR>';
    print "<HR><ADDRESS>HTTPush/$VERSION at $HTTPUSH_OPTIONS{'listen_ip'} Port $HTTPUSH_OPTIONS{'listen_port'}</ADDRESS>";
    print "</BODY></HTML>";

    } else {
      PrintForm($ssl_fd, $url_path, $url_host, $clientRequest);
    }

  if($CURRENT{'is_ssl'}) {
    Net::SSLeay::free ($CURRENT{'ssl'});
    Net::SSLeay::CTX_free ($CURRENT{'ctx'});
  }

if(defined $client_fd) {
  close $client_fd;
  close STDIN;
  close STDOUT;
}

  return undef;
}

sub my_error () {
  my ($error, $data, $code)=@_;

  lib::debug("Displaying error $error");

  if(! $code) {
    $code=502;
  }

  print "HTTP/1.1 $code an error code\r\n";
  print "Content-Type: text/html\r\n";
  print "Connection: Close\r\n";
  print "\r\n";
  print "<HTML>";

  print "<HEAD>";
  print HTMLPrintStyle();

  print "<TITLE>Error retrieving URL</TITLE></HEAD><BODY>";

  if($error==1) {
    print "<H1>Your request was malformed. Please check your proxy settings
           and try again. Remember, this is a HTTP/HTTPS proxy, nothing else.</H1><BR>";
    } elsif($error==2) {
    print "<H1>The address you are trying to access uses an unsupported URI
           scheme. Currently HTTPush only supports HTTP/HTTPS connections</H1><BR>";
    } elsif($error==3) {
    print "<H1>There has been an error connecting to the remote host</H1><I><strong>$data:</strong>The
           address does not exist or it is down.</I><BR>";
    } elsif($error==4) {
    print "<H1>There has been an error while trying to retrieve the learned item. The item doesn't exist anymore.
           </H1><BR>";
    } elsif($error==5) {
    print "<H1>SSL support is unavailable</H1>Please install SSLeay modules if you need HTTPS support.
           <BR>";
    } elsif($error==6) {
    print "<H1>XML support is unavailable</H1>Please install the XML::Twig module or disable traditional mode (drop the '-t') 
           in order to use this functionality.
           <BR>";
    } elsif($error==7) {
    print "<H1>Internal error, you found a bug!</H1>HTTPush encountered an internal error, the following details could help
to find where it came from:<BR>
<B><PRE>$data</PRE></B><BR>Please submit this information to our <A
HREF=\"http://sourceforge.net/tracker/?func=add&group_id=29026&atid=395271\">Bug Reporting System</A> to help us fix it in a future
release.<BR>Thanks for your support.";
    } else {
    print "<H1>There was an undefined error while trying to process your
           request.</H1><BR>";
    }

  print "<HR><ADDRESS>HTTPush/$VERSION at $HTTPUSH_OPTIONS{'listen_ip'} Port $HTTPUSH_OPTIONS{'listen_port'}</ADDRESS>";

  print "</BODY></HTML>";

  if($HTTPUSH_OPTIONS{'ssl'}) {
    Net::SSLeay::CTX_free ($CURRENT{'ctx'});
  }

  close Client;
  }

sub usage() {
  print STDERR "HTTPush v$VERSION by Lluis Mora <llmora\@s21sec.com>\n";
  print STDERR "Usage: httpush -p port [ -i ] [ -l ] [ -d level ] [ -I address ] [-e meta ] [ -h ] [ -t ]\n";
  print STDERR '               [ -P url ] [ -S url ] [ -n level ] [-x intr ][ -A allowed ]
 -p port    Listen on port "port"
 -i         In interactive mode (default), allow some data through (images, javascript...)
 -l         Learning mode. Allows transparent browsing while recording data sent and received
 -d level   Debug mode. Show debug information.
 -I address Listen on local IP address (default 127.0.0.1)
 -e meta    Meta-characters not to escape
 -h         Display usage.
 -t         Traditional mode (just the request interception, no plugins, no XML)
 -P url     Remote HTTP proxy URL
 -S url     Remote HTTPS proxy URL
 -n noise   Noise level threshold
 -x intr    Intrusive level threshold
 -A allowed Regexp describing hostnames we are allowed to play with
';

  exit(1);
  }

sub getout {
  exit;
  }

sub learn_add_command {
  my ($uri, $num)=@_;

  if($uri=~/^(.*)\?(.*)$/) {
    if(substr($uri,length($uri)-1,1) eq "&") {
      $uri=$uri."HTTPUSH_ITEM=$num";
      } else {
      $uri=$uri."&HTTPUSH_ITEM=$num";
      }
    } else {
    $uri=$uri."?HTTPUSH_ITEM=$num";
    }
  }

# Get request

sub GetRequest {
  my($req,$line,$content,%headers, $first, $second, $h);
  my ($r, $ssl_fd);
  my $end=0;

  while((! $end) && ($line=<STDIN>)) {
    #  lib::debug("read: $line");

    $line=~s/\n//;
    $line=~s/\r//;

    if(! $req) {
      $req=$line;
      } else {
      if($line eq "") {
        if($headers{'content-length'}) {
          read (STDIN,$content,$headers{'content-length'});
          # $content=uri_unescape($content,"\0-\377"); # YYY NORL!!!
          }
        $end=1;
        } else {
        if($line=~/([^:]+)\s*:\s*(.*)/) {
          $first=lc($1);
          $second=$2;
	
	  if($first eq "referer" && $second=~/^(.*)&HTTPUSH_ITEM=\d+$/) { 
	    $second=$1;
	    }

          if($first ne "connection" && $first ne "proxy-connection" ) { 
            $headers{"$first"}="$second";
            }
          }
        }
      }
    }

  # Done reading request, process it

  $r = new HTTP::Request;

  if($req=~/([^\s]+)\s*([^\s]+)\s*([^\s]+)/) {
    $r->method($1);
    $r->uri($2);
    $r->protocol($3);
    } else {
    # Error
    return undef;
    }

  $r->content($content);

  foreach $h (keys %headers) {
    $r->header($h,$headers{$h});
    }

  if($r->method() eq "CONNECT") {

    if($HTTPUSH_OPTIONS{'ssl'}) {
      $CURRENT{'is_ssl'}=1;
    } else {
      &my_error(5,"", 501);
      return undef;
    }

    $r=FakeSSL($r);
    }

  return $r;
  }

sub SplitURI {
  my $url=shift;

  if($url=~/^\/?https?:\/\/([^\/]+)(\/.*)?/) {
      return($1, $2);
    } else {
      return (undef, $url); # We found no known URL scheme, we assume it's all path
    }
  }

sub ParsePostRequest {
  my $form=shift;
  my($final, $x, $proxyRequest, %header, %cookie, %additional_header, %additional_cookie, %FORM, $var, $val, $k);
  my($client_data, $tmp);

  my @input=();

  $client_data="";


#  lib::debug("Received ($form)");

  $proxyRequest = new HTTP::Request;

  if(defined $form) {
    @input=split("&",$form);
  }

  my $cookie_counter=0;
  $final=0;

  foreach $x (@input) {
    if($x=~/([^\=\n\r]+)=(.*)/) {
      $var=$1;
		  $val=$2;

#      lib::debug("Pre-processing $var=$val");

      $val=~s/\+/ /g;
      $val=uri_unescape($val);

#      lib::debug("Processing $var=$val");

      if($var=~/^CookierAction$/) {
        $final=2;
        } elsif($var=~/^header_name_(\d+)$/) {
        $header{"$1"}{"name"}=$val;
        } elsif($var=~/^header_value_(\d+)$/) {
        $header{"$1"}{"value"}=$val;
        } elsif($var=~/^header_sticky_(\d+)$/) {
        $header{"$1"}{"sticky"}="1";
        } elsif($var=~/^cookie_name_(\d+)$/) {
        $cookie{"$1"}{"name"}=$val;
        } elsif($var=~/^cookie_value_(\d+)$/) {
        $cookie{"$1"}{"value"}=$val;
        } elsif($var=~/^cookie_sticky_(\d+)$/) {
        $cookie{"$1"}{"sticky"}="1";
        } elsif($var=~/^additional_header_name_(\d+)$/) {
        $additional_header{"$1"}{"name"}=$val;
        } elsif($var=~/^additional_header_value_(\d+)$/) {
        $additional_header{"$1"}{"value"}=$val;
        } elsif($var=~/^additional_header_sticky_(\d+)$/) {
        $additional_header{"$1"}{"sticky"}="1";
        } elsif($var=~/^additional_cookie_name_(\d+)$/) {
        $additional_cookie{"$1"}{"name"}=$val;
        } elsif($var=~/^additional_cookie_value_(\d+)$/) {
        $additional_cookie{"$1"}{"value"}=$val;
        } elsif($var=~/^additional_cookie_sticky_(\d+)$/) {
        $additional_cookie{"$1"}{"sticky"}="1";
        } else {
        $FORM{"$var"}=$val;
        }
      }
    }

  if($FORM{"method"}) {
    $proxyRequest->method($FORM{"method"});
    } else {
    return;
    }

  if($FORM{"proto"}) {
    $proxyRequest->protocol($FORM{"proto"});
    } else {
    return;
    }

  if($FORM{"url_host"} && $FORM{"url_path"}) {
    if($CURRENT{'is_ssl'}) {
      $proxyRequest->uri("https://".$FORM{"url_host"}.$FORM{"url_path"});
      } else {
      $proxyRequest->uri("http://".$FORM{"url_host"}.$FORM{"url_path"});
      }
    } else {
    return;
    }

  foreach $k (keys %header) {
    if($header{$k}{'name'}) {
      $proxyRequest->header($header{$k}{'name'},$header{$k}{'value'});

#    lib::debug("Req setting header ".$header{$k}{'name'}." to <".$header{$k}{'value'}.">");

#      if($header{$k}{'sticky'}) {
#        $client_data.=" H".MIME::Base64::encode_base64("$header{$k}{'name'}:$header{$k}{'value'}");
#        chop($client_data);
#        }
      }
    }

  $cookie_counter=0;
  foreach $k (keys %cookie) {
    if($cookie{$k}{'name'} && ($cookie{$k}{'name'} ne "HTTPUSHDATA")) {

      if($proxyRequest->header("Cookie")) {
        $tmp=$proxyRequest->header("Cookie")."; $cookie{$k}{'name'}=$cookie{$k}{'value'}";
        } else {
        $tmp="$cookie{$k}{'name'}=$cookie{$k}{'value'}";
        }

      $proxyRequest->header("Cookie",$tmp);

#      lib::debug("Req setting cookie ".$cookie{$k}{'name'}." to <".$cookie{$k}{'value'}.">");

#      if($cookie{$k}{'sticky'}) {
#        $client_data.=" C".MIME::Base64::encode_base64("$cookie{$k}{'name'}:$cookie{$k}{'value'}");
#        chop($client_data);
#        }

      $cookie_counter++;
      }
    }

  foreach $k (keys %additional_header) {
    if($additional_header{$k}{'name'}) {
      $proxyRequest->header($additional_header{$k}{'name'},$additional_header{$k}{'value'});

      if($additional_header{$k}{'sticky'}) {
        $client_data.=" H".MIME::Base64::encode_base64("$additional_header{$k}{'name'}:$additional_header{$k}{'value'}");
        chop($client_data);
        }
      }
    }

  $cookie_counter=0;
  foreach $k (keys %additional_cookie) {
    if($additional_cookie{$k}{'name'}) {

      if($proxyRequest->header("Cookie")) {
        $tmp=$proxyRequest->header("Cookie")."; $additional_cookie{$k}{'name'}=$additional_cookie{$k}{'value'}";
        } else {
        $tmp="$additional_cookie{$k}{'name'}=$additional_cookie{$k}{'value'}";
        }

      $proxyRequest->header("Cookie",$tmp);

      if($additional_cookie{$k}{'sticky'}) {
        $client_data.=" C".MIME::Base64::encode_base64("$additional_cookie{$k}{'name'}:$additional_cookie{$k}{'value'}");
        chop($client_data);
        }

      $cookie_counter++;
      }
    }

  if($FORM{"content"}) {
    $proxyRequest->content($FORM{"content"});
    }

  return ($final, $proxyRequest, $client_data);
  }

sub PrintForm {
  my($ssl, $url_path, $url_host, $r)=@_;
  my($headers, %h, $tmp, $k, $cookies, %cookies, $tmp2);

  hprint($ssl,"HTTP/1.1 200 OK\r\n");
  hprint($ssl,"Cache-Control: no-cache\r\n");
  hprint($ssl,"Pragma: no-cache\r\n");
  hprint($ssl,"Content-Type: text/html\r\n");
  hprint($ssl,"\r\n");

  hprint($ssl,"<HTML>");
  hprint($ssl,"<HEAD><TITLE>$HTTPUSH_OPTIONS{'title'} - ".$r->uri()."</TITLE>");
  hprint($ssl,'
  <SCRIPT>
  function change () {
    i=0;
    done=0;

    while(document.forms.main.elements["header_name_"+i] && (done == 0)) {
      if(document.forms.main.elements["header_name_"+i].value=="content-length") {
        document.forms.main.elements["header_value_"+i].value=document.forms.main.elements["content"].value.length;
        done=1;
      }
      
      i++;
    }
  }
  </SCRIPT>
  ');

  hprint($ssl,HTMLPrintStyle());

  hprint($ssl,'</HEAD>');

  hprint($ssl,"<BODY OnLoad=\"document.forms.main.go_button.focus(); change()\">");

  hprint($ssl,"<FORM NAME=\"main\" METHOD=\"POST\" ACTION=\"".$r->uri()."\">\n");

  # Request header

  hprint($ssl,"<TABLE WIDTH=100% CELLSPACING=\"0\" BORDER=\"0\">");
  hprint($ssl,"<TR WIDTH=\"100%\"><TD CLASS=\"header\">The Request</TD></TR>");
  hprint($ssl,"</TABLE>");

  # Request main details

  hprint($ssl,"<TABLE CLASS=\"request\" CELLSPACING=\"0\" BORDER=\"0\">");

  my $len=int(length($url_path)*1.2);

  if($len<20) {
    $len=20;
    }

  hprint($ssl,"<TR CLASS=\"red2\"><TD CLASS=\"mini2\"><BR><INPUT TYPE=\"submit\" NAME=\"go_button\" VALUE=\"Go!\"></TD><TD CLASS=\"mini2\">Method<BR><INPUT TYPE=\"text\" NAME=\"method\" VALUE=\"".$r->method()."\" SIZE=\"7\"></TD><TD CLASS=\"mini2\">&nbsp;</TD><TD CLASS=\"mini2\">URI<BR><INPUT TYPE=\"text\" NAME=\"url_path\" VALUE=\"$url_path\" SIZE=\"$len\"></TD><TD CLASS=\"mini2\">&nbsp;</TD><TD CLASS=\"mini2\">Protocol<BR><INPUT TYPE=\"text\" NAME=\"proto\" VALUE=\"".$r->protocol()."\"></TD></TR>");

  if($r->content) {
    my $content=uri_unescape($r->content);

    $content=~s/&/&amp;/gi;

    hprint($ssl,"<TR><TD CLASS=\"mini2\" COLSPAN=\"6\">POST Content<BR><TEXTAREA NAME=\"content\" ROWS=\"5\" COLS=\"100\" OnChange=\"change()\">$content</TEXTAREA></TD></TR>\n");
    }

  hprint($ssl,"</TABLE>");

  # Cookies

  $headers=$r->{'_headers'};
  %h=%$headers;

  $cookies="";

  foreach $k (keys %h) {
    if($k=~/^cookie\d*/) {
      my (@vals, $v, @v);
      @vals=split(";",$h{$k});
      foreach $v (@vals) {
        @v=split("=",$v,2);

        if($v[0]=~/^\s*(.*)\s*$/) {
          $v[0]=$1;
          }

        if($v[1]=~/^\s*(.*)\s*$/) {
          $v[1]=$1;
          }

        $cookies{$v[0]}=$v[1];
        }
      }
    }

  if(keys %cookies) {
    hprint($ssl,"<TABLE WIDTH=100% CELLSPACING=\"0\" BORDER=\"0\">");
    hprint($ssl,"<TR><TD CLASS=\"header\"><B>Cookies</B></TD></TR>");

    $tmp=0;

    foreach $k (sort keys %cookies) {
      $len=int(length($cookies{"$k"})*1.5);
      if($len<20) {
        $len=20;
        }

      if($k ne "HTTPUSHDATA") {
        hprint($ssl,"<TR><TD CLASS=\"mini\">$k&nbsp;&nbsp;<INPUT TYPE=\"hidden\" NAME=\"cookie_name_$tmp\" VALUE=\"$k\"><INPUT TYPE=\"text\" NAME=\"cookie_value_$tmp\" VALUE=\"$cookies{$k}\" SIZE=\"$len\"></TD></TR>\n");
        } else {
        hprint($ssl,"<INPUT TYPE=\"hidden\" NAME=\"cookie_name_$tmp\" VALUE=\"$k\"><INPUT TYPE=\"hidden\" NAME=\"cookie_value_$tmp\" VALUE=\"$cookies{$k}\" SIZE=\"$len\">\n");
        }
      $tmp++;
      }
    }

  hprint($ssl,"</TABLE>");

  # Request headers

  hprint($ssl,"<TABLE WIDTH=\"100%\" CELLSPACING=\"0\" BORDER=\"0\">\n");

  hprint($ssl,"<TR WIDTH=\"100%\"><TD CLASS=\"header\"><B>Request Headers</B></TD></TR>");

  # Print alphabetically sorted list of headers. We could group them here

  $tmp=0;
  foreach $k (sort keys %h) {
    if($k!~/^cookie\d*/) {
      $len=int(length($h{"$k"})*1.5);

      if($len<20) {
        $len=20;
        }

      hprint($ssl,"<TR WIDTH=\"100%\"><TD CLASS=\"mini\">$k&nbsp;&nbsp;<INPUT TYPE=\"hidden\" NAME=\"header_name_$tmp\" VALUE=\"$k\"><INPUT TYPE=\"text\" NAME=\"header_value_$tmp\" VALUE=\"$h{$k}\" SIZE=\"$len\"></TD></TR>\n");
      $tmp++;
      }
    }

  hprint($ssl,"</TABLE>");

  # User defined request headers

  hprint($ssl,"<TABLE WIDTH=\"100%\" CELLSPACING=\"0\" BORDER=\"0\">\n");
  hprint($ssl,"<TR WIDTH=\"100%\"><TD CLASS=\"header\">&nbsp;</TD><TD CLASS=\"header\"><B>Additional Headers</B></TD><TD CLASS=\"header\">&nbsp;</TD><TD CLASS=\"header\">&nbsp;</TD><TD CLASS=\"header\"><B>Additional Cookies</B></TD></TR>\n");

  # Print alphabetically sorted list of headers. We could group them here
  $tmp=0;

  if($cookies{HTTPUSHDATA}) {
    my %cheaders=ClientDataRead($cookies{HTTPUSHDATA},0);
    my %ccookies=ClientDataRead($cookies{HTTPUSHDATA},1);

    my @lcookies=keys %ccookies;
    my @lheaders= keys %cheaders;

    my ($cook,$head);

    $cook=pop(@lcookies);
    $head=pop(@lheaders);

    while($cook || $head) {
      if($head) {
        hprint($ssl,"<TR WIDTH=\"100%\"><TD><INPUT TYPE=\"checkbox\" NAME=\"additional_header_sticky_$tmp\" CHECKED></TD>");
        hprint($ssl,"<TD CLASS=\"mini\"><INPUT TYPE=\"text\" NAME=\"additional_header_name_$tmp\" VALUE=\"$head\">:&nbsp;<INPUT TYPE=\"text\" name=\"additional_header_value_$tmp\" VALUE=\"$cheaders{$head}\"></TD>\n");
        } else {
        hprint($ssl,"<TR WIDTH=\"100%\"><TD><INPUT TYPE=\"checkbox\" NAME=\"additional_header_sticky_$tmp\"></TD>");
        hprint($ssl,"<TD CLASS=\"mini\"><INPUT TYPE=\"text\" NAME=\"additional_header_name_$tmp\">:&nbsp;<INPUT TYPE=\"text\" name=\"additional_header_value_$tmp\"></TD>\n");
        }

      if($cook) {
        hprint($ssl,"<TD>&nbsp;</TD><TD><INPUT TYPE=\"checkbox\" NAME=\"additional_cookie_sticky_$tmp\" CHECKED></TD>");
        hprint($ssl,"<TD CLASS=\"mini\"><INPUT TYPE=\"text\" NAME=\"additional_cookie_name_$tmp\" VALUE=\"$cook\">:&nbsp;<INPUT TYPE=\"text\" name=\"additional_cookie_value_$tmp\" VALUE=\"$ccookies{$cook}\"></TD></TR>\n");
        } else {
        hprint($ssl,"<TD>&nbsp;</TD><TD><INPUT TYPE=\"checkbox\" NAME=\"additional_cookie_sticky_$tmp\"></TD>");
        hprint($ssl,"<TD CLASS=\"mini\"><INPUT TYPE=\"text\" NAME=\"additional_cookie_name_$tmp\">:&nbsp;<INPUT TYPE=\"text\" name=\"additional_cookie_value_$tmp\"></TD></TR>\n");
        }

      $cook=pop(@lcookies);
      $head=pop(@lheaders);

      $tmp++;
      }
    }

  for($tmp2=$tmp;$tmp2 < $tmp+5;$tmp2++) {
    hprint($ssl,"<TR WIDTH=\"100%\"><TD><INPUT TYPE=\"checkbox\" NAME=\"additional_header_sticky_$tmp2\" VALUE=\"\"></TD>");
    hprint($ssl,"<TD CLASS=\"mini\"><INPUT TYPE=\"text\" NAME=\"additional_header_name_$tmp2\">:&nbsp;<INPUT TYPE=\"text\" name=\"additional_header_value_$tmp2\"></TD>\n");

    hprint($ssl,"<TD>&nbsp;</TD><TD><INPUT TYPE=\"checkbox\" NAME=\"additional_cookie_sticky_$tmp2\" VALUE=\"\"></TD>");
    hprint($ssl,"<TD CLASS=\"mini\"><INPUT TYPE=\"text\" NAME=\"additional_cookie_name_$tmp2\">:&nbsp;<INPUT TYPE=\"text\" name=\"additional_cookie_value_$tmp2\"></TD></TR>\n");
    }

  hprint($ssl,"</TABLE>");

  # Next

  hprint($ssl,"<INPUT TYPE=\"hidden\" NAME=\"CookierAction\" VALUE=\"1\">");
  hprint($ssl,"<INPUT TYPE=\"hidden\" NAME=\"url_host\" VALUE=\"$url_host\">");
  hprint($ssl,"</FORM>");

  hprint($ssl,"</BODY>");
  hprint($ssl,"</HTML>");
  }

sub FakeSSL {
  my $r=shift;

  $CURRENT{'ssl'} = Net::SSLeay::new($CURRENT{'ctx'})      or die_now("SSL_new ($CURRENT{'ssl'}): $!");
  Net::SSLeay::set_fd($CURRENT{'ssl'}, fileno(Client));

  print "HTTP/1.1 200 OK\r\n\r\n";
  
  Net::SSLeay::accept($CURRENT{'ssl'}) and die_if_ssl_error('ssl accept');

  Debug ("Fake SSL session established (".Net::SSLeay::get_cipher($CURRENT{'ssl'}).")");

  my $x=GetSSLRequest($r);

  return $x;
  }

sub GetSSLRequest {
  my($pc)=@_;
  my $last_blank=0;

  my($req,$line,$content,%headers, $first, $second, $h, $got, $in, $i, $c);

  my $end=0;

  my $state=0;
  my $hstate=0;
  my $key="";
  my $val="";
  my $crlf=0;
  my $crlf2=0;
  my $clen=0;
  $content="";
  $last_blank="";
  my $skip=0;
  my $firstcontent=0;
  my $blank=0;

  while((! $end ) && ($in=Net::SSLeay::read($CURRENT{'ssl'}))) {

    for($i=0;$i<length($in);$i=$i+1) {

        $c=substr($in,$i,1);

        if($c eq "\r")
          {
          $crlf=1;
          }
        elsif($c eq "\n")
          {
          if($crlf==1)
            {
            $blank++;
            $crlf++;
            }
          }
        else
         {
         $blank=0;
         $crlf2=0;
         }

#      print STDERR ("READ (\"$c\" STATE $state HEADERS $hstate CRLF $crlf BLANKS $blank)\n");

      if($state == 0) # Request line
        {
        if($crlf==2)
          {
          $state++;
          $crlf=0;

          if($req=~/^\s*([^\r\n]*)[\r\n]*$/)
            {
            $req=$1;
            }

          next;
          }
        $req.=$c;
        }
      elsif($state == 1) # Headers
        {
        if(($hstate==0) && ($c eq ":")) 
          {
          $hstate++;
          }
        elsif($crlf==2 && $blank==2)
          {
          $state++;
          $blank=0;

          $firstcontent=1;
          }
        elsif($crlf==2 && $blank==1)
          {
          $key=lc($key);

	  if($key eq "referer" && $val=~/^(.*)&HTTPUSH_ITEM=\d+$/) { 
	    $val=$1;
	    }

          if($key ne "connection" && $key ne "proxy-connection")
            {
#            lib::debug("Setting $key - $val");

            if($val=~/^\s*([^\r\n]*)[\r\n]*$/)
              {
              $val=$1;
              }

#            lib::debug("Setting $key - $val");
            $headers{$key}="$val";
            }

            $key="";
            $val="";

            $hstate=0;

          }
        else
          {
          if($hstate==0) { $key.=$c; } else { $val.=$c; }
          }
        }

      if($state==2) # Content or end of request
        {
        if($clen < $headers{'content-length'})
          {
          if(! $firstcontent)
            {
            $clen++;
            $content.=$c;
            }
          }
        if($clen >= $headers{'content-length'})
          {
#          lib::debug("ENDING ($content)");
          $end=1;
          last; # Exit loop, the request has finished
          }
        $firstcontent=0;
        }
      }
#    lib::debug("Aqui ($end)");
#    print STDERR ("- READ (\"$c\" STATE $state HEADERS $hstate CRLF $crlf BLANKS $blank)\n");
    }

#   lib::debug("Found <$content>");

  # Done reading request, process it

#  lib::debug("Received ($content)");

  my $r = new HTTP::Request;

  if($req=~/([^\s]+)\s*([^\s]+)\s*([^\s]+)/) {
    $r->method($1);
    $r->uri("https://".$pc->uri()."$2");
    $r->protocol($3);
    } else {
    # Error
    return;
    }

  $r->content($content);

  foreach $h (keys %headers) {
    $r->header($h,$headers{$h});
    }

  if($r->header("User-Agent") =~/MSIE/) { # Disable Keep-Alives in MSIE SSL clients, to disable connection reuse
    $r->header("Connection","");
  }

  return $r;
  }

sub ClientDataRead {
  my ($val, $op)=@_;    # 0 - additional headers, 1 - additional cookies
  my (%data, @dec, @dec2, $v);

  @dec=split(" ",$val);

  foreach $val (@dec) {
    if($val=~/\s*H([^\s]*)/ && ($op==0)) {
      $v=MIME::Base64::decode_base64($1);
      @dec2=split(":",$v);
      if($dec2[0] && $dec2[1]) {
        $data{$dec2[0]}=$dec2[1];
        }
      }

    if($val=~/\s*C([^\s]*)/ && ($op==1)) {
      $v=MIME::Base64::decode_base64($1);
      @dec2=split(":",$v);
      if($dec2[0] && $dec2[1]) {
        $data{$dec2[0]}=$dec2[1];
        }
      }
    }
  return (%data);
  }

sub StoredList {
  my ($file, $id, $query_ptr)=@_;
  my (%query)=%{$query_ptr};
  my ($server,$path, $element, $twig);

  if(defined $file) {
    if(defined $id) {
      if($file && -f "$DATADIR/$file.xml") {
        $twig= new XML::Twig(KeepSpaces => 1);
        open(PFILE,"< $DATADIR/$file.xml");
        flock PFILE, 1;
        $twig->safe_parse(*PFILE) || do { flock PFILE, 8; &my_error(7,"2 $!"); return undef; };
        flock PFILE, 8;
        close(PFILE);

        $element=$twig->elt_id($id);

      } else {
        &my_error(4);
        return undef;
      }
    }

    StoredListServer($file, $element);

    if(defined $twig) {
      $twig->purge;
    }

  } else {
  # Top level view
  StoredListServers();
  }

  return undef;
}

# Data view functions

sub StoredView {
  my ($file, $id, $query_ptr)=@_;
  my (%query)=%{$query_ptr};

  my ($twig);

  if(defined $file) {
    if(defined $id) {
      if($file && -f "$DATADIR/$file.xml") {
        $twig= new XML::Twig(KeepSpaces => 1);
        open(PFILE,"< $DATADIR/$file.xml");
        flock PFILE, 1;
        $twig->safe_parse(*PFILE) || do { flock PFILE, 8; &my_error(7,"2 $!"); return undef;};
        flock PFILE, 8;
        close(PFILE);
      } else {
        &my_error(4);
        return undef;
      }

      my $element=$twig->elt_id($id);

      if(! $element) {
        $twig->purge;
        &my_error(4);
        return undef;
      }

      if($element->gi eq "request") {
        # View request details
        StoredViewRequest($element, $query{'path'});
      } elsif ($element->gi eq "vuln") {
        StoredViewVulnerability($element);
      }
      $twig->purge;
    }
  }
  return undef;
}

# StoredViewRequest: Display one request properties.
#

sub StoredViewRequest {
  my ($request, $path) = @_;
  my ($twig, $fqdn, $port, $client, $reqline, $resline, @req, @res, $res_code);

#  my @all_dirs = $twig->root->descendants('directory');
#  my $root_dir=@all_dirs[0];

  if(! $request) {
    &my_error(4);
    return undef;
  }

  # Start display

  print "HTTP/1.1 200 OK\r\n";
  print "Content-Type: text/html\r\n";
  print "Connection: Close\r\n";
  print "\r\n";
  print "<HTML>";
  print "<HEAD>";

  print HTMLPrintStyle();

  print "<TITLE>Request view</TITLE></HEAD><BODY>";

  # Description
  # FQDNs

  print "<TABLE CELLSPACING=\"0\" BORDER=\"0\">\n";

  my $my_time = strftime "%d/%m/%Y %H:%M:%S", localtime($request->att('timestamp'));

  my $sent=$request->first_child('sent');
  my $received=$request->first_child('received');
  my $who=$request->first_child('who');

  if($sent) {
    $sent=MIME::Base64::decode_base64($sent->text) || die("$!");
    @req=split("\n",$sent);
    $reqline=$req[0];
    chomp($reqline);
  }

  if($received) {
    $received=MIME::Base64::decode_base64($received->text) || die("$!");
    @res=split("\n",$received);
    $resline=$res[0];
    chomp($resline);
    if($resline=~/^HTTP\/\d+\.\d+\s+(\d+)/) {
      $res_code=$1;
    }
  }

  if($who) {
    $client=$who->text;
  }

  my $sent_len=length($sent);
  my $received_len=length($received);

  my $html_received=$received;
  my $html_sent=$sent;

  $html_received=~s/</&lt;/g;
  $html_received=~s/>/&gt;/g;

  $html_sent=~s/</&lt;/g;
  $html_sent=~s/>/&gt;/g;


  print "<TR><TD CLASS=\"header\">Date</TD><TD>&nbsp;</TD><TD>".$my_time."</TD><TD>&nbsp;</TD><TD CLASS=\"header\">Client</TD><TD>&nbsp;</TD><TD>".$client."</TD></TR>\n";

  # Hook here to run request plugin
#  print "<TR><TD CLASS=\"header\">Request</TD><TD>&nbsp;</TD><TD>".$sent_len." bytes</TD><TD>&nbsp;</TD><TD CLASS=\"header\"><FORM>".$plugin->print_select()."<INPUT TYPE=\"hidden\" VALUE=\"Run\"></FORM></TD><TD>&nbsp;</TD><TD></TD></TR>\n";
  print "<TR WIDTH=\"100%\"><TD COLSPAN=\"10\"><TABLE WIDTH=\"100%\" BGCOLOR=\"#dddddd\" BORDER=\"1\"><TR><TD><PRE>$html_sent</PRE></TD></TR></TABLE></TD></TR>";

  # Hook here to run response plugin
#  print "<TR><TD CLASS=\"header\">Response</TD><TD>&nbsp;</TD><TD>".$received_len." bytes</TD><TD>&nbsp;</TD><TD CLASS=\"header\"><FORM>".$plugin->print_select()."<INPUT TYPE=\"hidden\" VALUE=\"Run\"></FORM></TD><TD>&nbsp;</TD><TD></TD></TR>\n";
  print "<TR WIDTH=\"100%\"><TD COLSPAN=\"10\"><TABLE WIDTH=\"100%\" BGCOLOR=\"#eeee00\" BORDER=\"1\"><TR><TD><PRE>$html_received</PRE></TD></TR></TABLE></TD></TR>";

  print "</TABLE>";
  print "</BODY></HTML>";

  return undef;
}

# StoredViewVulnerability: Display one vulnerability properties.
#

sub StoredViewVulnerability {
  my ($vuln) = @_;

  if(! $vuln) {
    &my_error(4);
    return undef;
  }

  # Start display

  print "HTTP/1.1 200 OK\r\n";
  print "Content-Type: text/html\r\n";
  print "Connection: Close\r\n";
  print "\r\n";
  print "<HTML>";
  print "<HEAD>";

  print HTMLPrintStyle();

  print "<TITLE>Vulnerability view</TITLE></HEAD><BODY>";

  # Description
  # FQDNs

  print "<TABLE CELLSPACING=\"0\" BORDER=\"0\">\n";


  my $my_level=$vuln->att('level');
  my $title=$vuln->att('title');
  my $bid=$vuln->att('bid');
  my $cve=$vuln->att('cve');
  my $timestamp=$vuln->att('timestamp');
  my $plugin=$vuln->att('plugin');

  my $content=$vuln->text;

  if($content) {
    $content=MIME::Base64::decode_base64($content) || return undef;
  }

  my $my_time = strftime "%d/%m/%Y %H:%M:%S", localtime($timestamp);

  print "<TR><TD CLASS=\"header\">Vulnerability</TD><TD>&nbsp;</TD><TD COLSPAN=\"10\">".$title."</TD></TR>\n";
  print "<TR><TD CLASS=\"header\">Date</TD><TD>&nbsp;</TD><TD>".$my_time."</TD><TD>&nbsp;</TD><TD CLASS=\"header\">Plugin</TD><TD>&nbsp;</TD><TD>".$plugin."</TD><TD>&nbsp;</TD><TD CLASS=\"header\">BID</TD><TD>&nbsp;</TD><TD>".$bid."</TD><TD>&nbsp;</TD><TD CLASS=\"header\">CVE</TD><TD>&nbsp;</TD><TD>".$cve."</TD></TR>\n";
  print "<TR WIDTH=\"100%\"><TD COLSPAN=\"10\"><TABLE WIDTH=\"100%\" BGCOLOR=\"#dddddd\" BORDER=\"1\"><TR><TD>$content</TD></TR></TABLE></TD></TR>";

  print "</TABLE>";
  print "</BODY></HTML>";
  return undef;
}

# Data list functions

sub StoredListServers {
 my ($id,$file,$desc,$fqdn, $hostcount);

 print "HTTP/1.1 200 OK\r\n";
 print "Content-Type: text/html\r\n";
 print "Connection: Close\r\n";
 print "\r\n";
 print "<HTML>";
 print "<HEAD>";

 print HTMLPrintStyle();

 print "<TITLE>Browse servers</TITLE></HEAD>";

 open(INDEX,"< $DATADIR/.index") || die("Can't open index file");
 flock (INDEX, 1);

 print "<BODY>";
 print "<TABLE WIDTH=\"100%\" CELLSPACING=\"0\" BORDER=\"0\">\n";

 print "<TR WIDTH=\"100%\"><TD WIDTH=\"10%\" CLASS=\"header\">&nbsp;</TD><TD CLASS=\"header\"><B>FQDN</B</TD></TR>\n";

 $hostcount=0;

 while(<INDEX>) {
  if(/^(\d+)\s+\"([^\"]*)\"\s+(.*)$/) {
   $id=$1;
   $desc=$2 || "Undefined";
   $fqdn=$3 || "Undefined";

   print "<TR><TD><A HREF=\"/Browse/$id\">[+]</A></TD><TD>$fqdn</TD></TR>\n";
   $hostcount++;
   }
  }

  if($hostcount) {
    print "<TR><TD COLSPAN=\"3\"><center><i>$hostcount servers audited</i></center></TD></TR>";
  } else {
    print "<TR><TD COLSPAN=\"3\"><center><i>No servers have been audited</i></center></TD></TR>";
  }

 print "</TABLE>";
 print "</BODY></HTML>";
 flock (INDEX, 8);
 close(INDEX);

 return undef;
 }

sub RecPrintDir {
  my ($file, $element, $dir, $path, $indent, $server_url)=@_;
  my ($child_dir, $tmp_ret);

  my $bgcolorsw=0;
  my $ret=undef;

  # Print directories

  my $dir_path=$dir->att('path');
  my $requests=$dir->children('request');
  my $vulns=$dir->children('vuln');

  $path.=$dir_path;

  my $id=$dir->att('id');

  if(defined $element) {
    if($id == $element->att('id')) {
      $ret=$dir;
    }
  }

  my $print_path="&nbsp;&nbsp;"x($indent-1)."$path";

  print "<TR CLASS=\"h1\"><TD CLASS=\"h1\">";

  if(! defined $ret) {
    print "<A HREF=\"/Browse/$file/$id\">[+]</A>";
  } else {
    print "[ ]";
  }

  print "</TD><TD CLASS=\"h1\">$vulns</TD><TD CLASS=\"h1\"><A HREF=\"".$server_url.$path."\">$print_path</A> ($requests requests)</TD></TR>\n";

  my @pages = $dir->children('page');
  my $page;

  foreach $page (@pages) {
    my $this_is_the_one=0;
    my ($class, $print_page_name);
    my $requests=$page->children('request');
    my $vulns=$page->descendants('vuln');
    my $page_name=$page->att('name');

    my $current_item_path=$path.$page_name;
    my $id=$page->att('id');

    if(defined $element) {
      if($page->att('id') == $element->att('id')) {
        $ret=$page;
        $this_is_the_one=1;
      }
    }

    # Pick a nice color for the table background
    my ($page_name_name, $page_ext)=split(/\./, $page_name);

    $page_ext=lc($page_ext);

    if(grep($page_ext eq $_, @EXT_IMAGES)) {
      $class="lightgray";
    } elsif(grep($page_ext eq $_, @EXT_INTERESTING)) {
      $class="lightred";
    } elsif (! $page_ext) {
      $class="orange";
    } else {
      $class="h2";
    }

    $print_page_name="&nbsp;&nbsp;"x$indent."$page_name";

    print "<TR CLASS=\"$class\"><TD>";

    if(! $this_is_the_one) {
      print "<A HREF=\"/Browse/$file/$id\">[+]</A>";
    } else {
      print "[ ]";
    }

    print "</TD><TD>$vulns</TD><TD>$print_page_name ($requests requests)</TD></TR>\n";
  }

  my @directories = $dir->children('directory');

  foreach $child_dir (@directories) {
    $tmp_ret=RecPrintDir($file, $element, $child_dir, $path, $indent+1, $server_url);
    if(defined $tmp_ret) {
      $ret=$tmp_ret;
    }
  }

  return $ret;
}

sub RecGetItem {
  my ($item_path, $dir, $path)=@_;
  my ($child_dir, $page, $tmp_ret);

  my $ret=undef;

  # Print directories

  my $dir_path=$dir->att('path');
  my $requests=$dir->children('request');
  my $vulns=$dir->children('vuln');

  $path.=$dir_path;

  if($path eq $item_path) {
    $ret=$dir;
  }

  my @pages = $dir->children('page');

  foreach $page (@pages) {
    my $page_name=$page->att('name');

    my $current_item_path=$path.$page_name;

    if($current_item_path eq $item_path) {
      $ret=$page;
    }
  }

  my @directories = $dir->children('directory');

  foreach $child_dir (@directories) {
    $tmp_ret=RecGetItem($item_path, $child_dir, $path);
    if($tmp_ret) {
      $ret=$tmp_ret;
    }
  }

  return $ret;
}


# Display the server directories tree and malformed requests. Optionally expand the provided path.
# This function loads and parses the server XML tree, and displays it in XHTML.

sub StoredListServer {

 my ($file, $element)=@_;

 my ($id, $tmp, $dir, @directories, $twig, $my_level);

 if(! $file) {
   &my_error(4);
   return undef;
   }

 my $dir_indent=0;

if($file && -f "$DATADIR/$file.xml")
  {
  $twig= new XML::Twig(KeepSpaces => 1);
  open(PFILE,"< $DATADIR/$file.xml");
  flock PFILE, 1;
  $twig->safe_parse(*PFILE) || do { flock PFILE, 8; &my_error(7,"3 $!"); return undef; };
  flock PFILE, 8;
  close(PFILE);
  }
else
  {
  &my_error(4);
  return undef;
  }

 print "HTTP/1.1 200 OK\r\n";
 print "Content-Type: text/html\r\n";
 print "Connection: Close\r\n";
 print "\r\n";
 print "<HTML><HEAD>";

 print HTMLPrintStyle();

 print "<TITLE>Browse server</TITLE></HEAD>";
 print "<BODY>\n";

# Print out server information

# Description
# FQDNs
my $server=$twig->root->first_child('server');

print "<CENTER><TABLE CELLSPACING=\"0\" BORDER=\"0\">\n";

  my @fqdns=$server->children("fqdn");
  my $fqdn;

  foreach $fqdn (@fqdns) {
    print "<TR CLASS=\"h1\" WIDTH=\"100%\"><TD CLASS=\"h1\">Server</TD><TD WIDTH=\"10\">&nbsp;</TD><TD>".$fqdn->att('name')."</TD><TD WIDTH=\"10\">&nbsp;</TD><TD>&nbsp;</TD></TR>\n";
  }

# Ports

  my @ports=$server->children("port");
  my $port;

  foreach $port (@ports) {

    # Protocols, Vulns, OS, Software, Directory, Request

    my @protocols=$port->children("protocol");
    my $protocol_list=undef;
    if(@protocols) {
      my $protocol;
      foreach $protocol (@protocols) {
        if($protocol_list) {
          $protocol_list.=",";
        }
        $protocol_list.=$protocol->att('name');
      }
     } else { $protocol_list="&nbsp;"; }

    my @oss=$port->children("os");
    my $os_list=undef;

    if(@oss) {
      my $os;
      foreach $os (@oss) {
        if($os_list) {
          $os_list.=",";
        }
        $os_list.=$os->text;
      }
    } else { $os_list="&nbsp;"; }

    my @softwares=$port->children("sofware");

    my $software_list=undef;

    if(@softwares) {
      my $software;
      foreach $software (@softwares) {
        if($software_list) {
          $software_list.=",";
        }
        $software_list.=$software->text;
      }
    } else { $software_list="&nbsp;"; }


    my $directories=$port->descendants("directory");
    my $requests=$server->descendants("request");
    my $vulnerabilities=$server->descendants("vuln");

#    print "<TR WIDTH=\"100%\"><TD CLASS=\"header\">Port</TD><TD WIDTH=\"10\" CLASS=\"header\">&nbsp;</TD><TD CLASS=\"\">".$port->att('number')."</TD><TD WIDTH=\"10\" CLASS=\"header\">$protocol_list</TD><TD CLASS=\"header\">&nbsp;</TD></TR>\n";
#    print "<TR WIDTH=\"100%\"><TD CLASS=\"header\">Operating System</TD><TD WIDTH=\"10\" CLASS=\"header\">&nbsp;</TD><TD CLASS=\"\">".$oss_list."</TD><TD WIDTH=\"10\" CLASS=\"header\">$oss_list</TD><TD CLASS=\"header\">&nbsp;</TD></TR>\n";
#    print "<TR WIDTH=\"100%\"><TD CLASS=\"header\">Server software</TD><TD WIDTH=\"10\" CLASS=\"header\">&nbsp;</TD><TD CLASS=\"\">".$software_list."</TD><TD WIDTH=\"10\" CLASS=\"header\">$software_list</TD><TD CLASS=\"header\">&nbsp;</TD></TR>\n";

#    print "<TR WIDTH=\"100%\"><TD CLASS=\"header\">Directories</TD><TD WIDTH=\"10\" CLASS=\"header\">&nbsp;</TD><TD CLASS=\"\">".$directories."</TD><TD WIDTH=\"10\" CLASS=\"header\">&nbsp;</TD><TD CLASS=\"header\">&nbsp;</TD></TR>\n";
#    print "<TR WIDTH=\"100%\"><TD CLASS=\"header\">Requests</TD><TD WIDTH=\"10\" CLASS=\"header\">&nbsp;</TD><TD CLASS=\"\">".$requests."</TD><TD WIDTH=\"10\" CLASS=\"header\">&nbsp;</TD><TD CLASS=\"header\">&nbsp;</TD></TR>\n";
    print "<TR WIDTH=\"100%\" CLASS=\"h1\"><TD CLASS=\"h1\"><A HREF=\"/Browse/$file\">Total vulnerabilities</A></TD><TD WIDTH=\"10\">&nbsp;</TD><TD>".$vulnerabilities."</TD><TD WIDTH=\"10\">&nbsp;</TD><TD>&nbsp;</TD></TR>\n";

  }

  print "</TABLE></CENTER>";

# Print out directories

 print "<TABLE><TR VALIGN=\"TOP\"><TD WIDTH=\"50%\" ROWSPAN=\"2\">";
 print "<TABLE WIDTH=\"100%\" CELLSPACING=\"0\" BORDER=\"0\">\n";
 print "<TR VALIGN=\"TOP\"><TD CLASS=\"section\" NOWRAP=\"true\" COLSPAN=\"9\"><CENTER>Directories tree</CENTER></TD></TR>";

# Print out results

my @all_dirs = $twig->root->descendants('directory');
my $root_dir=$all_dirs[0];

# Print directories

my $tmp_server=$twig->root->first_child('server');
my @tmp_fqdns=$tmp_server->children("fqdn");
my $tmp_fqdn=$tmp_fqdns[0]->att('name');
my @tmp_ports=$tmp_server->children("port");
my $tmp_port=$tmp_ports[0];
my $tmp_port_number=$tmp_port->att('number');
my @tmp_protocols=$tmp_port->children("protocol");
my $tmp_protocol=$tmp_protocols[0]->att('name');


if(($tmp_protocol eq "http") && ($tmp_port_number==80)) {
  $tmp_port_number="";
} elsif (($tmp_protocol eq "http") && ($tmp_port_number==443)) {
  $tmp_port_number="";
} else {
  $tmp_port_number=":".$tmp_port_number;
}

my $server_url=$tmp_protocol."://".$tmp_fqdn.$tmp_port_number;

my $current_item=&RecPrintDir($file, $element, $root_dir, "", 0, $server_url);

if(! defined $current_item) {
  $current_item=$server;
}

my $path;

if($current_item->gi eq "directory") {
  $path=$current_item->att('path');
} elsif ($current_item->gi eq "page") {
  $path=$current_item->att('name');
} else {
  $path="";
}

print "</TABLE></TD>";

# Print current path requests
print "<TD WIDTH=\"100%\">";
print "<TABLE WIDTH=\"100%\" CELLSPACING=\"0\" BORDER=\"0\">\n";

if($current_item && ($current_item->gi ne "server")) {

  print "<TR VALIGN=\"TOP\"><TD CLASS=\"section\" NOWRAP=\"true\" COLSPAN=\"11\"><CENTER>Requests for $path</CENTER></TD></TR>";
  print "<TR VALIGN=\"TOP\"><TD CLASS=\"header\">&nbsp;</TD><TD CLASS=\"header\">&nbsp;</TD><TD CLASS=\"header\" NOWRAP=\"true\"><B>Date</B></TD><TD WIDTH=\"10\" CLASS=\"header\" NOWRAP=\"true\">&nbsp;</TD><TD CLASS=\"header\" NOWRAP=\"true\">Result code</TD><TD WIDTH=\"10\" CLASS=\"header\" NOWRAP=\"true\">&nbsp;</TD><TD CLASS=\"header\" NOWRAP=\"true\">Sent</TD><TD WIDTH=\"10\" CLASS=\"header\" NOWRAP=\"true\">&nbsp;</TD><TD CLASS=\"header\" NOWRAP=\"true\">Received</TD><TD WIDTH=\"10\" CLASS=\"header\" NOWRAP=\"true\">&nbsp;</TD><TD CLASS=\"header\" NOWRAP=\"true\">Request line</TD></TR>\n";

  my @requests = $current_item->children('request');
  my $request_id=0;
  my $request;
  my (@req,@res,$reqline, $resline, $res_code);

  foreach $request (@requests) {
    my $my_time = strftime "%d/%m/%Y %H:%M:%S", localtime($request->att('timestamp'));

    my $sent=$request->first_child('sent');
    my $received=$request->first_child('received');
    my $who=$request->first_child('who');
    my $id=$request->att('id');

    if($sent) {
      $sent=MIME::Base64::decode_base64($sent->text) || die("$!");
      @req=split("\n",$sent);
      $reqline=$req[0];
      chomp($reqline);
    }

    if($received) {
      $received=MIME::Base64::decode_base64($received->text) || die("$!");
      @res=split("\n",$received);
      $resline=$res[0];
      chomp($resline);
      if($resline=~/^HTTP\/\d+\.\d+\s+(\d+)/) {
        $res_code=$1;
      }
    }

    my $sent_len=length($sent);
    my $received_len=length($received);

    my $class="orange";

    if(substr($res_code,0,1) eq "2") {
    $class="green";
    } elsif(substr($res_code,0,1) eq "4") {
    $class="yellow";
    } elsif(substr($res_code,0,1) eq "5") {
    $class="lightred";
    }

    print "<TR CLASS=\"$class\"><TD><A HREF=\"/View/$file/$id\">[*]</TD><TD>&nbsp;</TD><TD NOWRAP=\"true\">$my_time</TD><TD>&nbsp;</TD><TD>$res_code</TD><TD>&nbsp;</TD><TD>$sent_len</TD><TD>&nbsp;</TD><TD>$received_len</TD><TD>&nbsp;</TD><TD>$reqline</TD></TR>";
    $request_id++;
  }
}

print "</TABLE></TD></TR>";

print "<TR>";

# Print vulnerabilities

print "<TD WIDTH=\"100%\">";
print "<TABLE WIDTH=\"100%\" CELLSPACING=\"0\" BORDER=\"0\">\n";
print "<TR VALIGN=\"TOP\"><TD CLASS=\"section\" NOWRAP=\"true\" COLSPAN=\"11\"><CENTER>Vulnerabilities</CENTER></TD></TR>";
print "<TR VALIGN=\"TOP\"><TD CLASS=\"header\">&nbsp;</TD><TD CLASS=\"header\">&nbsp;</TD><TD CLASS=\"header\" NOWRAP=\"true\"><B>Date</B></TD><TD WIDTH=\"10\" CLASS=\"header\" NOWRAP=\"true\">&nbsp;</TD><TD CLASS=\"header\" NOWRAP=\"true\">Title</TD><TD WIDTH=\"10\" CLASS=\"header\" NOWRAP=\"true\">&nbsp;</TD><TD CLASS=\"header\" NOWRAP=\"true\">Bugtraq ID</TD><TD WIDTH=\"10\" CLASS=\"header\" NOWRAP=\"true\">&nbsp;</TD><TD CLASS=\"header\" NOWRAP=\"true\">CVE</TD><TD WIDTH=\"10\" CLASS=\"header\" NOWRAP=\"true\">&nbsp;</TD><TD CLASS=\"header\" NOWRAP=\"true\">Plugin</TD></TR>\n";

if($current_item) {
  my @vulnerabilities = $current_item->descendants('vuln');

  if(@vulnerabilities) {
    my $vulnerability;
    foreach $vulnerability (@vulnerabilities) {
      my $my_time = strftime "%d/%m/%Y %H:%M:%S", localtime($vulnerability->att('timestamp'));

      my $bid=$vulnerability->att('bid');

      my $id=$vulnerability->att('id');

      my $plugin=$vulnerability->att('plugin');
      my $cve=$vulnerability->att('cve');
      my $my_level=$vulnerability->att('level');
      my $title=$vulnerability->att('title');

      my $content=MIME::Base64::decode_base64($vulnerability->text());

      my $class="orange";

      if($my_level==0) {
      $class="green";
      } elsif($my_level==1) {
      $class="yellow";
      } elsif($my_level==2) {
      $class="lightred";
      }

      print "<TR CLASS=\"$class\"><TD><A HREF=\"/View/$file/$id\">[*]</TD><TD>&nbsp;</TD><TD NOWRAP=\"true\">$my_time</TD><TD>&nbsp;</TD><TD>$title</TD><TD>&nbsp;</TD><TD>$bid</TD><TD>&nbsp;</TD><TD>$cve</TD><TD>&nbsp;</TD><TD>$plugin</TD></TR>";
    }
  } else {
    print "<TR CLASS=\"yellow\"><TD COLSPAN=\"11\">No vulnerabilities found.</TD></TR>";
  }
}

print "</TABLE></TD></TR>";

############

print "</TABLE>";

print "</BODY></HTML>";

$twig->purge;

return undef;

}

sub QuerySplit {
  my $query=shift;
  my (%ret, $pair, $key, $val);

  return if (! defined $query);

  foreach $key (split('&',$query)) {
   if($key=~/^([^=]+)\=(.*)$/) {
    $key=$1;
    $val=$2;
    $ret{$key}=$val;
    }
   }

  return %ret;
}

sub HTMLPrintStyle { 
'
<STYLE TYPE="text/css">
BODY {font-family: arial,sans-serif; background-color: white; font-size: 8pt}
TD.h {font-family: arial,sans-serif; font-weight: bold}
TD.d {font-family: arial,sans-serif; color: #990000}
TD.mini {font-size: 55%; font-weight: bold;}
TD.mini2 {font-size: 75%; font-weight: bold; color: #ffffff;}
TD.h1 {color: #ffffff; font-weight: bold}
TD.h2 {color: #444444}
TR.h2 {background-color: #eeeeee}
TR.h22 {background-color: #bbbbbb}
TR.h1 {font-weight: bold; background-color: #7777aa;}
TR.green {font-weight: bold; background-color: #22dd22;}
TR.lightred {font-weight: bold; background-color: #dd6666;}
TR.red {font-weight: bold; background-color: #dd2222;}
TR.orange {font-weight: bold; background-color: #ffcc00;}
TR.yellow {font-weight: bold; background-color: #ffff00;}
TR.lightgray {font-weight: bold; background-color: #cccccc;}
TR.white {font-weight: bold; background-color: #ffffff;}
TD.header {background-color: #000088; color: #ffffff; font-weight: bold}
TD.section {background-color: #ffffff; color: #222222; font-weight: bold}
TABLE.request {background-color: #994444;}
red2 {color: #994444}
</STYLE>';
}

sub Record {
my ($plugin, $event, $timestamp, $client_addr, $client_port, $request, $response)=@_;
my $request_uri=$request->uri();
my $new_file=0;
my ($proto, $host, $port, $uri);

$0="httpush [recording]";

if(! $HTTPUSH_OPTIONS{'xml'}) {
  return;
}

 # Get the values we want to positionate in the XML tree

 if($request_uri)
   {
   ($proto, $host, $port, $uri)=$request_uri=~/^([^:]+)\:\/\/([^:\/]+):?(\d+)?(\/.*)$/;
   }

 # Find our data file

$0="httpush [recording - retrieving datafile]";

 my ($file, $file_id)=StoredRetrieveFilename($host,1);

 if(! $file_id) {
   $file_id=lib::get_unique_id();
   $file="$file_id.xml";
   $new_file=1;

   # Update the index file, so that if a request comes for our new host before we have finished processing it, it ends up in the same file
   open(INDEX,">> $DATADIR/.index");
   flock INDEX, 2;

   open(PFILE,"> $DATADIR/$file");
   flock PFILE, 2;

   print INDEX "$file_id \"\" $host\n";
   flock INDEX, 8;
   close(INDEX);
 }

 if(! $port) {
   if($proto eq "https") {
     $port=443;
   } else {
     $port=80;
   }
 }

 $0="httpush [recording - parsing datafile]";

  my $twig= new XML::Twig(KeepSpaces => 1);

  if((! $new_file) && $file && -f "$DATADIR/$file") {

    open(PFILE,"< $DATADIR/$file");
    lib::debug("Waiting for lock on $file....");
    flock PFILE, 2;
    lib::debug("Got lock....");

    $twig->safe_parse(*PFILE) || do { flock PFILE, 8; close(PFILE); &my_error(7,"4 $!"); };
#    flock PFILE, 8;
#    close(PFILE);

    $new_file=0;
   } elsif ($new_file) {
  # Create a new skel XML tree

  $twig->parse("<?xml version=\"1.0\" ?><!DOCTYPE httpush SYSTEM \"httpush.dtd\"><httpush id=\"".lib::get_unique_id()."\"><general><store></store></general><server fileid=\"".$file_id."\" id=\"".lib::get_unique_id()."\"></server></httpush>");

#  PluginRun($file_id, $twig->root->first_child("server"), "server", $request_uri);
  }
  else
  {
  &my_error(7,"5");
  }

 my $erequest = XML::Twig::Elt->new('request', { timestamp => "$timestamp"});
 $erequest->set_att('id',lib::get_unique_id());

 my $ewho = XML::Twig::Elt->new('who');
 my $erawsent = XML::Twig::Elt->new('sent');
 my $erawreceived = XML::Twig::Elt->new('received');

 $ewho->paste("last_child", $erequest);
 $erawsent->paste("last_child", $erequest);
 $erawreceived->paste("last_child", $erequest);

 $ewho->set_att('address', "$client_addr");
 $ewho->set_att('port', "$client_port");
 $ewho->set_att('id',lib::get_unique_id());

 $erawsent->set_text(MIME::Base64::encode_base64($request->as_string));
 $erawsent->set_att('id',lib::get_unique_id());

 $erawreceived->set_text(MIME::Base64::encode_base64($response->as_string));
 $erawreceived->set_att('id',lib::get_unique_id());

 $0="httpush [recording - inserting data]";

 ParserRequestInsert($plugin, $twig, $erequest, $proto, $host, $port, $uri, $file_id, $request_uri);

 # Run any plugins for this stage before we save to file...

 $0="httpush [recording - running plugins]";

 $plugin->run($file_id, $erawsent, "sent", $request_uri, $erawsent->text);
 $plugin->run($file_id, $erawreceived, "received", $request_uri, $erawreceived->text); 
 $plugin->run($file_id, $erequest, "request", $request_uri, $ewho->att('address'),$ewho->att('port'), $erawsent->text, $erawreceived->text);

 my $raw=$twig->sprint;

 lib::debug("Updating $file (saving request ".$erequest->att('id').")...");

 open(OUT,"> $DATADIR/$file") || die("open: $@");
 print OUT "$raw";
 close(OUT);

# rename("$DATADIR/$file.tmp.$$","$DATADIR/$file");

 lib::debug("Releasing lock");

 flock PFILE, 8;
 close(PFILE);

# open(OUT,"> $DATADIR/$file.tmp.$$") || die("open: $@");
# flock OUT, 2;

# if($new_file)
#   {
#   &StoredReindex();
#   }

$twig->purge;
}

sub StoredRetrieveFilename {
 my ($tmp, $method)=@_;
 my ($file, $id, $fqdn, $done, $f, $line);

 $done=0;

 open(INDEX,"< $DATADIR/.index") || die("open");
 flock INDEX, 1;

 while((! $done) && ($line=<INDEX>)) {
  if($line=~/^(\d+)\s+\"([^\"]*)\"\s+(.*)$/) {
   $id=$1;
   $file=$id.".xml";
   $fqdn=$3;

   if($method==0  || (! $method))
     {
     if($id == $tmp)
       {
       $done=1;
       last;
       }
     }

   if($method==1)
     {
     foreach $f (split('\s',$fqdn))
       {
       if($f eq $tmp)
         {
         $done=1;
         last;
         }
       }
     }
   }
  }
flock INDEX, 8;
close(INDEX);

if(! $done)
  {
  return(undef, undef);
  }

return($file, $id);
}

sub ParserRequestInsert {
my ($plugin, $twig, $erequest, $proto, $host, $port, $uri, $file_id, $request_uri) =@_;
my ($s, $p, $done, $path,$page, $p_ok, $efqdn, $pg, $pg_ok, $edir, $d);

$s=$twig->root->first_child('server');

my ($my_uri,$args)=split("\\?",$uri,2);

if($my_uri=~/^(.*\/)([^\/]*)$/)
  {
  $path=$1;
  $page=$2;
  }
else
  {
  $path=$my_uri;
  $page=undef;
  }

$path=xml_escape($path);
$page=xml_escape($page);

$done=0;
my $tmp;

foreach $tmp ($s->children('fqdn')) {
  if($tmp->att('name') eq "$host") {
    $done=1;
  }
}

if(! $done) {
  $efqdn = XML::Twig::Elt->new('fqdn');

  $efqdn->paste(last_child, $s);
  $efqdn->set_att('name',$host);
  $efqdn->set_att('id', lib::get_unique_id());

  $plugin->run($file_id, $efqdn, "fqdn", $request_uri);
}

$done=0;

foreach $p ($s->children('port'))
  {
  if($p->att('number') eq $port)
    {
    $p_ok=$p;
    $done=1;
    last;
    }
  }

if(! $done)
  {
  $p_ok= XML::Twig::Elt->new('port',{number => $port});
  $p_ok->paste(last_child, $s);
  $p_ok->set_att('id',lib::get_unique_id());

  $plugin->run($file_id, $p_ok, "port", $request_uri);
  }

$done=0;
my $t;

foreach $t ($p_ok->children('protocol'))
  {
  if($t->att('name') eq $proto)
    {
    $done=1;
    last;
    }
  }

if(! $done)
  {
  my $eproto = XML::Twig::Elt->new('protocol');
  $eproto->paste(last_child, $p_ok);
  $eproto->set_att('name', $proto);
  $eproto->set_att('id',lib::get_unique_id());

  $plugin->run($file_id, $eproto, "protocol", $request_uri);
  }

$d=$p_ok->first_child('directory');

if(! $d)
  {
  $d=XML::Twig::Elt->new('directory',{path => "/"});
  $d->paste(last_child,$p_ok);
  $d->set_att('id',lib::get_unique_id());

  $plugin->run($file_id, $d, "directory", $request_uri);
  }

$edir = ParserDirectoryFind($plugin, $d, $path, $file_id, $request_uri);

if($page)
   {
  $done=0;

  foreach $pg ($edir->children('page'))
    {
    if($pg->att('name') eq $page)
      {
      $pg_ok=$pg;
      $done=1;
      last;
      }
    }

  if(! $done)
    {
    $pg_ok = XML::Twig::Elt->new('page', {name => "$page"});
    $pg_ok->paste(last_child, $edir);
    $pg_ok->set_att('id',lib::get_unique_id());

    $plugin->run($file_id, $pg_ok, "page", $request_uri);
    }

   $erequest->paste(last_child,$pg_ok);
  }
else
  {
   $erequest->paste(last_child,$edir);
  }
}

sub ParserDirectoryFind {
  my ($plugin, $edirectory, $path, $file_id, $request_uri)=@_;
  my ($d, @dirs, $current, $done);

  @dirs=split('/',$path);

  $done=0;

  if(@dirs) {
    $current=shift(@dirs)."/";

    lib::debug("Trying directory $current ($path)");

    foreach $d ($edirectory, $edirectory->children('directory')) {
      if($d->att('path') eq "$current") {
	      $edirectory=$d;
        $done=$d;
        last;
      }
    }

    $d=$done;

    if(! $done) {
      lib::debug("Creating new node for path \"$current\"");
      $d=XML::Twig::Elt->new('directory',{path => "$current"});
      $d->paste(last_child,$edirectory);
      $d->set_att('id',lib::get_unique_id());

      $plugin->run($file_id, $d, "directory", $request_uri);
    }

    return(ParserDirectoryFind($plugin, $d,join('/',@dirs), $file_id, $request_uri));
  }

  return($edirectory);
}

sub ParserSectionFind {
  my ($section, $path, $create) = @_;

  my ($d, @sections, $current, $done);

  @sections=split('/',$path);

  $done=0;

  if(@sections) {
    $current=shift(@sections)."/";

    lib::debug("Trying section $current ($path)");

    foreach $d ($section, $section->children('section')) {
      if($d->att('name') eq "$current") {
	      $section=$d;
        $done=$d;
        last;
      }
    }

    $d=$done;

    if(! $done) {
      if(! $create) {
        return(undef);
      }

      lib::debug("Creating new node for section \"$current\"");
      $d=XML::Twig::Elt->new('section',{name => "$current"});
      $d->paste(last_child,$section);

#      $d->set_att('id',lib::get_unique_id());
#      $plugin->run($file_id, $d, "directory", $request_uri);
    }

    return(ParserSectionFind($d,join('/',@sections), $create));
  }

  return($section);
}

sub StoredReindex {
  my $file="";
  my @files;
  my $counter=1;

  open(INDEX,"> $DATADIR/.index.tmp.$$");
  flock INDEX, 2;

  my $twig= new XML::Twig(KeepSpaces => 1, TwigHandlers => { 'server' => \&StoredReindexDo});

  opendir(DIR, "$DATADIR/") || do { flock INDEX, 8; close(INDEX); die "can't open directory data: $!"; };
  @files = grep { !/(^\.|~$)/ && /\.xml$/ && -f "$DATADIR/$_" } readdir(DIR);
  closedir DIR;

  lib::debug("Reindexing in progress...");

  foreach $file (@files) {
    lib::debug("Indexing file $file");
    open(PFILE,"< $DATADIR/$file") || next;
    lib::debug("Waiting for reindex shared lock");
    flock PFILE, 1;
    lib::debug("Got reindex shared lock");
    $twig->safe_parse(*PFILE) || do { die("$!"); flock PFILE, 8; close(PFILE); flock INDEX, 8; close(INDEX); next };
    lib::debug("Releasing reindex shared lock");
    flock PFILE, 8;
    close(PFILE);
    $twig->purge();
  }

  flock INDEX, 8;
  close(INDEX);

  if (! -e "$DATADIR/.index.tmp.$$" || -f "$DATADIR/.index.tmp.$$") {
    open(OUT, ">> $DATADIR/.index");
    flock OUT, 2;
    rename ("$DATADIR/.index.tmp.$$","$DATADIR/.index") || do { close (OUT); flock OUT, 8; die("can't create index file"); };
    flock OUT, 8;
    close(OUT);
  } else {
    lib::announce("Something strange with .index.tmp file, refusing to create it");
    exit(1);
  }
lib::debug("End of indexing");
}

sub StoredReindexDo {
  my ($twig,$server)=@_;
  my ($desc, $tmp, $desc2, $ch, $fileid);

  $desc=$server->first_child_text('desc');

  $desc2="";

  foreach $ch ($server->children('fqdn')) {
    if($tmp=$ch->att('name')) {
      $desc2.="$tmp ";
    }
  }

  $fileid=$server->att('fileid');

  print INDEX "$fileid \"$desc\" $desc2\n";

  $twig->purge();
}

sub hprint ($$) {
  my ($ssl, $data) = @_;

  if($CURRENT{'is_ssl'}) {
    Net::SSLeay::write ($CURRENT{'ssl'}, $data); 
  } else {
    print $data;
  }
}

sub xml_escape {
  my $text=shift;

  $text=~s/&/&amp;/g;
  $text=~s/</&lt;/g;
  $text=~s/>/&gt;/g;
  $text=~s/"/&quot;/g;
  $text=~s/'/&apos;/g;

  return($text);
}

sub xml_unescape {
  my $text=shift;

  $text=~s/&amp;/&/g;
  $text=~s/&lt;/</g;
  $text=~s/&gt;/>/g;
  $text=~s/&quot;/"/g;
  $text=~s/&apos;/'/g;

  return($text);
}


sub plugin_encode {
  my $data = shift;
  $data=MIME::Base64::encode_base64($data);
  $data=~s/[\n\r]//g;
  return($data);
}

sub RecordAllowed {
  my $uri=shift;

  return 1 if(! defined $HTTPUSH_OPTIONS{'allowed'});

  if($uri=~/$HTTPUSH_OPTIONS{'allowed'}/) {
    return 1;
  } else {
    return 0;
  }
}
