package lib::plugin::broker;

require 'lib.pm';
require lib::plugin::buffer;
require lib::plugin::node;
require lib::plugin::message;

use IPC::Open2;
use strict;
no strict 'subs';

my $DATADIR="data"; # XXXXXXXXXXXXXX TEMPORAL ###############

# Constructors

sub new {
  my($class, %init) = @_;

  my $self;

  $self = bless {
    'count' => 0,
    'master' => new lib::plugin::node,
    'list' => [],
    'noise' => 0,
    'intrusive' => 0,
    'plugin_obj' => undef
  }, $class;

  foreach (keys %init) {
    if(exists $self->{$_}) {
      $self->{$_}=$init{$_};
    }
  }

  return $self;
}

sub start {
  my ($self) = @_;

  my $plugin=$self->{'plugin_obj'};

  lib::debug("Starting broker (".$plugin->{'basedir'}.")");

#  pipe (BROKER_READ, BROKER_WRITE);
#  socketpair (BROKER_READ, BROKER_WRITE, AF_UNIX, SOCK_STREAM, PF_UNSPEC) || die "socketpair: $!";
  socketpair (HTTPUSH, BROKER, 1, 1, 0) || die "socketpair: $!";

#  $self->{'master'}->{'write'} = new lib::plugin::buffer(( 'fd' => *BROKER_WRITE));

  $self->plugin_load();

  if(! fork()) {
    close HTTPUSH;
    $self->{'master'}->{'io'} = new lib::plugin::buffer(('fd' => *BROKER));
    $self->run($plugin);
    exit;
    }

  close BROKER;
  $self->{'master'}->{'io'} = new lib::plugin::buffer(('fd' => *HTTPUSH));
}

sub register {
  my($self, $filename) = @_;
  my($description, $plugin_count);

  my ($child, $parent);

#  my $pid = open2($rfh, $wfh, "$filename") || next;

#  socketpair($rfh, $wfh, AF_UNIX, SOCK_STREAM, PF_UNSPEC) || die "socketpair: $!";
  socketpair ($child, $parent, 1, 1, 0) || die "socketpair: $!";

  my $child_fd=fileno($child);
  my $parent_fd=fileno($parent);

  if(! fork()) {

    open(STDIN,  "<&=$parent_fd") || die "can't dup client to stdin: $!";
    open(STDOUT, ">&=$parent_fd") || die "can't dup client to stdout: $!";

    exec $filename || die("Couldn't execute plugin \"$filename\": $!");
  }

  close $parent;
  my $node = new lib::plugin::node('io' => new lib::plugin::buffer('fd' => $child), 'exec' => $filename);

  lib::debug("Identifying plugin \"$filename\"...");

#  print $wfh "identify $HTTPUSH_OPTIONS{'plugin_version'}\n\n";
#  $self->{'list'}{$plugin_count}{'exec'}=$filename;

  my $message=new lib::plugin::message(('method' => 'identify', 'version' => lib::plugin::Version));

  if($node->send_message($message)) {
    lib::debug("finished send message");

    my $message=undef;

    while(! defined $message) {
      $message=$node->recv_message(('method' => 'register'));
    }

    lib::debug("RECEIVED MESSAGE $message, method <$message->{'method'}>");

    if(defined $message) {
      my $item;
      foreach $item (@{$message->{'headers'}}) {
        my ($key, $val)=@{$item};

        if($key && $val) {

          if($key eq "name") {
            $node->{name}=$val;
          } elsif ($key eq "description") {
            $node->{description}=$val;
          } elsif ($key eq "version") {
            $node->{version}=$val;
          } elsif ($key eq "event") {
            lib::debug("Registering for event <$val>");
            $node->{events}{$val}=1;
          } elsif ($key eq "noise") {
            $node->{noise}=$val;
          } elsif ($key eq "intrusive") {
            $node->{intrusive}=$val;
          }
        }
      }

      $self->insert($node);

      lib::debug("Loaded plugin $plugin_count => $filename ($node->{description}) - NL: $node->{noise} INT: $node->{intrusive}...");
    }
  }

}

sub run {
  my ($self, $plugin) = @_;

  my ($found, $r_fds, $plugin_count, $message);


  while(1) {
    $0="httpush [broker]";

    # Check if the main program asked for something
    # Check if the plugins reported something

#    lib::debug("getting messages from master");

    if($message=$self->{'master'}->recv_message) {
      lib::debug("got message from master ($message->{'method'})");
      $self->process_master($message)
    }

    my $i=0;

#    lib::debug("getting messages from slaves");

    while($i < $self->{'count'}) {
      my $node=@{$self->{'list'}}[$i];
      if(defined $node) {

#      lib::debug("checking slave $i");

        if($message=$node->recv_message) {
          lib::debug("got message from slave $i [name $node->{'name'}]");
          $self->process_slave($node, $message);
        }
      } else {
        lib::debug("node $i not defined");
      }
      $i++;
    }
  }
}

sub plugin_load {
  my ($self) = @_;
  my (@plugin_files, $plugin_file);

  my $plugin=$self->{'plugin_obj'};

  lib::debug("loading plugins from ".$plugin->{'basedir'});

  opendir (PDIR, $plugin->{'basedir'});
  @plugin_files = grep { !/^\./ && !/~$/ && !/^CVS$/ && -x $plugin->{'basedir'}."$_" } readdir(PDIR);
  closedir(PDIR);

  $|=1;

  foreach $plugin_file (@plugin_files) {
    $self->register($plugin->{'basedir'}.$plugin_file);
  }

}

sub insert {
  my ($self, $node) = @_;
  push (@{$self->{'list'}}, $node);
  $self->{'count'}++;
  lib::debug("plugin count: ".$self->{'count'}." == ".@{$self->{'list'}});
  return 1;
}

return 1;

sub process_master {
  my ($self, $message) = @_;
  my ($line, $event);

  $event = $message->{'method'};

  lib::debug("starting pluginmaster for event $event, will copy data to slaves");

  $0="httpush [broker - master]";

  my $node;

  lib::debug("=> Testing against ".@{$self->{'list'}}." plugins");

  foreach $node (@{$self->{'list'}}) {

    lib::debug("Should run for \"$event\"? $node->{'events'}{$event} - $node->{'noise'} / $self->{'noise'} - $node->{'intrusive'} / $self->{'intrusive'}");

    if((! $node->{'events'}{$event})
      || ($node->{'noise'} > $self->{'noise'})
      || ($node->{'intrusive'} > $self->{'intrusive'})) {
     lib::debug("Declining to run plugin [$node->{'name'}]");

     next;
    }

    lib::debug("Running $node->{name} for event $event");

    $node->send_message($message);
  }
}

sub process_slave {
  my ($self, $node, $message)=@_;

  my ($element_id,$twig, $tmp_req, $x, $item, $key, $val);

  $0="httpush [broker - slave]";
  lib::debug("starting plugin_slave, will get messages from slaves and process them [$message->{'method'}]");

  return if (!defined $message);

  my $m=$message->{'method'};

  if($m eq "req") {
    my $http_req=new HTTP::Request;

    my ($id, $record, @h);
    $record=0;

    foreach $item (@{$message->{'headers'}}) {
      my ($key, $val)=@{$item};

      if($key eq "content") {
        $http_req->content("$val");
      } elsif ($key eq "id") {
        $id=$val;
      } elsif ($key eq "record") {
        if($val) {
          $record=2;
        }
      } elsif ($key eq "method") {
        $http_req->method($val);
      } elsif ($key eq "uri") {
        $http_req->uri($val);
      } elsif ($key eq "protocol") {
        $http_req->protocol($val);
      } elsif ($key eq "header") {
        @h=split(":", $val,2);
        $http_req->header($h[0], $h[1]);
      }
    }

    lib::debug("Running REQ on ".$http_req->uri);

    my $response=::proxy_process($self->{'plugin_obj'}, undef, $http_req, $record, undef, undef);

    if((defined $response) && ($record==0)) {
      lib::debug("Sending response to plugin");

      my $res_message=new lib::plugin::message(('method' => 'res'));
      $res_message->add_header("content",$response->as_string);

      if($id) {
        $res_message->add_header("id",$id);
      }

      $node->send_message($res_message);
    } else {
      lib::debug("No response to REQ or record set");
    }
  } elsif($message->{'method'}  eq "vuln") { # Plugin informs of a vulnerability
    my ($content, $my_level, $title, $bid, $cve, $element_id);

    foreach $item (@{$message->{'headers'}}) {
      my ($key, $val)=@{$item};

      if($key eq "content") {
        $content=$val;
      } elsif ($key eq "level") {
        $my_level=$val;
      } elsif ($key eq "title") {
        $title=$val;
      } elsif ($key eq "bid") {
        $bid=$val;
      } elsif ($key eq "cve") {
        $cve=$val;
      } elsif ($key eq "id") {
        $element_id=$val;
      }
    }

    # Get XML element

    lib::debug("Adding VULN for element_id $element_id");

    if(defined($element_id)) {
      my ($file, $id)=split('/',$element_id, 2);
      $file=~y/0-9/0-9/cd;
      $id=~y/0-9/0-9/cd;

      if(defined($file) && defined($id)) {
        if($file && -f "$DATADIR/$file.xml") {
          $twig= new XML::Twig(KeepSpaces => 1);
          open(PFILE,"< $DATADIR/$file.xml");
          lib::debug("Waiting for shared lock on $file.xml");
          flock PFILE, 1;
          lib::debug("Got shared lock on $file.xml");
          $twig->safe_parse(*PFILE) || do { flock PFILE, 8; &my_error(7,"1: $!"); };

				lib::debug("Releasing shared lock on $file.xml");
          flock PFILE, 8;
          close(PFILE);
        } else {
          &my_error(4);
        }

        my $el=$twig->elt_id($id);

        if(defined($el)) {
          # Insert vulnerability

          lib::debug("Inserting vulnerability...");
          lib::announce("Plugin \"$node->{'name'}\" found vulnerability \"$title\"");

          my $vuln = XML::Twig::Elt->new('vuln');

          if(! $cve) {
           $cve="-";
          }

          if(! $bid) {
           $bid="-";
          }

          $vuln->paste(last_child, $el);
          $vuln->set_att('id',lib::get_unique_id());
          $vuln->set_att('level',$my_level);
          $vuln->set_att('cve',$cve);
          $vuln->set_att('bid',$bid);
          $vuln->set_att('plugin',$node->{'name'});
          $vuln->set_att('title',$title);
          $vuln->set_att('timestamp',time());
          $vuln->set_text(MIME::Base64::encode_base64($content));

          my $raw=$twig->sprint;

          lib::debug("Updating $file.xml...");
          open(LFILE, "> $DATADIR/$file.xml.lck");
          flock LFILE, 2;

          open(OUT,"> $DATADIR/$file.tmp.$$") || do { unlink("$DATADIR/$file.xml.lck"); flock LFILE, 8; close(LFILE);  die("open: $@"); };
          flock OUT, 2;
          print OUT "$raw";
          flock OUT, 8;
          close(OUT);

          rename("$DATADIR/$file.tmp.$$","$DATADIR/$file.xml");
          unlink("$DATADIR/$file.xml.lck");
          flock LFILE, 8;
          close(LFILE);
        } else {
          lib::debug("Identification not found...");
        }
        $twig->purge;
      }
    }
  } elsif($message->{'method'}  eq "store") { # Plugin informs of data we want to store
    my ($content, $section, $element_id, $my_content);

    foreach $item (@{$message->{'headers'}}) {
      my ($key, $val)=@{$item};

      if($key eq "content") {
        $content=$val;
      } elsif ($key eq "section") {
        $section=$val;
      } elsif ($key eq "id") {
        $element_id=$val;
      }
    }

    # Get XML element

    lib::debug("Adding DATA for element_id $element_id");

    if(defined($element_id)) {
      my ($file, $id)=split('/',$element_id, 2);
      $file=~y/0-9/0-9/cd;
      $id=~y/0-9/0-9/cd;

      if(defined($file) && defined($id)) {
        if($file && -f "$DATADIR/$file.xml") {
          $twig= new XML::Twig(KeepSpaces => 1);
          open(PFILE,"< $DATADIR/$file.xml");
          lib::debug("Waiting for shared lock on $file.xml");
          flock PFILE, 1;
          lib::debug("Got shared lock on $file.xml");
          $twig->safe_parse(*PFILE) || do { flock PFILE, 8; &my_error(7,"1: $!"); };

				lib::debug("Releasing shared lock on $file.xml");
          flock PFILE, 8;
          close(PFILE);
        } else {
          &my_error(4);
        }

        # Get the current data in the section (create subsections as necessary)

        my $store=$twig->first_elt('store');

        my $el=::ParserSectionFind($store, $section, 1);

        if(defined $el) {
          $my_content=MIME::Base64::decode_base64($el->text);

          my @my_content_data = split(/\n/,$my_content);
          push(@my_content_data, split(/\n/, $content));

          my @in;
          my $prev = "not equal to $in[0]";
          @my_content_data = grep($_ ne "" && $_ ne $prev && ($prev = $_, 1), @my_content_data);

          $el->set_text(MIME::Base64::encode_base64(join("\n",@my_content_data)));

          lib::debug("Inserting storage data... (".join("\n",@my_content_data).")");

          my $raw=$twig->sprint;

          lib::debug("Updating $file.xml...");

          open(LFILE, ">> $DATADIR/$file.xml");
          flock LFILE, 2;

          open(OUT,"> $DATADIR/$file.tmp.$$") || do { flock LFILE, 8; close(LFILE); die("open: $@"); };
          flock OUT, 2;
          print OUT "$raw";
          flock OUT, 8;
          close(OUT);

          rename("$DATADIR/$file.tmp.$$","$DATADIR/$file.xml");
          flock LFILE, 8;
          close(LFILE);
        }
        $twig->purge;
      }
    }
  } elsif($message->{'method'}  eq "retrieve") { # Plugin wants to retrieve data from the store
    my ($section, $id);

    foreach $item (@{$message->{'headers'}}) {
      my ($key, $val)=@{$item};

      if ($key eq "section") {
        $section=$val;
      } elsif ($key eq "id") {
        $element_id=$val;
      }
    }

    # Get XML element

    lib::debug("Retrieving DATA from \"$section\" $element_id");

    if(defined($section) && defined($element_id)) {
      my ($file, $id)=split('/',$element_id, 2);
      $file=~y/0-9/0-9/cd;
      $id=~y/0-9/0-9/cd;

      if(defined($file) && defined($id)) {
        if($file && -f "$DATADIR/$file.xml") {
          $twig= new XML::Twig(KeepSpaces => 1);
          open(PFILE,"< $DATADIR/$file.xml");
          lib::debug("Waiting for shared lock on $file.xml");
          flock PFILE, 1;
          lib::debug("Got shared lock on $file.xml");
          $twig->safe_parse(*PFILE) || do { flock PFILE, 8; &my_error(7,"1: $!"); };

				lib::debug("Releasing shared lock on $file.xml");
          flock PFILE, 8;
          close(PFILE);
        } else {
          &my_error(4);
        }

        # Get the current data in the section (create subsections as necessary)

        my $store=$twig->first_elt('store');

        my $el=::ParserSectionFind($store, $section, 0);

        my $content="";

        if(defined $el) {
          $content=MIME::Base64::decode_base64($el->text);
        }

        my $res_message=new lib::plugin::message(('method' => 'data'));

        $res_message->add_header("id",$element_id);
        $res_message->add_header("section",$section);
        $res_message->add_header("content",$content);

        $node->send_message($res_message);

        $twig->purge;
      }
    }
  }

}

return 1;
