#!/usr/bin/env python
# Copyright (C) 2005-2006 Sivan Greenberg. <sivan@ubuntu.com>
# 
# This software is distributed under the terms and conditions of the GNU General
# Public License. See http://www.gnu.org/copyleft/gpl.html for details.
import os
import sys
import pwd
import re
import ConffileParser


DEBUG_PRINT = True


# change this for other distro's then Debian / Ubuntu
ADDUSER_CONF = '/etc/adduser.conf'

def getSizeSpecifier(size):
    mysize = float(size)
    new_size, specifier =  (mysize, 'bytes')
    if mysize > 1024:
        new_size, specifier =  (round(mysize/1024,2), 'KB')
    if mysize > 1048576:
        new_size, specifier =  (round(mysize/1048576,2), 'MB')
    if mysize > 1073741824:
        new_size, specifier =  (round(mysize/1073741824,2), 'GB')
    return (int(new_size), specifier)


class UserInfo:
    def __init__(self):
        self.myuid = os.getuid()
        self.pwd_tuple = pwd.getpwuid(self.myuid)

    def username(self):
        return self.pwd_tuple[0]
    def uid(self):
        return self.pwd_tuple[2]
    def gid(self):
        return self.pwd_tuple[3]
    def comment(self):
        return self.pwd_tuple[4]
    def homedir(self):
        return self.pwd_tuple[5]
    def shell(self):
        return self.pwd_tuple[6]


class DirInfo(list):
    """ 
    Recusively calculates directory size 
    usage:
        a = DirInfo("/home/yourhomedir")
        a.calculate()
        # a now is a list which it's first member is the path, 
        # the second is the size, the third is the most suitable 
        # size representation it has been calculated in (largest) , and the forth
        # is the original size in bytes
        
    result:

        print a
        ['/home/yourhomedir', 1.0, 'MB', 1048576 ]
        
    """
    def __init__(self,dirname):
        self.size = 0
        self.append(dirname)
        self.Done = False
        self.currentFile = ""
        self.currentFileSize = 0

    def calculate(self):
        for root, dirs, files in os.walk(self[0]):
            for f in files:
                path = os.path.join(root, f)
                if os.path.isfile(path):
                    self.currentFile = path
                    self.currentFileSize = os.path.getsize(path)
                    self.size += self.currentFileSize
                    yield True
        new_size, specifier = getSizeSpecifier(self.size)
        self.append(new_size)
        self.append(specifier)
        self.append(self.size)
        self.Done = True
        yield False

class ListFiles(list):
    def __init__(self, regexp, path):
        match = re.compile(regexp,re.IGNORECASE)
        filelist = os.listdir(path)
        if DEBUG_PRINT:
            print "* filelist = %s " % filelist
            print "* match = %s" % match
        tmp = [f for f in filelist if match.match(f)]
        if DEBUG_PRINT:
            print "* tmp = %s" % tmp
        for i in tmp:
            self.append(i)

    def reportCurrentFileSize(self):
        return self.currentFileSize

    def reportCurrentFileName(self):
        return self.currentFile

    def isDone(self):
        return self.Done

class UserHomesInfo(dict):
    """
    Creates a dictionary of all regular users  home directories with 
    the following data:
     1) key => Account username.
     2) value => Tuple (path, size, specifier)

    It does that, by scanning through /etc/passwd for non system
    users and then feeds the home path to DirInfo.

    If the user is not running with a root uid, then thi will just
    have one entry the corrosponds to the executing user's home.

    If executed as root, calculating all user's homes, it will have
    another last key that will point to a tuple of the form:

      1) key => "total" (string)
      2) value => Tuple (bytesize , size, specifier)
    """

    def __init__(self):
        self.Done = False
        self.myCurrentFile = ""
        self.mypid = os.getuid()
        self.pwd_tuple = pwd.getpwuid(self.mypid)
        self.totalSize = 0
        self.AdduserConf = ConffileParser.ParseHashFile(ADDUSER_CONF)
        self.ValidRange = range(int(self.AdduserConf['FIRST_UID']),int(self.AdduserConf['LAST_UID']))
    
    def calculate(self):
        if self.mypid==0:
            pwd_data = pwd.getpwall()
            for t in pwd_data:
                if int(t[2]) in self.ValidRange: 
                    if os.path.isdir(t[5]):
                        tmpInfo = DirInfo(t[5])
                        g = tmpInfo.calculate()
                        while g.next():
                            self.totalSize = self.totalSize + tmpInfo.currentFileSize
                            self.myCurrentFile = tmpInfo.currentFile
                            yield True
                        self[t[0]] = tmpInfo
            tmpInfo = DirInfo(self.pwd_tuple[5])
            g = tmpInfo.calculate()
            while g.next():
                self.totalSize = self.totalSize + tmpInfo.currentFileSize
                self.myCurrentFile = tmpInfo.currentFile
            self[self.pwd_tuple[0]] = tmpInfo
            self["total"] = (self.totalSize, getSizeSpecifier(self.totalSize))
        else:
            if os.path.isdir(self.pwd_tuple[5]):
                tmpInfo = DirInfo(self.pwd_tuple[5])
                g = tmpInfo.calculate()
                while g.next():
                    self.totalSize = self.totalSize + tmpInfo.currentFileSize
                    self.myCurrentFile = tmpInfo.currentFile
                    yield True
                self[self.pwd_tuple[0]] = tmpInfo
                self["total"] = (self.totalSize, getSizeSpecifier(self.totalSize))
        self.Done = True
        yield False

    def reportTotalSize(self):
        return self.totalSize

    def reportCurrentFile(self):
        return self.myCurrentFile

    def isDone(self):
        return Done
       
        
if __name__ == "__main__":
        Homes = UserHomesInfo()
        g = Homes.calculate()
        while g.next():
            sizeTuple = getSizeSpecifier(Homes.reportTotalSize())
            str = Homes.reportCurrentFile()[0:80]
            if len(str) < 80:
                str = str + (" " * (80 - len(str)))
            sys.stdout.write("%s..[%s%s]  \r" % (str, sizeTuple[0], sizeTuple[1]))
            sys.stdout.flush()
        TotalTuple = Homes.pop("total")
#        print "Total size for all home dirs : %s" % TotalTuple[0]
#        print "After getSizeSpecifier : %s%s " % (TotalTuple[1][0],TotalTuple[1][1])
        print "Total Size : [%s%s] " % (TotalTuple[1][0],TotalTuple[1][1])

        for home in Homes: 
            print "Username : %s" % home
            print "Home dir : %s" % Homes[home][0]
            print "Size : %s%s" % (Homes[home][1],Homes[home][2])

