/****************************************************************************
 *
 * Copyright (c) 2001-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#ifndef _ICAL_H_
#define    _ICAL_H

#include <xpl.h>

#if defined(_MSC_VER) || defined(NETWARE) || defined(LIBC)
#pragma pack(push, 8)
#endif /* _MSC_VER */

enum {
    ICAL_RULE_YEARLY,
    ICAL_RULE_MONTHLY,
    ICAL_RULE_WEEKLY,
    ICAL_RULE_DAILY,
    ICAL_RULE_HOURLY,
    ICAL_RULE_MINUTELY,
    ICAL_RULE_SECONDLY
};

enum {
    ICAL_NO_STATE,
    ICAL_VTIMEZONE,
    ICAL_VEVENT,
    ICAL_VJOURNAL,
    ICAL_VTODO,
    ICAL_VALARM
};

enum {
    ICAL_TZ_NO_STATE,
    ICAL_TZ_STANDARD,
    ICAL_TZ_DAYLIGHT,
};

enum {
    ICAL_DTSTART,
    ICAL_DTEND,
    ICAL_DTSTAMP,
    ICAL_DTCOMPLETED,
    ICAL_RECURID
};

enum {
    ICAL_ROLE_CHAIR,
    ICAL_ROLE_REQUIRED_PARTICIPANT,
    ICAL_ROLE_OPTIONAL_PARTICIPANT,
    ICAL_ROLE_NON_PARTICIPANT
};

enum {
    ICAL_CUTYPE_INDIVIDUAL,
    ICAL_CUTYPE_GROUP,
    ICAL_CUTYPE_RESOURCE,
    ICAL_CUTYPE_ROOM,
    ICAL_CUTYPE_UNKNOWN
};

enum {
    ICAL_PARTSTAT_NEEDS_ACTION,
    ICAL_PARTSTAT_ACCEPTED,
    ICAL_PARTSTAT_DECLINED,
    ICAL_PARTSTAT_TENTATIVE,
    ICAL_PARTSTAT_DELEGATED,
    ICAL_PARTSTAT_COMPLETED,
    ICAL_PARTSTAT_IN_PROCESS
};

enum {
    ICAL_FREEBUSY_FREE,
    ICAL_FREEBUSY_BUSY,
    ICAL_FREEBUSY_BUSY_TENTATIVE,
    ICAL_FREEBUSY_BUSY_UNAVAILABLE
};

enum {
    ICAL_METHOD_REQUEST,
    ICAL_METHOD_REPLY,
    ICAL_METHOD_CANCEL
};

enum {
    ICAL_PROD_UNKOWN,
    ICAL_PROD_MODWEB
};

typedef struct _ICalVTimeZone {
    unsigned char *TZName;          /* Exact name of timezone               */

    unsigned long TZState;          /* Parsing state                        */
    unsigned long TimezoneID;       /* Timezone ID                  */

    signed long WDay;               /* Weekday switching from DST           */
    signed long DSTWDay;            /* Weekday switching to DST             */
    signed long Day;                /* Day switching from DST               */
    signed long DSTDay;             /* Day switching to DST                 */
    signed long Month;              /* Month switching from DST             */
    signed long DSTMonth;           /* Month switching to DST               */
    signed long Hour;               /* Hour switching from DST              */
    signed long DSTHour;            /* Hour switching to DST                */
    signed long Offset;             /* Offset from UTC                      */
    signed long DSTOffset;          /* DST Offset from UTC                  */

    struct _ICalVTimeZone *Next;    /* Next parsed timezone                 */
} ICalVTimeZone;

typedef struct _ICalVRule {
    unsigned long Frequency;        /* Frequency of rule                    */
    unsigned long Interval;         /* Frequency interval                   */
    unsigned long Count;            /* Maximum number of occurrences        */
    unsigned long Until;            /* Maximum date rule may go to          */

    unsigned char WeekStart;        /* When does week start                 */

    signed char Days[12][12][31];   /* 12 Years                             */
    signed char Hours[24][60];      /* 24 hours in minute increments        */
    signed char DaysTop;            /* ID to match in Days[]                */
    signed char HoursTop;           /* ID to match in Hours[]               */
} ICalVRule;

typedef struct _ICalVRuleIterator {
    long Day;                       /* Day of this rule iteration           */
    long Month;                     /* Month of this rule iteration         */
    long Year;                      /* Year of this rule iteration          */
    long Hour;                      /* Hour of this rule iteration          */
    long Minute;                    /* Minute of this rule iteration        */
    long Second;                    /* Second of this rule iteration        */

    unsigned long UTC;              /* StartUTC for rule iteration          */

    long Duration;                  /* seconds len of rule iteration        */

    /* This is for the iteration counting: */
    unsigned long Count;            /* Iteration counter                    */
    unsigned long DayIndex;         /* Iteration counter                    */
    unsigned long MonthIndex;       /* Iteration counter                    */
    unsigned long YearIndex;        /* Iteration counter                    */
    unsigned long HourIndex;        /* Iteration counter                    */
    unsigned long MinuteIndex;      /* Iteration counter                    */
} ICalVRuleIterator;

typedef struct _ICalVAttendee {
    unsigned char *Name;            /* CN                                   */
    unsigned char *Address;         /* MailTo                               */
    unsigned char *Delegated;       /* Attendee has delegated to            */
    unsigned char *DelegatedFrom;   /* Event delegated from                 */

    BOOL RSVP;                      /* RSVP Requested?                      */

    unsigned long Type;             /* Chair, required, etc...              */
    unsigned long Role;             /* Individual, resource, etc...         */
    unsigned long State;            /* Accepted, declined, etc...           */
    struct _ICalVAttendee *Next;
} ICalVAttendee;

typedef struct _ICalVOrganizer {
    unsigned char *Name;            /* CN                                   */
    unsigned char *Address;         /* MailTo                               */
} ICalVOrganizer;

typedef struct _ICalVDateTime {
    unsigned long UTC;

    long Day;
    long Month;
    long Year;
    long Hour;
    long Minute;
    long Second;

    unsigned long TimezoneID;       /* ID DateTime timezone                 */
} ICalVDateTime;

typedef struct _ICalVAlarm {
    unsigned long Action;
    unsigned char *Description;
    unsigned long RepeatCount;
    unsigned long Trigger;

    ICalVDateTime TriggerTime;

    struct _ICalVAlarm *Next;
} ICalVAlarm;

typedef struct _ICalVFreeBusy {
    ICalVDateTime Date;

    unsigned long Duration;
    unsigned long Type;

    struct _ICalVFreeBusy *Next;
} ICalVFreeBusy;

typedef struct _ICalObject {
    unsigned long Type;             /* ICAL_Vxxx type                       */
    unsigned long Method;           /* Method, if any                       */
    unsigned long ProdID;           /* Identifies whether it came from modweb or not */

    /* Descriptive component properties */
    unsigned char *Summary;
    unsigned char *Description;
    unsigned char *Location;
    unsigned char *Comment;

    unsigned long PercentComplete;
    unsigned long Priority;

    /* Timezone component properties */
    ICalVTimeZone *Timezone;        /* Pointer to timezone array            */

    /* Date/Time component properties */
    ICalVDateTime Start;            /* Start time description in UTC        */
    ICalVDateTime End;              /* End time description in UTC          */
    ICalVDateTime Stamp;            /* Timestamp in UTC                     */
    ICalVDateTime Completed;        /* Timestamp in UTC                     */

    unsigned long Duration;         /* Duration                             */

    ICalVFreeBusy *FreeBusy;        /* List of free/busy events             */

    BOOL Transparent;               /* Is it visible on calendar            */

    /* Recurrence component properties */
    ICalVRule *Rule;                /* Recurrence Rule                      */

    /* Relationship component properties */
    ICalVOrganizer *Organizer;      /* Duh!                                 */
    ICalVAttendee *Attendee;        /* Linked list of attendees             */

    unsigned char *Contact;
    unsigned char *UID;

    ICalVAlarm *Alarm;

    /* Change management properties */
    unsigned long Sequence;

    ICalVDateTime RecurrenceID;     /* To cancel recurring events           */

    /* For owner use */
    void *UserData;

    unsigned long UserLong;

    struct _ICalObject *Next;
} ICalObject;

#ifdef __cplusplus
extern "C" {
#endif
ICalVRule *ICalNewRule(ICalObject *Parent);

BOOL ICalFreeRule(ICalObject *Parent);
ICalVAttendee *ICalNewAttendee(ICalVAttendee *Previous, unsigned char *Name, unsigned char *Address, unsigned char *Delegated, unsigned char *DelegatedFrom, BOOL RSVP, unsigned long Type, unsigned long Role, unsigned long State);
BOOL ICalFreeAttendees(ICalObject *Parent);
ICalVTimeZone *ICalNewTimezone(ICalObject *Parent);
BOOL ICalFreeTimezone(ICalObject *Parent);
ICalVFreeBusy *ICalNewFreeBusy(ICalObject *Parent);
BOOL ICalFreeFreeBusy(ICalObject *Parent);
ICalObject *ICalNewObject(ICalObject *Parent);
BOOL ICalFreeObjects(ICalObject *Parent);
ICalVOrganizer *ICalNewOrganizer(unsigned char *Name, unsigned char *Address);
BOOL ICalFreeOrganizer(ICalObject *Parent);

/* Field parsers */
unsigned char *ICalGrabArgument(BOOL *IsValue, unsigned char *String, unsigned char *Text);
unsigned char *ICalGrabValue(long *Value, unsigned char *Text);
unsigned char *ICalGrabPeriod(ICalVDateTime *Date, long *Duration, unsigned char *Text);
unsigned char *ICalGrabWDay(long *Offset, unsigned char *WDay, unsigned char *Text);

/* Encoders */
BOOL ICalEncodeArgument(unsigned char *Src, FILE *Dest, unsigned long BreakLen, unsigned long Start);

/* Component parsers */
BOOL ICalParsePercent(ICalObject *ICal, unsigned char *Text);
BOOL ICalParseSequence(ICalObject *ICal, unsigned char *Text);
BOOL ICalParsePriority(ICalObject *ICal, unsigned char *Text);
BOOL ICalParseTransparency(ICalObject *ICal, unsigned char *Text);
BOOL ICalParseFreeBusy(ICalObject *ICal, unsigned char *Text);
BOOL ICalParseDuration(ICalObject *ICal, long *Duration, unsigned char *Text);
BOOL ICalParseOrganizer(ICalObject *ICal, unsigned char *Text);
BOOL ICalParseUID(ICalObject *ICal, unsigned char *Text);
BOOL ICalParseContact(ICalObject *ICal, unsigned char *Text);
BOOL ICalParseAttendee(ICalObject *ICal, unsigned char *Text);
BOOL ICalParseTimezoneRule(ICalVTimeZone *Timezone, unsigned char *Rule, BOOL IsDST);
BOOL ICalParseLocation(ICalObject *ICal, unsigned char *Text);
BOOL ICalParseDescription(ICalObject *ICal, unsigned char *Text);
BOOL ICalParseSummary(ICalObject *ICal, unsigned char *Text);
unsigned long ICalParseRuleDateTime(ICalVRule *Rule, ICalVTimeZone *Timezone, unsigned char *DateTime);
BOOL ICalParseDTComponent(ICalObject *ICal, unsigned char *DateTime, unsigned char Type);

/* Component generators */
BOOL ICalGenerateAttendeeFile(ICalObject *ICal, FILE *Object);


/* RRule Processing */
BOOL ICalProcessYearlyRule(ICalObject *ICal, ICalVRule *Rule, unsigned char *RuleText);
BOOL ICalProcessMonthlyRule(ICalObject *ICal, ICalVRule *Rule, unsigned char *RuleText);
BOOL ICalProcessDailyRule(ICalObject *ICal, ICalVRule *Rule, unsigned char *RuleText);
BOOL ICalProcessWeeklyRule(ICalObject *ICal, ICalVRule *Rule, unsigned char *RuleText);
BOOL ICalProcessHourlyRule(ICalObject *ICal, ICalVRule *Rule, unsigned char *RuleText);
BOOL ICalProcessMinutelyRule(ICalObject *ICal, ICalVRule *Rule, unsigned char *RuleText);
BOOL ICalProcessSecondlyRule(ICalObject *ICal, ICalVRule *Rule, unsigned char *RuleText);
BOOL ICalParseRecurrenceRule(ICalObject *ICal, unsigned char *RuleText);

/* RRule retrieval */
BOOL ICalHasRule(ICalObject *ICal);
BOOL ICalFirstRuleInstance(ICalObject *ICal, ICalVRuleIterator *Iteration);
BOOL ICalNextRuleInstance(ICalObject *ICal, ICalVRuleIterator *Iteration);

/* Main entry */
BOOL ICalGenerateObjectFile(ICalObject *ICal, FILE *Object, unsigned char *Identifier, unsigned char *RRule);
ICalObject *ICalParseObject(FILE *Object, unsigned char *MemoryObject, unsigned long Size);

#ifdef __cplusplus
}
#endif

#if defined(_MSC_VER) || defined(NETWARE) || defined(LIBC)
#pragma pack(pop)
#endif /* _MSC_VER */

#endif /* _ICAL_H_ */
