/****************************************************************************
 *
 * Copyright (c) 1997-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#include <xpl.h>
#include <mdb.h>

#include "calagent.h"

#define PRODUCT_NAME "NetMail Calendar Agent"
#define PRODUCT_DESCRIPTION "Provides automatic processing of calendar events users receive."
#define PRODUCT_VERSION "$Revision: 1.6 $"

static BOOL ReadCalVariable(unsigned int variable, unsigned char *data, size_t *length);
static BOOL WriteCalVariable(unsigned int variable, unsigned char *data, size_t length);

static BOOL CalDMCCommandHelp(unsigned char *arguments, unsigned char **response, BOOL *closeConnection);
static BOOL CalShutdown(unsigned char *arguments, unsigned char **response, BOOL *closeConnection);
static BOOL SendCalStatistics(unsigned char *arguments, unsigned char **response, BOOL *closeConnection);

ManagementCommands CalManagementCommands[] = {
    { DMCMC_HELP, CalDMCCommandHelp }, 
    { DMCMC_SHUTDOWN, CalShutdown },
    { DMCMC_STATS, SendCalStatistics }, 
    { DMCMC_DUMP_MEMORY_USAGE, ManagementMemoryStats }, 
};

ManagementVariables CalManagementVariables[] = {
    { DMCMV_SERVER_THREAD_COUNT, DMCMV_SERVER_THREAD_COUNT_HELP, ReadCalVariable, NULL }, 
    { DMCMV_CONNECTION_COUNT, DMCMV_CONNECTION_COUNT_HELP, ReadCalVariable, NULL }, 
    { DMCMV_NMAP_ADDRESS, DMCMV_NMAP_ADDRESS_HELP, ReadCalVariable, NULL }, 
    { DMCMV_OFFICIAL_NAME, DMCMV_OFFICIAL_NAME_HELP, ReadCalVariable, NULL }, 
    { DMCMV_REVISIONS, DMCMV_REVISIONS_HELP, ReadCalVariable, NULL }, 
    { DMCMV_VERSION, DMCMV_VERSION_HELP, ReadCalVariable, NULL }, 
};

ManagementVariables *
GetCalManagementVariables(void)
{
    return(CalManagementVariables);
}

int 
GetCalManagementVariablesCount(void)
{
    return(sizeof(CalManagementVariables) / sizeof(ManagementVariables));
}

ManagementCommands *
GetCalManagementCommands(void)
{
    return(CalManagementCommands);
}

int 
GetCalManagementCommandsCount(void)
{
    return(sizeof(CalManagementCommands) / sizeof(ManagementCommands));
}

static BOOL 
CalShutdown(unsigned char *arguments, unsigned char **response, BOOL *closeConnection)
{
    XplThreadID id;

    if (response) {
        if (!arguments) {
            if (Cal.nmap.conn) {
                *response = MemStrdup("Shutting down.\r\n");
                if (*response) {
                    id = XplSetThreadGroupID(Cal.id.group);

                    Cal.state = CAL_STATE_UNLOADING;

                    if (Cal.nmap.conn) {
                        ConnClose(Cal.nmap.conn, 1);
                        Cal.nmap.conn = NULL;
                    }

                    if (closeConnection) {
                        *closeConnection = TRUE;
                    }

                    XplSetThreadGroupID(id);
                }
            } else if (Cal.state != CAL_STATE_RUNNING) {
                *response = MemStrdup("Shutdown in progress.\r\n");
            }

            if (*response) {
                return(TRUE);
            }

            return(FALSE);
        }

        *response = MemStrdup("arguments not allowed.\r\n");
        return(TRUE);
    }

    return(FALSE);
}

static BOOL 
CalDMCCommandHelp(unsigned char *arguments, unsigned char **response, BOOL *closeConnection)
{
    BOOL responded = FALSE;

    if (response) {
        if (arguments) {
            switch(toupper(arguments[0])) {
                case 'M': {
                    if (XplStrCaseCmp(arguments, DMCMC_DUMP_MEMORY_USAGE) == 0) {
                        if ((*response = MemStrdup(DMCMC_DUMP_MEMORY_USAGE_HELP)) != NULL) {
                            responded = TRUE;
                        }

                        break;
                    }
                }

                case 'S': {
                    if (XplStrCaseCmp(arguments, DMCMC_SHUTDOWN) == 0) {
                        if ((*response = MemStrdup(DMCMC_SHUTDOWN_HELP)) != NULL) {
                            responded = TRUE;
                        }

                        break;
                    } else if (XplStrCaseCmp(arguments, DMCMC_STATS) == 0) {
                        if ((*response = MemStrdup(DMCMC_STATS_HELP)) != NULL) {
                            responded = TRUE;
                        }

                        break;
                    }
                }

                default: {
                    break;
                }
            }
        } else if ((*response = MemStrdup(DMCMC_HELP_HELP)) != NULL) {
            responded = TRUE;
        }

        if (responded || ((*response = MemStrdup(DMCMC_UNKOWN_COMMAND)) != NULL)) {
            return(TRUE);
        }
    }

    return(FALSE);
}

static BOOL 
SendCalStatistics(unsigned char *arguments, unsigned char **response, BOOL *closeConnection)
{
    MemStatistics poolStats;

    if (!arguments && response) {
        memset(&poolStats, 0, sizeof(MemStatistics));

        *response = MemMalloc(sizeof(PRODUCT_NAME) 
                            + sizeof(PRODUCT_SHORT_NAME) 
                            + 112);

        MemPrivatePoolStatistics(Cal.nmap.pool, &poolStats);

        if (*response) {
            sprintf(*response, "%s (%s: v%d.%d.%d)\r\n%lu:%lu:%lu:%lu:%d:%d\r\n", 
                    PRODUCT_NAME, 
                    PRODUCT_SHORT_NAME, 
                    PRODUCT_MAJOR_VERSION, 
                    PRODUCT_MINOR_VERSION, 
                    PRODUCT_LETTER_VERSION, 
                    poolStats.totalAlloc.count, 
                    poolStats.totalAlloc.size, 
                    poolStats.pitches, 
                    poolStats.strikes, 
                    XplSafeRead(Cal.server.active), 
                    XplSafeRead(Cal.nmap.worker.active));

            return(TRUE);
        }

        if ((*response = MemStrdup("Out of memory.\r\n")) != NULL) {
            return(TRUE);
        }
    } else if ((arguments) && ((*response = MemStrdup("arguments not allowed.\r\n")) != NULL)) {
        return(TRUE);
    }

    return(FALSE);
}

static BOOL 
ReadCalVariable(unsigned int variable, unsigned char *data, size_t *length)
{
    size_t count;
    unsigned char *ptr;

    switch (variable) {
        case 0: {
            if (data && (*length > 12)) {
                sprintf(data, "%010d\r\n", XplSafeRead(Cal.server.active));
            }

            *length = 12;
            break;
        }

        case 1: {
            if (data && (*length > 12)) {
                sprintf(data, "%010d\r\n", XplSafeRead(Cal.nmap.worker.active));
            }

            *length = 12;
            break;
        }

        case 2: {
            count = strlen(Cal.nmap.address) + 2;
            if (data && (*length > count)) {
                sprintf(data, "%s\r\n", Cal.nmap.address);
            }

            *length = count;
            break;
        }

        case 3: {
            count = strlen(Cal.officialName) + 2;
            if (data && (*length > count)) {
                sprintf(data, "%s\r\n", Cal.officialName);
            }

            *length = count;
            break;
        }

        case 4: {
            unsigned char    version[30];

            PVCSRevisionToVersion(PRODUCT_VERSION, version);
            count = strlen(version) + 14;

            if (data && (*length > count)) {
                ptr = data;

                PVCSRevisionToVersion(PRODUCT_VERSION, version);
                ptr += sprintf(ptr, "calagent.c: %s\r\n", version);

                *length = ptr - data;
            } else {
                *length = count;
            }

            break;
        }

        case 5: {
            DMC_REPORT_PRODUCT_VERSION(data, *length);
            break;
        }
    }

    return(TRUE);
}
