/****************************************************************************
 *
 * Copyright (c) 1997-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

/* Compile-time options */

/* Product defines */

#include <config.h>

#define	PRODUCT_SHORT_NAME	"WebAdmin"
#define	PRODUCT_NAME			"WebAdmin"
#define	PRODUCT_DESCRIPTION	"Provides template & MDB-based tree management"
#define	PRODUCT_VERSION		"2.0"

#include <xpl.h>
#include <connio.h>

#include <openssl/ssl.h>
#include <openssl/err.h>
#include <openssl/pem.h>
#include <openssl/x509.h>
#include <openssl/x509v3.h>

#include <mdb.h>
#include <memmgr.h>
#include <hulautil.h>

#include "webadminp.h"

// Minutes * 60 (1 second granularity)
#define	CONNECTION_TIMEOUT	(15*60)

#if defined(DEBUG)
int		WAScreen;
#endif

/* Globals */
WebAdminGlobals WebAdmin;
XplSemaphore	ExitSemaphore;
XplSemaphore	ServerSemaphore;
XplSemaphore	ShutdownSemaphore;
XplSemaphore	CacheSemaphore;
XplSemaphore	SessionDBSemaphore;
XplSemaphore	TSessionDBSemaphore;
BOOL				Unloading								= FALSE;
BOOL				Exiting									= FALSE;
BOOL				*WAExiting								= &Exiting;
int				ServerSocket							= -1;
int				ServerSocketSSL						= -1;
char				Hostname[256];
char				ManagementURL[256]					= "";
int				TGid;
char				NMAP_ADDR[80]							= "127.0.0.1";
BOOL				AllowClientSSL							= TRUE;
BOOL				UseNMAPSSL								= FALSE;
int				WEBADMIN_PORT							= 89;
int				WEBADMIN_PORT_SSL						= 449;
unsigned long	BindIPAddress							= 0;
BOOL				IgnoreConfig							= FALSE;
BOOL				IgnoreRoles								= TRUE;
BOOL				ExitOnLoad								= FALSE;

unsigned char	ConfigDN[MDB_MAX_OBJECT_CHARS+1]	= "";
unsigned char	DefaultContext[MDB_MAX_OBJECT_CHARS+1]	= "";
unsigned char	LocalTree[MDB_MAX_OBJECT_CHARS+1]		= "";

#ifdef NETWARE

#define	WA_CONFIG_INIT

#endif

#if defined(LINUX) || defined(S390RH) || defined(REDHAT)

#define	WA_CONFIG_INIT

#endif

#ifdef SOLARIS

#define	WA_CONFIG_INIT

#endif

#if defined(WIN32)

#define	WA_CONFIG_INIT		{																																					\
											HKEY HKey;																																\
											if (RegOpenKeyEx(HKEY_LOCAL_MACHINE, "Software\\Hula\\WebAdmin", 0, KEY_READ, &HKey)==ERROR_SUCCESS) {	\
												DWORD		BufSize=XPL_MAX_PATH;																									\
												RegQueryValueEx(HKey, "SnapinDir", NULL, NULL, ModuleDir, &BufSize);												\
												RegCloseKey(HKey);																												\
											}																																			\
											if (RegOpenKeyEx(HKEY_LOCAL_MACHINE, "Software\\Hula\\WebAdmin", 0, KEY_READ, &HKey)==ERROR_SUCCESS) {	\
												DWORD		BufSize=XPL_MAX_PATH;																									\
												RegQueryValueEx(HKey, "TemplateDir", NULL, NULL, TemplateDir, &BufSize);										\
												RegCloseKey(HKey);																												\
											}																																			\
										}

#endif

#define	WA_GENERATED_CERT							XPL_DEFAULT_BIN_DIR "/webadmin/cert.int"
unsigned char ServerCertificatePath[XPL_MAX_PATH]   = XPL_DEFAULT_BIN_DIR "/webadmin/osslcert.pem";
unsigned char ServerPKeyPath[XPL_MAX_PATH]			= XPL_DEFAULT_BIN_DIR "/webadmin/osslpriv.pem";
unsigned char ModuleDir[XPL_MAX_PATH+1]				= XPL_DEFAULT_LIB_DIR;
unsigned char TemplateDir[XPL_MAX_PATH+1]			= XPL_DEFAULT_LIB_DIR "/webadmin";
unsigned char WorkDir[XPL_MAX_PATH+1]				= XPL_DEFAULT_LIB_DIR "/webadmin";
unsigned char LogoDir[XPL_MAX_PATH+1]				= XPL_DEFAULT_LIB_DIR "/webadmin/logo";

unsigned char FormLoginRedirectURL[XPL_MAX_PATH+1]= "/";
unsigned char FormLogoutRedirectURL[XPL_MAX_PATH+1]= "/";
BOOL				UseFormLogoutRedirectURL			= FALSE;
unsigned long Configuration							= 0;
unsigned char OfficialDomain[256 + 1] = "";
unsigned char *DefaultTitle							= NULL;
SSL_CTX *SSLContext								= NULL;

X509					*ServerCertificate					= NULL;
EVP_PKEY				*ServerPKey								= NULL;

static X509 *
LoadX509Certificate(unsigned char *FileOrPtr, unsigned long Length, BOOL IsFile)
{
	X509			*Certificate = NULL;

	if (IsFile) {
		FILE				*CertFile;

		CertFile = fopen(FileOrPtr, "rb");
		if (!CertFile) {
			return(NULL);
		}
		Certificate = PEM_read_X509(CertFile, NULL, NULL, NULL);
		fclose(CertFile);
	} else {
		BIO	*Bio;

		Bio = BIO_new_mem_buf(FileOrPtr, Length);
		Certificate = PEM_read_bio_X509(Bio, NULL, NULL, NULL);
		BIO_free(Bio);
	}

	return(Certificate);
}

static EVP_PKEY *
LoadPrivateKey(unsigned char *FileOrPtr, unsigned long Length, BOOL IsFile)
{
	EVP_PKEY		*PrivateKey = NULL;

	if (IsFile) {
		FILE					*PKeyFile;

		PKeyFile = fopen(FileOrPtr, "rb");
		if (!PKeyFile) {
			return(NULL);
		}
		PrivateKey = PEM_read_PrivateKey(PKeyFile, NULL, NULL, NULL);
		fclose(PKeyFile);
	} else {
		BIO	*Bio;

		Bio = BIO_new_mem_buf(FileOrPtr, Length);
		PrivateKey = PEM_read_bio_PrivateKey(Bio, NULL, NULL, NULL);
		BIO_free(Bio);
	}

	return(PrivateKey);
}

#define	SetPtrToValue(Ptr,String)	Ptr=String;while(isspace(*Ptr)) Ptr++;if ((*Ptr=='=') || (*Ptr==':')) Ptr++; while(isspace(*Ptr)) Ptr++;

int ClientRead(ConnectionStruct *Client, const unsigned char *Buf, int Len);
int ClientReadSSL(ConnectionStruct *Client, const unsigned char *Buf, int Len);
int ClientWrite(ConnectionStruct *Client, const unsigned char *Buf, int Len);
int ClientWriteSSL(ConnectionStruct *Client, const unsigned char *Buf, int Len);
int NMAPRead(SessionStruct *Session, const unsigned char *Buf, int Len);
int NMAPReadSSL(SessionStruct *Session, const unsigned char *Buf, int Len);
int NMAPWrite(SessionStruct *Session, const unsigned char *Buf, int Len);
int NMAPWriteSSL(SessionStruct *Session, const unsigned char *Buf, int Len);

SessionFunc SessionFuncTbl[4] = {
	NMAPWrite, NMAPWriteSSL,
	NMAPRead, NMAPReadSSL,
};
ConnectionFunc ConnectionFuncTbl[4] = {
	ClientWrite, ClientWriteSSL,
	ClientRead, ClientReadSSL
};

#undef DEBUG

int
WADirectClientRead(ConnectionStruct *Client, const unsigned char *Buf, int Len)
{
	return(DoClientRead(Client, Buf, Len));
}

int
WADirectClientWrite(ConnectionStruct *Client, const unsigned char *Buf, int Len)
{
	return(DoClientWrite(Client, Buf, Len));
}

int
WADirectNMAPRead(SessionStruct *Session, const unsigned char *Buf, int Len)
{
	return(DoNMAPRead(Session, Buf, Len));
}

int
WADirectNMAPWrite(SessionStruct *Session, const unsigned char *Buf, int Len)
{
	return(DoNMAPWrite(Session, Buf, Len));
}


#define	SocketReadyTimeout(Socket, Timeout, Exiting)					\
	{																					\
		int                  ret;												\
		fd_set               readfds;											\
		struct timeval       timeout;											\
																						\
		FD_ZERO(&readfds);														\
		FD_SET((Socket), &readfds);											\
		timeout.tv_usec = 0;														\
		timeout.tv_sec = (Timeout);											\
		ret = select(FD_SETSIZE, &readfds, NULL, NULL, &timeout);	\
																						\
		if (Exiting) {																\
			return(0);																\
		}																				\
																						\
		if (ret < 1) {																\
			return(-1);																\
		}																				\
	}

int
ClientRead(ConnectionStruct *Client, const unsigned char *Buf, int Len)
{
#ifndef WIN32
    int ret;
    struct pollfd pfd;

    pfd.fd = Client->s;
    pfd.events = POLLIN;
    ret = poll(&pfd, 1, CONNECTION_TIMEOUT * 1000);
    if (ret > 0) {
        if (!(pfd.revents & (POLLERR | POLLHUP | POLLNVAL))) {
            do {
                ret = IPrecv(Client->s, (unsigned char *)Buf, Len, 0);
                if (ret >= 0) {
                    break;
                } else if (errno == EINTR) {
                    continue;
                }

                ret = -1;
                break;
            } while (TRUE);
        } else {
            ret = -1;
        }
    } else {
        ret = -1;
    }

    if (Exiting) {
        return(0);
    }

    if (ret < 1) {
        return(-1);
    }

	return(ret);
#else
	SocketReadyTimeout(Client->s, CONNECTION_TIMEOUT, Exiting);
	return(IPrecv(Client->s, (unsigned char *)Buf,Len, 0));    
#endif
}

int
ClientWrite(ConnectionStruct *Client, const unsigned char *Buf, int Len)
{
	return(IPsend(Client->s, (char *)Buf, Len, 0));
}

int
ClientReadSSL(ConnectionStruct *Client, const unsigned char *Buf, int Len)
{
	int		llen;

	do {
		llen = SSL_read(Client->CSSL, (void *)Buf, Len);

		switch (SSL_get_error(Client->CSSL, llen)) {
			case SSL_ERROR_NONE:
			case SSL_ERROR_WANT_READ:
			case SSL_ERROR_WANT_WRITE: {
				/* These aren't really errors.  Just call again */

				break;
			}

			default:
			case SSL_ERROR_ZERO_RETURN:
			case SSL_ERROR_SYSCALL:
			case SSL_ERROR_SSL: {
				/* We had a real error, so its time to leave */

				llen = -1;
			}
		}
	} while (llen == 0 && Len != 0 && !Exiting);

	if (!Exiting) {
		return(llen);
	}

	return(-1);
}

int
ClientWriteSSL(ConnectionStruct *Client, const unsigned char *Buf, int Len)
{
	int	err;

	err = SSL_write(Client->CSSL, (void*)Buf, Len);
	if (err==-1) {
		return(-1);
	}
	return(Len);
}

int
NMAPRead(SessionStruct *Session, const unsigned char *Buf, int Len)
{

#ifndef WIN32
    int ret;
    struct pollfd pfd;

    pfd.fd = Session->NMAPs;
    pfd.events = POLLIN;
    ret = poll(&pfd, 1, CONNECTION_TIMEOUT * 1000);
    if (ret > 0) {
        if (!(pfd.revents & (POLLERR | POLLHUP | POLLNVAL))) {
            do {
                ret = IPrecv(Session->NMAPs, (unsigned char *)Buf, Len, 0);
                if (ret >= 0) {
                    break;
                } else if (errno == EINTR) {
                    continue;
                }

                ret = -1;
                break;
            } while (TRUE);
        } else {
            ret = -1;
        }
    } else {
        ret = -1;
    }

    if (Exiting) {
        return(0);
    }

    if (ret < 1) {
        return(-1);
    }

	return(ret);
#else
	SocketReadyTimeout(Session->NMAPs, CONNECTION_TIMEOUT, Exiting);
	return(IPrecv(Session->NMAPs, (unsigned char *)Buf, Len, 0));

#endif
}

int
NMAPWrite(SessionStruct *Session, const unsigned char *Buf, int Len)
{
	return(IPsend(Session->NMAPs, (char *)Buf, Len, 0));
}

int
NMAPReadSSL(SessionStruct *Session, const unsigned char *Buf, int Len)
{
	int		llen;

	do {
		llen = SSL_read(Session->NSSL, (void*)Buf, Len);

		switch (SSL_get_error(Session->NSSL, llen)) {
			case SSL_ERROR_NONE:
			case SSL_ERROR_WANT_READ:
			case SSL_ERROR_WANT_WRITE: {
				/* These aren't really errors.  Just call again */

				break;
			}

			default:
			case SSL_ERROR_ZERO_RETURN:
			case SSL_ERROR_SYSCALL:
			case SSL_ERROR_SSL: {
				/* We had a real error, so its time to leave */

				llen = -1;
			}
		}
	} while (llen == 0 && Len != 0 && !Exiting);

	if (!Exiting) {
		return(llen);
	}

	return(-1);
}

int
NMAPWriteSSL(SessionStruct *Session, const unsigned char *Buf, int Len)
{
	int	err;

	err = SSL_write(Session->NSSL, (void*)Buf, Len);
	if (err==-1) {
		return(-1);
	}
	
	return(Len);
}

BOOL
EndClientConnection(ConnectionStruct *Client)
{
	if (Client->NMAPProtected) {
		return(FALSE);
	}

	if (Client) {
		if (Client->State == STATE_ENDING) {
			return(FALSE);
		}
		Client->State=STATE_ENDING;

		if (Client->s!=-1) {
			WAFlushClient(Client);
			IPclose(Client->s);
		}

		if (Client->ClientSSL) {
			if (Client->CSSL) {
			    SSL_shutdown(Client->CSSL);
				SSL_free(Client->CSSL);
				Client->CSSL = NULL;
			}
		}

		if (Client->FormSeparator) {
			free(Client->FormSeparator);
		}


		if (Client->PostData) {
			fclose(Client->PostData);
			if (Client->TSession) {
				DestroyTSession(Client->TSession);
			}
		} else {
			ReleaseTSession(Client->TSession);
		}

		ReleaseSession(Client->Session);
		ReleaseCachedConnection(Client);
	}
	XplWaitOnLocalSemaphore(ExitSemaphore);
	XplExitThread(EXIT_THREAD, 0);
	return(TRUE);
}

BOOL
WAProtectNMAP(ConnectionStruct *Client, BOOL Protect)
{
	Client->NMAPProtected=Protect;
	return(TRUE);
}

BOOL	
WAFlushClient(ConnectionStruct *Client)
{
	int	count;
	int	sent=0;

 	while ((sent<Client->SBufferPtr) && (!Exiting)) {
		count=DoClientWrite(Client, Client->SBuffer+sent, Client->SBufferPtr-sent);
		if ((count<1) || (Exiting)) {
			if (!Client->NMAPProtected) {
				return(EndClientConnection(Client));
			} else {
				count=Client->SBufferPtr-sent;
			}
		}
		sent+=count;
	}
	Client->SBufferPtr=0;
	Client->SBuffer[0]='\0';
	return(TRUE);
}

BOOL
WASendClient(ConnectionStruct *Client, const unsigned char *Data, int Len)
{
	int	sent=0;

	if (!Len)
		return(TRUE);

	if (!Data)
		return(FALSE);

	while((sent<Len) && !Exiting) {
		if (Len-sent+Client->SBufferPtr>=MTU) {
			memcpy(Client->SBuffer+Client->SBufferPtr, Data+sent, (MTU-Client->SBufferPtr));
			sent+=(MTU-Client->SBufferPtr);
			Client->SBufferPtr+=(MTU-Client->SBufferPtr);
			if (!WAFlushClient(Client)) {
				return(FALSE);
			}
		} else {
			memcpy(Client->SBuffer+Client->SBufferPtr, Data+sent, Len-sent);
			Client->SBufferPtr+=Len-sent;
			sent=Len;
		}
	}
	Client->SBuffer[Client->SBufferPtr]='\0';
	return(TRUE);
}

BOOL
WASendClientEscaped(ConnectionStruct *Client, const unsigned char *Data, int Len)
{
	unsigned char		*Begin = (unsigned char *)Data;
	unsigned char		*ptr = (unsigned char *)Data;
	unsigned char		Buffer[255];
	unsigned long		i;

	/*
		Make sure there are no chars that can't be sent as HTML
	*/
	while (ptr && ((ptr - Data) < Len)) {
		switch (*ptr) {
			case '\"':
			case '<':
			case '>':
			case '\'': {
				WASendClient(Client, Begin, ptr - Begin);
				i = snprintf(Buffer, sizeof(Buffer), "&#%2d;", *ptr);
				WASendClient(Client, Buffer, i);

				ptr++;
				Begin = ptr;

				break;
			}
			default: {
				ptr++;
				break;
			}
		}
	}
	if (ptr != Begin) {
		WASendClient(Client, Begin, ptr - Begin);
	}

	return(TRUE);
}

BOOL
WASendNMAPServer(SessionStruct *Session, const unsigned char *Data, int Len)
{
	int	count;
	int	sent=0;

	while (sent<Len) {
// Fix me - handle error gracefully
		count=DoNMAPWrite(Session, Data+sent, Len-sent);
		if ((count<1) || (Exiting)) {
			IPclose(Session->NMAPs);
			Session->NMAPs=-1;
			return(FALSE);
		}
		sent+=count;
	}

	return(TRUE);
}

int
WAGetNMAPAnswer(SessionStruct *Session, unsigned char *Reply, int ReplyLen, BOOL CheckForResult)
{
	BOOL				Ready=FALSE;
	int				count;
	int				Result;
	unsigned char	*ptr;

	if (Session->NMAPs==-1)
		return(-1);

	if ((Session->NBufferPtr>0) && ((ptr=strchr(Session->NBuffer, 0x0a))!=NULL)) {
		*ptr='\0';
		if (ReplyLen>(long)strlen(Session->NBuffer)) {
			strcpy(Reply, Session->NBuffer);
		} else {
			strncpy(Reply, Session->NBuffer, ReplyLen-1);
			Reply[ReplyLen-1]='\0';
		}
		Session->NBufferPtr=strlen(ptr+1);
		memmove(Session->NBuffer, ptr+1, Session->NBufferPtr+1);
		if ((ptr=strrchr(Reply, 0x0d))!=NULL)
			*ptr='\0';
		Ready=TRUE;
	} else {
		while (!Ready) {
// FIX ME - handle error graceful
			if (Exiting) {
				return(-1);
			}
			count=DoNMAPRead(Session, Session->NBuffer+Session->NBufferPtr, BUFSIZE-Session->NBufferPtr);
// FIX ME - handle error graceful
			if ((count<1) || (Exiting)) {
				return(-1);
			}
			Session->NBufferPtr+=count;
			Session->NBuffer[Session->NBufferPtr]='\0';
			if ((ptr=strchr(Session->NBuffer,0x0a))!=NULL) {
				*ptr='\0';
				count = min((ptr - Session->NBuffer)+1, ReplyLen);
				memcpy(Reply, Session->NBuffer, count);

				Session->NBufferPtr=strlen(ptr+1);
				memmove(Session->NBuffer, ptr+1, Session->NBufferPtr+1);
				if ((ptr=strrchr(Reply, 0x0d))!=NULL)
					*ptr='\0';
				Ready=TRUE;
			}
		}
	}

	if (CheckForResult) {
		if (Reply[0]=='6') {
			if (Reply[1]=='0' && Reply[2]=='0' && Reply[3]=='0') {
				//Session->NMAPChanged=TRUE;
				return(WAGetNMAPAnswer(Session, Reply, ReplyLen, CheckForResult));
			}
		}
		if ((Reply[4]!=' ') && (Reply[4]!='-')) {
			ptr=strchr(Reply,' ');
			if (!ptr) {
				return(-1);
			}
		} else {
			ptr=Reply+4;
		}
		*ptr='\0';
		Result=atoi(Reply);
		if (Result==5001) {
			IPclose(Session->NMAPs);
			Session->NMAPs=-1;
		}
		memmove(Reply, ptr+1, strlen(ptr+1)+1);
	} else {
		Result=atoi(Reply);
	}
	return(Result);
}

BOOL
WAReadClientLine(ConnectionStruct *Client)
{
	BOOL				Ready=FALSE;
	unsigned char	*ptr;
	int				count;

	if ((Client->BufferPtr>0) && ((ptr=strchr(Client->Buffer, 0x0a))!=NULL)) {
		*ptr='\0';
		memcpy(Client->Command, Client->Buffer, ptr-Client->Buffer+1);
		Client->BufferPtr-=strlen(Client->Command)+1;
		memmove(Client->Buffer, ptr+1, Client->BufferPtr);
		Client->Buffer[Client->BufferPtr]='\0';
		if ((ptr=strrchr(Client->Command, 0x0d))!=NULL) {
			*ptr='\0';
		}
		Ready=TRUE;
	} else {
		while (!Ready) {
			if (Exiting) {
				return(EndClientConnection(Client));
			}
			if (!Client->PostData) {
				count=DoClientRead(Client, Client->Buffer+Client->BufferPtr, BUFSIZE-Client->BufferPtr);
				if ((count<1) || (Exiting)) {
					return(EndClientConnection(Client));
				}
			} else {
				count=fread(Client->Buffer+Client->BufferPtr, 1, BUFSIZE-Client->BufferPtr, Client->PostData);
				if (feof(Client->PostData) || ferror(Client->PostData) || (Exiting)) {
					return(EndClientConnection(Client));
				}
			}
			if (count>(long)Client->ContentLength) {
				Client->ContentLength=0;
			} else {
				Client->ContentLength-=count;
			}
			Client->BufferPtr+=count;
			Client->Buffer[Client->BufferPtr]='\0';
			if ((ptr=strchr(Client->Buffer,0x0a))!=NULL) {
				*ptr='\0';
				memcpy(Client->Command, Client->Buffer, ptr-Client->Buffer+1);
				Client->BufferPtr-=strlen(Client->Command)+1;
				memmove(Client->Buffer, ptr+1, Client->BufferPtr);
				Client->Buffer[Client->BufferPtr]='\0';
				if ((ptr=strrchr(Client->Command, 0x0d))!=NULL) 
					*ptr='\0';
				Ready=TRUE;
			} else if (Client->BufferPtr>BUFSIZE) {
				return(EndClientConnection(Client));
			}
		}
	}
	return(TRUE);
}

static BOOL
ReadHeader(ConnectionStruct *Client)
{
	BOOL					inHeader;
	unsigned char		*lineStart;
	unsigned char		*lineEnd;
	int					count;
	unsigned long		fragmentLen;

	Client->ContentLength = 0;
	if (Client->FormSeparator) {
		MemFree(Client->FormSeparator);
	}
	Client->FormSeparator = NULL;
	Client->SessionUID = 0;

	inHeader = TRUE;
	lineStart = Client->Buffer;

	do {
		lineEnd = strchr(lineStart, 0x0a);
		if (lineEnd != NULL && (lineEnd > lineStart)) {
			;
		} else {
			/* we don't have a full line in the buffer */
			if (lineStart != Client->Buffer) {
				/* there is data in the buffer that we don't need anymore. make as much room as possible for the read */
				if (lineStart < (Client->Buffer + Client->BufferPtr))	{
					/* Preserve the unseen line fragment and move it to the beginning of the buffer */
					fragmentLen = (Client->Buffer + Client->BufferPtr) - lineStart;
					memmove(Client->Buffer, lineStart, fragmentLen + 1);  /* move the NULL */
					Client->BufferPtr = fragmentLen;
				} else {
					Client->BufferPtr = 0;
				}

				lineStart = Client->Buffer;
			}

			do {
				if (Client->BufferPtr < BUFSIZE) {
					count = DoClientRead(Client, Client->Buffer + Client->BufferPtr, BUFSIZE - Client->BufferPtr);

					if ((count > 0) && !Exiting) {
						Client->BufferPtr += count;
						Client->Buffer[Client->BufferPtr] = '\0';
						lineEnd = strchr(Client->Buffer, 0x0a);
						if (lineEnd != NULL) {
							 
							break;
						}

						/* There is still not a newline, go get more */
						continue;
					}

					return(EndClientConnection(Client));
				}

				/* the buffer is full and there is not a complete line. */
				/* since we are not interested in header line of this length */
				/* throw everything away up to the next newline */

				Client->BufferPtr = 0;
		
				do {
					if (Client->BufferPtr < BUFSIZE) {
						count = DoClientRead(Client, Client->Buffer + Client->BufferPtr, BUFSIZE - Client->BufferPtr);
						if ((count > 0) && !Exiting) {
							Client->BufferPtr += count;
							Client->Buffer[Client->BufferPtr] = '\0';
							lineEnd = strchr(Client->Buffer, 0x0a);
							if (lineEnd != NULL) {
								lineStart = lineEnd + 1;
								if (lineStart != Client->Buffer) {
									if (lineStart < (Client->Buffer + Client->BufferPtr))	{
										/* Preserve the line fragment in the buffer and make as much room as possible for the read */
										fragmentLen = (Client->Buffer + Client->BufferPtr) - lineStart;
										memmove(Client->Buffer, lineStart, fragmentLen + 1);  /* copy the NULL */
										Client->BufferPtr = fragmentLen;
									} else {
										Client->BufferPtr = 0;
									}

									lineStart = Client->Buffer;
								}
								break;
							}

							/* There is still not a newline, go get more */
							continue;
						}
						return(EndClientConnection(Client));
					}

					/* the buffer is full again and there is still not a complete line */
						
					Client->BufferPtr = 0;
					continue;

				} while (TRUE);

				/* the long line has been skipped */
				lineEnd = strchr(Client->Buffer, 0x0a);
				if (lineEnd == NULL) {
					/* the buffer does not contain a full line */
					continue;
				}

				break;
			} while (TRUE);
		}

		/* We have a complete header line in the buffer */
		*lineEnd = '\0';
		if (*(lineEnd - 1) == 0x0d) {
			*(lineEnd - 1) = '\0';
		}
		
		switch(toupper(lineStart[0])) {
			case '\0': {
				inHeader = FALSE;
				break;
			}

			case 'C': {
				if (WAQuickNCmp(lineStart, "connection: keep-alive", 22)) {
					Client->KeepAlive = TRUE;
				} else if (WAQuickNCmp(lineStart, "content-length: ", 16)) {
					Client->ContentLength = atol(lineStart + 16);
				} else if (WAQuickNCmp(lineStart, "content-type: ", 14)) {
					unsigned char	*ptr;

					if (WAQuickNCmp(lineStart + 14, "multipart/form-data", 19)) {
						Client->EncodingType = FORM_DATA_ENCODED;
						ptr = lineStart + 13;
						do {
							ptr = strchr(ptr + 1, 'b');

							if (ptr) {
								if (WAQuickNCmp(ptr, "boundary=", 9)) {
									if (ptr[9] != '"') {
										Client->FormSeparator = MemStrdup(ptr + 7);
									} else {
										Client->FormSeparator = MemStrdup(ptr + 8);
										ptr = strchr(ptr+10, '"');
										if (ptr) {
											*ptr = '\0';
										}
									}
									Client->FormSeparator[0] = '-';
									Client->FormSeparator[1] = '-';
									ptr = NULL;
								}
							}
						} while (ptr);
					} else {
						Client->EncodingType = FORM_URL_ENCODED;
					}
				} else if (WAQuickNCmp(lineStart, "cookie: ", 8)) {
					unsigned char	*ptr;

					ptr = lineStart + 7;
					do {
						ptr = strchr(ptr + 1, '=');
						if (ptr) {
							unsigned char		*CookieName = ptr - 1;

							while (*CookieName >= '0' && *CookieName <= '9' && CookieName > lineStart + 7) {
								CookieName--;
							}

							if ((CookieName - 2 > lineStart + 7) && WAQuickNCmp(CookieName - 2, "SID", 3)) {
							    unsigned long		CookieID = atol(CookieName + 1);
							    
							    if (CookieID == 0) {
								Client->SessionUID = atol(ptr + 1);
							    }
							}
						}
					} while (ptr);

					/*
						Find the next avaliable free spot for a cookie.  Don't bother checking
						the 10th spot, because if thats all thats left we are just going to use
						it anyway.
					*/
#if 0
					Client->CookieID = 0;
					while (Client->SessionUID[Client->CookieID] != 0 && Client->CookieID < 9) {
						Client->CookieID++;
					}
#endif	
				}
				break;
			}

			case 'A': {
				if (WAQuickNCmp(lineStart, "Authorization: Basic ", 21)) {
					int	len;
					len = strlen(lineStart + 21);
					
					if (len > (256 * 2)) {
					    len = 256 * 2;
					}

					memcpy(Client->Credentials, lineStart + 21, len);
					Client->Credentials[WADecodeBase64(Client->Credentials, len, &len)] = '\0';
					WADebug("Client Credentials:%s\n", Client->Credentials);
				} else if (WAQuickNCmp(lineStart, "Accept-Language: ", 17)) {

					/* Copy up to 20 characters of the language list. */
					HulaStrNCpy(Client->Language, lineStart + 17, 20);
					Client->Language[19] = '\0';

				} else if (WAQuickNCmp(lineStart, "Accept: ", 8) && strstr(lineStart, "wap")) {
					Client->DeviceType = DEVICE_WML;
				}
				break;
			}
		}
		
#if 0
		if (AuthHosts == 0) {
			;
		} else {
			if ((AuthIDHeader[0] == lineStart[0]) && WAQuickNCmp(lineStart, AuthIDHeader, AuthIDHeaderLen)) {
				unsigned char *ptr;
				unsigned long i;

				ptr = lineStart + AuthIDHeaderLen;
				while (*ptr != '\0') {
					if (!isspace(*ptr)) {
						HulaStrNCpy(Client->Credentials, sizeof(Client->Credentials), ptr);
						for (i = 0; i < AuthHosts; i++) {
							if (AuthAddress[i] == Client->cs.sin_addr.s_addr) {
								Client->TrustCredentials = TRUE;
								break;
							}
						}
						break;
					}
					ptr++;
				} 
			}
		}
#endif

		lineStart = lineEnd + 1;

	} while (inHeader);

	if (lineStart == Client->Buffer) {
		;
	} else if (lineStart < (Client->Buffer + Client->BufferPtr))	{
		fragmentLen = Client->BufferPtr - (lineStart - Client->Buffer);
		memmove(Client->Buffer, lineStart, fragmentLen + 1); /* copy the NULL */
		Client->BufferPtr = fragmentLen;
	} else {
		Client->BufferPtr = 0;
	}

	if (Client->ContentLength > 0) {
		if ((unsigned)Client->BufferPtr > Client->ContentLength) {
			Client->ContentLength = 0;
		} else {
			Client->ContentLength -= Client->BufferPtr;
		}
	}
	return(TRUE);

}


static BOOL
HandleConnection(void *param)
{
	ConnectionStruct		*Client = (ConnectionStruct *)param;
	int						count;
	int						ReplyInt;
	BOOL						Ready;
	unsigned char			*ptr;
	unsigned char			*URL = NULL;
	unsigned long			URLLen;
	unsigned char			Buffer[BUFSIZE+1];
	unsigned char			Answer[BUFSIZE+1];
	URLStruct				URLData;

	SetCurrentNameSpace(NWOS2_NAME_SPACE);

	XplSignalLocalSemaphore(ExitSemaphore);

	WADebug("HandleConnection() called\n");

	if (Client->ClientSSL) {
		if (SSL_accept(Client->CSSL) == -1) {
			return(EndClientConnection(Client));
		}
	}

	while (TRUE) {
		Ready = FALSE;
		if ((Client->BufferPtr > 0) && ((ptr = strchr(Client->Buffer, 0x0a)) != NULL)) {
			*ptr = '\0';

			memcpy(Client->Command, Client->Buffer, ptr-Client->Buffer + 1);
			Client->BufferPtr = strlen(ptr + 1);

			memmove(Client->Buffer, ptr + 1, Client->BufferPtr + 1);
			if ((ptr = strrchr(Client->Command, 0x0d)) != NULL) {
				*ptr = '\0';
			}

			Ready = TRUE;
		} else {
			while (!Ready) {
				if (Exiting) {
					return(EndClientConnection(Client));
				}

				WADebug("Doing DoClientRead\n");

				count = DoClientRead(Client, Client->Buffer + Client->BufferPtr, BUFSIZE - Client->BufferPtr);
				WADebug("Done DoClientRead\n");

				if ((count < 1) || (Exiting)) {
					return(EndClientConnection(Client));
				}

				Client->BufferPtr += count;
				Client->Buffer[Client->BufferPtr] = '\0';

				if ((ptr = strchr(Client->Buffer, 0x0a)) != NULL) {
					*ptr = '\0';

					memcpy(Client->Command, Client->Buffer, ptr - Client->Buffer + 1);

					Client->BufferPtr = strlen(ptr + 1);
					memmove(Client->Buffer, ptr + 1, Client->BufferPtr);
					Client->Buffer[Client->BufferPtr] = '\0';

					if ((ptr = strrchr(Client->Command, 0x0d)) != NULL) {
						*ptr = '\0';
					}

					Ready = TRUE;
				} else if (Client->BufferPtr > BUFSIZE) {
					return(EndClientConnection(Client));
				}
			}
		}

		//WADebug("[%d.%d.%d.%d] %s\n",	Client->cs.sin_addr.s_net,Client->cs.sin_addr.s_host,Client->cs.sin_addr.s_lh,Client->cs.sin_addr.s_impno,Client->Command);
		WADebug("Cmd:%s\n", Client->Command);

		/* Handle busy redirection */
#if 0
		if (Configuration & CONFIG_REDIRECT_ONLY) {
			ReadHeader(Client);
			KeepAlive=FALSE;
			if (Client->ClientSSL) {
				ReplyInt=snprintf(Answer, sizeof(Answer), MSGREDIRECT_TO, RedirectURLSSL, RedirectURLSSL, RedirectURLSSL);
			} else {
				ReplyInt=snprintf(Answer, sizeof(Answer), MSGREDIRECT_TO, RedirectURL, RedirectURL, RedirectURL);
			}
			WASendClient(Client, Answer, ReplyInt);
		}
#endif

		/* Read the rest of the HTTP header */

		ReadHeader(Client);
		Client->SentHeader = FALSE;

		URLLen = strlen(Client->Command);

		/*
			Parse the request header

			We aren't strict about the commands, but we do need to make sure that
			URL will not be pointing at random memory, so if we don't get enuf to
			ensure that we just clean it up.
		*/
		switch(toupper(Client->Command[0])) {
			case 'H': { 	/* HEAD */
				if (URLLen > 5) {
					URL = Client->Command + 5;
				} else {
					URL = Client->Command + URLLen;
					URL[0] = '/';
				}
				break;
			}

			case 'G': {		/* GET */
				if (URLLen > 4) {
					URL = Client->Command + 4;
				} else {
					URL = Client->Command + URLLen;
					URL[0] = '/';
				}
				break;
			}

			case 'P': {		/* POST */
				if (URLLen > 5) {
					URL = Client->Command + 5;
				} else {
					URL = Client->Command + URLLen;
					URL[0] = '/';
				}
				break;
			}

			default: {
				break;
			}
		}

		if (URL == NULL) {
			WASendClient(Client, MSGERRBADREQUEST, MSGERRBADREQUEST_LEN);
			EndClientConnection(Client);
		}

		/* Take off the HTTP/1.x at the end */
		/* Most common case first */

		ptr = strchr(URL, ' ');
		if (ptr) {
			*ptr = '\0';
		}

		Client->URLExtra = "";
		if ((ptr = strchr(URL, '+')) != NULL) {
			Client->URLExtra = ptr + 1;
			*ptr = '\0';
		}

		switch(tolower(URL[1])) {
			case 'a': {	/* Public access */
				if (!PublicTemplate) {
					Send404();
					break;
				}

				if (DecodeURL(URL + 3, &URLData)) {
					HandlePublicURL(Client, &URLData);
				} else if (URL[2] == '?' || URL[2] == '\0') {
				    SessionStruct *Session=malloc(sizeof(SessionStruct));
					memset(Session, 0, sizeof(SessionStruct));

					switch (Client->DeviceType) {
						default:
						case DEVICE_HTML: {
							WAEncodeURL(Session, Buffer, URL_TYPE_PUBLIC, DISPLAY_TEMPLATE, PublicTemplate->InitialTemplate, 0, 0, 0);
							break;
						}

						case DEVICE_WML: {
							WAEncodeURL(Session, Buffer, URL_TYPE_PUBLIC, DISPLAY_TEMPLATE, PublicTemplate->InitialWMLTemplate, 0, 0, 0);
							break;
						}
					}
					free (Session);

					WARedirectClient(Client, Buffer);
				} else {
					Send404();
				}

				break;
			}

			case 'w': {	/* Regular template request */
				if (DecodeURL(URL + 3, &URLData) && ((Client->Session=RetrieveSession(URLData.SessionID, URLData.SessionUID)) != NULL)) {
					/* Check access security */

#if 0
					if ((Client->SessionUID == URLData.SessionUID) || (Client->DeviceType != DEVICE_HTML)) {
						HandleURL(Client, Client->Session, &URLData);
					} else {
						SendTimeoutPage(Client, &URLData);
					}
#endif
					HandleURL(Client, Client->Session, &URLData);

				} else {
					if (URLData.ReloginOK) {
						Client->URL = URL;
						SendReloginPage(Client, &URLData);
					} else {
						SendTimeoutPage(Client, &URLData);
					}
				}

				break;
			}

			case 'p': {	/* Image request */
				if (DecodeURL(URL + 3, &URLData)) {
					HandleURL(Client, NULL, &URLData);
				} else {
					Send404();
				}
				break;
			}

			case 'r': {	/* Redirect request */
				if (DecodeURL(URL + 3, &URLData)) {
					WARedirectClient(Client, Client->URLExtra);
				} else {
					Send404();
				}
				break;
			}

			case 'i': {
				if (!WAQuickNCmp(URL + 1, "index", 5)) {
					Send404();
					break;
				}
			}
			/* Fall through */

			case '\0': {	/* Index request */
				WADebug("Login request, homepage\n");

				if (!PublicTemplate) {
					snprintf(Answer, sizeof(Answer), "/%d/l", (int) time(NULL));
					WARedirectClient(Client, Answer);
				} else {
					snprintf(Answer, sizeof(Answer), "/%c", URL_TYPE_PUBLIC);
					WARedirectClient(Client, Answer);
				}
				break;
			}

			case 'f': {	/* Form-based login; also start of group for regular login */
				BOOL				Disabled;
				unsigned char	FieldName[128] = "\0";
				unsigned char	Username[256] = "\0";
				unsigned char	Password[256] = "\0";
				unsigned char	Template[256] = "\0";
				unsigned char	InitialObject[MDB_MAX_OBJECT_CHARS + 1] = "\0";
				unsigned char	Page[256] = "\0";
				unsigned char	Key[32 + 1] = "\0";
				unsigned char	returnURL[256] = "\0";
				unsigned long	timeout = 0;
				unsigned long	ValueSize;

				if (URL[2] == '?') {	/* GET method */
					ptr = URL + 3;

					while (ptr && (ptr = GetURLFormName(ptr, FieldName, sizeof(FieldName))) != NULL) {
						ValueSize = BUFSIZE;
						ptr = GetURLFormValue(ptr, Client->Temp, &ValueSize);

						if (ValueSize > 0) {
						    switch(toupper(FieldName[0])) {
							case 'U' :
							    HulaStrNCpy(Username, Client->Temp, sizeof(Username));
							    break;
							case 'P' :
							    if (WAQuickNCmp(FieldName, "Pass", 4)) {
								HulaStrNCpy(Password, Client->Temp, sizeof(Password));
							    } else if (WAQuickNCmp(FieldName, "Page", 4)) {
								HulaStrNCpy(Page, Client->Temp, sizeof(Page));
							    }
							    break;
							case 'I' :
							    HulaStrNCpy(Template, Client->Temp, sizeof(Template));
							    break;
							case 'O': {		/* Object */
							    HulaStrNCpy(InitialObject, Client->Temp, sizeof(InitialObject));
							    break;
							}
							case 'K': {		/* Key */
							    HulaStrNCpy(Key, Client->Temp, sizeof(Key));
							    break;
							}
							case 'R': {		/* ReturnURL */
							    HulaStrNCpy(returnURL, Client->Temp, sizeof(returnURL));
							    break;
							}
							case 'T': {		/* Timeout */
							    timeout = atol(Client->Temp);
							    break;
								}
							}
						}
					}
				} else if (URL[2] == '\0') {
					while (WAGetFormName(Client, FieldName, sizeof(FieldName))) {
						ValueSize = BUFSIZE;
						WAGetFormValue(Client, Client->Temp, &ValueSize);

						if (ValueSize > 0) {
						    switch(toupper(FieldName[0])) {
							case 'U' :
							    HulaStrNCpy(Username, Client->Temp, sizeof(Username));
							    break;
							case 'P' :
							    if (WAQuickNCmp(FieldName, "Pass", 4)) {
								HulaStrNCpy(Password, Client->Temp, sizeof(Password));
							    } else if (WAQuickNCmp(FieldName, "Page", 4)) {
								HulaStrNCpy(Page, Client->Temp, sizeof(Page));
							    }
							    break;
							case 'I' :
							    HulaStrNCpy(Template, Client->Temp, sizeof(Template));
							    break;
							case 'O': {		/* Object */
							    HulaStrNCpy(InitialObject, Client->Temp, sizeof(InitialObject));
							    break;
							}
							case 'K': {		/* Key */
							    HulaStrNCpy(Key, Client->Temp, sizeof(Key));
							    break;
							}
							case 'R': {		/* ReturnURL */
							    HulaStrNCpy(returnURL, Client->Temp, sizeof(returnURL));
							    break;
							}
							case 'T': {		/* Timeout */
							    timeout = atol(Client->Temp);
							    break;
								}
							}
						}
					}
				} else {
					Send404();
					break;
				}

				if (Username[0] != '\0' && Password[0] != '\0') {
					snprintf(Client->Credentials, sizeof(Client->Credentials), "%s:%s", Username, Password);
				}
				/* Fall through */

			case '1':
			case '2':
			case '3':
			case '4':
			case '5':
			case '6':
			case '7':
			case '8':
			case '9':
			case '0': 	/* Login redirection */

				Disabled = FALSE;

				/* 
					We expect login credentials, if they're not provided or don't work send the guy back
					to the 404 page.

					If a key is provided we need to look for a session with the same key (and user name)
					before creating a session.  If we find it we will use that session instead of creating
					a new one.  Since CreateSession requires the user name and password to be passed in
					we are going to check them before hand, but only if the Key is set.
				*/

				if ((Client->Credentials[0] != '\0') && (ptr = strchr(Client->Credentials, ':')) != NULL) {
					*ptr = '\0';

					if (Key[0] != '\0') {
						unsigned char			UserDN[MDB_MAX_OBJECT_CHARS+1];

						WAX500toMDB(Client->Credentials, UserDN, NULL);
						Client->Session = RetrieveSessionByKey(Key, UserDN, ptr + 1);
					}

					if (Client->Session == NULL) {
						CreateSession(Client, Client->Credentials, ptr + 1, &Disabled);
					}
				}

				if (Disabled) {
					WADebug("Login: FEATURE_WEBADMIN disabled\n");
					Client->KeepAlive = FALSE;
					WASendClient(Client, "HTTP/1.1 200 Ok\r\nPragma: no-cache\r\nCache-Control: no-cache\r\nContent-type: text/html\r\nConnection: close\r\n", 114);
					WASendClient(Client, "Content-type: text/html\r\n\r\n<META HTTP-EQUIV=\"Content-Type\" CONTENT=\"text/html; charset=utf-8\">", 94);
					WASendClient(Client, "<H1>WebAdm feature disabled</H1>", 33);
				} else if (!Client->Session) {
					if (isdigit(URL[1])) {
						WADebug("Login: No or wrong credentials; sending 401 error\n");
						Client->KeepAlive = FALSE;
						switch (Client->DeviceType) {

							case DEVICE_HTML:
							default: {
								WASendClient(Client, "HTTP/1.1 401 Unauthorized\r\nPragma: no-cache\r\nCache-Control: no-cache\r\nContent-type: text/html\r\nConnection: close\r\n", 114);
								WASendClient(Client, "WWW-Authenticate: Basic realm=\"", 31);
								ReplyInt=snprintf(Answer, sizeof(Answer), "%d", (int) time(NULL));
								WASendClient(Client, Answer, ReplyInt);
								WASendClient(Client, "\"\r\nContent-type: text/html\r\n\r\n<META HTTP-EQUIV=\"Content-Type\" CONTENT=\"text/html; charset=utf-8\">", 97);
								WASendClient(Client, "<H1>Authentication required</H1>", 32);
								break;
							}

							case DEVICE_WML: {
								WASendClient(Client, "HTTP/1.1 401 Unauthorized\r\nPragma: no-cache\r\nCache-Control: no-cache\r\nContent-type: text/vnd.wap.wml\r\nConnection: close\r\n", 121);
								WASendClient(Client, "WWW-Authenticate: Basic realm=\"", 31);
								ReplyInt=snprintf(Answer, sizeof (Answer), "%d", (int) time(NULL));
								WASendClient(Client, Answer, ReplyInt);
								WASendClient(Client, "\"\r\n\r\n<?xml version=\"1.0\" encoding=\"UTF-8\"?><!DOCTYPE wml PUBLIC \"-//WAPFORUM//DTD WML 1.1//EN\" \"http://www.wapforum.org/DTD/wml_1.1.xml\"><wml><head><meta http-equiv=\"Cache-Control\" content=\"max-age=0\"/></head><card title=\"Login Page\"></card></wml>", 247);
								break;
							}
						}
					} else {
						WARedirectClient(Client, FormLoginRedirectURL);
					}
				} else {
					if (!Client->TSession) {
						/* Did he provide a key? */
						if (Key[0] != '\0') {
							SetSessionKey(Client->Session, Key);
						}

						/* Is he asking for a template other than his default? */
						if (Template[0] != '\0') {
							unsigned long	TemplateID;

							TemplateID = WAFindTemplate(Template);
							if (TemplateID != -1) {
								WASetSessionTemplate(TemplateID, Client->Session->Language, Client->Session);
							}
						}

						/* Did he ask for a specific object? */
						if (InitialObject[0] != '\0') {
							WAX500toMDB(InitialObject, Client->Session->CurrentObject, NULL);

							if (MDBIsObject(Client->Session->CurrentObject, Client->Session->V)) {
								MDBGetObjectDetails(Client->Session->CurrentObject, Client->Session->CurrentClass, NULL, NULL, Client->Session->V);
							} else {
								Client->Session->CurrentObject[0] = '\0';
							}
						}

						/* Did he provide a returnURL? */
						if (returnURL[0] != '\0') {
							HulaStrNCpy(Client->Session->returnURL,
								    returnURL,
								    sizeof(Client->Session->returnURL));
						}

						/* Did he provide a timeout value? */
						if (timeout != 0) {
							Client->Session->ConnectionTimeout = timeout * 60;
						}

						/* let's build a URL and redirect the guy to the template's default page */
						switch (Client->DeviceType) {
							case DEVICE_HTML: 
							default: {
								signed long			pageID = -1;
								unsigned long		i;
								unsigned long		TemplateID = Client->Session->TemplateID;

								if (Page[0] != '\0') {
									/*
										Is he asking for a page other than the default?  If so we
										need to find it.  We have to remeber that there is a very
										good chance that it isn't in the default template, but a
										helper template instead.
									*/

									for (i = 0; i < TemplateCount && (pageID == -1); i++) {
										WASetSessionTemplate(i, Client->Session->Language, Client->Session);
										pageID = (unsigned long) WAFindTemplatePage(Page, Client->Session->TemplateID);
									}
								}

								if (pageID == -1) {
									/* We didn't find it, so restore the defaults */
									WASetSessionTemplate(TemplateID, Client->Session->Language, Client->Session);
									pageID = Templates[Client->Session->TemplateID]->InitialTemplate;
								}

								WAEncodeURL(Client->Session, Buffer, URL_TYPE_LINK, DISPLAY_TEMPLATE, pageID, 0, 0, 0);
								break;
							}

							case DEVICE_WML: {
								WAEncodeURL(Client->Session, Buffer, URL_TYPE_LINK, DISPLAY_TEMPLATE, Templates[Client->Session->TemplateID]->InitialWMLTemplate, 0, 0, 0);
								break;
							}
						}

						RedirectAfterLogin(Client, Buffer);
					} else {
						snprintf(Client->Temp, sizeof(Client->Temp), "%s/T%lu", WorkDir, Client->TSession->SessionID);
						Client->PostData=fopen(Client->Temp, "rb");

						/* Original content length */
						fgets(Client->Temp, BUFSIZE, Client->PostData);
						Client->ContentLength=atol(Client->Temp);

						/* Separator */
						fgets(Client->Temp, BUFSIZE, Client->PostData);
						ChopNL(Client->Temp);
						if (Client->Temp[0]!='\0') {
							Client->FormSeparator=strdup(Client->Temp+1);
						}

						if (Client->Temp[0]=='f') {
							Client->EncodingType=FORM_DATA_ENCODED;
						} else {
							Client->EncodingType=FORM_URL_ENCODED;
						}

						/* Original request URL */
						fgets(Client->Temp, BUFSIZE, Client->PostData);
						ChopNL(Client->Temp);

						if (!DecodeURL(Client->Temp + 3, &URLData)) {
							switch (Client->DeviceType) {
								case DEVICE_HTML: 
								default: {
									WAEncodeURL(Client->Session, Buffer, URL_TYPE_LINK, DISPLAY_TEMPLATE, Templates[Client->Session->TemplateID]->InitialTemplate, 0, 0, 0);
									break;
								}

								case DEVICE_WML: {
									WAEncodeURL(Client->Session, Buffer, URL_TYPE_LINK, DISPLAY_TEMPLATE, Templates[Client->Session->TemplateID]->InitialWMLTemplate, 0, 0, 0);
									break;
								}
							}
							RedirectAfterLogin(Client, Buffer);
						} else {
//							Client->Session->SessionID=URLData.SessionID;
//							Client->Session->SessionUID=URLData.SessionUID;
							Client->Cookie=Client->Session->SessionUID;
							HandleURL(Client, Client->Session, &URLData);
						}
					}
				}
				break;
			}

			default: {
				Send404();
				break;
			}
		}
		
		WAFlushClient(Client);

		if (!Client->KeepAlive) {
			EndClientConnection(Client);
		} else {
			ReleaseSession(Client->Session);
		}
	}
	return(TRUE);
}

static int
RequestUnload(void)
{
	int	SemCount;
	XplThreadID id;

	id = XplSetThreadGroupID(TGid);
	TGid = id;
	
	if (Unloading) {
		return(0);
	}

	Unloading = TRUE;

	printf("\r%s: Preparing to unload; please be patient, this may take a minute.\n", PRODUCT_NAME);

	/* Wait 'til everyone's closed */

	Exiting=TRUE;

	/* fixme - SocketShutdown; */

	XplExamineLocalSemaphore(ExitSemaphore, SemCount);
	while (SemCount>1) {
		XplDelay(500);
		XplExamineLocalSemaphore(ExitSemaphore, SemCount);
	}

	if (ServerSocket!=-1) {
		IPclose(ServerSocket);
	}
	ServerSocket=-1;

	if(ServerSocketSSL!=-1) {
		IPclose(ServerSocketSSL);
	}
	ServerSocketSSL=-1;

	XplCloseLocalSemaphore(ExitSemaphore);

	XplCloseLocalSemaphore(CacheSemaphore);
	ClearConnectionCache();

	DestroyAllSessions();
	XplCloseLocalSemaphore(SessionDBSemaphore);
	XplCloseLocalSemaphore(TSessionDBSemaphore);
	SessionDBShutdown();

	FreeTemplates();
	FreeModules();
	if (DefaultTitle) {
		free(DefaultTitle);
	}

	/* Cleanup SSL */

	if (ServerCertificate) {
		X509_free(ServerCertificate);
	}

	if (ServerPKey) {
		EVP_PKEY_free(ServerPKey);
	}

	if (SSLContext) {
		SSL_CTX_free(SSLContext);
	}
	ERR_free_strings();
	ERR_remove_state(0);
	EVP_cleanup();

	//MDBShutdown();

    MemoryManagerClose(WA_SERVER_OBJECT);

	printf("\r%s Shutdown complete\n", PRODUCT_NAME);

	XplSetThreadGroupID(TGid);
	return(0);
}

#if defined(NOVELL) || defined(LIBC) || defined(WIN32)
int _NonAppCheckUnload(void)
{
	int			s;
	static BOOL	checked = FALSE;
	XplThreadID	id;

	if (!checked) {
		checked = Exiting = TRUE;

		XplWaitOnLocalSemaphore(ShutdownSemaphore);

		id = XplSetThreadGroupID(TGid);
		if (ServerSocket != -1) {
			s = ServerSocket;
			ServerSocket = -1;

			IPclose(s);
		}
		XplSetThreadGroupID(id);

		XplWaitOnLocalSemaphore(ServerSemaphore);
	}

	return(0);
}
#endif
static void
WebAdmShutdownSigHandler(int sigtype)
{
	if (!Unloading) {
		RequestUnload();
	}

#if 0
#if !defined(NETWARE) && !defined(WIN32)
	#if defined(LINUX) || defined(S390RH)
		killpg(0, 9);
	#else
		exit(0);
	#endif
#endif
#endif

    return;
}

static const char *
GetUnpriviledgedUser(void)
{
    if (HULA_USER[0] == '\0') {
	return NULL;
    } else {
	return HULA_USER;
    }
}

static int
ServerSocketInit(void)
{
    int code;
    struct sockaddr_in	server_sockaddr;
    
    ServerSocket=IPsocket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
    if (ServerSocket == -1) {
	XplConsolePrintf("hulawebadmin: Unable to allocate socket\n");
	return -1;
    }
    code=1;
    setsockopt(ServerSocket, SOL_SOCKET, SO_REUSEADDR, (char *)&code, sizeof(code));
    
    memset(&server_sockaddr, 0, sizeof(struct sockaddr));
    
    server_sockaddr.sin_family = AF_INET;
    server_sockaddr.sin_port = htons((unsigned short)WEBADMIN_PORT);
    server_sockaddr.sin_addr.s_addr=BindIPAddress;

    /* Get root privs back for the bind.  It's ok if this fails - 
     * the user might not need to be root to bind to the port */
    XplSetEffectiveUserId(0);

    code = IPbind(ServerSocket, (struct sockaddr *)&server_sockaddr, sizeof(server_sockaddr));

    if (XplSetEffectiveUser(GetUnpriviledgedUser()) < 0) {
	XplConsolePrintf("hulawebadmin: Could not drop to unpriviledged user '%s'\n", GetUnpriviledgedUser());
	return -1;
    }

    if (code < 0) {
	XplConsolePrintf("hulawebadmin: Unable to bind to port %d\n", WEBADMIN_PORT);
	return -1;
    }

    return 0;
}

static int
ServerSocketSSLInit(void)
{
    int code;
    struct sockaddr_in	server_sockaddr;
    
    ServerSocketSSL=IPsocket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
    if (ServerSocketSSL == -1) {
	XplConsolePrintf("hulawebadmin: Unable to allocate SSL socket\n");
	return -1;
    }

    code=1;
    setsockopt(ServerSocketSSL, SOL_SOCKET, SO_REUSEADDR, (char *)&code, sizeof(code));
    
    memset(&server_sockaddr, 0, sizeof(struct sockaddr));
    
    server_sockaddr.sin_family = AF_INET;
    server_sockaddr.sin_port = htons((unsigned short)WEBADMIN_PORT_SSL);
    server_sockaddr.sin_addr.s_addr=BindIPAddress;
    

    /* Get root privs back for the bind.  It's ok if this fails - 
     * the user might not need to be root to bind to the port */
    XplSetEffectiveUserId(0);

    code = IPbind(ServerSocketSSL, (struct sockaddr *)&server_sockaddr, sizeof(server_sockaddr));

    if (XplSetEffectiveUser(GetUnpriviledgedUser()) < 0) {
	XplConsolePrintf("hulawebadmin: Could not drop to unpriviledged user '%s'\n", GetUnpriviledgedUser());
	return -1;
    }
    if (code < 0) {
	XplConsolePrintf("hulawebadmin: Unable to bind to SSL port %d\n", WEBADMIN_PORT_SSL);
	return -1;
    }

    return 0;
}

static void
ServerStartup(void *ignored)
{
	int						code, arg, ds;
	struct sockaddr_in	client_sockaddr;
	ConnectionStruct		*Client;
	XplThreadID				ID;
    
	code = IPlisten(ServerSocket, 2048);
	if (code == -1) {
	    IPclose(ServerSocket);
	    ServerSocket=-1;
	    raise(SIGTERM);
	    return;
	}

	while (!Exiting) {
		arg = sizeof(client_sockaddr);
		ds = IPaccept(ServerSocket, (struct sockaddr *)&client_sockaddr, &arg);
		if (ds == -1 ) {
			IPclose(ServerSocket);
			ServerSocket=-1;
			break;
		}
		if (!Exiting) {
			Client=GetCachedConnection();
			if (Client) {
				Client->s=ds;
				Client->cs=client_sockaddr;
				Client->State=STATE_FRESH;
				Client->BufferPtr=0;
				Client->SBufferPtr=0;
				Client->KeepAlive=FALSE;
				Client->ClientSSL = 0;
				Client->Credentials[0]='\0';
				Client->Session=NULL;
				Client->TSession=NULL;
				Client->PostData=NULL;
				Client->FormSeparator=NULL;
				Client->NMAPProtected=FALSE;
				Client->DeviceType = DEVICE_HTML;
				Client->Cookie=0;
				Client->ContentType = NULL;

				// Set TCP non blocking io
				code=1;
				setsockopt(ds, IPPROTO_TCP, 1, (char *)&code, sizeof(code));

				XplBeginThread(&ID, HandleConnection, MODWEB_STACK_SPACE, (void *) Client, code);
			} else {
				IPsend(ds, MSGERRNOMEMORY, MSGERRNOMEMORY_LEN, 0);
				IPclose(ds);
			}
		} else {
			IPclose(ds);
		}
	}

	XplSignalLocalSemaphore(ServerSemaphore);
	XplWaitOnLocalSemaphore(ShutdownSemaphore);

	return;
}

static void
ServerStartupSSL(void *ignored)
{
	int						code, arg, ds;
	struct sockaddr_in	client_sockaddr;
	ConnectionStruct		*Client;
	XplThreadID				ID;

	code = IPlisten(ServerSocketSSL, 2048);
	if (code == -1) {
		IPclose(ServerSocketSSL);
		ServerSocketSSL=-1;
		raise(SIGTERM);
		return;
	}

	while (!Exiting) {
		arg = sizeof(client_sockaddr);
		ds = IPaccept(ServerSocketSSL, (struct sockaddr *)&client_sockaddr, &arg);
		if (ds == -1 ) {
			IPclose(ServerSocketSSL);
			ServerSocketSSL=-1;
			return;
		}
		if (!Exiting) {
			Client=GetCachedConnection();
			if (Client) {
				Client->s=ds;
				Client->cs=client_sockaddr;
				Client->State=STATE_FRESH;
				Client->BufferPtr=0;
				Client->SBufferPtr=0;
				Client->KeepAlive=FALSE;
				Client->ClientSSL = 1;
				Client->CSSL = SSL_new(SSLContext);
				Client->Credentials[0]='\0';
				Client->Session=NULL;
				Client->TSession=NULL;
				Client->PostData=NULL;
				Client->FormSeparator=NULL;
				Client->NMAPProtected=FALSE;
				Client->DeviceType = DEVICE_HTML;
				Client->Cookie=0;
				Client->ContentType = NULL;

				code=SSL_set_fd(Client->CSSL, Client->s);

				if (code!=-1) {
					code=1;
					setsockopt(ds, IPPROTO_TCP, 1, (unsigned char *)&code, sizeof(code));
					XplBeginThread(&ID, HandleConnection, MODWEB_STACK_SPACE, (void *) Client, code);
				} else {
					free(Client);
					IPclose(ds);
				}
			} else {
				IPsend(ds, MSGERRNOMEMORY, MSGERRNOMEMORY_LEN, 0);
				IPclose(ds);
			}
		} else {
			IPclose(ds);
		}
	}
}

static BOOL
ReadConfiguration(void)
{
	MDBHandle			ConfigHandle;
	MDBValueStruct		*Config = NULL;
	unsigned char		*ptr;

	WADebug("ReadConfiguration() called\n");

	WA_CONFIG_INIT;

	ConfigHandle = MDBAuthenticate("WebAdmin", NULL, NULL);
	if (ConfigHandle) {
		Config = MDBCreateValueStruct(ConfigHandle, NULL);
	}

	if (!Config) {
		return(FALSE);
	}

	if (!MDBGetServerInfo(DefaultContext, LocalTree, NULL)) {
		printf("\r%s: Could not find server object\n", PRODUCT_NAME);
		MDBDestroyValueStruct(Config);
		MDBRelease(ConfigHandle);
		return(FALSE);
	}

	if (!MDBRead(DefaultContext, WA_A_CONFIG_DN, Config)) {
		/* Cleanup the default context */
		ptr = strrchr(DefaultContext, '\\');
		if (ptr) {
			*ptr = '\0';
		}

		IgnoreConfig = TRUE;
		MDBDestroyValueStruct(Config);
		MDBRelease(ConfigHandle);
		return(FALSE);
	}

	HulaStrNCpy(ConfigDN, Config->Value[0], sizeof(ConfigDN));
	MDBFreeValues(Config);

	if (MDBRead(ConfigDN, WA_A_CONFIG, Config)) {
		unsigned long			i;

		for (i = 0; i < Config->Used; i++) {
			if (WAQuickCmp(Config->Value[i], "RolesInstalled")) {
				IgnoreRoles = FALSE;
			} else if (XplStrNCaseCmp(Config->Value[i], "JIT=", 4) == 0) {
				WebAdmin.CheckDiskForTemplateUpdates = atol(Config->Value[i] + 4);
			}
		}

		MDBFreeValues(Config);
	}

	if (MDBRead(ConfigDN, WA_A_PORT, Config)) {
		WEBADMIN_PORT = atol(Config->Value[0]);

		MDBFreeValues(Config);
	}

	if (MDBRead(ConfigDN, WA_A_SSLPORT, Config)) {
		WEBADMIN_PORT_SSL = atol(Config->Value[0]);

		MDBFreeValues(Config);
	}

	if (MDBRead(ConfigDN, WA_A_PATH, Config)) {
	    HulaStrNCpy(ModuleDir, Config->Value[0], sizeof(ModuleDir));
	    snprintf(TemplateDir, sizeof (TemplateDir), "%s/webadmin", Config->Value[0]);
	    snprintf(WorkDir, sizeof (WorkDir), "%s/work", Config->Value[0]);
	    snprintf(LogoDir, sizeof (WorkDir), "%s/logo", Config->Value[0]);
	    
	    MDBFreeValues(Config);
	}

    WA_CONFIG_INIT

	if (MDBRead(ConfigDN, WA_A_CERT_PATH, Config)) {
		HulaStrNCpy(ServerCertificatePath, Config->Value[0], sizeof(ServerCertificatePath));

		MDBFreeValues(Config);
	}

	if (MDBRead(ConfigDN, WA_A_KEY_PATH, Config)) {
	    HulaStrNCpy(ServerPKeyPath, Config->Value[0], sizeof(ServerPKeyPath));
	    
	    MDBFreeValues(Config);
	}

	/* Cleanup the default context */
	ptr = strrchr(DefaultContext, '\\');
	if (ptr) {
		*ptr = '\0';
	}

	MDBDestroyValueStruct(Config);
	MDBRelease(ConfigHandle);

	return(TRUE);
}

static BOOL
StoreConfiguration(MDBHandle ConfigHandle)
{
	MDBValueStruct		*V = NULL;
	unsigned char		Buffer[1024];
	unsigned char		Config[MDB_MAX_OBJECT_CHARS+1];

	WADebug("StoreConfiguration() called\n");

	V = MDBCreateValueStruct(ConfigHandle, NULL);

	if (!V) {
		return(FALSE);
	}

	if (!MDBGetServerInfo(Config, NULL, NULL)) {
		printf("\r%s: Could not find server object\n", PRODUCT_NAME);

		MDBDestroyValueStruct(V);
		return(FALSE);
	}

	if (!MDBRead(Config, WA_A_CONFIG_DN, V)) {
		MDBDestroyValueStruct(V);

		return(FALSE);
	}

	HulaStrNCpy(Config, V->Value[0], sizeof(Config));
	MDBFreeValues(V);

	XplPrintHostIPAddress(Buffer, sizeof(Buffer));

	MDBAddValue(Buffer, V);
	MDBWrite(Config, WA_A_ADDRESS, V);
	MDBFreeValues(V);

	snprintf(Buffer, sizeof(Buffer), "%d", WEBADMIN_PORT);
	MDBAddValue(Buffer, V);
	MDBWrite(Config, WA_A_PORT, V);
	MDBFreeValues(V);

	snprintf(Buffer, sizeof(Buffer), "%d", WEBADMIN_PORT_SSL);
	MDBAddValue(Buffer, V);
	MDBWrite(Config, WA_A_SSLPORT, V);
	MDBFreeValues(V);

	MDBAddValue(ModuleDir, V);
	MDBWrite(Config, WA_A_PATH, V);
	MDBFreeValues(V);

	MDBAddValue(ServerCertificatePath, V);
	MDBWrite(Config, WA_A_CERT_PATH, V);
	MDBFreeValues(V);

	MDBAddValue(ServerPKeyPath, V);
	MDBWrite(Config, WA_A_KEY_PATH, V);

	MDBDestroyValueStruct(V);

	return(TRUE);
}

static BOOL
AddV3Extension(X509 *Certificate, int nid, char *Value)
{
	X509_EXTENSION		*Extension;
	X509V3_CTX			Context;

//	X509V3_set_ctx_nodb(&Context);

	/* The issuer and subject cert is the same since we're self-signed */
	X509V3_set_ctx(&Context, Certificate, Certificate, NULL, NULL, 0);
	Extension = X509V3_EXT_conf_nid(NULL, &Context, nid, Value);
	if (!Extension) {
		return(FALSE);
	}

	X509_add_ext(Certificate, Extension, -1);
	X509_EXTENSION_free(Extension);
	return(TRUE);
}

static void
GenerateCertificateCallback(int p, int n, void *arg)
{
#if 0
	char	c='B';

	switch(p) {
		case 0: c='.'; break;
		case 1: c='+'; break;
		case 2: c='*'; break;
		case 3: c='\n'; break;
	}
	fputc(c, stderr);
#endif

	XplDelay(55);

	return;
}

static BOOL
GenerateCertificate()
{
	X509					*Cert;
	X509_NAME			*Name = NULL;
	EVP_PKEY				*PrivKey;
	RSA					*RSAData;
	FILE					*CertFile;
	unsigned char		Buffer[256];

	do {
		unsigned long		len;

		srand((unsigned) time(NULL));

		len = snprintf(Buffer, sizeof(Buffer), "%d", rand());
		RAND_seed(Buffer, len);
	} while (RAND_status() != 1);

	/* Generate a private key */
	if ((PrivKey = EVP_PKEY_new()) == NULL) {
		return(FALSE);
	}

	if ((Cert = X509_new()) == NULL) {
		EVP_PKEY_free(PrivKey);

		return(FALSE);
	}

	RSAData = RSA_generate_key(512, RSA_F4, GenerateCertificateCallback, NULL);
	if (!EVP_PKEY_assign_RSA(PrivKey, RSAData)) {
		X509_free(Cert);
		EVP_PKEY_free(PrivKey);

		return(FALSE);
	}

	RSAData = NULL;

	X509_set_version(Cert, 2);
	ASN1_INTEGER_set(X509_get_serialNumber(Cert), 0x27051977);
	X509_gmtime_adj(X509_get_notBefore(Cert), 0);
	X509_gmtime_adj(X509_get_notAfter(Cert), 86400*3650);
	X509_set_pubkey(Cert, PrivKey);

	Name = X509_get_subject_name(Cert);
	X509_NAME_add_entry_by_txt(Name, "C", MBSTRING_ASC, "US", -1, -1, 0);

	XplPrintHostIPAddress(Buffer, sizeof(Buffer));

	X509_NAME_add_entry_by_txt(Name, "CN", MBSTRING_ASC, Buffer, -1, -1, 0);

	/* This makes it "self-signed" */
	X509_set_issuer_name(Cert, Name);

	/* Add extensions */
	AddV3Extension(Cert, NID_basic_constraints, "critical,CA:TRUE");
	AddV3Extension(Cert, NID_key_usage, "critical,digitalSignature,keyEncipherment,keyCertSign");

	AddV3Extension(Cert, NID_subject_key_identifier, "hash");

	if (!X509_sign(Cert, PrivKey, EVP_md5())) {
		X509_free(Cert);
		EVP_PKEY_free(PrivKey);

		return(FALSE);
	}

	CertFile = fopen(WA_GENERATED_CERT, "wb");
	if (CertFile) {
		PEM_write_X509(CertFile, Cert);

		fprintf(CertFile, "\r\n");

		PEM_write_PrivateKey(CertFile, PrivKey, NULL, NULL, 0, NULL, NULL);

		fclose(CertFile);

		X509_free(Cert);
		EVP_PKEY_free(PrivKey);

		return(TRUE);
	} else {
		X509_free(Cert);
		EVP_PKEY_free(PrivKey);

		return(FALSE);
	}
}

#define		ReportMDBError(String)																																	\
{																																												\
	if (V->ErrNo != 0 && V->ErrNo != ERR_ATTRIBUTE_ALREADY_EXISTS && V->ErrNo != ERR_CLASS_ALREADY_EXISTS && V->ErrNo != ERR_ENTRY_ALREADY_EXISTS) {	\
		printf("\r%s: %s, ErrNo:%lu\n", PRODUCT_NAME, (String), V->ErrNo);																		\
		exit(0);																																								\
	} else {																																									\
		V->ErrNo = 0;																																						\
	}																																											\
}																																												\

static BOOL
WAInstall(int argc, char *argv[])
{
	MDBHandle			ExtendHandle;
	MDBValueStruct		*V;
	MDBValueStruct		*Naming;
	MDBValueStruct		*Mandatory;
	MDBValueStruct		*Optional;
	MDBValueStruct		*Superclass;
	MDBValueStruct		*Attribute;
	MDBValueStruct		*Data;
	unsigned char		ObjectName[MDB_MAX_OBJECT_CHARS+1];
	unsigned char		Tree[MDB_MAX_OBJECT_CHARS+1];
	unsigned char		ServerDN[MDB_MAX_OBJECT_CHARS+1];
	unsigned char		*ServerName = ServerDN;

	/* Can not be used with other arguments */

	if (argc < 3) {
		printf("\r%s: You must supply a user name and password\n", PRODUCT_NAME);
		exit(0);
	}

	if (!MDBInit()) {
		printf("\r%s:  Unable to contact directory service\r\n", PRODUCT_NAME);
		exit(-1);
	}

	printf("\r\n%s: Extending Schema\n", PRODUCT_NAME);

	/* Login */
	WAX500toMDB(argv[2], ObjectName, NULL);
	ExtendHandle = MDBAuthenticate("WebAdmin", ObjectName, argv[3]);
	if (!ExtendHandle) {
		printf("\r%s: Authentication failed\n", PRODUCT_NAME);
		exit(0);
	}

	V = MDBCreateValueStruct(ExtendHandle, NULL);
	Naming = MDBCreateValueStruct(ExtendHandle, NULL);
	Mandatory = MDBCreateValueStruct(ExtendHandle, NULL);
	Optional = MDBCreateValueStruct(ExtendHandle, NULL);
	Superclass = MDBCreateValueStruct(ExtendHandle, NULL);
	Attribute = MDBCreateValueStruct(ExtendHandle, NULL);
	Data = MDBCreateValueStruct(ExtendHandle, NULL);

	if (!V || !Naming || !Mandatory || !Optional || !Superclass || !Attribute || !Data) {
		printf("\r%s: Authentication failed\n", PRODUCT_NAME);
		exit(0);
	}

	/* Extend Schema */
	MDBDefineAttribute(WA_A_ROLE_NAME,	"2.16.840.1.113719.2.407.4.1", MDB_ATTR_SYN_STRING, TRUE, TRUE, FALSE, V);			ReportMDBError("Failed to define attribute: " WA_A_ROLE_NAME);

	MDBDefineAttribute(WA_A_CONFIG,		"2.16.840.1.113719.2.407.4.2", MDB_ATTR_SYN_STRING, FALSE, TRUE, TRUE, V);			ReportMDBError("Failed to define attribute: " WA_A_CONFIG);
	MDBDefineAttribute(WA_A_PORT,			"2.16.840.1.113719.2.407.4.3", MDB_ATTR_SYN_STRING, TRUE, TRUE, TRUE, V);			ReportMDBError("Failed to define attribute: " WA_A_PORT);
	MDBDefineAttribute(WA_A_SSLPORT,		"2.16.840.1.113719.2.407.4.4", MDB_ATTR_SYN_STRING, TRUE, TRUE, TRUE, V);			ReportMDBError("Failed to define attribute: " WA_A_SSLPORT);
	MDBDefineAttribute(WA_A_PATH,			"2.16.840.1.113719.2.407.4.5", MDB_ATTR_SYN_STRING, TRUE, TRUE, TRUE, V);			ReportMDBError("Failed to define attribute: " WA_A_PATH);
	MDBDefineAttribute(WA_A_CERT_PATH,	"2.16.840.1.113719.2.407.4.6", MDB_ATTR_SYN_STRING, TRUE, TRUE, TRUE, V);			ReportMDBError("Failed to define attribute: " WA_A_CERT_PATH);
	MDBDefineAttribute(WA_A_KEY_PATH,	"2.16.840.1.113719.2.407.4.7", MDB_ATTR_SYN_STRING, TRUE, TRUE, TRUE, V);			ReportMDBError("Failed to define attribute: " WA_A_KEY_PATH);
	MDBDefineAttribute(WA_A_ADDRESS,		"2.16.840.1.113719.2.407.4.8", MDB_ATTR_SYN_STRING, TRUE, TRUE, TRUE, V);			ReportMDBError("Failed to define attribute: " WA_A_ADDRESS);

	MDBDefineAttribute(WA_A_CONFIG_DN,	"2.16.840.1.113719.2.407.4.9", MDB_ATTR_SYN_DIST_NAME, TRUE, TRUE, TRUE, V);		ReportMDBError("Failed to define attribute: " WA_A_CONFIG_DN);
	MDBAddAttribute(WA_A_CONFIG_DN, C_NCP_SERVER, V);														ReportMDBError("Failed to add attribute " WA_A_CONFIG " to class: " WA_C_SERVER_OBJECT);

	MDBAddValue(C_TOP, Superclass);
	MDBAddValue(A_ORGANIZATIONAL_UNIT_NAME, Naming);
	MDBAddValue(A_ORGANIZATIONAL_UNIT_NAME, Mandatory);
	MDBAddValue(C_ORGANIZATION, V);
	MDBAddValue(C_ORGANIZATIONAL_UNIT, V);
	MDBDefineClass(WA_C_SERVER_OBJECT,	"2.16.840.1.113719.2.407.6.1", TRUE, Superclass, V, Naming, Mandatory, Optional, V);	ReportMDBError("Failed to define class: " WA_C_SERVER_OBJECT);
	MDBFreeValues(V);

	MDBAddValue(WA_C_SERVER_OBJECT, V);
	MDBDefineClass(WA_C_ROLE,				"2.16.840.1.113719.2.407.6.2", FALSE, Superclass, V, Naming, Mandatory, Optional, V);	ReportMDBError("Failed to define class: " WA_C_ROLE);
	MDBFreeValues(V);

	MDBAddAttribute(WA_A_ROLE_NAME, WA_C_ROLE, V);				ReportMDBError("Failed to add attribute " WA_A_ROLE_NAME " to class: " WA_C_ROLE);
	MDBAddAttribute(WA_A_CONFIG, WA_C_ROLE, V);					ReportMDBError("Failed to add attribute " WA_A_CONFIG " to class: " WA_C_ROLE);
	MDBAddAttribute(WA_A_PORT, WA_C_SERVER_OBJECT, V);			ReportMDBError("Failed to add attribute " WA_A_PORT " to class: " WA_C_SERVER_OBJECT);
	MDBAddAttribute(WA_A_SSLPORT, WA_C_SERVER_OBJECT, V);		ReportMDBError("Failed to add attribute " WA_A_SSLPORT " to class: " WA_C_SERVER_OBJECT);
	MDBAddAttribute(WA_A_PATH, WA_C_SERVER_OBJECT, V);			ReportMDBError("Failed to add attribute " WA_A_PATH " to class: " WA_C_SERVER_OBJECT);
	MDBAddAttribute(WA_A_CERT_PATH, WA_C_SERVER_OBJECT, V);	ReportMDBError("Failed to add attribute " WA_A_CERT_PATH " to class: " WA_C_SERVER_OBJECT);
	MDBAddAttribute(WA_A_KEY_PATH, WA_C_SERVER_OBJECT, V);	ReportMDBError("Failed to add attribute " WA_A_KEY_PATH " to class: " WA_C_SERVER_OBJECT);
	MDBAddAttribute(WA_A_ADDRESS, WA_C_SERVER_OBJECT, V);		ReportMDBError("Failed to add attribute " WA_A_ADDRESS " to class: " WA_C_SERVER_OBJECT);
	MDBAddAttribute(WA_A_CONFIG, WA_C_SERVER_OBJECT, V);		ReportMDBError("Failed to add attribute " WA_A_CONFIG " to class: " WA_C_SERVER_OBJECT);

	MDBDestroyValueStruct(Naming);
	MDBDestroyValueStruct(Mandatory);
	MDBDestroyValueStruct(Optional);
	MDBDestroyValueStruct(Superclass);

	/* Create Objects */

	Tree[0] = '\\';
	if (!MDBGetServerInfo(ServerDN, (Tree + 1), NULL)) {
		printf("\r%s: Could not find server object\n", PRODUCT_NAME);

		MDBDestroyValueStruct(Attribute);
		MDBDestroyValueStruct(Data);
		MDBDestroyValueStruct(V);
		MDBRelease(ExtendHandle);
		exit(0);
	}

	ServerName = strrchr(ServerDN, '\\');
	if (ServerName) {
		ServerName++;
	} else {
		ServerName = ServerDN;
	}

	if (argc > 4) {
		snprintf(ObjectName, sizeof(ObjectName), "%s\\%s %s", argv[4], ServerName, WA_SERVER_OBJECT);
	} else {
		unsigned char		*ptr;

		ptr = strrchr(ObjectName, '\\');
		if (ptr) {
		    int size;
		    ptr++;
		    size = sizeof (ObjectName) - (ptr - ObjectName);
		    snprintf(ptr, size, "%s %s", ServerName, WA_SERVER_OBJECT);
		} else {
			printf("\r%s: Please specify a container\n", PRODUCT_NAME);
		}
	}

	/*
		If the object already exists then we simply bump the use count.  If
		not then we create it.  When we try to uninstall we drop the use count
	*/

	if (!MDBIsObject(ObjectName, V)) {
		MDBCreateObject(ObjectName, WA_C_SERVER_OBJECT, Attribute, Data, V);
		ReportMDBError("Failed to create object: " WA_C_SERVER_OBJECT);

		MDBAddValue(ObjectName, V);
		if (!MDBWriteDN(ServerDN, WA_A_CONFIG_DN, V)) {
			printf("\r%s: Could not write configuration, ErrNo:%lu\n", PRODUCT_NAME, V->ErrNo);

			MDBDestroyValueStruct(Attribute);
			MDBDestroyValueStruct(Data);
			MDBDestroyValueStruct(V);
			MDBRelease(ExtendHandle);
			exit(0);
		}
		MDBFreeValues(V);
	} else {
		unsigned long		i;
		unsigned long		UseCount = 1;	/* The object is there, so it must be at least one */

		/* The object already exists.  Read the use count, and bump it */

		MDBRead(ObjectName, WA_A_CONFIG, V);
		for (i = 0; i < V->Used; i++) {
			if (WAQuickNCmp(V->Value[i], "UseCount:", 9)) {
				UseCount = atol(V->Value[i] + 9);
				MDBFreeValue(i, V);
			}
		}

		/* Bump it and store the new one (Abuse ServerDN) */
		UseCount++;
		snprintf(ServerDN, sizeof(ServerDN), "UseCount:%lu", UseCount);
		MDBAddValue(ServerDN, V);
		MDBWrite(ObjectName, WA_A_CONFIG, V);
	}

	/* We have to do this after storing the WA_A_CONFIG_DN on the NCP server */
	StoreConfiguration(ExtendHandle);

	MDBDestroyValueStruct(Attribute);
	MDBDestroyValueStruct(Data);
	MDBDestroyValueStruct(V);

	printf("\r%s: Install complete\n", PRODUCT_NAME);

	MDBRelease(ExtendHandle);

	//MDBShutdown();
	exit(0);
}

static BOOL
WAUninstall(int argc, char *argv[])
{
	MDBHandle			ExtendHandle;
	MDBValueStruct		*V;
	unsigned char		ObjectName[MDB_MAX_OBJECT_CHARS + 1];

	/* Can not be used with other arguments */

	if (argc < 3) {
		printf("\r%s: You must supply a user name and password\n", PRODUCT_NAME);
		exit(0);
	}

	if (!MDBInit()) {
		printf("\r%s:  Unable to contact directory service\r\n", PRODUCT_NAME);
		exit(-1);
	}

	printf("\r\n%s: Removing Schema Extensions\n", PRODUCT_NAME);

	/* Login */
	WAX500toMDB(argv[2], ObjectName, NULL);

	ExtendHandle = MDBAuthenticate("WebAdmin", ObjectName, argv[3]);
	if (!ExtendHandle) {
		printf("\r%s: Authentication failed\n", PRODUCT_NAME);
		exit(0);
	}

	V = MDBCreateValueStruct(ExtendHandle, NULL);
	if (!V) {
		printf("\r%s: Could not find server object\n", PRODUCT_NAME);

		MDBRelease(ExtendHandle);
		exit(0);
	}

	/* Remove Objects */
	if (!MDBGetServerInfo(ObjectName, NULL, NULL)) {
		printf("\r%s: Could not find server object\n", PRODUCT_NAME);

		MDBDestroyValueStruct(V);
		MDBRelease(ExtendHandle);
		exit(0);
	}

	if (MDBRead(ObjectName, WA_A_CONFIG_DN, V)) {
		unsigned long		i;
		unsigned long		UseCount = 1;	/* The object is there, so it must be at least one */
		unsigned char		WAObject[MDB_MAX_OBJECT_CHARS+1];

		/* Check UseCount.  If it is greater than one, then we simply decrement it and leave */
		HulaStrNCpy(WAObject, V->Value[0], sizeof(WAObject));
		MDBFreeValues(V);

		MDBRead(WAObject, WA_A_CONFIG, V);
		for (i = 0; i < V->Used; i++) {
			if (WAQuickNCmp(V->Value[i], "UseCount:", 9)) {
				UseCount = atol(V->Value[i] + 9);
				MDBFreeValue(i, V);
			}
		}

		if (UseCount <= 1) {
			MDBDeleteObject(WAObject, TRUE, V);
			ReportMDBError("Failed to delete configuration object");

			MDBFreeValues(V);
			MDBWrite(ObjectName, WA_A_CONFIG_DN, V);
		} else {
			unsigned char		Buffer[256];

			/* Decrement it and store the new one */
			UseCount--;
			snprintf(Buffer, sizeof (Buffer), "UseCount:%lu", UseCount);
			MDBAddValue(Buffer, V);
			MDBWrite(WAObject, WA_A_CONFIG, V);
		}
	}

	/* Remove Schema */
	if (MDBUndefineClass(WA_C_SERVER_OBJECT, V)) {
		MDBUndefineClass(WA_C_ROLE, V);					ReportMDBError("Failed to remove class: " WA_C_ROLE);

		MDBUndefineAttribute(WA_A_ROLE_NAME, V);		ReportMDBError("Failed to remove attribute: " WA_A_ROLE_NAME);
		MDBUndefineAttribute(WA_A_CONFIG, V);			ReportMDBError("Failed to remove attribute: " WA_A_CONFIG);

		MDBUndefineAttribute(WA_A_PORT, V);				ReportMDBError("Failed to remove attribute: " WA_A_PORT);
		MDBUndefineAttribute(WA_A_SSLPORT, V);			ReportMDBError("Failed to remove attribute: " WA_A_SSLPORT);
		MDBUndefineAttribute(WA_A_PATH, V);				ReportMDBError("Failed to remove attribute: " WA_A_PATH);
		MDBUndefineAttribute(WA_A_CERT_PATH, V);		ReportMDBError("Failed to remove attribute: " WA_A_CERT_PATH);
		MDBUndefineAttribute(WA_A_KEY_PATH, V);		ReportMDBError("Failed to remove attribute: " WA_A_KEY_PATH);
		MDBUndefineAttribute(WA_A_ADDRESS, V);			ReportMDBError("Failed to remove attribute: " WA_A_ADDRESS);

		MDBUndefineAttribute(WA_A_CONFIG_DN, V);	/*	ReportMDBError("Failed to remove attribute: " WA_A_CONFIG_DN); */
	}

	MDBDestroyValueStruct(V);

	printf("\r%s: Uninstall complete\n", PRODUCT_NAME);

	MDBRelease(ExtendHandle);

	//MDBShutdown();
	exit(0);
}

static BOOL
WAInstallRoles(int argc, char *argv[])
{
	MDBHandle			RolesHandle;
	MDBValueStruct		*V;
	MDBValueStruct		*Attribute;
	MDBValueStruct		*Data;
	unsigned char		ObjectName[MDB_MAX_OBJECT_CHARS+1];
	unsigned char		Tree[MDB_MAX_OBJECT_CHARS+1];
	unsigned char		ServerDN[MDB_MAX_OBJECT_CHARS+1];

	/* Can not be used with other arguments */

	if (argc < 3) {
		printf("\r%s: You must supply a user name and password\n", PRODUCT_NAME);
		exit(0);
	}

	if (!MDBInit()) {
		printf("\r%s:  Unable to contact directory service\r\n", PRODUCT_NAME);
		exit(-1);
	}

	printf("\r\n%s: Creating Role Objects\n", PRODUCT_NAME);

	/* Login */
	WAX500toMDB(argv[2], ObjectName, NULL);
	RolesHandle = MDBAuthenticate("WebAdmin", ObjectName, argv[3]);
	if (!RolesHandle) {
		printf("\r%s: Authentication failed\n", PRODUCT_NAME);
		exit(0);
	}

	V = MDBCreateValueStruct(RolesHandle, NULL);
	Attribute = MDBCreateValueStruct(RolesHandle, NULL);
	Data = MDBCreateValueStruct(RolesHandle, NULL);

	if (!V || !Attribute || !Data) {
		printf("\r%s: Authentication failed\n", PRODUCT_NAME);
		exit(0);
	}

	/* Find our object */

	Tree[0] = '\\';
	if (!MDBGetServerInfo(ServerDN, (Tree + 1), NULL)) {
		printf("\r%s: Could not find server object\n", PRODUCT_NAME);

		MDBDestroyValueStruct(Attribute);
		MDBDestroyValueStruct(Data);
		MDBDestroyValueStruct(V);
		MDBRelease(RolesHandle);
		exit(0);
	}

	if (!MDBRead(ServerDN, WA_A_CONFIG_DN, V)) {
		printf("\r%s: Could not locate a Web Administration Object\n", PRODUCT_NAME);

		MDBDestroyValueStruct(Attribute);
		MDBDestroyValueStruct(Data);
		MDBDestroyValueStruct(V);
		MDBRelease(RolesHandle);

		exit(0);
	}

	HulaStrNCpy(ObjectName, V->Value[0], sizeof(ObjectName));
	MDBFreeValues(V);

	/* Store the flag to use roles */
	MDBAdd(ObjectName, WA_A_CONFIG, "RolesInstalled", V);

	/* Create role objects */
	LoadTemplates(TemplateDir);
	if (Templates) {
		long		i;
		unsigned char		*PathEnd;

		PathEnd = ObjectName + strlen(ObjectName);

		for (i = 0; i < TemplateCount; i++) {
		    int size;
		    size = sizeof(ObjectName) - (PathEnd - ObjectName);
		    snprintf(PathEnd, size, "\\%s", Templates[i]->Name);
		    MDBAddStringAttribute(WA_A_ROLE_NAME, Templates[i]->Name, Attribute, Data)
			
			if (MDBCreateObject(ObjectName, WA_C_ROLE, Attribute, Data, V)) {
			    printf("\r%s: Created object for role:\"%s\"\n", PRODUCT_NAME, Templates[i]->Name);
			}
		    
		    ReportMDBError("Failed to create object: WA_C_ROLE");
		    
		    MDBFreeValues(Attribute);
		    MDBFreeValues(Data);
		}

		FreeTemplates();
	}

	MDBDestroyValueStruct(Attribute);
	MDBDestroyValueStruct(Data);
	MDBDestroyValueStruct(V);

	printf("\r%s: Install complete\n", PRODUCT_NAME);

	MDBRelease(RolesHandle);

	//MDBShutdown();
	exit(0);
}

static void
MakeWebAdminPath(unsigned char *Path)
{
    unsigned char    *ptr=Path;
    unsigned char    *ptr2;
    struct stat        sb;

    ptr=strchr(Path, '/');
    if (!ptr)
        ptr=strchr(Path, '\\');

    while (ptr) {
        *ptr='\0';
        if (stat(Path, &sb)!=0) {
            XplMakeDir(Path);
        }
        *ptr='/';
        ptr2=ptr;
        ptr=strchr(ptr2+1, '/');
        if (!ptr)
        ptr=strchr(ptr2+1, '\\');
    }

    if (stat(Path, &sb)!=0) {
        XplMakeDir(Path);
    }
}

XplServiceCode(WebAdmShutdownSigHandler)

int
XplServiceMain(int argc, char *argv[])
{
	int					code;
	XplThreadID			ID;	
	BOOL					changed = FALSE;
	unsigned char		*user = NULL;
	unsigned char		*pass = NULL;
    unsigned char       *ptr;

    if (XplSetEffectiveUser(GetUnpriviledgedUser()) < 0) {
        XplConsolePrintf("hulawebadmin: Could not drop to unpriviledged user '%s', exiting.\n", GetUnpriviledgedUser());
        return -1;
    }

    /* init globals */
    WebAdmin.CheckDiskForTemplateUpdates = FALSE;

	MemoryManagerOpen(WA_SERVER_OBJECT);

	if ((argc > 1 && WAQuickNCmp(argv[1], "install", 7)) || (argc > 1 && WAQuickNCmp(argv[1], "-install", 8))) {
		WAInstall(argc, argv);
	} else if ((argc > 1 && WAQuickNCmp(argv[1], "uninstall", 9)) || (argc > 1 && WAQuickNCmp(argv[1], "-uninstall", 10))) {
		WAUninstall(argc, argv);
	} else if ((argc > 1 && WAQuickNCmp(argv[1], "roles", 5)) || (argc > 1 && WAQuickNCmp(argv[1], "-roles", 6))) {
		WAInstallRoles(argc, argv);
	} else {
		long		i;

		/*
			We walk through the arguments twice.  The first time we just look for -i
			and if we don't find it we read our configuration.  We need to do this
			before looking at anything else, so that anything specified on the
			command line will overwrite the settings we get from ReadConfiguration.
		*/

		for (i = 1; i < argc; i++)  {
			if (WAQuickNCmp(argv[i], "-i", 2)) {
				/* Ignore */

				printf("\r%s ignoring stored settings\n", PRODUCT_NAME);

				IgnoreConfig = TRUE;
			} else if (WAQuickCmp(argv[i], "-v")) {
				/* Version */

				printf("\r\n\n\n%s [%s]\n",PRODUCT_NAME, PRODUCT_VERSION);
				printf("\r  Author:      Micah Gorrell [micah@novonyx.com]\n");

				exit(0);
			} else if (WAQuickCmp(argv[i], "-?") || WAQuickCmp(argv[i], "/?") || WAQuickCmp(argv[i], "-h") || WAQuickCmp(argv[i], "/h") || WAQuickCmp(argv[i], "-help") || WAQuickCmp(argv[i], "--help")) {
				/* Help */

				printf("\r\n%s usage: [arguments] [username] [password]\n", PRODUCT_NAME);
				printf("\rwebadmin [-v] [-p] [-s] [-t] [-c] [-k] [-i] [-x] [-?] [-install | -uninstall | -roles]\n\r\n");
				printf("\r-v             -Display version information\n");
				printf("\r-p:<port>      -Use <port> to accept HTTP connections (Default:89)\n");
				printf("\r-s:<port>      -Use <port> to accept secure (HTTPS) connections (Default:444)\n");
				printf("\r-t:<path>      -Use <path> to load templates and snapins\n");
				printf("\r-c:<path>      -Use <path> to load SSL certificate\n");
				printf("\r-k:<path>      -Use <path> to load SSL private key\n");
				printf("\r-i             -Ignore configuration objects\n");
				printf("\r-x             -Exit after setting configuration.\n");
				printf("\r               Arguments will be stored if a [username] and [password] are\n");
				printf("\r               given, and the install option has been used previously.\n\r\n");
				printf("\r-install <username> <password> [context]\n");
				printf("\r               -Extend the schema and create WebAdmin configuration object\n");
				printf("\r-uninstall <username> <password>\n");
				printf("\r               -Remove the schema and WebAdmin configuration object\n");
				printf("\r-roles <username> <password>\n");
				printf("\r               -Create a Role object for each exisiting template\n");

				exit(0);
			}
		}

		IPInit();

		if (!MDBInit()) {
			printf("\r%s:  Unable to contact directory service\r\n", PRODUCT_NAME);
			exit(-1);
		}

		if (!IgnoreConfig) {
			ReadConfiguration();
		}

		for (i = 1; i < argc; i++)  {
			if (WAQuickNCmp(argv[i], "-p:", 3)) {
			    long		newPort = atol(argv[i] + 3);

				/* Port */
				if (newPort == 0) {
					printf("\r%s: Invalid port\r\n", PRODUCT_NAME);
				} else if (WEBADMIN_PORT != newPort) {
					changed = TRUE;
					WEBADMIN_PORT = newPort;
				}
			} else if (WAQuickNCmp(argv[i], "-s:", 3)) {
				long		newPort = atol(argv[i] + 3);

				/* SSL Port */
				if (newPort == 0) {
					printf("\r%s: Invalid port\r\n", PRODUCT_NAME);
				} else if (WEBADMIN_PORT_SSL != newPort) {
					changed = TRUE;
					WEBADMIN_PORT_SSL = newPort;
				}
			} else if (WAQuickNCmp(argv[i], "-t:", 3)) {
				/* Templates and snapins Path */

				if (!WAQuickCmp(ModuleDir, argv[i] + 3)) {
					changed = TRUE;
				}

				HulaStrNCpy(ModuleDir, argv[i] + 3, sizeof(ModuleDir));
				snprintf(TemplateDir, sizeof(TemplateDir), "%s/webadmin", argv[i] + 3);
				snprintf(WorkDir, sizeof(WorkDir), "%s/work", argv[i] + 3);
				snprintf(LogoDir, sizeof(LogoDir), "%s/logo", argv[i] + 3);
			} else if (WAQuickNCmp(argv[i], "-c:", 3)) {
				/* Cert Path */

				if (!WAQuickCmp(ServerCertificatePath, argv[i] + 3)) {
					changed = TRUE;
					HulaStrNCpy(ServerCertificatePath, argv[i] + 3, sizeof (ServerCertificatePath));
				}
			} else if (WAQuickNCmp(argv[i], "-k:", 3)) {
				/* Private key path */

				if (!WAQuickCmp(ServerPKeyPath, argv[i] + 3)) {
					changed = TRUE;
					HulaStrNCpy(ServerPKeyPath, argv[i] + 3, sizeof(ServerPKeyPath));
				}
			} else if (WAQuickNCmp(argv[i], "-x", 2)) {
				/* Exit */

				ExitOnLoad = TRUE;
			} else if (WAQuickNCmp(argv[i], "-i", 2) || WAQuickNCmp(argv[i], "-v", 2) || WAQuickNCmp(argv[i], "-?", 2) || WAQuickNCmp(argv[i], "-h", 2)) {
				/* Nothing, just prevent warning */
			} else if (argv[i][0] == '-') {
				printf("\r%s: Invalid option: %s\n", PRODUCT_NAME, argv[i]);
			} else {
				/* We assume its either the user name or password */
				if (!user) {
					user = argv[i];
				} else {
					pass = argv[i];
				}
			}
		}
	}

	TGid = XplGetThreadGroupID();
	WADebugInit();

	XplOpenLocalSemaphore(ExitSemaphore, 1);
	XplOpenLocalSemaphore(ShutdownSemaphore, 1);
	XplOpenLocalSemaphore(ServerSemaphore, 0);
	XplOpenLocalSemaphore(CacheSemaphore, 1);
	XplOpenLocalSemaphore(SessionDBSemaphore, 1);
	XplOpenLocalSemaphore(TSessionDBSemaphore, 1);

#if 0
	signal(SIGTERM, WebAdmShutdownSigHandler);
	signal(SIGINT, WebAdmShutdownSigHandler);
#else
	XplSignalHandler(WebAdmShutdownSigHandler);
#endif

	/*	openlog("WebAdm", 0, LOG_MAIL);	*/

	/* Get our defaults */
	if (!IgnoreConfig) {
		if (user && pass) {
			MDBHandle			ConfigHandle;
			unsigned char		object[MDB_MAX_OBJECT_CHARS+1];

			WAX500toMDB(user, object, NULL);
			ConfigHandle = MDBAuthenticate("WebAdmin", object, pass);

			if (ConfigHandle) {
				StoreConfiguration(ConfigHandle);

				MDBRelease(ConfigHandle);
			} else {
				printf("\r%s: Authentication failed\n", PRODUCT_NAME);
				user = NULL;
				pass = NULL;
			}
		}
	}

	if (ExitOnLoad) {
		/* The user doesn't want us to load afterall */

		exit(0);
	}

	if (ServerSocketInit() < 0) {
	    XplConsolePrintf("hulawebadmin: Exiting\n");
	    return 1;
	}

    MakeWebAdminPath(ModuleDir);

    ptr = strrchr(ServerCertificatePath, '/');
    if (ptr) {
        *ptr = '\0';

        MakeWebAdminPath(ServerCertificatePath);

        *ptr = '/';
    }

    ptr = strrchr(ServerPKeyPath, '/');
    if (ptr) {
        *ptr = '\0';

        MakeWebAdminPath(ServerPKeyPath);

        *ptr = '/';
    }

	/*
		If they provided a user name and password on the command line, store the ip
		address, port, and paths in the directory.  If they didn't, but something
		changed, print a notice saying it will not be saved.
	*/

	/* Initialize any modules and databases we require */
	SessionDBInit();
	LoadTemplates(TemplateDir);
	if (Templates==NULL) {
		printf("\r%s: No templates found; exiting\n", PRODUCT_NAME);
		if (DefaultTitle) {
			free(DefaultTitle);
		}
		XplCloseLocalSemaphore(ExitSemaphore);
		XplCloseLocalSemaphore(CacheSemaphore);
		XplCloseLocalSemaphore(SessionDBSemaphore);
		XplCloseLocalSemaphore(TSessionDBSemaphore);
		SessionDBShutdown();
		exit(-1);
	}

	LoadModules(ModuleDir, TRUE);

	XplBeginThread(&ID, (void *)SessionMonitor, MODWEB_STACK_SPACE, NULL, code);

	if (AllowClientSSL) {
		unsigned char	*Data = NULL;

		SSL_load_error_strings();
		SSL_library_init();
		SSLContext = SSL_CTX_new(SSLv23_server_method());

		/* Set data to cipher list if required */
		if (Data) {
			SSL_CTX_set_cipher_list(SSLContext, Data);
		}

		/*
			Load the user configured certs
		*/
		if ((ServerCertificatePath[0] != '\0') && (ServerPKeyPath[0] != '\0')) {
			ServerCertificate = LoadX509Certificate(ServerCertificatePath, 0, TRUE);
			ServerPKey = LoadPrivateKey(ServerPKeyPath, 0, TRUE);

			if (!ServerCertificate || !ServerPKey) {
				AllowClientSSL = FALSE;
			} else {
				printf("\r%s: Secure mode - using external certificate\n", PRODUCT_NAME);
			}
		}

		if (!AllowClientSSL) {
			/*
				We failed to load a user specified cert, so lets try to load a generated cert.
			*/

			ServerCertificate = LoadX509Certificate(WA_GENERATED_CERT, 0, TRUE);
			ServerPKey = LoadPrivateKey(WA_GENERATED_CERT, 0, TRUE);

			if (!ServerCertificate || !ServerPKey) {
				AllowClientSSL = FALSE;
			} else {
				AllowClientSSL = TRUE;
				printf("\r%s: using internal certificate\n", PRODUCT_NAME);
			}
		}

		if (!AllowClientSSL) {
			/*
				It looks like we don't have a generated cert, so lets create one
				and load that.
			*/

			printf("\r%s: generating certificate\n", PRODUCT_NAME);
			/* Generate a self signed cert, in case there isn't one provided */
			SSL_load_error_strings();
			SSL_library_init();

			GenerateCertificate();

			ServerCertificate = LoadX509Certificate(WA_GENERATED_CERT, 0, TRUE);
			ServerPKey = LoadPrivateKey(WA_GENERATED_CERT, 0, TRUE);

			if (!ServerCertificate || !ServerPKey) {
				AllowClientSSL = FALSE;
			} else {
				AllowClientSSL = TRUE;
				printf("\r%s: using internal certificate\n", PRODUCT_NAME);
			}
		}

		if (AllowClientSSL) {
			if (SSLContext) {
				if (SSL_CTX_use_certificate(SSLContext, ServerCertificate) > 0) {
					if (SSL_CTX_use_PrivateKey(SSLContext, ServerPKey) > 0) {
						if (SSL_CTX_check_private_key(SSLContext)) {
						    if(ServerSocketSSLInit() >= 0) {
							/* Done binding to ports, drop privs permanently */
							if (XplSetRealUser(GetUnpriviledgedUser()) < 0) {
							    XplConsolePrintf("hulawebadmin: Could not drop to unpriviledged user '%s', exiting\n", GetUnpriviledgedUser());
							    return -1;
							}

							XplBeginThread(&ID, (void *)ServerStartupSSL, MODWEB_STACK_SPACE, NULL, code);
						    } else {
							AllowClientSSL = FALSE;
						    }
						} else {
							printf("\r%s: PrivateKey check failed\n", PRODUCT_NAME);
							AllowClientSSL=FALSE;
						}
					} else {
						printf("\r%s: Could not load private key\n", PRODUCT_NAME);
						AllowClientSSL=FALSE;
					}
				} else {
					printf("\r%s: Could not load public key\n", PRODUCT_NAME);
					AllowClientSSL=FALSE;
				}
			} else {
				printf("\r%s: Could not generate SSL context\n", PRODUCT_NAME);
				AllowClientSSL=FALSE;
			}
		} else {
			printf("\r%s: Could not load public key\n", PRODUCT_NAME);
		}
	}

	/* Done binding to ports, drop privs permanently */
	if (XplSetRealUser(GetUnpriviledgedUser()) < 0) {
	    XplConsolePrintf("hulawebadmin: Could not drop to unpriviledged user '%s', exiting\n", GetUnpriviledgedUser());
	    return -1;
	}

	/* Warn the user if their settings where not stored */
	if (changed && (!user || !pass)) {
		printf("\r%s: unable to store command line options.  Credentials not supplied.\n", PRODUCT_NAME);
	}

	XplStartMainThread(PRODUCT_SHORT_NAME, &ID, ServerStartup, MODWEB_STACK_SPACE, NULL, code);

    MemoryManagerClose(WA_SERVER_OBJECT);

	XplUnloadApp(XplGetThreadID());

	return(0);
}
