/****************************************************************************
 *
 * Copyright (c) 1997-2004 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#include <xpl.h>
#include <hulautil.h>
#include <openssl/md5.h>

#include <memmgr.h>

#include <nmlib.h>
#include <msgapi.h>
#include <nmap.h>

struct {
    RegistrationStates state;

    unsigned char access[NMAP_HASH_SIZE];

    MDBHandle directoryHandle;

    BOOL debug;
} NMAPLibrary = {
    REGISTRATION_LOADING, 
    { '\0' }, 
    NULL, 

    FALSE
};

int 
NMAPSendCommand(Connection *conn, const unsigned char *command, size_t length)
{
    register const char *b;
    register size_t i;
    register size_t r;

    b = command;
    r = length;
    if (r > 0) {
        if (r < conn->send.remaining) {
            memcpy(conn->send.write, b, r);

            conn->send.write += r;
            conn->send.remaining -= r;

            CONN_TCP_FLUSH(conn, conn->send.read, conn->send.write, i);
            if (i > 0) {
                conn->send.read = conn->send.write = conn->send.buffer;
                conn->send.remaining = CONN_TCP_MTU;

                return(length);
            }
        } else {
            if (conn->send.remaining) {
                memcpy(conn->send.write, b, conn->send.remaining);
                b += conn->send.remaining;
                r -= conn->send.remaining;

                conn->send.write += conn->send.remaining;
                conn->send.remaining = 0;
            }

            CONN_TCP_FLUSH(conn, conn->send.read, conn->send.write, i);
            if (i > 0) {
                conn->send.read = conn->send.write = conn->send.buffer;
                conn->send.remaining = CONN_TCP_MTU;

                CONN_TCP_FLUSH(conn, b, b + r, i);
                if (i > 0) {
                    return(length);
                }
            }
        }
    }

    return((length == 0)? 0: -1);
}

int
NMAPSendCommandF(Connection *conn, const char *format, ...)
{
    int i;
    va_list ap;
    Connection *c = conn;

    do {
        if (CONN_BUFSIZE < c->send.remaining) {
            va_start(ap, format);
            i = XplVsnprintf(c->send.write, CONN_BUFSIZE, format, ap);
            va_end(ap);

            if (i >= 0) {
                c->send.write += i;
                c->send.remaining -= i;

                CONN_TCP_FLUSH(c, c->send.read, c->send.write, i);
                if (i > 0) {
                    c->send.read = c->send.write = c->send.buffer;
                    c->send.remaining = CONN_TCP_MTU;

                    return(i);
                }
            }
        }

        CONN_TCP_FLUSH(c, c->send.read, c->send.write, i);
        if (i > 0) {
            c->send.read = c->send.write = c->send.buffer;
            c->send.remaining = CONN_TCP_MTU;

            continue;
        }

        return(-1);
    } while (TRUE);
}

int
NMAPReadAnswer(Connection *conn, unsigned char *response, size_t length, BOOL checkForResult)
{
    int len;
    int result = -1;
    size_t count;
    unsigned char *cur;

    len = ConnReadAnswer(conn, response, length);
    if (len > 0) {
        cur = response + 4;
        if (checkForResult && ((*cur == ' ') || (*cur == '-') || ((cur = strchr(response, ' ')) != NULL))) {
            *cur++ = '\0';

            result = atoi(response);

            count = len - (cur - response);
            memmove(response, cur, count);
            response[count] = '\0';
        } else {
            result = atoi(response);
        }
    }

    return(result);
}
                                                                                                                                                                            
int
NMAPReadAnswerLine(Connection *conn, unsigned char *response, size_t length, BOOL checkForResult)
{
    int len;
    int result = -1 ; 
    size_t count;
    unsigned char *cur;

    len = ConnReadLine(conn, response, length);
    if (len > 0) {
        cur = response + 4;
        if (checkForResult && ((*cur == ' ') || (*cur == '-') || ((cur = strchr(response, ' ')) != NULL))) {
            *cur++ = '\0';

	    result = atoi(response);
	    
            count = len - (cur - response);
            memmove(response, cur, count);
            response[count] = '\0';
        } else {
            result = atoi(response);
        }
    }

    return(result);
}
                                                                                                                                                                            
Connection *
NMAPConnect(unsigned char *address, struct sockaddr_in *addr)
{
    int ccode;
    Connection *conn;

    if (address || addr) {
        conn = ConnAlloc(TRUE);
    } else {
        return(NULL);
    }

    if (conn) {
        memset(&conn->socketAddress, 0, sizeof(struct sockaddr_in));

        if (address) {
            conn->socketAddress.sin_family = AF_INET;
            conn->socketAddress.sin_addr.s_addr = inet_addr(address);
            conn->socketAddress.sin_port = htons(NMAP_PORT);
        } else {
            conn->socketAddress.sin_family = addr->sin_family;
            conn->socketAddress.sin_addr.s_addr = addr->sin_addr.s_addr;
            conn->socketAddress.sin_port = addr->sin_port;
        }

        conn->socket = IPsocket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
        if (conn->socket != -1) {
            ccode = IPconnect(conn->socket, (struct sockaddr *)&(conn->socketAddress), sizeof(struct sockaddr_in));

            if (!ccode) {
                /* Set TCP non blocking io */
                ccode = 1;
                setsockopt(conn->socket, IPPROTO_TCP, 1, (unsigned char *)&ccode, sizeof(ccode));

                return(conn);
            }

            IPclose(conn->socket);
            conn->socket = -1;
        }

        ConnFree(conn);
    }

    return(NULL);
}

BOOL 
NMAPAuthenticate(Connection *conn, unsigned char *response, int length)
{
    int ccode;

    ccode = NMAPReadAnswer(conn, response, length, TRUE);
    switch (ccode) {
        case 1000: {
            return(TRUE);
        }

        case 4242: {
            int                i;
            unsigned char    *ptr;
            unsigned char    *salt;
            unsigned char    digest[16];
            unsigned char    message[33];
            MD5_CTX        context;

            ptr = strchr(response, '<');
            if (ptr) {
                salt = ++ptr;

                if ((ptr = strchr(ptr, '>')) != NULL) {
                    *ptr = '\0';
                }

                MD5_Init(&context);

                MD5_Update(&context, salt, strlen(salt));
                MD5_Update(&context, NMAPLibrary.access, NMAP_HASH_SIZE);

                MD5_Final(digest, &context);

                ConnWrite(conn, "AUTH ", 5);

                for (i = 0; i < 16; i++) {
                    sprintf(message + (i * 2), "%02x", digest[i]);
                }

                ConnWrite(conn, message, 32);
                ConnWrite(conn, "\r\n", 2);

                if ((i = ConnFlush(conn)) == 39) {
                    if ((i = NMAPReadAnswer(conn, response, length, TRUE)) == 1000) {
                        return(TRUE);
                    }
                }
            }

            /*
                Fall through to the default case statement.
            */
        }

        default: {
            break;
        }
    }

    return(FALSE);
}

void 
NMAPQuit(Connection *conn)
{
    ConnWrite(conn, "QUIT\r\n", 6);

    CONN_TCP_CLOSE(conn);

    return;
}

RegistrationStates 
NMAPRegister(unsigned char *dn, unsigned long queue, unsigned short port)
{
    int j;
    int ccode;
    unsigned long i;
    unsigned char response[CONN_BUFSIZE + 1];
    BOOL result;
    MDBValueStruct *servers = NULL;
    Connection *conn = NULL;

    if (dn) {
        NMAPLibrary.state = REGISTRATION_ALLOCATING;

        servers = MDBCreateValueStruct(NMAPLibrary.directoryHandle, MsgGetServerDN(NULL));
    } else {
        NMAPLibrary.state = REGISTRATION_FAILED;
        return(NMAPLibrary.state);
    }

    if (servers != NULL) {
        result = MsgReadIP(dn, MSGSRV_A_NMAP_SERVER, servers);
    } else {
        NMAPLibrary.state = REGISTRATION_FAILED;
        return(NMAPLibrary.state);
    }

    if (result) {
        result = FALSE;
        NMAPLibrary.state = REGISTRATION_CONNECTING;

        for (i = 0; (i < servers->Used) && (NMAPLibrary.state == REGISTRATION_CONNECTING); i++) {
            NMAPLibrary.state = REGISTRATION_CONNECTING;

            do {
                conn = NMAPConnect(servers->Value[i], NULL);
                if (conn) {
                    NMAPLibrary.state = REGISTRATION_REGISTERING;
                    break;
                }

                for (j = 0; (j < 15) && (NMAPLibrary.state == REGISTRATION_CONNECTING); j++) {
                    XplDelay(1000);
                }
            } while (NMAPLibrary.state == REGISTRATION_CONNECTING);

            if (NMAPLibrary.state == REGISTRATION_REGISTERING) {
                if (NMAPAuthenticate(conn, response, CONN_BUFSIZE)) {
                    ccode = ConnWriteF(conn, "QWAIT %lu %d %s%s%lu\r\n", queue, ntohs(port), MsgGetServerDN(NULL), dn, queue);
                    if (ccode > 0) {
                        ConnFlush(conn);

                        if (NMAPReadAnswer(conn, response, CONN_BUFSIZE, TRUE) == 1000) {
                            NMAPLibrary.state = REGISTRATION_COMPLETED;
                        }
                    }
                }

                NMAPQuit(conn);
            }

            if (conn) {
                ConnFree(conn);
            }
        }

        if (NMAPLibrary.state != REGISTRATION_COMPLETED) {
            NMAPLibrary.state = REGISTRATION_FAILED;
        }
    } else {
        NMAPLibrary.state = REGISTRATION_FAILED;
    }

    MDBDestroyValueStruct(servers);

    return(NMAPLibrary.state);
}

BOOL 
NMAPInitialize(MDBHandle directoryHandle)
{
    BOOL result = FALSE;
    MDBValueStruct *v;

    if (directoryHandle) {
        NMAPLibrary.directoryHandle = directoryHandle;

        v = MDBCreateValueStruct(NMAPLibrary.directoryHandle, NULL);
        if (v) {
            MDBRead(MSGSRV_ROOT, MSGSRV_A_ACL, v);
            if (v->Used) {
                result = HashCredential(MsgGetServerDN(NULL), v->Value[0], NMAPLibrary.access);
            }

            MDBDestroyValueStruct(v);
        }
    }

    return(result);
}
