/****************************************************************************
 *
 * Copyright (c) 2001-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

/** \file antispam.c Code for the anti-spam agent, including spamassassin.
 */

#include <config.h>

#include <xpl.h>
#include <memmgr.h>
#include <logger.h>
#include <hulautil.h>
#include <mdb.h>
#include <nmap.h>
#include <nmlib.h>
#include <msgapi.h>
#include <libical.h>
#include <streamio.h>
#include <openssl/md5.h>
#include <openssl/ssl.h>
#include <openssl/err.h>
#include <connio.h>

#include "antispam.h"

#if defined(RELEASE_BUILD)
#define ASpamClientAlloc() MemPrivatePoolGetEntry(ASpam.nmap.pool)
#else
#define ASpamClientAlloc() MemPrivatePoolGetEntryDebug(ASpam.nmap.pool, __FILE__, __LINE__)
#endif

static void SignalHandler(int sigtype);

#define QUEUE_WORK_TO_DO(c, id, r) \
        { \
            XplWaitOnLocalSemaphore(ASpam.nmap.semaphore); \
            if (XplSafeRead(ASpam.nmap.worker.idle)) { \
                (c)->queue.previous = NULL; \
                if (((c)->queue.next = ASpam.nmap.worker.head) != NULL) { \
                    (c)->queue.next->queue.previous = (c); \
                } else { \
                    ASpam.nmap.worker.tail = (c); \
                } \
                ASpam.nmap.worker.head = (c); \
                (r) = 0; \
            } else { \
                XplSafeIncrement(ASpam.nmap.worker.active); \
                XplSignalBlock(); \
                XplBeginThread(&(id), HandleConnection, 24 * 1024, XplSafeRead(ASpam.nmap.worker.active), (r)); \
                XplSignalHandler(SignalHandler); \
                if (!(r)) { \
                    (c)->queue.previous = NULL; \
                    if (((c)->queue.next = ASpam.nmap.worker.head) != NULL) { \
                        (c)->queue.next->queue.previous = (c); \
                    } else { \
                        ASpam.nmap.worker.tail = (c); \
                    } \
                    ASpam.nmap.worker.head = (c); \
                } else { \
                    XplSafeDecrement(ASpam.nmap.worker.active); \
                    (r) = -1; \
                } \
            } \
            XplSignalLocalSemaphore(ASpam.nmap.semaphore); \
        }


ASpamGlobals ASpam;

static BOOL 
ASpamClientAllocCB(void *buffer, void *data)
{
    register ASpamClient *c = (ASpamClient *)buffer;

    memset(c, 0, sizeof(ASpamClient));

    return(TRUE);
}

static void 
ASpamClientFree(ASpamClient *client)
{
    register ASpamClient *c = client;

    if (c->conn) {
        ConnClose(c->conn, 1);
        ConnFree(c->conn);
        c->conn = NULL;
    }

    MemPrivatePoolReturnEntry(c);

    return;
}

static void 
FreeClientData(ASpamClient *client)
{
    if (client && !(client->flags & ASPAM_CLIENT_FLAG_EXITING)) {
        client->flags |= ASPAM_CLIENT_FLAG_EXITING;

        if (client->conn) {
            ConnClose(client->conn, 1);
            ConnFree(client->conn);
            client->conn = NULL;
        }

        if (client->envelope) {
            MemFree(client->envelope);
        }
    }

    return;
}

static int 
CmpAddr(const void *c, const void *d)
{
    int r = 0;
    int clen;
    int dlen;
    int ci;
    int di;
    unsigned char cc;
    unsigned char dc;
    const unsigned char *candidate = *(unsigned char**)c;
    const unsigned char *domain = *(unsigned char**)d;

    if (candidate && domain) {
        ci = clen = strlen(candidate) - 1;
        di = dlen = strlen(domain) - 1;

        while ((ci >= 0) && (di >= 0) && (r == 0)) {
            cc = toupper(candidate[ci]);
            dc = toupper(domain[di]);
            if (cc == dc) {
                --ci;
                --di;
            } else if (cc < dc) {
                r = -1;
            } else {
                r = 1;
            }
        }

        if (r == 0) {
            if (clen < dlen) {
                r = -1;
            } else if ((clen > dlen) && (candidate[ci] != '.') && (candidate[ci] != '@')) {
                r = 1;
            } else {
                r = 1;
            }
        }

        return(r);
    }

    return(0);
}

static int 
MatchAddr(unsigned char *candidate, unsigned char *domain)
{
    int r = 0;
    int clen;
    int dlen;
    int ci;
    int di;
    unsigned char cc;
    unsigned char dc;

    if (candidate && domain) {
        ci = clen = strlen(candidate) - 1;
        di = dlen = strlen(domain) - 1;

        while ((ci >= 0) && (di >= 0) && (r == 0)) {
            cc = toupper(candidate[ci]);
            dc = toupper(domain[di]);
            if (cc == dc) {
                --ci;
                --di;
            } else if(cc < dc) {
                r = -1;
            } else {
                r = 1;
            }
        }

        if ((r == 0) && (clen != dlen)) {
            if (clen < dlen) {
                r = -1;
            } else if ((candidate[ci] != '.') && (candidate[ci] != '@')) {
                r = 1;
            }
        }

        return(r);
    }

    return(0);
}

/** Searches for the passed ip address in the disallowed list.
 */
static unsigned char 
*IsSpammer(unsigned char *from)
{
    int cmp;
    int start;
    int end;
    int middle = 0;
    BOOL matched = FALSE;

    start = 0;
    end = ASpam.disallow.used - 1;

    while ((end >= start) && !matched) {
        middle = (end - start) / 2 + start;
        cmp = MatchAddr(from, ASpam.disallow.list->Value[middle]);
        if (cmp == 0) {
            matched = TRUE;
        } else if (cmp < 0) {
            end = middle - 1;
        } else {
            start = middle + 1;
        }
    }

    if(matched) {
        return(ASpam.disallow.list->Value[middle]);
    }

    return(NULL);
}

/** Connects to spamd to process the message.  Depending on your settings, analyzed
 * messages are left untouched, or requeued with additional spam information in the
 * headers, or they may be deleted from the queue entirely.
 *
 * \param queueID is the queue-unique id of the message currently being processed.
 * NMAP provides this in the callback.
 *
 * \param client contains all the information for this particular session with NMAP.
 * 
 * \retval infected: returns a boolean summarizing whether or not the message was
 * determined to be spam.
 *
 * If you want verbose output from spamassassin portion of the antispam agent then 
 * #define VERBOSE_SPAMASSASSIN
 */
#define VERBOSE_SPAMASSASSIN
static BOOL
CheckSpamAssassin(ASpamClient *client, const char *queueID, unsigned long msgFlags)
{   
    int ccode;
    BOOL infected;
	BOOL success;
	double score = 0;
    Connection *conn; /* Holds the current connection to spamd. */

#ifdef VERBOSE_SPAMASSASSIN
    XplConsolePrintf("AntiSpam: (%s) CheckSpamAssassin\n", queueID);
#endif
    
    infected = FALSE;

    conn = ConnAlloc(TRUE);
    if (conn) {
        /* Copy connection information from the global set up in ReadConfiguration. */
        memcpy(&conn->socketAddress, &ASpam.spamassassin.addr, sizeof(struct sockaddr_in));
        if (ConnConnect(conn, NULL, 0, NULL) >= 0) {
            char *ptr;
            char *ptr2;
            unsigned long size;
            
            size = 0;
            
            if (((ccode = NMAPSendCommandF(client->conn, "QRETR %s MESSAGE\r\n", queueID)) != -1)
                && ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE)) != -1)
                && ccode == 2023) {
                /* Successfully retrieved the message body. Need to parse out the size. */

                ptr = strchr (client->line, ' ');
                if (ptr) {
                    *ptr = '\0';
                }
                
                size = atol(client->line);
            }

	    if (   (size != 0)
		&& ((ccode =  ConnWrite(conn, "PROCESS SPAMC/1.3\r\n", 19)        ) != -1)
		&& ((ccode = ConnWriteF(conn, "Content-length: %ld\r\n\r\n", size)) != -1)
		&& ((ccode =  ConnFlush(conn)                                     ) != -1)
		/* Write the spamc protocol header to the connection. */
		&& ((ccode = ConnReadToConn(client->conn, conn, size)             ) != -1)
		&& ((ccode = ConnFlush(conn)                                      ) != -1)
		/* Write the message body to the connection. */
		&& ((ccode = ConnReadAnswer(conn, client->line, CONN_BUFSIZE)     ) != -1)) {
		/* Read response from spamd. */
		#ifdef VERBOSE_SPAMASSASSIN
		XplConsolePrintf("AntiSpam: (%s) Answer from socket: \"%s\"\n", queueID, client->line);
		#else
		;
		#endif
	    } else {
		goto ErrConnFree;
	    }

	    if (   ((ptr  = strchr(client->line, ' ')) != NULL)
		&& ((ptr2 = strchr(ptr,          ' ')) != NULL)) {
		/* Correct response from spamd is "SPAMD/1.1 0 EX_OK", look for the '0'
		 * to confirm success. 
		 */
		++ptr;
		*ptr2++ = '\0';
		ccode = atol(ptr);
	    } else {
                goto ErrConnFree;
            }

            if (ccode == 0) {
		/* Now we need to parse the header for the spam score. The response header will be
		 * of the form:
		 *  SPAMD/1.1 0 EX_OK
		 *  Content-length: 1018             <-----------------------------+
		 *  Spam: True ; 998.8 / 5.0                                       |
		 *                                                                 |
		 * We have already read the first line, so we are on this line here. We need to
		 * grab the content length, and then the score.
		 */
		
		size = 0;
		while ((ccode = ConnReadAnswer(conn, client->line, CONN_BUFSIZE)) != -1) {
		    if (client->line[0] == '\0') {
			/* Looking for the blank line separating the spamd protocol header from
			 * the message.
			 */
			if (size == 0) {
			    /* We should have parsed the size and score before hitting the blank line
			     */
			    goto ErrConnFree;
			} else {
			    /* Correctly parsed the header.
			     */
			    break;
			}
		    } else if (strncmp(client->line, "Content-length: ", strlen("Content-length: ")) == 0) {
			/* Looking for the size information.
			 */
			ptr = client->line + strlen("Content-length: ");
			size = atol(ptr);
			if (size == 0) {
			    goto ErrConnFree;
			}
		    } else if (strncmp(client->line, "Spam: ", strlen("Spam: ")) == 0) {
			/* Looking for the spam score.
			 */
			ptr = strchr(client->line, ';');
			score = atof(++ptr);
			#ifdef VERBOSE_SPAMASSASSIN
			XplConsolePrintf("AntiSpam: (%s) Current message received a spam score of %f\n", queueID, score);
			#endif
		    }
		}
		if (score >= ASpam.spamassassin.headerThreshhold) {
		    infected = TRUE;
		    if (score < ASpam.spamassassin.dropThreshhold && size != 0) {
			/* The spam score is greater than the header threshhold,
			 * but lower than the drop threshhold.  Create a new message
			 * with spam headers added before deleting this one.
			 *
			 * We will use the envelope that we read out in the beginning
			 * to create the new message envelope.
			 */
			if (   ((ccode = NMAPSendCommandF(client->conn, "QCREA %d\r\n", Q_INCOMING)      ) != -1 )
			    && ((ccode = NMAPReadAnswer  (client->conn, client->line, CONN_BUFSIZE, TRUE)) == 1000)
			    /* Create the new queue entry. */
			    ) {
                            ;
			} else {
			    #ifdef VERBOSE_SPAMASSASSIN
			    XplConsolePrintf("AntiSpam: (%s) error while creating new queue entry.\n", queueID);
			    #endif
			    NMAPSendCommand(client->conn, "QABRT\r\n", 7);
			    NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE);
			    goto ErrConnFree;
			}
			ptr = client->envelope;
			while (*ptr) {
			    /* Parse the old envelope and send all but the FROM line with 
			     * RAW commands, send the from using QSTOR FROM. 
			     */
			    ptr2 = strchr(ptr, '\n');
			    if (ptr2 == NULL){
				/* error in the envelope structure. */
				NMAPSendCommand(client->conn, "QABRT\r\n", 7);
				goto ErrConnFree;
			    }
			    *ptr2 = '\0';
			    switch (*ptr) {
                                case QUEUE_FLAGS: {
                                    /* Add our own flag to the envelope structure.
                                     */
                                    success = (((ccode = NMAPSendCommandF(client->conn, "QSTOR RAW X%ld\r\n", (msgFlags | MSG_FLAG_SPAM_CHECKED))) != -1)
					    && ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE)) == 1000));
                                }
				case QUEUE_FROM: {
				    success = (((ccode = NMAPSendCommandF(client->conn, "QSTOR FROM %s\n", ++ptr)      ) != -1)
					    && ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE)) == 1000));
                                    break;
                                }
				default: {
				    success = (((ccode = NMAPSendCommandF(client->conn, "QSTOR RAW %s\n", ptr)         ) != -1)
					    && ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE)) == 1000));
                                }
			    }
                            *ptr2++ = '\n';
                            ptr = ptr2;
			    if (success) {
                                ;
			    } else {
				NMAPSendCommand(client->conn, "QABRT\r\n", 7);
				goto ErrConnFree;
			    }
			}
			if (  ((ccode = NMAPSendCommandF(client->conn, "QSTOR MESSAGE %ld\r\n",    size)) != -1)
			    &&((ccode =   ConnReadToConn(conn,         client->conn,               size)) != -1)
			    &&((ccode =   NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE)) == 1000)
			    &&((ccode =  NMAPSendCommand(client->conn, "QRUN\r\n",                    6)) != -1)
			    &&((ccode =   NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE)) == 1000)) {
			    /* Copy out the message data from spamassassin.
			     * Tell nmap to process the new message. 
			     */
			    #ifdef VERBOSE_SPAMASSASSIN
			    XplConsolePrintf("AntiSpam: (%s) Created new queue entry with spam headers.\n", queueID);
			    #else
			    ;
			    #endif
			} else {
			    NMAPSendCommand(client->conn, "QABRT\r\n", 7);
			    goto ErrConnFree;
			}
		    }
		    /* Finally, drop the original spam message. If more processing is 
		     * ever added after this, then error checking should be added here.
		     */
		    #ifdef VERBOSE_SPAMASSASSIN
		    XplConsolePrintf("AntiSpam: (%s) Dropped (original) spam message from the queue.\n", queueID);
		    #endif
		    ccode = NMAPSendCommandF(client->conn, "QDELE %s\r\n", queueID); /* expect ccode != -1*/
		    ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE); /* expect ccode == 1000 */
		}
	    } else {
                #ifdef VERBOSE_SPAMASSASSIN
                XplConsolePrintf("AntiSpam: (%s) Recieved a non-zero status code from spamd: %d", queueID, ccode);
                #endif
            }
        }
        ErrConnFree:
        ConnFree(conn);
    }    

    return infected;
}

/** Callback function.  Whenever a new message arrives in the queue that
 * this agent has registered itself on, NMAP calls back to this function
 * and provides the agent with the unique hash of the new message.  The 
 * connection then remains open, and NMAP goes into slave state while
 * the agent does all necessary processing.
 * \param client The connection initiated by the NMAP store.
 */
static __inline int 
ProcessConnection(ASpamClient *client)
{
    int ccode;
    int length;
    unsigned long source = 0;
    unsigned long msgFlags = 0;
    unsigned char *ptr;
    unsigned char *cur;
    unsigned char *line;
    unsigned char *blockedAddr = NULL;
    unsigned char qID[16];
    BOOL copy;
    BOOL blocked = FALSE;

    /* Read out the connection answer and determine the size of the message envelope. */
    if (((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE)) != -1) 
            && (ccode == 6020) 
            && ((ptr = strchr(client->line, ' ')) != NULL)) {
        *ptr++ = '\0';

        strcpy(qID, client->line);

        length = atoi(ptr);
        client->envelope = MemMalloc(length + 3);
    } else {
        NMAPSendCommand(client->conn, "QDONE\r\n", 7);
        NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE);
        return(-1);
    }

    /* Rename the thread to something useful, and read in the entire envelope. */
    if (client->envelope) {
        sprintf(client->line, "ASpam: %s", qID);
        XplRenameThread(XplGetThreadID(), client->line);

        ccode = ConnRead(client->conn, client->envelope, length);
    } else {
        NMAPSendCommand(client->conn, "QDONE\r\n", 7);
        NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE);
        return(-1);
    }

    /* Finish the handshake with the NMAP store. */
    if ((ccode != -1) 
            && ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE)) == 6021)) {
        client->envelope[length] = '\0';

        cur = client->envelope;
    } else {
        MemFree(client->envelope);
        client->envelope = NULL;

        NMAPSendCommand(client->conn, "QDONE\r\n", 7);
        NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE);
        return(-1);
    }

    /* NMAP store is now in slave mode. Start processing. */

    ptr = strstr(client->envelope, "\r\n"QUEUES_FLAGS);
    if ( ptr 
        && ((msgFlags = atol(ptr + 3)) != 0)
        && (!(msgFlags & MSG_FLAG_SPAM_CHECKED))) {
        ;
    } else {
        /* This message has already been processed, or there was
         * an error in parsing the envelope. Either way, we bail
         * out.
         */
        MemFree(client->envelope);
        client->envelope = NULL;

        NMAPSendCommand(client->conn, "QDONE\r\n", 7);
        NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE);

        return 0;
    }

    if (ASpam.allow.used || ASpam.disallow.used) {
        unsigned char *tmpNull = NULL;
        unsigned char tmpChar;
        while (*cur) { 
            /* Parsing the envelope to see if the sender is blacklisted.
             */
            copy = TRUE;
            line = strchr(cur, 0x0A);
            if (line) {
                if (line[-1] == 0x0D) {
                    tmpNull = line - 1;
                } else {
                    tmpNull = line;
                }

                tmpChar = *tmpNull;
                *tmpNull = '\0';
                line++;
            } else {
                line = cur + strlen(cur);
                tmpNull = NULL;
            }

            switch (cur[0]) {
                case QUEUE_FLAGS: {
                    copy = FALSE;
                    ccode = NMAPSendCommandF(client->conn, "QMOD RAW "QUEUES_FLAGS"%ld\r\n", (msgFlags | MSG_FLAG_SPAM_CHECKED));
                    break;
                }
                case QUEUE_FROM: {
                    ptr = strchr(cur + 1, ' ');
                    if (ptr) {
                        *ptr = '\0';
                        
                        blockedAddr = IsSpammer(cur + 1);
                        if (!blockedAddr) {
                            *ptr = ' ';
                        } else {
                            blocked = TRUE;
                            
                            LoggerEvent(ASpam.handle.logging, LOGGER_SUBSYSTEM_QUEUE, LOGGER_EVENT_SPAM_BLOCKED, LOG_NOTICE, 0, cur + 1, NULL, source, 0, NULL, 0);
                            
                            if (ASpam.flags & ASPAM_FLAG_RETURN_TO_SENDER) {
                                ccode = NMAPSendCommandF(client->conn, "QRTS %s %s %lu %d Mail from user or domain %s is blocked at this site.\r\n", 
                                                         cur + 1, cur + 1, (long unsigned int)(DSN_HEADER|DSN_FAILURE), DELIVER_BLOCKED, blockedAddr);
                            }
                            
                            if ((ccode != -1) && (ASpam.flags & ASPAM_FLAG_NOTIFY_POSTMASTER)) {
                                if (((ccode = NMAPSendCommandF(client->conn, "QCOPY %s\r\n", qID)) != -1) 
                                    && ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE)) != -1) 
                                    && ((ccode = NMAPSendCommand(client->conn, "QSTOR FLAGS %ld\r\n", (msgFlags | MSG_FLAG_SPAM_CHECKED))) != -1) 
                                    && ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE)) != -1) 
                                    && ((ccode = NMAPSendCommand(client->conn, "QSTOR FROM - -\r\n", 16)) != -1) 
                                    && ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE)) != -1) 
                                    && ((ccode = NMAPSendCommandF(client->conn, "QSTOR LOCAL %s %s 0\r\n", ASpam.postmaster, ASpam.postmaster)) != -1) 
                                    && ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE)) != -1) 
                                    && ((ccode = NMAPSendCommand(client->conn, "QRUN\r\n", 6)) != -1)) {
                                    ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE);
                                } else {
                                    NMAPSendCommand(client->conn, "QABRT\r\n", 7);
                                }
                            }
                            
                            *ptr = ' ';
                        }
                    }
                    
                    break;
                }
                    
                case QUEUE_RECIP_REMOTE:
                case QUEUE_RECIP_LOCAL:
                case QUEUE_RECIP_MBOX_LOCAL: {
                    if (blocked) {
                        copy = FALSE;
                    }
                    
                    break;
                }
                    
                case QUEUE_ADDRESS: {
                    source = atol(cur + 1);
                    break;
                }
            }
        
            if (copy && (ccode != -1)) {
                ccode = NMAPSendCommandF(client->conn, "QMOD RAW %s\r\n", cur);
            }
            cur = line;
            if (tmpNull) {
                /* Restore the local copy of the envelope. */
                *tmpNull = tmpChar;
            }
        }
    }

    if (!blocked && ASpam.spamassassin.enabled) { 
        blocked = CheckSpamAssassin(client, qID, msgFlags);
    }

    if ((ccode != -1) 
            && ((ccode = NMAPSendCommand(client->conn, "QDONE\r\n", 7)) != -1)) {
        ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE);
    }

    if (client->envelope) {
        MemFree(client->envelope);
        client->envelope = NULL;
    }

    return(0);
}

static void 
HandleConnection(void *param)
{
    int ccode;
    long threadNumber = (long)param;
    time_t sleep = time(NULL);
    time_t wokeup;
    ASpamClient *client;

    if ((client = ASpamClientAlloc()) == NULL) {
        XplConsolePrintf("hulaantispam: New worker failed to startup; out of memory.\r\n");

        NMAPSendCommand(client->conn, "QDONE\r\n", 7);

        XplSafeDecrement(ASpam.nmap.worker.active);

        return;
    }

    do {
        XplRenameThread(XplGetThreadID(), "ASpam Worker");

        XplSafeIncrement(ASpam.nmap.worker.idle);

        XplWaitOnLocalSemaphore(ASpam.nmap.worker.todo);

        XplSafeDecrement(ASpam.nmap.worker.idle);

        wokeup = time(NULL);

        XplWaitOnLocalSemaphore(ASpam.nmap.semaphore);

        client->conn = ASpam.nmap.worker.tail;
        if (client->conn) {
            ASpam.nmap.worker.tail = client->conn->queue.previous;
            if (ASpam.nmap.worker.tail) {
                ASpam.nmap.worker.tail->queue.next = NULL;
            } else {
                ASpam.nmap.worker.head = NULL;
            }
        }

        XplSignalLocalSemaphore(ASpam.nmap.semaphore);

        if (client->conn) {
            if (ConnNegotiate(client->conn, ASpam.nmap.ssl.context)) {
                ccode = ProcessConnection(client);
            } else {
                NMAPSendCommand(client->conn, "QDONE\r\n", 7);
                NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE);
            }
        }

        if (client->conn) {
            ConnFlush(client->conn);
        }

        FreeClientData(client);

        /* Live or die? */
        if (threadNumber == XplSafeRead(ASpam.nmap.worker.active)) {
            if ((wokeup - sleep) > ASpam.nmap.sleepTime) {
                break;
            }
        }

        sleep = time(NULL);

        ASpamClientAllocCB(client, NULL);
    } while (ASpam.state == ASPAM_STATE_RUNNING);

    FreeClientData(client);

    ASpamClientFree(client);

    XplSafeDecrement(ASpam.nmap.worker.active);

    XplExitThread(TSR_THREAD, 0);

    return;
}

static void 
AntiSpamServer(void *ignored)
{
    int i;
    int ccode;
    XplThreadID id;
    Connection *conn;

    XplSafeIncrement(ASpam.server.active);

    XplRenameThread(XplGetThreadID(), "Anti-Spam Server");

    while (ASpam.state < ASPAM_STATE_STOPPING) {
        if (ConnAccept(ASpam.nmap.conn, &conn) != -1) {
            if (ASpam.state < ASPAM_STATE_STOPPING) {
                conn->ssl.enable = FALSE;

                QUEUE_WORK_TO_DO(conn, id, ccode);
                if (!ccode) {
                    XplSignalLocalSemaphore(ASpam.nmap.worker.todo);

                    continue;
                }
            }

            ConnWrite(conn, "QDONE\r\n", 7);
            ConnClose(conn, 0);

            ConnFree(conn);
            conn = NULL;

            continue;
        }

        switch (errno) {
            case ECONNABORTED:
#ifdef EPROTO
            case EPROTO: 
#endif
            case EINTR: {
                if (ASpam.state < ASPAM_STATE_STOPPING) {
                    LoggerEvent(ASpam.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_ACCEPT_FAILURE, LOG_ERROR, 0, "Server", NULL, errno, 0, NULL, 0);
                }

                continue;
            }

            default: {
                if (ASpam.state < ASPAM_STATE_STOPPING) {
                    XplConsolePrintf("hulaantispam: Exiting after an accept() failure; error %d\r\n", errno);

                    LoggerEvent(ASpam.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_ACCEPT_FAILURE, LOG_ERROR, 0, "Server", NULL, errno, 0, NULL, 0);

                    ASpam.state = ASPAM_STATE_STOPPING;
                }

                break;
            }
        }

        break;
    }

    /* Shutting down */
    ASpam.state = ASPAM_STATE_UNLOADING;

    XplConsolePrintf("hulaantispam: Shutting down.\r\n");

    id = XplSetThreadGroupID(ASpam.id.group);

    if (ASpam.nmap.conn) {
        ConnClose(ASpam.nmap.conn, 1);
        ASpam.nmap.conn = NULL;
    }

    if (ASpam.nmap.ssl.enable) {
        ASpam.nmap.ssl.enable = FALSE;

        if (ASpam.nmap.ssl.conn) {
            ConnClose(ASpam.nmap.ssl.conn, 1);
            ASpam.nmap.ssl.conn = NULL;
        }

        if (ASpam.nmap.ssl.context) {
            ConnSSLContextFree(ASpam.nmap.ssl.context);
            ASpam.nmap.ssl.context = NULL;
        }
    }

    ConnCloseAll(1);

    if (ManagementState() == MANAGEMENT_RUNNING) {
        ManagementShutdown();
    }

    for (i = 0; (XplSafeRead(ASpam.server.active) > 1) && (i < 60); i++) {
        XplDelay(1000);
    }

    for (i = 0; (ManagementState() != MANAGEMENT_STOPPED) && (i < 60); i++) {
        XplDelay(1000);
    }

    XplConsolePrintf("hulaantispam: Shutting down %d queue threads\r\n", XplSafeRead(ASpam.nmap.worker.active));

    XplWaitOnLocalSemaphore(ASpam.nmap.semaphore);

    ccode = XplSafeRead(ASpam.nmap.worker.idle);
    while (ccode--) {
        XplSignalLocalSemaphore(ASpam.nmap.worker.todo);
    }

    XplSignalLocalSemaphore(ASpam.nmap.semaphore);

    for (i = 0; XplSafeRead(ASpam.nmap.worker.active) && (i < 60); i++) {
        XplDelay(1000);
    }

    if (XplSafeRead(ASpam.server.active) > 1) {
        XplConsolePrintf("hulaantispam: %d server threads outstanding; attempting forceful unload.\r\n", XplSafeRead(ASpam.server.active) - 1);
    }

    if (XplSafeRead(ASpam.nmap.worker.active)) {
        XplConsolePrintf("hulaantispam: %d threads outstanding; attempting forceful unload.\r\n", XplSafeRead(ASpam.nmap.worker.active));
    }

    LoggerClose(ASpam.handle.logging);
    ASpam.handle.logging = NULL;

    /* shutdown the scanning engine */

    XplCloseLocalSemaphore(ASpam.nmap.worker.todo);
    XplCloseLocalSemaphore(ASpam.nmap.semaphore);

    if (ASpam.allow.list) {
        MDBDestroyValueStruct(ASpam.allow.list);
        ASpam.allow.list = NULL;
        ASpam.allow.used = 0;
    }

    if (ASpam.disallow.list) {
        MDBDestroyValueStruct(ASpam.disallow.list);
        ASpam.disallow.list = NULL;
        ASpam.disallow.used = 0;
    }

    MsgShutdown();

    ConnShutdown();

    MemPrivatePoolFree(ASpam.nmap.pool);

    MemoryManagerClose(MSGSRV_AGENT_ANTISPAM);

    XplConsolePrintf("hulaantispam: Shutdown complete\r\n");

    XplSignalLocalSemaphore(ASpam.sem.main);
    XplWaitOnLocalSemaphore(ASpam.sem.shutdown);

    XplCloseLocalSemaphore(ASpam.sem.shutdown);
    XplCloseLocalSemaphore(ASpam.sem.main);

    XplSetThreadGroupID(id);

    return;
}

/** Reads configuration details for the agent into global structs. 
 * There are two essential configuration options that need to be set in
 * the Novonyx:Configuration attribute of the Antispam Agent object.
 * 
 * \c HeaderThreshhold: \c <num>  is the minimum score before adding spam
 * headers to the message. Messages with a spam score greater than or
 * equal to this threshhold will have spam headers added to them. These
 * headers can then be used in the rule server.
 *
 * \c DropThreshhold: \c <num> is the minimum score before dropping the 
 * message from the queue altogether. Messages with a score greater than
 * or equal to this threshhold will not be delivered.
 *
 * \c SpamdPort: \c <num> is the port that spamd is listening on.
 *
 * \c SpamdHost: \c <host> is the host name of the computer that spamd is
 * running on.
 */
static BOOL 
ReadConfiguration(void)
{
    unsigned long used;
    unsigned char *ptr;
    int ccode;
    MDBValueStruct *config; /* Temporarily holds options from the database */

    ASpam.allow.list    = MDBCreateValueStruct(ASpam.handle.directory, MsgGetServerDN(NULL));
    ASpam.disallow.list = MDBCreateValueStruct(ASpam.handle.directory, MsgGetServerDN(NULL));
    config              = MDBCreateValueStruct(ASpam.handle.directory, MsgGetServerDN(NULL));
    if (ASpam.allow.list && ASpam.disallow.list && config) {
        ASpam.allow.used = 0;
        ASpam.disallow.used = 0;
    } else {
        if (config) {
            MDBDestroyValueStruct(config);
        }

        return(FALSE);
    }

    if (MDBRead(MSGSRV_AGENT_ANTISPAM, MSGSRV_A_ACTION, config) > 0) {
        LoggerEvent(ASpam.handle.logging, LOGGER_SUBSYSTEM_CONFIGURATION, LOGGER_EVENT_CONFIGURATION_STRING, LOG_INFO, 0, "MSGSRV_A_ACTION", config->Value[0], 0, 0, NULL, 0);

        if (atoi(config->Value[0]) & 0x01) {
            ASpam.flags |= ASPAM_FLAG_RETURN_TO_SENDER;
        }

        if (atoi(config->Value[0]) & 0x02) {
            ASpam.flags |= ASPAM_FLAG_NOTIFY_POSTMASTER;
        }

        MDBFreeValues(config);
    }
    /* Set up default spam assassin options. */
    ASpam.spamassassin.spamdHost = gethostbyname("localhost");
    ASpam.spamassassin.spamdPort        = 783;
    ASpam.spamassassin.headerThreshhold = -9999;
    ASpam.spamassassin.dropThreshhold   =  9999;
    ASpam.spamassassin.enabled          = FALSE;

    if (MDBRead(MSGSRV_AGENT_ANTISPAM, MSGSRV_A_CONFIGURATION, config) > 0) {
        /* Find out which queue to register with.  Otherwise remain Q_INCOMING */
        for (used = 0; used < config->Used; used++) {
            if ((ccode = XplStrNCaseCmp(config->Value[used], "Queue: ", 7)) == 0) {
                ASpam.nmap.queue = atol(config->Value[used] + 7);
                continue;
            }
            if ((ccode = XplStrNCaseCmp(config->Value[used], "HeaderThreshhold: ", 18)) == 0) {
                ASpam.spamassassin.headerThreshhold = atof(config->Value[used] + 18);
                ASpam.spamassassin.enabled = TRUE;
                continue;
            }
            if ((ccode = XplStrNCaseCmp(config->Value[used], "DropThreshhold: ", 16)) == 0) {
                ASpam.spamassassin.dropThreshhold = atof(config->Value[used] + 16);
                ASpam.spamassassin.enabled = TRUE;
                continue;
            }
            if ((ccode = XplStrNCaseCmp(config->Value[used], "SpamdPort: ", 11)) == 0) {
                ASpam.spamassassin.spamdPort = atoi(config->Value[used] + 11);
                ASpam.spamassassin.enabled = TRUE;
                continue;
            }
            if ((ccode = XplStrNCaseCmp(config->Value[used], "SpamdHost: ", 11)) == 0) {
                /* Currently only works on localhost. */
                ASpam.spamassassin.spamdHost = gethostbyname(config->Value[used] + 11);
                ASpam.spamassassin.enabled = TRUE;
                continue;
            }
            if ((ccode = XplStrNCaseCmp(config->Value[used], "SpamassassinEnabled: ", 21)) == 0) {
                if (atoi(config->Value[used] + 21)) {
                    ASpam.spamassassin.enabled = TRUE;
                } else {
                    ASpam.spamassassin.enabled = FALSE;
                }
                continue;
            }
        }
        
        MDBFreeValues(config);
    }

    if (MDBRead(MSGSRV_SELECTED_CONTEXT, MSGSRV_A_POSTMASTER, config) > 0) {
        ptr = strrchr(config->Value[0], '\\');
        if (ptr) {
            ptr++;
        } else {
            ptr = config->Value[0];
        }

        strcpy(ASpam.postmaster, ptr);

        MDBFreeValues(config);
    } else {
        ASpam.flags &= ~ASPAM_FLAG_NOTIFY_POSTMASTER;
    }

    if ((ASpam.disallow.used = MDBRead(MSGSRV_AGENT_ANTISPAM, MSGSRV_A_EMAIL_ADDRESS, ASpam.disallow.list)) > 0) {
        qsort(ASpam.disallow.list->Value, ASpam.disallow.used, sizeof(unsigned char*), CmpAddr);

        MDBFreeValues(config);
    }

    MDBSetValueStructContext(NULL, config);
    if (MDBRead(MSGSRV_ROOT, MSGSRV_A_ACL, config)>0) { 
        HashCredential(MsgGetServerDN(NULL), config->Value[0], ASpam.nmap.hash);
    }

    MDBDestroyValueStruct(config);

    if (ASpam.spamassassin.spamdHost) {
        ASpam.spamassassin.addr.sin_family = AF_INET;
        memcpy(&ASpam.spamassassin.addr.sin_addr.s_addr, ASpam.spamassassin.spamdHost->h_addr_list[0], sizeof(ASpam.spamassassin.addr.sin_addr.s_addr));
    }
    ASpam.spamassassin.addr.sin_port = htons(ASpam.spamassassin.spamdPort);


    return(TRUE);
}

#if defined(NETWARE) || defined(LIBC) || defined(WIN32)
static int 
_NonAppCheckUnload(void)
{
    static BOOL    checked = FALSE;
    XplThreadID    id;

    if (!checked) {
        checked = TRUE;
        ASpam.state = ASPAM_STATE_UNLOADING;

        XplWaitOnLocalSemaphore(ASpam.sem.shutdown);

        id = XplSetThreadGroupID(ASpam.id.group);
        ConnClose(ASpam.nmap.conn, 1);
        XplSetThreadGroupID(id);

        XplWaitOnLocalSemaphore(ASpam.sem.main);
    }

    return(0);
}
#endif

static void 
SignalHandler(int sigtype)
{
    switch(sigtype) {
        case SIGHUP: {
            if (ASpam.state < ASPAM_STATE_UNLOADING) {
                ASpam.state = ASPAM_STATE_UNLOADING;
            }

            break;
        }

        case SIGINT:
        case SIGTERM: {
            if (ASpam.state == ASPAM_STATE_STOPPING) {
                XplUnloadApp(getpid());
            } else if (ASpam.state < ASPAM_STATE_STOPPING) {
                ASpam.state = ASPAM_STATE_STOPPING;
            }

            break;
        }

        default: {
            break;
        }
    }

    return;
}

static int 
QueueSocketInit(void)
{
    ASpam.nmap.conn = ConnAlloc(FALSE);
    if (ASpam.nmap.conn) {
        memset(&(ASpam.nmap.conn->socketAddress), 0, sizeof(ASpam.nmap.conn->socketAddress));

        ASpam.nmap.conn->socketAddress.sin_family = AF_INET;
        ASpam.nmap.conn->socketAddress.sin_addr.s_addr = MsgGetAgentBindIPAddress();

        /* Get root privs back for the bind.  It's ok if this fails -
        * the user might not need to be root to bind to the port */
        XplSetEffectiveUserId(0);

        ASpam.nmap.conn->socket = ConnServerSocket(ASpam.nmap.conn, 2048);
        if (XplSetEffectiveUser(MsgGetUnprivilegedUser()) < 0) {
            XplConsolePrintf("hulaantispam: Could not drop to unprivileged user '%s'\r\n", MsgGetUnprivilegedUser());
            ConnFree(ASpam.nmap.conn);
            ASpam.nmap.conn = NULL;
            return(-1);
        }

        if (ASpam.nmap.conn->socket == -1) {
            XplConsolePrintf("hulaantispam: Could not bind to dynamic port\r\n");
            ConnFree(ASpam.nmap.conn);
            ASpam.nmap.conn = NULL;
            return(-1);
        }

        if (NMAPRegister(MSGSRV_AGENT_ANTISPAM, ASpam.nmap.queue, ASpam.nmap.conn->socketAddress.sin_port) != REGISTRATION_COMPLETED) {
            XplConsolePrintf("hulaantispam: Could not register with hulanmap\r\n");
            ConnFree(ASpam.nmap.conn);
            ASpam.nmap.conn = NULL;
            return(-1);
        }
    } else {
        XplConsolePrintf("hulaantispam: Could not allocate connection.\r\n");
        return(-1);
    }

    return(0);
}

XplServiceCode(SignalHandler)

int
XplServiceMain(int argc, char *argv[])
{
    int                ccode;
    XplThreadID        id;

    if (XplSetEffectiveUser(MsgGetUnprivilegedUser()) < 0) {
        XplConsolePrintf("hulaantispam: Could not drop to unprivileged user '%s', exiting.\n", MsgGetUnprivilegedUser());
        return(1);
    }

    XplSignalHandler(SignalHandler);

    ASpam.id.main = XplGetThreadID();
    ASpam.id.group = XplGetThreadGroupID();

    ASpam.state = ASPAM_STATE_INITIALIZING;
    ASpam.flags = 0;

    ASpam.nmap.conn = NULL;
    ASpam.nmap.queue = Q_INCOMING;
    ASpam.nmap.pool = NULL;
    ASpam.nmap.sleepTime = (5 * 60);
    ASpam.nmap.ssl.conn = NULL;
    ASpam.nmap.ssl.enable = FALSE;
    ASpam.nmap.ssl.context = NULL;
    ASpam.nmap.ssl.config.options = 0;
    ASpam.spamassassin.enabled = FALSE;

    ASpam.handle.directory = NULL;
    ASpam.handle.logging = NULL;

    strcpy(ASpam.nmap.address, "127.0.0.1");

    XplSafeWrite(ASpam.server.active, 0);

    XplSafeWrite(ASpam.nmap.worker.idle, 0);
    XplSafeWrite(ASpam.nmap.worker.active, 0);
    XplSafeWrite(ASpam.nmap.worker.maximum, 100000);

    if (MemoryManagerOpen(MSGSRV_AGENT_ANTISPAM) == TRUE) {
        ASpam.nmap.pool = MemPrivatePoolAlloc("AntiSpam Connections", sizeof(ASpamClient), 0, 3072, TRUE, FALSE, ASpamClientAllocCB, NULL, NULL);
        if (ASpam.nmap.pool != NULL) {
            XplOpenLocalSemaphore(ASpam.sem.main, 0);
            XplOpenLocalSemaphore(ASpam.sem.shutdown, 1);
            XplOpenLocalSemaphore(ASpam.nmap.semaphore, 1);
            XplOpenLocalSemaphore(ASpam.nmap.worker.todo, 1);
        } else {
            MemoryManagerClose(MSGSRV_AGENT_ANTISPAM);

            XplConsolePrintf("hulaantispam: Unable to create connection pool; shutting down.\r\n");
            return(-1);
        }
    } else {
        XplConsolePrintf("hulaantispam: Unable to initialize memory manager; shutting down.\r\n");
        return(-1);
    }

    ConnStartup(CONNECTION_TIMEOUT, TRUE);

    MDBInit();
    ASpam.handle.directory = (MDBHandle)MsgInit();
    if (ASpam.handle.directory == NULL) {
        XplBell();
        XplConsolePrintf("hulaantispam: Invalid directory credentials; exiting!\r\n");
        XplBell();

        MemoryManagerClose(MSGSRV_AGENT_ANTISPAM);

        return(-1);
    }

    NMAPInitialize(ASpam.handle.directory);

    SetCurrentNameSpace(NWOS2_NAME_SPACE);
    SetTargetNameSpace(NWOS2_NAME_SPACE);

    ASpam.handle.logging = LoggerOpen("hulaantispam");
    if (!ASpam.handle.logging) {
            XplConsolePrintf("hulaantispam: Unable to initialize logging; disabled.\r\n");
    }

    ReadConfiguration();

    if (ASpam.spamassassin.enabled || ASpam.allow.used || ASpam.disallow.used) {
        if (QueueSocketInit() < 0) {
            XplConsolePrintf("hulaantispam: Exiting.\r\n");

            MemoryManagerClose(MSGSRV_AGENT_ANTISPAM);

            return -1;
        }

        /* initialize scanning engine here */

        if (XplSetRealUser(MsgGetUnprivilegedUser()) < 0) {
            XplConsolePrintf("hulaantispam: Could not drop to unprivileged user '%s', exiting.\r\n", MsgGetUnprivilegedUser());

            MemoryManagerClose(MSGSRV_AGENT_ANTISPAM);

            return 1;
        }

        ASpam.nmap.ssl.enable = FALSE;
        ASpam.nmap.ssl.config.method = SSLv23_client_method;
        ASpam.nmap.ssl.config.options = SSL_OP_DONT_INSERT_EMPTY_FRAGMENTS;
        ASpam.nmap.ssl.config.mode = SSL_MODE_AUTO_RETRY;
        ASpam.nmap.ssl.config.cipherList = NULL;
        ASpam.nmap.ssl.config.certificate.type = SSL_FILETYPE_PEM;
        ASpam.nmap.ssl.config.certificate.file = MsgGetTLSCertPath(NULL);
        ASpam.nmap.ssl.config.key.type = SSL_FILETYPE_PEM;
        ASpam.nmap.ssl.config.key.file = MsgGetTLSKeyPath(NULL);

        ASpam.nmap.ssl.context = ConnSSLContextAlloc(&(ASpam.nmap.ssl.config));
        if (ASpam.nmap.ssl.context) {
            ASpam.nmap.ssl.enable = TRUE;
        }

        NMAPSetEncryption(ASpam.nmap.ssl.context);

        if ((ManagementInit(MSGSRV_AGENT_ANTISPAM, ASpam.handle.directory)) 
                && (ManagementSetVariables(GetASpamManagementVariables(), GetASpamManagementVariablesCount())) 
                && (ManagementSetCommands(GetASpamManagementCommands(), GetASpamManagementCommandsCount()))) {
            XplBeginThread(&id, ManagementServer, DMC_MANAGEMENT_STACKSIZE, NULL, ccode);
        }


        if (ccode) {
            XplConsolePrintf("hulaantispam: Unable to startup the management interface.\r\n");
        }

        ASpam.state = ASPAM_STATE_RUNNING;
    } else {
        XplConsolePrintf("hulaantispam: spamassassin not enabled and no hosts allowed or disallowed; unloading\r\n");

        /* Linux Advanced Server 2.1 has a race condition if we exit too early. */
        XplDelay(30 * 1000);

        ASpam.state = ASPAM_STATE_STOPPING;
    }

    XplStartMainThread(PRODUCT_SHORT_NAME, &id, AntiSpamServer, 8192, NULL, ccode);
    
    XplUnloadApp(XplGetThreadID());
    return(0);
}
