/****************************************************************************
 *
 * Copyright (c) 2001-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>

#include <xpl.h>
#include <memmgr.h>
#include <logger.h>
#include <hulautil.h>
#include <mdb.h>
#include <nmap.h>
#include <nmlib.h>
#include <msgapi.h>
#include <libical.h>
#include <streamio.h>
#include <openssl/md5.h>
#include <openssl/ssl.h>
#include <openssl/err.h>
#include <connio.h>

#include "generic.h"

#if defined(RELEASE_BUILD)
#define GAgentClientAlloc() MemPrivatePoolGetEntry(GAgent.nmap.pool)
#else
#define GAgentClientAlloc() MemPrivatePoolGetEntryDebug(GAgent.nmap.pool, __FILE__, __LINE__)
#endif

static void SignalHandler(int sigtype);

#define QUEUE_WORK_TO_DO(c, id, r) \
        { \
            XplWaitOnLocalSemaphore(GAgent.nmap.semaphore); \
            if (XplSafeRead(GAgent.nmap.worker.idle)) { \
                (c)->queue.previous = NULL; \
                if (((c)->queue.next = GAgent.nmap.worker.head) != NULL) { \
                    (c)->queue.next->queue.previous = (c); \
                } else { \
                    GAgent.nmap.worker.tail = (c); \
                } \
                GAgent.nmap.worker.head = (c); \
                (r) = 0; \
            } else { \
                XplSafeIncrement(GAgent.nmap.worker.active); \
                XplSignalBlock(); \
                XplBeginThread(&(id), HandleConnection, 24 * 1024, XplSafeRead(GAgent.nmap.worker.active), (r)); \
                XplSignalHandler(SignalHandler); \
                if (!(r)) { \
                    (c)->queue.previous = NULL; \
                    if (((c)->queue.next = GAgent.nmap.worker.head) != NULL) { \
                        (c)->queue.next->queue.previous = (c); \
                    } else { \
                        GAgent.nmap.worker.tail = (c); \
                    } \
                    GAgent.nmap.worker.head = (c); \
                } else { \
                    XplSafeDecrement(GAgent.nmap.worker.active); \
                    (r) = -1; \
                } \
            } \
            XplSignalLocalSemaphore(GAgent.nmap.semaphore); \
        }


GAgentGlobals GAgent;

static BOOL 
GAgentClientAllocCB(void *buffer, void *data)
{
    register GAgentClient *c = (GAgentClient *)buffer;

    memset(c, 0, sizeof(GAgentClient));

    return(TRUE);
}

static void 
GAgentClientFree(GAgentClient *client)
{
    register GAgentClient *c = client;

    if (c->conn) {
        ConnClose(c->conn, 1);
        ConnFree(c->conn);
        c->conn = NULL;
    }

    MemPrivatePoolReturnEntry(c);

    return;
}

static void 
FreeClientData(GAgentClient *client)
{
    if (client && !(client->flags & GAGENT_CLIENT_FLAG_EXITING)) {
        client->flags |= GAGENT_CLIENT_FLAG_EXITING;

        if (client->conn) {
            ConnClose(client->conn, 1);
            ConnFree(client->conn);
            client->conn = NULL;
        }

        if (client->envelope) {
            MemFree(client->envelope);
        }
    }

    return;
}

static __inline int 
ProcessConnection(GAgentClient *client)
{
    int ccode;
    int length;
    unsigned char *ptr;
    unsigned char qID[16];

    if (((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE)) != -1) 
            && (ccode == 6020) 
            && ((ptr = strchr(client->line, ' ')) != NULL)) {
        *ptr++ = '\0';

        strcpy(qID, client->line);

        length = atoi(ptr);
        client->envelope = MemMalloc(length + 3);
    } else {
        NMAPSendCommand(client->conn, "QDONE\r\n", 7);
        NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE);
        return(-1);
    }

    if (client->envelope) {
        sprintf(client->line, "GAgent: %s", qID);
        XplRenameThread(XplGetThreadID(), client->line);

        ccode = ConnRead(client->conn, client->envelope, length);
    } else {
        NMAPSendCommand(client->conn, "QDONE\r\n", 7);
        NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE);
        return(-1);
    }

    if ((ccode != -1) 
            && ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE)) == 6021)) {
        client->envelope[length] = '\0';

        /* NMAP queue agent specific code */
        ;
    } else {
        MemFree(client->envelope);
        client->envelope = NULL;

        NMAPSendCommand(client->conn, "QDONE\r\n", 7);
        NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE);
        return(-1);
    }

    if ((ccode != -1) 
            && ((ccode = NMAPSendCommand(client->conn, "QDONE\r\n", 7)) != -1)) {
        ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE);
    }

    MemFree(client->envelope);
    client->envelope = NULL;

    return(0);
}

static void 
HandleConnection(void *param)
{
    int ccode;
    long threadNumber = (long)param;
    time_t sleep = time(NULL);
    time_t wokeup;
    GAgentClient *client;

    if ((client = GAgentClientAlloc()) == NULL) {
        XplConsolePrintf("hulaagent: New worker failed to startup; out of memory.\r\n");

        NMAPSendCommand(client->conn, "QDONE\r\n", 7);

        XplSafeDecrement(GAgent.nmap.worker.active);

        return;
    }

    do {
        XplRenameThread(XplGetThreadID(), "GAgent Worker");

        XplSafeIncrement(GAgent.nmap.worker.idle);

        XplWaitOnLocalSemaphore(GAgent.nmap.worker.todo);

        XplSafeDecrement(GAgent.nmap.worker.idle);

        wokeup = time(NULL);

        XplWaitOnLocalSemaphore(GAgent.nmap.semaphore);

        client->conn = GAgent.nmap.worker.tail;
        if (client->conn) {
            GAgent.nmap.worker.tail = client->conn->queue.previous;
            if (GAgent.nmap.worker.tail) {
                GAgent.nmap.worker.tail->queue.next = NULL;
            } else {
                GAgent.nmap.worker.head = NULL;
            }
        }

        XplSignalLocalSemaphore(GAgent.nmap.semaphore);

        if (client->conn) {
            if (ConnNegotiate(client->conn, GAgent.nmap.ssl.context)) {
                ccode = ProcessConnection(client);
            } else {
                NMAPSendCommand(client->conn, "QDONE\r\n", 7);
                NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE);
            }
        }

        if (client->conn) {
            ConnFlush(client->conn);
        }

        FreeClientData(client);

        /* Live or die? */
        if (threadNumber == XplSafeRead(GAgent.nmap.worker.active)) {
            if ((wokeup - sleep) > GAgent.nmap.sleepTime) {
                break;
            }
        }

        sleep = time(NULL);

        GAgentClientAllocCB(client, NULL);
    } while (GAgent.state == GAGENT_STATE_RUNNING);

    FreeClientData(client);

    GAgentClientFree(client);

    XplSafeDecrement(GAgent.nmap.worker.active);

    XplExitThread(TSR_THREAD, 0);

    return;
}

static void 
GenericAgentServer(void *ignored)
{
    int i;
    int ccode;
    XplThreadID id;
    Connection *conn;

    XplSafeIncrement(GAgent.server.active);

    XplRenameThread(XplGetThreadID(), "Generic Agent Server");

    GAgent.state = GAGENT_STATE_RUNNING;

    while (GAgent.state < GAGENT_STATE_STOPPING) {
        if (ConnAccept(GAgent.nmap.conn, &conn) != -1) {
            if (GAgent.state < GAGENT_STATE_STOPPING) {
                conn->ssl.enable = FALSE;

                QUEUE_WORK_TO_DO(conn, id, ccode);
                if (!ccode) {
                    XplSignalLocalSemaphore(GAgent.nmap.worker.todo);

                    continue;
                }
            }

            ConnWrite(conn, "QDONE\r\n", 7);
            ConnClose(conn, 0);

            ConnFree(conn);
            conn = NULL;

            continue;
        }

        switch (errno) {
            case ECONNABORTED:
#ifdef EPROTO
            case EPROTO: 
#endif
            case EINTR: {
                if (GAgent.state < GAGENT_STATE_STOPPING) {
                    LoggerEvent(GAgent.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_ACCEPT_FAILURE, LOG_ERROR, 0, "Server", NULL, errno, 0, NULL, 0);
                }

                continue;
            }

            default: {
                if (GAgent.state < GAGENT_STATE_STOPPING) {
                    XplConsolePrintf("hulaagent: Exiting after an accept() failure; error %d\r\n", errno);

                    LoggerEvent(GAgent.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_ACCEPT_FAILURE, LOG_ERROR, 0, "Server", NULL, errno, 0, NULL, 0);

                    GAgent.state = GAGENT_STATE_STOPPING;
                }

                break;
            }
        }

        break;
    }

    /* Shutting down */
    GAgent.state = GAGENT_STATE_UNLOADING;

    XplConsolePrintf("hulaagent: Shutting down.\r\n");

    id = XplSetThreadGroupID(GAgent.id.group);

    if (GAgent.nmap.conn) {
        ConnClose(GAgent.nmap.conn, 1);
        GAgent.nmap.conn = NULL;
    }

    if (GAgent.nmap.ssl.enable) {
        GAgent.nmap.ssl.enable = FALSE;

        if (GAgent.nmap.ssl.conn) {
            ConnClose(GAgent.nmap.ssl.conn, 1);
            GAgent.nmap.ssl.conn = NULL;
        }

        if (GAgent.nmap.ssl.context) {
            ConnSSLContextFree(GAgent.nmap.ssl.context);
            GAgent.nmap.ssl.context = NULL;
        }
    }

    ConnCloseAll(1);

    if (ManagementState() == MANAGEMENT_RUNNING) {
        ManagementShutdown();
    }

    for (i = 0; (XplSafeRead(GAgent.server.active) > 1) && (i < 60); i++) {
        XplDelay(1000);
    }

    for (i = 0; (ManagementState() != MANAGEMENT_STOPPED) && (i < 60); i++) {
        XplDelay(1000);
    }

    XplConsolePrintf("hulaagent: Shutting down %d queue threads\r\n", XplSafeRead(GAgent.nmap.worker.active));

    XplWaitOnLocalSemaphore(GAgent.nmap.semaphore);

    ccode = XplSafeRead(GAgent.nmap.worker.idle);
    while (ccode--) {
        XplSignalLocalSemaphore(GAgent.nmap.worker.todo);
    }

    XplSignalLocalSemaphore(GAgent.nmap.semaphore);

    for (i = 0; XplSafeRead(GAgent.nmap.worker.active) && (i < 60); i++) {
        XplDelay(1000);
    }

    if (XplSafeRead(GAgent.server.active) > 1) {
        XplConsolePrintf("hulaagent: %d server threads outstanding; attempting forceful unload.\r\n", XplSafeRead(GAgent.server.active) - 1);
    }

    if (XplSafeRead(GAgent.nmap.worker.active)) {
        XplConsolePrintf("hulaagent: %d threads outstanding; attempting forceful unload.\r\n", XplSafeRead(GAgent.nmap.worker.active));
    }

    LoggerClose(GAgent.handle.logging);
    GAgent.handle.logging = NULL;

    /* shutdown the scanning engine */

    XplCloseLocalSemaphore(GAgent.nmap.worker.todo);
    XplCloseLocalSemaphore(GAgent.nmap.semaphore);

    MsgShutdown();

    ConnShutdown();

    MemPrivatePoolFree(GAgent.nmap.pool);

    MemoryManagerClose("GenericAgent"); /* Use the appropriate MSGSRV_AGENT_NAME macro. */

    XplConsolePrintf("hulaagent: Shutdown complete\r\n");

    XplSignalLocalSemaphore(GAgent.sem.main);
    XplWaitOnLocalSemaphore(GAgent.sem.shutdown);

    XplCloseLocalSemaphore(GAgent.sem.shutdown);
    XplCloseLocalSemaphore(GAgent.sem.main);

    XplSetThreadGroupID(id);

    return;
}

static BOOL 
ReadConfiguration(void)
{
    MDBValueStruct *config;

    config = MDBCreateValueStruct(GAgent.handle.directory, MsgGetServerDN(NULL));
    if (config) {
        if (MDBRead(".", MSGSRV_A_OFFICIAL_NAME, config)) {
            strcpy(GAgent.officialName, config->Value[0]);

            MDBFreeValues(config);
        } else {
            GAgent.officialName[0] = '\0';
        }

        MDBSetValueStructContext(NULL, config);
        if (MDBRead(MSGSRV_ROOT, MSGSRV_A_ACL, config) > 0) { 
            HashCredential(MsgGetServerDN(NULL), config->Value[0], GAgent.nmap.hash);
        }
    } else {
        return(FALSE);
    }

    MDBDestroyValueStruct(config);

    return(TRUE);
}

#if defined(NETWARE) || defined(LIBC)
static int 
_NonAppCheckUnload(void)
{
    int            s;
    static BOOL    checked = FALSE;
    XplThreadID    id;

    if (!checked) {
        checked = TRUE;
        GAgent.state = GAGENT_STATE_UNLOADING;

        XplWaitOnLocalSemaphore(GAgent.sem.shutdown);

        id = XplSetThreadGroupID(GAgent.id.group);
        ConnClose(GAgent.nmap.conn, 1);
        XplSetThreadGroupID(id);

        XplWaitOnLocalSemaphore(GAgent.sem.main);
    }

    return(0);
}
#endif

static void 
SignalHandler(int sigtype)
{
    switch(sigtype) {
        case SIGHUP: {
            if (GAgent.state < GAGENT_STATE_UNLOADING) {
                GAgent.state = GAGENT_STATE_UNLOADING;
            }

            break;
        }

        case SIGINT:
        case SIGTERM: {
            if (GAgent.state == GAGENT_STATE_STOPPING) {
                kill(getpid(), 9);
            } else if (GAgent.state < GAGENT_STATE_STOPPING) {
                GAgent.state = GAGENT_STATE_STOPPING;
            }

            break;
        }

        default: {
            break;
        }
    }

    return;
}

static int 
QueueSocketInit(void)
{
    GAgent.nmap.conn = ConnAlloc(FALSE);
    if (GAgent.nmap.conn) {
        memset(&(GAgent.nmap.conn->socketAddress), 0, sizeof(GAgent.nmap.conn->socketAddress));

        GAgent.nmap.conn->socketAddress.sin_family = AF_INET;
        GAgent.nmap.conn->socketAddress.sin_addr.s_addr = MsgGetAgentBindIPAddress();

        /* Get root privs back for the bind.  It's ok if this fails -
        * the user might not need to be root to bind to the port */
        XplSetEffectiveUserId(0);

        GAgent.nmap.conn->socket = ConnServerSocket(GAgent.nmap.conn, 2048);
        if (XplSetEffectiveUser(MsgGetUnprivilegedUser()) < 0) {
            XplConsolePrintf("hulaagent: Could not drop to unprivileged user '%s'\r\n", MsgGetUnprivilegedUser());
            ConnFree(GAgent.nmap.conn);
            GAgent.nmap.conn = NULL;
            return(-1);
        }

        if (GAgent.nmap.conn->socket == -1) {
            XplConsolePrintf("hulaagent: Could not bind to dynamic port\r\n");
            ConnFree(GAgent.nmap.conn);
            GAgent.nmap.conn = NULL;
            return(-1);
        }

        /* Use the appropriate MSGSRV_AGENT_NAME macro in place of "GenericAgent" for NMAPRegister argument one. */
        if (NMAPRegister("GenericAgent", GAgent.nmap.queue, GAgent.nmap.conn->socketAddress.sin_port) != REGISTRATION_COMPLETED) {
            XplConsolePrintf("hulaagent: Could not register with hulanmap\r\n");
            ConnFree(GAgent.nmap.conn);
            GAgent.nmap.conn = NULL;
            return(-1);
        }
    } else {
        XplConsolePrintf("hulaagent: Could not allocate connection.\r\n");
        return(-1);
    }

    return(0);
}

XplServiceCode(SignalHandler)

int
XplServiceMain(int argc, char *argv[])
{
    int                ccode;
    XplThreadID        id;

    if (XplSetEffectiveUser(MsgGetUnprivilegedUser()) < 0) {
        XplConsolePrintf("hulaagent: Could not drop to unprivileged user '%s', exiting.\n", MsgGetUnprivilegedUser());
        return(1);
    }

    XplSignalHandler(SignalHandler);

    GAgent.id.main = XplGetThreadID();
    GAgent.id.group = XplGetThreadGroupID();

    GAgent.state = GAGENT_STATE_INITIALIZING;
    GAgent.flags = 0;

    GAgent.nmap.conn = NULL;
    GAgent.nmap.queue = Q_INCOMING;
    GAgent.nmap.pool = NULL;
    GAgent.nmap.sleepTime = (5 * 60);
    GAgent.nmap.ssl.conn = NULL;
    GAgent.nmap.ssl.enable = FALSE;
    GAgent.nmap.ssl.context = NULL;
    GAgent.nmap.ssl.config.options = 0;

    GAgent.handle.directory = NULL;
    GAgent.handle.logging = NULL;

    strcpy(GAgent.nmap.address, "127.0.0.1");

    XplSafeWrite(GAgent.server.active, 0);

    XplSafeWrite(GAgent.nmap.worker.idle, 0);
    XplSafeWrite(GAgent.nmap.worker.active, 0);
    XplSafeWrite(GAgent.nmap.worker.maximum, 100000);

    /* Use the appropriate MSGSRV_AGENT_NAME macro in place of "GenericAgent" for MemoryManagerOpen argument one. */
    if (MemoryManagerOpen("GenericAgent") == TRUE) {
        GAgent.nmap.pool = MemPrivatePoolAlloc("GenericAgent Connections", sizeof(GAgentClient), 0, 3072, TRUE, FALSE, GAgentClientAllocCB, NULL, NULL);
        if (GAgent.nmap.pool != NULL) {
            XplOpenLocalSemaphore(GAgent.sem.main, 0);
            XplOpenLocalSemaphore(GAgent.sem.shutdown, 1);
            XplOpenLocalSemaphore(GAgent.nmap.semaphore, 1);
            XplOpenLocalSemaphore(GAgent.nmap.worker.todo, 1);
        } else {
            /* Use the appropriate MSGSRV_AGENT_NAME macro in place of "GenericAgent" for MemoryManagerClose argument one. */
            MemoryManagerClose("GenericAgent");

            XplConsolePrintf("hulaagent: Unable to create connection pool; shutting down.\r\n");
            return(-1);
        }
    } else {
        XplConsolePrintf("hulaagent: Unable to initialize memory manager; shutting down.\r\n");
        return(-1);
    }

    ConnStartup(CONNECTION_TIMEOUT, TRUE);

    MDBInit();
    GAgent.handle.directory = (MDBHandle)MsgInit();
    if (GAgent.handle.directory == NULL) {
        XplBell();
        XplConsolePrintf("hulaagent: Invalid directory credentials; exiting!\r\n");
        XplBell();

        /* Use the appropriate MSGSRV_AGENT_NAME macro in place of "GenericAgent" for MemoryManagerClose argument one. */
        MemoryManagerClose("GenericAgent");

        return(-1);
    }

    NMAPInitialize(GAgent.handle.directory);

    SetCurrentNameSpace(NWOS2_NAME_SPACE);
    SetTargetNameSpace(NWOS2_NAME_SPACE);

	GAgent.handle.logging = LoggerOpen("hulageneric");
	if (!GAgent.handle.logging) {
		XplConsolePrintf("hulaagent: Unable to initialize logging; disabled.\r\n");
	}

    ReadConfiguration();

    if (QueueSocketInit() < 0) {
        XplConsolePrintf("hulaagent: Exiting.\r\n");

        /* Use the appropriate MSGSRV_AGENT_NAME macro in place of "GenericAgent" for MemoryManagerClose argument one. */
        MemoryManagerClose("GenericAgent");

        return -1;
    }

    /* initialize scanning engine here */


    if (XplSetRealUser(MsgGetUnprivilegedUser()) < 0) {
        XplConsolePrintf("hulaagent: Could not drop to unprivileged user '%s', exiting.\r\n", MsgGetUnprivilegedUser());

        /* Use the appropriate MSGSRV_AGENT_NAME macro in place of "GenericAgent" for MemoryManagerClose argument one. */
        MemoryManagerClose("GenericAgent");

        return 1;
    }

    GAgent.nmap.ssl.enable = FALSE;
    GAgent.nmap.ssl.config.method = SSLv23_client_method;
    GAgent.nmap.ssl.config.options = SSL_OP_DONT_INSERT_EMPTY_FRAGMENTS;
    GAgent.nmap.ssl.config.mode = SSL_MODE_AUTO_RETRY;
    GAgent.nmap.ssl.config.cipherList = NULL;
    GAgent.nmap.ssl.config.certificate.type = SSL_FILETYPE_PEM;
    GAgent.nmap.ssl.config.certificate.file = MsgGetTLSCertPath(NULL);
    GAgent.nmap.ssl.config.key.type = SSL_FILETYPE_PEM;
    GAgent.nmap.ssl.config.key.file = MsgGetTLSKeyPath(NULL);

    GAgent.nmap.ssl.context = ConnSSLContextAlloc(&(GAgent.nmap.ssl.config));
    if (GAgent.nmap.ssl.context) {
        GAgent.nmap.ssl.enable = TRUE;
    }

    NMAPSetEncryption(GAgent.nmap.ssl.context);

    /* Use the appropriate MSGSRV_AGENT_NAME macro in place of "GenericAgent" for MemoryManagerClose argument one. */
    if ((ManagementInit("GenericAgent", GAgent.handle.directory)) 
            && (ManagementSetVariables(GetGAgentManagementVariables(), GetGAgentManagementVariablesCount())) 
            && (ManagementSetCommands(GetGAgentManagementCommands(), GetGAgentManagementCommandsCount()))) {
        XplBeginThread(&id, ManagementServer, DMC_MANAGEMENT_STACKSIZE, NULL, ccode);
    }


    if (ccode) {
        XplConsolePrintf("hulaagent: Unable to startup the management interface.\r\n");
    }
    
    XplStartMainThread(PRODUCT_SHORT_NAME, &id, GenericAgentServer, 8192, NULL, ccode);
    
    XplUnloadApp(XplGetThreadID());
    return(0);
}
