/****************************************************************************
 *
 * Copyright (c) 1997-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#include <xpl.h>
#include <mdb.h>

#include "generic.h"

#define PRODUCT_NAME "Hula NMAP Agent"
#define PRODUCT_DESCRIPTION "Provides mail-store access and message queueing services via the NMAP protocol."
#define PRODUCT_VERSION "$Revision: 1.11 $"

static BOOL ReadGAgentVariable(unsigned int variable, unsigned char *data, size_t *length);
static BOOL WriteGAgentVariable(unsigned int variable, unsigned char *data, size_t length);

static BOOL GAgentDMCCommandHelp(unsigned char *arguments, unsigned char **response, BOOL *closeConnection);
static BOOL GAgentShutdown(unsigned char *arguments, unsigned char **response, BOOL *closeConnection);
static BOOL SendGenericAgentStatistics(unsigned char *arguments, unsigned char **response, BOOL *closeConnection);

ManagementCommands GAgentManagementCommands[] = {
    { DMCMC_HELP, GAgentDMCCommandHelp }, 
    { DMCMC_SHUTDOWN, GAgentShutdown },
    { DMCMC_STATS, SendGenericAgentStatistics }, 
    { DMCMC_DUMP_MEMORY_USAGE, ManagementMemoryStats }, 
};

ManagementVariables GAgentManagementVariables[] = {
    { DMCMV_REVISIONS, DMCMV_REVISIONS_HELP, ReadGAgentVariable, NULL }, 
    { DMCMV_CONNECTION_COUNT, DMCMV_CONNECTION_COUNT_HELP, ReadGAgentVariable, NULL }, 
    { DMCMV_IDLE_CONNECTION_COUNT, DMCMV_IDLE_CONNECTION_COUNT_HELP, ReadGAgentVariable, NULL }, 
    { DMCMV_SERVER_THREAD_COUNT, DMCMV_SERVER_THREAD_COUNT_HELP, ReadGAgentVariable, NULL }, 
    { DMCMV_NMAP_QUEUE, DMCMV_NMAP_QUEUE_HELP, ReadGAgentVariable, NULL }, 
    { DMCMV_NMAP_ADDRESS, DMCMV_NMAP_ADDRESS_HELP, ReadGAgentVariable, NULL }, 
    { DMCMV_OFFICIAL_NAME, DMCMV_OFFICIAL_NAME_HELP, ReadGAgentVariable, NULL }, 
    { DMCMV_VERSION, DMCMV_VERSION_HELP, ReadGAgentVariable, NULL }, 
};

ManagementVariables *
GetGAgentManagementVariables(void)
{
    return(GAgentManagementVariables);
}

int 
GetGAgentManagementVariablesCount(void)
{
    return(sizeof(GAgentManagementVariables) / sizeof(ManagementVariables));
}

ManagementCommands *
GetGAgentManagementCommands(void)
{
    return(GAgentManagementCommands);
}

int 
GetGAgentManagementCommandsCount(void)
{
    return(sizeof(GAgentManagementCommands) / sizeof(ManagementCommands));
}

static BOOL 
GAgentShutdown(unsigned char *arguments, unsigned char **response, BOOL *closeConnection)
{
    XplThreadID id;

    if (response) {
        if (!arguments) {
            if (GAgent.nmap.conn) {
                *response = MemStrdup("Shutting down.\r\n");
                if (*response) {
                    id = XplSetThreadGroupID(GAgent.id.group);

                    GAgent.state = GAGENT_STATE_UNLOADING;

                    if (GAgent.nmap.conn) {
                        ConnClose(GAgent.nmap.conn, 1);
                        GAgent.nmap.conn = NULL;
                    }

                    if (closeConnection) {
                        *closeConnection = TRUE;
                    }

                    XplSetThreadGroupID(id);
                }
            } else if (GAgent.state != GAGENT_STATE_RUNNING) {
                *response = MemStrdup("Shutdown in progress.\r\n");
            }

            if (*response) {
                return(TRUE);
            }

            return(FALSE);
        }

        *response = MemStrdup("arguments not allowed.\r\n");
        return(TRUE);
    }

    return(FALSE);
}

static BOOL 
GAgentDMCCommandHelp(unsigned char *arguments, unsigned char **response, BOOL *closeConnection)
{
    BOOL responded = FALSE;

    if (response) {
        if (arguments) {
            switch(toupper(arguments[0])) {
                case 'M': {
                    if (XplStrCaseCmp(arguments, DMCMC_DUMP_MEMORY_USAGE) == 0) {
                        if ((*response = MemStrdup(DMCMC_DUMP_MEMORY_USAGE_HELP)) != NULL) {
                            responded = TRUE;
                        }

                        break;
                    }
                }

                case 'S': {
                    if (XplStrCaseCmp(arguments, DMCMC_SHUTDOWN) == 0) {
                        if ((*response = MemStrdup(DMCMC_SHUTDOWN_HELP)) != NULL) {
                            responded = TRUE;
                        }

                        break;
                    } else if (XplStrCaseCmp(arguments, DMCMC_STATS) == 0) {
                        if ((*response = MemStrdup(DMCMC_STATS_HELP)) != NULL) {
                            responded = TRUE;
                        }

                        break;
                    }
                }

                default: {
                    break;
                }
            }
        } else if ((*response = MemStrdup(DMCMC_HELP_HELP)) != NULL) {
            responded = TRUE;
        }

        if (responded || ((*response = MemStrdup(DMCMC_UNKOWN_COMMAND)) != NULL)) {
            return(TRUE);
        }
    }

    return(FALSE);
}

static BOOL 
SendGenericAgentStatistics(unsigned char *arguments, unsigned char **response, BOOL *closeConnection)
{
    MemStatistics poolStats;

    if (!arguments && response) {
        memset(&poolStats, 0, sizeof(MemStatistics));

        *response = MemMalloc(sizeof(PRODUCT_NAME)
                    + sizeof(PRODUCT_SHORT_NAME)
                    + 172);

        MemPrivatePoolStatistics(GAgent.nmap.pool, &poolStats);

        if (*response) {
            sprintf(*response, "%s (%s: v%d.%d.%d)\r\n%lu:%lu:%lu:%lu:%d:%d:%d\r\n", 
                    PRODUCT_NAME, 
                    PRODUCT_SHORT_NAME, 
                    PRODUCT_MAJOR_VERSION, 
                    PRODUCT_MINOR_VERSION, 
                    PRODUCT_LETTER_VERSION, 
                    poolStats.totalAlloc.count, 
                    poolStats.totalAlloc.size, 
                    poolStats.pitches, 
                    poolStats.strikes, 
                    XplSafeRead(GAgent.server.active), 
                    XplSafeRead(GAgent.nmap.worker.active), 
                    XplSafeRead(GAgent.nmap.worker.idle));

            return(TRUE);
        }

        if ((*response = MemStrdup("Out of memory.\r\n")) != NULL) {
            return(TRUE);
        }
    } else if ((arguments) && ((*response = MemStrdup("arguments not allowed.\r\n")) != NULL)) {
        return(TRUE);
    }

    return(FALSE);
}

static BOOL 
ReadGAgentVariable(unsigned int variable, unsigned char *data, size_t *length)
{
    size_t count;
    unsigned char *ptr;

    switch (variable) {
        case 0: {
            unsigned char version[30];

            PVCSRevisionToVersion(PRODUCT_VERSION, version);
            count = strlen(version) + 12;

            if (data && (*length > count)) {
                ptr = data;

                PVCSRevisionToVersion(PRODUCT_VERSION, version);
                ptr += sprintf(ptr, "generic.c: %s\r\n", version);

                *length = ptr - data;
            } else {
                *length = count;
            }

            break;
        }

        case 1: {
            if (data && (*length > 12)) {
                sprintf(data, "%010d\r\n", XplSafeRead(GAgent.nmap.worker.active));
            }

            *length = 12;
            break;
        }

        case 2: {
            if (data && (*length > 12)) {
                sprintf(data, "%010d\r\n", XplSafeRead(GAgent.nmap.worker.idle));
            }

            *length = 12;
            break;
        }

        case 3: {
            if (data && (*length > 12)) {
                sprintf(data, "%010d\r\n", XplSafeRead(GAgent.server.active));
            }

            *length = 12;
            break;
        }

       case 4: {
            if (data && (*length > 12)) {
                sprintf(data, "%010lu\r\n", GAgent.nmap.queue);
            }

            *length = 12;
            break;
        }

        case 5: {
            count = strlen(GAgent.nmap.address) + 2;
            if (data && (*length > count)) {
                sprintf(data, "%s\r\n", GAgent.nmap.address);
            }

            *length = count;
            break;
        }

        case 6: {
            count = strlen(GAgent.officialName) + 2;
            if (data && (*length > count)) {
                sprintf(data, "%s\r\n", GAgent.officialName);
            }

            *length = count;
            break;
        }

        case 7: {
            DMC_REPORT_PRODUCT_VERSION(data, *length);
            break;
        }
    }

    return(TRUE);
}

static BOOL 
WriteGAgentVariable(unsigned int variable, unsigned char *data, size_t length)
{
    BOOL result = TRUE;

    if (!data || !length) {
        return(FALSE);
    }

    switch (variable) {
		case 0:
		case 1:
		case 2:
		case 3:
		case 4:
		case 5:
		case 6:
		case 7:
        default: {
            result = FALSE;
            break;
        }
    }

    return(result);
}
