/****************************************************************************
 *
 * Copyright (c) 2001-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <xpl.h>
#include <openssl/ssl.h>
#include <openssl/err.h>
#include <stdio.h>

/* Port to listen */
#define IMAP4_PORT  143
#define IMAP4_PORT_SSL 993

/* return values for command functions */
#define CLIENT_CONTINUE 0
#define CLIENT_TERMINATE -1

/* supported IMAP commands */

/* IMAP client commands - any state */
#define IMAP_COMMAND_CAPABILITY "CAPABILITY"
#define IMAP_COMMAND_NOOP "NOOP"
#define IMAP_COMMAND_LOGOUT "LOGOUT"

/* IMAP client commands - not authenticated state */
#define IMAP_COMMAND_STARTTLS "STARTTLS"
#define IMAP_COMMAND_AUTHENTICATE "AUTHENTICATE"
#define IMAP_COMMAND_LOGIN "LOGIN"

/* IMAP client commands - authenticated state */
#define IMAP_COMMAND_SELECT "SELECT"
#define IMAP_COMMAND_EXAMINE "EXAMINE"
#define IMAP_COMMAND_CREATE "CREATE"
#define IMAP_COMMAND_DELETE "DELETE"
#define IMAP_COMMAND_RENAME "RENAME"
#define IMAP_COMMAND_SUBSCRIBE "SUBSCRIBE"
#define IMAP_COMMAND_UNSUBSCRIBE "UNSUBSCRIBE"
#define IMAP_COMMAND_LIST "LIST"
#define IMAP_COMMAND_LSUB "LSUB"
#define IMAP_COMMAND_STATUS "STATUS"
#define IMAP_COMMAND_APPEND "APPEND"

/* IMAP client commands - selected state */
#define IMAP_COMMAND_CHECK "CHECK"
#define IMAP_COMMAND_CLOSE "CLOSE"
#define IMAP_COMMAND_EXPUNGE "EXPUNGE"
#define IMAP_COMMAND_SEARCH "SEARCH"
#define IMAP_COMMAND_UID_SEARCH "UID SEARCH"
#define IMAP_COMMAND_FETCH "FETCH"
#define IMAP_COMMAND_UID_FETCH "UID FETCH"
#define IMAP_COMMAND_STORE "STORE"
#define IMAP_COMMAND_UID_STORE "UID STORE"
#define IMAP_COMMAND_COPY "COPY"
#define IMAP_COMMAND_UID_COPY "UID COPY"

/* IMAP ACL commands rfc2086 */
#define IMAP_COMMAND_SETACL "SETACL"
#define IMAP_COMMAND_DELETEACL "DELETEACL"
#define IMAP_COMMAND_GETACL "GETACL"
#define IMAP_COMMAND_LISTRIGHTS "LISTRIGHTS"
#define IMAP_COMMAND_MYRIGHTS "MYRIGHTS"

/* IMAP Quota commands rfc2087 */
#define IMAP_COMMAND_SETQUOTA "SETQUOTA"
#define IMAP_COMMAND_GETQUOTA "GETQUOTA"
#define IMAP_COMMAND_GETQUOTAROOT "GETQUOTAROOT"

/* IMAP Namespace command rfc2342 */
#define IMAP_COMMAND_NAMESPACE "NAMESPACE"

/* non-standard commands */
#define IMAP_COMMAND_PROXYAUTH "PROXYAUTH"  /* depricated - should use SASL */
#define IMAP_COMMAND_QUIT "QUIT"

#define IMAP_HELP_NOT_DEFINED "%s - HELP not defined.\r\n"


/* Internal stuff */
#define CR     0x0d
#define LF     0x0a


#define STATE_FRESH  (1<<0)
#define STATE_AUTH  (1<<1)
#define STATE_SELECTED (1<<2)
#define STATE_EXITING (1<<3)


#define F_UID (1<<0)
#define F_FLAGS (1<<1)
#define F_ENVELOPE (1<<2)
#define F_XSENDER (1<<3)
#define F_INTERNALDATE (1<<4)
#define F_BODYSTRUCTURE (1<<5)
#define F_RFC822_TEXT (1<<6)
#define F_RFC822_SIZE (1<<7)
#define F_RFC822_HEADER (1<<8)
#define F_RFC822_BOTH (1<<9)
#define F_BODY (1<<10)
#define F_BODY_HEADER (1<<11)
#define F_BODY_HEADER_PARTIAL (1<<12)
#define F_BODY_TEXT (1<<13)
#define F_BODY_TEXT_PARTIAL (1<<14)
#define F_BODY_BOTH (1<<15)
#define F_BODY_BOTH_PARTIAL (1<<16)
#define F_BODY_MIME (1<<17)
#define F_BODY_MIME_PARTIAL (1<<18)
#define F_BODY_HEADER_FIELDS (1<<19)
#define F_BODY_HEADER_FIELDS_NOT (1<<20)
#define F_BODY_PART (1<<21)
#define F_BODY_SEEN (1<<22) 

#define F_PARSE_ERROR (1<<30)
#define F_SYSTEM_ERROR (1<<31)

#define F_NEED_HEADER (F_BODY_HEADER | F_BODY_HEADER_PARTIAL | F_RFC822_HEADER | F_RFC822_BOTH | F_ENVELOPE | F_BODY_HEADER_FIELDS | F_BODY_HEADER_FIELDS_NOT | F_BODY_MIME | F_BODY_MIME_PARTIAL)
#define F_NEED_MIME (F_BODY_PART)

#define F_ERROR (F_PARSE_ERROR | F_SYSTEM_ERROR)

#define BUFSIZE    1023
#define MTU     (1480*3)

/* Fetch BodyPart request types  */
#define REQUEST_TYPE_MIME    1
#define REQUEST_TYPE_TEXT    2
#define REQUEST_TYPE_HEADER   3
#define REQUEST_TYPE_FIELDS   4
#define REQUEST_TYPE_FIELDS_NOT  5

/* NMAP MIME Line Types */
#define REGULAR_PART  0
#define RFC822_PART   1
#define RFC822_PART_END 0
#define MULTI_PART   0
#define MULTI_PART_END  2

/* For UIDtoNUM       */
#define UID_EQUAL   0
#define UID_LESSEQUAL  1
#define UID_EQUALGREATER 2

/* Minutes * 60 (1 second granularity) */
#define CONNECTION_TIMEOUT (40*60)

typedef struct iofuncs {
    GenericWriteFunc write;
    GenericWriteFunc writeSSL;
    GenericReadFunc read;
    GenericReadFunc readSSL;
}IOFuncs;



#define DoNMAPWrite(client, Buf, Len, Flags)    FuncTbl.write((void *)client->NMAPs, Buf, Len)
#define DoNMAPRead(client, Buf, Len, Flags)    FuncTbl.read((void *)client->NMAPs, Buf, Len, CONNECTION_TIMEOUT)
#define DoClientWrite(client, Buf, Len, Flags)   client->ClientWrite(client->ClientSktCtx, Buf, Len)
#define DoClientRead(client, Buf, Len, Flags)   client->ClientRead(client->ClientSktCtx, Buf, Len, CONNECTION_TIMEOUT)

#define UID_HIGHEST     ULONG_MAX

/* RFC822 Address parsing */
#define BROKEN_ADDRESS_HOST  ".Syntax-Error."
#define BROKEN_ADDRESS_MAILBOX "UnexpectedDataAfterAddress"
extern const unsigned char *WSpecials;

#define I2C_ESC    0x1b /* ESCape */
#define I2C_MULTI    0x24 /* multi-byte character set */
#define I2CS_94_JIS_BUGROM 0x48 /* 4/8 some software does this */
#define I2CS_94_JIS_KANA 0x49 /* 4/9 JIS X 0201-1976 right half */
#define I2CS_94_JIS_ROMAN 0x4a /* 4/a JIS X 0201-1976 left half */
#define I2CS_94x94_JIS_OLD 0x40 /* 4/0 JIS X 0208-1978 */
#define I2CS_94x94_JIS_NEW 0x42 /* 4/2 JIS X 0208-1983 */
#define I2C_G0_94    0x28 /* G0 94-character set */
#define I2CS_94    0x000 /* 94 character set */
#define I2CS_94_ASCII  0x42 /* 4/2 ISO 646 USA (ASCII) */
#define I2CS_ASCII   (I2CS_94 | I2CS_94_ASCII)



/* For the commands working on a range */
#define WORK_SINGLE  1
#define WORK_LIST  2
#define WORK_RANGE  3

#define IMAP_STACK_SIZE  (1024*65)

#define ChopNL(String)  { unsigned char *pTr; pTr=strchr((String), 0x0a); if (pTr) *pTr='\0'; pTr=strrchr((String), 0x0d); if (pTr) *pTr='\0'; }
#define SetPtrToValue(Ptr,String) Ptr=String;while(isspace(*Ptr)) Ptr++;if ((*Ptr=='=') || (*Ptr==':')) Ptr++; while(isspace(*Ptr)) Ptr++;

#define SendUnseenFlags(CLIENT, BUFFER, LEN) \
    if (CLIENT->UnseenFlags) { \
        unsigned long i; \
        unsigned long MsgFlags; \
        for (i = 0; i < CLIENT->UnseenFlags[1]; i++) { \
            MsgFlags = CLIENT->UnseenFlags[2 + (i * 2) + 1]; \
            LEN = sprintf(BUFFER, "* %lu FETCH (FLAGS (%s%s%s%s%s%s", NMAPtoNUM(CLIENT, CLIENT->UnseenFlags[2 + (i * 2)]),  \
                     (MsgFlags & MSG_STATE_ANSWERED) ? "\\Answered " : "", \
                     (MsgFlags & MSG_STATE_DELETED) ? "\\Deleted " : "", \
                     (MsgFlags & MSG_STATE_RECENT) ? "\\Recent " : "", \
                     (MsgFlags & MSG_STATE_READ) ? "\\Seen " : "", \
                     (MsgFlags & MSG_STATE_DRAFT) ? "\\Draft " : "", \
                     (MsgFlags & MSG_STATE_PRIOHIGH) ? "\\Flagged " : ""); \
            if (BUFFER[LEN - 1]==' ') { \
               LEN--; \
            } \
            LEN += sprintf(BUFFER + LEN, "))\r\n"); \
            SendClient(CLIENT, BUFFER, LEN); \
        } \
        MemFree(CLIENT->UnseenFlags); \
        CLIENT->UnseenFlags = NULL; \
    }

#define UNSEEN_FLAGS_BLOCK_SIZE 16


#define RememberFlags(UNSEEN_FLAGS, MSG_NUM, MSG_FLAGS) \
    if (UNSEEN_FLAGS ==  NULL) { \
        UNSEEN_FLAGS = MemMalloc((1 + UNSEEN_FLAGS_BLOCK_SIZE) * 2 * sizeof(unsigned long)); \
        if (UNSEEN_FLAGS) { \
            UNSEEN_FLAGS[2] = MSG_NUM; \
            UNSEEN_FLAGS[3] = MSG_FLAGS; \
            /* Set Counters */ \
            UNSEEN_FLAGS[0] = UNSEEN_FLAGS_BLOCK_SIZE;   /* Allocated */ \
            UNSEEN_FLAGS[1] = 1;           /* Used */ \
        } \
    } else { \
        if (UNSEEN_FLAGS[1] < UNSEEN_FLAGS[0]) { \
            UNSEEN_FLAGS[2 + (UNSEEN_FLAGS[1] * 2)]   = MSG_NUM; \
            UNSEEN_FLAGS[2 + (UNSEEN_FLAGS[1] * 2) + 1]  = MSG_FLAGS; \
            /* Set Counter */ \
            UNSEEN_FLAGS[1]++; \
        } else { \
            unsigned long *Tmp; \
            Tmp = MemRealloc(UNSEEN_FLAGS, ((1 + UNSEEN_FLAGS[0] + UNSEEN_FLAGS_BLOCK_SIZE) * 2 * sizeof(unsigned long))); \
            if (Tmp) { \
                UNSEEN_FLAGS = Tmp; \
                UNSEEN_FLAGS[2 + (UNSEEN_FLAGS[1] * 2)]  = MSG_NUM; \
                UNSEEN_FLAGS[2 + (UNSEEN_FLAGS[1] * 2) + 1] = MSG_FLAGS; \
                /* Set Counters */ \
                UNSEEN_FLAGS[0] += UNSEEN_FLAGS_BLOCK_SIZE; \
                UNSEEN_FLAGS[1]++; \
            } \
        } \
    }

#define FreeUnseenFlags(CLIENT) \
    if (CLIENT->UnseenFlags) { \
        MemFree(CLIENT->UnseenFlags); \
        CLIENT->UnseenFlags = NULL; \
    }


typedef struct _ImapGlobal {
    ProtocolCommandTree commands;

} ImapGlobal;

typedef enum /* all symbols */
{
    UIDsy = 1,    SEARCHsy,        CHARSETsy,     BODYsy,
    TEXTsy,       TOsy,            FROMsy,        SUBJECTsy,
    CCsy,         BCCsy,           HEADERsy,      NEWsy,
    OLDsy,        RECENTsy,        ANSWEREDsy,    UNANSWEREDsy,
    DELETEDsy,    UNDELETEDsy,     DRAFTsy,       UNDRAFTsy,
    FLAGGEDsy,    UNFLAGGEDsy,     SEENsy,        UNSEENsy,
    KEYWORDsy,    UNKEYWORDsy,     LARGERsy,      SMALLERsy,
    NOTsy,        ORsy,            BEFOREsy,      ONsy,
    SINCEsy,      SENTBEFOREsy,    SENTONsy,      SENTSINCEsy,
    ALLsy,        spacesy,         largestsy,     nz_numbersy,  
    atomsy,       stringsy,        endofbufsy,    crlfsy,
    crsy,         lfsy,            lbracesy,      rbracesy,
    commasy,      colonsy,         lparensy,      rparensy,
    lbracksy,     rbracksy,        numbersy,      datesy,
    errorsy
} SYMBOL;



#define IDENTSIZE 100
typedef struct
{
    SYMBOL sy;                           /* current symbol */
    unsigned char ch;                    /* current char */
    int offsetIn;                        /* current offset of bufferIn (Command) */
    int offsetOut;                       /* current offset of bufferIn */
    unsigned char bufferOut[ BUFSIZE+1]; /* bufferOut to NMAP */
    int identStart;                      /* identifier offset start */
    unsigned long identLen;              /* identifier length */
    unsigned char ident[ IDENTSIZE+2];   /* current identifier */
    BOOL resultByPosition;               /* search results by position or by uid */
    /*int numSearchKeys; */              /* number of search keys */
    unsigned char *Command;              /* short cut to Client->Command */
    void *Client;                        /* JIC baggage */
    unsigned long msgCount;
    unsigned long bvByteSize;
    unsigned long identNum;
    int messageSet;
} IMAPSEARCH;

#define ALLOCATED_FLAGS 1 << 0
#define ALLOCATED_PART_NUMBERS 1 << 1
#define ALLOCATED_FIELD_POINTERS 1 << 2
#define ALLOCATED_FIELDS 1 << 3

typedef struct {
    unsigned long start;
    unsigned long len;
} PartialRequestStruct;

#define FETCH_FIELD_ALLOC_THRESHOLD 3

typedef struct {
    unsigned long count;
    unsigned char **field;
    unsigned char *fieldEmbedded[FETCH_FIELD_ALLOC_THRESHOLD];
    unsigned char *fieldRequest;
    BOOL skip;
} HeaderFieldStruct;

typedef long (* FetchResponder)(void *param1, void *param2, void *param3);

#define PART_DEPTH_ALLOC_THRESHOLD 5

typedef struct {
    unsigned long depth;
    unsigned long *part;
    unsigned long partEmbedded[PART_DEPTH_ALLOC_THRESHOLD];
    long mimeOffset;
    unsigned long mimeLen;
    unsigned long partOffset;
    unsigned long partLen;
    BOOL isMessage;
    unsigned long messageHeaderLen;
    FetchResponder responder;
} BodyPartRequestStruct;

typedef struct {
    unsigned long imapId;
    unsigned long nmapId;
    unsigned long totalSize;
    unsigned long headerSize;
    unsigned long bodySize;
    unsigned long flags;
    unsigned long internalDate;
    unsigned char *header;
    MDBValueStruct *mimeInfo;
} MessageInfo;

typedef struct {
    FetchResponder responder;

    unsigned long hasAllocated;
    BOOL isPartial;
    PartialRequestStruct partial;
    HeaderFieldStruct fieldList;
    BodyPartRequestStruct bodyPart;
} FetchFlagStruct;

#define FETCH_FLAG_ALLOC_THRESHOLD 12

typedef struct {
    unsigned long flags;

    unsigned char *messageSet;
    unsigned char *nextRange;
    unsigned long rangeStart;
    unsigned long rangeEnd;

    unsigned long flagCount;
    FetchFlagStruct *flag;
    FetchFlagStruct flagEmbedded[FETCH_FLAG_ALLOC_THRESHOLD];
    unsigned long hasAllocated;

    unsigned char sendBuf[1024];
    unsigned char responseBuf[1024];

    MessageInfo currentMessage;
} FetchStruct;

typedef struct _RFC822AddressStruct {
    unsigned char      *Personal;
    unsigned char      *Mailbox;
    unsigned char      *Host;
    unsigned char      *ADL;
    struct _RFC822AddressStruct *Next;
} RFC822AddressStruct;

typedef struct {
    int                  s;       /* Socket       */
    int                  NMAPs;     /* Socket to NMAP server  */
    int                  State;     /* Connection state    */
    struct sockaddr_in   cs;      /* Client info      */
    SSL                  *CSSL;
    SSL                  *NSSL;
    unsigned char         MBox[XPL_MAX_PATH];  /* Greeting (for APOP)   */
    int                   BufferPtr;    /* Current input pointer  */
    int                   NBufferPtr;    /* Current NMAP input pointer */
    unsigned char         User[65];    /* Username       */
    unsigned char         Buffer[BUFSIZE+1]; /* Input buffer     */
    unsigned char         NBuffer[BUFSIZE+1]; /* NMAP Input buffer    */
    unsigned char         *Command;    /* Current command    */
    unsigned long         CommandLen;    /* Allocated size of command */
    unsigned char         Ident[65];    /* Ident for current command */
    unsigned char         SBuffer[MTU+1];  /* Send Buffer [speedup]  */
    int                   SBufferPtr;    /* Send Buffer Pointer   */
    int                   LoginCount;    /* Internal       */
    unsigned long         Messages;    /* Number of messages loaded */
    unsigned long         NMAPCount;    /* Total number of messages   */
    unsigned long         *UIDList;    /* UID Cache      */
    unsigned long         *IDList;     /* ID Cache       */
    BOOL                  ReadOnly;    /* NMAP MBOX is read-only  */
    IMAPSEARCH            *Search;
    unsigned char         ClientSSL;
    unsigned char         NMAPSSL;
    unsigned long         NMAPAsyncEvent;  /* Async Events received from NMAP */

    /* Cleanup variables */
    RFC822AddressStruct  *RFC822AddressList; /* Used by SendRFC822Address */
    unsigned char        *Header;     /* Used by HandleFetch()  */
    unsigned char        *MailSet;    /* Used by HandleFetch()  */
    unsigned char        **MultiPartTypeList; /* Used by BodyStructure  */
    long                 *RFCSizeList;   /* Used by BodyStructure  */
    unsigned char        *MIMEResponse;   /* Used by BodyStructure  */
    unsigned long        *UnseenFlags;   /* Used by HandleConnection() */

    /* Generic IP read and write functions - four since client could be SSL and NMAP non-ssl */
    /* Overloaded with NON-SSL,SSL,And Traceable recv and write functions*/
    GenericReadFunc       ClientRead;
    GenericWriteFunc      ClientWrite;
    void                  *ClientSktCtx; /* either a socket, or SSL context used as first parameter */

#if 0 // for now NMAP does not support SSL 
    GenericReadFunc       NMAPRead;
    GenericWriteFunc      NMAPWrite;
    void *NMAPSktCtx; /* either a socket, or SSL context used as first parameter */
#endif

} ImapClient;



typedef unsigned long (* FetchFlagParser)(unsigned char **ptr, FetchFlagStruct *flag);

typedef struct {
    unsigned char *name;
    unsigned long nameLen;
    unsigned long value;
    FetchFlagParser parser;
    FetchResponder responder;
} FetchFlag;

extern IOFuncs FuncTbl;
extern unsigned char *IMSearchVersion;
extern unsigned char Hostname[];
extern unsigned long HostnameLen;
extern unsigned char *MonthNames[];

extern MDBHandle IMAPDirectoryHandle;



BOOL                SendClient(ImapClient *Client, unsigned char *Data, int Len);
BOOL                FlushClient(ImapClient *Client);
BOOL                SendNMAPServer(ImapClient *Client, unsigned char *Data, int Len);
int                 GetNMAPAnswer(ImapClient *Client, unsigned char *Reply, int ReplyLen, BOOL CheckForResult);
int                 NBrecv(int s, unsigned char *buf, int len, int flags);
long                UIDtoNUM(ImapClient *Client, unsigned long UID);
BOOL                UIDtoNUMRange(ImapClient *Client, unsigned long StartUID, unsigned long EndUID, long *StartIndex, long *EndIndex);
long                IMAPSearch(ImapClient *Client, BOOL UIDMode);
void                InitSearchModule(void);

unsigned char       *RFC822SkipComment(unsigned char **String, unsigned long Trim);
void                RFC822SkipWhitespace(unsigned char **String);
unsigned char       *RFC822Quote(unsigned char *src);
unsigned char       *RFC822Copy(unsigned char *Source);
unsigned char       *RFC822ParseWord(unsigned char *s, const unsigned char *Delimiters);
unsigned char       *RFC822ParsePhrase(unsigned char *s);
RFC822AddressStruct *RFC822ParseAddrSpec(unsigned char *Address, unsigned char **ret, unsigned char *DefaultHost);
unsigned char       *RFC822ParseDomain(unsigned char *Address, unsigned char **end);
RFC822AddressStruct *RFC822ParseRouteAddress(unsigned char *Address, unsigned char **ret, unsigned char *DefaultHost);
RFC822AddressStruct *RFC822ParseMailbox(unsigned char **Address, unsigned char *DefaultHost);
RFC822AddressStruct *RFC822ParseGroup(RFC822AddressStruct **List, RFC822AddressStruct *Last, unsigned char **Address, char *DefaultHost, unsigned long Depth);
RFC822AddressStruct *RFC822ParseAddress(RFC822AddressStruct **List, RFC822AddressStruct *Last, unsigned char **Address, unsigned char *DefaultHost, unsigned long Depth);
BOOL                RFC822ParseAddressList(RFC822AddressStruct **List, unsigned char *Address, unsigned char *DefaultHost);
void                RFC822FreeAddressList(RFC822AddressStruct *List);
long                SendAsyncNewMessages(ImapClient *client);
unsigned char       *GrabArgument(ImapClient *client, unsigned char *Input, unsigned char **Destination);
BOOL                LoadUIDList(ImapClient *client);
BOOL                EndClientConnection(ImapClient *client);


void FetchCleanup(void);
BOOL FetchInit(void);

/* IMAP command prototypes */
int ImapCommandCapability(void *param);
int ImapCommandNoop(void *param);
int ImapCommandLogout(void *param);
int ImapCommandStartTls(void *param);
int ImapCommandAuthenticate(void *param);
int ImapCommandLogin(void *param);
int ImapCommandSelect(void *param);
int ImapCommandExamine(void *param);
int ImapCommandCreate(void *param);
int ImapCommandDelete(void *param);
int ImapCommandRename(void *param);
int ImapCommandSubscribe(void *param);
int ImapCommandUnsubscribe(void *param);
int ImapCommandList(void *param);
int ImapCommandLsub(void *param);
int ImapCommandStatus(void *param);
int ImapCommandAppend(void *param);
int ImapCommandCheck(void *param);
int ImapCommandClose(void *param);
int ImapCommandExpunge(void *param);
int ImapCommandSearch(void *param);
int ImapCommandUidSearch(void *param);
int ImapCommandFetch(void *param);
int ImapCommandUidFetch(void *param);
int ImapCommandStore(void *param);
int ImapCommandUidStore(void *param);
int ImapCommandCopy(void *param);
int ImapCommandUidCopy(void *param);
int ImapCommandSetAcl(void *param);
int ImapCommandDeleteAcl(void *param);
int ImapCommandGetAcl(void *param);
int ImapCommandListRights(void *param);
int ImapCommandMyRights(void *param);
int ImapCommandSetQuota(void *param);
int ImapCommandGetQuota(void *param);
int ImapCommandGetQuotaRoot(void *param);
int ImapCommandNameSpace(void *param);
int ImapCommandProxyAuth(void *param);
int ImapCommandQuit(void *param);
