/****************************************************************************
 *
 * Copyright (c) 1997-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#include <xpl.h>
#include <hulautil.h>
#include <logger.h>
#include <nmlib.h>
#include <mdb.h>
#include <xplresolve.h>

#include "mailprox.h"

/* NOTE:
 *
 * The IMAP proxy relies on the IMAP Pipelining capabilities
 * it will issue the fetch(message) request right when getting the response
 * to the fetch(uid) command. Same happens with the delete; we'll queue the
 * commands and after retrieving all the messages, flush and retrieve the
 * response. Read the RFC!
 *
 */
unsigned long 
ProxyIMAPAccount(MailProxyClient *client, ProxyAccount *proxy)
{
    int count;
    int ccode;
    int messages;
    int remaining;
    long j;
    long len;
    long id;
    long curUID;
    long stopUID;
    long startUID;
    long proxiedBytes;
    long proxiedMessages;
    unsigned char *ptr;
    unsigned char *uid;
    unsigned char tag[8];
    XplDnsRecord *dns = NULL;

    proxy->flags |= MAILPROXY_FLAG_STORE_UID;
    strcpy(client->lastUID, proxy->uid);

    len = strlen(proxy->host) - 1;
    if (proxy->host[len] == '.') {
        proxy->host[len] = '\0';
    }

    if ((client->conn = ConnAlloc(TRUE)) != NULL) {
        client->conn->socketAddress.sin_addr.s_addr = inet_addr(proxy->host);
        if (client->conn->socketAddress.sin_addr.s_addr == (unsigned long)-1) {
            ptr = strchr(proxy->host, '@');
            if (ptr) {
                ptr++;
            } else {
                ptr = proxy->host;
            }

            ccode = XplDnsResolve(ptr, &dns, XPL_RR_A);
            switch (ccode) {
                case XPL_DNS_SUCCESS: {
                    client->conn->socketAddress.sin_addr = dns->A.addr;
                    MemFree(dns);
                    break;
                }

                case XPL_DNS_BADHOSTNAME:
                case XPL_DNS_NORECORDS: {
                    proxy->flags |= MAILPROXY_FLAG_BAD_HOST;
                    LoggerEvent(MailProxy.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_BAD_HOSTNAME, LOG_ERROR, 0, proxy->host, client->q->user, 0, 0, NULL, 0);

                    MemFree(dns);
                    return(-1);
                }

                default:
                case XPL_DNS_FAIL:
                case XPL_DNS_TIMEOUT: {
                    MemFree(dns);

                    LoggerEvent(MailProxy.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_OUT_OF_MEMORY, LOG_ERROR, 0, __FILE__, client->q->user, sizeof(Connection), __LINE__, NULL, 0);
                    return(-1);
                }
            }
        }

        client->conn->socketAddress.sin_family = AF_INET;
        client->conn->socketAddress.sin_port = htons(proxy->port);
    } else {
        LoggerEvent(MailProxy.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_OUT_OF_MEMORY, LOG_ERROR, 0, __FILE__, client->q->user, sizeof(Connection), __LINE__, NULL, 0);
        return(-1);
    }

    /* fixme - this check only works on single server solutions */
    if (MsgGetHostIPAddress() != client->conn->socketAddress.sin_addr.s_addr) {
        if (!(proxy->flags & MAILPROXY_FLAG_SSL)) {
            ccode = ConnConnect(client->conn, NULL, 0, NULL);
        } else if (MailProxy.client.ssl.enable) {
            if (proxy->port == PORT_IMAP) {
                ccode = ConnConnect(client->conn, NULL, 0, NULL);
            } else {
                ccode = ConnConnect(client->conn, NULL, 0, MailProxy.client.ssl.context);
            }
        } else {
            ccode = -1;
        }
    } else {
        /* We don't want to proxy ourselves... */
        proxy->flags |= MAILPROXY_FLAG_BAD_HOST;
        proxy->flags &= ~MAILPROXY_FLAG_STORE_UID;

        LoggerEvent(MailProxy.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_OUT_OF_MEMORY, LOG_ERROR, 0, client->q->user, NULL, XplHostToLittle(client->conn->socketAddress.sin_addr.s_addr), 0, NULL, 0);
        return(-1);
    }

    /* IMAP greeting: * OK ... */
    if (ccode != -1) {
        if (proxy->flags & MAILPROXY_FLAG_SSL) {
            if (proxy->port == PORT_IMAP) {
                if (((ccode = ConnReadAnswer(client->conn, client->line, CONN_BUFSIZE)) != -1) 
                        && (client->line[0] == '*') 
                        && ((ccode = ConnWrite(client->conn, "0000 CAPABILITY\r\n", 17)) != -1) 
                        && ((ccode = ConnFlush(client->conn)) != -1) 
                        && ((ccode = ConnReadAnswer(client->conn, client->line, CONN_BUFSIZE)) != -1)) {
                    if (client->line[0] == '*') {
                        if (strstr(client->line, "STARTTLS") != NULL) {
                            ccode = ConnReadAnswer(client->conn, client->line, CONN_BUFSIZE);
                        } else {
                            ConnReadAnswer(client->conn, client->line, CONN_BUFSIZE);
                            ccode = -1;
                        }
                    } else {
                        ccode = -1;
                    }

                    if ((ccode != -1) 
                            && (XplStrNCaseCmp(client->line, "0000 OK", 7) == 0) 
                            && ((ccode = ConnWrite(client->conn, "0000 STARTTLS\r\n", 15)) != -1) 
                            && ((ccode = ConnFlush(client->conn)) != -1) 
                            && ((ccode = ConnReadAnswer(client->conn, client->line, CONN_BUFSIZE)) != -1) 
                            && (XplStrNCaseCmp(client->line, "0000 OK", 7) == 0)) {
                        client->conn->ssl.enable = TRUE;
                        ccode = ConnEncrypt(client->conn, MailProxy.client.ssl.context);
                    } else {
                        ccode = -1;
                    }
                } else {
                    ccode = -1;
                }
            } else {
                client->conn->ssl.enable = TRUE;
                if (((ccode = ConnReadAnswer(client->conn, client->line, CONN_BUFSIZE)) != -1) 
                        && (XplStrNCaseCmp(client->line, "* OK", 3) != 0)) {
                    ccode = -1;
                }
            }
        } else if (((ccode = ConnReadAnswer(client->conn, client->line, CONN_BUFSIZE)) != -1) 
                && (XplStrNCaseCmp(client->line, "* OK", 3) != 0)) {
            ccode = -1;
        }
    } else {
        proxy->flags |= MAILPROXY_FLAG_BAD_HANDSHAKE;
        LoggerEvent(MailProxy.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_IP_CONNECT_FAILURE, LOG_ERROR, 0, proxy->host, client->q->user, XplHostToLittle(client->conn->socketAddress.sin_addr.s_addr), 0, NULL, 0);
        return(-1);
    }

    if (ccode != -1) {
        startUID = stopUID = atol(proxy->uid) + 1;
    } else {
        proxy->flags |= MAILPROXY_FLAG_BAD_HANDSHAKE;
        LoggerEvent(MailProxy.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_NEGOTIATION_FAILED, LOG_ERROR, 0, proxy->host, client->q->user, XplHostToLittle(client->conn->socketAddress.sin_addr.s_addr), 0, NULL, 0);
        return(-1);
    }

    /* Expected IMAP conversation:
        Hula: 0000 LOGIN <user> <password>
        Hula: 0001 SELECT INBOX
        Hula: 0002 UID FETCH <last uid + 1>:* (UID)
        IMAP: 0000 OK LOGIN ... 
        IMAP: <untagged responses beginning with *>
        IMAP: 0001 OK [<state>] LOGIN ...
        IMAP: * <sequence number> FETCH (UID <uid>)
        IMAP: * <sequence number> FETCH (UID <uid>)
        IMAP: * <sequence number> FETCH (UID <uid>)
        IMAP: * <sequence number> FETCH (UID <uid>)
        IMAP: a003 OK UID FETCH ...

    */
    strcpy(tag, "0000 OK");
    if ((ccode != -1) 
            && ((ccode = ConnWriteF(client->conn, "0000 LOGIN %s %s\r\n", proxy->user, proxy->password)) != -1) 
            && ((ccode = ConnFlush(client->conn)) != -1) 
            && ((ccode = ConnReadAnswer(client->conn, client->line, CONN_BUFSIZE)) != -1) 
            && (XplStrNCaseCmp(client->line, tag, 7) == 0) 
            && ((ccode = ConnWrite(client->conn, "0001 SELECT INBOX\r\n", 19)) != -1) 
            && ((ccode = ConnFlush(client->conn)) != -1)) {
        do {
            ccode = ConnReadAnswer(client->conn, client->line, CONN_BUFSIZE);
            if (ccode != -1) {
                continue;
            }

            proxy->flags |= MAILPROXY_FLAG_BAD_HANDSHAKE;

            LoggerEvent(MailProxy.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_NETWORK_IO_FAILURE, LOG_ERROR, 0, proxy->host, client->q->user, XplHostToLittle(client->conn->socketAddress.sin_addr.s_addr), __LINE__, NULL, 0);
            return(-1);
        } while (client->line[0] == '*');

        tag[3] = '1';
    } else {
        if (ccode != -1) {
            XplSafeIncrement(MailProxy.stats.wrongPassword);

            proxy->flags |= MAILPROXY_FLAG_BAD_PASSWORD;
            LoggerEvent(MailProxy.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_WRONG_PASSWORD, LOG_ERROR, 0, client->q->user, proxy->password, XplHostToLittle(client->conn->socketAddress.sin_addr.s_addr), 0, NULL, 0);
        } else {
            proxy->flags |= MAILPROXY_FLAG_BAD_HANDSHAKE;
            LoggerEvent(MailProxy.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_NETWORK_IO_FAILURE, LOG_ERROR, 0, proxy->host, client->q->user, XplHostToLittle(client->conn->socketAddress.sin_addr.s_addr), __LINE__, NULL, 0);
        }

        return(-1);
    }

    if (XplStrNCaseCmp(client->line, tag, 7) == 0) {
        tag[3] = '2';

        messages = 0;
        remaining = 0;

        if ((ccode = ConnWriteF(client->conn, "0002 UID FETCH %lu:* (UID)\r\n", startUID)) != -1) {
            ccode = ConnFlush(client->conn);
        }
    } else {
        proxy->flags |= MAILPROXY_FLAG_BAD_HANDSHAKE;
        LoggerEvent(MailProxy.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_NETWORK_IO_FAILURE, LOG_ERROR, 0, proxy->host, client->q->user, XplHostToLittle(client->conn->socketAddress.sin_addr.s_addr), __LINE__, NULL, 0);
        return(-1);
    }

    while ((ccode != -1) 
            && ((count = ConnReadAnswer(client->conn, client->line, CONN_BUFSIZE)) != -1) 
            && (client->line[0] == '*') 
            && (sscanf(client->line, "* %lu FETCH", &id) == 1)) {
        uid = NULL;

        /* * <sequence number> FETCH (UID <uid>) */
        ptr = strchr(client->line, '(');
        if (ptr) {
            ptr++;

            do {
                while (*ptr && isspace(*ptr)) {
                    ptr++;
                }

                switch (toupper(*ptr)) {
                    case 'U': {
                        if ((toupper(ptr[1]) == 'I')
                                && (toupper(ptr[2]) == 'D')) {
                            ptr += 3;

                            while (*ptr && isspace(*ptr)) {
                                ptr++;
                            }

                            uid = ptr;
                            ptr = strchr(ptr, ')');
                            if (ptr) {
                                *ptr++ = '\0';
                                break;
                            }
                        }

                        uid = NULL;
                        ptr = NULL;
                        break;
                    }

                    default: {
                        uid = NULL;
                        ptr = NULL;
                        break;
                    }
                }
            } while (ptr && (ptr[0] != '\0'));
        }

        if ((uid) && ((curUID = atol(uid)) >= startUID)) {
            if (messages) {
                ccode = ConnWriteF(client->conn, ",%lu", id);
            } else {
                ccode = ConnWriteF(client->conn, "0003 FETCH %lu", id);
            }

            if (ccode != -1) {
                if (curUID > stopUID) {
                    stopUID = curUID;
                }

                messages++;
            }
        }

        continue;
    }

    if (XplStrNCaseCmp(client->line, tag, 7) == 0) {
        tag[3] = '3';
    } else {
        proxy->flags |= MAILPROXY_FLAG_BAD_HANDSHAKE;
        LoggerEvent(MailProxy.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_NETWORK_IO_FAILURE, LOG_ERROR, 0, proxy->host, client->q->user, XplHostToLittle(client->conn->socketAddress.sin_addr.s_addr), __LINE__, NULL, 0);
        return(-1);
    }

    if ((ccode != -1) 
            && (messages) 
            && ((ccode = ConnWrite(client->conn, " (BODY.PEEK[])\r\n", 16)) != -1) 
            && ((ccode = ConnFlush(client->conn)) != -1)) {
        client->nmap = NMAPConnect(MailProxy.nmap.address, NULL);
    } else if (ccode != -1) {
        /* We've already got all the messages, or there are none */
        LoggerEvent(MailProxy.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_PROXY_USER, LOG_WARNING, 0, client->q->user, proxy->host, 0, 0, NULL, 0);
        return(0);
    } else {
        proxy->flags |= MAILPROXY_FLAG_BAD_PROXY;
        LoggerEvent(MailProxy.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_NETWORK_IO_FAILURE, LOG_ERROR, 0, proxy->host, client->q->user, XplHostToLittle(client->conn->socketAddress.sin_addr.s_addr), __LINE__, NULL, 0);
        return(-1);
    }

    if ((client->nmap) && (NMAPAuthenticate(client->nmap, client->line, CONN_BUFSIZE) == TRUE)) {
        proxiedMessages = 0;
        proxiedBytes = 0;

        messages = 0;
    } else {
        if (client->nmap) {
            NMAPSendCommand(client->nmap, "QUIT\r\n", 6);
            NMAPReadAnswer(client->nmap, client->line, CONN_BUFSIZE, FALSE);
            LoggerEvent(MailProxy.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_NMAP_UNAVAILABLE, LOG_ERROR, 0, NULL, NULL, XplHostToLittle(client->conn->socketAddress.sin_addr.s_addr), 0, NULL, 0);
        } else {
            LoggerEvent(MailProxy.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_OUT_OF_MEMORY, LOG_ERROR, 0, __FILE__, client->q->user, sizeof(Connection), __LINE__, NULL, 0);
        }

        proxy->flags |= MAILPROXY_FLAG_BAD_PROXY;
        return(-1);
    }

    while ((ccode != -1) 
            && ((count = ConnReadAnswer(client->conn, client->line, CONN_BUFSIZE)) != -1) 
            && (client->line[0] == '*') 
            && (sscanf(client->line, "* %lu FETCH", &id) == 1)) {
        ptr = client->line + count - 1;
        if (*ptr == '}') {
            *ptr-- = '\0';
            while (ptr > client->line) {
                if (*ptr != '{') {
                    ptr--;
                    continue;
                }

                *ptr++ = '\0';
                break;
            }

            if (*ptr) {
                while (ptr && *ptr && isspace(*ptr)) {
                    ptr++;
                }
            }
        } else {
            ptr = NULL;
        }

        if (ptr && *ptr) {
            count = atol(ptr);
        } else {
            proxy->flags |= MAILPROXY_FLAG_BAD_PROXY;
            LoggerEvent(MailProxy.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_NEGOTIATION_FAILED, LOG_ERROR, 0, proxy->host, client->q->user, XplHostToLittle(client->conn->socketAddress.sin_addr.s_addr), 0, NULL, 0);
            return(-1);
        }

        ccode = NMAPSendCommandF(client->nmap, "QCREA\r\nQSTOR FLAGS 128\r\nQSTOR FROM %s %s -\r\nQSTOR LOCAL %s %s %d\r\nQSTOR MESSAGE %lu\r\n", MailProxy.postmaster, MailProxy.postmaster, client->q->user, client->q->user, NO_FORWARD, count);
        for (j = 0; (ccode != -1) && (j < 4); j++) {
            /* QSTOR does not have an immediate result [it comes once count bytes are sent] */
            if ((ccode = NMAPReadAnswer(client->nmap, client->line, CONN_BUFSIZE, FALSE)) != -1) {
                continue;
            }

            proxy->flags |= MAILPROXY_FLAG_BAD_PROXY;

            LoggerEvent(MailProxy.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_OUT_OF_MEMORY, LOG_ERROR, 0, client->q->user, "Create queue entry", ccode, 0, NULL, 0);
            XplConsolePrintf("hulamailprox: Could not create queue entry, NMAP error %d, user %s", ccode, client->q->user);

            return(-1);
        }

        if (((ccode = ConnReadToConn(client->conn, client->nmap, count)) != -1) 
                && ((ccode = ConnReadAnswer(client->nmap, client->line, CONN_BUFSIZE)) != -1) /* QSTOR MESSAGE */
                && ((ccode = ConnReadAnswer(client->conn, client->line, CONN_BUFSIZE)) != -1) 
                && (client->line[0] == ')')) {
            proxiedMessages++;
            proxiedBytes += count;

            if (((ccode = NMAPSendCommand(client->nmap, "QRUN\r\n", 6)) != -1)) {
                if ((ccode = NMAPReadAnswer(client->nmap, client->line, CONN_BUFSIZE, TRUE)) == NMAP_ENTRY_CREATED) {
                    if (!(proxy->flags & MAILPROXY_FLAG_LEAVE_MAIL)) {
                        if (messages) {
                            ccode = ConnWriteF(client->conn, ",%lu", id);
                        } else {
                            ccode = ConnWriteF(client->conn, "0004 STORE %lu", id);
                        }

                        if (ccode != -1) {
                            messages++;
                        }
                    }
                }
            }
        } else {
            NMAPSendCommand(client->nmap, "QABRT\r\n", 7);
        }
    }

    if (XplStrNCaseCmp(client->line, tag, 7) != 0) {
        proxy->flags |= MAILPROXY_FLAG_BAD_HANDSHAKE;
        LoggerEvent(MailProxy.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_NEGOTIATION_FAILED, LOG_ERROR, 0, proxy->host, client->q->user, XplHostToLittle(client->conn->socketAddress.sin_addr.s_addr), 0, NULL, 0);
        ConnFree(client->conn);
        client->conn = NULL;
        return(-1);
    }

    if (ccode != -1) {
        if (proxiedMessages) {
            sprintf(client->lastUID, "%lu", stopUID);

            if ((!(proxy->flags & MAILPROXY_FLAG_LEAVE_MAIL)) 
                    && ((ccode = ConnWrite(client->conn, " +FLAGS.SILENT (\\Deleted)\r\n", 27)) != -1) 
                    && ((ccode = ConnFlush(client->conn)) != -1)) {

                ccode = ConnReadAnswer(client->conn, client->line, CONN_BUFSIZE);
            }
        }

        proxiedBytes = (proxiedBytes + 1023) / 1024;

        LoggerEvent(MailProxy.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_PROXY_USER, LOG_WARNING, 0, client->q->user, proxy->host, proxiedMessages, proxiedBytes, NULL, 0);

        XplSafeAdd(MailProxy.stats.messages, proxiedMessages);
        XplSafeAdd(MailProxy.stats.kiloBytes, proxiedBytes);
    }

    return(proxiedMessages);
}
